# -*- coding: utf-8 -*-
#
# This file is for use with essm.
# Copyright (C) 2022 ETH Zurich, Swiss Data Science Center.
#
# essm is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public License as
# published by the Free Software Foundation; either version 2 of the
# License, or (at your option) any later version.
#
# essm is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with essm; if not, write to the
# Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
# MA 02111-1307, USA.
"""Variables defined in notebooks/extra_calculations/Intrinsic_permeability.ipynb and dependencies."""

from essm.variables import Variable
from sympy.physics.units import kilogram, meter, pascal, second


class L(Variable):
    """Length of the twig"""

    name = 'L'
    unit = meter
    assumptions = {'real': True}
    latex_name = 'L'
    default = None


class dP(Variable):
    """Pressure drop accross the twig"""

    name = 'dP'
    unit = pascal
    assumptions = {'real': True}
    latex_name = 'dP'
    default = None


class Q_m(Variable):
    """Mass flowrate"""

    name = 'Q_m'
    unit = kilogram / second
    assumptions = {'real': True}
    latex_name = 'Q_m'
    default = None


class Q_v(Variable):
    """Volume flowrate"""

    name = 'Q_v'
    unit = meter ** 3 / second
    assumptions = {'real': True}
    latex_name = 'Q_v'
    default = None


class dh(Variable):
    """Hydraulic head difference"""

    name = 'dh'
    unit = meter
    assumptions = {'real': True}
    latex_name = 'dh'
    default = None


class g(Variable):
    """Gravity"""

    name = 'g'
    unit = meter / second ** 2
    assumptions = {'real': True}
    latex_name = 'g'
    default = 9.81


class rho(Variable):
    """Density of fluid"""

    name = 'rho'
    unit = kilogram / meter ** 3
    assumptions = {'real': True}
    latex_name = '\\rho'
    default = 1000


class A(Variable):
    """Cross-sectional area of flow"""

    name = 'A'
    unit = meter ** 2
    assumptions = {'real': True}
    latex_name = 'A'
    default = None


class mu(Variable):
    """viscocity"""

    name = 'mu'
    unit = pascal * second
    assumptions = {'real': True}
    latex_name = '\\mu'
    default = None


class pi(Variable):
    """pi"""

    name = 'pi'
    unit = pascal * second
    assumptions = {'real': True}
    latex_name = '\\pi'
    default = 3.14


class K(Variable):
    """Conductivity in Sperry et al. 1988"""

    name = 'K'
    unit = kilogram / (meter * pascal * second)
    assumptions = {'real': True}
    latex_name = 'K'
    default = None
    expr = -L * Q_m / (A * dP)


class k(Variable):
    """Intrinsic permeabiility"""

    name = 'k'
    unit = meter ** 2
    assumptions = {'real': True}
    latex_name = 'k'
    default = None


__all__ = (
    'L',
    'dP',
    'Q_m',
    'Q_v',
    'dh',
    'g',
    'rho',
    'A',
    'mu',
    'pi',
    'K',
    'k',
)
