# *** -------------------------------------------------------------------------
# Data and Packages
# *** -------------------------------------------------------------------------

# load packages
library(mice)
library(miceadds)
library(mdmb)
library(jomo)
library(mitml)
library(lme4)

# load data
load(file = "example2.rda")

# set RNG seed
set.seed(5893)

# *** -------------------------------------------------------------------------
# Mising data methods
# *** -------------------------------------------------------------------------

# ***
# JM-HOM (JM1)
#

# convert w into a factor
dat.jm <- within(dat, w <- factor(w, levels = c(0, 1)))

# specify imputation model
fml.jm.hom <- list(
  x + y ~ 1 + (1|ID),
  w ~ 1
)

# run JM
imp.jm.hom <- mitml::jomoImpute(
  data = dat.jm, formula = fml.jm.hom, n.burn = 5000, n.iter = 50, m = 100
)

# fit analysis model
lst.jm.hom <- mitml::mitmlComplete(imp.jm.hom)
fit.jm.hom <- with(lst.jm.hom, lmer(y ~ (cwc(x, ID) + gm(x, ID)) * w + (1+cwc(x, ID)|ID)))

# pool results
mitml::testEstimates(fit.jm.hom, extra.pars = TRUE)

# ***
# JM (JM2)
#

# convert w into a factor
dat.jm <- within(dat, w <- factor(w, levels = c(0, 1)))

# specify imputation model
fml.jm <- list(
  x + y ~ 1 + (1|ID),
  w ~ 1
)

# run JM
imp.jm <- mitml::jomoImpute(
  data = dat.jm, formula = fml.jm, random.L1 = "mean", n.burn = 5000, n.iter = 50, m = 100
)

# fit analysis model
lst.jm <- mitml::mitmlComplete(imp.jm)
fit.jm <- with(lst.jm, lmer(y ~ (cwc(x, ID) + gm(x, ID)) * w + (1+cwc(x, ID)|ID)))

# pool results
mitml::testEstimates(fit.jm, extra.pars = TRUE)

# ***
# FCS-RI (FCS1)
#

# specify imputation model
pred.ri <- mice::make.predictorMatrix(dat)
pred.ri["x", "ID"] <- -2
pred.ri["x", c("y", "w")] <- c(3, 1)

meth.ri <- mice::make.method(dat)
meth.ri["x"] <- "2l.pan"

# run FCS
imp.fcs.ri <- mice::mice(
  data = dat, method = meth.ri, predictorMatrix = pred.ri, m = 100, maxit = 10
)

# fit analysis model
lst.fcs.ri <- mitml::mids2mitml.list(imp.fcs.ri)
fit.fcs.ri <- with(lst.fcs.ri, lmer(y ~ (cwc(x, ID) + gm(x, ID)) * w + (1+cwc(x, ID)|ID)))

# pool results
mitml::testEstimates(fit.fcs.ri, extra.pars = TRUE)

# ***
# FCS (FCS2)
#

# create data set with variables for nonlinear terms
dat.fcs <- dat
dat.fcs$my <- miceadds::gm(dat.fcs$y, dat.fcs$ID)
dat.fcs$cy <- dat.fcs$y - dat.fcs$my
dat.fcs$my2 <- dat.fcs$my^2
dat.fcs$cy2 <- dat.fcs$cy^2
dat.fcs$my_w <- dat.fcs$my * dat.fcs$w
dat.fcs$cy_w <- dat.fcs$cy * dat.fcs$w

# specify imputation model
pred <- mice::make.predictorMatrix(dat.fcs)
pred[,] <- 0
pred["x", "ID"] <- -2
pred["x", c("my", "cy", "w", "my2", "cy2", "my_w", "cy_w")] <- 1

meth <- mice::make.method(dat.fcs)
meth["x"] <- "2l.pan"

# run FCS
imp.fcs <- mice::mice(
  data = dat.fcs, method = meth, predictorMatrix = pred, m = 100, maxit = 10
)

# fit analysis model
lst.fcs <- mitml::mids2mitml.list(imp.fcs)
fit.fcs <- with(lst.fcs, lmer(y ~ (cwc(x, ID) + gm(x, ID)) * w + (1+cwc(x, ID)|ID)))

# pool results
mitml::testEstimates(fit.fcs, extra.pars = TRUE)

# ***
# SMC-SM
#

# specify model sequence
y.fml <- y ~ 1 + cwc(x, ID)*w + gm(x, ID)*w + (1+cwc(x, ID)|ID)
x.fml <- x ~ 1 + w + (1|ID)

y.mod <- list(model = "mlreg", formula = y.fml, sampling_level = "ID")
x.mod <- list(model = "mlreg", formula = x.fml, sampling_level = "ID")
   
# run SMC-SM
imp.smc <- mdmb::frm_fb(
  dat, dep = y.mod, ind = list(x = x.mod), burnin = 5000, iter = 15000, Nimp = 100,
  aggregation = TRUE
)

# fit analysis model
lst.smc = as.mitml.list(frm2datlist(imp.smc))
fit.smc <- with(lst.smc, lmer(y ~ (cwc(x, ID) + gm(x, ID)) * w + (1+cwc(x, ID)|ID)))

# pool results
mitml::testEstimates(fit.smc, extra.pars = TRUE)

# ***
# BE
#

# specify model sequence
y.fml <- y ~ 1 + cwc(x, ID)*w + gm(x, ID)*w + (1+cwc(x, ID)|ID)
x.fml <- x ~ 1 + w + (1|ID)

y.mod <- list(model = "mlreg", formula = y.fml, sampling_level = "ID")
x.mod <- list(model = "mlreg", formula = x.fml, sampling_level = "ID")

# fit analysis model with BE
fit.be <- mdmb::frm_fb(
  dat, dep = y.mod, ind = list(x = x.mod), burnin = 5000, iter = 15000,
  aggregation = TRUE
)

summary(fit.be)

