# *** -------------------------------------------------------------------------
# Data and Packages
# *** -------------------------------------------------------------------------

# load packages
library(mice)
library(miceadds)
library(mdmb)
library(jomo)
library(mitml)
library(lme4)

# load data
load(file = "example1.rda")

# set RNG seed
set.seed(3741)

# *** -------------------------------------------------------------------------
# Mising data methods
# *** -------------------------------------------------------------------------

# ***
# JM
#

# convert w into a factor
dat.jm <- within(dat, w <- factor(w, levels = c(0, 1)))

# specify imputation model
fml.jm <- list(
  x + y ~ 1 + (1|ID),
  w ~ 1
)

# run JM
imp.jm <- mitml::jomoImpute(
  data = dat.jm, formula = fml.jm, n.burn = 2500, n.iter = 50, m = 100
)

# fit analysis model
lst.jm <- mitml::mitmlComplete(imp.jm)
fit.jm <- with(lst.jm, lmer(y ~ cwc(x, ID) + gm(x, ID) + w + (1|ID)))

# pool results
mitml::testEstimates(fit.jm, extra.pars = TRUE)

# ***
# FCS
#

# specify imputation model
pred <- mice::make.predictorMatrix(dat)
pred[, "ID"] <- -2
pred["x", c("y", "w")] <- c(3, 1)
pred["y", c("x", "w")] <- c(3, 1)
pred["w", c("x", "y")] <- c(1, 1)

meth <- mice::make.method(dat)
meth[c("x", "y")] <- "2l.pan"
meth["w"] <- "2lonly.pmm"

# run FCS
imp.fcs <- mice::mice(
  data = dat, method = meth, predictorMatrix = pred, m = 100, maxit = 10
)

# fit analysis model
lst.fcs <- mitml::mids2mitml.list(imp.fcs)
fit.fcs <- with(lst.fcs, lmer(y ~ cwc(x, ID) + gm(x, ID) + w + (1|ID)))

# pool results
mitml::testEstimates(fit.fcs, extra.pars = TRUE)

# ***
# SMC-SM
#

# specify model sequence
y.fml <- y ~ 1 + cwc(x, ID) + gm(x, ID) + w + (1|ID)
x.fml <- x ~ 1 + w + (1|ID)
w.fml <- w ~ 1

y.mod <- list(model = "mlreg", formula = y.fml, sampling_level = "ID")
x.mod <- list(model = "mlreg", formula = x.fml, sampling_level = "ID")
w.mod <- list(model = "logistic", formula = w.fml, variable_level = "ID")
   
# run SMC-SM
imp.smc <- mdmb::frm_fb(
  dat, dep = y.mod, ind = list(x = x.mod, w = w.mod), burnin = 2500, iter = 7500,
  Nimp = 100, aggregation = TRUE
)

# fit analysis model
lst.smc = as.mitml.list(frm2datlist(imp.smc))
fit.smc <- with(lst.smc, lmer(y ~ cwc(x, ID) + gm(x, ID) + w + (1|ID)))

# pool results
mitml::testEstimates(fit.smc, extra.pars = TRUE)

# ***
# BE
#

# specify model sequence
y.fml <- y ~ 1 + cwc(x, ID) + gm(x, ID) + w + (1|ID)
x.fml <- x ~ 1 + w + (1|ID)
w.fml <- w ~ 1

y.mod <- list(model = "mlreg", formula = y.fml, sampling_level = "ID")
x.mod <- list(model = "mlreg", formula = x.fml, sampling_level = "ID")
w.mod <- list(model = "logistic", formula = w.fml, variable_level = "ID")

# fit analysis model with BE
fit.be <- mdmb::frm_fb(
  dat, dep = y.mod, ind = list(x = x.mod, w = w.mod), burnin = 2500, iter = 7500,
  aggregation = TRUE
)

summary(fit.be)

