# -*- coding: utf-8 -*-
#
# Copyright (C) 2008-2014 Jonathan F. Donges
# Author: Jonathan F. Donges <donges@pik-potsdam.de>
# URL: <http://www.pik-potsdam.de/members/donges/software>

"""
Plots results of recurrence analysis of an ensemble of time series generated by the COPRA algorithm and the corresponding single proxy record.

This script provides analyses for this publication:

J.F. Donges, R.V. Donner, N. Marwan, S.F.M. Breitenbach, K. Rehfeld, and J. Kurths,
Nonlinear regime shifts in Holocene Asian monsoon variability: Potential impacts on cultural change and migratory patterns,
Climate of the Past 11, 709-741 (2015),
DOI: 10.5194/cp-11-709-2015

For the COPRA algorithm refer to:

S.F.M. Breitenbach  et al. (2012),
COnstructing Proxy Records from Age models (COPRA),
Climate of the Past 8, 1765–1779, doi: 10.5194/cp-8-1765-2012.
"""

#
#  Imports
#

#  Import sys for command line arguments
import sys

#  Import cPickle for loading and saving data
import cPickle

#  Import np for fast numerics
import numpy as np

#  Import pylab for plotting
import pylab

#
#  Settings
#

#  Settings for statistics

#  q-quantile
Q_QUANTILE = 0.10

#  Settings for plotting

#  List of letters from "A" to "Z"
figure_labels = map(chr, range(65, 91))

#  Toggle plotting of confidence band
SHOW_CONFIDENCE_BAND = True

#  Set labels
AGE_LABEL = "Age (ka B.P.)"
DATA_LABEL = r"$\Delta \delta^{18}$O" #  (‰ VPDB)

#  Settings for plotting additional information

# See Mayewski et al., Quat. Sci. Rev. (2004)
BOND_EVENTS = np.array([10.3, 9.4, 8.2, 5.9, 4.35, 2.8, 1.2, 0.5])
RCC_EPISODES = np.array([[8., 9.], [6., 5.], [3.8, 4.2], [2.5, 3.5], [1., 1.2],
                        [0.15, 0.6]])

#
#  Load COPRA results for plotting
#

#  Load from file
filename = sys.argv[1]
print "Loading results from", filename

file = open(filename, 'r')
storage = cPickle.load(file)
file.close()

#  Store parameters
FILENAME = storage["FILENAME"]
NAME = storage["NAME"]

T_TIME = storage["T_TIME"]
DELTA_TIME = storage["DELTA_TIME"]
DETRENDING_WINDOW_SIZE = storage["DETRENDING_WINDOW_SIZE"]

DIM = storage["DIM"]
TAU = storage["TAU"]
METRIC = storage["METRIC"]
RR = storage["RR"]

N_ENSEMBLE = storage["N_ENSEMBLE"]
SHUFFLE_EMBEDDED = storage["SHUFFLE_EMBEDDED"]

#  Store symbols
symbols = storage["symbols"]

#  Store raw input data
time = storage["time"]
values = storage["values"]

#  Store axes
step_sequence = storage["step_sequence"]

#  Store results
results = storage["results"]
surrogate_results = storage["surrogate_results"]

#  Derived properties
print "Number of realizations for significance test:", N_ENSEMBLE

#
#  Load RAW data results for plotting
#

filename_raw = sys.argv[2]
print "Loading results from", filename_raw

file = open(filename_raw, 'r')
storage_raw = cPickle.load(file)
file.close()

#  Get names of records stored in file
NAMES_raw = storage_raw["NAMES"]

record_index = NAMES_raw.index("Dongge")

#  Store raw input data
time_raw = storage_raw["time_list"][record_index] / 1000.
data_raw = storage_raw["data_list"][record_index]

#  Store axes
step_sequence_raw = storage_raw["step_sequence"][record_index]

#  Store results
results_raw = storage_raw["results"]
surrogate_results_raw = storage_raw["surrogate_results"]

#  Store ensemble size of surrogate results
N_ENSEMBLE_RAW = storage_raw["N_ENSEMBLE"]

#  Get number of time series
n_time_series_raw = len(NAMES_raw)

#
#  Prepare plotting of COPRA ensemble results
#

#  Number of elements right and left of the quantile
#  Divide by two, such that the total probability of false rejection of null
#  hypothesis
#  is equal to Q_QUANTILE (two sided test).
q_elements = int(Q_QUANTILE / 2 * N_ENSEMBLE)

#  Get statistics for time series values
values.sort(axis=0)

values_median = np.median(values, axis=0)
values_left = values[q_elements - 1, :]
values_right = values[-q_elements, :]

#  Get statistics for results
median = {}
q_left = {}
q_right = {}

for measure in symbols.keys():
    results[measure].sort(axis=0)

    #  Get median
    median[measure] = np.median(results[measure], axis=0)

    #  Get q-quantiles
    q_left[measure] = results[measure][q_elements - 1, :]
    q_right[measure] = results[measure][-q_elements, :]

#
#  Calculate q-quantiles for COPRA surrogates
#

surr_q_left = {}
surr_q_right = {}

for measure in symbols.keys():
    #  Get quantiles from surrogates from all time series realizations for now
    surrogate_results[measure] = surrogate_results[measure].flatten()

    #  Sort
    surrogate_results[measure].sort()

    #  Get q-quantiles
    surr_q_left[measure] = surrogate_results[measure][q_elements - 1]
    surr_q_right[measure] = surrogate_results[measure][-q_elements]

#
#  Calculate q-quantiles for RAW data surrogates
#

q_left_raw = {}
q_right_raw = {}
surrogate_mean_raw = {}

for measure in symbols.keys():
    q_left_raw[measure] = []
    q_right_raw[measure] = []
    surrogate_mean_raw[measure] = []

#  Number of elements right and left of the quantile
#  Divide by two, such that the total probability of false rejection of null hypothesis
#  is equal to Q_QUANTILE (two sided test).
q_elements_raw = int(Q_QUANTILE / 2 * N_ENSEMBLE_RAW)

for i in xrange(n_time_series_raw):
    for measure in symbols.keys():
        surrogate_results_raw[measure][i].sort()
        #  Get mean
        surrogate_mean_raw[measure].append(surrogate_results_raw[measure][i].mean())

        #  Get q-quantiles
        q_left_raw[measure].append(surrogate_results_raw[measure][i][q_elements_raw - 1])
        q_right_raw[measure].append(surrogate_results_raw[measure][i][-q_elements_raw])

#
#  Rescale time axis
#

time /= 1000.

#
#  Plotting
#

#  Set plotting parameters (A4 size)
# params = { 'figure.figsize': (8.268,11.693),
#           'axes.labelsize': 12,
#           'text.fontsize': 12,
#           'xtick.labelsize': 12,
#           'ytick.labelsize': 12,
#           'legend.fontsize': 10,
#           'title.fontsize': 12,
#           'text.usetex': False,
#           'font': 'Helvetica',
#           'mathtext.bf': 'helvetica:bold',
#            'xtick.major.pad': 6,
#            'ytick.major.pad': 6,
#            'xtick.major.size': 5,
#            'ytick.major.size': 5,
#            'tick.labelsize': 'small'
#            }

#  Set plotting parameters (for Clim. Past paper)
params = { 'figure.figsize': (8.268, 6.),
          'axes.labelsize': 12,
          'text.fontsize': 12,
          'xtick.labelsize': 12,
          'ytick.labelsize': 12,
          'legend.fontsize': 10,
          'title.fontsize': 12,
          'text.usetex': False,
          'font': 'Helvetica',
          'mathtext.bf': 'helvetica:bold',
          'xtick.major.pad': 6,
          'ytick.major.pad': 6,
          'xtick.major.size': 5,
          'ytick.major.size': 5,
          'xtick.minor.size': 3,      # minor tick size in points
          'xtick.major.width': 1.,    # major tick width in points
          'xtick.minor.width': 1.,    # minor tick width in points
          'ytick.minor.size': 3,      # minor tick size in points
          'ytick.major.width': 1.,    # major tick width in points
          'ytick.minor.width': 1.,    # minor tick width in points
          'tick.labelsize': 'small'
           }

#pylab.rcParams.update(params)

#  Get overall minimum and maximum age
min_age = time.min()
max_age = time.max()

#
#  Plot overview figure
#

#  Create new figure
fig = pylab.figure()

#  Plot original data including dating uncertainties
ax = fig.add_subplot(len(symbols) + 1,1,1)

#  Plot COPRA records
pylab.plot(time, values_median, "#DB232E", linewidth=2)
pylab.fill_between(time, values_left, values_right, color="#F1B08E")

#  Plot original raw record
pylab.plot(time_raw, data_raw + 1., "k")

#  Plot additional information
#  Plot RCC episodes
for k in xrange(RCC_EPISODES.shape[0]):
    pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], fill=True,
                  edgecolor="0.8", color="0.8", zorder=0)

#  Plot Bond events
pylab.scatter(BOND_EVENTS, 2.0 * np.ones(len(BOND_EVENTS)), c='k', marker='o',
              s=80)

pylab.ylabel(DATA_LABEL, fontsize=16)
pylab.xlim(min_age, max_age)

#  Add figure label
ax.annotate(figure_labels[0], xy=(-0.14, 1.1),
            xycoords='axes fraction',
            horizontalalignment='left', verticalalignment='top',
            fontname='Helvetica', fontsize=14, fontweight='bold')

#  Render time labels invisible
#ax.axes.get_xaxis().set_visible(False)

#  Enable minor ticks
#ax.minorticks_on()

#  Add time labels and axis label to top subplot
#ax.xaxis.tick_top()
ax.xaxis.set_label_position("top")
ax.tick_params(labelbottom='off',labeltop='on')

pylab.xlabel(AGE_LABEL)

#  Add text
ax.text(0.99, 0.15, "COPRA", horizontalalignment='right',
        verticalalignment='top', transform = ax.transAxes)
ax.text(0.99, 0.80, "RAW", horizontalalignment='right',
        verticalalignment='top', transform = ax.transAxes)

#  Set number of ticks and tick labels
ax.locator_params(axis="y", tight=True, nbins=5)

#  Set colors for results subplots
colors_line = ["#3A89CB", "#9CD39A"]
colors_fill = ["#DDEAED", "#DBEDDA"]

#  Define shift of raw record results
shift_raw = [3, 0.2]

#  Plot windowed measures
j = 1
for measure in symbols.keys():
    j += 1
    ax = fig.add_subplot(len(symbols) + 1,1,j)

    #  Plot COPRA results
    pylab.plot(time[step_sequence], median[measure], colors_line[j - 2],
               linewidth=2)
    pylab.plot(time[step_sequence], q_left[measure], colors_line[j - 2],
               linewidth=1)
    pylab.plot(time[step_sequence], q_right[measure], colors_line[j - 2],
               linewidth=1)
    #pylab.fill_between(time[step_sequence], q_left[measure], q_right[measure], color=colors_fill[j - 2])

    #  Fill significant episodes for COPRA results
    pylab.fill_between(time[step_sequence], surr_q_right[measure],
                       q_right[measure], where=q_right[measure] > surr_q_right[measure], color=colors_line[j - 2])

    pylab.ylabel(symbols[measure], fontsize=16)

    #  Plot additional information
    #  Plot RCC episodes
    for k in xrange(RCC_EPISODES.shape[0]):
        pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], fill=True,
                      edgecolor="0.8", color="0.8", zorder=0)

    #  Plot Bond events
    pylab.scatter(BOND_EVENTS, 1.2 * q_right[measure].max() * np.ones(len(BOND_EVENTS)), c='k', marker='o', s=80)

    #  Plot COPRA confidence band
    if SHOW_CONFIDENCE_BAND:
        #pylab.axhline(y=surrogate_mean[measure][i], linewidth=1, color="r")
        #pylab.fill_between(time[step_sequence], surr_q_left[measure], surr_q_right[measure], color="0.75")
        #pylab.axhline(y=surr_q_left[measure], linewidth=1, color="r")
        #pylab.axhline(y=surr_q_right[measure], linewidth=1, color="r")
        pylab.axhspan(surr_q_left[measure], surr_q_right[measure], fill=True,
                      color=colors_fill[j - 2], zorder=0)

    #  Plot results for original raw data
    pylab.plot(time_raw[step_sequence_raw], results_raw[measure][record_index] + shift_raw[j - 2], "k")
    #  Fill significant episodes
    pylab.fill_between(time_raw[step_sequence_raw], q_right_raw[measure][record_index] + shift_raw[j - 2], results_raw[measure][record_index] + shift_raw[j - 2],
      where=results_raw[measure][record_index] > q_right_raw[measure][record_index], color=colors_line[j - 2])
    pylab.fill_between(time_raw[step_sequence_raw], q_left_raw[measure][record_index] + shift_raw[j - 2], results_raw[measure][record_index] + shift_raw[j - 2],
      where=results_raw[measure][record_index] < q_left_raw[measure][record_index], color=colors_line[j - 2])

    #  Plot RAW data confidence band
    if SHOW_CONFIDENCE_BAND:
        #pylab.fill_between(time_raw[step_sequence_raw], q_left_raw[measure][record_index] + shift_raw[j - 2], q_right_raw[measure][record_index] + shift_raw[j - 2], color="0.75")
        pylab.axhspan(q_left_raw[measure][record_index] + shift_raw[j - 2],
                      q_right_raw[measure][record_index] + shift_raw[j - 2],
                      fill=True, color=colors_fill[j - 2], zorder=0)

    pylab.xlim(min_age, max_age)

    #  Add figure label
    ax.annotate(figure_labels[j - 1], xy=(-0.14, 1.1),
                xycoords='axes fraction',
                horizontalalignment='left', verticalalignment='top',
                fontname='Helvetica', fontsize=14, fontweight='bold')

    #  Add text
    ax.text(0.99, 0.15, "COPRA", horizontalalignment='right',
            verticalalignment='top', transform = ax.transAxes)
    ax.text(0.99, 0.80, "RAW", horizontalalignment='right',
            verticalalignment='top', transform = ax.transAxes)

    #  Enable minor ticks
    #ax.minorticks_on()

    #  Set number of ticks and tick labels
    #ax.locator_params(axis="x", tight=True, nbins=9)
    ax.locator_params(axis="y", tight=True, nbins=5)

    #  Render time labels invisible
    if j < 3:
        ax.set_xticklabels([])

pylab.xlabel(AGE_LABEL)

#  Save figure
pylab.savefig(NAME + "_copra_overview.pdf")

pylab.show()
