# -*- coding: utf-8 -*-
#
# Copyright (C) 2008-2014 Jonathan F. Donges
# Author: Jonathan F. Donges <donges@pik-potsdam.de>
# URL: <http://www.pik-potsdam.de/members/donges/software>

"""
Performs recurrence analysis of an ensemble of time series generated by the
COPRA algorithm corresponding to a single proxy record.

This script provides analyses for this publication:

J.F. Donges, R.V. Donner, N. Marwan, S.F.M. Breitenbach, K. Rehfeld, and J. Kurths,
Nonlinear regime shifts in Holocene Asian monsoon variability: Potential impacts on cultural change and migratory patterns,
Climate of the Past 11, 709-741 (2015),
DOI: 10.5194/cp-11-709-2015

For the COPRA algorithm refer to:

S.F.M. Breitenbach  et al. (2012),
COnstructing Proxy Records from Age models (COPRA),
Climate of the Past 8, 1765–1779, doi: 10.5194/cp-8-1765-2012.
"""

#
#  Imports
#

import cPickle

#  Import np for fast numerics
import numpy as np

#  Import progress bar for easy progress bar handling
import progressbar

#  Import class for recurrence network analysis
from pyunicorn.timeseries import RecurrenceNetwork

#
#  Settings
#

#  Name of data directory
DATA_DIR = "../../data/copra_ensemble_data/"

#  Give filename
FILENAME = "dongge_realisations.txt"

#  Give name of record
NAME = "DONGGE"

#  Specify symbol used for commenting in data file
COMMENT_SYMBOL = "%"

#  Settings for the time dependent recurrence plot
#  Window length [a] / [ka]
T_TIME = 750.
#  Step size [a] / [ka]
DELTA_TIME = 50.

#  Settings for the embedding
DIM = 3 # embedding dimension
DELAY = 185. # embedding delay in units of [a] / [ka]

#  Settings for the recurrence plot
METRIC = "supremum"
RR = 0.05

#  Settings for significance testing

#  Ensemble size
N_ENSEMBLE = 100
#  Choose whether whole embedded state vectors or the scalar time series should be shuffled (Different null-hypothesis!)
SHUFFLE_EMBEDDED = True

#  Settings for detrending
DETREND = True
DETRENDING_WINDOW_SIZE = 1000. # measured in [a] / [ka]

#
#  Functions
#

def detrend_time_series(data, window_size):
    """
    """
    #  Get length of data array
    n = data.shape[0]
    #  Initialize a local copy of data array
    detrended_data = np.empty(n)

    #  Detrend data
    for j in xrange(n):
        #  Get distance of sample from boundaries of time series
        dist = min(j, n - 1 - j)

        if window_size / 2 > dist:
            half_size = dist
        else:
            half_size = window_size / 2

        detrended_data[j] = data[j] - data[j - half_size:j + half_size + 1].mean()

    return detrended_data

#
#  The main script
#

print "Analyzing ensemble of time series generated by COPRA"
print "----------------------------------------------------"

#
#  Import data
#

print "Loading data..."

#  Load data
data = np.loadtxt(DATA_DIR + FILENAME, comments=COMMENT_SYMBOL).transpose()

print "Mean sampling time:", np.diff(data[0,:]).mean()
print "STD sampling time:", np.diff(data[0,:]).std()

#  Take only every fourth data point, reduces time-resolution of COPRA records
#  Also, look at smaller number of 100 COPRA realizations
time = data[0, ::4]
values = data[1:101, ::4]

#  Remove NaNs
is_not_nan = np.all(np.logical_not(np.isnan(values)), axis=0)
time = time[is_not_nan]
values = values[:, is_not_nan]

#  Get time series properties
average_sampling_time = np.diff(time).mean()
n_realizations = values.shape[0]

if DETREND:
    #  Detrend data!
    print "Detrending data..."

    for i in xrange(n_realizations):
        values[i,:] = detrend_time_series(data=values[i,:], window_size=DETRENDING_WINDOW_SIZE / average_sampling_time)

#
#  Print some statistics
#

print "Average sampling time:"
print NAME, ": (", np.diff(time).mean(), "pm", np.diff(time).std(), ") a"

#
#  Analyze time dependent recurrence networks by moving a window over the time series
#

#  Create dictionary of symbols for each windowed measure to be calculated
symbols = {"Transitivity": "$\mathcal{T}$",
           "Average path length": "$\mathcal{L}$"}

#symbols = {"Average path length": "$\mathcal{L}$",
#           "n.s.i. average path length": "$\mathcal{L}^*$",
#           "Clustering": "$\mathcal{C}$",
#           "n.s.i. clustering": "$\mathcal{C}^*$"}

#symbols = {"Determinism": "$DET$",
#           "Laminarity": "$LAM$",
#           "Mean diagonal line length": "$L_{mean}$",
#           "Trapping time": "$TT$",
#           "Diagonal line entropy": "$ENTR$",
#           "Autocorrelation": "$ACF(1)$",
#           "Mean": "Mean",
#           "Standard deviation": "STD"}

#  Get window and step size in units of samples
T = int(T_TIME / average_sampling_time)
delta = int(DELTA_TIME / average_sampling_time)

#  Get embedding delay in units of samples
TAU = int(DELAY / average_sampling_time)

#  Get length of time series
t_max = len(time)

#  Get required time series length before embedding to achive window length T
#  in the recurrence plot
T_embedded = T + (DIM - 1) * TAU

#  Get number of steps
t_steps = int((t_max - T_embedded) / float(delta) + 1)

print "Length of dust record:", t_max
print "Size of moving window:", T
print "Step size:", delta
print "Number of steps for moving window:", t_steps
print "Embedding dimension:", DIM
print "Embedding delay:", TAU
print "Prescribed link density / recurrence rate:", RR

#  Initialize
step_sequence = np.zeros(t_steps, dtype=int)

results = {}
surrogate_results = {}

for measure in symbols.keys():
    results[measure] = np.zeros((n_realizations,t_steps))
    surrogate_results[measure] = np.zeros((n_realizations,N_ENSEMBLE))

print "Analysing time series realizations..."

#  Initialize progress bar
progress = progressbar.ProgressBar().start()

#  Run analysis for each realization separately
for i in xrange(n_realizations):
    #  Loop over moving windows
    for j in xrange(t_steps):
        #  Get time series section for current window
        time_series = values[i,j * delta:j * delta + T_embedded]
        step_sequence[j] = j * delta + T_embedded / 2

        #  Prepare recurrence network from original data
        rec_net = RecurrenceNetwork(time_series.copy(), dim=DIM, tau=TAU,
                                    metric=METRIC, normalize=False,
                                    silence_level=2, recurrence_rate=RR)

        #  Calculations for original recurrence network
        results["Transitivity"][i,j] = rec_net.transitivity()
        results["Average path length"][i,j] = rec_net.average_path_length()

        #results["Assortativity"][i,j] = rec_net.assortativity()
        #results["Diameter"][i,j] = rec_net.diameter()

        #  Calculate RQA measures
        #local_result["Determinism"][j] = rec_net.determinism()
        #local_result["Laminarity"][j] = rec_net.laminarity()
        #local_result["Mean diagonal line length"][j] = rec_net.average_diaglength()
        #local_result["Trapping time"][j] = rec_net.trapping_time()
        #local_result["Diagonal line entropy"][j] = rec_net.diag_entropy()
        #local_result["Autocorrelation"][j] = autocorrelation(time_series,
        #                                                      lag=1)
        #local_result["Mean"][j] = time_series.mean()
        #local_result["Standard deviation"][j] = time_series.std()

    #
    #  Calculate significance levels for network measures
    #

    #  Create a copy of time series realization for generating surrogates from
    surrogate_data = values[i,:].copy()

    if SHUFFLE_EMBEDDED:
        #  Get embedding of full time series
        surrogate_embedding = rec_net.embed_time_series(surrogate_data, DIM,
                                                        TAU)

    for j in xrange(N_ENSEMBLE):
        if SHUFFLE_EMBEDDED:
            #  Shuffle embedded time series along time axis, that is,
            #  whole embedded state vectors are
            #  shuffled around.
            permuted_indices = np.random.permutation(surrogate_embedding.shape[0])

            #  Use the first T state vectors from the shuffled and embedded
            #  time series as a surrogate for one window
            surrogate_series = surrogate_embedding[permuted_indices[:T],:]

            #  Prepare recurrence network from surrogate data for shuffled
            #  embedded time series
            rec_net = RecurrenceNetwork(surrogate_series.copy(), metric=METRIC,
                                        normalize=False, silence_level=2,
                                        recurrence_rate=RR)
        else:
            #  Shuffle dust time series
            permuted_indices = np.random.permutation(surrogate_data.shape[0])

            #  Use the first T_embedded states from the shuffled dust time
            #  series as a surrogate for one window
            surrogate_series = surrogate_data[permuted_indices[:T_embedded]]

            #  Prepare recurrence network from surrogate data for shuffled
            #  time series
            rec_net = RecurrenceNetwork(surrogate_series.copy(), dim=DIM,
                                        tau=TAU, metric=METRIC,
                                        normalize=False, silence_level=2,
                                        recurrence_rate=RR)

        #  Calculate measures for surrogate network
        surrogate_results["Average path length"][i,j] = rec_net.average_path_length()
        surrogate_results["Transitivity"][i,j] = rec_net.transitivity()

        #surrogate_results["Assortativity"][i,j] = rec_net.assortativity()
        #surrogate_results["Diameter"][i,j] = rec_net.diameter()

        #surrogate_result["Determinism"][i,j] = rec_net.determinism()
        #surrogate_result["Laminarity"][i,j] = rec_net.laminarity()

        #  Calculate RQA measures
        #local_surrogate_result["Determinism"][j] = rec_net.determinism()
        #local_surrogate_result["Laminarity"][j] = rec_net.laminarity()
        #local_surrogate_result["Mean diagonal line length"][j] = rec_net.average_diaglength()
        #local_surrogate_result["Trapping time"][j] = rec_net.trapping_time()
        #local_surrogate_result["Diagonal line entropy"][j] = rec_net.diag_entropy()
        #local_surrogate_result["Autocorrelation"][j] = autocorrelation(data, lag=1)
        #local_surrogate_result["Mean"][j] = data.mean()
        #local_surrogate_result["Standard deviation"][j] = data.std()

    #  Update progress bar every step
    progress.update(int(100 * i / float(n_realizations)))

#  Terminate progress bar
progress.finish()

#
#  Save results
#

print "Saving results..."

#  Initialize storage dictionary
storage = {}

#  Store parameters
storage["FILENAME"] = FILENAME
storage["NAME"] = NAME

storage["T_TIME"] = T_TIME
storage["DELTA_TIME"] = DELTA_TIME
storage["DETRENDING_WINDOW_SIZE"] = DETRENDING_WINDOW_SIZE

storage["DIM"] = DIM
storage["TAU"] = TAU

storage["METRIC"] = METRIC
storage["RR"] = RR

storage["N_ENSEMBLE"] = N_ENSEMBLE
storage["SHUFFLE_EMBEDDED"] = SHUFFLE_EMBEDDED

#  Store symbols
storage["symbols"] = symbols

#  Store raw input data
storage["time"] = time
storage["values"] = values

#  Store axes
storage["step_sequence"] = step_sequence

#  Store results
storage["results"] = results
storage["surrogate_results"] = surrogate_results

#  Save to file
filename = "results_copra_ensemble_DIM_" + str(DIM) + "_TAU_" + str(TAU) + "_M_" + str(N_ENSEMBLE) + "_DETREND_" + str(DETREND) + "_" + NAME + ".pickle"
file = open(filename, 'w')
cPickle.dump(storage, file)
file.close()
