# -*- coding: utf-8 -*-
#
# Copyright (C) 2008-2014 Jonathan F. Donges
# Author: Jonathan F. Donges <donges@pik-potsdam.de>
# URL: <http://www.pik-potsdam.de/members/donges/software>

"""
Plots results of recurrence analysis of multiple ensembles of time series
generated by the COPRA algorithm.

This script provides analyses for this publication:

J.F. Donges, R.V. Donner, N. Marwan, S.F.M. Breitenbach, K. Rehfeld, and J. Kurths,
Nonlinear regime shifts in Holocene Asian monsoon variability: Potential impacts on cultural change and migratory patterns,
Climate of the Past 11, 709-741 (2015),
DOI: 10.5194/cp-11-709-2015

For the COPRA algorithm refer to:

S.F.M. Breitenbach  et al. (2012),
COnstructing Proxy Records from Age models (COPRA),
Climate of the Past 8, 1765–1779, doi: 10.5194/cp-8-1765-2012.
"""

#
#  Imports
#

#  Import sys for command line arguments
import sys

#  Import cPickle for loading and saving data
import cPickle

#  Import np for fast numerics
import numpy as np

#  Import pylab for plotting
import pylab

#
#  Settings
#

#  Settings for statistics

#  q-quantile
Q_QUANTILE = 0.10

#  Settings for plotting

#  List of letters from "A" to "Z"
figure_labels = map(chr, range(65, 91))

#  Toggle plotting of confidence band
SHOW_CONFIDENCE_BAND = True

#  Set labels
AGE_LABEL = "Age (ka B.P.)"
DATA_LABEL = r"$\Delta \delta^{18}$O" #  (‰ VPDB)

#  Settings for plotting additional information

# See Mayewski et al., Quat. Sci. Rev. (2004)
BOND_EVENTS = np.array([10.3, 9.4, 8.2, 5.9, 4.35, 2.8, 1.2, 0.5])
RCC_EPISODES = np.array([[8., 9.], [6., 5.], [3.8, 4.2], [2.5, 3.5], [1., 1.2],
                        [0.15, 0.6]])

#
#  List of data files
#

#  List files in the desired plotting order
DATA_FILE_NAMES = ["results_copra_ensemble_DIM_3_TAU_25_M_100_DETREND_True_DIMARSHIM.pickle",
                   "results_copra_ensemble_DIM_3_TAU_54_M_100_DETREND_True_QUNF.pickle",
                   "results_copra_ensemble_DIM_3_TAU_14_M_100_DETREND_True_HOTI.pickle",
                   "results_copra_ensemble_DIM_3_TAU_36_M_100_DETREND_True_MAWMLUH.pickle",
                   "results_copra_ensemble_DIM_3_TAU_22_M_100_DETREND_True_TIANMEN.pickle",
                   "results_copra_ensemble_DIM_3_TAU_46_M_100_DETREND_True_DONGGE.pickle",
                   "results_copra_ensemble_DIM_3_TAU_48_M_100_DETREND_True_LIANHUA.pickle",
                   "results_copra_ensemble_DIM_3_TAU_15_M_100_DETREND_True_HESHANG.pickle",
                   "results_copra_ensemble_DIM_3_TAU_18_M_100_DETREND_True_JIUXIAN.pickle",
                   "results_copra_ensemble_DIM_3_TAU_33_M_100_DETREND_True_LIANG-LUAR1.pickle",
                   "results_copra_ensemble_DIM_3_TAU_33_M_100_DETREND_True_LIANG-LUAR2.pickle",
                   "results_copra_ensemble_DIM_3_TAU_33_M_100_DETREND_True_LIANG-LUAR3.pickle"]

n_time_series = len(DATA_FILE_NAMES)

#
#  Load results for plotting
#

#  Initialize dicts
storage = {}
FILENAME = {}
NAME = {}
N_ENSEMBLE = {}
symbols = {}
time = {}
values = {}
step_sequence = {}
results = {}
surrogate_results = {}

#  Load from files
for i in xrange(n_time_series):
    print "Loading results from", DATA_FILE_NAMES[i]

    file = open(DATA_FILE_NAMES[i], 'r')
    storage[i] = cPickle.load(file)
    file.close()

    #  Store parameters
    FILENAME[i] = storage[i]["FILENAME"]
    NAME[i] = storage[i]["NAME"]
    N_ENSEMBLE[i] = storage[i]["N_ENSEMBLE"]

    #  Store symbols
    symbols = storage[i]["symbols"] # !!! Assume that the same symbols are used in all files!

    #  Store raw input data
    time[i] = storage[i]["time"]
    values[i] = storage[i]["values"]

    #  Store axes
    step_sequence[i] = storage[i]["step_sequence"]

    #  Store results
    results[i] = storage[i]["results"]
    surrogate_results[i] = storage[i]["surrogate_results"]

    #  Derived properties
    print "Number of realizations for significance test:", N_ENSEMBLE[i]

#  Overwrite names by prettier ones
NAME = ["Dimarshim",
        "Qunf",
        "Hoti",
        "Mawmluh",
        "Tianmen",
        "Dongge",
        "Lianhua",
        "Heshang",
        "Jiuxian",
        "Liang-Luar 1",
        "Liang-Luar 2",
        "Liang-Luar 3"]

#
#  Prepare plotting of ensemble results
#

#  Init dicts
q_elements = {}
values_median = {}
values_left = {}
values_right = {}
median = {}
q_left = {}
q_right = {}
surr_q_left = {}
surr_q_right = {}

for i in xrange(n_time_series):
    #  Number of elements right and left of the quantile
    #  Divide by two, such that the total probability of false rejection of
    #  null hypothesis
    #  is equal to Q_QUANTILE (two sided test).
    q_elements[i] = int(Q_QUANTILE / 2 * N_ENSEMBLE[i])

    #  Get statistics for time series values
    values[i].sort(axis=0)

    values_median[i] = np.median(values[i], axis=0)
    values_left[i] = values[i][q_elements[i] - 1, :]
    values_right[i] = values[i][-q_elements[i], :]

    #  Get statistics for results
    median[i] = {}
    q_left[i] = {}
    q_right[i] = {}

    for measure in symbols.keys():
        results[i][measure].sort(axis=0)

        #  Get median
        median[i][measure] = np.median(results[i][measure], axis=0)

        #  Get q-quantiles
        q_left[i][measure] = results[i][measure][q_elements[i] - 1, :]
        q_right[i][measure] = results[i][measure][-q_elements[i], :]

    #
    #  Calculate q-quantiles for surrogates
    #

    surr_q_left[i] = {}
    surr_q_right[i] = {}

    for measure in symbols.keys():
        #  Get quantiles from surrogates from all time series realizations for
        #  now
        surrogate_results[i][measure] = surrogate_results[i][measure].flatten()

        #  Sort
        surrogate_results[i][measure].sort()

        #  Get q-quantiles
        surr_q_left[i][measure] = surrogate_results[i][measure][q_elements[i] - 1]
        surr_q_right[i][measure] = surrogate_results[i][measure][-q_elements[i]]

#
#  Rescale time axis
#

min_age_local = np.zeros(n_time_series)
max_age_local = np.zeros(n_time_series)

for i in xrange(n_time_series):
    time[i] /= 1000.

    #  Get local min and max age
    min_age_local[i] = time[i].min()
    max_age_local[i] = time[i].max()

#  Get overall minimum and maximum age
min_age = min_age_local.min()
max_age = max_age_local.max()

#
#  Plot overview figure for each time series
#

#  Set plotting parameters (for Clim. Past paper)
params = { 'figure.figsize': (7.,7.),
          'axes.labelsize': 12,
          'text.fontsize': 12,
          'xtick.labelsize': 12,
          'ytick.labelsize': 12,
          'legend.fontsize': 10,
          'title.fontsize': 12,
          'text.usetex': False,
          'font': 'Helvetica',
          'mathtext.bf': 'helvetica:bold',
           'xtick.major.pad': 6,
           'ytick.major.pad': 6,
           'xtick.major.size': 5,
           'ytick.major.size': 5,
           'tick.labelsize': 'small'
           }

#pylab.rcParams.update(params)

for i in xrange(n_time_series):
    #  Create new figure
    fig = pylab.figure()

    #  Plot original data including dating uncertainties
    ax = fig.add_subplot(len(symbols) + 1,1,1)

    pylab.plot(time[i], values_median[i], "k")
    pylab.fill_between(time[i], values_left[i], values_right[i], color="0.75")

    #  Plot additional information
    #  Plot RCC episodes
    for k in xrange(RCC_EPISODES.shape[0]):
        pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], hatch="/",
                      fill=False, edgecolor="0.5")
    #  Plot Bond events
    pylab.scatter(BOND_EVENTS,
                  0.9 * np.ones(len(BOND_EVENTS)) * values_right[i].max(),
                  c='k', marker='*', s=80)

    pylab.ylabel(DATA_LABEL)
    pylab.xlim(min_age, max_age)

    #  Add figure label
    ax.annotate(figure_labels[0], xy=(-0.14, 1.1),
                xycoords='axes fraction',
                horizontalalignment='left', verticalalignment='top',
                fontname='Helvetica', fontsize=14, fontweight='bold')

    #  Render time labels invisible
    ax.axes.get_xaxis().set_visible(False)

    #  Plot windowed measures
    j = 1
    for measure in symbols.keys():
        j += 1
        ax = fig.add_subplot(len(symbols) + 1, 1, j)

        pylab.plot(time[i][step_sequence[i]], median[i][measure], "k")
        pylab.fill_between(time[i][step_sequence[i]], q_left[i][measure],
                           q_right[i][measure], color="0.75")

        pylab.ylabel(symbols[measure])

        if SHOW_CONFIDENCE_BAND:
            #pylab.axhline(y=surrogate_mean[measure][i], linewidth=1, color="r")
            #pylab.fill_between(time[step_sequence], surr_q_left[measure], surr_q_right[measure], color="0.75")
            pylab.axhline(y=surr_q_left[i][measure], linewidth=1, color="r")
            pylab.axhline(y=surr_q_right[i][measure], linewidth=1, color="r")

        #  Plot additional information
        #  Plot RCC episodes
        for k in xrange(RCC_EPISODES.shape[0]):
            pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], hatch="/",
                          fill=False, edgecolor="0.5")
        #  Plot Bond events
        pylab.scatter(BOND_EVENTS,
                 0.9 * np.ones(len(BOND_EVENTS)) * q_right[i][measure].max(),
                 c='k', marker='*', s=80)

        pylab.xlim(min_age, max_age)

        #  Add figure label
        ax.annotate(figure_labels[j - 1], xy=(-0.14, 1.1),
                    xycoords='axes fraction',
                    horizontalalignment='left', verticalalignment='top',
                    fontname='Helvetica', fontsize=14, fontweight='bold')

        #  Render time labels invisible
        if j < 3:
            ax.axes.get_xaxis().set_visible(False)

    pylab.xlabel(AGE_LABEL)

    #  Save figure
    pylab.savefig(NAME[i] + "_copra_overview.pdf")

#
#  Plot comparison figures for single time series / windowed measures
#

#  Set plotting parameters
params = { 'figure.figsize': (8.268,11.693),
          'axes.labelsize': 12,
          'text.fontsize': 12,
          'xtick.labelsize': 12,
          'ytick.labelsize': 12,
          'legend.fontsize': 10,
          'title.fontsize': 12,
          'text.usetex': False,
          'font': 'Helvetica',
          'mathtext.bf': 'helvetica:bold',
           'xtick.major.pad': 6,
           'ytick.major.pad': 6,
           'xtick.major.size': 5,
           'ytick.major.size': 5,
           'tick.labelsize': 'small'
           }

#pylab.rcParams.update(params)

#
#  Comparison of time series (COPRA)
#

fig = pylab.figure()

for i in xrange(n_time_series):
    ax = fig.add_subplot(n_time_series, 1, i+1)

    pylab.plot(time[i], values_median[i], "#DB232E", linewidth=2)
    pylab.fill_between(time[i], values_left[i], values_right[i],
                       color="#F1B08E")

    pylab.xlim(min_age, max_age)

    #  Plot additional information
    #  Plot RCC episodes
    #for k in xrange(RCC_EPISODES.shape[0]):
    #    pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], fill=True, edgecolor="0.8", color="0.8", zorder=0)

    #  Plot Bond events
    #pylab.scatter(BOND_EVENTS, 2.0 * np.ones(len(BOND_EVENTS)), c='k', marker='o', s=80)

    #  Add record name
    ax.text(0.99, 0.9, NAME[i], horizontalalignment='right',
            verticalalignment='top',
            transform = ax.transAxes)

    #  Add figure label
    ax.annotate(figure_labels[i], xy=(-0.14, 1.1),
                xycoords='axes fraction',
                horizontalalignment='left', verticalalignment='top',
                fontname='Helvetica', fontsize=14, fontweight='bold')

    #  Remove spines
    for loc, spine in ax.spines.items():
        #  For first subplot
        if i == 0:
            if loc in ['left','top']:
                spine.set_position(('outward',10)) # outward by 10 points
            elif loc in ['right','bottom']:
                spine.set_color('none') # don't draw spine
            else:
                raise ValueError('unknown spine location: %s'%loc)

            # turn off ticks where there is no spine
            ax.xaxis.set_ticks_position('top')
            ax.yaxis.set_ticks_position('left')
        #  For every even numbered subplot
        elif i % 2 == 0:
            if loc in ['left','bottom']:
                spine.set_position(('outward',10)) # outward by 10 points
            elif loc in ['right','top']:
                spine.set_color('none') # don't draw spine
            else:
                raise ValueError('unknown spine location: %s'%loc)

            # turn off ticks where there is no spine
            #ax.xaxis.set_ticks_position('bottom')
            ax.yaxis.set_ticks_position('left')
        #  For every uneven numbered subplot
        elif i % 2 == 1:
            if loc in ['right','bottom']:
                spine.set_position(('outward',10)) # outward by 10 points
            elif loc in ['left','top']:
                spine.set_color('none') # don't draw spine
            else:
                raise ValueError('unknown spine location: %s'%loc)

            # turn off ticks where there is no spine
            #ax.xaxis.set_ticks_position('bottom')
            ax.yaxis.set_ticks_position('right')

    #  Add time ticks and label to uppermost subplot
    if i == 0:
        ax.xaxis.set_label_position("top")
        ax.tick_params(labelbottom='off', labeltop='on')
        pylab.xlabel(AGE_LABEL)

    #  Move y-axis tick labels and axis label to right side for every second
    #  subplot
    if i % 2 == 1:
        ax.yaxis.tick_right()
        ax.yaxis.set_label_position("right")

    #  Render time tick labels invisible
    if 0 < i < n_time_series - 1:
        ax.set_xticklabels([])

    #  Enable minor ticks
    ax.minorticks_on()

    #  Set number of ticks and tick labels
    ax.locator_params(axis="x", tight=True, nbins=9)
    ax.locator_params(axis="y", tight=True, nbins=4)

    pylab.ylabel(DATA_LABEL)

pylab.xlabel(AGE_LABEL)

#  Save figure
pylab.savefig("time_series_copra.pdf")

#
#  Comparison of windowed measures
#

#  Set colors for results subplots
colors_line = ["#3A89CB", "#9CD39A"]
colors_fill = ["#DDEAED", "#DBEDDA"]

#  Initialize counter
j = 0

for measure in symbols.keys():
    fig = pylab.figure()

    for i in xrange(n_time_series):
        ax = fig.add_subplot(n_time_series, 1, i + 1)

        pylab.plot(time[i][step_sequence[i]], median[i][measure],
                   colors_line[j], linewidth=2)
        pylab.plot(time[i][step_sequence[i]], q_left[i][measure],
                   colors_line[j], linewidth=1)
        pylab.plot(time[i][step_sequence[i]], q_right[i][measure],
                   colors_line[j], linewidth=1)

        #  Fill significant episodes for COPRA results
        pylab.fill_between(time[i][step_sequence[i]], surr_q_right[i][measure],
                        q_right[i][measure],
                        where=q_right[i][measure] > surr_q_right[i][measure],
                        color=colors_line[j])

        #  Plot COPRA confidence band
        if SHOW_CONFIDENCE_BAND:
            pylab.axhspan(surr_q_left[i][measure], surr_q_right[i][measure],
                          fill=True, color=colors_fill[j], zorder=0)

        pylab.xlim(min_age, max_age)
        if j == 1:
            pylab.ylim(0.4, 0.8)

        #  Plot additional information
        #  Plot RCC episodes
        #for k in xrange(RCC_EPISODES.shape[0]):
        #    pylab.axvspan(RCC_EPISODES[k,0], RCC_EPISODES[k,1], fill=True, edgecolor="0.8", color="0.8", zorder=0)

        #  Plot Bond events
        #pylab.scatter(BOND_EVENTS, 2.0 * np.ones(len(BOND_EVENTS)), c='k', marker='o', s=80)

        #  Add figure label
        ax.annotate(figure_labels[i], xy=(-0.14, 1.1),
                    xycoords='axes fraction',
                    horizontalalignment='left', verticalalignment='top',
                    fontname='Helvetica', fontsize=14, fontweight='bold')

        #  Add record name
        ax.text(0.99, 0.9, NAME[i], horizontalalignment='right',
                verticalalignment='top', transform = ax.transAxes)

        #  Remove spines
        for loc, spine in ax.spines.items():
            #  For first subplot
            if i == 0:
                if loc in ['left','top']:
                    spine.set_position(('outward',10)) # outward by 10 points
                elif loc in ['right','bottom']:
                    spine.set_color('none') # don't draw spine
                else:
                    raise ValueError('unknown spine location: %s'%loc)

                # turn off ticks where there is no spine
                ax.xaxis.set_ticks_position('top')
                ax.yaxis.set_ticks_position('left')
            #  For every even numbered subplot
            elif i % 2 == 0:
                if loc in ['left','bottom']:
                    spine.set_position(('outward',10)) # outward by 10 points
                elif loc in ['right','top']:
                    spine.set_color('none') # don't draw spine
                else:
                    raise ValueError('unknown spine location: %s'%loc)

                # turn off ticks where there is no spine
                #ax.xaxis.set_ticks_position('bottom')
                ax.yaxis.set_ticks_position('left')
            #  For every uneven numbered subplot
            elif i % 2 == 1:
                if loc in ['right','bottom']:
                    spine.set_position(('outward',10)) # outward by 10 points
                elif loc in ['left','top']:
                    spine.set_color('none') # don't draw spine
                else:
                    raise ValueError('unknown spine location: %s'%loc)

                # turn off ticks where there is no spine
                #ax.xaxis.set_ticks_position('bottom')
                ax.yaxis.set_ticks_position('right')

        #  Add time ticks and label to uppermost subplot
        if i == 0:
            ax.xaxis.set_label_position("top")
            ax.tick_params(labelbottom='off',labeltop='on')
            pylab.xlabel(AGE_LABEL)

        #  Move y-axis tick labels and axis label to right side for every second subplot
        if i % 2 == 1:
            ax.yaxis.tick_right()
            ax.yaxis.set_label_position("right")

        #  Render time tick labels invisible
        if 0 < i < n_time_series - 1:
            ax.set_xticklabels([])

        #  Enable minor ticks
        ax.minorticks_on()

        #  Set number of ticks and tick labels
        ax.locator_params(axis="x", tight=True, nbins=9)
        ax.locator_params(axis="y", tight=True, nbins=4)

        pylab.ylabel(symbols[measure])

    pylab.xlabel(AGE_LABEL)

    #  Save figure
    pylab.savefig("RN_" + measure + "_copra.pdf")

    #  Increase counter
    j += 1
