import numpy as np
import math
import scipy.special as sc
import scipy.integrate as integrate


def my_range(start, end, step):
    while start <= end:
        yield start
        start += step     
       

G=6.67*10**-11 #newton constant in m^3 kg^-1 s^-2
CMtoM=10**-2
MtoCM=10**2
GEVtoKG=1.783*10**-27

elements = None  # Initialize elements as None

def set_elements(composition):
    global elements
    elements = np.array(composition)

def multicapture(R,M,dens,v,mn,sig,mdm): # has Eq. 2.15 of paper built in, nuclei-level cross section
    elementcap=0
    if taumulti(R,M,mn,mdm,sig) < 3/2:
        for i in my_range(0, elements.size/2-1, 1):
            elementcap=elementcap+weakcapture(R,M,dens,v,elements[i,1],i,mn,sig,mdm)
        return elementcap
    if taumulti(R,M,mn,mdm,sig) >= 3/2:
        return strongcapture(R,M,dens,v,mn,smaveragedmulti(R,M,mn,mdm,sig),sig,mdm)
    
def multicapture_noA(R,M,dens,v,mn,sig,mdm): # no nucleon to nuclei enhancement
    elementcap=0
    if taumulti_noA(R,M,mn,mdm,sig) < 3/2:
        for i in my_range(0, elements.size/2-1, 1):
            elementcap=elementcap+weakcapture_noA(R,M,dens,v,elements[i,1],i,mn,sig,mdm)
        return elementcap
    if taumulti_noA(R,M,mn,mdm,sig) >= 3/2:
        return strongcapture_noA(R,M,dens,v,mn,smaveragedmulti_noA(R,M,mn,mdm,sig),sig,mdm)
        
        
        
        
        
 # weak interaction regime treatment
 
def weakcapture(R,M,dens,v,mA,i,mn,sig,mdm):
    mu=mdm/mA    
    if (mu/(1+mu**2))*(vesc(R,M)/v)**2 <= 10**-5: # for the case of large mass mismatch and weak interaction regime
        return totalcapsinglesmall(R,M,dens,v,mA,mn,i,sig,mdm)
    if (mu/(1+mu**2))*(vesc(R,M)/v)**2 > 10**-5:
        return totalcapsingle(R,M,dens,v,mA,i,mn,sig,mdm)
    
def weakcapture_noA(R,M,dens,v,mA,i,mn,sig,mdm):
    mu=mdm/mA    
    if (mu/(1+mu**2))*(vesc(R,M)/v)**2 <= 10**-5: # for the case of large mass mismatch and weak interaction regime
        return totalcapsinglesmall_noA(R,M,dens,v,mA,mn,i,sig,mdm)
    if (mu/(1+mu**2))*(vesc(R,M)/v)**2 > 10**-5:
        return totalcapsingle_noA(R,M,dens,v,mA,mn,i,sig,mdm) 
        
def nsm(M,mA):
    return (M/(GEVtoKG))/mA

def mured(msm,mdm):
    return msm*mdm/(msm+mdm)

def nucleisig(mA,mn,mdm,sig): # in cm^2
    return mA**2*(mured(mA,mdm)/mured(mn,mdm))**2*sig

def sigmatr(M,R,mA): # in cm^2, assumes all of object is made of mass element A in this function
    return np.pi*(R*MtoCM)**2/nsm(M,mA)

# single element tau under A^2/A^4 enhancement of cross section
def singletau(R,M,mA,i,mn,mdm,sig):
    return (3/2)*nucleisig(mA,mn,mdm,sig)*elements[i,0]/(sigmatr(M,R,elements[i,1]))

# single element tau without A^2/A^4 enhancement of cross section
def singletau_noA(R,M,mA,i,mn,mdm,sig):
    return (3/2)*sig*elements[i,0]/(sigmatr(M,R,elements[i,1])) 

    
    
def vesc(R,M):
    return (np.sqrt(2*G*M/(R))) # m/s units

def beta(mdm,mA):
    if mdm==mA:
    	return 0.999 # offset to avoid beta=1
    else:
    	return (4*mdm*mA)/(mdm+mA)**2 # no units

def vN(R,M,mdm,mA,N):
    return vesc(R,M)*(1-beta(mdm,mA)/2)**(-N/2) # m/s units

def kap(v,R,M,mdm,mA,N): # m^2/s^2 units
    return ((2*v**2+3*vesc(R,M)**2)-(2*v**2+3*vN(R,M,mdm,mA,N)**2)*np.exp(-3*(vN(R,M,mdm,mA,N)**2-vesc(R,M)**2)/(2*v**2)))

def numberdens(dens,mdm):
    return (10**6*dens/mdm) # m^-3


# framework

def xibar(v): # in m^2/s^2
    return v**2/3

def xiesc(R,M): # in m^2/s^2
    return vesc(R,M)**2/2

def ximax(R,M,mdm,mA,N):  # in m^2/s^2
    return xiesc(R,M)*((1-beta(mdm,mA))**-1*((beta(mdm,mA)**-1*np.log(1/(1-beta(mdm,mA))))**(N-1))-1)

def ximin(R,M,mdm,mA,N):  # in m^2/s^2
    return xiesc(R,M)*(((beta(mdm,mA)**-1*np.log(1/(1-beta(mdm,mA))))**(N-1))-1)

def pdef(v,R,M,mdm,mA,N): # unitless
    return np.exp(-ximin(R,M,mdm,mA,N)/xibar(v))-np.exp(-ximax(R,M,mdm,mA,N)/xibar(v))

def qdef(v,R,M,mdm,mA,N): # in m^2/s^2
    return (xibar(v)+xiesc(R,M)+ximin(R,M,mdm,mA,N))*np.exp(-ximin(R,M,mdm,mA,N)/xibar(v))-(xibar(v)+xiesc(R,M)+ximax(R,M,mdm,mA,N))*np.exp(-ximax(R,M,mdm,mA,N)/xibar(v))

def kappanew(v,R,M,mdm,mA,N): # in m^2/s^2
    return (xiesc(R,M)/beta(mdm,mA)**N)*(np.log(1/(1-beta(mdm,mA))))**(N-1)*pdef(v,R,M,mdm,mA,N)-(beta(mdm,mA)**-1-1)*qdef(v,R,M,mdm,mA,N)

def capnnewnoP(R,M,N,dens,v,mA,sig,mdm): # units s^-1
    return ((np.pi*R**2*np.sqrt(8/np.pi)*numberdens(dens,mdm)/(np.sqrt(xibar(v))))*kappanew(v,R,M,mdm,mA,N))

def integrandpnsingle(M,mdm,mn,mA,i,N,R,sig,y):
    if singletau(R,M,mA,i,mn,mdm,sig) < 10**-10:
        return 2*(y*(y*singletau(R,M,mA,i,mn,mdm,sig))**N)/np.math.factorial(N) # set the exponential to 1 if tau is very small; it will be approx exp^0=1
    else:
        return 2*((y*np.exp(-y*singletau(R,M,mA,i,mn,mdm,sig))*(y*singletau(R,M,mA,i,mn,mdm,sig))**N)/np.math.factorial(N))

def pnsingle(M,mdm,mn,mA,i,N,R,sig):
    return integrate.quad(lambda y: integrandpnsingle(M,mdm,mn,mA,i,N,R,sig,y), 0, 1,limit=200)[0]


def totalcapsingle(R,M,dens,v,mA,i,mn,sig,mdm):  # units s^-1
    multicapsum=0
    multicapsum2=0
    Nmaxi=10
    for nscatter in my_range(1, Nmaxi, 1): # (start, stop, step size)
        multicapsum=0    
        for j in my_range(1, nscatter, 1): # (start, stop, step size)
            multicapsum=multicapsum+capnnewnoP(R,M,j,dens,v,mA,sig,mdm)
        multicapsum2=multicapsum2+multicapsum*pnsingle(M,mdm,mn,mA,i,nscatter,R,sig)
    return multicapsum2

def K(mdm,mA,R,M,v):
    mu=mdm/mA
    vsqratio=vesc(R,M)**2/v**2
    return (9*vesc(R,M)**4*mu)/(((2+3*vsqratio)*v**4)*(1+mu)**2)

def cnsmall(M,mdm,mA,mn,i,N,dens,v,R,sig):
    return cap_geo(dens,v,M,R,mdm)*pnsingle(M,mdm,mn,mA,i,N,R,sig)*N*K(mdm,mA,R,M,v)

def totalcapsinglesmall(R,M,dens,v,mA,mn,i,sig,mdm):  # units s^-1
    multicapsum=0
    for nscatter in my_range(1, 10, 1): # (start, stop, step size) up to nsum 10
        multicapsum=multicapsum+cnsmall(M,mdm,mA,mn,i,nscatter,dens,v,R,sig)
    return multicapsum

# weak regime no A enhancement

def integrandpnsingle_noA(M,mdm,mn,mA,i,N,R,sig,y):
    if singletau(R,M,mA,i,mn,mdm,sig) < 10**-10: 
        return 2*(y*(y*singletau_noA(R,M,mA,i,mn,mdm,sig))**N)/np.math.factorial(N) # set the exponential to 1 if tau is very small; it will be approx exp^0=1
    else:
        return 2*((y*np.exp(-y*singletau_noA(R,M,mA,i,mn,mdm,sig))*(y*singletau_noA(R,M,mA,i,mn,mdm,sig))**N)/np.math.factorial(N))

    
def pnsingle_noA(M,mdm,mn,mA,i,N,R,sig):
    return integrate.quad(lambda y: integrandpnsingle_noA(M,mdm,mn,mA,i,N,R,sig,y), 0, 1,limit=200)[0]

def totalcapsingle_noA(R,M,dens,v,mA,mn,i,sig,mdm):  # units s^-1
    multicapsum=0
    multicapsum2=0
    Nmaxi=10
    for nscatter in my_range(1, Nmaxi, 1): # (start, stop, step size)
        multicapsum=0    
        for j in my_range(1, nscatter, 1): # (start, stop, step size)
            multicapsum=multicapsum+capnnewnoP(R,M,j,dens,v,mA,sig,mdm)
        multicapsum2=multicapsum2+multicapsum*pnsingle_noA(M,mdm,mn,mA,i,nscatter,R,sig)
    return multicapsum2

def cnsmall_noA(M,mdm,mA,mn,i,N,dens,v,R,sig):
    return cap_geo(dens,v,M,R,mdm)*pnsingle_noA(M,mdm,mn,mA,i,N,R,sig)*N*K(mdm,mA,R,M,v)

def totalcapsinglesmall_noA(R,M,dens,v,mA,mn,i,sig,mdm):  # units s^-1
    multicapsum=0
    for nscatter in my_range(1, 10, 1): # (start, stop, step size) up to nsum 10
        multicapsum=multicapsum+cnsmall_noA(M,mdm,mA,mn,i,nscatter,dens,v,R,sig)
    return multicapsum   
    
            
# strong interaction capture regime            


def strongcapture(R,M,dens,v,mn,mA,sig,mdm):
    escv=np.sqrt(2*G*M/(R))/1000
    if taumulti(R,M,mn,mdm,sig) <= 100:
        return min(totalcap(R,M,dens,v,mn,mA,sig,mdm),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if taumulti(R,M,mn,mdm,sig) > 100 and taumulti(R,M,mn,mdm,sig) < 10**6:
        return min(intermedcapregime(R,M,dens,v,mn,mA,sig,mdm),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if taumulti(R,M,mn,mdm,sig) >= 10**6:
        return largecaptureregime(R,M,dens,v,mdm,mn,mA,sig)
    
    
def strongcapture_noA(R,M,dens,v,mn,mA,sig,mdm):
    escv=np.sqrt(2*G*M/(R))/1000
    if taumulti_noA(R,M,mn,mdm,sig) <= 100:
        return min(totalcap_noA(R,M,dens,v,mn,mA,sig,mdm),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if taumulti_noA(R,M,mn,mdm,sig) > 100 and taumulti_noA(R,M,mn,mdm,sig) < 10**6:
        return min(intermedcapregime_noA(R,M,dens,v,mn,mA,sig,mdm),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if taumulti_noA(R,M,mn,mdm,sig) >= 10**6:
        return largecaptureregime_noA(R,M,dens,v,mdm,mn,mA,sig)    

    
    
def Nreq(R,M,v,mdm,mA): 
    mu=mdm/mA
    if mu <= 10**12:
        return np.log(vesc(R,M)**2/(v**2+vesc(R,M)**2))/np.log(fmu(mdm,mA))
    if mu > 10**12:
        return np.log(vesc(R,M)**2/(v**2+vesc(R,M)**2))/(2*mu/(mu**2+mu+1))    

    
def intermedcapregime(R,M,dens,v,mn,mA,sig,mdm):
    mu=mdm/mA   
    if Nsum(sig,M,R,mdm,mn) > 10*Nreq(R,M,v,mdm,mA) and mu < 10**4:
        return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: pNlarge(R,M,y,mdm,mn,sig), 1, Nsum(sig,M,R,mdm,mn))[0] 
    else:
        return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: (capnopn(R,M,y,dens,v,mA,mdm)/cap_geo(dens,v,M,R,mdm))*pNlarge(R,M,y,mdm,mn,sig), 1, Nsum(sig,M,R,mdm,mn),limit=500)[0] # normalization with cap_geo helps stability of integral

def intermedcapregime_noA(R,M,dens,v,mn,mA,sig,mdm):
    mu=mdm/mA   
    if Nsum_noA(sig,M,R,mdm,mn) > 10*Nreq(R,M,v,mdm,mA) and mu < 10**4:
        return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: pNlarge_noA(R,M,y,mdm,mn,sig), 1, Nsum_noA(sig,M,R,mdm,mn))[0] 
    else:
        return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: (capnopn(R,M,y,dens,v,mA,mdm)/cap_geo(dens,v,M,R,mdm))*pNlarge_noA(R,M,y,mdm,mn,sig), 1, Nsum_noA(sig,M,R,mdm,mn))[0] # normalization with cap_geo helps stability of integral



    
def largecaptureregime(R,M,dens,v,mdm,mn,mA,sig):
    mu=mdm/mA
    escv=np.sqrt(2*G*M/(R))/1000
    if mu <= 1:
        return min(CNlightfull(R,M,dens,v,mdm,mn,mA,sig),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if mu > 1:
        return CNheavyfull(R,M,dens,v,mdm,mn,mA,sig)
    
def largecaptureregime_noA(R,M,dens,v,mdm,mn,mA,sig):
    mu=mdm/mA
    escv=np.sqrt(2*G*M/(R))/1000
    if mu <= 1:
        return min(CNlightfull_noA(R,M,dens,v,mdm,mn,mA,sig),fcapfull(v,mA,mdm,escv)*cap_geo(dens,v,M,R,mdm))
    if mu > 1:
        return CNheavyfull_noA(R,M,dens,v,mdm,mn,mA,sig)
    


def pNlarge(R,M,N,mdm,mn,sig):
    return (2*(N+1)/taumulti(R,M,mn,mdm,sig)**2)*np.heaviside(taumulti(R,M,mn,mdm,sig)-N,1)

def pNlarge_noA(R,M,N,mdm,mn,sig):
    return (2*(N+1)/taumulti_noA(R,M,mn,mdm,sig)**2)*np.heaviside(taumulti_noA(R,M,mn,mdm,sig)-N,1)


# heavy cases

def CNheavylarge(R,M,N,v,mdm,mA,sig):
    return 3*N*vesc(R,M)**4*CNheavybracket(R,M,N,v,mdm,mA,sig)/CNheavydenom(R,M,N,v,mdm,mA)

def CNheavybracket(R,M,N,v,mdm,mA,sig):
    vsqratio=vesc(R,M)**2/v**2
    mu=mdm/mA
    return 8*mu*(3*(mu-1)*mu+1)+N**3*(64-3*(9*vsqratio**2*(vsqratio-6)+76*vsqratio))+4*(mu-3)*N**2*(9*vsqratio**2-30*vsqratio+16)-4*N*(3*(mu-2)*mu+5)*(3*vsqratio-4)

def CNheavydenom(R,M,N,v,mdm,mA):
    mu=mdm/mA
    return 8*mu**4*(3*vesc(R,M)**2*v**2+2*v**4)

def CNheavy(R,M,N,v,mdm,mn,mA,sig):
    return pNlarge(R,M,N,mdm,mn,sig)*min(1,np.abs(CNheavylarge(R,M,N,v,mdm,mA,sig)))

def CNheavyfull(R,M,dens,v,mdm,mn,mA,sig):
    return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: CNheavy(R,M,y,v,mdm,mn,mA,sig), 1, Nsum(sig,M,R,mdm,mn),limit=500)[0]    

def CNheavy_noA(R,M,N,v,mdm,mn,mA,sig):
    return pNlarge_noA(R,M,N,mdm,mn,sig)*min(1,np.abs(CNheavylarge(R,M,N,v,mdm,mA,sig)))

def CNheavyfull_noA(R,M,dens,v,mdm,mn,mA,sig):
    return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: CNheavy_noA(R,M,y,v,mdm,mn,mA,sig), 1, Nsum_noA(sig,M,R,mdm,mn),limit=200)[0]    



# light cases

def CNlightlarge(R,M,N,v,mdm,mA,sig):
    mu=mdm/mA    
    return 3*mu*N*vesc(R,M)**4*CNlightbracket(R,M,N,v,mdm,mA,sig)/CNlightdenom(R,M,v,mdm,mA)

def CNlightbracket(R,M,N,v,mdm,mA,sig):
    vsqratio=vesc(R,M)**2/v**2
    mu=mdm/mA
    return 24+mu**3*N**3*(64-3*(9*vsqratio**2*(vsqratio-6)+76*vsqratio))-4*mu**2*(3*mu-1)*N**2*(9*vsqratio**2-30*vsqratio+16)-4*mu*(mu*(5*mu-6)+3)*N*(3*vsqratio-4)+8*mu*(mu-3)

def CNlightdenom(R,M,v,mdm,mA):
    mu=mdm/mA
    return 8*(3*vesc(R,M)**2*v**2+2*v**4)

def CNlight(R,M,N,v,mdm,mn,mA,sig):
    return pNlarge(R,M,N,mdm,mn,sig)*min(1,np.abs(CNlightlarge(R,M,N,v,mdm,mA,sig)))

def CNlightfull(R,M,dens,v,mdm,mn,mA,sig):
    return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: CNlight(R,M,y,v,mdm,mn,mA,sig), 1, Nsum(sig,M,R,mdm,mn))[0]  

def CNlight_noA(R,M,N,v,mdm,mn,mA,sig):
    return pNlarge_noA(R,M,N,mdm,mn,sig)*min(1,np.abs(CNlightlarge(R,M,N,v,mdm,mA,sig)))

def CNlightfull_noA(R,M,dens,v,mdm,mn,mA,sig):
    return cap_geo(dens,v,M,R,mdm)*integrate.quad(lambda y: CNlight_noA(R,M,y,v,mdm,mn,mA,sig), 1, Nsum_noA(sig,M,R,mdm,mn))[0]        
    
    

# geometric

def cap_geo(dens,v,M,R,mdm):
    return np.pi*R**2*(dens/mdm)*10**6*v*(1+(3/2)*(vesc(R,M)**2/v**2))




# tau and average interaction mass under A^2/A^4 enhancement of cross section

def taumulti(R,M,mn,mdm,sig): # unitless, includes weighting for mass fraction of each element A
    newtau=0
    for i in my_range(0, elements.size/2-1, 1):
        newtau=newtau+nucleisig(elements[i,1],mn,mdm,sig)*elements[i,0]/(sigmatr(M,R,elements[i,1]))
    return (3/2)*newtau

def taumultifree(R,M,i,mn,mdm,sig):
    return (3/2)*nucleisig(elements[i,1],mn,mdm,sig)*elements[i,0]/(sigmatr(M,R,elements[i,1]))

def smaveragedmulti(R,M,mn,mdm,sig):
    smavg=0
    for i in my_range(0, elements.size/2-1, 1):
        smavg=smavg+taumultifree(R,M,i,mn,mdm,sig)*elements[i,1]
    return smavg/taumulti(R,M,mn,mdm,sig)


# tau and average interaction mass no enhancement of cross section

def taumulti_noA_targetcomp(R,M,mn,mdm,sig): # unitless, includes weighting for mass fraction of each element A
    newtau=0
    for i in my_range(0, elements.size/2-1, 1):
        newtau=newtau+sig*elements[i,0]/(sigmatr(M,R,elements[i,1]))
    return (3/2)*newtau # optical depth ignoring number of targets available
    
def taumulti_noA(R,M,mn,mdm,sig): # unitless, includes weighting for mass fraction of each element A
    return min(taumulti_noA_targetcomp(R,M,mn,mdm,sig),totaltargets(R,M,mn,mdm,sig)) # min of optical depth or approx number of targets available

def totaltargets(R,M,mn,mdm,sig):
    return nsm(M,smaveragedmulti_noA(R,M,mn,mdm,sig))**(1/3) 

def taumultifree_noA(R,M,i,mn,mdm,sig):
    return (3/2)*sig*elements[i,0]/(sigmatr(M,R,elements[i,1]))

def smaveragedmulti_noA(R,M,mn,mdm,sig):
    smavg=0
    for i in my_range(0, elements.size/2-1, 1):
        smavg=smavg+taumultifree_noA(R,M,i,mn,mdm,sig)*elements[i,1]
    return smavg/taumulti_noA_targetcomp(R,M,mn,mdm,sig)


# framework

def pn(M,mn,mdm,N,R,sig): # no units
    return 2*(N+1)*(1-sc.gammaincc(N+2,taumulti(R,M,mn,mdm,sig)))/taumulti(R,M,mn,mdm,sig)**2

def pn_noA(M,mn,mdm,N,R,sig): # no units
    return 2*(N+1)*(1-sc.gammaincc(N+2,taumulti_noA(R,M,mn,mdm,sig)))/taumulti_noA(R,M,mn,mdm,sig)**2

def alpha(mdm,mA):
    mu=mdm/mA
    return 1-(2*mu/(1+mu)**2)

def capn(R,M,N,dens,v,mn,mA,sig,mdm): # units s^-1
    vsqratio=vesc(R,M)**2/v**2
    return cap_geo(dens,v,M,R,mdm)*pn(M,mn,mdm,N,R,sig)*(1-(np.exp(-(3/2)*vsqratio*(alpha(mdm,mA)**(-N)-1))*(1+(3/2)*vsqratio*alpha(mdm,mA)**-N))/(1+(3/2)*vsqratio))

def capn_noA(R,M,N,dens,v,mn,mA,sig,mdm): # units s^-1
    vsqratio=vesc(R,M)**2/v**2
    return cap_geo(dens,v,M,R,mdm)*pn_noA(M,mn,mdm,N,R,sig)*(1-(np.exp(-(3/2)*vsqratio*(alpha(mdm,mA)**(-N)-1))*(1+(3/2)*vsqratio*alpha(mdm,mA)**-N))/(1+(3/2)*vsqratio))


def capnopn(R,M,N,dens,v,mA,mdm): # units s^-1
    vsqratio=vesc(R,M)**2/v**2
    return cap_geo(dens,v,M,R,mdm)*(1-(np.exp(-(3/2)*vsqratio*(alpha(mdm,mA)**(-N)-1))*(1+(3/2)*vsqratio*alpha(mdm,mA)**-N))/(1+(3/2)*vsqratio))

def Nsum(sig,M,R,mdm,mn):
    return max(10,np.exp(1)*taumulti(R,M,mn,mdm,sig))

def Nsum_noA(sig,M,R,mdm,mn):
    return max(10,np.exp(1)*taumulti_noA(R,M,mn,mdm,sig))

def totalcap(R,M,dens,v,mn,mA,sig,mdm):  # units s^-1
    multicapsum=0
    for nscatter in my_range(1, Nsum(sig,M,R,mdm,mn), 1): # (start, stop, step size)
        multicapsum=multicapsum+capn(R,M,nscatter,dens,v,mn,mA,sig,mdm)
    return multicapsum

def totalcap_noA(R,M,dens,v,mn,mA,sig,mdm):  # units s^-1
    multicapsum=0
    for nscatter in my_range(1, Nsum_noA(sig,M,R,mdm,mn), 1): # (start, stop, step size)
        multicapsum=multicapsum+capn_noA(R,M,nscatter,dens,v,mn,mA,sig,mdm)
    return multicapsum
    
    
# reflection factors

def NT(v,escv):
    return 12+1.8*np.log(np.sqrt(1+v**2/(escv*1000)**2))

def velratio(v,escv):
    return (escv*1000)**2/((escv*1000)**2+v)

def muT(v,escv):
    return (np.sqrt(2*velratio(v,escv)**(1/NT(v,escv))-1)-velratio(v,escv)**(1/NT(v,escv)))/(velratio(v,escv)**(1/NT(v,escv))-1)

def muM(v,escv):
    return 1.56*(1-1/(1+0.53*np.log(np.sqrt(1+v**2/(escv*1000)**2))))

def fM(v,escv):
    return 0.22*(1+3.55/(1+0.23*np.log(np.sqrt(1+v**2/(escv*1000)**2))))

def fmu(mdm,mA):
    return 1-(2*(mdm/mA))/(1+(mdm/mA))**2

def fcaplightesc(v,mA,mdm,escv): # unitless, escv in km/s
    return np.sqrt((4/(np.pi))*np.log(fmu(mdm,mA))/(np.log(((escv*1000)**2/(v**2+(escv*1000)**2)))))

def fcapmidesc(v,mA,mdm,escv): # unitless, escv in km/s
    return (muT(v,escv)-fcaplightesc(v,mA,mdm,escv))/(muT(v,escv)-1)+(-(mdm/mA)+(mdm/mA)*fcaplightesc(v,mA,mdm,escv))/(muT(v,escv)-1)

def fcapheavyesc(v,mA,mdm,escv): # unitless, escv in km/s
    return (mdm/mA)/((mdm/mA)-muM(v,escv)+muM(v,escv)/fM(v,escv))

def fcapfull(v,mA,mdm,escv):
    if (mdm/mA) < muT(v,escv):
        return fcaplightesc(v,mA,mdm,escv)
    if muT(v,escv) < (mdm/mA):
        if (mdm/mA) < muM(v,escv):
            return fcapmidesc(v,mA,mdm,escv)
    if (mdm/mA) > muM(v,escv):
        return fcapheavyesc(v,mA,mdm,escv)
