//clang++ vel_channel2D.cpp -std=c++11 -O3

#include <iostream>
#include <vector>
#include <chrono>
#include <numeric>
#include <math.h>
#include <cmath>
#include <algorithm>

#define idx(i,j) (cellcount_x*j + i)
#define idxvelx(i,j) ((cellcount_x+1)*j + i)

//save just the sum of c_A and c_B in time. Each row corresponds to a new timestep that we save.
void save_totals(unsigned long long itime, std::vector<double> &ca, std::vector<double> &cb, int dxinit, int dyinit, int interp_steps)
{
    double catot=std::accumulate(ca.begin(), ca.end(),0.0);
    double cbtot=std::accumulate(cb.begin(), cb.end(),0.0);
    char fname[100];
    FILE *fid;
    sprintf(fname,"data/ctots_%d_%d_%d.dat", (128+8*dxinit), (160+8*dyinit), 42400);
    if (itime==0)
    {
        fid=fopen(fname, "w");
    }
    if (itime!=0)
    {
        fid=fopen(fname, "a");
    }
    fprintf(fid, "%llu %g %g \n",((itime/interp_steps*100)+42400), catot, cbtot);
    fclose(fid);
}

//saving entire concentration field
//given a `color' and a field, this has each COLUMN with a position in space.
void save_conc(unsigned long long itime, std::vector<double> &c,  int cellcount_x, int cellcount_y, int dxinit, int dyinit, int interp_steps, int interp_grid, int color)
{
    //this step is so that our output files don't grow when we make interp_grid finer
    int outputpts_x=int(cellcount_x/interp_grid);
    int outputpts_y=int(cellcount_y/interp_grid);

    char fname[100];
    FILE* fid;
    std::sprintf(fname, "./data/c_%d_%llu_%d_%d.dat", color, ((itime/interp_steps*100)+42400), (128+8*dxinit), (160+8*dyinit));
    fid=fopen(fname, "w");
    for (int j=0; j < outputpts_y; ++j)
    {
        for (int i=0; i< outputpts_x; ++i)
        {
            std::fprintf(fid, "%g ", c[idx((interp_grid*i), (interp_grid*j))]);
        }
        std::fprintf(fid, " \n");
    }
    std::fclose(fid);
}

int pbc(int cellcount, int idx)
{

    if (idx<0)
    {
        return idx+cellcount;
    }
    else
    {
        return idx-floor(idx/cellcount)*cellcount;
    }
}

void read_velocity(int &velsteps, int cellcount_x, int cellcount_y, std::vector<double> &extvelx, std::vector<double> &extvelxstep, std::vector<double> &extvely, std::vector<double> &extvelystep, int interp_steps,int &velinterpstep)
{
    char fname[100];
    FILE *fin;
    std::cout<<velsteps<<std::endl;
    
    double ev;
    int ii, jj;
    int interp_grid=int(cellcount_x/256.0);
    double grid_stepsq=(1.0/(1.0*interp_grid*interp_grid));
    
    //Read in the velocity field Ux
    std::sprintf(fname, "/Users/abbyplummer/Desktop/k46/uf_k46_%d.data", velsteps);
    fin=fopen(fname,"r");
    
    //these are in units of meters per day-- need to convert if not using physical units.
    for (int k=0; k< ((256+1)*320); k++)
    {
        fscanf(fin, "%lf \n", &ev);
        //take the index as it reads through the NOT interpolated file, and convert it to sparse i,j coords
        //This allows us to slot it in to the right place in our extvelx vector, with zeros in between that we fill in with interpolations
        ii=(k%(256+1))*interp_grid;
        jj=(k*interp_grid-ii)/(256+1);
        extvelx[idxvelx(ii,jj)]=ev;
        //subtract here on extvelxstep. Will add on the next step.
        extvelxstep[idxvelx(ii,jj)]=-1.0*ev/(interp_steps*1.0);
    }
    
    //increment by interp_grid-- this hits all points except for the ones on the far RHS where i=cellcount_x+1
    //i and j are therefore the real indexing numbers for our expanded interpolated grid.
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int i=0; i< cellcount_x; i+=interp_grid)
        {
            //now, interpolate in space between the points.
            for (int n=0; n< interp_grid; n++)
            {
                for (int m=0; m< interp_grid; m++)
                {
                    //exclude the origin explicitly-- we already know this point
                    //linear interpolation in x(m) and y(n).
                    //only need to call pbc's on y, because we are not addressing the 257th row in x.
                    if ((n!=0 || m!=0))
                    {
                        extvelx[idxvelx((i+m),(j+n))]=extvelx[idxvelx(i,j)]*(interp_grid-m)*(interp_grid-n)*grid_stepsq+extvelx[idxvelx((i+interp_grid),j)]*m*(interp_grid-n)*grid_stepsq+extvelx[idxvelx(i,pbc(cellcount_y,(j+interp_grid)))]*(interp_grid-m)*n*grid_stepsq+extvelx[idxvelx((i+interp_grid),pbc(cellcount_y,(j+interp_grid)))]*m*n*grid_stepsq;
                        //We will be subtracting this off as we go from interpolated time step to interpolated time step
                        extvelxstep[idxvelx((i+m),(j+n))]=-1.0*extvelx[idxvelx((i+m),(j+n))]/(interp_steps*1.0);
                    }
                }
            }
        }
    }
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int n=1; n< interp_grid; n++)
        {
            extvelx[idxvelx(cellcount_x, (j+n))]= extvelx[idxvelx(cellcount_x, j)]*(interp_grid-n)/(interp_grid*1.0)+extvelx[idxvelx(cellcount_x, pbc(cellcount_y,(j+interp_grid)))]*n/(interp_grid*1.0);
            extvelxstep[idxvelx(cellcount_x,(j+n))]=-1.0*extvelx[idxvelx(cellcount_x,(j+n))]/(interp_steps*1.0);
        }
    }
    fclose(fin);
    
    //Read in the next timestep
    //open up the next file and feed it in.
    std::sprintf(fname, "/Users/abbyplummer/Desktop/k46/uf_k46_%d.data",(velsteps+100));
    fin=fopen(fname,"r");
    
    for (int k=0; k< ((256+1)*320); k++)
    {
        fscanf(fin, "%lf \n", &ev);
        //take the index as it reads through the NOT interpolated file, and convert it to sparse i,j coords
        //This allows us to slot it in to the right place in our extvelx vector, with zeros in between that we fill in with interpolations
        ii=(k%(256+1))*interp_grid;
        jj=(k*interp_grid-ii)/(256+1);
        //ADD to what was already there.
        extvelxstep[idxvelx(ii,jj)]+=ev/(interp_steps*1.0);
    }
    //i and j are therefore the real indexing numbers for our expanded interpolated grid.
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int i=0; i< cellcount_x; i+=interp_grid)
        {
            //now, interpolate in space between the points.
            for (int n=0; n< interp_grid; n++)
            {
                for (int m=0; m< interp_grid; m++)
                {
                    //exclude the origin explicitly-- we already know this point
                    //linear interpolation in x(m) and y(n).
                    //only need to call pbc's on y, because we are not addressing the 257th row in x.
                    if ((n!=0 || m!=0))
                    {
                        extvelxstep[idxvelx((i+m),(j+n))]+=1.0/(interp_steps*1.0)*(extvelx[idxvelx(i,j)]*(interp_grid-m)*(interp_grid-n)*grid_stepsq+extvelx[idxvelx((i+interp_grid),j)]*m*(interp_grid-n)*grid_stepsq+extvelx[idxvelx(i,pbc(cellcount_y,(j+interp_grid)))]*(interp_grid-m)*n*grid_stepsq+extvelx[idxvelx((i+interp_grid),pbc(cellcount_y,(j+interp_grid)))]*m*n*grid_stepsq);
                    }
                }
            }
        }
    }
    //make sure to skip n=0, because this point has already been done.
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int n=1; n< interp_grid; n++)
        {
            extvelxstep[idxvelx(cellcount_x, (j+n))]+= 1.0/(interp_steps*1.0)*(extvelx[idxvelx(cellcount_x, j)]*(interp_grid-n)/(interp_grid*1.0)+extvelx[idxvelx(cellcount_x, pbc(cellcount_y,(j+interp_grid)))]*n/(interp_grid*1.0));
        }
    }
    fclose(fin);
    
    //Read in Uy velocity field now with same timesteps
    std::sprintf(fname, "/Users/abbyplummer/Desktop/k46/vf_k46_%d.data",velsteps);
    fin=fopen(fname,"r");
    for (int k=0; k< (256*(320+1)); k++)
    {
        fscanf(fin, "%lf \n", &ev);
        //take the index as it reads through the NOT interpolated file, and convert it to sparse i,j coords
        //This allows us to slot it in to the right place in our extvelx vector, with zeros in between that we fill in with interpolations
        ii=(k%256)*interp_grid;
        jj=(k*interp_grid-ii)/256;
        extvely[idx(ii,jj)]=ev;
        //subtract here on extvelxstep. Will add on the next step.
        extvelystep[idx(ii,jj)]=-1.0*ev/(interp_steps*1.0);
    }
    //increment by interp_grid-- this hits all points except for the ones on the far RHS where i=cellcount_x+1
    //i and j are therefore the real indexing numbers for our expanded interpolated grid.
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int i=0; i< cellcount_x; i+=interp_grid)
        {
            //now, interpolate in space between the points.
            for (int n=0; n< interp_grid; n++)
            {
                for (int m=0; m< interp_grid; m++)
                {
                    //exclude the origin explicitly-- we already know this point
                    //linear interpolation in x(m) and y(n).
                    //only need to call pbc's on x, because we are not addressing the 361st row in y.
                    if ((n!=0 || m!=0))
                    {
                        extvely[idx((i+m),(j+n))]=extvely[idx(i,j)]*(interp_grid-m)*(interp_grid-n)*grid_stepsq+extvely[idx(pbc(cellcount_x,(i+interp_grid)),j)]*m*(interp_grid-n)*grid_stepsq+extvely[idx(i,(j+interp_grid))]*(interp_grid-m)*n*grid_stepsq+extvely[idx(pbc(cellcount_x,(i+interp_grid)),(j+interp_grid))]*m*n*grid_stepsq;
                        //We will be subtracting this off as we go from interpolated time step to interpolated time step
                        extvelystep[idx((i+m),(j+n))]=-1.0*extvely[idx((i+m),(j+n))]/(interp_steps*1.0);
                    }
                }
            }
        }
    }
    for (int i=0; i< cellcount_x; i+=interp_grid)
    {
        for (int m=1; m< interp_grid; m++)
        {
            extvely[idx((i+m), cellcount_y)]= extvely[idx(i, cellcount_y)]*(interp_grid-m)/(interp_grid*1.0)+extvely[idx(pbc(cellcount_x, (i+interp_grid)),cellcount_y)] *m/(interp_grid*1.0);
            extvelystep[idx((i+m),cellcount_y)]=-1.0*extvely[idx((i+m),cellcount_y)]/(interp_steps*1.0);
        }
    }
    fclose(fin);
    
    //Read in the next timestep
    //open up the next file and feed it in.
    std::sprintf(fname, "/Users/abbyplummer/Desktop/k46/vf_k46_%d.data",(velsteps+100));
    fin=fopen(fname,"r");
        
    for (int k=0; k< (256*(320+1)); k++)
    {
        fscanf(fin, "%lf \n", &ev);
        //take the index as it reads through the NOT interpolated file, and convert it to sparse i,j coords
        //This allows us to slot it in to the right place in our extvelx vector, with zeros in between that we fill in with interpolations
        ii=(k%256)*interp_grid;
        jj=(k*interp_grid-ii)/256;
        //ADD to what was already there.
        extvelystep[idx(ii,jj)]+=ev/(interp_steps*1.0);
    }
    
    //i and j are therefore the real indexing numbers for our expanded interpolated grid.
    for (int j=0; j< cellcount_y; j+=interp_grid)
    {
        for (int i=0; i< cellcount_x; i+=interp_grid)
        {
            //now, interpolate in space between the points.
            for (int n=0; n< interp_grid; n++)
            {
                for (int m=0; m< interp_grid; m++)
                {
                    //exclude the origin explicitly-- we already know this point
                    //linear interpolation in x(m) and y(n).x.
                    if ((n!=0 || m!=0))
                    {
                        extvelystep[idx((i+m),(j+n))]+=1.0/(interp_steps*1.0)*(extvely[idx(i,j)]*(interp_grid-m)*(interp_grid-n)*grid_stepsq+extvely[idx(pbc(cellcount_x,(i+interp_grid)),j)]*m*(interp_grid-n)*grid_stepsq+extvely[idx(i,(j+interp_grid))]*(interp_grid-m)*n*grid_stepsq+extvely[idx(pbc(cellcount_x,(i+interp_grid)),(j+interp_grid))]*m*n*grid_stepsq);
                    }
                }
            }
        }
    }
    for (int i=0; i< cellcount_x; i+=interp_grid)
    {
        for (int m=1; m< interp_grid; m++)
        {
            extvelystep[idx((i+m), cellcount_y)]+= 1.0/(interp_steps*1.0)*(extvely[idx(i, cellcount_y)]*(interp_grid-m)/(interp_grid*1.0)+extvely[idx(pbc(cellcount_x, (i+interp_grid)),cellcount_y)] *m/(interp_grid*1.0));
        }
    }
    fclose(fin);
    
    //increment velsteps, so we start at a new velocity next time. this is only to keep track of the file names, which increment in 100's.
    velsteps+=100;
    //set velinterpstep to one. We will count this as 1, 2, 3, so that we only have to call one program, find velocity, even for the first step.
    velinterpstep=1;
}

void step_velocity(int cellcount_x, int cellcount_y, std::vector<double> &extvelx, std::vector<double> &extvelxstep, std::vector<double> &extvely, std::vector<double> &extvelystep, int &velinterpstep)
{
    for (int j=0; j<(cellcount_y+1); j++)
    {
        for (int i=0; i<(cellcount_x+1); i++)
        {
        //to each extvel, add one `unit' of the next velocity timestep, according to how many interpolation steps total there are, and subtract one.
            if (j!=cellcount_y)
            {
                extvelx[idxvelx(i,j)]+=extvelxstep[idxvelx(i,j)];
            }
            if (i!=cellcount_x)
            {
                extvely[idx(i,j)]+=extvelystep[idx(i,j)];
            }
        }
    }

    //increment the number of times extvelstep has been added.
    velinterpstep+=1;
}

void find_velocity(int &velsteps, int cellcount_x, int cellcount_y, std::vector<double> &extvelx, std::vector<double> &extvelxstep,std::vector<double> &extvely, std::vector<double> &extvelystep,int interp_steps, int &velinterpstep)
{
    //if veltinterpstep is either equal to 0 (as it is initially) or equal to interp_steps (as it is after the correct number of steps), read a new velocity step.
    if (velinterpstep%interp_steps!=0)
    {
        step_velocity(cellcount_x, cellcount_y, extvelx, extvelxstep, extvely, extvelystep, velinterpstep);
    }
    else if (velinterpstep%interp_steps==0)
    {
        read_velocity(velsteps, cellcount_x, cellcount_y, extvelx, extvelxstep, extvely, extvelystep, interp_steps, velinterpstep);
    }
}

//calculate dc_a/dt* Delta t (if color=0) or dc_b/dt Delta t(if color=1), given c_a and c_b at a certain timestep
double derivdt(int color, std::vector<double> &ca, std::vector<double> &cb, double mu_param, double diff_param, double s, int cellcount_x, int cellcount_y, int i, int j)
{
    if (color==0)
    {
        return mu_param*ca[idx(i,j)]*(1-ca[idx(i,j)]-cb[idx(i,j)])+diff_param*(ca[idx(pbc(cellcount_x,(i+1)), j)]-2*ca[idx(i,j)]+ca[idx(pbc(cellcount_x,(i-1)),j)]+ca[idx(i,pbc(cellcount_y,(j+1)))]-2*ca[idx(i,j)]+ca[idx(i,pbc(cellcount_y,(j-1)))])+s*mu_param*ca[idx(i,j)]*cb[idx(i,j)];
    }
    if (color==1)
    {
        return mu_param*cb[idx(i,j)]*(1-ca[idx(i,j)]-cb[idx(i,j)])+diff_param*(cb[idx(pbc(cellcount_x,(i+1)), j)]-2*cb[idx(i,j)]+cb[idx(pbc(cellcount_x,(i-1)),j)]+cb[idx(i,pbc(cellcount_y,(j+1)))]-2*cb[idx(i,j)]+cb[idx(i,pbc(cellcount_y,(j-1)))])-s*mu_param*ca[idx(i,j)]*cb[idx(i,j)];
    }
    else
    {
        std::cout<<"error in derivdt"<<std::endl;
        return 0;
    }
}

double dvelcdt(std::vector<double> &c, std::vector<double> &extvelx, std::vector<double> &extvely, double dtdx, int cellcount_x, int cellcount_y, int i, int j)
{
//  We evaluate at F_(i+1/2)- F_(i-1/2). We have velocities given on faces
    //sum contribution from dvelx and dvely
    return -dtdx*0.5*((extvelx[idxvelx((i+1),j)]*(c[idx(pbc(cellcount_x, (i+1)), j)]+c[idx(i,j)])- extvelx[idxvelx(i,j)]*(c[idx(pbc(cellcount_x, (i-1)),j)]+c[idx(i,j)]))+(extvely[idx(i,(j+1))]*(c[idx(i, pbc(cellcount_y, (j+1)))]+c[idx(i,j)])- extvely[idx(i,j)]*(c[idx(i,pbc(cellcount_y, (j-1)))]+c[idx(i,j)])));
}

//goes in: cap (past info step), ca (current info step), can (empty buffer), and params.
//comes out: cap and ca, both updated by one step.
void abstep(std::vector<double> &cap, std::vector<double> &cbp, std::vector<double> &ca, std::vector<double> &cb, std::vector<double> &can, std::vector<double> &cbn, double mu_param, double diff_param, double dtdx,  double s,  int cellcount_x, int cellcount_y, std::vector<double> &extvelx, std::vector<double> &extvely)
{
    //fill new concentration vector with information, drawing from t=0 and t=-1
    for (int i=0; i< cellcount_x; i++)
    {
        for (int j=0; j< cellcount_y; j++)
        {
            can[idx(i,j)]=ca[idx(i,j)]+(1.5*(derivdt(0, ca, cb, mu_param, diff_param, s, cellcount_x, cellcount_y, i, j)+dvelcdt(ca, extvelx, extvely, dtdx, cellcount_x, cellcount_y, i, j))- 0.5*(derivdt(0, cap, cbp, mu_param, diff_param, s, cellcount_x, cellcount_y, i, j)+dvelcdt(cap, extvelx, extvely, dtdx, cellcount_x, cellcount_y, i, j)));
            cbn[idx(i,j)]=cb[idx(i,j)]+(1.5*(derivdt(1, ca, cb, mu_param, diff_param, s, cellcount_x, cellcount_y, i,j)+dvelcdt(cb, extvelx, extvely, dtdx, cellcount_x, cellcount_y, i, j))- 0.5*(derivdt(1, cap, cbp, mu_param, diff_param, s, cellcount_x, cellcount_y, i, j)+dvelcdt(cbp, extvelx, extvely, dtdx, cellcount_x, cellcount_y, i, j)));
        }
    }
    //swap can and cap
    can.swap(cap);
    cbn.swap(cbp);
    //swap ca and cap, so that the newest step is in ca and the oldest step is in cap
    ca.swap(cap);
    cb.swap(cbp);
}

void initialize(int cellcount_x, int cellcount_y, int interp_grid, std::vector<double> &ca, std::vector<double> &cb, int dx, int dy)
{
    double width=8*interp_grid;
    double offset=width;
    for (int j=0; j< cellcount_y; j++)
    {
        for (int i=0; i< cellcount_x; i++)
        {
            ca[idx(i,j)]=exp(-pow(((i-cellcount_x/2.0)-offset*dx),2)/(2*width*width))*exp(-pow(((j-cellcount_y/2.0)-offset*dy),2)/(2*width*width))+ exp(-pow(((i-(cellcount_x/2.0+cellcount_x))-offset*dx),2)/(2*width*width))*exp(-pow(((j-cellcount_y/2.0)-offset*dy),2)/(2*width*width))+ exp(-pow(((i-(cellcount_x/2.0-cellcount_x))-offset*dx),2)/(2*width*width))*exp(-pow(((j-cellcount_y/2.0)-offset*dy),2)/(2*width*width))+ exp(-pow(((i-cellcount_x/2.0)-offset*dx),2)/(2*width*width))*exp(-pow(((j-(cellcount_y/2.0-cellcount_y))-offset*dy),2)/(2*width*width))+ exp(-pow(((i-cellcount_x/2.0)-offset*dx),2)/(2*width*width))*exp(-pow(((j-(cellcount_y/2.0+cellcount_y))-offset*dy),2)/(2*width*width));
            cb[idx(i,j)]=1.0-ca[idx(i,j)];
        }
    }
}

int main(int argc, char const *argv[])
{
    //Make a folder for it.
    system("mkdir -p data");
    
    /* PARAMETERS */
    
    //mu=1/generation time (approx 1 per day in physical units)
    double mufactor=1.0;
    //factor determines where the Gaussian is centered.
    int dxinit=std::atof(argv[1]);
    int dyinit=std::atof(argv[2]);
    double mu=1.0/86400.0*mufactor;
    double s=0.0;

    double diffusivity=5.0;
    
    /* TIMESTEPPING */
    //The velocity field timestep is 10800 seconds (every 3 hours).
    //interp_steps is the number of times we will run the population genetics code PER velocity timestep, as we move through it.
    int interp_steps=2000;
    double timestep=10800.0/(interp_steps*1.0);
    std::cout<<"Timestep: "<<timestep<<std::endl;
    
    /* DOMAIN SIZE */
    //in physical units, in the direction that pbcs are domain_size=128 km
    double domain_length_x=128000.0;
    
    //Interpgrid describes how we divide up space within each of Mara's sampling box (so that nothing gets too sharp or blows up)
    int interp_grid=20;
    
    //256 is the number of velocity data points that we get in the x direction.
    int cellcount_x=256*interp_grid;
    //more boxes in the y direction, but we will only analyze data in a box of size 256 x 256, because close to the edges (32 boxes on either side), there's a variable grid size to prevent eddies back scattering and closed boundaries. This is how the velocity was generated-- we just use that velocity field with PBCs.
    int cellcount_y=320*interp_grid;
    
    std::cout<<"Interpolated gridpoints: "<<int(cellcount_x/256.0)<<std::endl;
    //in physical units, 500 m
    double dx=domain_length_x/(cellcount_x*1.0);
    
    //if 86400/timestep*n, this is running for n days (and also n generations).
    unsigned long long timemax=int(86400.0/(1.0*timestep));
    //output every velocity step
    int outputevery=interp_steps;

    //physical time is itime times timestep
    unsigned long long itime=0;
    int startday=1;
    int velsteps=42400;
    //velinterpstep must be initialized with zero so that we first read the file. But otherwise it ranges from 1 to interp_steps
    int velinterpstep=0;

    double mu_param=mu*timestep;
    double diff_param=diffusivity*timestep/(dx*dx);
    double dtdx=timestep/dx;
    
    //concentration fields
    std::vector<double> ca(cellcount_x*cellcount_y);
    std::vector<double> cb(cellcount_x*cellcount_y);
    //vectors to hold new steps (that we won't refer to explicitly)
    std::vector<double> can(cellcount_x*cellcount_y);
    std::vector<double> cbn(cellcount_x*cellcount_y);
    //vectors to hold past timesteps
    std::vector<double> cap(cellcount_x*cellcount_y);
    std::vector<double> cbp(cellcount_x*cellcount_y);
    
    //external velocity at each timestep
    std::vector<double> extvelx((cellcount_x+1)*cellcount_y);
    std::vector<double> extvelxstep((cellcount_x+1)*cellcount_y);
    
    std::vector<double> extvely(cellcount_x*(cellcount_y+1));
    std::vector<double> extvelystep(cellcount_x*(cellcount_y+1));
    
    typedef std::chrono::high_resolution_clock myclock;
    myclock::time_point beginning = myclock::now();

    //set up
    initialize(cellcount_x, cellcount_y, interp_grid, ca, cb, dxinit, dyinit);
    //read in so that extvel contains the correct velocity.
    find_velocity(velsteps, cellcount_x, cellcount_y, extvelx, extvelxstep, extvely, extvelystep, interp_steps, velinterpstep);
    //save initial condition
    save_totals(itime, ca, cb, dxinit, dyinit, interp_steps);
    save_conc(itime, ca, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps, interp_grid,0);
    save_conc(itime, cb, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps, interp_grid,1);

    //initialize past step to be t=0. this is a copying step.
    cap=ca;
    cbp=cb;
    //take one euler step so that you have t=1, and put it in concentration "new" vector.
    for (int j=0; j<cellcount_y; j++)
    {
        for (int i=0; i<cellcount_x; i++)
        {
            can[idx(i,j)]=ca[idx(i,j)]+(derivdt(0, ca, cb, mu_param, diff_param, s, cellcount_x, cellcount_y,i, j)+dvelcdt(ca, extvelx, extvely,dtdx, cellcount_x, cellcount_y,i,j));
            cbn[idx(i,j)]=cb[idx(i,j)]+(derivdt(1, ca, cb, mu_param, diff_param, s, cellcount_x, cellcount_y, i, j)+dvelcdt(cb,extvelx, extvely, dtdx,cellcount_x, cellcount_y,i,j));
        }
    }
    can.swap(ca);
    cbn.swap(cb);
    
    //ready to do two-step adams bashforth for timemax steps.
    for (itime=1; itime<(timemax+1); itime++)
    {
        //need to step the velocity here (or read new one, depending on the timestep).
        find_velocity(velsteps, cellcount_x, cellcount_y, extvelx, extvelxstep, extvely, extvelystep, interp_steps, velinterpstep);
        abstep(cap, cbp, ca, cb, can, cbn, mu_param, diff_param, dtdx, s, cellcount_x, cellcount_y, extvelx, extvely);
        if (itime%outputevery==0)
        {
            save_totals(itime, ca, cb, dxinit, dyinit, interp_steps);
            save_conc(itime, ca, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps,interp_grid,0);
            save_conc(itime, cb, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps, interp_grid,1);
    
            if (*std::min_element(ca.begin(), ca.end())<0.0 || *std::min_element(cb.begin(), cb.end())<0.0)
            {
                std::cout<<"Numerically unstable- negative concentration"<<std::endl;
                break;
            }
        }
    }
    
    save_totals(itime, ca, cb, dxinit, dyinit, interp_steps);
    save_conc(itime, ca, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps,interp_grid, 0);
    save_conc(itime, cb, cellcount_x, cellcount_y, dxinit, dyinit, interp_steps, interp_grid,1);
          
    typedef std::chrono::duration<double> fsec;
    fsec duration = myclock::now() - beginning;
    std::cout << "It took me " << duration.count() << " seconds." <<std::endl;
    
    return 0;
}
