#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import xarray as xr
import numpy as np
import wind_powercurves2 as wpc
import glob
import os
import datetime as dt
from joblib import Parallel, delayed
import sys

def calc_power(in_file, land_mask, offshore_mask, out_dir, t0):
    t1 = dt.datetime.now()
    out_file = out_dir + os.path.basename(in_file).replace("wspd","power")
    if os.path.isfile(out_file):
        print(out_file, "exists")
        return 1
    with xr.open_dataset(in_file) as ws:
        pc_out = ws.drop_vars(["score"])
        print(in_file, "loading wind speed", dt.datetime.now()-t0, dt.datetime.now()-t1)
        wind_sp = ws["score"].data
        print(in_file, "calulating power curve", dt.datetime.now()-t0, dt.datetime.now()-t1)
        power_c = wpc.powercurve_land4(wind_sp)*land_mask + wpc.powercurve_offshore4(wind_sp)*offshore_mask
        
        pc_out["power"] = xr.DataArray(data = power_c, coords = ws["score"].coords)
        print(in_file, "writing file", dt.datetime.now()-t0, dt.datetime.now()-t1)
        pc_out.to_netcdf(out_file)
        print(in_file, "done", dt.datetime.now()-t0, dt.datetime.now()-t1)

in_dir = "/path/to/ERA5_wspd_150m/"
mask_dir = "/path/to/wind_masks_projected/"
out_dir = "/path/to/ERA5_powercurve/"

in_files = sorted(glob.glob(in_dir + "*.nc"))

#masks
land_mask = xr.open_rasterio(mask_dir + "usable-area+buffer301m_ERA5.tif").squeeze()
offshore_mask = xr.open_rasterio(mask_dir + "offshore_usable_HeightGt-1000_shippingGt5000_NotProtected_mask_ERA5.tif").squeeze()

mask_sum = land_mask + offshore_mask

#limit mask_sum to max 1, flip upside down and convert to np.array
land_mask = (land_mask/mask_sum.where(mask_sum>1,1)).data[::-1,:]
offshore_mask = (offshore_mask/mask_sum.where(mask_sum>1,1)).data[::-1,:]

# shell mode
#in_file = sys.argv[1]
#print(in_file)
#t0 = dt.datetime.now()
#calc_power(in_file, land_mask, offshore_mask, out_dir, t0)

# parallel mode
num_cores = 12
t0 = dt.datetime.now()
parallel_input = [(in_file, land_mask, offshore_mask, out_dir, t0) for in_file in in_files]
Parallel(n_jobs=num_cores)(delayed(calc_power)(*args) for args in parallel_input) 
        
         
         
