#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Created on Wed Feb 16 10:15:26 2022

@author: mphilipp
"""

import pandas as pd
import xarray as xr
import numpy as np
import glob
import time
import pytz 
import timezonefinder 
import datetime

bias_corrected_path = "/path/to/bias/corrected/data/"
sun_times_output_path= "/path/to/output/sun_times.nc"
def calculate_sun_times():
    '''
    Computes the times of sunrise, sunnoon, sunset as well as daylength for each day as well as timezone. 

    Yields
    ------
    None.

    '''

    # Create dummy data set, filled with zeros
    dummy = xr.open_dataset(bias_corrected_path+"sfcWind_ICHEC-EC-EARTH_KNMI-RACMO22E_rcp85_2021-2030_.nc")["sfcWind"]
    emptydata = np.zeros((366,len(dummy.lat.values),len(dummy.lon.values)),dtype=np.float32)
    days = np.arange(1,367)
    
    sun_times = xr.Dataset(data_vars=dict(
                                 sunrise =(["dayofyear","lat","lon"],emptydata),
                                 sunnoon =(["dayofyear","lat","lon"],emptydata),
                                 sunset =(["dayofyear","lat","lon"],emptydata),
                                 daylength =(["dayofyear","lat","lon"],emptydata),
                                 timezone = (["lat","lon"],emptydata[0,:,:])),
        
                                 coords=dict(lon=dummy.lon.values,lat=dummy.lat.values,dayofyear=days))
   
    # Create lat and lon grid 
    lon_grid = np.array(len(sun_times.timezone.lat.values)*[sun_times.timezone.lon.values])
    lat_grid = np.array(len(sun_times.timezone.lon.values)*[sun_times.timezone.lat.values]).T
    dayofyear_grid = np.ones(emptydata.shape)
    
    # Get timezone for every pixel
    tf = timezonefinder.TimezoneFinder()
    for iy, ix in np.ndindex(sun_times.timezone.shape):
        # Fill day of year grid
        dayofyear_grid[:,iy,ix]= np.arange(1,367)
        # Get Timezone
        timezone_str = tf.timezone_at(lat=float(sun_times.lat[iy].values), lng=float(sun_times.lon[ix].values))
        tz = pytz.timezone(timezone_str)
        tz = int(tz.utcoffset(datetime.datetime.utcnow()).seconds/3600)
        if tz > 12:
            tz = tz-24
        sun_times.timezone[iy,ix]=tz    
        
    # doy = np.array([(d - d.replace(day=1, month=1)).days + 1 for d in df.index])  # day of year, in melodist

    # Day angle and declination after Bourges (1985):
    day_angle_b = np.deg2rad((360. / 365.25) * (dayofyear_grid - 79.346))
    
    declination = np.deg2rad(
        0.3723 + 23.2567 * np.sin(day_angle_b) - 0.7580 * np.cos(day_angle_b)
        + 0.1149 * np.sin(2*day_angle_b) + 0.3656 * np.cos(2*day_angle_b)
        - 0.1712 * np.sin(3*day_angle_b) + 0.0201 * np.cos(3*day_angle_b)
    )
    
    # Equation of time with day angle after Spencer (1971):
    day_angle_s = 2 * np.pi * (dayofyear_grid - 1) / 365.
    eq_time = 12. / np.pi * (
        0.000075 +
        0.001868 * np.cos(  day_angle_s) - 0.032077 * np.sin(  day_angle_s) -
        0.014615 * np.cos(2*day_angle_s) - 0.040849 * np.sin(2*day_angle_s)
        )
    
    standard_meridian = sun_times.timezone.values * 15.
    delta_lat_time = (lon_grid - standard_meridian) * 24. / 360.
    
    omega_nul_arg = -np.tan(np.deg2rad(lat_grid)) * np.tan(declination)
    omega_nul = np.arccos(omega_nul_arg)
    sunrise = 12. * (1. - (omega_nul) / np.pi) - delta_lat_time - eq_time
    sunset  = 12. * (1. + (omega_nul) / np.pi) - delta_lat_time - eq_time

    # as an approximation, solar noon is independent of the below mentioned
    # cases:
    sunnoon  = 12. * (1.) - delta_lat_time - eq_time
    
    # $kf 2015-11-13: special case midnight sun and polar night
    # CASE 1: MIDNIGHT SUN
    # set sunrise and sunset to values that would yield the maximum day
    # length even though this a crude assumption
    pos = omega_nul_arg < -1
    sunrise[pos] = sunnoon[pos] - 12
    sunset[pos]  = sunnoon[pos] + 12

    # CASE 2: POLAR NIGHT
    # set sunrise and sunset to values that would yield the minmum day
    # length even though this a crude assumption
    pos = omega_nul_arg > 1
    sunrise[pos] = sunnoon[pos]
    sunset[pos]  = sunnoon[pos]

    daylength = sunset - sunrise
        
    # adjust if required
    sunrise[sunrise < 0] += 24
    sunset[sunset > 24] -= 24

    sun_times.sunrise.values = sunrise
    sun_times.sunnoon.values = sunnoon
    sun_times.sunset.values = sunset
    sun_times.daylength.values = daylength
    
    # Spit at 29th of february and reuse the 28th feb for the 29th
    sun_times.sunrise[60:] = sun_times.sunrise[59:-1].values
    sun_times.sunnoon[60:] = sun_times.sunnoon[59:-1].values
    sun_times.sunset[60:] = sun_times.sunset[59:-1].values
    sun_times.daylength[60:] = sun_times.daylength[59:-1].values

    sun_times.sunrise[59] = sun_times.sunrise[58]
    sun_times.sunnoon[59] = sun_times.sunnoon[58]
    sun_times.sunset[59] = sun_times.sunset[58]
    sun_times.daylength[59] = sun_times.daylength[58]
    
    # Save
    sun_times.to_netcdf(sun_times_output_path)

if __name__ == "__main__":     
     calculate_sun_times()
