"""
Write NEXUS files suitable as input for SplitsTree, BEAST.
The following options can be set with the parameter "--options":
    - Exclude cognate sets marked as Parallel Derivation
    - Exclude cognate sets marked as Ideophonic
    - Exclude cognate sets marked as Loan event
    - Exclude cognate sets marked as Parallel Loan event
    - Include Parallel Loan events as independent cognate sets

by using a '1' for True and a '0' for False.
"""

import pathlib
import re
import string
import time
from string import ascii_uppercase
from textwrap import dedent
from cldfbench.cli_util import add_dataset_spec, get_dataset
from collections import defaultdict, OrderedDict
from nexus import NexusWriter


NEXUS_COMMENTS = [
    ('excludePllDerivation', "Exclude cog. sets: Pll. Derivation: {}"),
    ('excludeIdeophonic', "Exclude cog. sets: Ideophonic: {}"),
    ('excludeDubious', "Exclude cog. sets: Dubious: {}"),
    ('excludeLoanword', "Exclude cog. sets: Loan event: {}"),
    ('excludePllLoan', "Exclude cog. sets: Pll Loan: {}"),
    ('includePllLoan', "Include Pll Loan as independent cog. sets: {}"),
]


def register(parser):
    add_dataset_spec(parser)
    parser.add_argument(
        '--options',
        type=str,
        help='''a string separated by a space of 0\'s and 1\'s specifying:
        excludePllDerivation excludeIdeophonic excludeLoanword
        excludePllLoan includePllLoan''',
        default='1 1 0 0 1',
    )


def run(args):

    opts = args.options.split()
    if len(opts) != 5:
        args.log.error('Please check the number of options - run "cldfbench iecor.make_nexus -h" for help')
        return 1

    start_time = time.time()

    opts = [bool(int(x)) for x in opts]

    spec_dict = {
        'excludePllDerivation': opts[0],
        'excludeIdeophonic': opts[1],
        'excludeLoanword': opts[2],
        'excludePllLoan': opts[3],
        'includePllLoan': opts[4],
    }

    dialect_full_name = "NeighborNet"

    for k, v in NEXUS_COMMENTS:
        if k in spec_dict:
            print(v.format(spec_dict[k]))

    print()
    q = input('Are these settings correct? [Y/n]:')
    if len(q) and q[0].lower() != 'y':
        return

    nexwriter = NexusWriter()

    ds = get_dataset(args)
    ds_cldf = ds.cldf_reader()

    languages = sorted(ds_cldf["LanguageTable"], key=lambda x: x['sort_order'])
    language_names = [lg["ascii_name"] for lg in languages]

    meanings = sorted(ds_cldf["ParameterTable"], key=lambda x: x['Name'])
    max_len = max([len(lg) for lg in language_names])

    cladeLevels = {}
    for c in ds_cldf.get('clades.csv'):
        cl = []
        for lv in [c['level0_name'], c['level1_name'], c['level2_name'], c['level3_name']]:
            if lv is not None and len(lv):
                cl.append(lv)
        if len(cl) == 1 and cl[0] != c['clade_name']:
            cl.append(c['clade_name'])
        if len(cl) == 3:
            cl[2] = c['clade_name']
        cladeLevels[':'.join(cl)] = (c['clade_level0'], c['clade_level1'])

    args.log.info('construct data')
    matrix, cognate_class_names, assumptions = construct_matrix(
        ds_cldf, languages, meanings, **spec_dict)

    # Export data to compose for BEAST:
    exportBEAUti = []

    nexcom = getNexusComments(dialect_full_name, **spec_dict)

    nexwriter.add_comment("Generated by: {}".format("IE-CoR CLDF dataset"))
    nexwriter.add_comment("")
    nexwriter.add_comment(ds.metadata.citation)
    for c in nexcom:
        nexwriter.add_comment(c)

    exportBEAUti.append("#NEXUS\n\n[ Generated by: {} ]\n".format("IE-CoR CLDF dataset"))
    exportBEAUti.append(nexus_comment(ds.metadata.citation))
    exportBEAUti.append("".join(["[ {} ]\n".format(c) for c in nexcom]))

    max_len += 2  # taxlabels are quoted
    exportBEAUti.append(dedent("""\
        begin characters;
          dimensions nchar={};
          format symbols="01" missing=?;""".format(len(cognate_class_names))))
    labels = []
    for i, cc in enumerate(cognate_class_names):
        labels.append("    {} {}".format(i + 1, cc))
    exportBEAUti.append("  matrix")

    matrixComments = getMatrixCommentsFromCognateNames(
        cognate_class_names)

    padding = " " * (max_len + 4)
    for c in matrixComments:
        if len(c.strip()) > 0:
            exportBEAUti.append("{}[{}]".format(padding, c))
            nexwriter.add_collabels(c)
        else:
            exportBEAUti.append("")
            nexwriter.add_collabels("")

    exportBEAUti.append("\n")

    for row_ in matrix:
        language_name, row = row_[0], row_[1:]
        def quoted(s):
            return "'{}'".format(s)
        for i, ccn in enumerate(cognate_class_names):
            nexwriter.add(quoted(row_[0]), ccn, row_[i+1])
        exportBEAUti.append("    {} {}{}".format(
                      quoted(language_name),
                      " " * (max_len - len(quoted(language_name))),
                      "".join(row)))

    exportBEAUti.append("\n")
    for c in matrixComments:
        if len(c.strip()) > 0:
            exportBEAUti.append("{}[{}]".format(padding, c))
        else:
            exportBEAUti.append("")

    exportBEAUti.append("  ;\nend;\n")
    exportBEAUti.append("begin assumptions;")
    for charset in assumptions:
        exportBEAUti.append("    " + charset)
    exportBEAUti.append("end;")
    exportBEAUti.append(getNexusLocations(languages))
    exportBEAUti.append(getCharstateLabels(cognate_class_names))
    exportBEAUti.append(cladeMembership(ds_cldf, languages))
    exportBEAUti.append(computeCalibrations(ds_cldf, languages))

    args.log.info("processed {} cognate sets from {} languages and {} meanings".format(
                  len(cognate_class_names), len(matrix), len(meanings)))

    fname = 'IE-CoR_{}'.format(args.options.replace(' ', ''))

    nexwriter.write_to_file("{}.nex".format(fname),
                            interleave=False, charblock=True, preserve_order=True)

    with open(ds.dir / '{}_BEAUti.nex'.format(fname), 'w', encoding='utf-8') as f:
        for s in exportBEAUti:
            f.write(s + '\n')


def nexus_comment(s):
    lines = s.split("\n")
    maxlen = max(len(e) for e in lines)
    return "\n".join("[ "+e.ljust(maxlen)+" ]" for e in lines)


def getNexusComments(dialect_full_name, **spec_dict):
    lines = ["Nexus dialect: {}".format(dialect_full_name)]
    for k, v in NEXUS_COMMENTS:
        if k in spec_dict:
            lines.append(v.format(spec_dict[k]))
    lines.append("File generated: {}".format(
                 time.strftime("%Y-%m-%d %H:%M:%S", time.localtime())))
    return lines


def construct_matrix(ds_cldf,
                     languages,                # [Language]
                     meanings,                 # [Meaning]
                     excludePllDerivation,     # bool
                     excludeIdeophonic,        # bool
                     excludeLoanword,          # bool
                     excludePllLoan,           # bool
                     includePllLoan,           # bool
                     **spec_dict):             # don't care

    cs = {c['ID']: c for c in ds_cldf.get('cognatesets.csv') if c['ID']}

    loans = {c['Cognateset_ID']: c for c in ds_cldf.get('loans.csv') if c['Cognateset_ID']}

    cognates_form_ids = set(c['Form_ID'] for c in ds_cldf.get('cognates.csv'))

    forms = {f['ID']: f for f in ds_cldf['FormTable']}
    parameters = {p['ID']: p for p in meanings}

    cognate_classes = defaultdict(list)
    cj_for_meaning = defaultdict(list)

    for c in ds_cldf.get('cognates.csv'):
        if c['Cognateset_ID'] not in cs:
            continue
        if excludePllDerivation:
            if cs[c['Cognateset_ID']]['parallelDerivation']:
                continue
        if excludeIdeophonic:
            if cs[c['Cognateset_ID']]['Ideophonic']:
                continue
        if excludeLoanword:
            if c['Cognateset_ID'] in loans:
                continue
        elif excludePllLoan:
            if c['Cognateset_ID'] in loans and loans[c['Cognateset_ID']]['Parallel_loan_event']:
                continue
        frm = forms[c['Form_ID']]
        param = parameters[frm['Parameter_ID']]
        c['Parameter_Name'] = param['Name']
        c['Parameter_ID'] = param['ID']
        c['Language_ID'] = frm['Language_ID']
        if c['Parameter_ID'] in cognate_classes:
            if c['Cognateset_ID'] not in cognate_classes[c['Parameter_ID']]:
                cognate_classes[c['Parameter_ID']].append(c['Cognateset_ID'])
        else:
            cognate_classes[c['Parameter_ID']].append(c['Cognateset_ID'])
        cj_for_meaning[c['Parameter_ID']].append(c)

    data = dict()
    pllloan_lexemes = []
    pllloan_cnt_clades = {}
    pllloan_cnt_lexemes = {}
    lex_order_due_lgs = [x['ID'] for x in languages]

    # get for all languages the clade ids for sorting cognate classes
    clade_names = {c['clade_name']: c['ID'] for c in ds_cldf.get('clades.csv')}
    languageClades = {c['ID']: clade_names[c['clade_name']] for c in ds_cldf['LanguageTable']}

    for meaning in meanings:
        cj_for_current_meaning = cj_for_meaning[meaning["ID"]]
        plls = []
        for cc in cognate_classes[meaning["ID"]]:
            matches = [c['Language_ID'] for c in cj_for_current_meaning if c['Cognateset_ID'] == cc]
            if matches:
                if includePllLoan:
                    pllloan_clds = set(languageClades[m] for m in matches if m in languageClades)
                    pllloan_cnt_clades[str(cc)] = len(pllloan_clds)
                    if cc in loans and loans[cc]['Parallel_loan_event']:
                        cjs = [(c['Language_ID'], c) for c in cj_for_current_meaning if c['Cognateset_ID'] == cc]
                        l_cnt = 25
                        if meaning['ID'] not in data:
                            data.setdefault(meaning['ID'], dict())
                        # sort lexemes according language list order
                        lex_cnt = 0
                        for cj in sorted(cjs, key=lambda x: lex_order_due_lgs.index(x[0])):
                            pllloan_lexemes.append(cj[1]['Form_ID'])
                            data[meaning['ID']].setdefault(
                                ("Z", "{}{}".format(cc, ascii_uppercase[l_cnt]), cj[1]['Form_ID']), list())\
                                .append(cj[1]['Language_ID'])
                            l_cnt -= 1
                            lex_cnt += 1
                        pllloan_cnt_lexemes[str(cc)] = lex_cnt
                    else:
                        data.setdefault(meaning['ID'], dict())[cc] = matches
                else:
                    data.setdefault(meaning['ID'], dict())[cc] = matches

    meaningIDsForLanguages = defaultdict(set)

    # # adds a cc code for all singletons
    # # (lexemes which are not registered as
    # # belonging to a cognate class), and add to cognate_class_dict
    for lexeme in ds_cldf['FormTable']:
        meaningIDsForLanguages[lexeme['Language_ID']].add(lexeme['Parameter_ID'])
        if lexeme['ID'] not in cognates_form_ids and lexeme['ID'] not in pllloan_lexemes:
            cc = ("U", lexeme['ID'])  # use tuple for sorting
            data[lexeme['Parameter_ID']].setdefault(
                cc, list()).append(lexeme['Language_ID'])

    def cognate_class_name_formatter(cc, gloss):
        # gloss = cognate_class_dict[cc]
        if isinstance(cc, (int, str)):
            return "{}_cognate_{}".format(gloss, cc)
        if isinstance(cc, (list, tuple)):
            if len(cc) == 2:
                return "{}_lexeme_{}".format(gloss, cc[1])
            elif len(cc) == 3:
                return "{}_cognate_{}_pllloanlexeme_{}".format(gloss, re.sub(r'[A-Z]', '', cc[1]), cc[2])
        return "{}_ERROR_{}".format(gloss, str(cc))

    def get_cognate_class_id_for_dataTable(cnt, cc):
        # is used for the later sorting of dict keys and preserving
        # the order of passed cognate set IDs
        if isinstance(cc, (int, str)):
            return "{}___{}".format(str(cnt).zfill(3), str(cc))
        if isinstance(cc, (list, tuple)):
            if len(cc) == 2:
                return "{}___{}".format(str(cnt).zfill(3), str(cc[1]))
            elif len(cc) == 3:
                return "{}___{}___{}".format(str(cnt).zfill(3), str(cc[1]), str(cc[2]))
        return "000___0"

    matrix, cognate_class_names, assumptions = list(), list(), list()
    make_header = True
    col_num = 0

    for language in languages:
        row = [language['ascii_name']]
        meaningIDsForLanguage = meaningIDsForLanguages[language['ID']]
        for meaning in meanings:
            is_lg_missing_mng = not meaning['ID'] in meaningIDsForLanguage
            if is_lg_missing_mng:
                row.append("?")
            else:
                row.append("0")
            if make_header:
                col_num += 1
                start_range = col_num
                cognate_class_names.append("{}_group".format(meaning['Name']))

            if meaning['ID'] in data:
                cnt = 0  # needed for preserving the cc order after sorting meaning keys in dict
                data_mng_id = data[meaning['ID']]

                # generate sort order for cognate class ids = order by (cladeCount, lexCount)
                cc_sortorder = {}
                for cc0 in data_mng_id.keys():
                    cc_m = '0'
                    if isinstance(cc0, (list, tuple)):
                        cc = str(cc0[1])
                        if len(cc0) == 3:
                            cc_m = '!'
                    else:
                        cc = str(cc0).zfill(6)
                    lexCount = len(data_mng_id[cc0])
                    if lexCount == 1:
                        cc_ = re.sub(r'[A-Z]', '', cc)
                        if cc_m == '!' and cc_ in pllloan_cnt_clades:
                            cc_sortorder["!{:04d}_{:06d}_{}".format(
                                pllloan_cnt_clades[cc_], pllloan_cnt_lexemes[cc_], cc.zfill(6))] = cc0
                        else:
                            cc_sortorder["{}{:04d}_{:06d}_{}".format(cc_m, 1, 1, cc.zfill(6))] = cc0
                    else:
                        clds = set()
                        cc_ = re.sub(r'[A-Z]', '', cc)
                        if cc_m == '!' and cc_ in pllloan_cnt_clades:
                            cc_sortorder["!{:04d}_{:06d}_{}".format(
                                pllloan_cnt_clades[cc_], 1, cc.zfill(6))] = cc0
                        else:
                            for lg in data_mng_id[cc0]:
                                if lg in languageClades:
                                    clds.add(languageClades[lg])
                            cc_sortorder["{:04d}_{:06d}_{}".format(len(clds), lexCount, cc.zfill(6))] = cc0

                for cc0 in sorted(cc_sortorder, reverse=True):
                    cc = cc_sortorder[cc0]
                    if make_header:
                        col_num += 1
                        cognate_class_names.append(
                            cognate_class_name_formatter(cc, meaning['Name']))
                    if is_lg_missing_mng:
                        row.append("?")
                    elif language['ID'] in data_mng_id[cc]:
                        row.append("1")
                    else:
                        row.append("0")
                    cnt += 1
            if make_header:
                end_range = col_num
                assumptions.append(
                    "charset {} = {}-{};".format(meaning['Name'], start_range, end_range))

        matrix.append(row)
        make_header = False

    return matrix, cognate_class_names, assumptions


def getMatrixCommentsFromCognateNames(cognate_class_names):

    meaningRow = []  # Pieces to be joined with ''
    flagRow = ''
    idBucket = []  # Matrix of id chars

    meaning = ''
    meaningLength = 0

    def nextMeaning(m):
        if m != meaning:
            label = (' ' + meaning)[:meaningLength].ljust(meaningLength, ' ')
            meaningRow.append(label)
            return m, 1
        return meaning, meaningLength + 1

    groupRegex = r'^(.+)_group$'
    lexemeRegex = r'^(.+)_lexeme_(.+)$'
    cognateRegex = r'^(.+)_cognate_(.+)$'
    pllloanRegex = r'^(.+)_cognate_(.+?)_pllloanlexeme_(.+)$'

    for name in cognate_class_names:
        groupMatch = re.match(groupRegex, name)
        if groupMatch:
            meaning, meaningLength = nextMeaning(groupMatch.group(1))
            flagRow += ' '
            idBucket.append('|')
            continue
        pllloanMatch = re.match(pllloanRegex, name)
        if pllloanMatch:
            meaning, meaningLength = nextMeaning(pllloanMatch.group(1))
            flagRow += 'P'
            idBucket.append("{}|{}".format(pllloanMatch.group(2), pllloanMatch.group(3)))
            continue
        lexemeMatch = re.match(lexemeRegex, name)
        if lexemeMatch:
            meaning, meaningLength = nextMeaning(lexemeMatch.group(1))
            flagRow += 'L'
            idBucket.append("{}|".format(lexemeMatch.group(2)))
            continue
        cognateMatch = re.match(cognateRegex, name)
        if cognateMatch:
            meaning, meaningLength = nextMeaning(cognateMatch.group(1))
            flagRow += 'C'
            idBucket.append("{}|".format(cognateMatch.group(2)))
            continue
        meaning, meaningLength = nextMeaning('')
        flagRow += '?'
        idBucket.append('')
    nextMeaning('')

    # Create idRows by transposing ids:
    idRows = []
    if len(idBucket) > 0:
        idMaxLenCC = max(*[len(i.split('|')[0]) for i in idBucket])
        idMaxLenLEX = max(*[len(i.split('|')[1]) for i in idBucket])
        idRows = []
        for id in idBucket:
            (ccid, lexid) = id.split('|')
            idRows.append([x for x in "{}|{}".format(
                ccid.rjust(idMaxLenCC, '-'), lexid.ljust(idMaxLenLEX, '-'))])
        idRows = [''.join(row) for row in zip(*idRows)]

    commentRows = []

    def addComment(c):
        commentRows.append(c)

    # add nexus set ids on top of the comment
    nex_ccs = []
    max_len = len(str(len(cognate_class_names)))
    for n in range(1, len(cognate_class_names) + 1):
        nex_ccs.append(str(n).rjust(max_len, '-'))
    for r in [''.join(row) for row in zip(*nex_ccs)]:
        addComment(r)

    addComment(''.join(meaningRow))
    commentRows.append("")
    addComment(flagRow)
    commentRows.append("")
    for idRow in idRows:
        addComment(idRow)

    return commentRows


def getNexusLocations(languages):
    lines = []
    for language in languages:
        if language['Latitude'] is None or language['Longitude'] is None:
            continue
        lines.append("  {} = {} {};".format(
            language['ascii_name'], language['Latitude'], language['Longitude']))
    return "\nbegin locations;\n{}\nend;\n".format("\n".join(lines))


def getCharstateLabels(cognate_class_names):
    labels = []
    for i, cc in enumerate(cognate_class_names):
        labels.append("    {} {}".format(i + 1, cc))
    return "\nbegin charstatelabels;\n{}\nend;\n".format("\n".join(labels))


def cladeMembership(ds_cldf, languages):
    clades = ds_cldf.get('clades.csv')
    taxsets_refcolors = defaultdict(list)
    taxonsetNames = []
    for clade in clades:
        # only clades for export have a taxonsetName ##
        if clade['taxonsetName']:
            level = -1
            if clade['clade_level0'] > 0:
                level += 1
            if clade['clade_level1'] > 0:
                level += 1
            if clade['clade_level2'] > 0:
                level += 1
            if clade['clade_level3'] > 0:
                level += 1
            taxonsetNames.append((clade['taxonsetName'], level, clade['clade_level0'], clade['clade_level1'],
                                  clade['clade_level2'], clade['clade_level3']))
    for ts in taxonsetNames:
        tsname = ts[0]
        level = ts[1]
        for clade in clades:
            if level == 0:
                if clade['clade_level0'] == ts[2]:
                    taxsets_refcolors[tsname].append(clade['color'][1:])
            elif level == 1:
                if clade['clade_level0'] == ts[2] and clade['clade_level1'] == ts[3]:
                    taxsets_refcolors[tsname].append(clade['color'][1:])
            elif level == 2:
                if clade['clade_level0'] == ts[2] and clade['clade_level1'] == ts[3] \
                        and clade['clade_level2'] == ts[4]:
                    taxsets_refcolors[tsname].append(clade['color'][1:])
            else:
                if clade['clade_level0'] == ts[2] and clade['clade_level1'] == ts[3] \
                        and clade['clade_level2'] == ts[4] and clade['clade_level3'] == ts[5]:
                    taxsets_refcolors[tsname].append(clade['color'][1:])
    taxsets = []
    for ts in taxonsetNames:
        lgs = []
        for lg in languages:
            if lg['Color'] in taxsets_refcolors[ts[0]]:
                lgs.append(lg['ascii_name'])
        if len(lgs) >= 1:
            taxsets.append("taxset ts{} = {};".format(ts[0], " ".join(lgs)))

    return "begin sets;\n{}\nend;\n".format("\n".join(taxsets))


def computeCalibrations(ds_cldf, languages):
    def getDistribution(abstractDistribution):

        def yearToFloat(year):
            if year is None:
                return 0.0
            return round(float(year) / 1000, 3)

        # no uniformUpper/uniformLower distribution ##
        if abstractDistribution['distribution'][0] == 'N':
            mean = yearToFloat(abstractDistribution['normalMean'])
            stDev = yearToFloat(abstractDistribution['normalStDev'])
            if mean == 0. and stDev == 0.:
                return None
            return "normal({:.3f},{:.3f})".format(mean, stDev)
        if abstractDistribution['distribution'][0] == 'L':
            mean = yearToFloat(abstractDistribution['logNormalMean'])
            stDev = abstractDistribution['logNormalStDev'] or 0.0
            if mean == 0. and stDev == 0.:
                return None
            return "lognormal({:.3f},{:.3f})".format(mean, stDev)
        if abstractDistribution['distribution'][0] == 'O':
            mean = yearToFloat(abstractDistribution['logNormalMean'])
            stDev = abstractDistribution['logNormalStDev'] or 0.0
            offset = yearToFloat(abstractDistribution['logNormalOffset'])
            if mean == 0. and stDev == 0. and offset == 0.:
                return None
            return "offsetlognormal({:.3f},{:.3f},{:.3f})".format(offset, mean, stDev)
        return None

    calibrations = []
    # clades do not have any calibrations ##
    for language in languages:
        if language['historical']:
            cal = getDistribution(language)
            if cal is not None:
                calibrations.append("calibrate {} = {}".format(language['ascii_name'], cal))

    return "begin assumptions;\n{}\nend;\n".format("\n".join(calibrations))
