/**
 * 
 */
package searching.gui;

import java.awt.Color;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Scanner;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JSpinner;
import javax.swing.JTextField;

import com.toedter.calendar.IDateEditor;
import com.toedter.calendar.JDateChooser;
import com.toedter.calendar.JTextFieldDateEditor;

import searching.params.ComparisonType;
import searching.params.SearchParam;
import utility.Debug;
import utility.Util;

/**
 * This is a custome Component
 * This component represents one entry of the search.
 * It consists of four parts:<br>
 * 1. a dropdown to select the variable to be used for filtering in the database according to certain criteria<br>
 * 2. a dropdown to select the comparison type of the variable<br>
 * 3. a input-component, to specify the search<br>
 * 4. a delete button, to remove this searchfilter<br>
 *  
 * @author Erik
 *
 */
public class SearchParamPanel extends ScalePanel {
	/**
	 * 
	 */
	private static final long serialVersionUID = -9051796379678491596L;
	
	private SearchParam param;
	private WideComboBox<SearchParam> variableChoice;
	private JComponent inputComponent;
	private WideComboBox<ComparisonType> comparisonMethod;
	private JButton deleteButton;
	
	private boolean hasClearedOutPreviewNumberOnce = false;
	private boolean blockVariableChangeListener = true;
	
	/**
	 * Create the panel.
	 */
	public SearchParamPanel(SearchParam startParam) {
		//setLayout(new BorderLayout(0, 0));
		super();
		
		variableChoice = new WideComboBox<SearchParam>(Util.SortArrayAfterToString(SearchParam.values()));		
		
		variableChoice.addItemListener(new ItemListener() {
			
			@Override
			public void itemStateChanged(ItemEvent e) {
				if (e.getStateChange() == ItemEvent.SELECTED) {
					if (blockVariableChangeListener) {
						blockVariableChangeListener = false;
					} else {
						Update((SearchParam)e.getItem());
					}
				}

			}
		});
		
		
		add(variableChoice);
		
		
		SetComponentBounds(variableChoice, 0, 0, 0.25f, 1);
		
		
		deleteButton = new JButton(new ImageIcon(SearchParamPanel.class.getResource("/resources/Thrashcan.png")));
		
		
		deleteButton.addActionListener(new ActionListener() {
			
			@Override
			public void actionPerformed(ActionEvent e) {
				MainFrame.singelton.RemoveSearchParam(SearchParamPanel.this);
			}
		});
		
		add(deleteButton);
		
		deleteButton.setMargin(new Insets(0, 0, 0, 0));
		SetComponentBounds(deleteButton, 0.9f + 0.01f, 0.15f , 0.1f - 0.01f * 2, 0.75f);
		
		//this.setVisible(true);
		Update(startParam);
		//Update(SearchParam.values()[0]);
	}
	
	public SearchParamPanel() {
		this(SearchParam.values()[0]);
	}
	
	/***
	 * Loads a SearchParamPanel From file
	 * @param reader non empty scanner of the file.
	 */
	public SearchParamPanel(Scanner reader) {
		this(SearchParam.valueOf(reader.nextLine()));
		String compMeth = reader.nextLine(), inpArgs = reader.nextLine();
		
		//Update(SearchParam.valueOf(searchParam));
		comparisonMethod.setSelectedItem(ComparisonType.valueOf(compMeth));
		//final String inpArgs = reader.nextLine();
		LoadInputComponentValueFromString(inpArgs);
		
	}
	
	/**
	 * Changes the Variable to filter
	 * @param title name of new filter criteria (must equal one of the SearchParam.GetTitle())
	 */
	public void Update(String title) {
		for(SearchParam s : SearchParam.values()) {
			if (title.equals(s.GetTitle())) {
				Update(s);
				break;
			}
		}
		
	}

	
	/**
	 * Changes the variable to filter to a specific SearchParam
	 * @param param the new SearchParam
	 */
	public void Update(SearchParam param) {
		
		hasClearedOutPreviewNumberOnce = false;
		if (inputComponent != null) {
			remove(inputComponent);
		}
		
		this.param = param;
		
		
		variableChoice.setSelectedItem(param);
		
		if (comparisonMethod != null) remove(comparisonMethod);
		
		comparisonMethod = new WideComboBox<>(Util.SortArrayAfterToString(param.GetComparisonCollection().GetComparisonTypes()));
		add(comparisonMethod);
		SetComponentBounds(comparisonMethod, 0.25f, 0, 0.25f, 1);
		
		
		switch(param.GetInputType()) {
		case STRING:
			final JTextField txtField = new JTextField("enter condition");
			txtField.addFocusListener(new FocusListener() {
				
				@Override
				public void focusLost(FocusEvent e) {
					if (txtField.getText().equals("")) txtField.setText("enter condition");
				}
				
				@Override
				public void focusGained(FocusEvent e) {
					if (txtField.getText().equals("enter condition")) txtField.setText("");
				}
			});
			inputComponent = txtField;
			break;
		case DATE:
			JDateChooser calenderField = new JDateChooser();
			IDateEditor dateEditor = calenderField.getDateEditor();
	        if (dateEditor instanceof JTextFieldDateEditor) {
	            JTextFieldDateEditor txtFld = (JTextFieldDateEditor) dateEditor;
	            //txtFld.setBackground(Color.BLACK);
	            txtFld.addPropertyChangeListener("foreground", event -> {
	                if (Color.BLACK.equals(event.getNewValue())) {
	                    txtFld.setForeground(MainFrame.singelton.GetDarkMode() ? Color.WHITE : Color.BLACK);
	                }
	            });
	        }
			inputComponent = calenderField;
			break;
		case NUMERIC:
			JSpinner spinner = new JSpinner();
			JSpinner.DefaultEditor editor = (JSpinner.DefaultEditor)spinner.getEditor();
			final JTextField spinnerTextField = editor.getTextField();
			spinnerTextField.addFocusListener(new FocusListener() {
				
				@Override
				public void focusLost(FocusEvent e) {
					
				}
				
				@Override
				public void focusGained(FocusEvent e) {
					if (spinnerTextField.getText().equals("0") && !hasClearedOutPreviewNumberOnce) {
						spinnerTextField.setText("");
						hasClearedOutPreviewNumberOnce = true;
					}
				}
			});
			inputComponent = spinner;
			break;
		case GROUP_ID:
			ComboSelections groupIDInputs = new ComboSelections("Select", new String[] {"A01", "A02", "A03", "A04", "A05", "A06", "A07", "A08", "A09", "A10", "A11", "A12", "A13", "A14", "A15", "A16", "A18", "A19", "A21", "F01", "F02"});
			inputComponent = groupIDInputs;
			break;
		case MODALITY:
			ComboSelections modalityTypesInputs = new ComboSelections("Select", new String[] {"Behavorial", "Calcium Imaging", "ECG|Pulse", "EDA", "EEG", "Eyetracking", "Histology", "Hormone measurements", "LFP", "MRI", "Questionaires", "Respiration", "Single cell recording", "TMS"});
			inputComponent = modalityTypesInputs;
			break;
		case RESOURCE_TYPE:
			JComboBox<String> resourceTypesInput = new JComboBox<String>(new String[] {"Analysed", "Measured", "Simulated"});
			inputComponent = resourceTypesInput;
			break;
		case SUBJECT_SPECIES:
			JComboBox<String> speciesTypesInput = new JComboBox<String>(new String[] {"Humans", "Pigeons", "Mice", "Rats", "Crows", "Jackdaws"});
			inputComponent = speciesTypesInput;
			break;
		case SUBJECT_TYPE:
			JComboBox<String> subjectTypesInput = new JComboBox<String>(new String[] {"Healthy test subject", "Patient", "Healthy control subject"});
			inputComponent = subjectTypesInput;
			break;
		case SUBJECT_SEX:
			JComboBox<String> subjectSexInput = new JComboBox<String>(new String[] {"female", "male", "diverse", "undefined"});
			inputComponent = subjectSexInput;
			break;
		default:
			Debug.Error("Unimplemented Searchparam: " + param);
			break;
		}
		
		add(inputComponent);
		SetComponentBounds(inputComponent, 0.5f, 0, 0.4f, 1);
		
		revalidate();
		repaint();
	}
	
	/**
	 * Sets the Value of the input-component
	 * @param input
	 */
	@SuppressWarnings("unchecked")
	public void LoadInputComponentValueFromString(String input) {
		
		switch (param.GetInputType()) {
		case DATE:
			input = input.substring(input.lastIndexOf("datetime(\"") + 10, input.length() - 2);
			try {
				Date inputDate = new SimpleDateFormat("yyyy-MM-dd").parse(input);
				((JDateChooser)inputComponent).setDate(inputDate);
				//((JTextField)((JDateChooser)inputComponent).getDateEditor().getUiComponent()).setText( new SimpleDateFormat(((JDateChooser)inputComponent).getDateFormatString()).format(inputDate) + " AA");
			} catch (ParseException e) {
				Debug.Error(e);
			}
			break;
		case NUMERIC:
			((JSpinner)inputComponent).setValue(Integer.parseInt(input));
			break;
		case STRING:
			((JTextField)inputComponent).setText(input);
			break;
		case MODALITY:
		case GROUP_ID:
			if (input.length() == 0) break;
			String[] checked = input.split(";;");
			((ComboSelections)inputComponent).ToggleEntries(checked);
			
			break;
		default:
			if (inputComponent instanceof JComboBox) {
				((JComboBox<String>)inputComponent).setSelectedItem(input);
			} else {
				Debug.Error("Unhandled Data Type while loading!");
			}
			
			break;
		}
	}
	
	/**
	 * Gets the value of the input-component as string
	 * Keep care (some might be decoded and not for straight forwards use [for ex.: Modality])
	 * @return returns the Value as String
	 */
	@SuppressWarnings("unchecked")
	public String GetInput() {
		ComparisonType compType = comparisonMethod.getItemAt(comparisonMethod.getSelectedIndex());
		
		String input = null;
		switch (param.GetInputType()) {
		case DATE:
			input = new SimpleDateFormat("yyyy-MM-dd").format(((JDateChooser)inputComponent).getDate());
			return "datetime(\"" + param.GetSQLName() +"\") " + compType.GetResultingSQL("datetime(\"" + input + "\")");
					
		case NUMERIC:
			input = ((JSpinner)inputComponent).getValue().toString();
			break;
		case STRING:
			input = ((JTextField)inputComponent).getText();
			break;
		case MODALITY:
			input = "";
			String[] modInp = ((ComboSelections)inputComponent).GetSelectedStrings();
			for(int i = 0; i < modInp.length; i++) {
				input += modInp[i];
				if (i != modInp.length - 1) input += ";;";
			}
			
			
			break;
		case GROUP_ID:
			input = "";
			String[] groupInp = ((ComboSelections)inputComponent).GetSelectedStrings();
			for(int i = 0; i < groupInp.length; i++) {
				input += groupInp[i];
				if (i != groupInp.length - 1) input += ";;";
			}
			
			
			break;
		case RESOURCE_TYPE:
		case SUBJECT_SPECIES:
		case SUBJECT_TYPE:
		case SUBJECT_SEX:
			input = (String)((JComboBox<String>)inputComponent).getSelectedItem();
			break;
		default:
			Debug.Error("Unkown Input Type! [SearchParamLabel::ExtractSQL_Command]");
			break;		
		}
		return input;
	}
	
	/**
	 * creates the restrictive part of a SQL query based on this SearchParamPanel.<br><br>
	 * The bold part will be the result.<br>
	 * SELECT someColumn FROM bla WHERE <b>name Like '%Schmid%'</b>;
	 *  
	 * 
	 * @return sql-where-part as string
	 */
	public String ExtractSQL_Command() {
		ComparisonType compType = comparisonMethod.getItemAt(comparisonMethod.getSelectedIndex());
		//SearchParam searchParam = ((SearchParam)variableChoice.getSelectedItem());
		switch (((SearchParam)variableChoice.getSelectedItem()).GetInputType()) {
		
		case MODALITY:
		case GROUP_ID:
			String[] queryStrings = GetInput().split(";;");
			String query = "(";
			for(String param : queryStrings) {
				query +=  "\"" + this.param.GetSQLName() + "\" LIKE '%" + param + "%'";
				if (param != queryStrings[queryStrings.length - 1]) 
					query += " " + compType.GetResultingSQL_RAW() + " ";
			}
			query += ")";
			return query;
		case DATE:
			return GetInput();
		default:
			return "\"" + param.GetSQLName() + "\" " + compType.GetResultingSQL(GetInput());
		}
	}
	
	@Override
	public String toString() {
		return param.baseToString() + "\n" + ((ComparisonType)comparisonMethod.getSelectedItem()).baseToString() + "\n" +  GetInput() + "\n";
	}
	

}
