/**
 * 
 */
package searching.gui;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Point;
import java.awt.SecondaryLoop;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.Date;
import java.util.LinkedList;
import java.util.Scanner;

import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTable;
import javax.swing.ProgressMonitor;
import javax.swing.ScrollPaneConstants;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;
import javax.swing.border.EmptyBorder;
import javax.swing.filechooser.FileFilter;
import javax.swing.table.DefaultTableModel;

import com.formdev.flatlaf.FlatDarkLaf;
import com.formdev.flatlaf.FlatLightLaf;

import utility.Database;
import utility.DatabaseInformation;
import utility.Debug;
import utility.Util;
import utility.XLSXhelper;

/**
 * This is the main Frame of the Database-Search-Application
 * @author Erik
 *
 */
public class MainFrame extends JFrame {
	public static MainFrame singelton;
	
	public static final String VERSION = "20220517";
	
	
	//private static final String CONFIG_FILE_PATH = "_dataSearch.config";
	private Database database;
	/**
	 * 
	 */
	private static final long serialVersionUID = 4151986736147273678L;
	
	
	private JPanel contentPane;
	private LinkedList<SearchParamPanel> queryLabels = new LinkedList<>();
	private LayoutBox searchParams;
	private JLabel outputTitle;
	private JTable outputTable;
	
	
	private LinkedList<String> resultPaths = new LinkedList<>();
	private JLabel databaseConnectionStatusLabel;
	private JLabel databaseLastChangeLabel;
	private JPanel rightSouthPanel;
	private JButton exitButton;
	
	private String tableCreatingSqlStatement = "";
	private String orderBySqlAddition = "";
	private LinkedList<String> removedPaths = new LinkedList<>();
	
	
	private boolean darkmode = false;
	private boolean fetchResultsLeathalProblemOccured = false;
	
	
	/**
	 * Launch the application.
	 */
	public static void Main(final String[] args) {	
		
		EventQueue.invokeLater(new Runnable() {
			public void run() {
				try {
					singelton = new MainFrame();
					singelton.setVisible(true);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		});
	}

	/**
	 * Create the frame and setup the UI
	 */
	public MainFrame() {
		SetDarkmode(true);
		
		setSize(new Dimension(785, 533));
		
		
		
		setTitle("SFB1280 - Meta Search");
		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		setBounds(100, 100, 800, 600);
		
		setIconImage(new ImageIcon(MainFrame.class.getResource("/resources/Logo_SFB_1280_transparent.png")).getImage());

		
		JMenuBar menuBar = new JMenuBar();
		setJMenuBar(menuBar);
		
		JMenu mnSearch = new JMenu("Search");
		menuBar.add(mnSearch);
		
		JMenuItem mntmSaveSearch = new JMenuItem("Save Search");
		mntmSaveSearch.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				SaveSearchParamter();
				
			}
		});
		mnSearch.add(mntmSaveSearch);
		
		JMenuItem mntmLoadSearch = new JMenuItem("Load Search");
		mntmLoadSearch.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				LoadSearchParameter();
			}
		});
		mnSearch.add(mntmLoadSearch);
		
		JMenuItem mntmAddSearchParameter = new JMenuItem("Add Searchparameter");
		mntmAddSearchParameter.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				AddNewSearchParam(new SearchParamPanel());
			}
		});
		mnSearch.add(mntmAddSearchParameter);
		
		JMenuItem mntmRemoveSearchparameter = new JMenuItem("Remove Searchparameter");
		mntmRemoveSearchparameter.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				RemoveLastSearchParam();
			}
		});
		mnSearch.add(mntmRemoveSearchparameter);
		
		JMenuItem mntmExecuteSearch = new JMenuItem("Execute Search");
		mntmExecuteSearch.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				Search();
			}
		});
		mnSearch.add(mntmExecuteSearch);
		
		JMenu mnExport = new JMenu("Export");
		menuBar.add(mnExport);
		
		JMenuItem mntmFetchDatasets = new JMenuItem("Fetch Datasets");
		mntmFetchDatasets.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				FetchResults();
			}
		});
		mnExport.add(mntmFetchDatasets);
		
		JMenuItem mntmExportAsCsv = new JMenuItem("Export as CSV");
		mntmExportAsCsv.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				SaveAsCSV();
			}
		});
		mnExport.add(mntmExportAsCsv);

		JMenuItem mntmExportAsXSLS = new JMenuItem("Export as XSLS");
		mntmExportAsXSLS.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				SaveAsExcel();
			}
		});
		mnExport.add(mntmExportAsXSLS );

		JMenu mnDatabase = new JMenu("Database");
		menuBar.add(mnDatabase);
		
		JMenuItem mntmConnectDatabase = new JMenuItem("Connect Database");
		mntmConnectDatabase.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				String path = ChooseDatabase();
				if (path == null) return;
				ConnectDatabase(path);
			}
		});
		mnDatabase.add(mntmConnectDatabase);
		
		JMenu mnWindow = new JMenu("Window");
		menuBar.add(mnWindow);
		
		JCheckBoxMenuItem chckbxmntmDarkmode = new JCheckBoxMenuItem("Darkmode");
		chckbxmntmDarkmode.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				SetDarkmode(!darkmode);
			}
		});
		chckbxmntmDarkmode.setSelected(true);
		mnWindow.add(chckbxmntmDarkmode);
		contentPane = new JPanel();
		contentPane.setBorder(new EmptyBorder(5, 5, 5, 5));
		setContentPane(contentPane);
		contentPane.setLayout(new BorderLayout(0, 0));
		
		// SplitPane
		
		JSplitPane splitPane = new JSplitPane();
		splitPane.addPropertyChangeListener(JSplitPane.DIVIDER_LOCATION_PROPERTY, new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent e) {
				MyRepaint(searchParams);
			}
		});
		splitPane.setResizeWeight(1.0);
		contentPane.add(splitPane, BorderLayout.CENTER);
		
		JPanel rightPanel = new JPanel();
		rightPanel.setMaximumSize(new Dimension(300, 32767));
		splitPane.setRightComponent(rightPanel);
		rightPanel.setLayout(new BorderLayout(0, 0));
		
		//Box verticalBox = Box.createVerticalBox();
		//output.add(verticalBox);
		
		outputTitle = new JLabel("Results: ");
		outputTitle.setHorizontalAlignment(SwingConstants.CENTER);
		outputTitle.setAlignmentX(CENTER_ALIGNMENT);
		rightPanel.add(outputTitle, BorderLayout.NORTH);
		
		JScrollPane searchOutputScrollPane = new JScrollPane(JScrollPane.VERTICAL_SCROLLBAR_AS_NEEDED, JScrollPane.HORIZONTAL_SCROLLBAR_AS_NEEDED);
		
		rightPanel.add(searchOutputScrollPane);
		
		outputTable = new JTable();
		outputTable.setModel(new DefaultTableModel(
			new Object[][] {
			},
			new String[] {
				"Subject ID", "Modality", "Group", "Datatype", "Experiment Title", "Delete"
			}
		) {
			boolean[] columnEditables = new boolean[] {
				false, false, false, false, false, false
			};
			public boolean isCellEditable(int row, int column) {
				return columnEditables[column];
			}
		});
		outputTable.getColumnModel().getColumn(5).setPreferredWidth(60);
		
		outputTable.getTableHeader().addMouseListener(new MouseAdapter() { 
			@Override
			public void mouseClicked(MouseEvent event) {
				if (tableCreatingSqlStatement == null || tableCreatingSqlStatement.length() == 0) return;
				
				Point point = event.getPoint();
				int column = outputTable.columnAtPoint(point);
				
				if (orderBySqlAddition.length() == 0) {
					orderBySqlAddition = "ORDER BY " + ColumnIndexToSqlColumnNameString(column) + " ASC";
				} else if (orderBySqlAddition.endsWith("ASC")) {
					orderBySqlAddition = "ORDER BY " + ColumnIndexToSqlColumnNameString(column) + " DESC";
				} else {
					orderBySqlAddition = "";
				}
				
				String statement = tableCreatingSqlStatement.substring(0, tableCreatingSqlStatement.length() - 1) + " " + orderBySqlAddition + ";";
				
				Search(statement);
				
			}
			
			private String ColumnIndexToSqlColumnNameString(int index) {// #JHeaderSort
				switch(index) {
				case 0:
					return "\"Subject ID\"";
				case 1:
					return "\"Modality\"";
				case 2:
					return "\"Shared with\"";
				case 3:
					return "\"FileNames\"";
				case 4:
					return "\"Experiment title\"";
				default:
					Debug.Error("Unkown Column index at #JHeaderSort");
					return null;
					
				}
			}
			
		});
		
		outputTable.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent evt) {
				int row = outputTable.rowAtPoint(evt.getPoint());
		        int col = outputTable.columnAtPoint(evt.getPoint());
		        //#RmvJTable
		        Debug.Log("[#RmvJTable] Remove: Row|Col " + row + " | " + col);
		        
		        if (row >= 0 && col == 5) {
		        	RemoveTableRow(row);
		        }
			}
			
			private void RemoveTableRow(int index) {
				int i = -1;
				String rmvPath = null;
				for(String path : resultPaths) {
					if (removedPaths.contains(path)) {
						continue;
					} else {
						i++;
						if (i == index) rmvPath = path;
					}
				}
				
				String resultLableTxt = outputTitle.getText();
				if (!resultLableTxt.equals("No Results")) {
					int amount = Integer.parseInt(resultLableTxt.substring(resultLableTxt.indexOf("[") + 1, resultLableTxt.indexOf("]")));
					amount--;
					if (amount <= 0) {
						outputTitle.setText("No Results");
						outputTitle.setForeground(Color.red);
					} else {
						outputTitle.setText("Results [" + amount + "]");
						outputTitle.setForeground(Color.LIGHT_GRAY);
					}
				}
				
				removedPaths.add(rmvPath);
				((DefaultTableModel)outputTable.getModel()).removeRow(index);
			}
		});
		
		searchOutputScrollPane.setViewportView(outputTable);
		
		rightSouthPanel = new JPanel();
		rightPanel.add(rightSouthPanel, BorderLayout.SOUTH);
		rightSouthPanel.setLayout(new BorderLayout(0, 0));
		
		exitButton = new JButton("Exit");
		exitButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				int closeDialog = JOptionPane.showConfirmDialog(null, "Exiting the program might discard unsafed changes, continue?", "Exit?", JOptionPane.YES_NO_OPTION);
				if (closeDialog == JOptionPane.OK_OPTION) System.exit(0);
			}
		});
		rightSouthPanel.add(exitButton, BorderLayout.EAST);
		
		JPanel rightSouthCenterInformationPanel = new JPanel();
		rightSouthPanel.add(rightSouthCenterInformationPanel, BorderLayout.CENTER);
		rightSouthCenterInformationPanel.setLayout(new BorderLayout(0, 0));
		
		databaseConnectionStatusLabel = new JLabel("Database: //C:/Cool//Path//Here//Which//Could//Be//Long.db");
		databaseConnectionStatusLabel.setHorizontalAlignment(SwingConstants.LEFT);
		databaseConnectionStatusLabel.addComponentListener(new ComponentAdapter() {
			@Override
			public void  componentResized(ComponentEvent e) {
				FitDatabasePathInLabel(databaseConnectionStatusLabel);
				databaseConnectionStatusLabel.setMinimumSize(new Dimension(0, 0));
			}
		});
		rightSouthCenterInformationPanel.add(databaseConnectionStatusLabel, BorderLayout.NORTH);
		
		
		
		databaseLastChangeLabel = new JLabel("Last Change: 99.99.2021");
		rightSouthCenterInformationPanel.add(databaseLastChangeLabel, BorderLayout.CENTER);
		
		JLabel lblVersion = new JLabel("Version " + VERSION);
		rightSouthCenterInformationPanel.add(lblVersion, BorderLayout.SOUTH);
		
		
		
		InitSearchParams();
		
		// Left Panel
		JPanel leftPanel = new JPanel();
		leftPanel.setLayout(new BorderLayout());
		splitPane.setLeftComponent(leftPanel);
		
		// Scroll Pane
		
		JScrollPane searchParamsScrollPane = new JScrollPane(searchParams, ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS, ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);
		leftPanel.add(searchParamsScrollPane, BorderLayout.CENTER);
		//splitPane.setLeftComponent(searchParamsScrollPane); //DEPRECATED
		
		// LeftSideBottom Half Panel
		
		JPanel leftSideBottomHalf = new JPanel();
		leftSideBottomHalf.setLayout(new BorderLayout());
		leftPanel.add(leftSideBottomHalf, BorderLayout.SOUTH);
		
		JPanel bottomSideTrippleSplitPanel = new JPanel();
		leftSideBottomHalf.add(bottomSideTrippleSplitPanel, BorderLayout.CENTER);
		bottomSideTrippleSplitPanel.setLayout(new BorderLayout(0, 0));
		
		JPanel searchPanel = new JPanel();
		bottomSideTrippleSplitPanel.add(searchPanel, BorderLayout.NORTH);
		searchPanel.setLayout(new BorderLayout(0, 0));
		
		// Search Controll Panel
		
		JPanel searchControllPanel = new JPanel();
		FlowLayout flowLayout = (FlowLayout) searchControllPanel.getLayout();
		flowLayout.setAlignment(FlowLayout.LEFT);
		searchPanel.add(searchControllPanel, BorderLayout.CENTER);
		
		JButton addNewSearchParam = new JButton("+");
		addNewSearchParam.setHorizontalAlignment(SwingConstants.LEFT);
		addNewSearchParam.setToolTipText("Add another search parameter");
		searchControllPanel.add(addNewSearchParam);
		
		JButton searchButton = new JButton("Search");
		searchButton.setToolTipText("Search with given search parameters.");
		searchButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				
				Search();
			}
		});
		
		JButton removeSearchParam = new JButton("-");
		removeSearchParam.setToolTipText("Remove the last search parameter");
		removeSearchParam.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				RemoveLastSearchParam();
			}
		});
		searchControllPanel.add(removeSearchParam);
		searchControllPanel.add(searchButton);
		
		JLabel searchText = new JLabel("Add or remove conditions for search");
		searchText.setFont(new Font("Tahoma", Font.PLAIN, 13));
		searchPanel.add(searchText, BorderLayout.NORTH);
		
		JPanel outerSearchSafePanel = new JPanel();
		bottomSideTrippleSplitPanel.add(outerSearchSafePanel, BorderLayout.CENTER);
		outerSearchSafePanel.setLayout(new BorderLayout(0, 0));
		
		JPanel searchSafePanel = new JPanel();
		FlowLayout flowLayout_2 = (FlowLayout) searchSafePanel.getLayout();
		flowLayout_2.setAlignment(FlowLayout.LEFT);
		outerSearchSafePanel.add(searchSafePanel, BorderLayout.CENTER);
		
		JButton saveSearchButton = new JButton("Save Search");
		saveSearchButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				SaveSearchParamter();
			}
		});
		saveSearchButton.setToolTipText("Save the current set of searchparameters to a file for later use.");
		searchSafePanel.add(saveSearchButton);
		
		JButton loadSearchButton = new JButton("Load Search");
		loadSearchButton.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				LoadSearchParameter();
			}
		});
		loadSearchButton.setToolTipText("Loads a set of Searchparameters from a .search file.");
		searchSafePanel.add(loadSearchButton);
		
		JLabel searchSafePanelText = new JLabel("Save or load the current searchparameters");
		searchSafePanelText.setFont(new Font("Tahoma", Font.PLAIN, 13));
		outerSearchSafePanel.add(searchSafePanelText, BorderLayout.NORTH);
		
		JPanel fetchDataPanel = new JPanel();
		bottomSideTrippleSplitPanel.add(fetchDataPanel, BorderLayout.SOUTH);
		fetchDataPanel.setLayout(new BorderLayout(0, 0));
		
		
		// Export Panel
		
		JPanel outputControlPanel = new JPanel();
		fetchDataPanel.add(outputControlPanel, BorderLayout.CENTER);
		FlowLayout flowLayout_1 = (FlowLayout) outputControlPanel.getLayout();
		flowLayout_1.setAlignment(FlowLayout.LEFT);
		outputControlPanel.setAlignmentY(Component.BOTTOM_ALIGNMENT);
		
		JButton copyOutputToClipboard = new JButton("Fetch Data");
		copyOutputToClipboard.setToolTipText("Copies all resulting datasets (not only the meta data) to your local system.");
		copyOutputToClipboard.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				FetchResults();
			}
		});
		copyOutputToClipboard.setAlignmentY(Component.BOTTOM_ALIGNMENT);
		outputControlPanel.add(copyOutputToClipboard);
		
		JButton exportOutputToFile = new JButton("Export for Excel");
		exportOutputToFile.setToolTipText("Exports the result into an microsoft Excel format");
		exportOutputToFile.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				//SaveAsCSV();
				SaveAsExcel();
			}
		});
		outputControlPanel.add(exportOutputToFile);
		
		JLabel fetchDataPanelText = new JLabel("Copy data to local hard drive or save results as csv");
		fetchDataPanel.add(fetchDataPanelText, BorderLayout.NORTH);
		fetchDataPanelText.setFont(new Font("Tahoma", Font.PLAIN, 13));
		

		addNewSearchParam.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent arg0) {
				AddNewSearchParam(new SearchParamPanel());
				
			}
		});
		
		
		//---------------------------- CODE ---------------------------------
		TryConnectDatabase();
		AddNewSearchParam(new SearchParamPanel());
		//TODO Sinnvollerers Searchparameter
		
	}
	
	/**
	 * Sets the Darkmode
	 * 
	 * @param darkmode true for a dark theme, false for a ligher theme. (Dark is the defaut theme)
	 */
	
	public void SetDarkmode(boolean darkmode) {
		/** All look&Feel
		 *  javax.swing.plaf.metal.MetalLookAndFeel
			com.sun.java.swing.plaf.motif.MotifLookAndFeel
			com.sun.java.swing.plaf.windows.WindowsLookAndFeel
			com.sun.java.swing.plaf.windows.WindowsClassicLookAndFeel
		 */
		this.darkmode = darkmode;
		
		try {
			if (this.darkmode)
				UIManager.setLookAndFeel(new FlatDarkLaf());
			else
				UIManager.setLookAndFeel(new FlatLightLaf());
			
			SwingUtilities.updateComponentTreeUI(this);
		} catch (UnsupportedLookAndFeelException e1) {
			Debug.Error(e1);
		}
	}
	
	/**
	 * Retruns whether the application is currently in darkmode
	 * @return
	 */
	public boolean GetDarkMode() {
		return darkmode;
	}
	
	/**
	 * initalises the SeachParam-LayoutBox;
	 */
	private void InitSearchParams() {
		searchParams = new LayoutBox(false, 40, 3);
	}
	
	/**
	 * Returns whether a Database is connected or not
	 * @return whether a Database is connected
	 */
	public boolean IsDatabaseConnected() {
		return database != null;
	}
	
	/**
	 * Tries to connect to a Database.
	 * Firstly tries to connect to any SFB Database on the local System in subdirectories of the Project (Depth 4)
	 * After that it tries to the Database on the SFB server
	 * Lastly asks the user to choose a Database to connect with.
	 */
	private void TryConnectDatabase() {
		
		String databasePath = null;
		
//		File config = new File(CONFIG_FILE_PATH);
//		
//		
//		if (config.exists()) {
//			Scanner sc;
//			try {
//				sc = new Scanner(config);
//				databasePath = sc.nextLine();
//				sc.close();
//				if (!Files.exists(Paths.get(databasePath))) 
//					databasePath = null;
//				
//			} catch (FileNotFoundException e) {
//				databasePath = null;
//				Debug.Log("No .config file found");
//			}
//		}
		
		databasePath = SearchForDatabaseOnLocalSystem(4);
		
		if (databasePath == null)
			databasePath = SearchForDatabaseOnServer();
		
		if (databasePath == null) 
			databasePath = ChooseDatabase();
		
		
		ConnectDatabase(databasePath);
		
	}
	
	/**
	 * Search for the Database on the local System via Breath-First-Search (while considering a maximum Depth)
	 * @param maxSearchDepth
	 * @return returns the path to the Database, when found, otherwise null
	 */
	private String SearchForDatabaseOnLocalSystem(int maxSearchDepth) {
		return _SearchForDatabaseOnLocalSystem(new File("."), 0, maxSearchDepth);
	}
	
	/**
	 * Recusive method of SearchForDatabaseOnLocalSystem.
	 * (Internal usage only)
	 * @param currentDir
	 * @param depth
	 * @param maxDepth
	 * @return
	 */
	private String _SearchForDatabaseOnLocalSystem(File currentDir, int depth, int maxDepth) {
		
		File databaseTest = new File(currentDir.getAbsolutePath() + "\\" + DatabaseInformation.DATABASE_NAME  + ".db");
		if (databaseTest.exists())
			return databaseTest.getAbsolutePath();
		
		File databaseTestZip = new File(currentDir.getAbsolutePath() + "\\" + DatabaseInformation.DATABASE_NAME  + ".zip");
		if (databaseTestZip.exists()) {
			Util.UnzipSingleFile(databaseTestZip, databaseTest, true);
			return databaseTest.getAbsolutePath();
		}
		
		if (depth >= maxDepth) return null;
		
		for(File subDir : currentDir.listFiles(new java.io.FileFilter() {
			@Override
			public boolean accept(File file) {
				return file.isDirectory();
			}
		})) {
			String result = _SearchForDatabaseOnLocalSystem(subDir, depth + 1, maxDepth);
			if (result != null)
				return result;
		}
		return null;
	}
	
	/**
	 * Tries to Connect to the Database on the SFB Server
	 * @return the String of the database-location if the connection was successful, returns null otherwise
	 */
	private String SearchForDatabaseOnServer() {
		File serverDatabase = new File(DatabaseInformation.DATABASE_SERVER_STATIC_PATH);
		if (serverDatabase.exists())
			return DatabaseInformation.DATABASE_SERVER_STATIC_PATH;
		File serverDatabaseZip = new File(DatabaseInformation.DATABASE_SERVER_STATIC_PATH_ZIP);
		if (serverDatabaseZip.exists()) {
			Util.UnzipSingleFile(serverDatabaseZip, serverDatabase, true);
			return DatabaseInformation.DATABASE_SERVER_STATIC_PATH;
		}
		return null;
	
	}
	
	/**
	 * Connects the Application to a Database, given by a path.
	 * This method is essential for any further operations on the database.
	 * @param databasePath path to the database file.
	 */
	private void ConnectDatabase(String databasePath) {
		if (databasePath != null) {
			database = new Database(new File(databasePath));
			
			final int maxStringLen = 35;
			String cuttedDatabaseString = databasePath;
			if (cuttedDatabaseString.length() > maxStringLen) {
				cuttedDatabaseString = "..." + cuttedDatabaseString.substring(cuttedDatabaseString.length() - (maxStringLen - 3)); 
			}
			
			databaseConnectionStatusLabel.setText("Database: " + cuttedDatabaseString);
			databaseConnectionStatusLabel.setToolTipText(database.GetDatabaseFile().getAbsolutePath());
			
			try {
				String date = database.execute("SELECT Date FROM " + DatabaseInformation.LAST_MODIFIED_TABLE_NAME + ";").getString(1);
				databaseLastChangeLabel.setText("Last Change: " + date);
				
				SimpleDateFormat f = new SimpleDateFormat("yyyy-MM-dd HH:mm");
				Date d = f.parse(date);
				if (d.getTime() + 1000l * 60l * 60l * 24l * 30l < System.currentTimeMillis()) {
					JOptionPane.showMessageDialog(null, "The Database you use is older than a month. Consider using a more recent one.");
				}
				
			} catch (SQLException | ParseException e) {
				Debug.Error(e);
				Debug.Error("Unable to extract Last-Modified-Date");
			}
			
			
		} else {
			databaseConnectionStatusLabel.setText("Database: NOT CONNECTED");
			databaseLastChangeLabel.setText("Last Change: NOT CONNECTED");
		}
		
		rightSouthPanel.validate();
		rightSouthPanel.repaint();
	}
	
	/**
	 * Asks the user to choose a sql-Database on from the local System.
	 * @return if choosen, then the path of the database is returned, false otherwise
	 */
	private String ChooseDatabase() {
		JFileChooser databaseChooser = new JFileChooser();
		databaseChooser.setFileFilter(new FileFilter() {
			
			@Override
			public String getDescription() {
				return ".db (Database) or .zip (Zipped Database)";
			}
			
			@Override
			public boolean accept(File f) {
				return f.getName().endsWith(".db") || f.getName().endsWith(".zip") || f.isDirectory();
			}
		});
		databaseChooser.setDialogTitle("Select Database");
		int result = databaseChooser.showDialog(null, "Select");
		
		if (result != JFileChooser.APPROVE_OPTION) // If not accepted was pressed
			return null;
		
		String fileName = databaseChooser.getSelectedFile().getAbsolutePath();
		
		if (fileName.endsWith(".zip")) {
			Util.UnzipSingleFile(new File(fileName), new File(fileName.replace("zip", ".db")), true);
			fileName = fileName.replace("zip", ".db");
		}
//		File config = new File(CONFIG_FILE_PATH);
//		if (config.exists())
//			config.delete();
//		
//		FileWriter writer = null;
//		
//		try {
//			config.createNewFile();
//			writer = new FileWriter(config);
//			writer.append(fileName);
//			writer.close();
//		} catch (IOException e) {
//			Debug.Error(e);
//			Debug.Error("Cannot Create Configfile!");
//			
//			if (writer != null) // Cleanup writer if needed
//				try {
//					writer.close();
//				} catch (IOException e1) {
//					Debug.Error(e1);
//				}
//		}
		
//		if (fileName != null && fileName.contains("CLASSIFIED PATH OF SFB SHARE")) {
//			fileName = null;
//			JOptionPane.showMessageDialog(null, "Currently you are not allowed to use the database on the share!");
//		}
		
//		
		return fileName;
	}
	
	/**
	 * Puts the Database path fittingly into a Label
	 * @param comp the lable, holding the database path
	 */
	private void FitDatabasePathInLabel(JLabel comp) {
		if (database == null) return;
		int maxWidthPxl = exitButton.getX() - comp.getX();
		String path = "", fullPath = database.GetDatabaseFile().getAbsolutePath();
		int iteration = 1;
		
		while(comp.getGraphics().getFontMetrics().stringWidth(path) < maxWidthPxl && iteration <= fullPath.length()) {
			
			path = fullPath.substring(fullPath.length() - iteration, fullPath.length());
			iteration++;
		}
		if (iteration <= fullPath.length()) {
			path = "..." + path.substring(3);
		}
		comp.setText(path);
	}
	
	/**
	 * Saves the current Searchfilters to a file.
	 * @param file file to save the search
	 */
	private void SaveSearchParams(File file) {
		if (file.exists())file.delete();
		try {
			file.createNewFile();
			FileWriter writer = new FileWriter(file);
			
			for(SearchParamPanel searchParam : queryLabels) {
				writer.append(searchParam.toString());
			}
			
			writer.close();
			JOptionPane.showMessageDialog(null, "Search successfully saved");
		} catch (IOException e) {
			Debug.Error(e);
			JOptionPane.showMessageDialog(null, "[FATAL] failed to save search");
		}
	}
	
	/**
	 * Loads a Searchfilter from a file
	 * @param file to load from
	 */
	private void LoadSearchParams(File file) {
		
		
		try {
			Scanner sc = new Scanner(file);
			
			while(queryLabels.size() > 0) {
				RemoveLastSearchParam();
			}
			
			while(sc.hasNextLine()) {
				SearchParamPanel newParam = new SearchParamPanel(sc);
				AddNewSearchParam(newParam);
			}
			
			
			sc.close();
		} catch (FileNotFoundException e) {
			Debug.Error(e);
		}
		
		searchParams.revalidate();
		searchParams.repaint();
		MyRepaint(searchParams);
	}
	
	/**
	 * Removes the last of the SearchParamPanels
	 */
	public void RemoveLastSearchParam() {
		
		if (searchParams.getComponentCount() > 0)
			RemoveSearchParam(searchParams.getComponent(searchParams.getComponentCount() - 1));
	}
	
	/**
	 * Removes a specific SearchParamPanel
	 * @param remove the SearchParamPanel to remove
	 */
	public void RemoveSearchParam(Component remove) {
		if(searchParams.getComponentCount() > 0) {
			
			searchParams.remove(remove);
			queryLabels.remove(remove);
			
			searchParams.revalidate();
			searchParams.repaint();
		}
	}
	
	/**
	 * Adds a new SearchParam to the query list
	 * @param newLabel newly Created Searchparam
	 */
	public void AddNewSearchParam(SearchParamPanel newLabel) {
		newLabel.setVisible(true);
		searchParams.add(newLabel);
		//searchParams.SetComponentBounds(newLabel, 0, 0, 1f, 0.1f);
		
		queryLabels.add(newLabel);
		MyRepaint(searchParams);
		//searchParams.paintImmediately(searchParams.getBounds());
	}
	
	/**
	 * Java awt automatic repaint and revalidate are poorly. So Ive created a hacky way to enforce the complete revalidate and rerender of a specific component 
	 * @param comp component to render.
	 */
	public static void MyRepaint(Component comp) {
		Toolkit.getDefaultToolkit().getSystemEventQueue().postEvent(new ComponentEvent(comp, ComponentEvent.COMPONENT_RESIZED));
	}
	
	/**
	 * Downloads the Searchresults to the local drive.
	 */
	public void FetchResults() {
		if (resultPaths.size() == 0) return;
		
		JFileChooser chooseFile = new JFileChooser();
		chooseFile.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		
		int result = chooseFile.showSaveDialog(null);
		
		if(result == JFileChooser.APPROVE_OPTION) {
			final File destinationDir = chooseFile.getSelectedFile();
			final LinkedList<String> copiedPaths = new LinkedList<>();
			
			final ProgressMonitor progressMonitor = new ProgressMonitor(null, "Copy data", "", 0, resultPaths.size() + 1);
			progressMonitor.setMillisToDecideToPopup(0);
			progressMonitor.setMillisToPopup(0);
			
			
			final SecondaryLoop loop = Toolkit.getDefaultToolkit().getSystemEventQueue().createSecondaryLoop();
			final Boolean problemOccured = true;
			
			Thread copyThread = new Thread(new Runnable() {
				
				@Override
				public void run() {
					fetchResultsLeathalProblemOccured = false;
					int i = 1;
					progressMonitor.setProgress(1);
					
					for(String fileName : resultPaths) {
						if (removedPaths.contains(fileName)) continue;
						
						File copyFile = new File(fileName);
						
						//Debug.Log(destinationDir.getAbsolutePath() + "\\" + Util.GetCopiedPathExtension(copyFile));
						
						String newDir = destinationDir.getAbsolutePath() + "\\" + Util.GetCopiedPathExtension(copyFile);
						copiedPaths.add(newDir);
						
						
						if (!Util.CopyDirectory(copyFile.getParent(), newDir)) {
							fetchResultsLeathalProblemOccured = true;
							break;
						}
						i++;
						
						Debug.Log(i + "/" + resultPaths.size());
						
						progressMonitor.setProgress(i);
						if (progressMonitor.isCanceled()) {
							Debug.Log("CANCELED");
							break;
						}
						
					}
					
					if (!progressMonitor.isCanceled())
						progressMonitor.setProgress(progressMonitor.getMaximum());
					
					loop.exit();
				}
			});
			
			copyThread.start();
			loop.enter();
			
			
			
			if (progressMonitor.isCanceled()) {
				int res = JOptionPane.showConfirmDialog(null, "Data transfer canceled. Dispose existing copies? ", "Dispose Copies", JOptionPane.YES_NO_OPTION);
				
				final ProgressMonitor deleteMonitor = new ProgressMonitor(null, "Disposing data", "", 0, copiedPaths.size() + 1);
				deleteMonitor.setMillisToDecideToPopup(0);
				deleteMonitor.setMillisToPopup(500);
				
				
				if (res == JOptionPane.YES_OPTION) {
					
					final SecondaryLoop deleteLoop = Toolkit.getDefaultToolkit().getSystemEventQueue().createSecondaryLoop();
					Thread deleteThread = new Thread(new Runnable() {
						
						@Override
						public void run() {
							int i = 1;
							deleteMonitor.setProgress(1);
							
							for(String dir : copiedPaths) {
								Util.DeleteDirectory(new File(dir));
								deleteMonitor.setProgress(++i);
								if (deleteMonitor.isCanceled())break;
							}
							deleteLoop.exit();
						}
					});
					
					deleteThread.start();
					deleteLoop.enter();
					
					
				}
			
			} else {
				if (!fetchResultsLeathalProblemOccured)
					JOptionPane.showMessageDialog(null, "Data successfully copied!");
			}
			
		}
	}

	/**
	 * Saves the current Search where the where the user wants.
	 */
	private void SaveSearchParamter() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {
			
			@Override
			public String getDescription() {
				return ".search";
			}
			
			@Override
			public boolean accept(File f) {
				return f.isDirectory() || f.getName().contains(".search");
			}
		});
		
		int result = fileChooser.showSaveDialog(null);
		
		if (result == JFileChooser.APPROVE_OPTION) {
			String path = fileChooser.getSelectedFile().getAbsolutePath();
			if(!path.contains(".search")) path += ".search";
			SaveSearchParams(new File(path));
		}
	}
	
	/**
	 * Loads a user-selected Search
	 */
	private void LoadSearchParameter() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileFilter(new FileFilter() {
			
			@Override
			public String getDescription() {
				return ".search";
			}
			
			@Override
			public boolean accept(File f) {
				return f.isDirectory() || f.getName().contains(".search");
			}
		});
		
		int result = fileChooser.showDialog(null, "Select Search to load");
		
		if (result == JFileChooser.APPROVE_OPTION) {
			LoadSearchParams(fileChooser.getSelectedFile());
		}
	}
	
	/**
	 * Searches the Database with the set filters
	 */
	private void Search() {
		//if (queryLabels.isEmpty()) return;
		
		StringBuilder statement = new StringBuilder("SELECT DISTINCT Path, \"Subject ID\", \"Modality\", \"Group ID\", \"FileNames\", \"Experiment title\" FROM data WHERE "); //, \"DataCite-Format\"
		for(SearchParamPanel param : queryLabels) {
			statement.append(param.ExtractSQL_Command());
			if (param != queryLabels.getLast()) statement.append(" and ");
		}
		statement.append(";");
		
		
		
		if (queryLabels.isEmpty()) statement = statement.delete(statement.length() - 7, statement.length() - 1); // Removes The "WHERE "
		
		tableCreatingSqlStatement = statement.toString();
		orderBySqlAddition = "";
		
		removedPaths.clear();
		
		Search(statement.toString());
		
	}
	
	public void Search(String statement) {
		Debug.Log(statement);
		
		resultPaths.clear();
		
		try {
			ResultSet searchResultSet = database.execute(statement);
			//searchResultSet.
			
			DefaultTableModel outputTableModel  = (DefaultTableModel) outputTable.getModel();
			outputTableModel.setRowCount(0);
			
			if (!searchResultSet.isClosed()) {

				searchResultSet.next();
				int i = 0;
				do {
					if (removedPaths.contains(searchResultSet.getString(1))) continue; // If the currnet entry was manually removed it will not show up
					
					resultPaths.add(searchResultSet.getString(1));
					Debug.Log(searchResultSet.getString(1));
					outputTableModel.addRow(new Object[] {
							searchResultSet.getString(2),
							searchResultSet.getString(3).replace("[", "").replace("]", "").replace("\"", ""),
							searchResultSet.getString(4),
							Util.Trim(searchResultSet.getString(5).replace(".",  "").replace(" ", "").replace(";", ", ").replace("exe", "").replace("jar", ""), ' ', ','),
							searchResultSet.getString(6),
							
							
							"X"
							
							//TODO GO ON BERE
						});
					//Debug.Log(set.getString(2));
					
					i++;
				} while(searchResultSet.next());
				
				i--;
				
				outputTitle.setText("Results: [" + (i + 1) + "]");
				outputTitle.setForeground(Color.LIGHT_GRAY);
			} else {
				outputTitle.setText("No Results");
				outputTitle.setForeground(Color.red);
			}
			
			
		} catch (SQLException e) {
			Debug.Error(e);
			Debug.Error("False Statement: " + statement);
			//searchOutput.setText("An error occured. But it lead to no major problems.");
		}
	}

	private void SaveAsExcel(){
		JFileChooser chooseFile = new JFileChooser();
		//chooseFile.setFileSelectionMode(JFileChooser.FILES_ONLY);

		chooseFile.setFileFilter(new FileFilter() {

			@Override
			public String getDescription() {
				return ".xlsx";
			}

			@Override
			public boolean accept(File f) {
				return f.isDirectory() || f.getName().contains(".xlsx");
			}
		});

		int result = chooseFile.showSaveDialog(null);

		if (result == JFileChooser.APPROVE_OPTION) {
			File excelFile = chooseFile.getSelectedFile();
			if (!excelFile.getAbsolutePath().endsWith(".xlsx"))
				excelFile = new File(excelFile.getAbsolutePath() + ".xlsx");
			SaveToExcelWithXLSXhelper(excelFile);
			
		}
	}
	private void SaveToExcelWithXLSXhelper(File excelFile) {
		LinkedList<Object[]> data = new LinkedList<Object[]>();
		data.add(new Object[] {"Subject ID", "Group ID", "Experiment title", "Creator", "Contributor", "Record date", "Resource Type", "Modality", "Datatype", "Shared with", "Experiment Description", "Subject Species", "Subject type", "Subject sex", "Subject age", "Animal|Ethics approval No.", "Extra information", "Path"});

		
		String searchStatement = "SELECT \"Subject ID\", \"Group ID\", \"Experiment title\", \"Creator\", \"Contributor\", \"Record date\", \"Resource Type\", \"Modality\", \"FileNames\", \"Shared with\", \"Experiment Description\", \"Subject Species\", \"Subject type\", \"Subject sex\", \"Subject age\", \"Animal|Ethics approval No.\", \"Extra information\", \"Path\""
				+ tableCreatingSqlStatement.substring(tableCreatingSqlStatement.indexOf(" FROM"));
		searchStatement = searchStatement.substring(0, searchStatement.length() - 1) + " " + orderBySqlAddition + ";";

		ResultSet resultSet = database.executeSafe(searchStatement);

		try {
			while(resultSet.next()) {
				if (removedPaths.contains(resultSet.getString(18))) continue; // if table row was manually removed;
				Object[] excelEntry = new Object[] {
						resultSet.getString(1), // Subject ID
						resultSet.getString(2),
						resultSet.getString(3),
						resultSet.getString(4),// Creator
						resultSet.getString(5),
						resultSet.getString(6),// record date
						resultSet.getString(7),
						resultSet.getString(8).replace("[", "").replace("]", "").replace("\"", ""), // Modality
						Util.Trim(resultSet.getString(9).replace(".", "").replace(" ", "").replace(";", ", ").replace("exe", "").replace("jar", ""), ' ', ','), // Filenames
						resultSet.getString(10).replace("[", "").replace("]", "").replace("\"", ""), // Shared with
						resultSet.getString(11).replace("[", "").replace("]", "").replace("\"", ""), // Description
						resultSet.getString(12),
						resultSet.getString(13),
						resultSet.getString(14),
						resultSet.getString(15),
						resultSet.getString(16),
						resultSet.getString(17),
						resultSet.getString(18).replace("meta.json", "") // Path
				};
				data.add(excelEntry);

			}
		} catch (SQLException e) {
			Debug.Error(e);
		}
		
		try {
			Object[][] dataFormatted = new Object[data.size()][];
			for(int i = 0; i < data.size(); i++) {
				dataFormatted[i] = data.get(i);
				for(int i2 = 0; i2 < dataFormatted[i].length; i2++)
					dataFormatted[i][i2] = ((String)(dataFormatted[i][i2])).replaceAll("[^\\p{ASCII}]", "");
			}
			
			XLSXhelper.create(dataFormatted, excelFile);
			JOptionPane.showMessageDialog(null, "Excel sheet saved successfully");
		} catch (IOException e) {
			e.printStackTrace();
		}
	}
	
	/* JXL soloution
	@Deprecated
	private void SaveToExcelWithJExcel(File excelFile) {
		try {
			WritableWorkbook workbook = Workbook.createWorkbook(excelFile);
			WritableSheet sheet = workbook.createSheet("Data", 0);
			
			LinkedList<Object[]> data = new LinkedList<Object[]>();
			data.add(new Object[] {"Subject ID", "Group ID", "Experiment title", "Creator", "Contributor", "Record date", "Resource Type", "Modality", "Datatype", "Shared with", "Experiment Description", "Subject Species", "Subject type", "Subject sex", "Subject age", "Animal|Ethics approval No.", "Extra information", "Path"});

			
			String searchStatement = "SELECT \"Subject ID\", \"Group ID\", \"Experiment title\", \"Creator\", \"Contributor\", \"Record date\", \"Resource Type\", \"Modality\", \"FileNames\", \"Shared with\", \"Experiment Description\", \"Subject Species\", \"Subject type\", \"Subject sex\", \"Subject age\", \"Animal|Ethics approval No.\", \"Extra information\", \"Path\""
					+ tableCreatingSqlStatement.substring(tableCreatingSqlStatement.indexOf(" FROM"));
			searchStatement = searchStatement.substring(0, searchStatement.length() - 1) + " " + orderBySqlAddition + ";";

			ResultSet resultSet = database.executeSafe(searchStatement);

			try {
				while(resultSet.next()) {
					if (removedPaths.contains(resultSet.getString(18))) continue; // if table row was manually removed;
					Object[] excelEntry = new Object[] {
							resultSet.getString(1), // Subject ID
							resultSet.getString(2),
							resultSet.getString(3),
							resultSet.getString(4),// Creator
							resultSet.getString(5),
							resultSet.getString(6),// record date
							resultSet.getString(7),
							resultSet.getString(8).replace("[", "").replace("]", "").replace("\"", ""), // Modality
							Util.Trim(resultSet.getString(9).replace(".", "").replace(" ", "").replace(";", ", ").replace("exe", "").replace("jar", ""), ' ', ','), // Filenames
							resultSet.getString(10).replace("[", "").replace("]", "").replace("\"", ""), // Shared with
							resultSet.getString(11).replace("[", "").replace("]", "").replace("\"", ""), // Description
							resultSet.getString(12),
							resultSet.getString(13),
							resultSet.getString(14),
							resultSet.getString(15),
							resultSet.getString(16),
							resultSet.getString(17),
							resultSet.getString(18).replace("meta.json", "") // Path
					};
					data.add(excelEntry);

				}
			} catch (SQLException e) {
				Debug.Error(e);
			}
			
			int row = 0;
			for(Object[] line : data) {
				for(int collum = 0; collum < line.length; collum++) {
					sheet.addCell(new Label(collum, row, line[collum].toString()));
				}
				row++;
			}
			
			workbook.write();
			workbook.close();
			
			JOptionPane.showMessageDialog(null, "Excel sheet saved successfully");
		} catch (IOException e) {
			Debug.Error(e);
		} catch (RowsExceededException e) {
			Debug.Error(e);
		} catch (WriteException e) {
			Debug.Error(e);
		}
	}*/
	
	/* APACHE POI SOLUTION
	@Deprecated
	private void SaveToExcelWithApachePOI(JFileChooser chooseFile) {
		XSSFWorkbook workbook = new XSSFWorkbook();
		XSSFSheet sheet = workbook.createSheet("Data");

		Map<String, Object[]> data = new TreeMap<String, Object[]>();
		data.put("1", new Object[] {"Subject ID", "Group ID", "Experiment title", "Creator", "Contributor", "Record date", "Resource Type", "Modality", "Datatype", "Shared with", "Experiment Description", "Subject Species", "Subject type", "Subject sex", "Subject age", "Animal|Ethics approval No.", "Extra information", "Path"});

		String searchStatement = "SELECT \"Subject ID\", \"Group ID\", \"Experiment title\", \"Creator\", \"Contributor\", \"Record date\", \"Resource Type\", \"Modality\", \"FileNames\", \"Shared with\", \"Experiment Description\", \"Subject Species\", \"Subject type\", \"Subject sex\", \"Subject age\", \"Animal|Ethics approval No.\", \"Extra information\", \"Path\""
				+ tableCreatingSqlStatement.substring(tableCreatingSqlStatement.indexOf(" FROM"));
		searchStatement = searchStatement.substring(0, searchStatement.length() - 1) + " " + orderBySqlAddition + ";";

		ResultSet resultSet = database.executeSafe(searchStatement);

		int resultSetRow = 2;

		try {
			while(resultSet.next()) {
				if (removedPaths.contains(resultSet.getString(18))) continue; // if table row was manually removed;
				Object[] excelEntry = new Object[] {
						resultSet.getString(1), // Subject ID
						resultSet.getString(2),
						resultSet.getString(3),
						resultSet.getString(4),// Creator
						resultSet.getString(5),
						resultSet.getString(6),// record date
						resultSet.getString(7),
						resultSet.getString(8).replace("[", "").replace("]", "").replace("\"", ""), // Modality
						Util.Trim(resultSet.getString(9).replace(".", "").replace(" ", "").replace(";", ", ").replace("exe", "").replace("jar", ""), ' ', ','), // Filenames
						resultSet.getString(10).replace("[", "").replace("]", "").replace("\"", ""), // Shared with
						resultSet.getString(11).replace("[", "").replace("]", "").replace("\"", ""), // Description
						resultSet.getString(12),
						resultSet.getString(13),
						resultSet.getString(14),
						resultSet.getString(15),
						resultSet.getString(16),
						resultSet.getString(17),
						resultSet.getString(18).replace("meta.json", "") // Path
				};
				data.put(resultSetRow+"", excelEntry);
				resultSetRow++;

			}
			//Iterate over data and write to sheet
			Set<String> keyset = data.keySet();
			int rownum = 0;
			for (String key : keyset) {
				Row row = sheet.createRow(rownum++);
				Object[] objArr = data.get(key);
				int cellnum = 0;
				for (Object obj : objArr) {
					Cell cell = row.createCell(cellnum++);
					if (obj instanceof String)
						cell.setCellValue((String) obj);
					else if (obj instanceof Integer)
						cell.setCellValue((Integer) obj);
				}
			}
			try {
				//Write the workbook in file system
				String fileName = chooseFile.getSelectedFile().getAbsolutePath();
				if (!fileName.endsWith(".xlsx")) fileName += ".xlsx";
				FileOutputStream out = new FileOutputStream(new File(fileName));

				workbook.write(out);
				out.close();
				System.out.println("XLSL written successfully on disk.");
			} catch (Exception e) {
				e.printStackTrace();
			}
		} catch(SQLException e) {
			Debug.Error(e);

		}
	}*/

	/**
	 * Saves the result of the query as a CSV sheet with information out of the meta.json file.
	 * <br>Informations: Subject ID, Datatype, Group, Datatype, Path
	 */
	private void SaveAsCSV() {
		JFileChooser chooseFile = new JFileChooser();
		//chooseFile.setFileSelectionMode(JFileChooser.FILES_ONLY);
		
		chooseFile.setFileFilter(new FileFilter() {
			
			@Override
			public String getDescription() {
				return ".csv";
			}
			
			@Override
			public boolean accept(File f) {
				return f.isDirectory() || f.getName().contains(".csv");
			}
		});
		
		int result = chooseFile.showSaveDialog(null);
		
		if (result == JFileChooser.APPROVE_OPTION) {
			File f = chooseFile.getSelectedFile();
			if (!f.getAbsolutePath().endsWith(".csv"))
				f = new File(f.getAbsolutePath() + ".csv");
			if (f.exists()) f.delete();
			
			try {
				f.createNewFile();
			} catch (IOException e1) {
				Debug.Error(e1);
			}
			
			OutputStreamWriter writer = null;
			try {
				if (f.exists())f.delete();
				f.createNewFile();
				writer = new OutputStreamWriter(new FileOutputStream(f), StandardCharsets.US_ASCII);
				//DefaultTableModel outputTableModel  = (DefaultTableModel) outputTable.getModel();
				
				String searchStatement = "SELECT \"Subject ID\", \"Group ID\", \"Experiment title\", \"Creator\", \"Contributor\", \"Record date\", \"Resource Type\", \"Modality\", \"FileNames\", \"Shared with\", \"Experiment Description\", \"Subject Species\", \"Subject type\", \"Subject sex\", \"Subject age\", \"Animal|Ethics approval No.\", \"Extra information\", \"Path\""
										+ tableCreatingSqlStatement.substring(tableCreatingSqlStatement.indexOf(" FROM"));
				searchStatement = searchStatement.substring(0, searchStatement.length() - 1) + " " + orderBySqlAddition + ";";
				
				ResultSet resultSet = database.executeSafe(searchStatement);
				
				int row = 0;
				
				writer.append("\"Subject ID\", \"Group ID\", \"Experiment title\", \"Creator\", \"Contributor\", \"Record date\", \"Resource Type\", \"Modality\", \"Datatype\", \"Shared with\", \"Experiment Description\", \"Subject Species\", \"Subject type\", \"Subject sex\", \"Subject age\", \"Animal|Ethics approval No.\", \"Extra information\", \"Path\"\n");
				while(resultSet.next()) {
					if(removedPaths.contains(resultSet.getString(18))) continue; // if table row was manually removed;
					
					writer.append(
							"\"" + resultSet.getString(1) // Subject ID 
							+ "\", \"" + resultSet.getString(2) 
							+ "\", \"" + resultSet.getString(3)
							+ "\", \"" + resultSet.getString(4) // Creator
							+ "\", \"" + resultSet.getString(5) 
							+ "\", \"" + resultSet.getString(6)// record date
							+ "\", \"" + resultSet.getString(7)
							+ "\", \"" + resultSet.getString(8).replace("[", "").replace("]", "").replace("\"", "") // Modality
							+ "\", \"" + Util.Trim(resultSet.getString(9).replace(".",  "").replace(" ", "").replace(";", ", ").replace("exe", "").replace("jar", ""), ' ', ',') // Filenames
							+ "\", \"" + resultSet.getString(10).replace("[", "").replace("]", "").replace("\"", "") // Shared with
							+ "\", \"" + resultSet.getString(11).replace("[", "").replace("]", "").replace("\"", "") // Description
							+ "\", \"" + resultSet.getString(12)
							+ "\", \"" + resultSet.getString(13)
							+ "\", \"" + resultSet.getString(14)
							+ "\", \"" + resultSet.getString(15)
							+ "\", \"" + resultSet.getString(16)
							+ "\", \"" + resultSet.getString(17)
							+ "\", \"" + resultSet.getString(18).replace("meta.json", "") + "\"\n"); // Path

					
				}
				writer.close();
				
				JOptionPane.showMessageDialog(null, "Success: CSV-file was successfully created");
				
			} catch (IOException | SQLException e1) {
				Debug.Error(e1);
				JOptionPane.showMessageDialog(null, "Failed: CSV-file could not be created");
				if (writer != null)
					try {
						writer.close();
					} catch (IOException e2) {
						Debug.Error(e2);
					}
			}
			
		}
	}
}
