/**
 * 
 */
package searching.gui;

import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;

import javax.swing.JPanel;

/**
 * This is a custom component.
 * It will layout the components inside from top to bottom or left to right.
 * Spacing and componet size can be determinated
 * @author Erik
 *
 */
public class LayoutBox extends JPanel{
	
	private boolean horizontal;
	private int sizePxl, spacePxl;
	
	private int componentCount = 0;
	
	/**
	 * Creates a LayoutBox
	 * @param horizontal whether the Component layouts the components Horizontal
	 * @param sizePxl the size (width, or height; depends on whether the box is horizontal or not) of the inner components in Pixel
	 * @param spacePxl Space in Pixel between the inner components
	 */
	public LayoutBox(boolean horizontal, int sizePxl, int spacePxl) {
		setLayout(null);
		this.horizontal = horizontal;
		this.sizePxl = sizePxl;
		this.spacePxl = spacePxl;
		
		addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(ComponentEvent e) {
				int i = 0;
				for(Component comp : getComponents()) {
					SetPosition(i, comp);
					i++;
				}
				repaint();
			}
		});
	}
	
	
	@Override
	public Component add(Component c) {
		super.add(c);
		SetPosition(componentCount, c);
		
		componentCount ++;
		UpdateSize();
		
		c.revalidate();
		c.repaint();
		
		return c;
	}
	
	/**
	 * Sets a certain Component on a certain order in the box
	 * @param position index (starting at 0)
	 * @param c the component which will be set at a certain position
	 */
	public void SetPosition(int position, Component c) {
		int start = (sizePxl + spacePxl) * position;
		
		if (horizontal) {
			c.setBounds(start, 0, sizePxl, (int)getHeight());
		} else {
			c.setBounds(0, start, (int)getWidth(), sizePxl);
		}
	}
	
	
	@Override
	public void remove(Component c) {
		super.remove(c);
		
		componentCount--;
		UpdateSize();
		
		repaint();
		
		int i = 0;
		for(Component comp : getComponents()) {
			SetPosition(i, comp);
			i++;
		}
		repaint();
	}
	
	/**
	 * Updates the Dimensions of the Layoutbox. This helps when smth is rendered poorly or no repaint has been called
	 */
	public void UpdateSize() {
		if (horizontal) 
			setPreferredSize(new Dimension((sizePxl + spacePxl) * componentCount, (int)getHeight()));
		else
			setPreferredSize(new Dimension((int)getWidth(), (sizePxl + spacePxl) * componentCount));
	}
}
