package creation;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileFilter;
import java.io.FileReader;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.SQLException;
import java.util.Collections;
import java.util.LinkedList;

import org.json.JSONException;
import org.json.JSONObject;

import utility.Database;
import utility.DatabaseInformation;
import utility.Debug;
import utility.Util;

/**
 * Class which handles the creation of the meta.json Database
 * @author Erik
 *
 */
public class DatabaseCreator {
	public static final String VERSION = "20220609";
	
	public static final String PATH_SEPERATOR = "\\";
	public Database database;
	
	
	private String destinationDirectory, dataRootPath;
	
	/**
	 * 
	 * @param destinationDirectory The Directory where the database is going to be created in
	 * @param dataRootPath The root directory of the data.
	 */
	public DatabaseCreator(String destinationDirectory, String dataRootPath) {
		Debug.Log("Version: " + VERSION);
		System.out.println("Version: " + VERSION);
		
		this.destinationDirectory = destinationDirectory;
		this.dataRootPath = dataRootPath;
	}
	
	
	/**
	 * Creates a empty Databasefile with a certain name. (The name should be SFB1280Database)
	 * @param name The name of the database without the .db ending
	 */
	public void CreateEmptyDatabase(String name) {
		File newDatabase = new File(destinationDirectory + PATH_SEPERATOR + name + ".db");
	
		if (newDatabase.exists())
			newDatabase.delete();
			
		
		
		database = new Database(newDatabase);
		
		try {
			newDatabase.createNewFile();
		}catch(IOException ex) {
			Debug.Error(ex);
			Debug.Error("Unable to create Database!");
			return;
		}
		
		
		try {
			database.execute(DatabaseInformation.CREATE_TABLE_COMMAND);
			database.execute(DatabaseInformation.CREATE_LAST_MODIFIED_TABLE_COMMAND);
			
			database.execute("INSERT INTO " + DatabaseInformation.LAST_MODIFIED_TABLE_NAME + " (Date) Values (\"" + Debug.GetCurrentDate("yyyy-MM-dd HH:mm") + "\");");
		} catch (SQLException e) {
			Debug.Error(e);			
			Debug.Error("Failed to create table!");
			return;
		}
		
		Debug.Log("Created Table!");
	}
	
	/**
	 * Fills the Database with the meta files from the SFB-Server. (recusivly)
	 * This operation may take some time! (It could actually take up to a few hours computetime)
	 */
	public void CreateDataTable() {
		//_CreateDataTable(new File(dataRootPath), false);
		File[] aFiles = new File(dataRootPath).listFiles(new FileFilter() {
			
			@Override
			public boolean accept(File file) {
				return file.isDirectory() && file.getName().startsWith("A") && onlyDecimals(file.getName().substring(1));
			}
		});
		for(File f : aFiles) {
			_CreateDataTable(f, false);
		}
		
		try { // Sets Last Modified date
			database.execute("UPDATE \"data\" SET \"Subject age\" = NULL WHERE \"Subject age\" = 'NaN'");
			database.execute("DROP TABLE " + DatabaseInformation.LAST_MODIFIED_TABLE_NAME + ";");
			database.execute(DatabaseInformation.CREATE_LAST_MODIFIED_TABLE_COMMAND);
			database.execute("INSERT INTO " + DatabaseInformation.LAST_MODIFIED_TABLE_NAME + " (Date) Values (\"" + Debug.GetCurrentDate("yyyy-MM-dd HH:mm") + "\");");
		} catch (SQLException e) {
			Debug.Error(e);
		}
		
		Util.ZipSingleFile(database.GetDatabaseFile(), new File(database.GetDatabaseFile().getAbsolutePath().replace(".db", ".zip")), true);
		
		Debug.Log("Data Fetch Finished");
	}
	
	private boolean onlyDecimals(String s) {
		for(char c : s.toCharArray()) {
			if (!Character.isDigit(c)) return false;
		}
		return true;
	}
	
	/**
	 * Fetches the meta files from a certain directory into the database. This method runs recursively.
	 * @param dir The current Directory
	 * 
	 * @returns True when everything is fine and false whenever the lowest subfolder has no metafile and is not explicitly in the databids format
	 */
	private boolean _CreateDataTable(File dir, boolean databitsFormat) {
		/**File[] subDirs = dir.listFiles(new FileFilter() {
			public boolean accept(File file) {
				return file.isDirectory();
			}
		});*/
		
		LinkedList<File> subDirs = new LinkedList<>();
		Path anyMetaFile = null;
		
		Path dirNIO = Paths.get(dir.getAbsolutePath());
		try (DirectoryStream<Path> stream = Files.newDirectoryStream(dirNIO)) {
			for (Path entry: stream) {
				if (Files.isDirectory(entry)) subDirs.add(entry.toFile());
				if (entry.toString().endsWith("meta.json")) anyMetaFile = entry;
		    }
		} catch (IOException ex) {
			Debug.Error(ex);
		}
		
		if (Files.exists(Paths.get(dir.getAbsolutePath() + PATH_SEPERATOR + "participants.tsv")))
			databitsFormat = true; // Switch format to databids.
		
		//if (subDirs == null)Debug.Error("Cannot extract subdirectories: " + dir.getAbsolutePath());
		
		if (databitsFormat) { // Databids format
			if (anyMetaFile != null) {
				AddEntry(anyMetaFile.toFile());
			} else {
				for(File subDir : subDirs) {
					_CreateDataTable(subDir, databitsFormat);
				}
			}
		} else { // Normal Format or databids without "participants.tsv" File
		
			if (subDirs.isEmpty() ) {
				
				if (anyMetaFile != null) {
					AddEntry(anyMetaFile.toFile());
				} else {
					Debug.Warning("No Meta-Data found! In the case that the data is in databids format, the \"participants.tsv\" file is missing! Problem occurred in: " + dir.getAbsolutePath() + PATH_SEPERATOR);
					return false;
				}
			} else {
				
				
				for(File subDir : subDirs) {
					if (!_CreateDataTable(subDir, databitsFormat)) { // Check any metafile, because of undetected databids
						if (anyMetaFile != null) {
							AddEntry(anyMetaFile.toFile()); //Prob. Databids found
							//Debug.Warning("Presumed databids file found, but no \"participants.tsv\". If it is no Databids format please add the meta.json at the lowest subfolder! Problem occurred in: " + subDir.getAbsolutePath());
						}
					}
				}
			}
		}
		return true;
	}
	
	
	/**
	 * Adds a single Entry based on a meta.json to the Database
	 * @param json the path of the meta.json file to add to the database
	 */
	private void AddEntry(File json) {
		StringBuilder builder = new StringBuilder();
		
		BufferedReader reader = null;
		try {
			reader = new BufferedReader(new FileReader(json));
			String nxtLine;
			while((nxtLine = reader.readLine()) != null) {
				builder.append(nxtLine);
			}
			reader.close();
		} catch (IOException e) {
			Debug.Error(e);
			Debug.Error(json.getAbsolutePath() + ": FAILED" );
			Debug.Log(json.getAbsolutePath() + ": FAILED" );
			
			if (json.getParent() == null) {
				Debug.Error("File has no parent!");
			}
				
			
			File[] otherJsonFiles = json.getParentFile().listFiles(new FilenameFilter() {
				
				public boolean accept(File dir, String name) {
					return name.contains(".json");
				}
			});
			if (otherJsonFiles != null && otherJsonFiles.length > 0) {
				Debug.Error("Found Other JSON files: " + otherJsonFiles);
			}
			
			if (reader != null)
				try {
					reader.close();
				} catch (IOException e1) {
					Debug.Error(e1);
				}
			
			return;
		}
		
		File[] allFiles = json.getParentFile().listFiles(new FileFilter() {

			public boolean accept(File file) {
				return !file.getName().contains("meta.json");
			}
			
		});
		
		String fileNames = ""; ///////--------------------------------------------  FILE NAMES ---------------------------------------------------
//		if (allFiles != null) {
//			for(int i = 0; i < allFiles.length; i++) {
//				fileNames += allFiles[i].getName();
//				if (i < allFiles.length - 1) fileNames += "; ";
//			}
//			
//			if (fileNames.length() > DatabaseInformation.MAX_CHARACTER_AMOUNT_FILE_NAMES) {
//				Debug.Error(json.getAbsolutePath() + ": FAILED" );
//				Debug.Error("Too many other files to fit into 'FileNames'-field");
//				Debug.Log(json.getAbsolutePath() + ": FAILED" );
//				return;
//			
//			}
//		}
		
//		if (allFiles != null) {
//			Arrays.sort(allFiles, new Comparator<File>() {
//
//				@Override
//				public int compare(File arg0, File arg1) {
//					String name1 = arg0.getName(), name2 = arg1.getName();
//					name1 = name1.substring(name1.indexOf("."));
//					name2 = name2.substring(name2.indexOf("."));
//					return name1.compareTo(name2);
//				}
//				
//			});
//			
//			for(int i = 0; i < allFiles.length; i++) {
//				String name = allFiles[i].getName();
//				
//				if (name.indexOf('.') < 0) continue;
//				name = name.substring(name.lastIndexOf('.'));
//				if (!fileNames.contains(name)) {
//					fileNames += name;
//				
//					if (i < allFiles.length - 1) fileNames += "; ";
//				}
//			}
//			
//			if (fileNames.endsWith("; "))fileNames = fileNames.substring(0, fileNames.length() - 2);
//		
		fileNames = ListFileExtensionsRecusiv(json.toPath().getParent());
		
		if (fileNames.length() > DatabaseInformation.MAX_CHARACTER_AMOUNT_FILE_NAMES) {
			Debug.Error(json.getAbsolutePath() + ": FAILED" );
			Debug.Error("Too many other files to fit into 'FileNames'-field");
			Debug.Log(json.getAbsolutePath() + ": FAILED" );
			return;
		
		}
		//}
			
		
		
		
		
		JSONObject obj = new JSONObject(builder.toString());
		StringBuilder sqlInsertSceme = new StringBuilder("INSERT INTO data (Path, FileNames, ");
		StringBuilder sqlInsertValues = new StringBuilder("VALUES ('" + json.getAbsolutePath()+"', '" + fileNames + "', ");
		
		for(int i = 0; i < DatabaseInformation.ALL_FIELDS_JSON.length; i++){
			String value;
			try {
				value = obj.get(DatabaseInformation.ALL_FIELDS_JSON[i]).toString();
			} catch (JSONException e) { 
				//in case of old metaFiles with missing entries
				value = "";
			}
			sqlInsertSceme.append(DatabaseInformation.ALL_FIELDS_JSON_APOSTROPHE[i] + ", ");
			value = value.replace("'", "");
			sqlInsertValues.append("'" + value + "', ");
		}
		
		sqlInsertSceme.delete(sqlInsertSceme.length() - 2, sqlInsertSceme.length());
		sqlInsertValues.delete(sqlInsertValues.length() - 2, sqlInsertValues.length());
		
		sqlInsertSceme.append(") ");
		sqlInsertValues.append(");");
		
		try {
			String statement = sqlInsertSceme.toString() + sqlInsertValues.toString();
			database.execute(statement);
			
			
		} catch (SQLException e) {
			Debug.Error(e);
			Debug.Error(json.getAbsolutePath() + ": FAILED" );
			return;
		}
		
		Debug.Log(json.getAbsolutePath() + ": SUCCESS" );
		
	}
	
	private String ListFileExtensionsRecusiv(Path dir) {
		LinkedList<String> extensions = new LinkedList<>();
		_ListFileExtensionsRecusiv(dir, extensions);
		
		String concat = "";
		Collections.sort(extensions);
		for(String file : extensions) {
			concat += file;
			if (!file.equals(extensions.getLast())) {
				concat +="; ";
			}
		}
		return concat;
	}
	
	private void _ListFileExtensionsRecusiv(Path dir, LinkedList<String> extensions) {
		
		try {
			DirectoryStream<Path> files = Files.newDirectoryStream(dir);
			for(Path path : files) {
				if(Files.isDirectory(path)) {
					if (!path.toString().endsWith("DICOM\\") && !path.toString().contains("DICOM")) _ListFileExtensionsRecusiv(path, extensions);
				} else {
					if (path.toString().endsWith("meta.json")) continue;
					
					String fileName = path.toString();
					
					int dotPosition = fileName.lastIndexOf('.');
					if (dotPosition != -1 && dotPosition > fileName.lastIndexOf('\\')) {
						fileName = fileName.substring(dotPosition + 1);
					} else {
						fileName = "DICOM";
					}
					
					if (!extensions.contains(fileName))
						extensions.add(fileName);
					
				}
				
				
			}
		} catch (IOException e) {
			Debug.Error(e);
		}
	}
}
