# ## CPSS Optimization
# Here we design a CPSS (circular polarization selective structure) similar to the previous example
# using PSSFSS in conjunction with the CMAES optimizer from the 
# [CMAEvolutionStrategy](https://github.com/jbrea/CMAEvolutionStrategy.jl) package.  I've used CMAES
# in the past with good success on some tough optimization problems.  Here is the code that defines 
# the objective function:

# ```julia
# using PSSFSS
# using Dates: now
# 
# let bestf = typemax(Float64)
#     global objective
#     """
#         result = objective(x)
# 
#     """
#     function objective(x)
#         ao, bo, ai, bi, ac, bc, wo, ho, wi, hi, wc, hc, t1, t2 = x
#         (bo > ho > 2.1*wo && bi > hi > 2.1*wi && bc > hc > 2.1*wc) || (return 5000.0)
# 
#         outer(rot) = meander(a=ao, b=bo, w1=wo, w2=wo, h=ho, units=mm, ntri=400, rot=rot)
#         inner(rot) = meander(a=ai, b=bi, w1=wi, w2=wi, h=hi, units=mm, ntri=400, rot=rot)
#         center(rot) = meander(a=ac, b=bc, w1=wc, w2=wc, h=hc, units=mm, ntri=400, rot=rot)
# 
#         substrate = Layer(width=0.1mm, epsr=2.6)
#         foam(w) = Layer(width=w, epsr=1.05)
#         rot0 = 0
# 
#         strata = [
#                 Layer()
#                 outer(rot0)
#                 substrate
#                 foam(t1*1mm)
#                 inner(rot0 - 45)
#                 substrate
#                 foam(t2*1mm)
#                 center(rot0 - 2*45)
#                 substrate
#                 foam(t2*1mm)
#                 inner(rot0 - 3*45)
#                 substrate
#                 foam(t1*1mm)
#                 outer(rot0 - 4*45)
#                 substrate
#                 Layer() ]
#         steering = (θ=0, ϕ=0)
#         flist = 11.5:0.25:18.5
#         resultfile = logfile = devnull
#         showprogress = false
#         results = analyze(strata, flist, steering; showprogress, resultfile, logfile)
#         s11rr, s21ll, ar11db, ar21db = eachcol(extract_result(results, 
#                        @outputs s11db(R,R) s21db(L,L) ar11db(R) ar21db(L)))
#         RL = -s11rr
#         IL = -s21ll
#         RLgoal, ILgoal, ARgoal  = (0.4, 0.5, 0.6)
#         obj = max(maximum(RL) - RLgoal, 
#                   maximum(IL) - ILgoal, 
#                   maximum(ar11db) - ARgoal, 
#                   maximum(ar21db) - ARgoal)
#         if obj < bestf
#             bestf = obj
#             open("optimization_best.log", "a") do fid
#                 xround = map(t  -> round(t, digits=4), x)
#                 println(fid, round(obj,digits=5), " at x = ", xround, "  #", now())
#             end
#         end
#         return obj
#     end
# end
# ```

# We optimize at 29 frequencies between 11.5 and 18.5 GHz.  In the previously presented 
# Sjöberg and Ericsson design a smaller frequency range of 12 to 18 GHz was used for optimization.
# We have also adopted more ambitious goals for return loss, insertion loss, and axial ratio
# of 0.4 dB, 0.5 dB, and 0.6 dB, respectively. These should be feasible because for
# our optimization setup, we have relaxed the restriction that all unit cells must be
# identical squares.  With more "knobs to turn" (i.e., a larger search space), we expect to be able
# to do somewhat better in terms of bandwidth and worst-case performance.  For our setup there are 
# fourteen optimization variables in all.  
# As one can see from the code above, each successive sheet in the structure is rotated an additional
# 45 degrees relative to its predecessor.   The objective is defined as the maximum departure
# from the goal of RHCP reflected return loss, LHCP insertion loss, or reflected or transmitted axial ratio that
# occurs at any of the analysis frequencies (i.e. we are setting up for "minimax" optimization). Also,
# the `let` block allows the objective function to maintain persistent state in the
# variable `bestf` which is initialized to the largest 64-bit floating point value. Each time a set 
# of inputs results in a lower objective function value, `bestf` is updated with this value and 
# the inputs and objective function value are
# written to the file "optimization_best.log", along with a date/time stamp.  This allows the user
# to monitor the optimization and to terminate the it prematurely, if desired, without losing the 
# best result achieved so far. Each objective function evaluation takes about 4.5 seconds on my machine.

# Here is the code for running the optimization:

# ```julia
# using CMAEvolutionStrategy
# #  x = [a1,  b1,   a2, b2,  a3,  b3,  wo,  ho,  wi,   hi,  wc,   hc,  t1,  t2]
# xmin = [3.0, 3.0, 3.0, 3.0, 3.0, 3.0, 0.1, 0.1, 0.1,  0.1, 0.1,  0.1, 2.0, 2.0]
# xmax = [5.0, 5.0, 5.0, 5.0, 5.0, 5.0, 0.35,4.0, 0.35, 4.0, 0.35, 4.0, 6.0, 6.0]
# x0 = 0.5 * (xmin + xmax)
# isfile("optimization_best.log") && rm("optimization_best.log")
# popsize = 2*(4 + floor(Int, 3*log(length(x0))))
# opt = minimize(objective, x0, 1.0;
#            lower = xmin,
#            upper = xmax,
#            maxfevals = 9000,
#            popsize = popsize,
#            xtol = 1e-4,
#            ftol = 1e-6)
# ```

# Note that I set the population size to twice the normal default value.  Based
# on previous experience, using 2 to 3 times the default population size helps the 
# optimizer to do better on tough objective functions like the present one.
# The optimizer finished after about 12 hours, having used up its budget of 9000 objective function
# evaluations. During this time it reduced the objective function
# value from 35.75 dB to -0.14 dB. 

# Here are the first and last few lines of the file "optimization_best.log" created during the optimization run:
# ```
# 35.74591 at x = [3.0822, 3.851, 3.0639, 3.1239, 3.7074, 3.1435, 0.3477, 2.4549, 0.1816, 2.9164, 0.335, 1.9599, 4.6263, 3.8668]  #2022-09-20T17:58:29.168
# 34.98097 at x = [4.1331, 3.9279, 3.3677, 3.4181, 3.0029, 4.5767, 0.2332, 1.3751, 0.1181, 3.2087, 0.3212, 1.7239, 3.7246, 3.7646]  #2022-09-20T17:58:35.329
# 21.45525 at x = [3.0427, 3.1525, 4.2728, 4.8541, 4.1922, 3.426, 0.102, 1.1193, 0.35, 2.0465, 0.1142, 1.9158, 3.4733, 4.0413]  #2022-09-20T17:58:45.925
# 13.85918 at x = [4.2285, 4.3504, 3.873, 4.3875, 3.7093, 3.8152, 0.118, 2.1575, 0.31, 3.5789, 0.3475, 3.0538, 5.5819, 2.9443]  #2022-09-20T17:59:45.984
# 7.71171 at x = [3.3824, 3.7428, 4.0395, 3.0979, 4.4467, 3.6702, 0.1304, 0.8826, 0.2323, 1.8111, 0.1534, 2.5018, 4.4872, 2.8612]  #2022-09-20T17:59:56.203
# 7.34573 at x = [4.2534, 4.7094, 4.0162, 3.3676, 3.2118, 4.4815, 0.1251, 2.6005, 0.3312, 1.5494, 0.3153, 1.5827, 2.0242, 4.181]  #2022-09-20T18:00:00.968
# 3.07587 at x = [4.9501, 4.6063, 4.9145, 4.6475, 4.3812, 3.1389, 0.1147, 2.2538, 0.3489, 2.1218, 0.1052, 0.8864, 3.6838, 3.4847]  #2022-09-20T18:00:12.513
# 3.05626 at x = [4.2391, 4.9991, 4.4545, 4.3303, 3.8393, 3.4906, 0.1207, 1.4096, 0.1421, 2.5086, 0.3435, 1.3212, 3.1339, 2.7907]  #2022-09-20T18:01:51.282
# 2.41192 at x = [3.7758, 4.9686, 4.0366, 4.5324, 4.2108, 3.9565, 0.1304, 2.0133, 0.345, 0.753, 0.1746, 1.0458, 2.7633, 2.8851]  #2022-09-20T18:02:04.960
# 2.19734 at x = [3.5704, 3.3381, 4.8014, 3.9773, 4.95, 3.5487, 0.1265, 2.3151, 0.2854, 1.1389, 0.108, 1.4173, 3.8662, 2.5112]  #2022-09-20T18:02:59.465
# ...
# -0.13539 at x = [3.1913, 4.8684, 3.5625, 3.0614, 3.5238, 3.1521, 0.3499, 2.8795, 0.3371, 1.2669, 0.2726, 2.3683, 3.9736, 2.3058]  #2022-09-21T05:32:23.953
# -0.13572 at x = [3.2131, 4.8659, 3.5518, 3.0545, 3.5278, 3.1605, 0.35, 2.868, 0.3365, 1.2666, 0.2749, 2.3801, 3.9877, 2.2922]  #2022-09-21T05:32:34.131
# -0.13595 at x = [3.2096, 4.8665, 3.5728, 3.0472, 3.4952, 3.1517, 0.35, 2.8719, 0.3369, 1.2747, 0.274, 2.3782, 3.983, 2.2889]  #2022-09-21T05:34:20.813
# -0.13598 at x = [3.1954, 4.8681, 3.567, 3.0514, 3.4953, 3.1516, 0.35, 2.8779, 0.3366, 1.2694, 0.273, 2.3704, 3.9765, 2.2992]  #2022-09-21T05:35:16.363
# -0.13599 at x = [3.1931, 4.862, 3.5728, 3.0501, 3.5085, 3.1609, 0.3499, 2.8702, 0.3366, 1.2717, 0.2757, 2.3794, 3.9833, 2.2925]  #2022-09-21T05:35:41.565
# -0.13615 at x = [3.2057, 4.8631, 3.5908, 3.0483, 3.5005, 3.1543, 0.3498, 2.869, 0.3371, 1.2736, 0.2743, 2.3792, 3.9859, 2.2878]  #2022-09-21T05:36:42.633
# -0.13664 at x = [3.177, 4.8596, 3.598, 3.0483, 3.4378, 3.1504, 0.3498, 2.8722, 0.3373, 1.2771, 0.2753, 2.3737, 3.9777, 2.291]  #2022-09-21T05:38:59.901
# -0.13678 at x = [3.1939, 4.8634, 3.5887, 3.0558, 3.4491, 3.1507, 0.35, 2.8725, 0.3381, 1.2715, 0.2744, 2.3721, 3.9811, 2.2932]  #2022-09-21T05:43:47.228
# -0.13692 at x = [3.1669, 4.8616, 3.6063, 3.0555, 3.4097, 3.1535, 0.35, 2.8759, 0.3369, 1.2722, 0.2765, 2.3676, 3.9739, 2.2998]  #2022-09-21T05:56:14.160
# -0.13694 at x = [3.1666, 4.8568, 3.609, 3.053, 3.387, 3.1645, 0.35, 2.8688, 0.3368, 1.2743, 0.2806, 2.3778, 3.981, 2.2902]  #2022-09-21T05:57:50.537
# -0.137 at x = [3.1638, 4.8589, 3.603, 3.0576, 3.3991, 3.1654, 0.35, 2.8714, 0.3365, 1.2687, 0.2796, 2.3729, 3.978, 2.2978]  #2022-09-21T05:58:19.459
# -0.13701 at x = [3.1611, 4.859, 3.6064, 3.0629, 3.3962, 3.1529, 0.35, 2.8733, 0.3375, 1.2688, 0.2766, 2.3659, 3.9759, 2.3007]  #2022-09-21T05:58:34.762
# -0.13712 at x = [3.1711, 4.86, 3.6049, 3.0547, 3.4055, 3.1535, 0.35, 2.8727, 0.3386, 1.2797, 0.2774, 2.3738, 3.9777, 2.2907]  #2022-09-21T06:01:01.374
# -0.13717 at x = [3.1659, 4.8555, 3.6091, 3.0577, 3.3919, 3.1635, 0.35, 2.8698, 0.337, 1.2668, 0.2792, 2.3721, 3.9804, 2.2971]  #2022-09-21T06:10:03.148
# -0.13759 at x = [3.1576, 4.8557, 3.605, 3.0616, 3.3655, 3.1612, 0.3499, 2.8687, 0.3368, 1.2662, 0.2796, 2.3696, 3.9802, 2.2976]  #2022-09-21T06:10:13.351
# ```

# The final sheet geometries and performance of this design are shown below:

# ![](./assets/cpss_cmaesopt_sheets.png)

# ![](./assets/cpss_cmaesopt_rl_refl.png)

# ![](./assets/cpss_cmaesopt_ar_refl.png)

# ![](./assets/cpss_cmaesopt_il_trans.png)

# ![](./assets/cpss_cmaesopt_ar_trans.png)

# As hoped for, the performance meets the more stringent design goals over a broader bandwidth than the 
# Sjöberg and Ericsson design, presumably because of the greater design flexibility allowed here.
