"""
    Author: Julia Maia
    Date: June 2023

    This script computes the gravity and topography of Venus
    for a 3 layer viscosity model using the dynamic loading model
    as presented in Maia, Wieczorek and Plesa (2023)

"""
import sys
import os

import numpy as np

import dynamic_loading_functions as mdyn
import multitaper_localization_functions as mt

#%%
# DATA
# maximum computed spherical harmonic degree
LMAX = 32

# Using multitapers to locaized the data
# 9 windows with spectral bandwidth = 3
nwins = 9
lwin = 3
OBS = mt.localized_observations(nwins, lwin, LMAX)

#%%
# MODEL PARAMETERS
viscosities_norm = np.array([1e2, 1e0, 1e1])  # Relative viscosities

layers_R = (6051 - np.array([100, 300])) * 1e3  # radii of visocisty interfaces in m
ms_R = (6051 - 250) * 1e3  # radii of mass-sheet density anomalies in m
Te = 0  # elastic thickness in m

modelparams = {
    "mun": viscosities_norm,
    "Rvis": layers_R,
    "Rms": ms_R,
    "Te": Te,
}

# Computing the model
model = mdyn.computemodel(modelparams, OBS, LMAX)

print(
    "Saving spherical harmonic coeffients: \npredicited gravity (glm_global.sh) \ntopography (hlm_global.sh) \ndesity anomaly distribution (mass-sheet.sh)"
)
for i, k in enumerate(model):
    model[k].to_file(f"{k}.sh")


# ---------
# PLOTTING MAPS
# --------
# To plot the maps of the predicted gravity and mantle density anomaly distribution
# uncomment the lines below
# The code makes use of the package pygmt

# import pygmt
#
# grid_topo = model["hlm_global"].expand(lmax=100) / 1000
# grid_gravity = model["glm_global"].expand(lmax=100) * 1e5
# grid_masssheet = model["mass-sheet"].expand(lmax=100) / 1e6
#
# fig = pygmt.Figure()
# grid_masssheet.plotgmt(
#     projection="mollweide",
#     central_longitude=0.0,
#     cmap="vik",
#     cmap_limits=[-10, 10],
#     cmap_reverse=True,
#     tick_interval=None,
#     colorbar="right",
#     cb_label="mass-sheet",
#     cb_ylabel="1e6 kg/m2",
#     cb_tick_interval=2.0,
#     cb_offset=15,
#     grid=[60, 45],
#     fig=fig,
# )
# grid_gravity.plotgmt(
#     projection="mollweide",
#     central_longitude=0.0,
#     cmap="roma",
#     cmap_limits=[-100, 200],
#     cmap_reverse=True,
#     tick_interval=None,
#     colorbar="right",
#     cb_label="Predicted gravity",
#     cb_ylabel="mGal",
#     cb_tick_interval=50.0,
#     cb_offset=15,
#     grid=[60, 45],
#     fig=fig,
#     offset=[0, -3.5],
# )
# fig.show()
