/*
 * WeakHamiltonianNonEye.cpp, part of Hadrons (https://github.com/aportelli/Hadrons)
 *
 * Copyright (C) 2015 - 2020
 *
 * Author: Antonin Portelli <antonin.portelli@me.com>
 *
 * Hadrons is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * Hadrons is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Hadrons.  If not, see <http://www.gnu.org/licenses/>.
 *
 * See the full license in the file "LICENSE" in the top level distribution 
 * directory.
 */

/*  END LEGAL */

#include <Hadrons/Modules/MContraction/WeakHamiltonianNonEye.hpp>

using namespace Grid;
using namespace Hadrons;
using namespace MContraction;

/*
 * Weak Hamiltonian current-current contractions, Non-Eye-type.
 * 
 * These contractions are generated by the Q1 and Q2 operators in the physical
 * basis (see e.g. Fig 3 of arXiv:1507.03094).
 * 
 * Schematic:     
 *            q2             q3          |           q2              q3
 *          /--<--¬       /--<--¬        |        /--<--¬         /--<--¬       
 *         /       \     /       \       |       /       \       /       \      
 *        /         \   /         \      |      /         \     /         \     
 *       /           \ /           \     |     /           \   /           \    
 *    i *             * H_W         *  f |  i *             * * H_W         * f 
 *      \             *             |    |     \           /   \           /
 *       \           / \           /     |      \         /     \         /    
 *        \         /   \         /      |       \       /       \       /  
 *         \       /     \       /       |        \-->--/         \-->--/      
 *          \-->--/       \-->--/        |          q1               q4 
 *            q1             q4          |
 *                Connected (C)          |                 Wing (W)
 *
 * C: trace(q1*adj(q2)*g5*gL[mu]*q3*adj(q4)*g5*gL[mu])
 * W: trace(q1*adj(q2)*g5*gL[mu])*trace(q3*adj(q4)*g5*gL[mu])
 * 
 */

/******************************************************************************
 *                  TWeakHamiltonianNonEye implementation                     *
 ******************************************************************************/
// constructor /////////////////////////////////////////////////////////////////
TWeakHamiltonianNonEye::TWeakHamiltonianNonEye(const std::string name)
: Module<WeakHamiltonianPar>(name)
{}

// dependencies/products ///////////////////////////////////////////////////////
std::vector<std::string> TWeakHamiltonianNonEye::getInput(void)
{
    std::vector<std::string> in = {par().q1, par().q2, par().q3, par().q4};
    
    return in;
}

std::vector<std::string> TWeakHamiltonianNonEye::getOutput(void)
{
    std::vector<std::string> out = {};
    
    return out;
}

// setup ///////////////////////////////////////////////////////////////////////
void TWeakHamiltonianNonEye::setup(void)
{
    unsigned int ndim = env().getNd();

    envTmpLat(LatticeComplex,  "expbuf");
    envTmpLat(PropagatorField, "tmp1");
    envTmpLat(LatticeComplex,  "tmp2");
    envTmp(std::vector<PropagatorField>, "C_i_side_loop", 1, ndim, PropagatorField(env().getGrid()));
    envTmp(std::vector<PropagatorField>, "C_f_side_loop", 1, ndim, PropagatorField(env().getGrid()));
    envTmp(std::vector<LatticeComplex>,  "W_i_side_loop", 1, ndim, LatticeComplex(env().getGrid()));
    envTmp(std::vector<LatticeComplex>,  "W_f_side_loop", 1, ndim, LatticeComplex(env().getGrid()));
}

// execution ///////////////////////////////////////////////////////////////////
void TWeakHamiltonianNonEye::execute(void)
{
    LOG(Message) << "Computing Weak Hamiltonian (Non-Eye type) contractions '" 
                 << getName() << "' using quarks '" << par().q1 << "', '" 
                 << par().q2 << ", '" << par().q3 << "' and '" << par().q4 
                 << "'." << std::endl;
    
    auto                  &q1 = envGet(PropagatorField, par().q1);
    auto                  &q2 = envGet(PropagatorField, par().q2);
    auto                  &q3 = envGet(PropagatorField, par().q3);
    auto                  &q4 = envGet(PropagatorField, par().q4);
    Gamma                 g5  = Gamma(Gamma::Algebra::Gamma5);
    std::vector<TComplex> corrbuf;
    std::vector<Result>   result(n_noneye_diag); 
    unsigned int          ndim = env().getNd();

    envGetTmp(LatticeComplex,               expbuf); 
    envGetTmp(PropagatorField,              tmp1);
    envGetTmp(LatticeComplex,               tmp2);
    envGetTmp(std::vector<PropagatorField>, C_i_side_loop);
    envGetTmp(std::vector<PropagatorField>, C_f_side_loop);
    envGetTmp(std::vector<LatticeComplex>,  W_i_side_loop);
    envGetTmp(std::vector<LatticeComplex>,  W_f_side_loop);

    // Setup for C-type contractions.
    for (int mu = 0; mu < ndim; ++mu)
    {
        C_i_side_loop[mu] = MAKE_CW_SUBDIAG(q1, q2, GammaL(Gamma::gmu[mu]));
        C_f_side_loop[mu] = MAKE_CW_SUBDIAG(q3, q4, GammaL(Gamma::gmu[mu]));
    }

    // Perform C-type contractions.    
    SUM_MU(expbuf, trace(C_i_side_loop[mu]*C_f_side_loop[mu]))
    MAKE_DIAG(expbuf, corrbuf, result[C_diag], "HW_C")

    // Recycle sub-expressions for W-type contractions.
    for (unsigned int mu = 0; mu < ndim; ++mu)
    {
        W_i_side_loop[mu] = trace(C_i_side_loop[mu]);
        W_f_side_loop[mu] = trace(C_f_side_loop[mu]);
    }

    // Perform W-type contractions.
    SUM_MU(expbuf, W_i_side_loop[mu]*W_f_side_loop[mu])
    MAKE_DIAG(expbuf, corrbuf, result[W_diag], "HW_W")

    // IO
    saveResult(par().output, "HW_NonEye", result);
}
