#include <pybind11/pybind11.h>
#include <pybind11/stl.h>
#include <vector>
#include <string>
#include <sstream>

#include "overallDefine.h"
#include "filter.h"
#include "depthFirst.h"
#include "RNA.h"
#include <iostream>
#include <fstream>
#include <chrono>
#include <numeric>
#include <algorithm>

#define STRINGIFY(x) #x
#define MACRO_STRINGIFY(x) STRINGIFY(x)


std::vector<std::string> outputPathVec(RNA, std::vector<ONE_PATH>);

/* ============= Main functions ============== */
/* =========================================== */
std::vector<std::string> string_assembly(std::vector<std::string>& rnaTXT) {

	// --------------------- Version --------------------
	std::string versionStr = "2020.04.23,10:00";
	// --------------------- Version --------------------

	size_t CashingSize = 10000000; // >= 10000,  2^24 = 16,777,216
	size_t CashingVacuumSize = 10000; // >= 1
	size_t CashingVacuumSeqLen = 15; // >= 2


	RNA rna0(rnaTXT);


	// Write to output file that it is running.
	std::vector<ONE_PATH> pathVec{};
	
	std::chrono::time_point<std::chrono::steady_clock> startTIME, endTIME;
	startTIME = std::chrono::steady_clock::now();
	// ---------------------------------------------------------------
	// ------ function depthFirstSearch() is the main process ------
	pathVec = depthFirstSearch(rna0, startTIME,
		CashingSize, CashingVacuumSize, CashingVacuumSeqLen);
	// ---------------------------------------------------------------
	// ---------------------------------------------------------------
	endTIME = std::chrono::steady_clock::now();

	std::chrono::duration<double> running_time = endTIME - startTIME;
	double ElapsedTime = running_time.count();
	// Write final output.
	std::vector<std::string> output = outputPathVec(rna0, pathVec);
	return output;
}



std::vector<std::string> outputPathVec(RNA rna0, std::vector<ONE_PATH> pathVec) {

	std::stringstream pathVecFile;
    std::vector<std::string> output;



	if (pathVec.empty()) {
		int PAIdx = rna0.seqComb.size() - rna0.seqIdx0.size();

		pathVecFile << "  --- No pathway found ---  ";

		pathVecFile << ". Assembly Index = " << PAIdx;

		std::string output_str  = pathVecFile.str();

		output.push_back(output_str);
	}
	else {
		size_t k = 0;
		for (ONE_PATH& path : pathVec) {
			k++;

			std::stringstream pathVecFile;

			pathVecFile << "Pathway " << k << " ---";

			pathVecFile << " Assembly Index = " << path.PAIdx <<" --- ";
			for (size_t i = 0; i < path.dupN.size(); ++i) {
				pathVecFile << path.dupN[i] << " * " << path.dups[i] <<" --- ";
			}
			std::string output_str = pathVecFile.str();
			output.push_back(output_str);
		}
	}

    return output;
}



/* ============= PyBind Magic functions ============== */
namespace py = pybind11;

PYBIND11_MODULE(_string_assembly, m) {
    m.doc() = R"pbdoc(
        String Assembly
        -----------------------

        .. currentmodule:: string_assembly

        .. autosummary::
           :toctree: _generate

           string_assembly
    )pbdoc";


    m.def("string_assembly", &string_assembly, R"pbdoc(
        string_assembly

    )pbdoc");


#ifdef VERSION_INFO
    m.attr("__version__") = MACRO_STRINGIFY(VERSION_INFO);
#else
    m.attr("__version__") = "dev";
#endif
}
