# Copyright (C) 2021 University of Glasgow
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright notice,
#    this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
# AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
# LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
# CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
# SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
# INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
# CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
# ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

import matplotlib.pyplot as plt
import matplotlib.gridspec as gridspec
import seaborn as sns
import csv

plt.rc('font',**{'family':'serif','serif':['Helvetica']})
plt.rc('axes', axisbelow=True)
plt.rcParams['pdf.fonttype'] = 42

data = {}
count_data = {}
type_data = {}
status_counts = {}
rfc_counts = {}

with open("data/errata-by-submitter-alldata.csv") as errataBySubmitterFile:
    csv_reader = csv.reader(errataBySubmitterFile, delimiter=',')
    next(csv_reader)
    for row in csv_reader:
        name = row[0]
        filing_count = int(row[1])
        editorial = int(row[3])
        technical = int(row[2])
        verified = int(row[4])
        reported = int(row[5])
        held_for_update = int(row[6])
        rejected = int(row[7])
        areas = [int(x) for x in row[8:]]
        count_data[name] = [area_count/filing_count for area_count in areas]
        data[name] = filing_count
        type_data[name] = (technical, editorial)
        status_counts[name] = (verified, reported, held_for_update, rejected)

with open("data/errata-by-submitter-rfcs.csv") as errataBySubmitterRfcsFile:
    csv_reader = csv.reader(errataBySubmitterRfcsFile, delimiter=',')
    for row in csv_reader:
        name = row[0]
        rfc_count = int(row[2])
        rfc_counts[name] = rfc_count

names = sorted(data.keys(), key=lambda x: data[x], reverse=True)

with open("tables/top-filers-errata.tex", "w") as topErrataTable:

	print("""\\begin{table*}
\\footnotesize
\\begin{tabular}{lr|rrr|rr|r|r}
\\toprule
\\textbf{Submitter} & \\textbf{Filings} & \\textbf{Verified} & \\textbf{Held for Update} & \\textbf{Rejected} & \\textbf{Technical} & \\textbf{Editorial} & \\textbf{Areas} & \\textbf{RFCs authored} \\\\
\\midrule""", file=topErrataTable)

	for name in names[:22]:
		print(f"{name.title()} & {data[name]} & {status_counts[name][0]} ({(status_counts[name][0]/data[name])*100:.1f}\%) & {status_counts[name][2]} ({(status_counts[name][2]/data[name])*100:.1f}\%) & {status_counts[name][3]} ({(status_counts[name][3]/data[name])*100:.1f}\%) & {type_data[name][0]} ({(type_data[name][0]/data[name])*100:.1f}\%) & {type_data[name][1]} ({(type_data[name][1]/data[name])*100:.1f}\%) & {len(count_data[name])-count_data[name].count(0)} & {rfc_counts.get(name, 0)} \\\\", file=topErrataTable)

	print("""\\bottomrule
\\end{tabular}
\\caption{Top errata submitters.}
\\label{tab:top_submitters}
\\end{table*}""", file=topErrataTable)