# define variable 'ops': optimisation parameter set
# define variable 'setups': simulation setup set
# define variable 'optimiser': properties for the optimiser

import numpy as np
import pandas as pd
import math

# import just for IDE convenience
import propti as pr


# Fix the chid.
CHID_MCC = 'PMMA_MCC'
CHID_TGA_01 = 'PMMA_TGA_01'
CHID_TGA_02 = 'PMMA_TGA_02'
CHID_TGA_03 = 'PMMA_TGA_03'
CHID_TGA_04 = 'PMMA_TGA_04'
CHID_TGA_05 = 'PMMA_TGA_05'
# CHID_TGA_06 = 'PMMA_TGA_06'
# CHID_TGA_07 = 'PMMA_TGA_07'
# CHID_TGA_08 = 'PMMA_TGA_08'

# Define name of template file.
template_file_MCC = "Guess_MCC_Template.fds"
template_file_TGA = "Guess_TGA_Template.fds"

# Set the total amount of residue.
total_resi_frac = 0.010095524020195633

# Set experiment data file names.
exp_MCC_time_file = "NIST_MCC_60K_HRR_Time.csv"
exp_MCC_temp_file = "NIST_MCC_60K_HRR_Temp.csv"

exp_DSC_temp_file_01 = "FSRI_DSC_10K_HeatFlow_Temp.csv"
exp_DSC_temp_file_02 = "FSRI_DSC_20K_HeatFlow_Temp.csv"

exp_TGA_temp_file_01 = "LCPP_TGA_2-5K_NormalisedResidualMass_Temp.csv"
exp_TGA_temp_file_02 = "LCPP_TGA_5K_NormalisedResidualMass_Temp.csv"
exp_TGA_temp_file_03 = "LCPP_TGA_15K_NormalisedResidualMass_Temp.csv"
exp_TGA_temp_file_04 = "LCPP_TGA_20K_NormalisedResidualMass_Temp.csv"

exp_TGA_temp_file_05 = "NIST_TGA_10K_NormalisedResidualMass_Temp.csv"

# exp_TGA_temp_file_06 = "Sandia_TGA_1K_NormalisedResidualMass_Temp.csv"
# # exp_TGA_temp_file_07 = "Sandia_TGA_10K_NormalisedResidualMass_Temp.csv"
# exp_TGA_temp_file_08 = "Sandia_TGA_50K_NormalisedResidualMass_Temp.csv"


# Initialise parameter collections.
set_of_parameters = list()
environment_paras = list()




# Define the optimisation parameters.
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 1A, Ref Temp',
        place_holder='comp_1A_ref_temp',
        min_value=180.0,
        max_value=225.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 1A, Pyro Range',
        place_holder='comp_1A_pyro_range',
        min_value=50.0,
        max_value=250.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 1A',
        place_holder='HOR_1A',
        min_value=10.0,
        max_value=3500.0))

set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 1B, Ref Temp',
        place_holder='comp_1B_ref_temp',
        min_value=180.0,
        max_value=225.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 1B, Pyro Range',
        place_holder='comp_1B_pyro_range',
        min_value=50.0,
        max_value=250.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 1B',
        place_holder='HOR_1B',
        min_value=10.0,
        max_value=2500.0))


set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 2A, Ref Temp',
        place_holder='comp_2A_ref_temp',
        min_value=375.0,
        max_value=405.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 2A, Pyro Range',
        place_holder='comp_2A_pyro_range',
        min_value=50.0,
        max_value=300.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 2A',
        place_holder='HOR_2A',
        min_value=10.0,
        max_value=2500.0))

set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 2B, Ref Temp',
        place_holder='comp_2B_ref_temp',
        min_value=370.0,
        max_value=405.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 2B, Pyro Range',
        place_holder='comp_2B_pyro_range',
        min_value=50.0,
        max_value=300.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 2B',
        place_holder='HOR_2B',
        min_value=10.0,
        max_value=2500.0))


set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 3A, Ref Temp',
        place_holder='comp_3A_ref_temp',
        min_value=380.0,
        max_value=400.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 3A, Pyro Range',
        place_holder='comp_3A_pyro_range',
        min_value=50.0,
        max_value=200.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 3A',
        place_holder='HOR_3A',
        min_value=10.0,
        max_value=3500.0))

set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 3B, Ref Temp',
        place_holder='comp_3B_ref_temp',
        min_value=380.0,
        max_value=400.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA Comp. 3B, Pyro Range',
        place_holder='comp_3B_pyro_range',
        min_value=50.0,
        max_value=160.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, HOR 3B',
        place_holder='HOR_3B',
        min_value=10.0,
        max_value=3500.0))


set_of_parameters.append(
    pr.Parameter(
        name='PMMA, Specific Heat 100',
        place_holder='PMMA_spec_heat_100',
        min_value=0.00250,
        max_value=5.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, Specific Heat 200',
        place_holder='PMMA_spec_heat_200',
        min_value=0.00250,
        max_value=5.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, Specific Heat 300',
        place_holder='PMMA_spec_heat_300',
        min_value=0.00250,
        max_value=5.0))
set_of_parameters.append(
    pr.Parameter(
        name='PMMA, Specific Heat 420',
        place_holder='PMMA_spec_heat_420',
        min_value=0.00250,
        max_value=5.0))


set_of_parameters.append(
    pr.Parameter(
        name='PMMA Residue, Specific Heat',
        place_holder='PMMA_resi_spec_heat',
        min_value=0.0050,
        max_value=6.0))



# Set up parameters with dependency.


# MATL mass fractions
################

# Here components of a gas mixture, should add up to 1.
# Intended structure, floats are supposed to be guess from algorithm:
# a1 = 0.5
# a2 = (1 - a1) * 0.1
# a3 = (1 - a1 - a2) * 0.5
# a4 = (1 - a1 - a2 - a3) * 0.6
# ar = 1 - a1 - a2 - a3 - a4  # essentially the remainder
# ac = a1 + a2 + a3 + a4 + ar  # check, should be 1
# print(a1, a2, a3, a4, ar, ac)


# Component 1A
set_of_parameters.append(pr.Parameter(
    name='MATL_Mass_Fraction_1A',
    place_holder='matl_mass_frac_1A',
    min_value=0.01,
    max_value=0.5))

# Component 1B
set_of_parameters.append(pr.Parameter(
    name='Helper_MATL_1B',
    min_value=0.01,
    max_value=0.99))
environment_paras.append(pr.Parameter(
    name='MATL_Mass_Fraction_1B',
    place_holder='matl_mass_frac_1B',
    evaluate_value='(1 - MATL_Mass_Fraction_1A) * Helper_MATL_1B'))


# Component 2A
set_of_parameters.append(pr.Parameter(
    name='Helper_MATL_2A',
    min_value=0.01,
    max_value=0.99))
environment_paras.append(pr.Parameter(
    name='MATL_Mass_Fraction_2A',
    place_holder='matl_mass_frac_2A',
    evaluate_value='(1 - MATL_Mass_Fraction_1A - MATL_Mass_Fraction_1B) * Helper_MATL_2A'))

# Component 2B
set_of_parameters.append(pr.Parameter(
    name='Helper_MATL_2B',
    min_value=0.01,
    max_value=0.99))
environment_paras.append(pr.Parameter(
    name='MATL_Mass_Fraction_2B',
    place_holder='matl_mass_frac_2B',
    evaluate_value='(1 - MATL_Mass_Fraction_1A - MATL_Mass_Fraction_1B - MATL_Mass_Fraction_2A) * Helper_MATL_2B'))


# Component 3A
set_of_parameters.append(pr.Parameter(
    name='Helper_MATL_3A',
    min_value=0.01,
    max_value=0.99))
environment_paras.append(pr.Parameter(
    name='MATL_Mass_Fraction_3A',
    place_holder='matl_mass_frac_3A',
    evaluate_value='(1 - MATL_Mass_Fraction_1A - MATL_Mass_Fraction_1B - MATL_Mass_Fraction_2A - MATL_Mass_Fraction_2B) * Helper_MATL_3A'))

# Component 3B
set_of_parameters.append(pr.Parameter(
    name='Helper_MATL_3B',
    min_value=0.01,
    max_value=0.99))
environment_paras.append(pr.Parameter(
    name='MATL_Mass_Fraction_3B',
    place_holder='matl_mass_frac_3B',
    evaluate_value='(1 - MATL_Mass_Fraction_1A - MATL_Mass_Fraction_1B - MATL_Mass_Fraction_2A - MATL_Mass_Fraction_2B - MATL_Mass_Fraction_3A)'))



# Definition of parameters which is used by propti_prepare.py, it has no further meaning here.
ops = pr.ParameterSet(params=set_of_parameters)

# MCC Simulation, 60 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_MCC = pr.ParameterSet(params=set_of_parameters)
mps_MCC.append(pr.Parameter(name='CHID_MCC',
                          place_holder='CHID_MCC',
                          value=CHID_MCC))
mps_MCC.append(pr.Parameter(name='HeatingRate',
                          place_holder='heating_rate',
                          value=60.0))  # MCC 60 K/min

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_MCC.append(env_para)


# Define model-experiment data relation.
# MCC - energy release rate
################
r1_MCC = pr.Relation()
r1_MCC.id_label = "MCC - ERR curve"
r1_MCC.model.file_name = "{}_tga.csv".format(CHID_MCC)
r1_MCC.model.label_x = 'Temp'
r1_MCC.model.label_y = 'MCC'
r1_MCC.model.header_line = 1
r1_MCC.experiment.file_name = exp_MCC_temp_file
r1_MCC.experiment.label_x = 'Temperature'
r1_MCC.experiment.xoffset = -273.15
r1_MCC.experiment.label_y = 'Average'
r1_MCC.experiment.header_line = 0
r1_MCC.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                           scale_fitness=True)

# MCC - heat of combustion
################
r2_MCC = pr.Relation()
r2_MCC.id_label = "MCC - ERR integral"
r2_MCC.model.file_name = "{}_tga.csv".format(CHID_MCC)
r2_MCC.model.label_x = 'Time'
r2_MCC.model.label_y = 'MCC'
r2_MCC.model.header_line = 1
r2_MCC.experiment.file_name = exp_MCC_time_file
r2_MCC.experiment.label_x = 'Time'
r2_MCC.experiment.label_y = 'Average'
r2_MCC.experiment.header_line = 0
r2_MCC.fitness_method = pr.FitnessMethodIntegrate(n_points=770,
                                                  scale_fitness=True,
                                                  integrate_factor=(1/(1 - total_resi_frac)))


# Create simulation setup object for MCC.
s1 = pr.SimulationSetup(name=CHID_MCC,
                        execution_dir_prefix='/tmp/',
                        work_dir=CHID_MCC,
                        model_template=template_file_MCC,
                        model_parameter=mps_MCC,
                        model_executable=os.path.join(os.getcwd(),
                                                      'fds.pleiades.sh'),
                        relations=[r1_MCC, r2_MCC])



# TGA Simulation, 2.5 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_TGA_01 = pr.ParameterSet(params=set_of_parameters)
mps_TGA_01.append(pr.Parameter(name='TGA_01',
                               place_holder='CHID_TGA',
                               value=CHID_TGA_01))
mps_TGA_01.append(pr.Parameter(name='HeatingRate',
                               place_holder='heating_rate',
                               value=2.5))  # TGA 2.5 K/min.

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_TGA_01.append(env_para)


# Define model-experiment data relation.
# TGA - normalised residual mass
################
r_TGA_01 = pr.Relation()
r_TGA_01.id_label = 'TGA_01 - MLR curve'
r_TGA_01.fitness_weight = 2.0
r_TGA_01.model.file_name = '{}_tga.csv'.format(CHID_TGA_01)
r_TGA_01.model.label_x = 'Temp'
r_TGA_01.model.label_y = 'Total Mass'
r_TGA_01.model.header_line = 1
r_TGA_01.experiment.file_name = exp_TGA_temp_file_01
r_TGA_01.experiment.label_x = 'Temperature'
r_TGA_01.experiment.xoffset = -273.15
r_TGA_01.experiment.label_y = 'Average'
r_TGA_01.experiment.header_line = 0
r_TGA_01.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)


# Create simulation setup object for TGA.
s_TGA_01 = pr.SimulationSetup(name=CHID_TGA_01,
                              execution_dir_prefix='/tmp/',
                              work_dir=CHID_TGA_01,
                              model_template=template_file_TGA,
                              model_parameter=mps_TGA_01,
                              model_executable=os.path.join(os.getcwd(),
                                                          'fds.pleiades.sh'),
                              relations=[r_TGA_01])


# TGA Simulation, 5 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_TGA_02 = pr.ParameterSet(params=set_of_parameters)
mps_TGA_02.append(pr.Parameter(name='TGA_02',
                               place_holder='CHID_TGA',
                               value=CHID_TGA_02))
mps_TGA_02.append(pr.Parameter(name='HeatingRate',
                               place_holder='heating_rate',
                               value=5))  # TGA 5 K/min.

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_TGA_02.append(env_para)


# Define model-experiment data relation.
# TGA - normalised residual mass
################
r_TGA_02 = pr.Relation()
r_TGA_02.id_label = 'TGA_02 - MLR curve'
r_TGA_02.fitness_weight = 2.0
r_TGA_02.model.file_name = '{}_tga.csv'.format(CHID_TGA_02)
r_TGA_02.model.label_x = 'Temp'
r_TGA_02.model.label_y = 'Total Mass'
r_TGA_02.model.header_line = 1
r_TGA_02.experiment.file_name = exp_TGA_temp_file_02
r_TGA_02.experiment.label_x = 'Temperature'
r_TGA_02.experiment.xoffset = -273.15
r_TGA_02.experiment.label_y = 'Average'
r_TGA_02.experiment.header_line = 0
r_TGA_02.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)


# Create simulation setup object for TGA.
s_TGA_02 = pr.SimulationSetup(name=CHID_TGA_02,
                              execution_dir_prefix='/tmp/',
                              work_dir=CHID_TGA_02,
                              model_template=template_file_TGA,
                              model_parameter=mps_TGA_02,
                              model_executable=os.path.join(os.getcwd(),
                                                          'fds.pleiades.sh'),
                              relations=[r_TGA_02])


# TGA Simulation, 15 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_TGA_03 = pr.ParameterSet(params=set_of_parameters)
mps_TGA_03.append(pr.Parameter(name='TGA_03',
                               place_holder='CHID_TGA',
                               value=CHID_TGA_03))
mps_TGA_03.append(pr.Parameter(name='HeatingRate',
                               place_holder='heating_rate',
                               value=15))  # TGA 15 K/min.

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_TGA_03.append(env_para)


# Define model-experiment data relation.
# TGA - normalised residual mass
################
r_TGA_03 = pr.Relation()
r_TGA_03.id_label = 'TGA_03 - MLR curve'
r_TGA_03.fitness_weight = 2.0
r_TGA_03.model.file_name = '{}_tga.csv'.format(CHID_TGA_03)
r_TGA_03.model.label_x = 'Temp'
r_TGA_03.model.label_y = 'Total Mass'
r_TGA_03.model.header_line = 1
r_TGA_03.experiment.file_name = exp_TGA_temp_file_03
r_TGA_03.experiment.label_x = 'Temperature'
r_TGA_03.experiment.xoffset = -273.15
r_TGA_03.experiment.label_y = 'Average'
r_TGA_03.experiment.header_line = 0
r_TGA_03.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)

# Create simulation setup object for TGA.
s_TGA_03 = pr.SimulationSetup(name=CHID_TGA_03,
                              execution_dir_prefix='/tmp/',
                              work_dir=CHID_TGA_03,
                              model_template=template_file_TGA,
                              model_parameter=mps_TGA_03,
                              model_executable=os.path.join(os.getcwd(),
                                                          'fds.pleiades.sh'),
                              relations=[r_TGA_03])


# TGA Simulation, 20 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_TGA_04 = pr.ParameterSet(params=set_of_parameters)
mps_TGA_04.append(pr.Parameter(name='TGA_04',
                               place_holder='CHID_TGA',
                               value=CHID_TGA_04))
mps_TGA_04.append(pr.Parameter(name='HeatingRate',
                               place_holder='heating_rate',
                               value=20))  # TGA 20 K/min.

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_TGA_04.append(env_para)


# Define model-experiment data relation.
# TGA - normalised residual mass
################
r_TGA_04 = pr.Relation()
r_TGA_04.id_label = 'TGA_04 - MLR curve'
r_TGA_04.fitness_weight = 2.0
r_TGA_04.model.file_name = '{}_tga.csv'.format(CHID_TGA_04)
r_TGA_04.model.label_x = 'Temp'
r_TGA_04.model.label_y = 'Total Mass'
r_TGA_04.model.header_line = 1
r_TGA_04.experiment.file_name = exp_TGA_temp_file_04
r_TGA_04.experiment.label_x = 'Temperature'
r_TGA_04.experiment.xoffset = -273.15
r_TGA_04.experiment.label_y = 'Average'
r_TGA_04.experiment.header_line = 0
r_TGA_04.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)

# DSC - heat flow
################
r_DSC_04 = pr.Relation()
r_DSC_04.id_label = 'DSC_04 - heat flow curve'
r_DSC_04.fitness_weight = 2.0
r_DSC_04.model.file_name = '{}_tga.csv'.format(CHID_TGA_04)
r_DSC_04.model.label_x = 'Temp'
r_DSC_04.model.label_y = 'DSC'
r_DSC_04.model.header_line = 1
r_DSC_04.experiment.file_name = exp_DSC_temp_file_02
r_DSC_04.experiment.label_x = 'Temperature'
r_DSC_04.experiment.xoffset = -273.15
r_DSC_04.experiment.label_y = 'Average'
r_DSC_04.experiment.header_line = 0
r_DSC_04.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)


# Create simulation setup object for TGA.
s_TGA_04 = pr.SimulationSetup(name=CHID_TGA_04,
                              execution_dir_prefix='/tmp/',
                              work_dir=CHID_TGA_04,
                              model_template=template_file_TGA,
                              model_parameter=mps_TGA_04,
                              model_executable=os.path.join(os.getcwd(),
                                                          'fds.pleiades.sh'),
                              relations=[r_TGA_04, r_DSC_04])


# TGA Simulation, 10 K/min.
# -------------------------------
# Define general model parameter, including optimisation parameter set.
mps_TGA_05 = pr.ParameterSet(params=set_of_parameters)
mps_TGA_05.append(pr.Parameter(name='TGA_05',
                               place_holder='CHID_TGA',
                               value=CHID_TGA_05))
mps_TGA_05.append(pr.Parameter(name='HeatingRate',
                               place_holder='heating_rate',
                               value=10))  # TGA 10 K/min.

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps_TGA_05.append(env_para)


# Define model-experiment data relation.
# TGA - normalised residual mass
################
r_TGA_05 = pr.Relation()
r_TGA_05.id_label = 'TGA_05 - MLR curve'
r_TGA_05.fitness_weight = 2.0
r_TGA_05.model.file_name = '{}_tga.csv'.format(CHID_TGA_05)
r_TGA_05.model.label_x = 'Temp'
r_TGA_05.model.label_y = 'Total Mass'
r_TGA_05.model.header_line = 1
r_TGA_05.experiment.file_name = exp_TGA_temp_file_05
r_TGA_05.experiment.label_x = 'Temperature'
r_TGA_05.experiment.xoffset = -273.15
r_TGA_05.experiment.label_y = 'Average'
r_TGA_05.experiment.header_line = 0
r_TGA_05.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)

# DSC - heat flow
################
r_DSC_05 = pr.Relation()
r_DSC_05.id_label = 'DSC_05 - heat flow curve'
r_DSC_05.fitness_weight = 2.0
r_DSC_05.model.file_name = '{}_tga.csv'.format(CHID_TGA_05)
r_DSC_05.model.label_x = 'Temp'
r_DSC_05.model.label_y = 'DSC'
r_DSC_05.model.header_line = 1
r_DSC_05.experiment.file_name = exp_DSC_temp_file_01
r_DSC_05.experiment.label_x = 'Temperature'
r_DSC_05.experiment.xoffset = -273.15
r_DSC_05.experiment.label_y = 'Average'
r_DSC_05.experiment.header_line = 0
r_DSC_05.fitness_method=pr.FitnessMethodRMSE(n_points=770,
                                             x_def_range=[30.0, 600.0],
                                             scale_fitness=True)


# Create simulation setup object for TGA.
s_TGA_05 = pr.SimulationSetup(name=CHID_TGA_05,
                              execution_dir_prefix='/tmp/',
                              work_dir=CHID_TGA_05,
                              model_template=template_file_TGA,
                              model_parameter=mps_TGA_05,
                              model_executable=os.path.join(os.getcwd(),
                                                          'fds.pleiades.sh'),
                              relations=[r_TGA_05, r_DSC_05])


# # TGA Simulation, 1 K/min.
# # -------------------------------
# # Define general model parameter, including optimisation parameter set.
# mps_TGA_06 = pr.ParameterSet(params=set_of_parameters)
# mps_TGA_06.append(pr.Parameter(name='TGA_06',
#                                place_holder='CHID_TGA',
#                                value=CHID_TGA_06))
# mps_TGA_06.append(pr.Parameter(name='HeatingRate',
#                                place_holder='heating_rate',
#                                value=1))  # TGA 1 K/min.
#
# # Add parameters with a dependency as environment parameters.
# for env_para in environment_paras:
#     mps_TGA_06.append(env_para)
#
#
# # Define model-experiment data relation.
# # TGA - normalised residual mass
# ################
# r_TGA_06 = pr.Relation()
# r_TGA_06.id_label = 'TGA_06 - MLR curve'
# r_TGA_06.fitness_weight = 2.0
# r_TGA_06.model.file_name = '{}_tga.csv'.format(CHID_TGA_06)
# r_TGA_06.model.label_x = 'Temp'
# r_TGA_06.model.label_y = 'Total Mass'
# r_TGA_06.model.header_line = 1
# r_TGA_06.experiment.file_name = exp_TGA_temp_file_06
# r_TGA_06.experiment.label_x = 'Temperature'
# r_TGA_06.experiment.xoffset = -273.15
# r_TGA_06.experiment.label_y = 'Average'
# r_TGA_06.experiment.header_line = 0
# r_TGA_06.fitness_method=pr.FitnessMethodRMSE(n_points=770,
#                                              x_def_range=[30.0, 600.0],
#                                              scale_fitness=True)
#
#
# # Create simulation setup object for TGA.
# s_TGA_06 = pr.SimulationSetup(name=CHID_TGA_06,
#                               execution_dir_prefix='/tmp/',
#                               work_dir=CHID_TGA_06,
#                               model_template=template_file_TGA,
#                               model_parameter=mps_TGA_06,
#                               model_executable=os.path.join(os.getcwd(),
#                                                           'fds.pleiades.sh'),
#                               relations=[r_TGA_06])
#
#
# # TGA Simulation, 50 K/min.
# # -------------------------------
# # Define general model parameter, including optimisation parameter set.
# mps_TGA_08 = pr.ParameterSet(params=set_of_parameters)
# mps_TGA_08.append(pr.Parameter(name='TGA_08',
#                                place_holder='CHID_TGA',
#                                value=CHID_TGA_08))
# mps_TGA_08.append(pr.Parameter(name='HeatingRate',
#                                place_holder='heating_rate',
#                                value=50))  # TGA 50 K/min.
#
# # Add parameters with a dependency as environment parameters.
# for env_para in environment_paras:
#     mps_TGA_08.append(env_para)
#
#
# # Define model-experiment data relation.
# # TGA - normalised residual mass
# ################
# r_TGA_08 = pr.Relation()
# r_TGA_08.id_label = 'TGA_08 - MLR curve'
# r_TGA_08.fitness_weight = 2.0
# r_TGA_08.model.file_name = '{}_tga.csv'.format(CHID_TGA_08)
# r_TGA_08.model.label_x = 'Temp'
# r_TGA_08.model.label_y = 'Total Mass'
# r_TGA_08.model.header_line = 1
# r_TGA_08.experiment.file_name = exp_TGA_temp_file_08
# r_TGA_08.experiment.label_x = 'Temperature'
# r_TGA_08.experiment.xoffset = -273.15
# r_TGA_08.experiment.label_y = 'Average'
# r_TGA_08.experiment.header_line = 0
# r_TGA_08.fitness_method=pr.FitnessMethodRMSE(n_points=770,
#                                              x_def_range=[30.0, 600.0],
#                                              scale_fitness=True)
#
#
# # Create simulation setup object for TGA.
# s_TGA_08 = pr.SimulationSetup(name=CHID_TGA_08,
#                               execution_dir_prefix='/tmp/',
#                               work_dir=CHID_TGA_08,
#                               model_template=template_file_TGA,
#                               model_parameter=mps_TGA_08,
#                               model_executable=os.path.join(os.getcwd(),
#                                                           'fds.pleiades.sh'),
#                               relations=[r_TGA_08])


# Initialise empty simulation setup set.
setups = pr.SimulationSetupSet()

# Add all simulation setups to the simulation setup set.
setups.append(s1)        # MCC 60 K/min
setups.append(s_TGA_01)  # TGA 2.5 K/min
setups.append(s_TGA_02)  # TGA 5 K/min
setups.append(s_TGA_03)  # TGA 15 K/min
setups.append(s_TGA_04)  # TGA 20 K/min
setups.append(s_TGA_05)  # TGA 10 K/min
# setups.append(s_TGA_06)  # TGA 1 K/min
# setups.append(s_TGA_08)  # TGA 50 K/min

# Determine number of complexes (equal to number of optimisation parameters).
n_opti_para = len(set_of_parameters)
print(f'** Number of optimisation parameters: {n_opti_para}')

# Set values for optimiser.
optimiser = pr.OptimiserProperties(algorithm='sceua',
                                   repetitions=261783,
                                   backup_every=1,
                                   ngs=n_opti_para,
                                   num_subprocesses=4,
                                   mpi=True,
                                   max_loop_inc=1)


print('** Input file processed.')
