# define variable 'ops': optimisation parameter set
# define variable 'setups': simulation setup set
# define variable 'optimiser': properties for the optimiser

import numpy as np
import pandas as pd
import math

# import just for IDE convenience
import propti as pr


# Fix the chid.
CHID_Cone = 'Aalto_Cone_65kW'

# Define name of template file.
template_file_MCC = "Aalto_PMMA_Cone_65kW_Template.fds"

# Set experiment data file name.
exp_file_err = "Aalto_Cone_65kW_ERR_Processed.csv"
exp_file_backtemp = "Aalto_Cone_65kW_BackTemp_Processed.csv"

# Initialise parameter collections.
set_of_parameters = list()
environment_paras = list()


# Define the optimisation parameters.
set_of_parameters.append(pr.Parameter(
    name='PMMA, Emissivity',
    place_holder='PMMA_emissivity',
    min_value=0.5,
    max_value=0.999))
set_of_parameters.append(pr.Parameter(
    name='PMMA, Absorption Coefficient',
    place_holder='PMMA_absorption_coeff',
    min_value=10,
    max_value=4000))

set_of_parameters.append(pr.Parameter(
    name='PMMA, Conductivity 50',
    place_holder='PMMA_conductivity_50',
    min_value=0.005,
    max_value=0.7))
set_of_parameters.append(pr.Parameter(
    name='PMMA, Conductivity 150',
    place_holder='PMMA_conductivity_150',
    min_value=0.005,
    max_value=0.7))
set_of_parameters.append(pr.Parameter(
    name='PMMA, Conductivity 300',
    place_holder='PMMA_conductivity_300',
    min_value=0.005,
    max_value=0.7))
set_of_parameters.append(pr.Parameter(
    name='PMMA, Conductivity 420',
    place_holder='PMMA_conductivity_420',
    min_value=0.005,
    max_value=0.7))


set_of_parameters.append(pr.Parameter(
    name='PMMA Residue, Emissivity',
    place_holder='PMMA_resi_emissivity',
    min_value=0.4,
    max_value=0.999))
set_of_parameters.append(pr.Parameter(
    name='PMMA Residue, Conductivity',
    place_holder='PMMA_resi_conductivity',
    min_value=0.005,
    max_value=1.250))


set_of_parameters.append(pr.Parameter(
    name='PMMA Backing, Emissivity',
    place_holder='PMMA_backing_emissivity',
    min_value=0.4,
    max_value=0.999))
set_of_parameters.append(pr.Parameter(
    name='PMMA Backing, Conductivity',
    place_holder='PMMA_backing_conductivity',
    min_value=0.005,
    max_value=1.250))
set_of_parameters.append(pr.Parameter(
    name='PMMA Backing, Specific Heat',
    place_holder='PMMA_backing_spec_heat',
    min_value=0.0050,
    max_value=5.0))



# Definition of parameters which is used by propti_prepare.py, it has no further meaning here.
ops = pr.ParameterSet(params=set_of_parameters)


# Define general model parameter, including optimisation parameter set.
mps01 = pr.ParameterSet(params=set_of_parameters)
mps01.append(pr.Parameter(name='CHID_Cone',
                          place_holder='CHID_Cone',
                          value=CHID_Cone))

# Add parameters with a dependency as environment parameters.
for env_para in environment_paras:
    mps01.append(env_para)


# Define model-experiment data relation.
# Repetition 01
# Cone - energy release rate
################
r1 = pr.Relation()
r1.id_label = "Cone 65 - ERR curve"
r1.model.file_name = "{}_hrr.csv".format(CHID_Cone)
r1.model.label_x = 'Time'
r1.model.label_y = 'HRR'
r1.model.yfactor = 1/(0.1*0.1)  # Calculate HRRPUA
r1.model.header_line = 1
r1.experiment.file_name = exp_file_err
r1.experiment.label_x = 'Time'
r1.experiment.label_y = 'Average'
r1.experiment.header_line = 0
r1.fitness_method=pr.FitnessMethodRangeRMSE(n_points=300,
                                            x_def_range=[0, 301],
                                            scale_fitness=True)


# Create simulation setup object for MCC.
s1 = pr.SimulationSetup(name=CHID_Cone,
                        execution_dir_prefix='/tmp/',
                        work_dir=CHID_Cone,
                        model_template=template_file_MCC,
                        model_parameter=mps01,
                        model_executable=os.path.join(os.getcwd(),
                                                      'fds.pleiades.sh'),
                        relations=[r1])


# Initialise empty simulation setup set.
setups = pr.SimulationSetupSet()

# Add all simulation setups to the simulation setup set.
setups.append(s1)

# Determine number of complexes (equal to number of optimisation parameters).
n_opti_para = len(set_of_parameters)
print(f'** Number of optimisation parameters: {n_opti_para}')

# Set values for optimiser.
optimiser = pr.OptimiserProperties(algorithm='sceua',
                                   repetitions=15939,
                                   backup_every=1,
                                   ngs=n_opti_para,
                                   num_subprocesses=1,
                                   mpi=True,
                                   max_loop_inc=1)


print('** Input file processed.')
