!
! Copyright (C) 1991-2004  ; All Rights Reserved ; Colorado State University
! Colorado State University Research Foundation ; ATMET, LLC
! 
! This file is free software; you can redistribute it and/or modify it under the
! terms of the GNU General Public License as published by the Free Software 
! Foundation; either version 2 of the License, or (at your option) any later version.
! 
! This software is distributed in the hope that it will be useful, but WITHOUT ANY 
! WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
! PARTICULAR PURPOSE.  See the GNU General Public License for more details.
!
! You should have received a copy of the GNU General Public License along with this 
! code; if not, write to the Free Software Foundation, Inc., 
! 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
!======================================================================================

Subroutine mclatchy (iaction,m1  &
   ,prsnz,prsnzp,glat,rtgt,topt,rlongup  &
   ,zm,zt,press,tair,dn0,rv,zml,ztl,pl,tl,dl,rl,o3l,dzl)

use rconstants
use rrad3

implicit none

integer :: iaction,m1,k,lv,lf,lats,latn,lat  &
   ,isummer,iwinter,is,ind,index,ilev,ifld
integer, dimension(11) :: latind

real :: prsnz,prsnzp,glat,rtgt,topt,rlongup,rgasog,deltap  &
   ,wtnorth,wtsouth,wt,fjday,wtjul,wtjan
real, dimension(m1) :: zm,zt,press,tair,dn0,rv
real, dimension(nrad) ::   zml,ztl,pl,tl,dl,rl,o3l,dzl
real, dimension(33,9,6) :: mcdat,mclat
real, dimension(33,6) :: mcol
real, dimension(12) :: slat

data latind/1,1,2,3,4,5,5,6,7,8,9/
data slat/-90.,-70.,-60.,-45.,-25.,-15.,15.,25.,45.,60.,70.,90./

!  arctic winter
data ((mcdat(ilev,1,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101350.0,    249.1,  .1201E-02,  .4105E-07,  .1417E+01,  &
    1000.,  88416.0,    252.2,  .1190E-02,  .4067E-07,  .1221E+01,  &
    2000.,  77213.0,    250.9,  .1014E-02,  .4036E-07,  .1072E+01,  &
    3000.,  67274.0,    245.4,  .7333E-03,  .4221E-07,  .9549E+00,  &
    4000.,  58431.0,    239.9,  .4471E-03,  .4384E-07,  .8485E+00,  &
    5000.,  50583.0,    234.4,  .2254E-03,  .4527E-07,  .7518E+00,  &
    6000.,  43640.0,    228.9,  .9344E-04,  .4681E-07,  .6643E+00,  &
    7000.,  37520.0,    223.4,  .3123E-04,  .6740E-07,  .5852E+00,  &
    8000.,  32171.0,    217.9,  .1248E-04,  .8508E-07,  .5139E+00,  &
    9000.,  27435.0,    214.9,  .7875E-05,  .1505E-06,  .4448E+00,  &
   10000.,  23398.0,    214.4,  .5161E-05,  .2248E-06,  .3802E+00/
data ((mcdat(ilev,1,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  19951.0,    213.9,  .3533E-05,  .2983E-06,  .3249E+00,  &
   12000.,  17008.0,    213.2,  .2393E-05,  .3988E-06,  .2779E+00,  &
   13000.,  14490.0,    212.4,  .1538E-05,  .4330E-06,  .2376E+00,  &
   14000.,  12338.0,    211.6,  .1005E-05,  .4477E-06,  .2031E+00,  &
   15000.,  10499.0,    210.9,  .6644E-06,  .5076E-06,  .1735E+00,  &
   16000.,   8929.0,    210.1,  .5438E-06,  .5554E-06,  .1481E+00,  &
   17000.,   7590.0,    209.3,  .4610E-06,  .5497E-06,  .1264E+00,  &
   18000.,   6450.0,    208.4,  .3906E-06,  .5442E-06,  .1078E+00,  &
   19000.,   5475.0,    207.7,  .3307E-06,  .5208E-06,  .9185E-01,  &
   20000.,   4648.0,    207.6,  .2800E-06,  .4809E-06,  .7797E-01,  &
   21000.,   3945.0,    207.6,  .2373E-06,  .4337E-06,  .6619E-01/
data ((mcdat(ilev,1,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   3349.0,    207.6,  .2014E-06,  .3961E-06,  .5619E-01,  &
   23000.,   2843.0,    207.6,  .1705E-06,  .3594E-06,  .4770E-01,  &
   24000.,   2414.0,    207.6,  .1443E-06,  .2986E-06,  .4050E-01,  &
   25000.,   2050.0,    207.6,  .1226E-06,  .2633E-06,  .3439E-01,  &
   30000.,    905.1,    207.6,  .5169E-07,  .1178E-06,  .1519E-01,  &
   35000.,    417.1,    213.9,  .2317E-07,  .7227E-07,  .6804E-02,  &
   40000.,    199.0,    225.6,  .1045E-07,  .3221E-07,  .3075E-02,  &
   45000.,     98.8,    237.7,  .4933E-08,  .1021E-07,  .1449E-02,  &
   50000.,     50.8,    248.2,  .2412E-08,  .3378E-08,  .7094E-03,  &
   70000.,      3.5,    235.3,  .1791E-09,  .6756E-10,  .5259E-04,  &
  103000.,       .1,    201.2,  .1571E-11,  .3378E-13,  .4617E-06/
!  arctic summer
data ((mcdat(ilev,2,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101250.0,    278.1,  .9164E-02,  .4935E-07,  .1265E+01,  &
    1000.,  89502.0,    275.5,  .5963E-02,  .5366E-07,  .1129E+01,  &
    2000.,  79020.0,    272.9,  .4173E-02,  .5564E-07,  .1007E+01,  &
    3000.,  69671.0,    268.4,  .2664E-02,  .5743E-07,  .9030E+00,  &
    4000.,  61250.0,    261.9,  .1630E-02,  .5926E-07,  .8141E+00,  &
    5000.,  53667.0,    255.4,  .9583E-03,  .6303E-07,  .7317E+00,  &
    6000.,  46862.0,    248.9,  .5328E-03,  .6966E-07,  .6558E+00,  &
    7000.,  40778.0,    242.4,  .2829E-03,  .7316E-07,  .5861E+00,  &
    8000.,  35349.0,    235.9,  .1262E-03,  .7668E-07,  .5221E+00,  &
    9000.,  30525.0,    229.4,  .4040E-04,  .1063E-06,  .4636E+00,  &
   10000.,  26261.0,    226.7,  .1681E-04,  .1249E-06,  .4036E+00/
data ((mcdat(ilev,2,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  22598.0,    227.7,  .8268E-05,  .1739E-06,  .3458E+00,  &
   12000.,  19460.0,    228.6,  .4072E-05,  .2036E-06,  .2965E+00,  &
   13000.,  16770.0,    229.6,  .2006E-05,  .2532E-06,  .2544E+00,  &
   14000.,  12469.0,    230.1,  .7442E-06,  .2043E-06,  .1887E+00,  &
   15000.,  10752.0,    230.1,  .5726E-06,  .2358E-06,  .1628E+00,  &
   16000.,   9273.0,    230.1,  .4935E-06,  .2508E-06,  .1404E+00,  &
   17000.,   7999.0,    230.1,  .4274E-06,  .2899E-06,  .1211E+00,  &
   18000.,   6898.0,    230.1,  .3693E-06,  .3065E-06,  .1044E+00,  &
   19000.,   5950.0,    230.1,  .3198E-06,  .3085E-06,  .9007E-01,  &
   20000.,   5232.0,    230.1,  .2823E-06,  .3016E-06,  .7769E-01,  &
   21000.,   4428.0,    230.1,  .2395E-06,  .2746E-06,  .6702E-01/
data ((mcdat(ilev,2,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   3819.0,    230.1,  .2071E-06,  .2454E-06,  .5781E-01,  &
   23000.,   3295.0,    230.7,  .1788E-06,  .2312E-06,  .4976E-01,  &
   24000.,   2845.0,    231.9,  .1538E-06,  .2176E-06,  .4274E-01,  &
   25000.,   2459.0,    233.1,  .1330E-06,  .2035E-06,  .3674E-01,  &
   30000.,   1198.0,    239.1,  .9439E-07,  .1662E-06,  .1746E-01,  &
   35000.,    591.0,    251.6,  .3335E-07,  .8225E-07,  .8631E-02,  &
   40000.,    304.0,    266.9,  .1618E-07,  .3666E-07,  .4442E-02,  &
   45000.,    161.8,    278.9,  .8288E-08,  .1162E-07,  .2371E-02,  &
   50000.,     88.2,    281.8,  .4443E-08,  .3844E-08,  .1288E-02,  &
   70000.,      6.3,    220.6,  .4068E-09,  .7689E-10,  .9227E-04,  &
  104000.,       .1,    213.1,  .1788E-11,  .3844E-13,  .6525E-06/
!  sub-arctic winter
data ((mcdat(ilev,3,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101300.0,    257.1,  .1200E-02,  .4100E-07,  .1372E+01,  &
    1000.,  88780.0,    259.1,  .1200E-02,  .4100E-07,  .1193E+01,  &
    2000.,  77750.0,    256.4,  .1030E-02,  .4100E-07,  .1058E+01,  &
    3000.,  67980.0,    252.2,  .7470E-03,  .4300E-07,  .9366E+00,  &
    4000.,  59320.0,    246.8,  .4590E-03,  .4500E-07,  .8339E+00,  &
    5000.,  51580.0,    240.6,  .2340E-03,  .4700E-07,  .7457E+00,  &
    6000.,  44670.0,    233.9,  .9780E-04,  .4900E-07,  .6646E+00,  &
    7000.,  38530.0,    227.1,  .3290E-04,  .7100E-07,  .5904E+00,  &
    8000.,  33080.0,    220.4,  .1320E-04,  .9000E-07,  .5226E+00,  &
    9000.,  28290.0,    217.1,  .8370E-05,  .1600E-06,  .4538E+00,  &
   10000.,  24180.0,    217.1,  .5510E-05,  .2400E-06,  .3879E+00/
data ((mcdat(ilev,3,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  20670.0,    217.1,  .3790E-05,  .3200E-06,  .3315E+00,  &
   12000.,  17660.0,    217.1,  .2580E-05,  .4300E-06,  .2834E+00,  &
   13000.,  15100.0,    217.1,  .1670E-05,  .4700E-06,  .2422E+00,  &
   14000.,  12910.0,    217.1,  .1100E-05,  .4900E-06,  .2071E+00,  &
   15000.,  11030.0,    217.0,  .7330E-06,  .5600E-06,  .1770E+00,  &
   16000.,   9431.0,    216.7,  .6070E-06,  .6200E-06,  .1517E+00,  &
   17000.,   8058.0,    216.1,  .5200E-06,  .6200E-06,  .1300E+00,  &
   18000.,   6882.0,    215.5,  .4450E-06,  .6200E-06,  .1113E+00,  &
   19000.,   5875.0,    214.9,  .3810E-06,  .6000E-06,  .9529E-01,  &
   20000.,   5014.0,    214.3,  .3260E-06,  .5600E-06,  .8155E-01,  &
   21000.,   4277.0,    213.7,  .2790E-06,  .5100E-06,  .6976E-01/
data ((mcdat(ilev,3,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   3647.0,    213.1,  .2390E-06,  .4700E-06,  .5966E-01,  &
   23000.,   3109.0,    212.5,  .2040E-06,  .4300E-06,  .5100E-01,  &
   24000.,   2649.0,    212.0,  .1740E-06,  .3600E-06,  .4358E-01,  &
   25000.,   2256.0,    211.9,  .1490E-06,  .3200E-06,  .3722E-01,  &
   30000.,   1020.0,    216.6,  .6580E-07,  .1500E-06,  .1645E-01,  &
   35000.,    470.1,    223.1,  .2950E-07,  .9200E-07,  .7368E-02,  &
   40000.,    224.3,    235.3,  .1330E-07,  .4100E-07,  .3330E-02,  &
   45000.,    111.3,    247.9,  .6280E-08,  .1300E-07,  .1569E-02,  &
   50000.,     57.2,    258.9,  .3070E-08,  .4300E-08,  .7682E-03,  &
   70000.,      4.0,    245.4,  .2280E-09,  .8600E-10,  .5695E-04,  &
  103000.,       .1,    209.9,  .2000E-11,  .4300E-13,  .5000E-06/
!  sub-arctic summer
data ((mcdat(ilev,4,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101000.0,    287.0,  .9100E-02,  .4900E-07,  .1220E+01,  &
    1000.,  89600.0,    281.7,  .6000E-02,  .5400E-07,  .1110E+01,  &
    2000.,  79290.0,    276.4,  .4200E-02,  .5600E-07,  .9971E+00,  &
    3000.,  70000.0,    271.1,  .2690E-02,  .5800E-07,  .8985E+00,  &
    4000.,  61600.0,    265.7,  .1650E-02,  .6000E-07,  .8077E+00,  &
    5000.,  54100.0,    259.8,  .9730E-03,  .6400E-07,  .7244E+00,  &
    6000.,  47300.0,    252.8,  .5430E-03,  .7100E-07,  .6519E+00,  &
    7000.,  41300.0,    245.8,  .2900E-03,  .7500E-07,  .5849E+00,  &
    8000.,  35900.0,    238.8,  .1300E-03,  .7900E-07,  .5231E+00,  &
    9000.,  31070.0,    231.8,  .4180E-04,  .1100E-06,  .4663E+00,  &
   10000.,  26770.0,    225.6,  .1750E-04,  .1300E-06,  .4142E+00/
data ((mcdat(ilev,4,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  23000.0,    225.0,  .8560E-05,  .1800E-06,  .3559E+00,  &
   12000.,  19770.0,    225.0,  .4200E-05,  .2100E-06,  .3059E+00,  &
   13000.,  17000.0,    225.0,  .2060E-05,  .2600E-06,  .2630E+00,  &
   14000.,  14600.0,    225.0,  .1020E-05,  .2800E-06,  .2260E+00,  &
   15000.,  12500.0,    225.0,  .7770E-06,  .3200E-06,  .1943E+00,  &
   16000.,  10801.0,    225.0,  .6690E-06,  .3400E-06,  .1671E+00,  &
   17000.,   9280.0,    225.0,  .5750E-06,  .3900E-06,  .1436E+00,  &
   18000.,   7980.0,    225.0,  .4940E-06,  .4100E-06,  .1235E+00,  &
   19000.,   6860.0,    225.0,  .4250E-06,  .4100E-06,  .1062E+00,  &
   20000.,   5890.0,    225.0,  .3650E-06,  .3900E-06,  .9128E-01,  &
   21000.,   5070.0,    225.0,  .3140E-06,  .3600E-06,  .7849E-01/
data ((mcdat(ilev,4,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   4360.0,    225.1,  .2700E-06,  .3200E-06,  .6750E-01,  &
   23000.,   3750.0,    225.5,  .2320E-06,  .3000E-06,  .5805E-01,  &
   24000.,   3227.0,    226.6,  .1980E-06,  .2800E-06,  .4963E-01,  &
   25000.,   2780.0,    227.9,  .1700E-06,  .2600E-06,  .4247E-01,  &
   30000.,   1340.0,    234.9,  .7950E-07,  .1400E-06,  .1338E-01,  &
   35000.,    661.0,    247.2,  .3730E-07,  .9200E-07,  .6614E-02,  &
   40000.,    340.0,    262.3,  .1810E-07,  .4100E-07,  .3404E-02,  &
   45000.,    181.0,    274.1,  .9270E-08,  .1300E-07,  .1817E-02,  &
   50000.,     98.7,    276.9,  .4970E-08,  .4300E-08,  .9868E-03,  &
   70000.,      7.1,    216.8,  .4550E-09,  .8600E-10,  .7071E-04,  &
  104000.,       .1,    209.4,  .2000E-11,  .4300E-13,  .5000E-06/
!  mid-latitude winter
data ((mcdat(ilev,5,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101800.0,    272.2,  .3500E-02,  .6000E-07,  .1301E+01,  &
    1000.,  89730.0,    268.7,  .2500E-02,  .5400E-07,  .1162E+01,  &
    2000.,  78970.0,    265.2,  .1800E-02,  .4900E-07,  .1037E+01,  &
    3000.,  69380.0,    261.2,  .1160E-02,  .4900E-07,  .9230E+00,  &
    4000.,  60810.0,    255.7,  .6900E-03,  .4900E-07,  .8282E+00,  &
    5000.,  53130.0,    249.6,  .3780E-03,  .5800E-07,  .7411E+00,  &
    6000.,  46270.0,    243.6,  .1890E-03,  .6400E-07,  .6614E+00,  &
    7000.,  40160.0,    237.6,  .8570E-04,  .7700E-07,  .5886E+00,  &
    8000.,  34730.0,    231.6,  .3500E-04,  .9000E-07,  .5222E+00,  &
    9000.,  29920.0,    225.6,  .1600E-04,  .1200E-06,  .4619E+00,  &
   10000.,  25680.0,    220.6,  .7500E-05,  .1600E-06,  .4072E+00/
data ((mcdat(ilev,5,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  21990.0,    219.2,  .4440E-05,  .2100E-06,  .3496E+00,  &
   12000.,  18820.0,    218.7,  .2720E-05,  .2600E-06,  .2999E+00,  &
   13000.,  16100.0,    218.2,  .1720E-05,  .3000E-06,  .2572E+00,  &
   14000.,  13780.0,    217.7,  .1130E-05,  .3200E-06,  .2206E+00,  &
   15000.,  11780.0,    217.2,  .7640E-06,  .3400E-06,  .1890E+00,  &
   16000.,  10070.0,    216.7,  .6480E-06,  .3600E-06,  .1620E+00,  &
   17000.,   8610.0,    216.2,  .5550E-06,  .3900E-06,  .1388E+00,  &
   18000.,   7350.0,    215.7,  .4750E-06,  .4100E-06,  .1188E+00,  &
   19000.,   6280.0,    215.4,  .4060E-06,  .4300E-06,  .1017E+00,  &
   20000.,   5370.0,    215.2,  .3040E-06,  .4500E-06,  .8690E-01,  &
   21000.,   4580.0,    215.2,  .2970E-06,  .4300E-06,  .7421E-01/
data ((mcdat(ilev,5,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   3910.0,    215.2,  .2530E-06,  .4300E-06,  .6338E-01,  &
   23000.,   3340.0,    215.2,  .2160E-06,  .3900E-06,  .5415E-01,  &
   24000.,   2860.0,    215.2,  .1850E-06,  .3600E-06,  .4624E-01,  &
   25000.,   2430.0,    215.4,  .1570E-06,  .3400E-06,  .3950E-01,  &
   30000.,   1110.0,    217.3,  .7120E-07,  .1900E-06,  .1783E-01,  &
   35000.,    518.0,    227.9,  .3170E-07,  .9200E-07,  .7924E-02,  &
   40000.,    253.0,    244.0,  .1450E-07,  .4100E-07,  .3625E-02,  &
   45000.,    129.0,    258.9,  .6940E-08,  .1300E-07,  .1741E-02,  &
   50000.,     68.2,    265.6,  .3580E-08,  .4300E-08,  .8954E-03,  &
   70000.,      4.7,    230.9,  .2820E-09,  .8600E-10,  .7051E-04,  &
  103000.,       .1,    210.1,  .1990E-11,  .4300E-13,  .5000E-06/
!  mid-latitude summer
data ((mcdat(ilev,6,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101300.0,    294.0,  .1400E-01,  .6000E-07,  .1191E+01,  &
    1000.,  90200.0,    290.0,  .9300E-02,  .6000E-07,  .1080E+01,  &
    2000.,  80200.0,    285.0,  .5850E-02,  .6000E-07,  .9757E+00,  &
    3000.,  71000.0,    279.0,  .3430E-02,  .6200E-07,  .8846E+00,  &
    4000.,  62800.0,    273.0,  .1890E-02,  .6400E-07,  .7998E+00,  &
    5000.,  55400.0,    267.1,  .1000E-02,  .6600E-07,  .7211E+00,  &
    6000.,  48700.0,    261.0,  .6090E-03,  .6900E-07,  .6487E+00,  &
    7000.,  42600.0,    254.7,  .3710E-03,  .7500E-07,  .5830E+00,  &
    8000.,  37200.0,    248.2,  .2100E-03,  .7900E-07,  .5225E+00,  &
    9000.,  32400.0,    241.7,  .1180E-03,  .8600E-07,  .4669E+00,  &
   10000.,  28100.0,    235.2,  .6430E-04,  .9000E-07,  .4159E+00/
data ((mcdat(ilev,6,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  24300.0,    228.8,  .2190E-04,  .1100E-06,  .3693E+00,  &
   12000.,  20900.0,    222.3,  .6460E-05,  .1200E-06,  .3269E+00,  &
   13000.,  17900.0,    216.9,  .1660E-05,  .1500E-06,  .2882E+00,  &
   14000.,  15300.0,    215.8,  .9950E-06,  .1800E-06,  .2464E+00,  &
   15000.,  13000.0,    215.8,  .8400E-06,  .1900E-06,  .2104E+00,  &
   16000.,  11000.0,    215.8,  .7100E-06,  .2100E-06,  .1797E+00,  &
   17000.,   9500.0,    215.8,  .6140E-06,  .2400E-06,  .1535E+00,  &
   18000.,   8120.0,    216.0,  .5240E-06,  .2800E-06,  .1305E+00,  &
   19000.,   6950.0,    217.0,  .4460E-06,  .3200E-06,  .1110E+00,  &
   20000.,   5950.0,    218.2,  .3800E-06,  .3400E-06,  .9453E-01,  &
   21000.,   5100.0,    219.4,  .3240E-06,  .3600E-06,  .8056E-01/
data ((mcdat(ilev,6,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   4370.0,    220.6,  .2760E-06,  .3600E-06,  .6872E-01,  &
   23000.,   3760.0,    221.8,  .2360E-06,  .3400E-06,  .5867E-01,  &
   24000.,   3220.0,    223.0,  .2010E-06,  .3200E-06,  .5014E-01,  &
   25000.,   2770.0,    224.2,  .1720E-06,  .3000E-06,  .4288E-01,  &
   30000.,   1320.0,    234.2,  .7850E-07,  .2000E-06,  .1322E-01,  &
   35000.,    652.0,    245.3,  .3700E-07,  .9200E-07,  .6519E-02,  &
   40000.,    333.0,    257.5,  .1800E-07,  .4100E-07,  .3330E-02,  &
   45000.,    176.0,    269.7,  .9090E-08,  .1300E-07,  .1757E-02,  &
   50000.,     95.1,    276.2,  .4800E-08,  .4300E-08,  .9512E-03,  &
   70000.,      6.7,    219.1,  .4270E-09,  .8600E-10,  .6706E-04,  &
  104000.,       .1,    209.9,  .1990E-11,  .4300E-13,  .5000E-06/
!  subtropical winter
data ((mcdat(ilev,7,ifld),ifld=1,6),ilev=1,11)/  &
       0., 102100.0,    287.1,  .1125E-01,  .5800E-07,  .1233E+01,  &
    1000.,  90659.0,    284.2,  .7750E-02,  .5500E-07,  .1107E+01,  &
    2000.,  80378.0,    281.2,  .5545E-02,  .5150E-07,  .9934E+00,  &
    3000.,  71125.0,    274.7,  .2930E-02,  .5000E-07,  .9006E+00,  &
    4000.,  62740.0,    268.2,  .1675E-02,  .4800E-07,  .8142E+00,  &
    5000.,  55176.0,    261.7,  .9540E-03,  .5150E-07,  .7340E+00,  &
    6000.,  48367.0,    255.2,  .5245E-03,  .5350E-07,  .6599E+00,  &
    7000.,  42254.0,    248.8,  .2783E-03,  .5900E-07,  .5916E+00,  &
    8000.,  36786.0,    242.3,  .1425E-03,  .6450E-07,  .5289E+00,  &
    9000.,  31906.0,    235.8,  .6850E-04,  .7950E-07,  .4713E+00,  &
   10000.,  27563.0,    229.3,  .2825E-04,  .9950E-07,  .4187E+00/
data ((mcdat(ilev,7,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  23716.0,    222.9,  .1117E-04,  .1255E-06,  .3707E+00,  &
   12000.,  20315.0,    216.4,  .4400E-05,  .1515E-06,  .3270E+00,  &
   13000.,  17344.0,    213.7,  .1755E-05,  .1725E-06,  .2828E+00,  &
   14000.,  14781.0,    211.1,  .1058E-05,  .1825E-06,  .2439E+00,  &
   15000.,  12557.0,    208.5,  .7605E-06,  .1935E-06,  .2101E+00,  &
   16000.,  10671.0,    205.9,  .6425E-06,  .2035E-06,  .1805E+00,  &
   17000.,   9041.0,    203.3,  .5485E-06,  .2295E-06,  .1549E+00,  &
   18000.,   7651.0,    203.1,  .4615E-06,  .2500E-06,  .1311E+00,  &
   19000.,   6480.0,    205.4,  .3880E-06,  .2850E-06,  .1099E+00,  &
   20000.,   5498.0,    207.9,  .3060E-06,  .3200E-06,  .9213E-01,  &
   21000.,   4676.0,    210.4,  .2770E-06,  .3350E-06,  .7743E-01/
data ((mcdat(ilev,7,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   3984.0,    212.9,  .2345E-06,  .3550E-06,  .6520E-01,  &
   23000.,   3401.0,    214.9,  .1995E-06,  .3550E-06,  .5512E-01,  &
   24000.,   2907.0,    216.9,  .1700E-06,  .3500E-06,  .4669E-01,  &
   25000.,   2489.0,    218.9,  .1440E-06,  .3400E-06,  .3961E-01,  &
   30000.,   1169.0,    228.8,  .6535E-07,  .2150E-06,  .1780E-01,  &
   35000.,    568.0,    239.8,  .2980E-07,  .9200E-07,  .8255E-02,  &
   40000.,    286.0,    251.6,  .1405E-07,  .4100E-07,  .3960E-02,  &
   45000.,    148.8,    263.4,  .6870E-08,  .1300E-07,  .1967E-02,  &
   50000.,     79.4,    269.1,  .3580E-08,  .4300E-08,  .1027E-02,  &
   70000.,      5.4,    221.7,  .2905E-09,  .8600E-10,  .8440E-04,  &
  103000.,       .2,    191.1,  .1805E-11,  .4300E-13,  .3422E-05/
!  subtropical summer
data ((mcdat(ilev,8,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101350.0,    301.1,  .1650E-01,  .5800E-07,  .1159E+01,  &
    1000.,  90464.0,    293.7,  .1115E-01,  .5800E-07,  .1066E+01,  &
    2000.,  80504.0,    288.2,  .7570E-02,  .5700E-07,  .9686E+00,  &
    3000.,  71484.0,    282.7,  .4065E-02,  .5650E-07,  .8776E+00,  &
    4000.,  63311.0,    277.2,  .2275E-02,  .5550E-07,  .7937E+00,  &
    5000.,  55936.0,    271.7,  .1265E-02,  .5550E-07,  .7159E+00,  &
    6000.,  49292.0,    266.3,  .7345E-03,  .5600E-07,  .6443E+00,  &
    7000.,  43304.0,    259.3,  .4210E-03,  .5800E-07,  .5814E+00,  &
    8000.,  37913.0,    252.3,  .2300E-03,  .5900E-07,  .5233E+00,  &
    9000.,  33068.0,    245.3,  .1195E-03,  .6250E-07,  .4694E+00,  &
   10000.,  28729.0,    238.4,  .5665E-04,  .6450E-07,  .4198E+00/
data ((mcdat(ilev,8,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  24858.0,    231.4,  .1990E-04,  .7550E-07,  .3742E+00,  &
   12000.,  21414.0,    224.4,  .6270E-05,  .8150E-07,  .3324E+00,  &
   13000.,  18359.0,    217.5,  .1725E-05,  .9750E-07,  .2941E+00,  &
   14000.,  15665.0,    210.5,  .9905E-06,  .1125E-06,  .2953E+00,  &
   15000.,  13295.0,    203.5,  .7985E-06,  .1185E-06,  .2276E+00,  &
   16000.,  11248.0,    203.1,  .6735E-06,  .1285E-06,  .1929E+00,  &
   17000.,   9526.0,    205.2,  .5780E-06,  .1545E-06,  .1617E+00,  &
   18000.,   8081.0,    207.4,  .4860E-06,  .1850E-06,  .1358E+00,  &
   19000.,   6868.0,    209.6,  .4080E-06,  .2300E-06,  .1142E+00,  &
   20000.,   5846.0,    211.8,  .3440E-06,  .2650E-06,  .9618E-01,  &
   21000.,   4986.0,    213.9,  .2905E-06,  .3000E-06,  .8119E-01/
data ((mcdat(ilev,8,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   4258.0,    215.9,  .2460E-06,  .3200E-06,  .6870E-01,  &
   23000.,   3643.0,    217.9,  .2095E-06,  .3300E-06,  .5823E-01,  &
   24000.,   3121.0,    219.9,  .1780E-06,  .3300E-06,  .4944E-01,  &
   25000.,   2677.0,    221.9,  .1515E-06,  .3200E-06,  .4203E-01,  &
   30000.,   1270.0,    231.8,  .6900E-07,  .2200E-06,  .1909E-01,  &
   35000.,    622.9,    242.8,  .3245E-07,  .9200E-07,  .8939E-02,  &
   40000.,    316.2,    254.6,  .1580E-07,  .4100E-07,  .4327E-02,  &
   45000.,    165.7,    266.4,  .7945E-08,  .1300E-07,  .2167E-02,  &
   50000.,     89.1,    272.1,  .4190E-08,  .4300E-08,  .1140E-03,  &
   70000.,      6.1,    217.6,  .3630E-09,  .8600E-10,  .9739E-04,  &
  103000.,       .2,    180.1,  .1805E-11,  .4300E-13,  .3472E-05/
!  tropical
data ((mcdat(ilev,9,ifld),ifld=1,6),ilev=1,11)/  &
       0., 101300.0,    300.0,  .1900E-01,  .5600E-07,  .1167E+01,  &
    1000.,  90400.0,    294.1,  .1300E-01,  .5600E-07,  .1064E+01,  &
    2000.,  80500.0,    288.4,  .9290E-02,  .5400E-07,  .9689E+00,  &
    3000.,  71500.0,    283.6,  .4700E-02,  .5100E-07,  .8756E+00,  &
    4000.,  63300.0,    277.4,  .2660E-02,  .4700E-07,  .7951E+00,  &
    5000.,  55900.0,    270.7,  .1530E-02,  .4500E-07,  .7199E+00,  &
    6000.,  49200.0,    264.0,  .8600E-03,  .4300E-07,  .6501E+00,  &
    7000.,  43200.0,    257.3,  .4710E-03,  .4100E-07,  .5855E+00,  &
    8000.,  37800.0,    250.6,  .2500E-03,  .3900E-07,  .5258E+00,  &
    9000.,  32900.0,    243.8,  .1210E-03,  .3900E-07,  .4708E+00,  &
   10000.,  28600.0,    237.2,  .4900E-04,  .3900E-07,  .4202E+00/
data ((mcdat(ilev,9,ifld),ifld=1,6),ilev=12,22)/  &
   11000.,  24700.0,    230.4,  .1790E-04,  .4100E-07,  .3740E+00,  &
   12000.,  21300.0,    223.8,  .6080E-05,  .4300E-07,  .3316E+00,  &
   13000.,  18200.0,    217.0,  .1790E-05,  .4500E-07,  .2929E+00,  &
   14000.,  15600.0,    210.4,  .9860E-06,  .4500E-07,  .2578E+00,  &
   15000.,  13200.0,    203.6,  .7570E-06,  .4700E-07,  .2260E+00,  &
   16000.,  11100.0,    196.8,  .6370E-06,  .4700E-07,  .1972E+00,  &
   17000.,   9370.0,    195.6,  .5420E-06,  .6900E-07,  .1676E+00,  &
   18000.,   7890.0,    199.5,  .4480E-06,  .9000E-07,  .1382E+00,  &
   19000.,   6660.0,    203.6,  .3700E-06,  .1400E-06,  .1145E+00,  &
   20000.,   5650.0,    207.6,  .3080E-06,  .1900E-06,  .9515E-01,  &
   21000.,   4800.0,    211.5,  .2570E-06,  .2400E-06,  .7938E-01/
data ((mcdat(ilev,9,ifld),ifld=1,6),ilev=23,33)/  &
   22000.,   4090.0,    214.6,  .2160E-06,  .2800E-06,  .6645E-01,  &
   23000.,   3500.0,    216.9,  .1830E-06,  .3200E-06,  .5618E-01,  &
   24000.,   3000.0,    219.1,  .1550E-06,  .3400E-06,  .4763E-01,  &
   25000.,   2570.0,    221.3,  .1310E-06,  .3400E-06,  .4045E-01,  &
   30000.,   1220.0,    232.3,  .5950E-07,  .2400E-06,  .1831E-01,  &
   35000.,    600.0,    243.3,  .2790E-07,  .9200E-07,  .8600E-02,  &
   40000.,    305.0,    254.3,  .1360E-07,  .4100E-07,  .4181E-02,  &
   45000.,    159.0,    264.9,  .6800E-08,  .1300E-07,  .2097E-02,  &
   50000.,     85.4,    270.0,  .3580E-08,  .4300E-08,  .1101E-02,  &
   70000.,      5.8,    219.5,  .2990E-09,  .8600E-10,  .9210E-04,  &
  103000.,       .1,    209.9,  .1620E-11,  .4300E-13,  .5000E-06/

save

! field # from Mclatchy Soundings
!    1      : Height (m)
!    2      : pressure (Pa)
!    3      : potential temp (K)
!    4      : vapor density (kg/m^3)
!    5      : ozone density (kg/m^3)
!    6      : air density (kg/m^3)

if (iaction .eq. 1) then

   rgasog = rgas / g

! Copy (since no time interpolation needed) tropical sounding to mclat array
! (done only once on each node).

   do lv = 1,33
      do lf = 1,6
         mclat(lv,5,lf) = mcdat(lv,9,lf)
      enddo
   enddo

! Compute number of levels to be added above model top from Mclatchy soundings.
! Base this number on prsnz and prsnzp, which was earlier computed from
! PI01DN(NNZP(1),1), so that it is the same for all compute nodes
! (done only once on each node).

   if (prsnz .lt. 3000.) then

! If prsnz, the pressure at the highest model prognostic level, is less
! than 3000 Pa, do not add any levels.

      narad = 0

   else

! If prsnzp is greater than 3000 Pa, add one or more radiation levels.
! Make the top level be 1500 Pa.

      deltap = max(1500.,  &
                   (prsnz-prsnzp),  &
                   (prsnz-1500.1)/float(namax))
      narad = max(1,int((prsnz-1500.)/deltap))

   endif

elseif (iaction .eq. 2) then

! Interpolate arctic, sub-arctic, mid-latitude, and subtropical, Mclatchy
! soundings between summer and winter values by time of year using cosine
! func.  Assume that extreme values occur on January 16 and 1/2 year later.
! (Done once per grid/node each radiation time.)

   fjday = float(jday)
   wtjan = 0.5 * (1. + cos(6.283185 * (fjday-16.) / 365.))
   wtjul = 1. - wtjan

   do lats = 1,4
      latn = 10 - lats
      isummer = 2 * lats
      iwinter = isummer - 1
      do lv = 1,33
         do lf = 1,6
            mclat(lv,lats,lf) = wtjan * mcdat(lv,isummer,lf)  &
                              + wtjul * mcdat(lv,iwinter,lf)
            mclat(lv,latn,lf) = wtjan * mcdat(lv,iwinter,lf)  &
                              + wtjul * mcdat(lv,isummer,lf)
         enddo
      enddo
   enddo

! Adjust solar fluxes at top of atmosphere by current Earth-Sun distance

   do is = 1,nsolb
      solar1(is) = solar0(is) * solfac
   enddo

elseif (iaction .eq. 3) then

! At this point, subtropical, mid-latitude, sub-arctic, and arctic Mclatchy
! soundings have been interpolated between summer and winter values by time
! of year.  In this section of code, interpolate these 4 plus the all-year
! tropical sounding by latitude for the current i,j column in the grid.

   do ind = 1,11
      index = ind
      if (glat .lt. slat(index+1)) go to 10
   enddo
10      continue

   lat = latind(index)

   if (index .eq. 1 .or. index .eq. 6 .or. index .eq. 11) then

! For pure arctic or tropical latitudes, assign sounding values without
! interpolation.

      do lv = 1,33
         do lf = 1,6
            mcol(lv,lf) = mclat(lv,lat,lf)
         enddo
      enddo

   else

! For other latitudes, linearly interpolate between soundings according
! to latitude bands defined in array `slat'.

      wtnorth = (glat - slat(index)) / (slat(index + 1) - slat(index))
      wtsouth = 1. - wtnorth
      do lv = 1,33
         do lf = 1,6
            mcol(lv,lf) = wtsouth * mclat(lv,lat,lf)  &
                        + wtnorth * mclat(lv,lat+1,lf)
         enddo
      enddo
   endif

! Fill radiation column arrays (other than o3l) with variables from the model
! grid.


   do k = 1,m1-1
      zml(k) = topt + zm(k) * rtgt
      ztl(k) = topt + zt(k) * rtgt

      pl(k) = press(k)
      tl(k) = tair(k)
      dl(k) = dn0(k)
      rl(k) = rv(k) * dn0(k)
   enddo

! Compute pressures and heights of added levels for this column.

   deltap = (pl(m1-1) - 1500.) / float(narad)

   do k = m1,nrad
      pl(k) = pl(k-1) - deltap
   enddo

! Interpolate O3 from Mclatchy sounding to all levels in radiation column,
! and interpolate other variables (temperature, density, vapor mixing ratio)
! to added levels.

   lv = 1
   do k = 1,nrad
30       continue
      if (pl(k) .gt. mcol(1,2)) then
         o3l(k) = mcol(1,5)
         if (k .ge. m1) then
            tl(k) = mcol(1,3)
            rl(k) = mcol(1,4)
            dl(k) = mcol(1,6)
         endif
      elseif (pl(k) .le. mcol(lv,2) .and. pl(k) .ge. mcol(lv+1,2)) then
         wt = (pl(k) - mcol(lv,2)) / (mcol(lv+1,2) - mcol(lv,2))
         o3l(k) = mcol(lv,5) + (mcol(lv+1,5) - mcol(lv,5)) * wt
         if (k .ge. m1) then
            tl(k) = mcol(lv,3) + (mcol(lv+1,3) - mcol(lv,3)) * wt
            rl(k) = mcol(lv,4) + (mcol(lv+1,4) - mcol(lv,4)) * wt
            dl(k) = mcol(lv,6) + (mcol(lv+1,6) - mcol(lv,6)) * wt
         endif
      elseif (pl(k) .lt. mcol(33,2)) then
         o3l(k) = mcol(33,5)
         if (k .ge. m1) then
            tl(k) = mcol(33,3)
            rl(k) = mcol(33,4)
            dl(k) = mcol(33,6)
         endif
      elseif(pl(k) .lt. mcol(lv+1,2)) then
         lv = lv + 1
         if (lv .ge. 33) stop 'mclat1'
         go to 30
      else
         print*, 'pressure data improperly ordered'
         stop 'mclat2'
      endif
   enddo


! Compute heights of added levels by hydrostatic integration.

   do k = m1,nrad
      ztl(k) = ztl(k-1) + rgasog * (tl(k-1) - tl(k)  &
         + (pl(k-1) * tl(k) - pl(k) * tl(k-1)) / (pl(k) - pl(k-1))  &
         * log(pl(k)/pl(k-1)))
      zml(k-1) = .5 * (ztl(k) + ztl(k-1))
   enddo
   zml(nrad) = 2. * ztl(nrad) - zml(nrad-1)

! Compute dzl values.

   do k = 2,nrad
      dzl(k) = zml(k) - zml(k-1)
   enddo
   dzl(1)  = zml(2)

! Fill surface values

!   pl(1) = .5 * (press(1) + press(2))
   pl(1) = press(2) + (zm(1) - zt(3))  &
     / (zt(2) - zt(3)) * (press(2) - press(3))
   tl(1) = sqrt(sqrt(rlongup / stefan))

endif

return
END SUBROUTINE mclatchy







