/* (c) 2017 Open Source Geospatial Foundation - all rights reserved
 * This code is licensed under the GPL 2.0 license, available at the root
 * application directory.
 */
package org.geoserver.importer.rest.converters;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import net.sf.json.JSONObject;
import org.geoserver.importer.ImportTask;
import org.geoserver.importer.Importer;
import org.geoserver.importer.rest.converters.ImportJSONWriter.FlushableJSONBuilder;
import org.geoserver.rest.converters.BaseMessageConverter;
import org.geoserver.rest.util.MediaTypeExtensions;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.HttpOutputMessage;
import org.springframework.http.MediaType;
import org.springframework.http.converter.HttpMessageNotReadableException;
import org.springframework.http.converter.HttpMessageNotWritableException;
import org.springframework.stereotype.Component;

/** Convert {@link ImportTask} to/from JSON. */
@Component
public class ImportTaskJSONMessageConverter extends BaseMessageConverter<ImportTask> {

    Importer importer;

    @Autowired
    public ImportTaskJSONMessageConverter(Importer importer) {
        super(MediaType.APPLICATION_JSON, MediaTypeExtensions.TEXT_JSON);
        this.importer = importer;
    }

    @Override
    public int getPriority() {
        return super.getPriority() - 5;
    }

    @Override
    protected boolean supports(Class<?> clazz) {
        return ImportTask.class.isAssignableFrom(clazz);
    }

    //
    // Reading
    //
    @Override
    protected ImportTask readInternal(
            Class<? extends ImportTask> clazz, HttpInputMessage inputMessage)
            throws IOException, HttpMessageNotReadableException {
        try (InputStream in = inputMessage.getBody()) {
            ImportJSONReader reader = new ImportJSONReader(importer);
            JSONObject json = reader.parse(in);
            ImportTask task = reader.task(json);

            return task;
        }
    }

    //
    // writing
    //
    @Override
    @SuppressWarnings("PMD.CloseResource") // OS is managed by servlet container
    protected void writeInternal(ImportTask task, HttpOutputMessage outputMessage)
            throws IOException, HttpMessageNotWritableException {
        OutputStreamWriter outputStream = new OutputStreamWriter(outputMessage.getBody());
        FlushableJSONBuilder json = new FlushableJSONBuilder(outputStream);
        ImportJSONWriter writer = new ImportJSONWriter(importer);
        int expand = writer.expand(1);

        writer.task(json, task, true, expand);

        outputStream.flush();
    }
}
