--
-- Create the database for your local postgres machine
--
DROP SCHEMA IF EXISTS gw_beleid CASCADE;

CREATE SCHEMA gw_beleid;

SELECT set_config('search_path', 'gw_beleid,' || current_setting('search_path'), false);

CREATE TABLE codetable_country (
    code character varying  NOT NULL,
    country character varying NOT NULL
);

ALTER TABLE ONLY codetable_country
    ADD CONSTRAINT codetable_country_pkey PRIMARY KEY (code);

CREATE TABLE grondwaterlichamen_new (
    dataengine_id integer NOT NULL,
    gwl character varying (2058),
    shape geometry(MultiPolygon,31370),
    CONSTRAINT unique_on_gwl UNIQUE (gwl)
);

CREATE VIEW vw_grondwaterlichamen AS
 SELECT grondwaterlichamen_new.dataengine_id,
    grondwaterlichamen_new.gwl,
    grondwaterlichamen_new.shape,
    (select country FROM gw_beleid.codetable_country WHERE gw_beleid.codetable_country.code = 'BE') AS country
   FROM grondwaterlichamen_new
  WHERE ((grondwaterlichamen_new.gwl)::text ~~ 'BL%'::text);


CREATE VIEW vw_grondwaterlichamen_generated_id AS
 SELECT grondwaterlichamen_new.dataengine_id as generated_id,
    grondwaterlichamen_new.gwl,
    grondwaterlichamen_new.shape,
    (select country FROM gw_beleid.codetable_country WHERE gw_beleid.codetable_country.code = 'x') AS country
   FROM grondwaterlichamen_new
  WHERE ((grondwaterlichamen_new.gwl)::text ~~ 'BL%'::text);


CREATE VIEW "vw_GrondwaterlichamenCamelCase" AS
 SELECT grondwaterlichamen_new.dataengine_id as "Generated_Id",
    grondwaterlichamen_new.gwl as "Gwl",
    grondwaterlichamen_new.shape as "TheGeom"
   FROM grondwaterlichamen_new
  WHERE ((grondwaterlichamen_new.gwl)::text ~~ 'BL%'::text);


ALTER TABLE ONLY grondwaterlichamen_new
    ADD CONSTRAINT grondwaterlichamen_new_pkey PRIMARY KEY (dataengine_id);

CREATE INDEX grondwaterlichamen_new_sidx ON grondwaterlichamen_new USING gist (shape);

CREATE INDEX multipleColumns ON gw_beleid.grondwaterlichamen_new (dataengine_id, gwl);


--
-- Data for Name: grondwaterlichamen_new;
--

INSERT INTO grondwaterlichamen_new VALUES (1, '                                             <BR> CVS_0800_GWL_1 Ieperiaan Aquifer, Freatisch                      <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen        <BR> SS_1300_GWL_5 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter', '01060000208A7A0000030000000103000000010000002F000000008A4160B3ADFF40003D9BD5508006410000000060ADFF4000992A1825800641006E3480D7ACFF40807D1DB8AB800641005227A00EADFF40004703F8EF8006410040A4DF74ACFF4000470378658106410076BE9FE4ABFF4080F4DB5752830641000A8A1F58A9FF40806210D8F38206410000000020A7FF4000992A185083064100CAE53F13A4FF4000D0445876820641005227A06AA2FF40004703F8EF800641001C0DE047A2FF40004703F83B80064100000000E0A0FF40806210582980064100361AC0DF9DFF4000992A18507F0641002497FF4C9CFF40004703F8CF7D06410076BE9F259CFF400010E9370D7D064100000000A09AFF4000992A98F67C0641002E90A0A798FF40807D1DB8D37C064100B6627F9C96FF400010E9B7B17C0641000000006094FF4000470378897C0641006E34801092FF40807D1DB8D77B0641002E90A0FD90FF40004703F8AF7A0641002E90A0D290FF40807D1DB85679064100CAE53FC590FF40802BF697E2780641008A41609B90FF40004703F88F7706410076BE9F1891FF40807D1DB8137606410076BE9FDF90FF40807D1DB8CF750641000A8A1F6E93FF40004703F86F74064100764F1E6093FF40008273C66A74064100EE0DBE9F7AFF400011C7BA52710641000022FD7260FF4000F46CD6DA680641009CC4204232FF4080C8763E765E06410028A089D51BFF4000EBE2B6FC560641004850FC031DFF4000A4DF3EDF56064100F675E0DF36FF40804013E1045E0641007AC729344EFF4080C1A8A44463064100D066D55D64FF4080423E68846806410018B7D18773FF40803C2C54446A0641000EBE308781FF4080C3D32BC46D064100FCCBEE859DFF4000ED0D3E9973064100C420B08EA8FF4000AAF152EC73064100E4F21F18A9FF40807D1D385D74064100E4F21FF3A9FF40807D1D383875064100E4F21F80AAFF40807D1D38C575064100E4F21FB0ACFF40807D1D38E578064100E4F21F78ADFF40807D1D389D7A064100E4F21FC8ADFF40807D1D381D7D0641008A4160B3ADFF40003D9BD55080064101030000000100000006000000005C6DC56941FE4080923A0137A2064100DE9387023EFE40803F35DE51A7064100E8D9AC8434FE4080302AA9A8AC0641008AB0E17633FE408045B6F331AC064100DE718A253BFE4080B20C714EA70641005C6DC56941FE4080923A0137A206410103000000010000000400000000083D9BA747FE40802E6EA322970641007CD0B33947FE40009A081B7F99064100E09C111547FE4000728A0EB699064100083D9BA747FE40802E6EA322970641');
INSERT INTO grondwaterlichamen_new VALUES (2, '                                             <BR> CVS_0800_GWL_1 Ieperiaan Aquifer, Freatisch                      <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen      <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (4, '                                             <BR> CVS_0800_GWL_1 Ieperiaan Aquifer, Freatisch                    <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter        <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (5, '                                             <BR> CVS_0800_GWL_1 Ieperiaan Aquifer, Freatisch                    <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter      <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (6, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                              <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen      <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (3, '                                             <BR> CVS_0800_GWL_1 Ieperiaan Aquifer, Freatisch                      <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen    <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (93, '                                           <BR> CVS_0600_GWL_2 Ledo-Paniseliaan Aquifersysteem, Gespannen  <BR> CVS_0800_GWL_2 Ieperiaan Aquifer, Gespannen    <BR> KPS_0120_GWL_2 Duin- en kreekgebieden in de Oost-Vlaamse Polders, Freatisch  <BR> KPS_0160_GWL_2 verzilt Quartair en Oligoceen van Oost-Vlaamse Polders, Freatisch          <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen      <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (7, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                              <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen    <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (8, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                              <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen  <BR> SS_1300_GWL_2 Sokkel + Krijt Aquifersysteem, Lokaal freatisch, voedingsgebied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
INSERT INTO grondwaterlichamen_new VALUES (10, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                            <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter        <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen', '01060000208A7A0000030000000103000000010000000C010000008CB96B37DEEF40801D386716E6044100482EFF51CAEF4000992A18FEE50441006C3480F5C3EF408062105805E60441002C90A0DEA9EF4000D9CEF71FE804410080B740A497EF4080F4DB5700E9044100000000307CEF400010E9B717EE0441001C0DE08770EF4080F4DBD7B8F10441005CB1BFAC60EF4000992A18ECFC0441006C34802F5BEF4080BDC1177908054100A4DFBE6656EF400010E937D412054100F8065FB84BEF40807D1DB8D6180541001C0DE0B33CEF4080BDC117191A0541008048BF5932EF4080BDC117F219054100C05B202723EF4000D04458D218054100C05B208D19EF400010E9B7111805410094CB7F4EE7EE4080BDC1974E160541001483C0CED1EE4080B4377801160541006C3480A1C6EE4080BDC197751505410094CB7FCCA1EE40004703F83E120541006C3480AB8EEE400010E937BC11054100EC7C3F2D7DEE40807D1D38D011054100B8627F8B70EE40004703F80F1305410040A4DF8A61EE408062105885140541002C90A07E4FEE4080B437F8C717054100A4DFBE1246EE400010E9B769190541002497FFEC37EE40802BF697A91B05410040A4DF4216EE40806210D8C422054100C854C1DA09EE4000D9CE77FF2605410078BE9FE802EE40807D1DB8CF2905410014143FC8F8ED4080BDC197A32C05410040A4DF0CEDED4080F4DB57462F0541002497FF86D8ED4000992A985A310541005CB1BFC8C7ED400010E937D530054100D46F5F9BBCED4080621058493005410078BE9F9EB3ED40802BF697252D054100DC6800C7A9ED40806210583729054100088A1F9FABED400010E9B70D2205410088416041AAED40004703F8811A054100ECEBC0DBA2ED4000D9CE775C17054100B047E14890ED4080BDC117AB130541005C2041A176ED4000A2B43700110541002497FFEA54ED4000470378FA0C054100F875E02C47ED4000992A180F0A0541008048BFF148ED4080BDC117E701054100B8D100AC79ED40004703F821FA04410064AA60668AED40802BF69774FA044100482EFFB39CED4080B4377892FA044100B047E1F2BAED4080BDC197D5FA044100943A0137DDED40807D1DB8ACFB044100C05B2061F7ED408062105890FA04410064AA60682BEE40004703785EF40441000000005C36EE4080B43778EFF1044100EC7C3F834EEE40806210D844EB0441001C0DE0A16BEE40004703F829E7044100401361AD8EEE4080B437F844DD044100F8065F0EAAEE4000D044582FD704410080B740C0C5EE40807D1DB845D5044100A44E407BE8EE4080B437F8BCD10441001C0DE0B1FAEE400010E9B7A9D004410080B740FA0BEF400010E9B7CDCD044100D46F5F5111EF4000470378CCCA044100D4DEE02B0CEF4000470378DAC5044100088A1FE507EF40806210D8E5C1044100A44E4045F4EE400010E9B7D4BA044100D4DEE0E5EBEE4080F4DBD74DB5044100F875E084C5EE40802BF61729A8044100088A1FAFB1EE40802BF617509E04410078BE9F7695EE40802BF69745990441009CC420EC71EE4080F4DB579C9C0441002C90A06C45EE4080BDC117359D044100489D802820EE4080F4DBD7C39E04410080B740DA01EE4080F4DB57B59D044100482EFFA3DBED4080B43778489D044100EC7C3F91B0ED40004703F8AF9B0441005427A07391ED4080B437786F9A04410078BE9F4882ED4080F4DB57B599044100643BDF1973ED4080621058C8980441001483C08A60ED4080F4DBD74995044100DC6800AF39ED4000D9CEF7518C0441005CB1BF6421ED4000D9CE77AA8604410064AA60900BED4000470378FE81044100088A1F93F6EC40802BF617EA7D0441005CB1BF80EEEC4080F4DB575C7C044100381AC0DFC5EC4000992A18047104410050B81E5DAEEC40807D1DB8286B044100DC6800C773EC4000D0445844640441002C90A05451EC4080F4DBD70A610441006C34806521EC40802BF697465E0441002497FFC0F8EB4080B437F8435D0441002C90A044D7EB4000D9CEF7385C044100A44E404F77EB4000D9CE777F550441001C0DE0B142EB4080F4DBD7F1530441005CB1BFD21EEB40802BF697E952044100088A1F65F1EA4000992A98B95104410080B74068A8EA4000A2B4B73C54044100B84082D498EA408003E78C1C5604410094CB7F1C97EA40004703F8AB5504410050FC183F92EA400076E01C3D54044100EC7C3F9188EA400010E9376351044100C017261B7FEA40004C3789DE4E04410090ED7C1F7EEA4000D9CEF7864E0441003CDF4FED7DEA40806E1283754E044100F4285CBF7DEA40005A643B634E04410080D93D4F7DEA4080C117A6374E04410090C2F52078EA4080749318344C0441005817B70F78EA40806FF005014C044100A8F1D2F577EA4000B07268B44B04410048E17A5877EA4000A4703D774B0441009899995577EA40800C022B764B044100C0EC9E7A74EA4000CB10476F4A0441001C0DE08B73EA400010E9B798490441005CB1BF0E72EA40807D1DB8C047044100EC7C3F8370EA4080F4DB578E450441002497FF166FEA40802BF69791430441006C34801B6FEA40802BF697B1420441005427A0BF6EEA40004703F88F41044100E4F21F066FEA40004703F88B4004410094CB7FF26FEA40802BF697FD3E044100E4F21FB471EA4080F4DB570E3D044100C8E53F6273EA40802BF697D53B0441005CB1BF7E74EA40004703F8573A044100ACD85FAE76EA40807D1D3833390441001C0DE0F97AEA40806210589637044100ACD85F0880EA40802BF6970936044100C8E53F4284EA400010E9B79C340441006C3480158AEA40004703F8DB3204410028CB103D9CEA40801E856B912D0441007C3F35AA9CEA400085EB517C2D044100703D0ABB9CEA4000000080782D044100C8C342C99CEA408029CB10742D044100701283D49CEA40809BC4206F2D044100809543FD9CEA4080AF0367592D044100DC6800D79FEA40807D1DB8842C0441001C0DE00997EA40004703F88B2A0441000000006896EA400010E937632A0441006C34809595EA4080621058322A0441006C34801395EA4000992A98192A044100EC7C3F7394EA400010E937A729044100C05B207594EA4000992A983929044100000000D894EA40807D1DB800290441000000009894EA40004703F8CF280441001483C0F693EA40806210587A280441006C34804593EA40807D1DB83028044100B8627F4392EA4080F4DB57AE27044100A44E40A391EA40004703F8332704410094CB7FE290EA40807D1DB8AC2604410040A4DF4290EA40806210580226044100DC68005790EA40807D1D384B2504410094CB7FCC90EA40807D1DB8B024044100A44E409991EA40807D1DB8E02204410094CB7FBC91EA40806210586222044100E4F21F2C91EA400010E9B7F4210441005427A03990EA40806210589A210441006C34808B8FEA40806210589A200441001C0DE00B8FEA40004703F817200441005427A04F8FEA40802BF697A11F0441001C0DE0918FEA400010E9B76C1F04410094CB7FFA90EA40807D1D38EB1E0441001C0DE02192EA40802BF697851E044100A44E402395EA40004703F8C31D0441005427A0C795EA40806210586A1D044100E4F21FBE96EA40807D1DB8041D0441005427A08F96EA40807D1D38931C044100A44E406196EA400010E9B7181C044100DC6800E195EA40807D1D38B71B044100ACD85F1095EA40004703F81B1B0441005CB1BF8E93EA40807D1D381F1A04410094CB7FFC91EA40806210583219044100F875E03A91EA40807D1DB8E4180441000000006890EA40004703F8AF18044100381AC0C58FEA400010E9379F18044100DC6800618EEA40802BF69755180441001C0DE0518EEA40807D1DB82018044100ACD85F528EEA40006C787A0718044100141DC9BD8EEA4000C66D348117044100884160D78EEA40802BF6977517044100E4F21FE690EA40806210586A160441006C3480BB91EA400010E9B718160441002C90A0D092EA40807D1DB800160441000000007894EA40802BF69701160441001483C0CC95EA400010E9373B1604410078BE9FF096EA400010E9B77C1604410040A4DFA297EA40802BF697A51604410040A4DF9A99EA400010E937C3160441001C0DE0319BEA400010E937C7160441005427A0679CEA40004703F8BB160441005427A01F9EEA40004703F8A71604410094CB7F349FEA40004703F89F16044100A44E4079A0EA40802BF697D11604410060764F92A1EA4080423EE8EB1604410048E17A94A1EA40000AD7A3EC16044100986E12A3A1EA4080180456EE16044100A05E29A5A2EA400012143F0C1704410040F163C2A2EA4080E5D0A20F1704410088416061A3EA40004E6210221704410040355ED4A3EA40807D1D381117044100ACD85FF0A5EA400010E9B7141704410078BE9FA8A7EA40807D1DB8E416044100C8E53F82A9EA40004703F8A71604410078BE9FB8AAEA40004703F87F1604410000000040ACEA40004703F85F16044100E4F21F66ADEA40807D1D382F16044100DC6800B9ADEA4080621058E6150441006C34800BAEEA40802BF697B115044100ACD85F4EAEEA40004703F85B150441001C0DE0F1AEEA40004703F82F150441005427A0F7AFEA40807D1D38FB14044100C05B201DB1EA400010E937E71404410040A4DF52B2EA40807D1D38D314044100DC6800A9B3EA40004703F8CF14044100884160DFB4EA40807D1D389F140441002497FF66B6EA40807D1D38771404410094CB7F5CB7EA40806210583614044100A44E4021B8EA4000992A98F91304410094CB7FFCB9EA40004703F85F130441001C0DE009BCEA40807D1DB8881204410078BE9F48BEEA4000992A9899110441002497FF4EBFEA40807D1DB84411044100B8627F43C0EA400010E9B734110441005427A037C1EA400010E9B734110441006C34800BC2EA40807D1DB830110441002497FF50C3EA40802BF6973D11044100EC7C3F03C4EA40807D1D386F110441005427A057C5EA40807D1DB8A8110441005427A0C1C7EA4080621058DA11044100088A1F5DCAEA40807D1D38DF11044100381AC03DCEEA40802BF697FD11044100DC680059D5EA40004703F8DB12044100C8E53FD2D7EA40802BF697251304410064AA6048D9EA40806210583A13044100381AC08DDAEA400010E93747130441006C3480F3DBEA40004703F85713044100000000D8DCEA40004703F8471304410094CB7FBCDDEA40004703F82B13044100C8E53FD2DEEA40004703F803130441006C34809BE0EA40807D1D38B3120441005CB1BFBCE3EA40802BF6971512044100A44E4063ECEA40806210586B100441005427A091EDEA40807D1DB84810044100884160A7EEEA40807D1DB8241004410094CB7F9CEFEA400010E9B7F80F044100DC680031F0EA40807D1D38970F0441001483C094F0EA40807D1DB8280F044100C05B200DF2EA40004703F8D70E0441006C567D12F6EA4000E25817EC0D044100D088D240F6EA40801E16EAE90D0441005C8FC2A730EB4000234AFB5D1A044100F41FD29BA8EB40002DB21D1C2D044100FC5C6D8775EC4080A44E40DA3F04410004098A7151ED4000AE47619948044100A401BC4BCDEE4080A44E40DA3F044100F016484C95EF4000B37B720335044100088A1F67D9EF40004703F8653D044100CAE53F601EF04080B437F8E346044100000000C228F04080B43778FD4704410076BE9FAA53F040802BF6175851044100C05B207566F0400010E937F5530441008E06F0496FF040003411B6D5550441007268918D5AF0400080B7C04662044100000000403CF040004703F85F7204410096B20CF22BF04000925CFE7889044100000000D024F040004703F89F910441004AEA04831BF040006C787A81A0044100984C158719F04080FA5C6DFCA20441006844699B0FF040004703F8DFB004410054E3A5740CF04000F1F44A3AB804410000000080FBEF40004703F81FD00441008CB96B37DEEF40801D386716E60441010300000001000000A80000000040355E88E6E7408034803773E6034100948785E8E9E74000C7BAB8EFE6034100ACD85F76F1E7408062105806E80341001C0DE08B04E84080621058C2EA0341005427A0FF16E84000992A9859ED034100E4F21F2C23E840802BF69709EF034100A44E404135E8408062105826F1034100381AC00546E840807D1D385FF2034100C05B20FD56E8408062105836F30341005427A0296CE8400010E93757F303410094CB7F6C78E8400010E93787F3034100A44E40EB86E840807D1DB8C4F30341005427A02F93E840806210581EF4034100A44E40339FE840004703F877F40341001C0DE0F3AAE8408062105822F50341006C348083B2E8408062105882F5034100884160D7B6E840807D1D386BF5034100A44E4063B9E840807D1D383FF50341004C5986C6B6E84080EA73350CF803410090ED7C27B6E840803D0AD7EDF70341008816D9E6B4E840803D0AD78AF7034100E07A14B6B4E8400004560E8AF7034100088A1FA3B4E84080635DDC8CF70341002C651992B4E84000D1225B90F7034100B0726883B4E840808FC27594F7034100D42B6577B4E84080BC741399F70341001804566EB4E840807493189EF7034100B8F3FD0CB4E8408039B4C8E1F7034100085F9806B4E84080ABADD8E7F7034100489D8004B4E84000992A18EEF7034100CC10C706B4E8400062A156F4F7034100D0B3590DB4E84080658863FAF703410000BC0518B4E8408072F90F00F8034100246C7826B4E8400057EC2F05F803410038894138B4E84080E3A59B09F80341008C976E22B5E84000560E2D3AF803410020B0722CB5E84080976E123CF803410050B81E41B6E840806891ED6BF8034100DC8AFD49B6E840806FF0057FF8034100E4F21FECB4E840806210580AF9034100C05B20C5B3E8400010E93763F90341005CB1BF9EB2E840004703F8A3F9034100A44E400BB2E8400010E9B7D0F903410094CB7FE4B0E840802BF69721FA0341005427A02FB0E8408062105872FA034100ACD85F7EB0E840807D1DB800FB03410068B3EADBB0E84000D6566C7CFB034100701283D8B0E84000AAF1D27CFB03410018D9CEB3B0E840003108AC7FFB034100F853E3A1B0E84080EB51B883FB0341005C8FC271AFE84080726891A0FB034100E8FBA965AFE840805AD3BCA5FB034100C4D32B5DAFE84000F46C56ABFB03410094878558AFE840800F7A36B1FB0341003C0AD757AFE84000333333B7FB034100806ABC74AFE84080976E123AFC034100C420B076AFE840806210583EFC03410084C0CA79AFE840801283C041FC03410094438BD8AFE840801283C096FC034100C898BBE0AFE8400083C04A9CFC03410044696FECAFE84000D6566CA1FC034100F46C56FBAFE84000DCD701A6FC034100B86B090DB0E840801E85EBA9FC03410004560E21B0E84000728A0EADFC034100AC8BDB36B0E84080613255AFFC034100D022DB4DB0E84000E9B7AFB0FC034100FC5C6D65B0E840802A1815B1FC034100FCA9F17CB0E84080257582B0FC034100CC32C493B0E84000917EFBAEFC034100845149A9B0E8408095438BACFC034100441CEBBCB0E8400013F241A9FC034100C03923CEB0E840807DAE36A5FC03410048E17ADCB0E84080B81E85A0FC034100C09F1A4FB1E8408016D9CE73FC034100F8C26494B1E84000D34D6270FC03410040A4DFDAB1E840802BF697CDFC034100B8627FFBB3E840802BF69781FE0341001483C02EB5E840004703F8FBFE0341001851DA5FB5E84080B0E1E91EFF0341000CD7A360B5E84080567DAE1FFF03410094180462B5E8408045B67320FF0341008841601FB6E8400010E937A7FF03410010C7BA5EB6E84080B20CF1F7FF0341005CB1BF5EB6E840004703F8F7FF034100A44E40EBB5E840802BF6974100044100C8E53F0AB6E840807D1D389700044100A44E4009B6E840004703F8CB00044100489D809AB6E8400010E9B7F800044100DC6800AFB7E840802BF6970D010441007C36AB26B8E840009487851201044100FC87F437B8E84080E5D0A213010441006891ED4CB8E84080E926311401044100CCEEC94FB8E840005917371401044100381AC045B9E840806210581E0104410000000008BAE840807D1D385B0104410094CB7FFABAE840807D1DB8A001044100E4F21FACBBE84080F4DB57EE01044100C00E9CD7BCE840001895542602044100989999E1BCE8400095D4092902044100ACD85F50BDE840802BF69789020441005427A08FBDE8400010E937E302044100E4F21F4CBDE840807D1DB84C03044100C05B2025BCE84080621058BA0304410000000080BBE840004703F82704044100ACD85F9EBBE840802BF6978D040441001C0DE04BBBE84080F4DB57CE04044100DC6800E9BBE840807D1D38D7050441001C0DE0C9BCE840802BF6976506044100C8E53F2ABDE840807D1D38BB06044100EC7C3F49BDE840004703F8FB0604410064AA6048BDE840807D1DB82C07044100E4F21F06BDE840802BF6976107044100E4F21FC4BCE84080F4DB578E070441001C0DE081BCE8400010E937BB07044100E4F21F7CBBE840004703F80308044100E4F21F86BAE840802BF697610804410000000090B9E84080621058B208044100A44E40ABB8E840807D1D38DF0804410000000048B8E840004703F82709044100DC680077B8E840802BF6977509044100C05B2065B8E84080621058C609044100DC6800B1B7E840807D1D38FB0904410094CB7FFCB6E840004703F83B0A0441005CB1BFE6B5E840004703F86F0A04410094CB7F2CB4E840802BF697ED0A04410078BE9F98B3E840807D1D38370B0441006C3480F3B2E840807D1DB8A40B0441008841609FB2E840806210581A0C044100C05B207DB2E840004703F8870C04410040A4DF8AB2E840802BF697F50C04410094CB7FAAB2E84080621058220D0441005427A0C9B2E840004703F8630D044100EC7C3F49B3E840802BF697DD0D044100DC680037B3E840004703F8430E044100E4F21FE4B2E840807D1DB88C0E044100ACD85F0EB2E84080621058F20E044100A44E4043B0E840004703F8970F044100ACD85F16AEE840802BF6972910044100B8627F6DACE840807D1D388B10044100884160A7ABE840807D1DB80C110441001483C094ABE84080621058721104410000000040ABE8400010E9B71C120441005CB1BF1CABE8400010E937A312044100ACD85F36AAE8400010E9B718130441004CF38E5BA9E84000B8AF837F13044100E07A142AA7E840803F355EFC13044100F0A7C68FA2E84000894160DB1404410078BE9F22A2E84000508D97F1140441006CE7FB8DA1E840800C022B0C15044100DCF97E4AA1E84000FCA9F1181504410060BA4940A1E840004E62101A15044100A857CAE8A0E840007D3FB5291504410000000070A0E8408062105832150441006C567D5E9EE84080DB68805315044100A4DFBEBC98E84080BDC1970A110441002C90A02692E84000992A18170E044100ECEBC09382E8400010E9B7FF070441001483C07882E8400010E9B79B06044100F875E0CA69E84080EB513852060441008048BF2D55E84080F4DB573507044100C8E53F0C36E840004703F8C1050441001483C07A24E8400010E9B7D704044100643BDF710CE8400010E9B79002044100D46F5F7100E84080BDC117A001044100F875E022FCE740806210D845FD03410088D2DEE2EBE740806210D83DEE03410078BE9F10E9E740807D1DB87DE803410040355E88E6E7408034803773E603410103000000010000001000000000BCC11790D0E84000E63F24994D044100B8D1003AD0E840802BF6179B4D044100482EFFB3BBE84000D04458AF4F044100489D80F2B6E84080EB51B8E74F044100B0B662BBB4E84000EEEBC0AF48044100BCC117B0B5E8408003E70CC548044100E4F21F64B6E8400010E9B7D4480441005427A027BBE840807D1D382F49044100A44E4089BFE84080621058824904410060BA499AC1E8400093A982B74904410084EB51B0C2E84000B07268034A04410050B81EB5C2E840809DEFA7044A044100D44D628ACFE84080506B1A334D044100A0CDAA3FD0E84000ED0D3E7A4D0441005CB1BF6ED0E840807D1DB88C4D044100BCC11790D0E84000E63F24994D0441');
INSERT INTO grondwaterlichamen_new VALUES (11, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch', '01060000208A7A0000040000000103000000010000000A000000005227A04F3BF44080BDC117B6760A4100361AC0843BF44080BDC1179C760A4100A44E40A23CF44080F4DBD739760A4100CAE53FE73CF4408062105824760A4100E0BE0E753DF44000F241CFF5750A4100361AC0493EF440802BF697A9750A4100907EFB003FF4408099999995750A4100DC6800713FF4408062105882750A41006E34807D40F440806210585E750A41005227A04F3BF44080BDC117B6760A410103000000010000000C000000005227A05D44F440004703F82A750A4100DC68005146F44080F4DBD72D750A4100F697DD2C48F4400079E92647750A41005227A0274AF440004703F84B750A41007AC729B24AF4408039B4C865750A41001C0DE0274BF440802BF6976B750A41006E3480594BF440802BF6976F750A41001C0DE0A94BF44000A2B4B776750A4100E4F21FB44CF440802BF69788750A41001A2FDDB74CF44080081B1E89750A410030BB27A94CF44000DD24868F750A41005227A05D44F440004703F82A750A410103000000010000000A000000006E34807D40F440806210585E750A4100CAE53FF540F44000D9CE7752750A4100E4F21F0E41F440004703F84E750A4100361AC05E41F440004703F84A750A41005227A0DC41F44080F4DBD73A750A41006E34809B42F4408062105830750A41008A4160E642F440802BF6972C750A4100AED85F0B44F4408062105826750A41005227A05D44F440004703F82A750A41006E34807D40F440806210585E750A410103000000010000000600000000ACFA5C9939F44000CD3BCE6F770A41001C0DE09F39F440802BF6976B770A4100DC6800F739F440004703F841770A41000000002C3AF440004703F827770A41005227A04F3BF44080BDC117B6760A4100ACFA5C9939F44000CD3BCE6F770A41');
INSERT INTO grondwaterlichamen_new VALUES (88, '               <BR> BLKS_1000_GWL_1S Landeniaan Aquifersysteem, Freatisch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
INSERT INTO grondwaterlichamen_new VALUES (9, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                              <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen<BR> SS_1300_GWL_1 Kolenkalk, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (12, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                            <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter      <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (13, '                                     <BR> CVS_0160_GWL_1 Pleistoceen afzettingen, Freatisch                            <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter  <BR> SS_1300_GWL_1 Kolenkalk, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (14, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                                <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen      <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (15, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                                <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen    <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (16, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                                <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen  <BR> SS_1300_GWL_2 Sokkel + Krijt Aquifersysteem, Lokaal freatisch, voedingsgebied', '01060000208A7A00000A000000010300000003000000570B000000D881F31F50004100B5A6F94E1E034100000000C84E004100470378C721034100000080EC4D0041802BF617B325034100E5F29FA34F0041807D1DB8FC29034100000080454F0041802BF617272E034100E5F21FCD4D00418062105851320341001B0D60D74B0041807D1D38A53403410000008083490041004703786135034100000080D14600418062105803350341001B0DE08C410041004703F88A33034100C9E5BF773F0041004703785631034100371AC06C3B004100470378462E03410000000013340041806210D81C300341001B0D60D4330041807D1D38EE32034100E5F21F18330041806210585C360341001B0D607B320041802BF6179B360341001B0D60702E0041807D1DB8C933034100371A405B2C0041807D1DB817310341001B0DE0C7230041004703F8D830034100E5F21F30220041004703786634034100E5F21F30220041807D1DB8D936034100E5F29FAD220041807D1DB88B39034100E5F29FAD220041802BF697813B034100371AC0B7200041807D1D38193D034100E5F21F151B0041802BF697383D034100371AC038140041802BF697863A0341000000800E100041802BF697863A034100371AC0760E0041806210D8E9390341001B0D6009110041807D1DB8D437034100E5F21F0A170041802BF6179B360341001B0D60D61A0041806210585C360341001B0DE0631E0041004703F88A330341001B0DE0631E004100470378FD2F034100E5F29FA21E00410010E937D32B034100371AC0F51A004180621058DD290341000000000518004180621058362B034100371A400F16004180621058362B03410000000043120041807D1D38C8270341001B0DE02D100041807D1D38BD23034100371AC01D0D0041806210D8EB20034100371AC06B0A0041807D1D38AD200341001B0D60EE090041004703F8391E034100000080F60C0041802BF697201C0341001B0DE0860A0041807D1D38C716034100E5F29F4D060041802BF697BE14034100E4F21F97FDFF40802BF697BE14034100E4F21FB5F6FF40802BF697BE14034100CAE53FD3EFFF40802BF6979614034100AED85F41E9FF4080621058F613034100CAE53F80E5FF40807D1DB83615034100CAE53F30E5FF40807D1D38EF160341005227A070E6FF40802BF617101A034100361AC09FE3FF40004703F8E01C034100AED85FBCD7FF40004703F89719034100CAE53F3AD0FF4000470378DF17034100361AC027C7FF40806210586717034100361AC0B6C3FF40802BF697F01B0341001C0DE0C7C7FF40004703F8081D034100AED85F19CDFF4000470378111F034100CAE53F79CCFF40004703785A220341001C0DE038CBFF4080621058A3250341005227A07AD1FF40806210581429034100E4F21F7CD6FF40802BF6974D2D0341006E3480DAD0FF4000470378252D034100AED85F58C9FF40802BF617E52B034100000000A7C4FF40802BF697042A0341005227A034BCFF40004703F8AB27034100361AC052B5FF40004703F8AB27034100AED85F12B4FF40806210581429034100AED85F12B4FF4000470378FD2C0341001C0DE081B2FF40004703789630034100AED85F30ADFF40806210584630034100E4F21F0EA5FF40806210D83D2E03410092CB7FFB9BFF40802BF697752D0341000000001B9AFF40802BF6172E2F0341006E34803A98FF40802BF617EF320341006E34803A98FF40802BF617883603410092CB7FFB9BFF40807D1DB86838034100AED85F3DA2FF40806210D80839034100E4F21F7FA8FF40802BF617D13903410000000061AFFF4000470378113B0341001C0DE0C0AEFF40807D1D38BA3D0341005227A05FAAFF4080621058823E034100361AC0CDA3FF4080621058323E034100CAE53F0B9BFF40802BF617423D034100CAE53F3E94FF40802BF6174E3F0341005227A0138FFF40004703F8B142034100CAE53FF987FF40807D1D383F46034100CAE53F9584FF40807D1D3885450341000000000881FF40802BF6978842034100E4F21FAC7CFF40802BF61784400341001C0DE0FE7CFF4000470378CD3C0341006E3480C57EFF40802BF697A739034100E4F21F9C7EFF408062105812370341001C0DE00E7BFF40802BF61709330341006E34808871FF4000470378FC31034100E4F21FF36EFF40004703F80034034100E4F21FEB6FFF40004703F86C360341001C0DE03D70FF40807D1D38FA390341001C0DE08462FF40802BF617663C034100361AC0935BFF4000470378FB3E034100E4F21F5159FF4080621058A541034100000000295EFF40004703F8A9430341001C0DE00862FF40802BF697F4440341001C0DE0D96CFF40802BF697A646034100000000666BFF40806210D8F549034100361AC08F69FF40802BF6975D4C034100CAE53FCC6AFF4000470378E14C0341001C0DE0ED6AFF40004703F8F74E0341000000002B66FF40806210D89B520341006E34800965FF40807D1D382B53034100E4F21FA762FF40806210D801540341001C0DE05460FF4000470378C854034100CAE53F425EFF4080621058BF550341001C0DE03E5CFF40807D1D38FE56034100361AC0CA5AFF40802BF6979558034100000000C85AFF400010E9B7B559034100CAE53F665BFF40807D1D386E5A034100000000E45CFF40807D1D386F5B034100E4F21FF25EFF40802BF697485C034100AED85F8161FF40807D1D38B25C034100361AC08163FF4000470378AB5C0341005227A00165FF4000470378CC5C0341001C0DE0FF66FF40807D1DB8955D0341001C0DE0BD67FF40806210586E5E034100E4F21FBC68FF40802BF6172F5F034100361AC0396BFF40807D1DB848600341005227A0C96DFF40806210587A600341006E34807B71FF40802BF697EC5F0341001C0DE0FD73FF40004703F8155F034100000000DF76FF40807D1DB8BF5E034100361AC0FE7AFF4080621058125F034100E4F21F0F7EFF40807D1D38145F0341005227A03082FF40806210D8B65E0341005227A06487FF40806210D8595D034100361AC0D588FF400010E9B7FA5C034100361AC0D78AFF40806210D8435C0341005227A05A8CFF40802BF6972C5B034100361AC07E8DFF40802BF6179D59034100000000838DFF40004703F8E457034100CAE53FC68CFF40806210588C560341000000008A8BFF4080621058FB540341001C0DE05C8AFF4000470378C253034100E4F21F1F89FF40802BF697C152034100AED85F7188FF40004703F8D851034100E4F21F5388FF40806210D820510341005227A02489FF40806210588950034100AED85F158AFF40806210D851500341005227A0258BFF400047037842500341006E3480D58CFF40802BF69763500341006E3480548EFF40802BF697DC500341005227A0728FFF4080621058AD51034100E4F21F2091FF40802BF697CE52034100000000CE92FF40807D1DB8B7530341005227A0EC93FF40004703785054034100CAE53F8B95FF40802BF697F954034100CAE53F3A97FF40807D1DB87255034100CAE53F6A98FF40004703788B55034100AED85F1A9AFF40004703789455034100AED85F6C9CFF40806210D8DD540341001C0DE0ED9CFF40802BF6174E54034100E4F21FB09CFF40806210D85D53034100361AC0519CFF4000470378B552034100361AC0639CFF4000470378ED510341005227A0059DFF400010E9B72D51034100361AC0C69DFF40802BF617BE500341001C0DE0F79EFF40806210D85E50034100CAE53F29A1FF40802BF617F04F0341005227A046A6FF40004703F8F74E0341005227A01FADFF40807D1DB8014D0341005227A0E3B3FF400010E9B7F04A0341005227A0D7B4FF40802BF6975B48034100AED85F1FB7FF40807D1D3818480341006E348073B9FF40802BF617A94903410092CB7FBCBCFF40806210D8C249034100361AC0FFC1FF40004703F8B046034100AED85F62C6FF40807D1D383C440341005227A0D2CBFF40807D1DB86442034100E4F21F62D0FF40802BF69791420341001C0DE0D6D2FF40802BF6179644034100000000AAD2FF40802BF6976D46034100361AC066CDFF40004703F895490341005227A0D2CBFF40802BF6174E4C0341005227A0C9C7FF4080621058B54D03410000000067C3FF400010E9B7B94F034100000000D0BDFF40806210D8FF500341001C0DE028BDFF40807D1DB8815103410000000077BAFF40802BF61720520341001C0DE064B7FF4080621058D652034100E4F21FD3B4FF40807D1DB86C53034100AED85FE1B2FF40802BF69703540341005227A05FB1FF40806210D8AA54034100361AC03DB0FF400010E9375255034100CAE53F2AAEFF40802BF617A1560341006E3480F6ABFF40806210D80F58034100000000A2A9FF40802BF697B6590341000000006FA7FF4000470378D55A0341006E34806BA4FF40807D1DB81B5C0341005227A0C6A0FF40807D1DB8D95D034100361AC0839EFF4080621058E85E034100CAE53F109CFF40802BF6172F600341001C0DE09997FF40802BF6978462034100000000B594FF40004703F852640341001C0DE03293FF40807D1D381A65034100000000DF90FF40004703F8906603410092CB7F6A8FFF400010E93750680341006E3480888EFF40807D1DB80769034100AED85F758DFF400010E9373F6A034100AED85FB18CFF40004703F8CE6B034100361AC0FD8BFF40807D1DB83E6D0341000000005B8BFF40004703784E6E0341005227A0A789FF40802BF6979D6F034100AED85FD487FF40802BF697CC70034100E4F21FC286FF40802BF617AC71034100E4F21F5E85FF408062105833730341000000000B85FF40806210586B74034100CAE53F7985FF40807D1DB82B750341000000008885FF40806210D8AB750341001C0DE03686FF40806210582C760341001C0DE0B487FF40806210580D77034100361AC0218BFF40802BF69777780341006E3480FF8EFF40807D1D389A790341005227A08C91FF40004703F8E37A034100000000DA92FF40806210D8FC7B0341005227A06494FF40802BF6173E7E0341005227A02095FF40806210D8DE7F0341006E3480FC95FF40802BF6979F810341000000003A97FF4000470378A882034100CAE53FC79BFF40004703780384034100361AC0B69FFF40004703F865840341006E348037A3FF40802BF6174884034100361AC069A6FF40004703F89183034100000000ACA7FF400010E9B7B282034100000000E0A9FF40806210D83381034100AED85F04ACFF40004703F88C7F03410000000048ADFF40802BF6972D7E0341001C0DE02BAEFF40004703F8AD7C0341000000000FAEFF40807D1DB85D7B03410000000061AEFF40806210D89D7A034100361AC0F3AEFF40004703F88579034100E4F21FB6AFFF40806210589678034100CAE53FC7B0FF40004703F82E78034100AED85FE7B2FF408062105840780341005227A0E6B3FF40004703F898780341005227A054B4FF408062105871790341001C0DE0D0B4FF40806210D8F97A0341001C0DE00DB6FF40807D1DB8427C0341000000003BB7FF40802BF6977B7D034100000000D8B8FF40806210D8C47E0341005227A035BAFF40806210D8C57F0341006E348064BBFF40802BF6974E80034100361AC073BCFF4080621058A780034100CAE53FD9BDFF40807D1D383C81034100361AC0BDBFFF40807D1DB8D881034100000000F0C2FF4080621058A181034100CAE53F76C4FF40807D1DB8D88103410000000070C5FF40807D1D38FE81034100E4F21FFBC6FF40807D1DB8C8820341006E3480FCC7FF40807D1DB8B883034100E4F21F69C8FF40806210D8A8840341005227A0C7C8FF40806210D8988503410000000050C9FF40004703F8888603410000000070CAFF40802BF697A8870341000000003ECBFF40802BF6176988034100000000F0CCFF40807D1D381489034100000000D2CDFF40802BF617598903410000000070CFFF40004703F8C389034100000000F0D1FF4000470378F98903410000000070D4FF40807D1D38D789034100000000F0D6FF40004703784E890341006E348093D9FF40802BF6176988034100000000F0DBFF40806210587E87034100AED85F30DDFF40004703F88886034100361AC094DEFF40806210D898850341000000007EDFFF40806210D8A884034100AED85F36E0FF40807D1DB8B883034100361AC0A6E0FF40807D1DB8C8820341005227A0D5E0FF40807D1DB8D881034100CAE53FF2E0FF40802BF697E880034100000000F0E0FF40806210D8E37F034100AED85FCFE0FF4000470378087F034100000000F0E0FF40802BF697027E034100CAE53F19E2FF4000470378187E03410000000070E3FF40806210D8517E034100AED85FF3E3FF4000470378087F034100E4F21FC7E4FF40802BF697F87F034100AED85F17E6FF40802BF697E8800341006E3480E2E6FF40807D1DB8D881034100E4F21F4EE7FF40807D1DB8C882034100CAE53F93E7FF40807D1DB8B883034100000000ECE7FF40806210D8A88403410000000070E8FF40807D1D383B85034100361AC071E9FF40004703F88886034100CAE53F0FEAFF40004703F87887034100000000F0EAFF40802BF6978688034100CAE53FB4EBFF40802BF6175989034100361AC0BBEBFF40802BF617498A034100000000F0EAFF40802BF617F38A034100AED85F21EAFF400010E937398B03410000000070E8FF40807D1D38BC8B0341001C0DE0A4E6FF40807D1D38298C034100CAE53F4CE4FF4080621058198D03410000000070E3FF40004703F86B8D034100361AC087E2FF4080621058098E034100000000F0E0FF40004703F8358F034100361AC032E0FF4000470378E98F0341001C0DE02BDFFF40802BF697D99003410000000070DEFF40802BF6977A9103410000000015DDFF40802BF697B992034100CAE53F62DCFF40807D1DB8A993034100000000F0DBFF40004703F8D494034100CAE53FE3DAFF40806210D8899503410000000070D9FF4000470378CD950341006E3480A9D7FF40806210D8899503410000000070D4FF4000470378E694034100361AC09BD2FF40807D1DB8999403410000000070CFFF40802BF697149403410000000051CCFF40807D1DB8A99303410000000070CAFF40807D1D386D93034100000000F0C7FF4080621058309303410000000070C5FF40807D1D380F93034100000000F0C2FF40807D1DB8DF9203410000000070C0FF40802BF697B692034100361AC0E8BEFF40802BF697B9920341001C0DE055BDFF400010E9372A93034100361AC0E3B9FF40807D1D38D8930341001C0DE020B6FF40004703F8CD940341006E34800EB3FF40802BF617A4950341001C0DE0BBAFFF40807D1D3882960341006E34801AAEFF40807D1D38F996034100AED85FAAACFF4080621058F0960341006E34809AAAFF40802BF617C7960341001C0DE0BAA6FF40802BF69774960341005227A07BA2FF40806210D8F1950341000000002CA1FF40802BF617B995034100CAE53F6C9EFF40806210587795034100CAE53F6C9AFF40806210D844950341000000005C95FF40802BF69719950341001C0DE0AB93FF40802BF69710950341005227A0AA8FFF40807D1DB82194034100361AC02E8EFF40004703F8A4930341005227A0BF8CFF40802BF61734930341006E3480608AFF4000470378BA920341001C0DE05087FF40802BF6976892034100361AC09083FF40807D1D384692034100E4F21F2080FF40802BF6175C92034100361AC0EF7DFF40807D1DB86292034100361AC0DE7BFF4000470378B1920341006E3480CB77FF40802BF617C793034100361AC08875FF40806210D8C5940341005227A03674FF40802BF6178D950341000000005171FF40802BF697AB97034100CAE53F3C70FF400010E93783990341001C0DE0C771FF40806210585C9B034100E4F21FA374FF40806210586E9D034100361AC06078FF40806210D8989E03410092CB7FB079FF40807D1DB8C19E034100361AC0DE7CFF40806210D89B9F0341006E3480FC7CFF40004703F883A0034100CAE53F5A7CFF40807D1DB863A10341005227A0487BFF40802BF61703A20341005227A07679FF40802BF617BAA2034100CAE53FA577FF40004703F830A30341006E34804376FF40807D1D38D0A30341005227A07275FF40807D1DB81FA4034100000000FF72FF400047037876A50341001C0DE0EA70FF408062105805A7034100E4F21F566FFF40802BF697DCA8034100CAE53F516DFF40802BF697B3AA03410092CB7F6A6BFF40807D1DB852AD034100000000556BFF40004703F882AF034100CAE53FE06BFF40802BF6977BB1034100000000CC6CFF40806210583CB3034100CAE53F296DFF40807D1DB854B4034100E4F21FE76DFF40807D1D3845B50341005227A0C36FFF40802BF697C6B60341005227A0DD70FF40802BF6973FB9034100E4F21FCB6EFF40004703781EBA0341005227A0CA6CFF400010E9373DBA03410092CB7F3B6BFF40807D1D38CCB90341000000000E69FF40807D1DB8AAB8034100E4F21F2266FF40807D1DB8E0B60341000000006663FF40807D1DB826B5034100AED85F6961FF4080621058B5B3034100AED85F0D5FFF40802BF697FBB1034100000000325DFF40807D1D3802B0034100E4F21F265BFF40807D1DB840AE0341001C0DE09859FF40802BF69707AD0341006E34805C57FF40004703F875AB034100E4F21F4E55FF40802BF697B4AA034100CAE53F5E52FF40806210D882AA034100CAE53FEE4FFF40807D1D3861AA034100E4F21F4E4CFF40004703F846AA0341005227A08D47FF40802BF61734AA03410092CB7F4D44FF40004703F819AA034100CAE53F0E42FF40802BF697A8A9034100AED85FEE40FF40807D1DB87CA90341001C0DE05F3DFF40802BF697C5A80341005227A0313AFF4000470378E3A7034100CAE53F1338FF40802BF6172AA7034100E4F21F6635FF40807D1D38D8A5034100CAE53F2A32FF40004703F805A4034100000000ED30FF40802BF617DDA20341006E3480CF2FFF40807D1D38C4A10341006E3480B22FFF40004703F893A0034100361AC0D52FFF40004703F84B9F034100E4F21F5730FF40807D1D38C49E034100361AC06833FF40004703F83D9E034100361AC0EA35FF40004703788F9D034100AED85F8D36FF40807D1DB88F9C03410092CB7FCF36FF40806210D8AF9B034100361AC04237FF40004703F8679A034100361AC09537FF40004703F837990341001C0DE04737FF400010E9B75798034100CAE53FE935FF400010E9B7B6970341001C0DE0DA32FF40807D1DB8EC960341001C0DE07C2EFF40806210D8E995034100000000BE2AFF40004703783F95034100361AC09E27FF4000470378CD94034100AED85F2F23FF40807D1DB85294034100AED85F3F20FF40806210D82894034100AED85F101EFF40004703789F93034100E4F21FC21CFF4000470378D692034100361AC0F51AFF400010E9374D910341000000003919FF40004703F8E38F0341000000001B15FF400010E937E18E034100E4F21FAB11FF40802BF617AF8E034100361AC0EA0EFF4080621058AD8E0341001C0DE0EB0BFF4000470378138E034100E4F21F0E0AFF40802BF6170A8D034100AED85F5F07FF4080621058688C034100E4F21FAE04FF40807D1DB8C68C0341005227A04D03FF40004703F8ED8C0341001C0DE02D02FF40807D1D38BD8C0341005227A06F00FF40802BF617F48B0341005227A091FEFE40807D1DB80A8B034100CAE53FD2FCFE40802BF697B98A0341006E3480F1FBFE40802BF617F98A0341005227A0AFF9FE40806210D89F8B03410092CB7F0DF7FE40807D1D38568C0341001C0DE0CCF4FE40806210D87C8C0341005227A04BF3FE40004703F8EB8C034100E4F21F29F2FE4000470378DB8D034100AED85F56F2FE40807D1DB8F38E034100AED85FE4F2FE40802BF617CC8F0341006E348041F3FE4000470378FC900341000000006DF3FE40806210D8CC920341001C0DE049F3FE40004703F80C9403410092CB7FC5F1FE40807D1D38BC9503410000000032F0FE40806210580B97034100E4F21FCCEDFE40807D1D384A990341001C0DE076EDFE40807D1D38629B0341006E3480E5EEFE40807D1D38039C034100CAE53F25F0FE40004703F8339C034100361AC003F3FE40806210D8ED9C03410000000000F5FE40806210588F9E0341005227A06EF5FE40807D1DB81F9F034100000000ABF5FE40004703F89FA00341006E3480D6F5FE408062105868A20341006E348035F7FE40807D1D38E1A2034100AED85F83F9FE40806210D8DAA3034100E4F21F01FAFE400010E937CBA40341005227A05EF9FE40004703F8C2A5034100CAE53F89F6FE4000470378D1A70341005227A080F3FE40004703F81FAB0341000000005AF1FE40004703F8A6AD0341006E348037EFFE400010E9B795AE0341005227A056EDFE40802BF697CCAE034100E4F21F07EAFE40802BF69772AE034100000000B8E6FE4000470378E8AD0341001C0DE049E4FE40806210D80EAD034100CAE53FACE2FE40807D1DB805AC034100E4F21FCEE1FE40004703F834AB034100CAE53F10E1FE40004703784CAA034100CAE53F95E0FE40806210D84BA80341005227A038DFFE40806210D8DAA60341000000004BDCFE40806210D8B8A50341005227A01AD9FE40004703F8B6A5034100361AC019D5FE4000470378DCA5034100AED85F49D2FE40807D1DB8DAA5034100AED85FD9CEFE40802BF697B8A5034100361AC008CAFE40802BF697B5A503410000000047C8FE40802BF6974CA60341005227A0C3C6FE40807D1DB893A7034100CAE53FBEC4FE40806210D8AAA9034100E4F21F4EC3FE40806210D8A1A903410000000080C0FE40802BF617C0A80341005227A032BFFE40802BF61797A70341001C0DE093BDFE40004703F805A7034100000000E5B9FE40802BF69763A6034100361AC066B7FE40004703F891A5034100AED85FD8B6FE40802BF697E1A40341005227A04AB7FE40807D1DB801A40341005227A0DCB6FE408062105829A30341005227A0EEB5FE40802BF69748A2034100AED85F12B4FE4080621058AFA0034100AED85F94B3FE40806210D8DE9F034100E4F21F97B4FE4080621058CF9E0341001C0DE028B7FE40806210D8389E034100000000F9B8FE40004703F8419E034100CAE53F19BFFE40806210D87D9E0341001C0DE03AC1FE40802BF617F79D0341000000007DC2FE40806210D8279D0341005227A02FC3FE40802BF617289C0341006E348014C3FE400010E9B72F9A0341005227A0A7C2FE4080621058DF98034100E4F21F9AC0FE40004703F8C597034100AED85F3BBDFE400010E9B713970341006E34805CBAFE40806210D88196034100E4F21F4EB7FE40806210D8AF950341001C0DE07FB5FE40807D1DB8E694034100E4F21FF1B4FE40807D1D385694034100361AC0B4B4FE40004703F8E592034100361AC018B5FE40004703F84D91034100CAE53FCDB4FE40802BF697758F03410000000040B3FE4000470378448E0341006E348002B2FE4000470378338D0341001C0DE0C4B1FE40807D1D38438C034100361AC067B2FE40004703781B8B034100361AC0ABB3FE40802BF6178C8903410000000032B5FE40807D1DB8248703410000000014B5FE40802BF6974C86034100000000D9B2FE40806210D83284034100AED85F4DB0FE40802BF6174982034100AED85FA0AEFE40806210D8FF800341005227A002ADFE40807D1DB8FE7F034100361AC0F4AAFE40806210580D7F0341001C0DE068A7FE40806210D83A7D034100CAE53F0EA5FE40802BF617F17A034100CAE53F5FA4FE40802BF697807A034100AED85F2FA2FE40807D1D384F7A034100CAE53FBDA1FE40802BF617277B034100AED85FF9A1FE4000470378B77C0341005227A005A2FE40807D1DB83F7E0341005227A052A2FE40004703F86F7F034100E4F21F2FA2FE40802BF617E0800341006E34802BA1FE400010E9B73F820341001C0DE0689FFE400010E9B73E83034100E4F21FC59AFE40004703F87B84034100E4F21FA295FE40004703F878850341000000004093FE40004703782F86034100361AC0DD91FE40807D1DB8FE860341006E34809A90FE40802BF6173E880341006E3480A78FFE40807D1DB86D890341005227A0158FFE4000470378258A034100AED85F228EFE40004703F8648B034100AED85F008DFE40004703782C8C0341001C0DE03D8AFE40806210D8028D034100CAE53F0B86FE4080621058D88D0341000000000884FE400010E9370F8F0341000000004683FE40806210D8D68F034100000000E281FE40807D1D385E91034100E4F21FCD7FFE40802BF61735930341001C0DE0EA7DFE40802BF617049403410092CB7F2879FE40807D1D38B9940341001C0DE01775FE40807D1DB8CE940341005227A0196FFE40806210D8CA93034100361AC0AB6BFE40802BF697B892034100000000DC69FE40004703788F920341006E34809966FE40004703786593034100000000C665FE40802BF617BD940341006E3480B365FE40807D1D38C5950341005227A0A065FE4080621058ED96034100361AC0FD62FE40806210D8EB970341001C0DE01A5FFE40802BF697E998034100361AC0385EFE40802BF617B999034100361AC0545DFE40807D1DB8489B0341006E3480815DFE40004703F8989C0341006E34805E5DFE40802BF617C99D034100AED85F0B5EFE400010E9B7119F034100361AC0275FFE40802BF6979AA00341001C0DE08460FE40802BF697D3A10341005227A0F260FE40004703F8C3A20341005227A0205FFE40004703F87AA3034100CAE53FBF5BFE40806210D8D8A3034100361AC05C58FE40806210D8A6A4034100E4F21F4956FE40807D1DB80DA6034100CAE53FB555FE40802BF6978DA7034100AED85FD055FE40004703F885A90341005227A03D55FE40807D1DB89DAA034100000000AB52FE40807D1D3884AB0341001C0DE0C94FFE4000470378D2AB0341001C0DE0794CFE4080621058B0AB034100361AC0C948FE40802BF61786AB0341000000006846FE40807D1DB81CAC034100AED85F2545FE40004703F813AD034100CAE53F1244FE40004703784BAE034100AED85F8042FE40802BF697FAAE0341006E34807F40FE408062105839AF034100361AC05F3FFE40807D1DB810AF034100000000013EFE40807D1DB87FAE0341001C0DE04439FE40802BF697ACAC034100CAE53FC635FE4080621058FAAB0341006E34804632FE40802BF617A8AB0341001C0DE0F62EFE40004703F85DAB034100AED85F152BFE40807D1DB8BBAB0341005227A08226FE40004703F8A0AC0341005227A07F23FE400010E937AFAD0341005227A02D21FE40806210D85DAE034100CAE53F3B1DFE40004703781BAF0341005227A0CA1AFE40004703F839AF0341005227A06817FE40004703F8DFAF0341001C0DE07412FE40802BF61725B10341005227A0F10FFE40807D1DB853B20341000000003E0EFE400010E9B7B2B3034100CAE53F5A0DFE40004703782AB50341001C0DE0660DFE40802BF69782B60341006E3480000EFE40806210581BB9034100CAE53F0B0FFE400010E93744BB034100E4F21F3810FE40802BF6179DBC0341005227A09411FE400010E93716BE034100000000D212FE40802BF6172FBF0341001C0DE04013FE4000470378A7BF0341005227A0BF13FE40806210D82FC00341005227A0FD14FE40807D1DB810C1034100CAE53FFB16FE40802BF61722C2034100AED85F0819FE400047037863C3034100E4F21F761CFE40807D1DB87DC403410092CB7F0520FE40004703F8E7C4034100AED85F9622FE40802BF697B1C40341001C0DE0A826FE40004703F8E3C3034100000000A92AFE40004703780EC4034100361AC0872BFE40802BF61797C40341005227A0B52BFE408062105877C5034100A8E848AA2BFE4000992A9825C60341007C832F0FF2FD4000992A9825C6034100361AC0CEF0FD40807D1DB832C5034100361AC061F0FD4080621058FAC303410000000055F0FD40802BF617AAC20341006E348047F0FD40004703F8A9C10341005227A00AF0FD40807D1DB861C00341001C0DE0ABEFFD4080621058D9BF034100CAE53F8DEEFD40802BF69740BF03410092CB7F2DECFD40802BF61707BF034100361AC0BDE7FD4080621058B4BE0341000000003EE5FD400010E9B77ABE0341001C0DE03EE3FD400047037801BE034100361AC020E1FD40004703F827BD034100361AC083DFFD40806210D8DEBB03410000000047DFFD400047037886BA0341000000005ADFFD408062105856B9034100CAE53FDEDFFD40004703789EB7034100E4F21F92E0FD400010E9B716B6034100361AC085E1FD40802BF617AFB4034100CAE53FD8E2FD40802BF69722B30341000000005CE2FD40806210583FB20341000000004EE0FD40004703F84DB103410092CB7F9FDEFD40806210D8A4B003410092CB7FA1DCFD4000470378BBAF0341005227A084D9FD408062105849AE03410092CB7FB7D6FD4000470378FFAC0341005227A07AD3FD400010E9378DAB03410092CB7F2DD0FD40802BF61743AA034100CAE53F10CDFD40004703F8F8A8034100AED85FF4C8FD40807D1D381EA70341001C0DE0D5C5FD40802BF61754A6034100AED85FA7C2FD40802BF61792A5034100AED85F88BFFD40802BF61700A503410092CB7FB9BBFD40802BF69755A40341000000007AB8FD4000470378FBA30341001C0DE069B2FD40807D1DB8B7A30341005227A049AEFD40807D1D389DA30341001C0DE0D7A9FD40004703781AA4034100AED85F93A5FD40004703F8A7A5034100AED85F8FA2FD40806210581EA7034100AED85F4AA0FD400010E93705A9034100AED85FD69EFD400047037894AA034100361AC0A09DFD40004703F8C3AC034100CAE53FBC9CFD400010E9B78BAE0341001C0DE0659CFD40807D1DB81BB10341001C0DE0409DFD40802BF69724B3034100E4F21FFD9CFD40802BF697A4B4034100361AC0D99AFD4000470378E3B50341005227A0B797FD40802BF69791B6034100361AC0B593FD40807D1D381FB7034100E4F21F3590FD40004703F834B70341001C0DE0D58CFD40806210D8BAB6034100361AC0778AFD40807D1D38D9B50341006E34800989FD40807D1D3818B5034100361AC00B87FD40806210D80EB4034100AED85FBF84FD40807D1D387DB20341001C0DE02283FD40802BF61704B1034100361AC0C781FD40004703F8F2AE0341006E3480FB80FD40806210586AAD034100000000507FFD40004703F880AB034100361AC0A27DFD40806210D84FAA034100AED85F147CFD40807D1DB896A9034100E4F21F5579FD40004703F81CA9034100361AC0D573FD400047037899A80341005227A00570FD40802BF6176FA8034100CAE53F146CFD40807D1DB8D4A80341006E3480B068FD40806210D82AAA034100E4F21F2C66FD4000470378C1AB034100361AC0A862FD4080621058EFAC034100361AC0465FFD408062105895AD034100E4F21F365CFD4000470378ABAD034100AED85FE659FD40802BF6177AAD0341006E34808658FD40807D1D3851AD034100AED85FA555FD40004703789FAD034100361AC00252FD40807D1D387DAE034100E4F21FC24EFD40807D1D3893AE0341001C0DE0324DFD40807D1D3832AE0341006E3480544CFD40802BF69789AD0341006E3480074DFD40806210D851AC034100AED85F6A4EFD40802BF69742AB0341000000001E4FFD40806210D8DAA9034100CAE53F014FFD40802BF69782A8034100AED85FC54EFD40807D1D38D2A6034100361AC0284EFD40807D1DB879A50341005227A06C4CFD4000470378C8A30341005227A0A049FD40004703780EA2034100AED85F8346FD4080621058CCA00341005227A0D444FD40807D1D382BA0034100361AC00542FD4000470378919F0341006E3480373EFD40004703F8B69E034100E4F21F583BFD40802BF617459E034100AED85F0837FD4000470378FA9D0341006E34807831FD40004703F89E9D0341005227A0482CFD40802BF697539D034100AED85F7828FD40807D1D38319D0341005227A0C723FD4080621058469D034100CAE53F971FFD40807D1DB8339D034100361AC0081BFD40806210D8609C0341005227A06A16FD40806210D85D9B034100CAE53FCB13FD40802BF617049B0341001C0DE0CB0FFD40802BF697899A034100361AC05B0BFD40806210D8569A0341001C0DE08A07FD4000470378849A034100E4F21FC903FD400010E937029B03410092CB7FC700FD4000470378889B0341005227A0F4FCFC400010E9377E9C034100E4F21FD0F8FC40806210D8139E0341000000006DF6FC4000470378329F0341006E348058F5FC40004703F8F9A0034100E4F21F84F5FC4080621058BAA2034100361AC060F6FC40802BF6172BA40341005227A05CF7FC40806210D8DBA50341005227A056F7FC40807D1D3844A8034100CAE53F61F7FC40004703786CAA034100000000EEF6FC4080621058BCAB0341006E3480DAF5FC40806210D813AD0341001C0DE0B4F3FC40806210D83AAF0341001C0DE0BFF1FC40806210D821B1034100E4F21F7BEFFC400010E9B7F0B2034100E4F21F86EDFC400010E9B7DFB40341001C0DE0B1EBFC40807D1DB87EB6034100AED85FCCE9FC40806210D89DB80341006E3480E9E9FC40802BF617BEB9034100361AC0E6EAFC40806210D8E6BA0341001C0DE0F4EBFC40802BF697BFBB0341005227A0E3ECFC400010E93748BC0341001C0DE0D2EDFC40806210D8A0BC034100E4F21F82EFFC40004703F801BD0341005227A041F2FC40807D1DB85BBD034100AED85F02F6FC40802BF61746BD0341006E348072FAFC40806210D870BD0341001C0DE071FDFC40807D1DB8DABD034100E4F21F2F00FD40802BF69714BF034100000000EC00FD40807D1D385DC0034100E4F21F0901FD40004703788DC1034100AED85FA600FD4080621058A5C2034100AED85F54FFFC40802BF6975CC3034100000000B3FDFC40807D1DB8DBC30341006E3480C2FAFC40806210D8E1C30341006E3480A2F7FC40806210D8BFC3034100CAE53F72F4FC40806210D8ADC3034100CAE53F21F2FC4000470378ECC3034100000000F0EFFC400010E93753C4034100361AC05EEEFC40807D1D38C2C40341006E3480ACECFC408062105891C5034100CAE53F1BEBFC408062105800C60341001C0DE09AE7FC40807D1D38F6C50341000000000CE6FC40807D1D3875C50341005227A0BDE4FC4080621058B4C4034100AED85F50E3FC40807D1D388BC3034100E4F21F83E1FC40004703F859C20341005227A025DFFC408062105838C10341006E348016DEFC40802BF697C7C0034100361AC0B6DCFC40807D1DB89EC0034100E4F21F97D9FC40806210D84CC0034100361AC0D7D4FC40806210D8D1BF034100361AC027D0FC40806210D88EBF034100361AC057CCFC40004703785CBF034100AED85F27C8FC40806210D851BF034100AED85F27C4FC408062105817BF0341001C0DE017BFFC40802BF617ACBE0341005227A099B8FC40004703F89FBD034100CAE53F7BB5FC40004703F8C5BC0341000000004DB4FC40802BF6170DBC034100E4F21F5FB3FC40004703781CBB0341006E348052B2FC40802BF697BBB90341005227A014B2FC4080621058D3B8034100CAE53F47B2FC4080621058CBB70341006E348059B3FC40806210D8F3B60341000000005BB4FC400047037864B6034100361AC00CB7FC40004703F8D5B50341001C0DE07EBAFC40802BF61728B5034100AED85FB0BDFC40004703F8B1B40341005227A041C1FC40807D1D3864B40341001C0DE052C4FC40004703F80DB4034100361AC073C6FC4080621058C7B30341005227A0F5C8FC40806210D828B30341006E3480E9CAFC40806210D8B9B1034100CAE53FFCCBFC4080621058AAB00341006E34805FCDFC40802BF6176BAF0341005227A023CEFC4080621058D3AD034100E4F21FC7CDFC40004703F85AAC0341005227A07BCCFC40806210D881AA034100AED85F4ECBFC40004703F850A9034100AED85F10CAFC40802BF61770A80341001C0DE042C8FC40806210D85EA70341001C0DE0F5C5FC400010E93705A6034100361AC0B7C4FC408062105834A5034100E4F21FDAC3FC40807D1DB83BA4034100000000DDC2FC40004703F8FAA20341005227A0C0C1FC40802BF61782A1034100000000D4C0FC408062105811A003410092CB7F08C0FC40802BF697389E034100CAE53F1DBEFC40802BF617379C0341006E3480F0BBFC40802BF697CD9A0341005227A0A2B9FC40802BF617CC99034100AED85FF4B7FC40806210D80299034100AED85FA5B6FC40004703F889980341006E348077B4FC40802BF69790970341005227A009B2FC40004703F89E96034100361AC05BAEFC40004703789495034100361AC0CDA9FC40802BF6978994034100E4F21F7FA4FC40802BF617B693034100361AC0FFA1FC40802BF6975493034100E4F21FA0A0FC40802BF6971B93034100361AC0EF9CFC408062105811930341001C0DE0CE99FC40004703783F930341005227A04C96FC4080621058FD930341001C0DE04993FC40802BF697EB940341006E3480F790FC40807D1D38C2950341005227A0B48EFC40004703F8C896034100AED85F908CFC40806210D86798034100AED85F4C8BFC4080621058F7990341006E3480388AFC40806210D86E9B034100E4F21FD689FC40807D1DB85E9C034100361AC06289FC40802BF697BE9D0341005227A0EE88FC4000470378569F0341005227A0CB88FC40802BF6978EA0034100E4F21F9689FC4080621058D7A20341006E3480628AFC40802BF61750A4034100E4F21F8C8BFC40802BF617F9A6034100CAE53FE68BFC40802BF69751A903410092CB7F418CFC40802BF61742AB0341000000006D8CFC400047037812AD034100AED85F388CFC40802BF697EAAE0341005227A0258CFC40802BF6970AB0034100000000748AFC40807D1DB899B0034100CAE53FF487FC40802BF61760B0034100361AC00687FC40806210584FAF034100E4F21F2986FC40807D1DB84EAE0341005227A00B85FC40806210D845AD034100CAE53FBE83FC40004703F81CAC034100AED85F9182FC40004703F8CBAA034100AED85FD480FC40806210D882A90341001C0DE0767EFC400010E93769A8034100361AC0097CFC40004703781FA7034100E4F21FBC79FC40004703F80DA6034100CAE53F8E77FC400047037814A50341005227A0BF74FC40802BF69762A40341006E34809071FC40802BF697D8A3034100E4F21FF16DFC40807D1D386EA30341005227A0016BFC40004703780CA3034100E4F21F0267FC40806210D8A1A20341006E34802263FC40004703784FA2034100AED85F535EFC4080621058B4A10341005227A03459FC40802BF617F1A00341006E34807655FC40802BF697FE9F0341001C0DE0C852FC40806210D8E49E034100CAE53FAB50FC4080621058D39D034100CAE53F2C50FC40004703F8629D0341001C0DE08D4FFC40802BF697BA9C034100000000204FFC40807D1D38D29B034100AED85FF34EFC40806210D8799A0341006E3480254FFC40806210D899990341001C0DE0774FFC40004703F8A998034100000000AB4FFC40004703F86997034100361AC0CD4FFC40806210D85996034100CAE53F314FFC4080621058E194034100AED85FD34EFC40004703F80894034100E4F21FD64CFC40802BF697CF920341005227A0384AFC40806210D8A591034100AED85F9A48FC40807D1DB8DC90034100CAE53FAC45FC40806210D8FA8F0341001C0DE07E42FC40807D1DB8B88E03410092CB7F4040FC40807D1D38FF8D034100AED85F213FFC4000470378968D0341001C0DE0413EFC40806210D84D8D0341000612146E3CFC40808D06F0C48C03410096D4092040FC408079C7A9D88B034100508D97BE41FC4080ED7C3FBF8B0341006ABC744F42FC4080105839AC8B034100DCF97E2843FC400079E9268B8B03410090C2F52A43FC40801283C08A8B03410046B6F39D44FC4000AAF1D24A8B0341004A0C024945FC40005C8FC2FD8A034100FCA9F16445FC4000560E2DF18A034100986E12CB46FC40803D0AD74F8A034100BC05124647FC4080976E920F8A034100CAE53F6B48FC4080621058C6890341008CDB68B748FC4080A7574AA8890341005EBA49CE49FC4080BE9F1A5989034100BA490CDA4DFC4000D7A370E187034100263108E64DFC40007F6ABCDA8703410030992AF94FFC4000D6566CB186034100ECE236F94FFC40008BFD65B1860341006844691752FC4000F9A0E7818503410024287E1752FC4080AC8BDB8185034100BE30991752FC4000A8C6CB818503410078E9261852FC40806CE77B81850341008A4160FB52FC4000A8C64B0285034100E6D0220F54FC4080EB51B867840341008E976E1654FC4080F1D24D6284034100E6D0221D54FC4000CFF7535D8403410004560E6354FC4000B0726829840341009CC4208254FC40806210581284034100B6F3FDFE55FC408091ED7CB782034100AE47E19C56FC4080F3FDD4198203410084C0CAB758FC4000A245B6E97F0341004E6210AC58FC4000CDCCCCE57F034100A69BC49458FC4000DD2406DE7F034100285C8F7C5BFC4000D122DB107D034100F4FDD45C5CFC4080BC7493397C034100EA2631485DFC408047E17A657B03410084C0CAB95DFC40808FC2F5F27A03410052B81EC55DFC408095438BE77A034100F0F44A0C5EFC4080832FCC9F7A034100000000105EFC40806210589E7A0341005227A0A95EFC40806210588E7A034100CAE53F035FFC4080621058867A034100AED85F6E5FFC4080621058627A03410082E2C7AA5FFC400057EC2F447A0341008C6CE7C55FFC4000508D973A7A0341007EFB3ADB5FFC4000DFE00B337A0341009CC420F65FFC40004C3789297A034100E6D022FF5FFC4080A60A46207A0341000000000860FC40807D1DB81C7A0341005227A05960FC40004703F8E379034100ECE2365460FC4000FA7E6AB07903410096438B5C60FC4000FCA9F1A079034100B81E856560FC40803789414E790341008816D93660FC408047E17A1979034100C2F528F65FFC4000A4703DEF7803410086EB516A5FFC4000075F189F780341004C3789275FFC4000AAF1D27878034100F0A7C6D05EFC40002DB21D4778034100E6D022C35EFC4080F1D24D3F7803410016D9CE995EFC4080378941D57703410030DD24345EFC40808D976E5C77034100A223B9F55DFC40805BB13F1277034100744694F55DFC40004ED111127703410028ED0DF35DFC4080E17A140F770341004CC807F35DFC4000DF4F0D0F770341008E976E3A5DFC4080C420B03376034100BE9F1AAB5CFC40806CE7FBC6750341008204C5AA5CFC4000ABCFD5C6750341005227A0A15CFC40802BF697A975034100361AC0645DFC40004703F83375034100000000305EFC40807D1D38DF74034100CAE53FB35EFC4080621058B274034100000000E05FFC40802BF6976D74034100361AC01C61FC40807D1DB82C74034100506B9A3361FC400008AC9C247403410004560E4B61FC4000508D9720740341003E9B557E61FC400075021A0A74034100000000C861FC40004703F8EF73034100CAE53F6362FC40004703F8CF73034100AED85FF662FC40807D1D388F730341000000005063FC40802BF6972973034100CAE53FF363FC40004703F8A772034100361AC05C64FC40802BF69729720341005227A03165FC40004703F8AB71034100AED85F2666FC40806210582671034100CAE53F2B67FC40807D1DB8BC7003410014AE47B867FC4080D3BCE385700341005C8FC2D567FC40801804567C7003410054E3A5DE67FC40806B09F97670034100AED85FE667FC40004703F8737003410078E926DF68FC4000CD3BCE25700341007A14AE8F69FC408039B4C80C700341007268919B69FC4080BA490C0B700341001C5A64676AFC4000D34D62ED6F034100C420B0326CFC40008716D9986F034100D24D62596DFC40809ECD2AB96E034100AED85F5E72FC40802BF697FD6B034100E4F21FAD71FC408025E483796B034100221FF4C571FC4000DCD701676B034100F4FDD4CE71FC40801C5A64606B034100365EBAD171FC4080C420B05D6B0341002C8716E971FC40008716D9476B034100BE9F1AF371FC4000DBF97E3E6B0341006666668271FC4000068195B16A034100C2F5283471FC4000B29DEF8F6A03410048E17A2871FC4000D578E98A6A034100448B6C2571FC40800C022B5C6A034100FA7E6A2471FC4000508D974F6A0341002406812171FC400079E9262B6A0341007E6ABC2A71FC4080BA490C196A03410078E9262B71FC40005EBA49186A034100BE9F1A9F71FC4000295C8F506903410046B6F33B72FC4000FCA9F18868034100AAF1D24D72FC4000EA95327268034100AED85F3675FC40806210581669034100361AC00C7EFC40802BF6976161034100000000E080FC40004703F8A761034100E6AE25EB82FC4080627F59AF5F034100CAA1459286FC40803D0AD7D95C034100F6B9DA9786FC4000E3369AD45C0341002E6EA39B86FC4080FE43FACE5C03410070F0859D86FC4080764F1EC95C0341006ABC749D86FC40809F1A2FC35C034100D8A3709B86FC4080180456BD5C03410040A4DF1986FC40006C787A9D5C034100AED85F6E86FC40004703F84B5C034100000000A06AFC40806210588655034100361AC00470FC40806210582A50034100361AC0AC75FC40807D1DB8DC4E034100FA7E6AAE75FC4000EEEB40DC4E03410060764F7277FC4080DA8AFD874E03410084C0CAB577FC4000B072687B4E034100EAB7AF9877FC4000A8C64B534E034100AED85FE677FC40802BF6973D4E03410004560E9772FC40009487056D47034100B81E853F72FC4000D34D62F44603410092ED7CC170FC408016D9CEEA44034100C21726E46FFC40007D3FB5D143034100000000F87CFC40806210586241034100FC1873FC7CFC4000508D973D410341008204C5FC7CFC4080423E683D410341001461C3027DFC4080E09C9138410341000C2428077DFC4080A9825133410341001873D7097DFC4080F085C92D410341005CB1BF0A7DFC4080E3141D2841034100D022DB097DFC4000D7A3702241034100DE9387027DFC40801B0D600B41034100361AC0247DFC400010E9B7F03F034100D634EF537CFC4080C954C1E53E034100022B87427CFC40808FC2F5AE3E0341008A41600A7CFC4080D100DEF43D034100361AC0E47BFC40802BF6973D3D034100D49AE6C27BFC4000000000083D034100BC7493927BFC40008716D9673C034100D6C56D907BFC40004A7B03633C0341004872F98C7BFC4000D26F5F5E3C03410014AE47887BFC40806E12035A3C0341008E976E827BFC4080931804563C034100CCCCCCF67AFC4000022B87033C034100CAE53F7B7AFC4080621058923B0341005227A0D179FC400010E937FB3A0341005227A08979FC40802BF697893A034100AED85F8E78FC400010E9372F3A034100C420B0DA76FC40804772F99839034100A8C64BF975FC408095438B1439034100A4DFBE4B75FC40802EFF21B338034100CAE53FC374FC40004703F81F38034100CAE53F2374FC40807D1D38D736034100361AC0E473FC4080621058D2350341002EFF219273FC400030BBA7D934034100AED85F8E73FC4080621058CE34034100100BB58473FC4080B88D06B73403410084C0CA6373FC40002B87166234034100F038455973FC4080ED7C3F4E34034100AED85F0E73FC40802BF6979933034100AED85F9E72FC4080621058DE320341000000006072FC4080621058DE310341005227A07972FC4080621058E630034100B8AF037C72FC4080A52CC33E300341006E12838872FC4000B0726834300341000C022BB372FC4000FA7E6AE12F03410040355EB472FC4000819543DF2F03410004560EC772FC40803D0AD7962F034100A01A2F2F73FC4000D7A370032F034100C64B372F73FC4000D34D62032F03410078E9263973FC4000068195F12E0341008E5374BC72FC40008C4A6A312E0341005227A0B172FC40004703F8172E034100000000F071FC40802BF697812D034100361AC07C70FC40004703F8532C0341005227A0716FFC40004703F8872B034100F241CF1A6FFC400066F764032B034100448B6C196FFC4080749318E92A0341001A51DA166FFC4000363C3DB82A03410078E926116FFC4000D9CEF74B2A034100388941166FFC4080EFA7C6E329034100EE7C3FED6EFC40008716D96429034100105839F26EFC4080438B6C5E2903410052B81EB56EFC4080EB51B82D29034100FC1873126EFC40808D06F04628034100CAE53FFB6DFC40802BF6970928034100CAE53F636DFC40806210584E27034100000000306CFC40802BF6975126034100240681D96BFC4000A4DFBE1226034100C0CAA1916BFC408016D9CECC25034100B4C876086BFC408060E5D03F2503410022DBF9F46AFC400054E3A52E25034100621058E76AFC4000A69BC41D25034100560E2DD26AFC4080931804082503410092ED7CC76AFC4080703D0AFD2403410060E5D0C66AFC40806E1283F52403410076BE9F966AFC4000D122DBB924034100D8A370556AFC4000AE47E183240341006CE7FB896AFC4000022B8752240341005C8FC29F6AFC4000A8C64B262403410014AE47D36AFC4080F1D24D1F24034100105839EE6AFC40003333330924034100E61DA7F26AFC40806ABCF404240341008C4AEAF56AFC4080FC87740024034100FC87F4F76AFC40003945C7FB230341007E6ABCF86AFC4080490C02F723034100F6285CF96AFC408047E17AE2230341008816D9E86AFC408016D9CED523034100B07268EB6AFC4080B81E85BB230341001283C0FA6AFC4000333333B42303410076BE9F006BFC4000D34D62F222034100263108006BFC4080ED7C3FED220341000C022BE16AFC4080ED7C3F7522034100900F7ADF6AFC4080D0B3D970220341005839B4DC6AFC4000508D976C22034100E04F8DC96AFC4000B0726854220341004260E59C6AFC40002FDD244E22034100FC1873726AFC4080C5FE324822034100C420B0686AFC4080F3FDD4462203410016D9CE6D6AFC4000ABCF5542220341008816D98C6AFC4080BE9F1A27220341002406819B6AFC4080ED7C3F1A22034100D478E9906AFC40801A2FDD0C22034100285C8F886AFC4000A8C64BAB2103410086EB51886AFC4000295C8FA9210341000681957B6AFC408060E5D06321034100CAA1457A6AFC40809465885F21034100B29DEF776AFC4000F6285C5B210341007A14AE696AFC40000AD7A35C21034100BE0E9C266AFC4000C286A762210341002EB29DDB69FC40008941606921034100065F98CA69FC40802BF6975421034100CAE53F0B6AFC400010E937FB20034100CAE53FD369FC40807D1DB86C200341008816D9B769FC4000F1F44A5D200341003E0AD7A769FC40002DB29D4B20034100E4A59BA469FC4000B4C8764820034100A8C64BA169FC40801A2FDD452003410054E3A56569FC400052B81E1B2003410072F90FE968FC4080931804DD1F0341005227A0B168FC40802BF697A91F034100CAE53FCB68FC40004703F8271F0341005227A05968FC40802BF697A51E03410054C1A85F67FC40806210D82B1E03410036CD3B5E67FC4000A7E848291E034100C4B12E5967FC40806891EDE31D0341000AD7A35867FC400054E3A5B61D034100E02D905567FC4000ED0D3EB21D0341005227A04967FC40806210580E1D0341005227A0E966FC40802BF697DD1B0341008EE4F20867FC400013F2C1721B0341003AB4C80C67FC40005A643B6D1B0341005E4BC80B67FC4080742417691B034100361AC01C67FC40807D1D382F1B0341001C7C61DA66FC4080490C02DD1A0341000AD7A3E666FC4000D578E9B41A034100105839EA66FC4080C2F528A91A034100240681E166FC4080F3FDD4941A034100F831E6C266FC400073D7924D1A034100AED85FEE66FC4080621058E619034100024D849F66FC408091EDFCAB19034100FCA9F19C66FC408016D9CE90190341006CE7FB8F66FC40806E12837A190341009A99997566FC40807493184D190341004C3789D165FC400021B072CA18034100C420B0C065FC4080976E12A618034100B29DEF4D65FC4000894160D9170341006891ED4465FC4080B6F3FDCE1703410048E17A2065FC4000A8C64B8817034100C64B37BF64FC408060E5D0DA16034100000000B464FC40801C5A64B91603410060E5D0A664FC4080E17A149216034100DCF97E9464FC4080703D0A1A160341008C6CE77B64FC40809BC420E71503410008AC1C2264FC408045B6F35D15034100B6F3FD2064FC408047E17A45150341006E12831C64FC4080F1D24D3E150341009CC4202064FC40806E128332150341009A99991B64FC40801C5A64CF14034100D478E91A64FC408031772DCB14034100A4703D1964FC4080029A08C714034100E4A59B1664FC4080931804C3140341005A643BD963FC40002FDD247514034100E4A59BD463FC408062105870140341003AB4C8CE63FC4080703D0A6C14034100666666F062FC400008AC1CDF13034100B29DEF6162FC400085EB5129130341006ABC744162FC4080E3A59B9212034100DCB5843F62FC4000F46C568D12034100BA6B093C62FC408081044588120341002C87163762FC4000014D8483120341003AB4C83062FC40809F1A2F7F12034100263108D261FC40808B6CE747120341006666668261FC4000508D97D311034100B072687B61FC4080621058CE11034100F0A7C67961FC4080B6F3FDC611034100B81E854361FC4000355EBA77110341006EC5FE3E61FC400052B89E7211034100FCA9F13861FC4080438BEC6D110341004A7B833161FC4000EEEBC06911034100AE47E12861FC40007D3F3566110341006666665E60FC40808D976E1F11034100F0A7C6695FFC40002731086E10034100E8FBA9655FFC4000CFF7536B100341002EB29D5F5FFC40002FDD2468100341009CC420205FFC4080B81E854A100341006E1283165FFC40001214BF46100341005C8FC20B5FFC40803D0AD74310034100AE47E1925EFC40002506812910034100068195875EFC4000FCA9F12310034100C876BE9D5EFC40801E85EB0B10034100166A4DA05EFC4080CA32C40810034100A4703DA25EFC40002506810510034100FCA9F18A5EFC408047E17AFF0F0341005C8FC2975EFC40801283C0D90F03410048E17A885EFC40000AD7A3B30F03410088855A865EFC4080E1E995AF0F0341003A014D835EFC4000C4B1AEAB0F034100F6285C7F5EFC40806CE7FBA70F034100829543F75DFC400085EB51380F034100F0A7C6FD5DFC4080490C02F70E034100A245B6C75DFC408016D9CEB10E034100B81E85CB5DFC4080C0CAA1890E0341000AD7A3C65DFC40005839B4840E034100E6D022D15DFC4000D7A3704E0E0341008C6CE7D15DFC4080F1D24D460E0341003208ACC25DFC4080B81E85190E03410008AC1CB45DFC4000AC1C5A110E034100B29DEFB55DFC40004E6210F40D0341008A4160B15DFC40007B14AEE60D034100F0A7C69D5DFC40809F1A2FB90D034100A8C64B7F5DFC40000000009D0D034100AC1C5A385DFC40000AD7A31A0D034100A2D634355DFC40002B8716160D03410040A4DF305DFC4080F085C9110D034100B22E6E2B5DFC4080847CD00D0D03410022DBF9245DFC4000A4703D0A0D03410028ED0D205DFC4080F5B9DA070D034100CAE53FF35CFC40802BF697CD0C034100782D21B35CFC4000925CFEAC0C034100B29DEFB55CFC40801C5A64970C0341004A0C02AB5CFC4000894160910C0341005C2041795CFC40004ED1918F0C0341003CBD52425CFC4000780BA4730C0341002E90A0435CFC4000068195690C034100AED85F1E5DFC4080621058EE0B034100000000305DFC40807D1D381F0B0341005227A0415DFC40807D1DB87C0A034100CAE53F135DFC4080621058E209034100000000385CFC40807D1D383B09034100AED85F8E5BFC40807D1DB8C808034100361AC0E45AFC40806210583E08034100361AC08C5AFC40004703F88F07034100AED85F8E5AFC40802BF697F506034100000000705AFC400010E9376F06034100361AC0445BFC40807D1DB8A405034100AED85F365BFC40806210580E05034100022B87165BFC40802A1895EA04034100B4C8761C5BFC4080C420B0D5040341001804560A5BFC408014AE47BB040341000AF9A0DF5AFC40804469EFAC04034100361AC0D45AFC40807D1DB8A004034100361AC08C5AFC40806210581204034100000000E05AFC40807D1DB89C03034100CAE53F4B5BFC40004703F83703034100361AC0745BFC4080621058D6020341005227A0495CFC40004703F82F02034100CAE53F835CFC40802BF6977501034100AED85FE65CFC40807D1D38D700034100000000E85CFC400010E9B73000034100506B9AE85CFC40801E16EAF5FF024100C2F5282C5DFC4000CDCCCC85FF02410046B6F32E5DFC400011C73A82FF0241008816D9325DFC40806891ED7EFF02410090C2F5945DFC40809BC42039FF0241008A4160AD5DFC4000D9CEF73AFF024100BA490C325EFC408010583928FF0241009EEFA7545EFC4080B6F3FD26FF024100B4C876585FFC40002B8716FDFE0241002631085A60FC40806ABC74C6FE0241008E976E4661FC400033333375FE0241002EB29D4961FC4080E9263174FE02410084C0CA4D61FC400052B81E73FE024100986E123162FC4000295C8F3FFE024100DCF97EAA63FC4000FED47833FE0241000000003E67FC4000B4C87616FE024100B6F3FDEC67FC400008AC1CFAFD0241002406810768FC400079E926F8FD024100560E2D4468FC40806CE7FBEBFD024100BC74936669FC40806891EDBCFD0241008E976E006AFC40808B6CE7D2FD02410030DD24086AFC40803F355ED4FD0241009EEFA7406AFC40004C3709E2FD024100AED85F2E6BFC40807D1D3873FE024100CAE53F736CFC408062105806FF024100AED85F4E6DFC40004703F85FFF024100B20C71BA6DFC4080DC468384FF024100986E12E16DFC4000508D9798FF0241003CDF4F8370FC40005839B47300034100ECC0398870FC40003D9B55760003410022DBF9A270FC40003108AC8400034100A245B61F71FC408091ED7CC700034100FC87F43171FC400012143FD10003410008AC1CF871FC4000CBA1453B01034100E27A142C72FC40801C5A6434010341001058396072FC4000CDCCCC3301034100388941E672FC4080BE9F1A2201034100A245B63073FC4080BDC11731010341008816D99874FC40002DB29D79010341003AB4C8DA74FC4000F853E38601034100C420B0E674FC40004E62107D01034100EA26310A75FC408039B4C85F01034100BE9F1A1575FC4000A69BC45601034100DE2406C376FC4080D0B3591600034100B459F52B77FC4080C1A824C8FF0241008A4160F578FC40005A643B73FE024100703D0AF778FC4080E17A1472FE024100FC87F4567BFC40005B423EE2FC0241003CDF4F337CFC40803BDF4F51FC024100EE5A420C7DFC4080081B1EC3FB024100CAE53FB380FC40004703F807FD024100AED85FB683FC400010E9B7C4FD024100361AC0B485FC40004703F847FE024100361AC07C87FC40807D1D38A7FE024100AED85FAE8AFC40807D1DB8A0FF024100AED85FCE8CFC40802BF6976500034100361AC0CC8EFC40802BF6971D010341005227A03190FC40004703F89301034100361AC0EC90FC40802BF697DD010341003C4ED1FF90FC4080CA32C46401034100B6F3FD5691FC400033333319010341009C33A25B91FC4080D2DE601401034100F085C95E91FC4080CA32440F01034100D24D626091FC400023DBF909010341003AB4C80C91FC408014AE47D4FF024100C0CAA11B90FC408039B4C8EEFD0241006ABC741B90FC4000250681EEFD024100FC3A70E18FFC408067B36AC4FD024100EC9E3CE38FFC4000166ACD07FD024100F853E3FD8FFC408016D9CE82FC0241007C3F35FE8FFC408091ED7C80FC0241005EBA490090FC40002FDD2463FC02410054E3A5F78FFC400004568E39FC024100361AC0FC8FFC408062105806FC0241005227A01990FC40807D1DB89CFA024100361AC03490FC40806210581AF90241000000005890FC4080621058FEF7024100361AC0D490FC4080621058CEF60241005227A07191FC4080621058BEF50241005227A09992FC4080621058A6F40241000022FD8F93FC4080855AD3ECF30241005C8FC2E193FC4080BE9F1AC4F3024100D022DBE193FC400004560EC4F3024100ECE2364F94FC4080A679478AF30241002E90A0AD94FC4080F8316658F3024100CAE53FFB95FC40802BF697D1F2024100000000D896FC40802BF69795F2024100CAE53FE397FC40806210589EF20241000000002099FC40004703F8CFF2024100361AC0749AFC40802BF69709F3024100000000D09AFC40004703F8B3F202410098FF90D39AFC408016D9CE97F2024100DE2406FF9AFC400079E92685F20241004694F6069BFC4080E7FB2981F202410004C58F0D9BFC4000083D9B7CF202410054C1A8129BFC4000E2589777F20241002EFF21169BFC4000363C3D72F2024100CAA145169BFC40803A01CD71F202410066F7E4179BFC4000C4B1AE6CF20241008C6CE7179BFC400004560E67F20241009EEFA7CE9AFC40007B14AE8EF00241005C8FC2CD9AFC40800C022B8BF0024100F853E3859AFC40801E85EBC0EF0241002C8716839AFC4000250681BBEF02410052B81E3B9AFC408045B6F352EF024100BE0E9C369AFC40001B9EDE4DEF02410098BB96309AFC4080A0F83149EF024100343333299AFC4080703D0A45EF0241007A58A82399FC40804DF30EC7EE024100000000F098FC400010E9379BEE024100361AC08C99FC40802BF6973DEE024100CAE53FFB9AFC40004703F883ED0241005227A0499CFC40807D1D380FED0241005227A0019EFC40807D1D38D3EC024100AED85F0E9FFC408062105876EC024100000000F0A0FC40807D1DB8E4EB024100361AC0E4A1FC40004703F883EB024100FE65F73DA2FC4080F38ED32DEB024100CEF7531FA3FC4080BE9F1A87EA0241002497FF26A3FC4000BDE31480EA024100501E166AA3FC40007958A842EA024100AED85F7EA3FC40004703F833EA024100361AC0BCA4FC40802BF6975DE9024100AED85F6EA6FC40802BF69731E802410000000000A8FC408062105816E702410000000068A9FC4080621058F6E5024100000000D8AAFC400010E937D7E4024100361AC094ABFC40807D1DB830E402410000000038ACFC40004703F8E7E30241005227A061AEFC40004703F883E3024100361AC09CB0FC40807D1D38EFE2024100CAE53FFBB1FC408062105892E2024100789CA265B2FC4000CCEEC938E2024100986E128DB2FC4080C420B02FE2024100C2F5280EB3FC4000F853E3DEE10241002EB29D15B3FC40003108ACCCE1024100C8073D58B3FC4000E25897A1E1024100EE7C3F69B3FC4000508D9796E1024100787AA57BB3FC4080DE718A5FE1024100361AC09CB3FC408062105856E1024100361AC0A4B4FC40807D1DB8D0DF024100D8A37019B5FC408072F90FD7DE0241005C8FC283B5FC4000D9CEF74ADE0241003CBD527DB5FC4000EBE2362ADE024100F21FD27AB5FC408047E17A1DDE024100182653B0B5FC4000A69BC4C3DD024100FE43FA2EB6FC40808D976E4BDD024100CAE53F9BB6FC408062105806DD0241005227A0D9B7FC400010E9B76CDC02410000000050B9FC40807D1D38D3DB024100AED85F5EBAFC40806210585ADB024100000000C0BAFC40802BF6971DDB024100D42B65DCBAFC4080DA8AFDF8DA0241006ADE7141BBFC40809BC420CCDA024100BE9F1A8FBBFC400054E3A5A9DA024100643BDF93BBFC400021B0728BDA02410056302A90BBFC4080AC8B5B7DDA024100361AC0CCBBFC40004703F883DA02410000000050BCFC40004703F86FDA0241005227A091BCFC40806210582ADA024100C2F528A2BCFC4080499D00CBD9024100388941A2BCFC4000D9CEF7CAD9024100B4C876C8BCFC40005F98CCB7D90241009A2A1860BDFC4080C807BD6BD9024100000000B8BDFC40004703F85FD9024100361AC0C4BEFC40004703F843D9024100000000C0BFFC40807D1DB840D9024100361AC0B4C0FC40004703F8DBD8024100361AC0FCC1FC408062105836D8024100000000A8C2FC40802BF697F1D7024100361AC0ACC3FC40807D1DB8A0D70241005227A0B9C4FC40807D1D381FD7024100AED85F8EC5FC40802BF69789D602410000000098C5FC40004703F833D6024100AED85F1EC5FC40807D1D3807D6024100361AC094C4FC400010E9B7A0D5024100AED85FD6C4FC40807D1D383BD5024100AED85FBEC4FC4080621058E2D40241005227A001C3FC40807D1DB8F4D3024100AED85F06C2FC40802BF69751D3024100361AC004C1FC40807D1DB854D20241005227A0D1BFFC40802BF69791D1024100000000D0BEFC40004703F883D0024100AED85FCEBDFC40004703F893CF024100CAE53FE3BCFC40004703F8C3CE0241005227A001BCFC40807D1DB828CE02410000000010BBFC40004703F833CD024100AED85F1EBAFC40004703F84BCC024100361AC094B9FC400010E9B7A4CB024100361AC014B9FC400010E9B7D0CA024100361AC0D4B8FC40806210584ACA024100CAE53F3BB8FC40004703F8B7C90241005227A041B7FC40004703F8DFC80241005227A0C1B6FC40004703F8EBC7024100CAE53F8BB6FC40004703F80BC70241004850FC39B6FC4080E09C117AC60241006ABC743EB6FC4000DAAC7A78C60241009CC42044B6FC4000B4C87676C60241008816D919B6FC408079C7A91CC60241006E12830CB6FC408062105800C602410000000098B5FC4000D9CEF708C50241007E1D3841B5FC400010E9B750C4024100C876BE3BB5FC408074931845C4024100EE7C3FCFB4FC4080621058D7C3024100F44A5947B4FC40801CEBE289C302410024068135B4FC4080EB51B87FC3024100CAA14530B4FC4080C2F52872C30241004A0C0203B4FC40803D0AD7FCC20241003208ACE1B3FC40000A6822CCC2024100000000B8B3FC40808104458FC2024100000000B8B3FC400010E93773C20241004CA60AB5B3FC4080F1D2CD34C20241008E976EC2B3FC4000F853E3E5C102410084C0CAB5B3FC40808D976EC8C1024100EE7C3FBFB3FC4080749318A3C1024100FC3A70A5B3FC40005F984C8AC1024100221FF493B3FC40006E348079C10241005227A091B3FC40807D1D3873C102410030BB27DEB3FC4080976E9282C1024100A4703D18B4FC408060E5D08FC102410024287E17B4FC40804E40138EC1024100361AC05CB4FC40004703F89BC102410000000008B5FC40802BF697D9C10241005227A091B5FC40802BF697CDC1024100361AC02CB6FC40802BF697B9C1024100000000C8B6FC40807D1DB86CC1024100361AC0A4B7FC4080621058C6C0024100361AC01CB9FC40802BF69769BF024100325530DBB9FC4080136143FDBE0241007EFB3ADBB9FC4000363C3DFDBE0241004E6210C6BCFC4000DD2406CFBD02410084C0CA11BEFC4080C0CAA1EBBC024100F263CC3EBEFC40806B9AF7F2BC024100BE9F1A87BEFC40801283C0FEBC024100F6285C8FBEFC4000A4703DF9BC024100F4FDD40EC0FC408039B4C83ABD0241003208AC10C0FC4080E17A143BBD0241009CC4204CC2FC4000A4703D94BD0241004EF38E59C2FC40803480B78FBD02410000000090C2FC40004703F89BBD024100DA3D793BC4FC408072F98F21BD024100141DC973C4FC4080DD938753BD0241005227A091C4FC40807D1DB884BD024100CAE53F33C5FC4080621058F2BD024100361AC034C6FC40802BF6978DBE0241005227A0B1C7FC40802BF69731BF024100AED85F46C9FC40802BF697FDBF024100000000E8C9FC40004703F88BC0024100361AC084C9FC40807D1D383BC1024100B6A6792DC9FC4080AEB662A1C10241006ADE712DC9FC4080FAED6BA1C10241006CE7FB71C8FC4080BA490C2CC20241007AC7296CC8FC4080C7293A31C2024100C05B2068C8FC4080F2B0D036C20241006CE7FB65C8FC400055C1A83CC202410084C0CA65C8FC408099999942C20241005EBA4976C8FC4000355EBABEC2024100361AC077C8FC400020D2EFC3C20241005817B77AC8FC4000FFB2FBC8C202410052B81E7FC8FC40005C8FC2CDC2024100AE69DE84C8FC400055302AD2C20241003E0AD78BC8FC4080BE9F1AD6C2024100B4C8767EC9FC408047E17A4CC3024100F4FDD480C9FC40000681954DC3024100501E16B5C9FC408089B0E164C3024100AED85FA6CBFC40806210588AC4024100361AC0CCD4FC40807D1D38AFC0024100000000D0DAFC40806210587EC3024100CAE53F03DDFC40807D1D3853C202410000000098E0FC408062105882C3024100CAE53F13E4FC40802BF697DDC4024100CAE53F23E9FC40807D1DB8D0C602410000000000EAFC40807D1D382FC6024100361AC044EDFC40807D1D38BFC602410082514900EDFC408025E48350C602410084C0CA03EDFC4080643BDF50C602410094180408EDFC40005A643B51C60241001283C0F6ECFC4080847CD02AC60241008A4160D5ECFC4000508D97E0C502410052B81E79ECFC408074931860C5024100B88D0619ECFC4080779C2286C4024100643BDF13ECFC4000B4C8767AC402410018B7D1E2EBFC400039454752C40241001873D7A5EBFC408061325520C4024100CAE53F6BEBFC40802BF697C1C30241005227A029F1FC40806210582EC1024100D0D55614F2FC4080A8354DC1C102410004560EA1F1FC4000AC1C5AECC102410062105865F1FC4080C2F52807C2024100E63FA460F1FC4080F3FDD40CC2024100B29DEF5DF1FC4000F31FD212C2024100CE3B4E5DF1FC4080B3EAF318C2024100A69BC45EF1FC4000287E0C1FC2024100F0164862F1FC4000FBCBEE24C2024100C876BE67F1FC40808D976E2AC20241000E2DB296F1FC4080923A0151C202410000000070F1FC40807D1D386FC202410008AC1C5DF2FC4000D578E9F9C20241008816D984F2FC40808FC2F522C302410084C0CA37F1FC408060E5D01BC402410038894132F1FC4080C876BE20C4024100240681A5F0FC400052B81EB9C4024100C64B37A3F0FC40808B6CE7BBC4024100BA490CA0F0FC40005C8FC2C1C40241008A41606BF0FC4000A4703D52C5024100A44E406AF0FC4080EE5A4258C50241000C24286BF0FC4080F085495EC50241000434116EF0FC400030BB2764C5024100F831E672F0FC40007C61B269C5024100DCB58479F0FC4080C954C16EC50241007E6ABC81F0FC4000A0893073C50241003CBD528BF0FC400040A4DF76C50241004A0C0296F0FC4080EA04B479C502410092ED7CA1F0FC40809999997BC502410016FBCBF0F0FC4000992A1885C5024100AED85FE6F0FC40807D1DB8B0C5024100CAE53F23F4FC40807D1DB828C6024100AED85F4EF9FC40802BF69799C6024100361AC074F9FC40807D1D3883CB0241005227A0B9FAFC40004703F89BCB024100CAE53F03FDFC4080621058C2CB024100361AC034FFFC40004703F8D7CB024100361AC03C01FD40807D1DB8D0CB0241005227A04104FD40802BF69785CB0241002041F13C04FD4080506B1A7BCB0241001283C08404FD4000FED47879CB024100CC10C76B04FD40803D0AD73FCB0241001283C05604FD40806210580FCB024100703D0A8104FD408091ED7CE4CA02410014AE47B104FD4000508D97B3CA024100C264AA6609FD40007B142EF2C9024100D2DEE06609FD4080787A25F2C90241003889410A0AFD4080621058E0C902410076BE9F040AFD40000DE0ADD7C902410016D9CEF509FD4000D122DBC0C9024100621058090AFD4000DF4F8DBDC90241009EEFA74E09FD4080C420B09AC8024100AC1C5A4E09FD40809F1A2F9AC8024100E8FBA94308FD4080180456D9C6024100B4EA73E307FD40808863DD31C6024100AED85F0E0CFD40004703F87BC5024100BAB88D000CFD40006B2BF64DC5024100C876BE1F0DFD4080C64B372EC5024100EEEBC0100DFD40004ED1911BC5024100AE47E1000DFD40803D0AD707C502410060E5D0000DFD4000CDCCCC04C5024100F2D24D1C0DFD408072689101C502410070CE88220DFD40806891EDA7C40241000000005011FD400010E9B7E8C302410008CE197610FD400093A9026FC20241002653057610FD4080F606DF6EC202410014AE472910FD408045B6F3DDC102410090C2F52810FD40803F355EDDC1024100545227040FFD400063EEDAE7BF0241000A4625040FFD4080855AD3E7BF024100000000D80EFD408062105826BF0241000EBE30060EFD4080B7408232BE024100A69BC40C0DFD40806ABC74CABC024100FCA9F10A0DFD4000273108C7BC0241005EBA49FC0CFD40004E6210A0BC0241001873D7270DFD40806B9AF791BC02410086A757580DFD408013D04482BC024100EC51B8BC0DFD4080A679C761BC02410038D6C5BC0DFD4000EFC9C361BC024100CAE53F230EFD40802BF69745BC024100585BB1260EFD40006C787A3FBC024100AE47E1380EFD4000508D9739BC024100AC1C5A4A0EFD40008716D91ABC024100829543850EFD4080EFA7C606BC024100EC51B8910FFD4000728A0E14BA0241000000004010FD40807D1DB804BA0241007C832F4910FD40002CD49AF8B90241009CC4207A10FD4080B81E8502BA02410020F46C9110FD4080D40968E0B902410090C2F5A010FD4080E7FBA9C9B90241009CC420B610FD408045B6F3CDB90241005839B4EC11FD4000FCA9F11AB802410078E926ED11FD408014AE471AB802410022DBF91C12FD4000DF4F8DD0B70241002AA913C912FD40004703F814B702410022FDF61913FD4080840DCFBCB6024100AED85FB613FD40802BF697DDB60241005227A0E114FD40004703F84BB7024100AED85F5E16FD4080621058A6B70241003AB4C80317FD4080DB6800C1B70241008C28ED0317FD400095D409C1B70241003208ACA017FD4000022B87E9B70241006ABC741319FD4080438B6C1CB802410096218ECC19FD4080C876BE2EB8024100C2A8A4CC19FD4000A5BDC12EB8024100361AC0041AFD400010E93737B8024100B84082DF1BFD4000C4B12E80B802410096D409E01BFD408013614380B8024100388941EC1CFD4000AE47E1ACB8024100D022DB591DFD408045B6F3BEB8024100CE1951471EFD400073D792E4B8024100FA7E6A471EFD4080E2C798E4B80241005227A0911FFD408062105832B9024100361AC06C20FD40807D1D3863B9024100AED85F0E21FD40807D1DB880B90241005227A0E921FD4080621058EAB90241005C20413C22FD400079E9A601BA0241000000004822FD4000560E2D08BA024100FCCBEE5022FD4080635D5C0CBA0241005452275B22FD4000A392BA0FBA02410040355E6622FD40809F1A2F12BA024100A089B08F22FD400057EC2F19BA024100EC2FBB8F22FD408057CA3219BA024100AED85F4623FD40807D1DB84CBA024100CAE53F8B24FD400010E93787BA024100226C782225FD4000B150EB88BA024100AE47E19C25FD40007B14AE9DBA0241006ADE71A725FD40004C37099FBA024100A8C64BB225FD40800512949FBA024100325530BD25FD40003A924B9FBA02410088D2DEC725FD4000575B319EBA024100749318D225FD4000A8C64B9CBA024100C8293A1A26FD408033A2B48BBA0241005227A03126FD40004703F88BBA0241005227A0F127FD40004703F843BA02410070F085B332FD4080E5D0A204B70241008AB0E1B332FD408094F68604B7024100ACFA5C2D33FD4000083D1BE4B6024100567DAE8934FD40002FDD2487B6024100E2C798E535FD40805E294B2AB6024100E8FBA9E535FD408039B4482AB6024100AED85F7E36FD40802BF69715B6024100CAE53FB337FD40807D1D384BB6024100AED85F8E38FD40802BF697B1B60241000000007039FD40004703F81FB7024100361AC0843AFD40004703F827B70241005227A0C93BFD40807D1DB80CB7024100000000083DFD40802BF6970DB70241002A3A92263EFD400083C0CA2BB70241006ABC74CF3EFD40007D3F3559B7024100B0E1E9FC3EFD4080984C9561B70241005227A0A13FFD40807D1DB894B7024100CAE53F5B40FD40807D1D3807B8024100CAE53FAB40FD40807D1D38EFB8024100361AC0A441FD40004703F8BFB90241005227A01943FD40807D1D385FBA024100000000E044FD40802BF697C1BA0241003CBD526D45FD4000D34D62DCBA024100605452BE45FD4080051214F4BA024100C876BED345FD4080621058FABA024100D49AE63146FD4080C117A601BB024100AC3E576E46FD4000083D1B0DBB02410078E9268746FD400006819514BB024100B459F59B46FD40005C20C115BB0241000000000047FD400010E9B728BB024100AA13D05A47FD40803480B718BB024100A8C64BCD47FD4080BF7D9D21BB024100EE7C3FF547FD4080EB51B824BB024100B6F3FDC648FD4000B4C87606BB0241001648506949FD4000F697DDE6BA024100986E12EB49FD408075719BCDBA024100643BDF2F4AFD4080C64B37C0BA024100EE7C3FCF4AFD4000D9CEF799BA02410082E2C7654BFD40004CA68A6ABA024100388941814BFD4000D34DE261BA024100F46C569E4BFD4080635DDC65BA024100B0506B9E4BFD4080643BDF65BA0241005839B4E94BFD40003D9BD572BA024100C876BEB54CFD4000D9CEF795BA024100D42B65F74CFD40801E166A9EBA024100361AC0EC4DFD400010E937D3BA0241000C71AC064FFD408058A835C9BA024100DE2406D34FFD400004560ED6BA02410038F8C2DC4FFD4000ABCF55D6BA024100DA1B7CE64FFD40008F53F4D5BA024100941804F04FFD40801E85EBD4BA0241004A9D80D750FD40808CB96BB3BA024100081B9ED750FD40808B6C67B3BA0241005227A0C152FD40802BF697ADBA0241005227A0A954FD40806210589ABA024100AED85FBE57FD40004703F8B7BA024100AED85F065AFD4080621058EEBA024100AED85FD65BFD40807D1D38FBBA024100000000185EFD40004703F8CBBA02410096218EC25FFD40002A3A929BBA0241006E12830860FD4000A8C64BB1BA02410096438B1060FD400021B072B3BA0241008C6CE7A360FD40800C022BD5BA024100643BDFDF60FD4000508D97DCBA02410052B81E3761FD4000AC1C5ACFBA024100C0CAA1DD61FD40806210588ABA0241003889410662FD40803D0AD77FBA0241001283C07C62FD4080C64B3761BA02410026E483A962FD4000925CFE4ABA0241000000008065FD40807D1DB810BA0241000000006069FD40802BF697BDB9024100AED85F8E6CFD40802BF6975DB9024100CAE53FC36FFD40802BF69711B9024100AED85F2672FD40802BF697C9B802410062A1D63472FD4080C64BB787B80241002E211F6672FD40003C4E517FB80241000000009272FD40803D0AD777B8024100AE47E15273FD4000FED47811B802410024287EA573FD4000A1673389B7024100CAE53F7B74FD4080621058E6B60241000000007075FD400010E9B774B6024100361AC04C76FD40004703F823B602410052499DF677FD4080559FABBAB50241003889410279FD40803D0AD7A4B50241001283C09C7AFD40808FC2F513B5024100501E16B67AFD40807FD93D09B5024100CAE53F337BFD40807D1DB8E8B4024100C64B37B07BFD4080696F70C7B4024100C0CAA1CD7BFD4000B4C876C6B4024100E2361AFD7BFD40801D38E7C4B4024100B6F3FD9C7DFD4080C64B37B7B4024100B81E85D17DFD4000DD2406B6B402410092ED7C4B81FD40803F355EC7B402410046B6F35181FD4080EB51B8C7B40241000C022B5781FD4000CFF753C8B40241008C6CE75385FD4080BC74935EB502410060E5D00287FD408093180495B5024100D24D620487FD400033333395B5024100D49AE6D287FD4000E63FA4ABB50241008EE4F2D287FD400079E9A6ABB50241005227A0C188FD40807D1DB8DCB5024100AED85F5E8AFD40807D1D3837B6024100CAE53FB38BFD40802BF697A5B6024100AED85FD68CFD40802BF69771B7024100361AC03C8EFD40807D1DB820B7024100CAE53F838FFD40004703F8BFB6024100361AC09C91FD40806210583AB60241005CB1BFBE92FD4000F8C2E405B6024100DCF97EAC93FD4000A245B6F5B502410034EF385894FD4080270F0BEEB50241001283C0AC95FD40803D0AD7DEB502410052B81E1B96FD4080C64B37D0B5024100FCA9F12296FD4080621058CDB5024100E8FBA92396FD4000FA7E6A9BB5024100D478E97C96FD4000508D9794B50241009CC4208896FD4000FED47873B502410020F46C1298FD408069DEF15EB5024100CAE53FDB9AFD40004703F88BB50241005227A0719CFD400010E937E7B5024100AED85FA69DFD400010E9B73CB6024100AED85FCE9FFD408062105846B6024100E4141D65A0FD400090A0F841B60241004A0C02DFA0FD4080F3FDD453B6024100AAF1D207A1FD4080BC74934BB60241006ABC7497A1FD4000CBA1455BB60241008C6CE7B1A3FD408016D9CEB6B60241003AB4C828A5FD408095438B16B7024100448B6CEBA5FD400077BE9F17B70241004A0C02F7A5FD4000508D9719B70241002EB29D3DA7FD4080EB51B819B70241002EB29D2DA9FD4080C64B3727B7024100EE7C3F21AAFD4000D9CEF731B70241002C87169DABFD4000A167B331B702410000000048ACFD40802BF69771B7024100AED85FEEADFD40807D1D3893B7024100AED85F0EB0FD40802BF697D5B70241005227A0F9B2FD40802BF69751B80241005227A009B5FD40004703F8B7B8024100CAE53FB3B7FD40807D1D380BB9024100AED85F8EBAFD40802BF69739B9024100605452A6BAFD4080C0CA213BB902410008AC1CA4BAFD4000508D9757B9024100B4C876C0BFFD4000DF4F8DA0B9024100024D84F1BFFD408033A2B4A5B9024100AED85FC6C0FD40807D1D38C7B90241007E8CB930C3FD4000250601D4B902410008AC1C3EC3FD4000D9CEF7D6B902410056EC2F48C3FD40006744E9D4B9024100B8408249C3FD4000014D84D4B9024100361AC014C4FD40807D1DB8D8B9024100A6BDC1B6C9FD4080FAEDEBBEB60241000A6822EAC9FD4000A16733A6B6024100AED85FEECEFD40802BF6975DB40241005227A059D3FD40807D1DB8A0B2024100361AC0ACDCFD40807D1DB890B202410000000090DFFD40802BF6978DB20241005227A021E0FD40004703F803B3024100000000E0E1FD40004703F8EFB202410000000000E4FD400010E9B7E0B2024100CAE53F1BE7FD4080621058CAB2024100361AC07CE9FD40004703F8DBB2024100AED85F46ECFD400010E9B7E8B20241005227A019EFFD40807D1DB888B2024100C68F31E9EFFD4000992A9847B20241001E85EB59F0FD40007F6ABC5CB202410040355E66F0FD4000FA7E6A5EB2024100BE9F1A6DF0FD40008716D95EB202410008AC1C2EF1FD408099999966B20241005C204138F1FD40802F4CA666B202410060764F42F1FD4000B684FC65B2024100E27A144CF1FD400077BE9F64B20241002E211F79F1FD400074B5955CB2024100AED85F46F2FD40802BF69769B2024100000000D0F4FD400010E93793B2024100AED85F5EF6FD40807D1D3847B2024100384547BCF7FD40804013E1E7B102410074249724FAFD40806B9AF7A0B1024100CAE53FD3FBFD40806210588EB102410000000000FDFD40806210586EB1024100361AC01CFEFD40802BF697FDB0024100CAE53F63FFFD40004703F86FB00241005227A00101FE40807D1D380FB0024100000000A002FE4080621058CEAF0241005227A02905FE40004703F8C3AF024100CAE53F7307FE40004703F887AF024100CAE53FCB09FE40004703F83BAF024100AED85F560CFE40807D1DB834AF024100CAE53FF30DFE40004703F8C7AF024100000000780FFE40802BF69719B00241007081049910FE4000C286A7C2AF024100D8A3709111FE4080ED7C3FD6AF02410032772DD912FE40801058B9F1AF024100AED85F8E13FE40004703F847B0024100CAE53FB314FE40004703F81FB0024100EE7C3FC214FE4000ECC0B91AB0024100508D97D214FE40807493181CB0024100302AA9DC14FE40807346941CB0024100EEC9C3E614FE4000AC1C5A1CB002410034A2B4F014FE40801E856B1BB00241005EBA49FA14FE4000CDCCCC19B0024100E8FBA91F16FE4080438B6CDCAF02410038F8C22A16FE400001DE82D9AF0241007A14AE3416FE40007B14AED5AF02410052B81E3D16FE4080BA490CD1AF02410096218E4F16FE40009031F7C4AF024100FCA9F1A916FE4080832FCCC8AF024100D881731217FE400004568ED5AF024100361AC06C19FE40004703F897B0024100361AC0B419FE40802BF697FDB00241000A6822DD19FE408003E70C10B1024100703D0AE119FE4080F3FD5412B1024100EC51B8E919FE408001BC0516B10241009E5E29EB19FE4000FDF67516B10241005227A0611BFE40802BF697C1B10241005E29CBAA1BFE4000A392BAEEB1024100D24D62B81BFE40003108ACFFB10241006CE7FBBD1BFE4000A4703DFEB1024100C420B0C21BFE4080B81E850CB2024100AE47E1D01BFE4080C64B371EB2024100DACEF72F1CFE4000A4703D66B2024100388941921CFE40809DEFA77DB20241000CB5A6921CFE40005C20C17DB2024100361AC0CC1CFE40802BF697A1B20241005227A0611EFE40802BF69751B3024100361AC05C1FFE40807D1DB820B40241009E80267F1FFE4000BBB80D34B4024100C876BEE11FFE4080EB51B8AAB40241007E6ABCEC1FFE4080C64B37B0B4024100508D975420FE4000AAF1D2D0B402410060E5D05A20FE4080BC7493D2B40241007CF2B0CA20FE40001DC965EEB4024100AED85FFE22FE40807D1D382BB6024100CAE53F0325FE40802BF69789B7024100FC18731926FE40808638D629B80241008816D91E26FE4000022B8739B80241007A14AE1526FE4000355EBA54B80241001283C02026FE4080EB51B85DB80241002631083626FE40808D976E8DB80241009A99998D26FE40002FDD24C4B8024100CAA1459C26FE40002B8716D2B80241001C5A648B26FE40004E6210F5B8024100C0CAA1B926FE40803D0AD711B9024100643BDFCD26FE4080EB51B841B9024100388941CE26FE4000508D975EB90241003208ACD026FE408016D9CE66B9024100FED478D726FE4080C0CAA16CB9024100B4C876DE26FE4080E3A59B71B9024100703D0AE526FE400027310875B90241005EBA492227FE400027310891B90241007C3F352A27FE4000560E2D94B9024100F44A592B27FE408029CB9094B90241000000002827FE40004703F8FFB9024100CAE53F7B28FE40004703F8AFBA024100460378B328FE40004FAF14E5BA0241008A4160A728FE40803D0AD7EFBA024100643BDFB928FE4000FED47803BB02410042CF66E028FE4080C420B00FBB0241000CB5A6E028FE40003867C40FBB024100AED85FF328FE4080FC877421BB024100DCF97EEE28FE4000508D9736BB0241005C2041E628FE4000D7A3703ABB024100068195B128FE4080B6F3FD52BB0241006CE7FBE128FE40801E85EB6BBB0241009CC4201429FE4080BC749391BB02410086EB514729FE40000A6822EFBB024100CAE53F4B29FE40802BF6970DBC024100AED85F062AFE40004703F883BC02410020F46C642AFE40807CD0B3BCBC02410008AC1C762AFE400008AC1CD2BC02410008AC1CA82AFE4080621058F3BC0241008C6CE7052BFE4080438B6C3ABD024100E2361A9D2BFE4000F016486EBD024100361AC0942CFE4080621058F6BD0241004CC807722DFE4080A982D143BE024100DE2406732DFE40806E128347BE024100501E16752DFE40006D567D4CBE024100BAFC87782DFE4080A60A4651BE0241005EBA497D2DFE4080CA32C455BE024100829543832DFE4000AE47E159BE024100E4A59BDC2DFE4000B29DEF8EBE024100AE47E10A2EFE4000A8C64BCCBE0241008C6CE70F2EFE4000508D97D1BE0241001E85EB152EFE4080931804D6BE024100F085C9902EFE400079582823BF024100CAE53FD32EFE40806210589ABF024100AED85F2E30FE40802BF69731C00241003CBD525B30FE40809FABAD3FC0024100180456A630FE40808B6CE76CC0024100DC6800AD30FE40808A8E6470C0024100DE718AB430FE400067D56773C002410060E5D0BC30FE4000F853E375C0024100D8A370F930FE408066666685C002410022DBF98E31FE4000D7A370B1C002410094A9829831FE4000A69BC4B3C00241002E90A0A231FE40001B9E5EB5C0024100BE9F1AAD31FE4000341136B6C00241007EAEB6B731FE408038D645B6C0024100C8293AC231FE408096218EB5C0024100FA7E6ACC31FE4080976E12B4C00241004082E2CD31FE4080A757CAB3C0024100361AC0F431FE40004703F8BFC0024100AED85FB632FE400010E9374BC102410026C2869033FE408036AB3E82C10241008E976E9A33FE4000508D9789C102410022DBF99E33FE4080E3A59B8CC102410078E926A333FE4000F853E38EC1024100F853E3F533FE4000AC1C5AB8C1024100CEF753F533FE40000AD7A3C9C10241002C431C1E34FE40809DEFA7D5C1024100CAE53F6B34FE40807D1D382FC202410074D7128E34FE40803E575B37C20241001283C0A034FE4000D9CEF73FC20241002E6EA3B534FE4000E3369A40C2024100F2D24D8335FE4000C364AA70C20241008A41608B35FE4080EB51B878C202410016D9CE9135FE40808D976E93C202410054E3A59B35FE4000EBE2B696C20241008E976EA635FE40002FDD2499C202410052B81EFD35FE400052B81EA8C20241009CA2230936FE40805296A1A9C202410092CB7F1536FE400099BB16AAC2024100D066D52136FE4080B5157BA9C2024100F0A7C62D36FE4000AAF1D2A7C20241003889416836FE4080E7FBA99CC2024100703D0A9D36FE4000F6285CA3C2024100D85F76A636FE4080105839A4C2024100263108B036FE40806ABC74A4C2024100C876BEFF36FE4080C64B3789C20241009418042237FE4000DBF97E8AC2024100F6285C3B37FE4080F3FDD47FC2024100923A013D37FE40008716D985C20241009EEFA74037FE40000AD7A38BC2024100643BDF7137FE4000FED4788DC20241000000007437FE4080621058A5C2024100C0CAA19337FE4000B4C876B2C2024100C876BEC737FE4080C64B379BC2024100000000FE37FE4000508D9791C20241009CC4202638FE4080749318A4C2024100B6F3FD4838FE4080749318C8C20241009418045E38FE40005EBA49F5C2024100BC74938238FE40000AD7A3FEC2024100705F078E38FE400090A0F800C302410008AC1C9A38FE40005EBA4902C30241008E976EC438FE400027310805C30241008A4160CB38FE4000DD240608C3024100022B87F438FE40806ABC742EC3024100FC87F4F938FE40801361C332C302410096438B0039FE40009D11A536C3024100C2F5280839FE4000DD24063AC30241001E85EB3D39FE400054E3A54EC3024100787AA54739FE4000F1F4CA51C30241003889415239FE4080081B1E54C3024100DA3D795D39FE408004C58F55C30241004A0C026939FE40002B871656C3024100C05B20AA39FE40801E856B56C3024100AED85F4E3AFE40807D1D3887C30241005405A39B3AFE40809BC420A7C3024100508D97A43AFE40808B6CE7B5C3024100EA95B2A83AFE400030BB27BBC30241001C5A64AE3AFE4000948705C0C302410070CE88B53AFE4000D34D62C4C302410046B6F3BD3AFE4080E5D022C8C3024100EA2631DA3AFE4000D122DBD2C3024100643BDFFB3AFE4000FED478D1C302410052B81E093BFE4080C64B37D5C3024100986E120F3BFE4000A8C64BDCC3024100D8A3701D3BFE4080EC9EBCDCC30241005227A0593BFE40802BF697F5C3024100CAE53FC33BFE40807D1D385BC4024100CAE53F333CFE40806210580EC5024100361AC0E43CFE40807D1DB8A0C5024100000000E83DFE40004703F853C602410040355E7B3FFE400098DD1315C7024100448B6C853FFE4000022B873BC70241009CC420983FFE40806CE7FB58C7024100B4C876983FFE4000B4C87659C7024100FCA9F1A23FFE4000022B8763C70241009CC420A43FFE4000CFF75364C7024100388941DE3FFE4000CFF75389C7024100365EBADF3FFE40005A643B8AC7024100AE47E1E83FFE408039B4C88EC7024100448B6C3340FE4000B07268ADC702410004560E3740FE408016D9CEAEC7024100D8A3700F41FE40007B14AEFCC70241002406811541FE40000AD7A3FEC7024100BE9F1A1D41FE408095438B00C8024100621058EB41FE408060E5D02CC80241008816D92442FE408099999948C8024100022B872C42FE400061C3D34BC80241002631083542FE4080DA8A7D4EC8024100EA26313E42FE40004C378950C8024100EE7C3FF242FE408040826271C80241000000003843FE40807D1D388BC8024100EE0DBE3A43FE40808B6CE78BC8024100C420B09843FE4080976E12BEC802410022DBF99E43FE4080703D0AC1C8024100726891A943FE4000355EBAC4C8024100E27A143245FE40808D976E33C90241001804564A47FE4000B0726814CA0241002406812747FE4080976E1239CA02410000DE026147FE4000D7A37050CA0241000000006647FE4000B4C87652CA0241003EE8D94647FE400023DBF96BCA024100DACEF76346FE408039B4C825CB024100302AA96946FE40007E8CB92ACB024100F4DBD77046FE40805452272FCB0241006232557946FE4000FDF6F532CB024100441CEB8246FE400096B20C36CB024100F6285C8D46FE4080F4DB5738CB024100F8C2649846FE408039B4C839CB024100C876BEA346FE40801804563ACB0241002EB29DCA46FE40800C71AC3ACB024100AED85F9E46FE40807D1DB878CB024100CAE53F9347FE40802BF69791CB0241005227A0B149FE40004703F8F3CB024100361AC0BC4AFE408062105862CC0241005227A0614CFE408062105826CD024100CAE53F534FFE40807D1DB86CCE024100E6AE252F50FE4000F775E0B5CE024100E6D0227350FE40007F6ABCD5CE0241004E62107850FE40803D0AD7D7CE0241004E62100852FE4000D9CEF773CF024100F4FDD40F52FE400051DA9B76CF0241005A86381852FE4000C7BAB878CF0241002C87162152FE4000CBA1457ACF0241002AA913DC54FE4000DAAC7ADBCF024100AED85F3E55FE40802BF697EDCF024100000000405AFE40807D1D38AFD0024100764F1E4A5AFE4080FF907EA8D0024100000000AA5AFE4000355EBAB6D0024100EA2631B65AFE4080900FFAB7D00241009C33A2C25AFE40800B2428B8D00241006A4DF3CE5AFE4080136143B7D0024100143FC6DA5AFE40003CBD52B5D0024100CA54C1E55AFE4000F8C264B2D0024100E02D90EF5AFE408072F98FAED00241008C6CE7F75AFE4000FCA9F1A9D0024100903177255BFE408019E2D88BD0024100AED85F0E5CFE40004703F8BFD0024100AED85F3E5EFE40807D1D382FD10241000000003861FE4080621058C6D1024100CAE53F9B64FE40807D1D387FD2024100AED85F9667FE40807D1D3833D30241005227A0F968FE4080621058B2D3024100AED85F4E6AFE40802BF69765D4024100AED85FB66AFE40807D1DB83CD5024100000000806BFE40004703F8D3D50241005227A0196CFE408062105852D6024100CAE53F1B6DFE40802BF6970DD7024100AED85F166EFE40004703F897D7024100AED85F2E6EFE408062105812D80241005227A0116FFE4080621058F6D7024100AED85F6E70FE4080621058DAD70241009A081B0571FE40804160E59FD70241003208AC1E71FE4000AE47E19CD70241001895D42771FE400090A0789BD70241009465883071FE40806B9A7799D70241002E90A03871FE4080AF03E796D7024100DACEF73F71FE4000AAF1D293D7024100B4C876AE72FE4080976E12E4D6024100EA2631B272FE40002FDD24E2D6024100CEF7530B73FE4080B81E85B0D60241009418041273FE4080C2F528ACD602410030DD242474FE4000B07268DAD50241001058392674FE4000A245B6D8D5024100CEF7532B74FE4000A8C64BD3D5024100BAB88D5E74FE40809ECDAA8ED5024100361AC03C75FE40806210585ED50241005227A05176FE400010E93767D5024100AED85FB677FE40807D1DB8B4D50241005227A02179FE40004703F83FD60241005227A0197BFE40807D1D38C3D6024100AED85F8E7CFE40004703F8C7D6024100AED85F067DFE40004703F887D6024100AED85F8E7CFE40807D1D3883D6024100361AC0CC7BFE40807D1DB824D6024100CAE53FEB7AFE40802BF6975DD50241005227A0817AFE40807D1D38EBD4024100CAE53F937AFE40004703F8FFD3024100AED85FCE7AFE40004703F813D302410040C6DCF07AFE4000ED0D3EE7D2024100DCF97EF47AFE4000B4C876E3D20241002406815B7BFE4080EB51B862D20241006232556A7BFE40005839B449D2024100000000687CFE40807D1DB800D1024100C2F528277DFE400011363C00D0024100C0CAA18B7DFE4080EB51B8A3CF0241009CC420B27DFE4080C64B3772CF024100DCF97EE67DFE4080621058F9CE024100AE47E1947EFE4000B4C876B8CD0241001E3867B57EFE40809A779C85CD024100CAE53F117FFE400043ADE91FCD024100AE47E16E7FFE4000D9CEF7D2CC0241002C8716BF7FFE400008AC1C95CC024100B81E85DD7FFE400085EB5177CC024100448B6CF37FFE40801283C06CCC024100365EBA1980FE4080C64B374FCC0241000000005880FE4000508D9734CC024100365EBA9180FE4000FA7E6A2ECC024100B07268BF80FE4080490C0220CC0241004E6210E081FE4000CDCCCC03CC024100388941E281FE408072689103CC024100B094654082FE40003D9BD5F8CB0241005227A0E982FE40807D1D38EFCB0241000000009084FE40807D1D38B3CB024100AED85F1686FE40802BF69741CB0241005227A0B186FE40807D1DB8D4CA0241005227A0D986FE40807D1D3867CA0241005227A08186FE40807D1DB800CA024100361AC0F486FE40806210584EC9024100000000B087FE40004703F877C8024100000000F888FE40004703F873C7024100482EFF3089FE40007A362BF0C60241009CC4206E89FE4080EB51B8E5C6024100166A4D9289FE4000D6566CE1C602410044FAEDF789FE4080627F59D5C60241003E7958C08AFE4000273188BDC602410008AC1CEE8AFE4080749318B8C6024100388941248BFE4000FED478ABC60241000000003A8BFE4080C64B378CC60241001C7C61648BFE4080832F4C38C6024100DACEF78C8BFE408027A08906C60241001C5A64A38BFE4080621058F3C5024100CAC342B98BFE408026C286EDC5024100240681F78BFE4000D9CEF7DCC5024100643BDFD98CFE4080621058ADC5024100A69BC48A8DFE408074931883C5024100365EBAC18DFE4000B4C87666C5024100EA2631F48DFE40804160E524C50241007A36ABF98DFE4080A679C71DC5024100DAACFA378EFE40805AD3BC1EC50241003AB4C8788EFE408045B6F32FC502410046B6F3E18EFE4000DD24063FC50241007E8CB9DC8FFE408003788B72C50241000000000891FE40802BF697D9C5024100CAE53FEB91FE40004703F86FC6024100000000D093FE40004703F827C7024100CAE53F9B94FE40802BF69775C7024100CAE53F1B95FE40807D1DB814C8024100A2D6346A95FE4000A089B07DC8024100A01A2F6595FE40808B6CE7A0C80241001C7C616595FE4080545227A7C80241005AD3BC6795FE4000F1F44AADC8024100C4B12E6C95FE4080779C22B3C802410050AF947295FE4080006F81B8C802410012A5BD7A95FE408011A53DBDC8024100FA7E6A8495FE4000333333C1C8024100FA7E6A3297FE408045B6F353C90241003889413C97FE408013D0C456C9024100AC8BDB4697FE40003945C758C902410090C2F55197FE40801F63EE59C9024100A8E8485D97FE408057CA325AC902410096218E6897FE400073D79259C902410092ED7C7397FE4080976E1258C90241008A4160A598FE40005C8FC220C90241003889411E9AFE4080BE9F1ACBC80241000000004E9BFE40803D0AD77DC8024100388941A69BFE408074931859C80241004A0C02D39EFE4080621058A2C7024100E8FBA9D19FFE4000560E2D6AC7024100CCCCCCEAA3FE4000295C8F70C6024100285C8FE2A9FE40803789411AC60241003889413AABFE4080EB51B803C6024100643BDFE3AEFE4000508D97C3C5024100F4FDD43EAFFE4000CDCC4C37C5024100CEF7536FAFFE4080C0CA2130C50241001283C00AB2FE4080749318D4C5024100C0CAA17BB3FE4080E926311EC6024100CAA145FAB3FE4000AAF1D208C60241002EB29D03B4FE400040A4DF06C602410086EB510CB4FE40005EBA4904C60241004A0C02ABB4FE4000333333CDC5024100E8482E24B5FE4080F3FD54B6C50241005227A0A1B5FE4080621058AEC5024100361AC034B6FE40807D1D38C3C502410000000008B7FE40807D1D38AFC5024100361AC04CB8FE40807D1D3883C50241008CDB6896B8FE4000A69B447FC5024100EA2631E8B9FE408016D9CE83C50241002E211F83BAFE40801E16EA88C5024100AED85F36BBFE40807D1DB8A0C5024100BE0E9C4FBBFE400030BBA78FC50241004A0C0275BBFE4000F853E390C5024100D044D880BBFE4080A757CA90C502410048BF7D8CBBFE40801058B98FC50241000AF9A097BBFE400010E9B78DC502410090C2F5A1BBFE4080F38ED38AC5024100EA0434ABBBFE40809BC42087C502410052DA1BB3BBFE4000C7BAB882C5024100B4C876B9BBFE4080C729BA7DC502410050FC18BEBBFE408014AE4778C50241008AB0E1C0BBFE408026C28672C5024100EE0DBEC1BBFE40009B559F6CC50241009EEFA7C0BBFE4000355EBA66C502410004C58FB6BBFE4080DE710A4AC50241005227A0B9BBFE40004703F847C5024100361AC04CBCFE4080621058CAC4024100AED85FDEBCFE40802BF69795C4024100D40968E8BDFE408015FB4B70C40241004260E56EBEFE400077BE9F68C40241006ABC74EFBEFE408091ED7C79C40241005A643B29C0FE4080EFA7C6C6C4024100AC1C5A80C0FE400061C353EEC4024100361AC0A4C0FE40807D1DB808C502410014D0440BC1FE4000F8C2642DC50241009A999917C1FE4080B6F3FD32C5024100D8A37021C1FE4080C876BE36C50241001AC05B44C1FE4000CE19D141C5024100000000F0C1FE40807D1D387FC5024100361AC00CC3FE40004703F86BC5024100DA8AFD39C3FE400052B81E4CC5024100643BDF51C3FE40806E128349C502410030BB275CC3FE40006C78FA47C5024100D200DE65C3FE40007C61B245C50241005E29CB6EC3FE4080EB51B842C50241007E6ABC76C3FE400008AC1C3FC5024100D24D6226C4FE4080B81E85E2C402410086EB5100C7FE4080666666B6C4024100285C8FB8C8FE400023DBF9D5C402410076BE9FC2C9FE4000FCA9F1E8C4024100A2B437A7CAFE408001BC850CC50241005227A0B9CAFE40802BF69711C5024100F4DBD7D3CAFE400047037813C50241006CE7FB61CBFE4000508D9729C502410044696F6DCBFE4080ADD8DF2AC502410078E92679CBFE40007D3F352BC5024100CC5D4B07CCFE408045B67329C502410000000020CCFE40807D1D382BC502410000917E3ECCFE4000EF38C528C50241008C6CE743CCFE40005839B428C5024100E8D9AC4CCCFE40803C2C5428C50241003A234A55CCFE40801E856B27C502410072AC8B5ACCFE4000DDB58426C5024100361AC07CCDFE400010E9370FC50241005227A0E9CEFE40004703F81FC5024100CAE53F8BCFFE40802BF6975DC502410038F8C253D0FE40805F074E9DC5024100D478E94CD0FE4080C95441AEC502410076BE9F4CD0FE4000D9CEF7AEC5024100C876BE3FD0FE4080999999CEC502410038894192D0FE40803D0AD7E8C5024100240681A7D0FE4000FED478EDC5024100627FD9AAD0FE4000A9A4CEEDC5024100AE47E1D0D0FE4000508D97F1C5024100388941D2D0FE4080621058F6C5024100F8E461E4D0FE4080E7FBA936C602410076BE9FEAD0FE40803D0AD74CC6024100643BDF5FD1FE4080749318EFC60241002406817FD1FE4000FED47801C702410000000002D2FE40806210584DC7024100EE7C3FEBD2FE40002B8716AAC7024100C876BE49D3FE40000AD7A3CFC70241004694F625D4FE400043ADE911C8024100CAE53F0BD5FE400010E9B780C80241005227A019D7FE40807D1DB82CC902410000000008D9FE40807D1DB808CA024100AED85F66DBFE4080621058FECA024100361AC014DDFE40802BF697A9CB0241005227A0E9DEFE408062105886CC024100361AC034E0FE40802BF6973DCD0241005227A0B1E1FE40802BF69709CE024100361AC0DCE2FE40807D1DB8A8CE0241005227A061E4FE40807D1D3827CF024100AED85FD6E5FE4080621058B6CF024100361AC0D4E7FE40806210586AD002410072689160E8FE400089D25E95D0024100E8D9AC63E8FE4000C139A397D00241004CC8076CE8FE40004625F59BD0024100A245B675E8FE4080B81E859FD0024100DCF97E40E9FE4000A69BC4DDD002410004560E01EAFE4000C8983B17D10241005227A0E1EBFE40802BF697B5D102410000000008EEFE400010E9B74CD2024100361AC03CEFFE40807D1D38BBD2024100CAE53FFBF0FE40806210582ED3024100D0D55624F1FE4000DFE08B38D3024100AE47E1BAF1FE4000FED47865D3024100B6F3FDB8F7FE40000681953DD502410076BE9FB0F8FE4000FED4788BD50241002EB29D09FBFE40806210583ED6024100643BDF4FFCFE4000FED478A7D6024100C876BE0FFDFE4080621058EAD602410040355E6CFEFE408062105860D70241003E9B5534FFFE4000B9FC07A4D702410076BE9FA6FFFE4080EB51B8CAD702410076BE9F8A00FF4080C64B3719D802410012F241BE00FF4080F831662FD80241004A0C02DF00FF4000FED4783DD8024100C876BE5D01FF4080C64B3776D8024100EE7C3F6702FF4000D9CEF7ECD8024100240681DF02FF4000B4C87622D90241000000000C03FF408062105836D9024100EA73B54903FF40003B70CE51D902410040355E5A03FF4080C64B3759D9024100CAA1457203FF40008716D963D9024100B81E854F03FF40005839B4AAD90241005C8FC2DB03FF40806E1283D6D9024100ECE236D903FF40006E3400F2D9024100AE47E1D603FF4080C64B370BDA024100240681F703FF4000D9CEF71DDA024100B8AF035C05FF40800A462590DA024100000000E806FF40802BF69729DB0241000000005009FF40806210581ADC024100000000000CFF40807D1D38EFDC0241005227A0E10EFF40004703F8D3DD024100CAE53FBB11FF40802BF697C1DE024100E04F8D2014FF4080E2C79850DF024100560E2D9214FF400052B81E7BDF024100105839C414FF408091ED7C87DF024100240681CB14FF4080BA490C89DF024100CCCCCCD214FF4080C2F5288ADF024100285C8F2815FF408039B4C894DF0241000CB5A64E15FF400025068199DF024100A01A2F6F15FF40004C37899DDF024100D478E97A15FF408047E17A9EDF02410062C3D3CF15FF4080F085C9A1DF0241005227A03116FF40807D1D38AFDF024100000000C017FF40004703F8B7DF0241005227A0A91AFF4080621058DEDF0241000E2DB2B31BFF4000CAC342E0DF02410084C0CA171DFF408095438BF1DF0241007E6ABC1A1DFF40809DEFA7F1DF024100061214D91EFF40809BC420FEDF0241000000009820FF408062105826E0024100AED85F2622FF40807D1D3827E0024100361AC0CC23FF40004703F82FE00241005227A01125FF40807D1DB834E0024100361AC04426FF40802BF69735E002410070CE88A326FF40001AC0DB35E002410048E17AFA26FF40005EBA4938E0024100BC74930327FF4000EEEB4038E0024100508D970C27FF400054E3A537E0024100E8482E1B27FF4080E9263136E00241005227A05127FF408062105836E0024100AED85F8628FF408062105822E0024100AED85FAE2AFF40802BF697C5DF0241005227A0992BFF4080621058B6DF024100000000382DFF40806210589ADF024100AED85FC62EFF408062105862DF024100361AC0F42FFF40802BF697FDDE0241005227A01931FF40004703F883DE0241000000005832FF40004703F873DD024100CAE53FF332FF4080621058E2DC024100000000E833FF40802BF697ADDB024100AED85FCE34FF400010E9B770DA024100A69BC40835FF400040A45FFAD902410040355E1035FF4000FED478F3D902410052B81E4D35FF4080749318A0D902410076BE9F7C35FF40803D0AD753D90241002406817735FF4080EB51B852D902410040355E9635FF4080EB51B813D9024100E27A14BC36FF40009B551F88D60241005227A07937FF40004703F807D5024100CAE53F2B38FF40004703F8E7D402410066D5E73638FF4000674469D7D40241004E62104038FF40809BC420D6D4024100EE5A424A38FF4080A60A46D4D402410082E2C75338FF40807AA5ACD1D40241001E166A5C38FF400041F163CED40241004694F66338FF4000925C7ECAD40241005C20416A38FF40804E4013C6D40241000A46256F38FF408035CD3BC1D4024100022B877238FF4080E17A14BCD40241000022FDD038FF4080B81E85FCD3024100903177993DFF40806FF00531CD024100AE47E1E03DFF4000D9CEF7DCCC024100482EFF033EFF400068226CA9CC024100941804043EFF4080666666A9CC02410012F241D33EFF400033333379CB024100560E2D273FFF4000BD529611CB0241008A1F634D3FFF4000394547ECCA024100D8A3703F40FF40808FC2F5FFC90241005C2041CB40FF4080506B9A31C9024100CAE53F9B41FF40802BF697A5C8024100AED85F1646FF40004703F8CBC60241005227A0C96CFF40807D1DB8F0ED0241005227A0116DFF40802BF69721EE024100CAE53FD36DFF40807D1D3843EF0241005227A0296EFF40004703F877F00241005227A0D16EFF40807D1D3803F2024100361AC0EC6FFF40807D1D38E7F2024100CAE53FB375FF40807D1DB87CF70241002A3A923B77FF4000EBE2B6A2F8024100C0EC9E3B77FF40801283C0A2F80241005839B48678FF4080E7FBA9EEF90241005EBA49E779FF40801B0D6042FB0241004A5986E779FF4080764F9E42FB024100361AC04C81FF40802BF697D902034100CAE53F9B82FF40807D1DB888010341000000007083FF40807D1D380B01034100B4EA73EE83FF40006D56FDCD0003410004560ED184FF40803D0AD788000341004260E5D884FF40007F6ABC860003410040355E0A85FF40803D0AD76800034100C0CAA1B985FF40806210583200034100AE47E1D485FF40803D0AD72900034100E8D9ACD985FF40002FDD242700034100361AC07C86FF40807D1DB8FCFF0241000A8A1FD087FF4000D57869B1FF024100A245B66988FF4080F3FDD495FF024100EA2631D488FF4000AC1C5A7DFF0241008E976EEE88FF40000000007EFF0241003A014D1B89FF4080FB3AF075FF0241002C87168089FF400005341174FF0241009CC4209A89FF408074931874FF0241009CC420188AFF40803D0AD777FF02410098DD939A8AFF400060764F8FFF024100AE47E1AE8AFF4000D9CEF792FF0241007E6ABCB28AFF400006819593FF02410076BE9F428BFF4080EB51B8AAFF02410012F241878BFF4000AE47E1ADFF024100CAE53F3B8CFF4080621058C2FF024100E00B93A88CFF40000BB526BBFF024100240681AB8DFF4000D9CEF7C6FF02410040A4DFC18FFF408017B751DFFF02410052B81EC58FFF4000FED478DFFF024100D40968F38FFF4080A982D11E00034100FA7E6A2690FF40000AD7A364000341008816D93090FF40804160E5660003410052B81E0F91FF4000FED47871000341001283C0EC91FF4080EB51B86A00034100C0CAA13F93FF4000D9CEF76A00034100388941DC93FF40803D0AD77000034100C0CAA18594FF40806210585700034100DCF97E9A94FF40804BC80757000341001283C01295FF40801058395500034100643BDF5596FF4080C64B375B00034100621058E996FF40806891ED5D00034100A245B69997FF4080E926316700034100B6F3FDBC97FF40806210587000034100AE47E12498FF40806210587300034100C0CAA15998FF408062105880000341000C022B8D98FF4000508D9776000341002C8716C798FF4000CDCCCC7A0003410076BE9FFC98FF4080621058860003410096438B4299FF40005C8FC283000341001C5A648799FF40005C8FC288000341007C3F358C99FF4000CDCCCC88000341001283C09A99FF4080C64B378C00034100829543A599FF4000DD24068900034100C0CAA1059AFF4000AE47E18900034100DCF97E0A9AFF4000508D978A000341006CE7FB0B9AFF4000B29DEF89000341001C5A64B49AFF408044696F8B00034100361AC0449AFF40004703F8630103410016D9CEC79AFF408007CE997A01034100A8C64BCB9AFF40801058398E0103410092ED7CCB9AFF4000AC1C5A9001034100829543CB9AFF4080C420B09201034100C64B37BD9AFF40801C5A64DA01034100C2F5280A9BFF40808B6CE7DD01034100A245B6BD9BFF40801283C0D701034100C876BEBF9BFF4000D9CEF7D701034100CEF753C39BFF4000295C8FD701034100749318C89BFF40801C5A64D701034100BE3099799CFF400093A982C501034100361AC0AC9CFF4080621058CE010341006CE7FB30A1FF4080CC5D4B6402034100E8FBA931A1FF40800E2DB26402034100986E12A7A1FF400023DBF974020341005C8FC2A9A1FF40801C5A6475020341006210580BA4FF4000295C8FDD02034100C876BEB3A5FF40002FDD240B03034100BA6B09D3A5FF4000C9E5BF0D03034100AED85FFEA5FF40004703F81303034100000000D8A8FF40806210585E03034100000000F0AAFF40807D1DB89C030341005227A091ADFF4080621058C2030341005227A071AFFF400010E937AF030341005227A071B1FF4080621058660303410000000018B3FF40806210581E03034100AED85F06B5FF400010E9371303034100AED85FF6B6FF40802BF6977903034100361AC014B9FF40802BF6974504034100361AC084B7FF40806210588E050341005227A0D1BDFF40806210581208034100AED85F9EC0FF40807D1DB8D008034100164850E8C0FF40805CFE43A7080341006210586FC2FF4000CFF7532C080341000E2DB21BC3FF40806CE7FBDC07034100F6285CCBC2FF40806CE7FBAD0703410008AC1CF0C2FF4000DE02899E070341009CC42002C3FF4000D9CEF79607034100B07268E9C2FF408050FC988707034100361AC0B4C3FF40802BF69715070341005227A001BEFF4080621058020403410018B7D1ADBDFF400079E9A6E403034100CEF75357BDFF4080D3BCE3630303410040355E56BDFF4000B4C8766203034100285C8F48BDFF4080E5D02240030341008A416011BDFF40800C022B4103034100F0A7C63BBDFF408018045620030341009CC42028BDFF4000FED478EF02034100705F0724BDFF4080ADD85FF002034100105839D4BCFF40806891ED0103034100448B6C81BDFF4080F3FDD47B02034100C2F52882BDFF4000CBA1457B02034100C0CAA191C1FF40806CE7FB85FF0241001A2FDDCAC8FF4000FED47803FA0241008A41604BCCFF4080EB51B872F702410052B81E8DCCFF4000FED4783DF70241001A2FDD50D3FF4000D9CEF73DF202410076BE9F5CD3FF4000FCA9F141F2024100A067B3A9D7FF408013D044B6F302410038894154DEFF4080621058F7F50241007A14AE69DEFF4000D9CEF7EFF50241001804566EDEFF4000068195F1F5024100726891C9DEFF4000D9CEF7CEF5024100CE195105DFFF40801D3867BAF50241005227A0A1E0FF40807D1D3843F6024100CAE53FD3EBFF4080621058AAF1024100B6847CA9EBFF400042CFE675F1024100C0CAA137ECFF40007B14AE3FF1024100E4F21F0CECFF4080B662FF22F1024100643BDFEFEAFF4000508D9767F0024100441CEB09EBFF4080490C0260F0024100CAA14548EBFF4000D122DB4DF0024100B81E8531EBFF40806891ED3CF0024100BC749396EBFF40000AD7A321F002410040355E98EBFF40002FDD2421F0024100E8FBA9A1EBFF408045B6F31DF0024100ECE236DEEBFF408045B67305F002410012F241DEEBFF4080696F7005F0024100000000A8F0FF40807D1DB898EE024100000000D0F3FF40807D1DB8CCEF0241005227A021F8FF400010E937C3EC024100AC1C5A34FEFF4080C7293A13EE0241001283C026FAFF40801283C049F0024100E8FBA985F3FF40809BC420F5F3024100A0ABADD7F2FF4080BF7D9D31F40241003208ACD2F1FF40008941608CF40241008816D920F1FF408016D9CEEEF4024100C898BB68F1FF408059F5B9FCF402410050FC1860F2FF4000780BA42CF5024100000000C8F1FF40004703F86FF5024100361AC034F6FF40806210581EF6024100AED85F26FAFF40802BF69721F7024100789CA29EFDFF408023B9FCF1F7024100FCA9F1A2FDFF400083C0CAF3F70241005405A373FFFF4080C2F5A885F8024100EEC9C373FFFF40005839B485F80241001B0D6052000041807D1D38EFF8024100C364AA0D01004180F163CC5BF902410004560E3A01004100A245B679F902410045D870FA01004100FF217DF2F9024100903177FA01004180490C82F2F90241000000003003004180621058DEFA024100371AC04C040041807D1DB8A8FB0241002C6519730400418048BFFDDAFB024100CFF753EE05004180ED7C3F7AFF0241001283C0210600418062105806000341001F85EB9206004180ABADD820010341008941609906004100D9CEF73001034100E3A59B3207004100333333CA01034100E6AE25370700410074B515C8010341003E79583B0700410063EE5AC50103410052B81E3F070041004E6210C201034100713D0A55070041806E1283D6010341000000005C070041803D0AD7CF010341009CC4202A08004180EB51B89302034100894160B7080041803D0AD73803034100DBF97E0609004180C64B37950303410077BE9F470900410023DBF9E903034100894160D8090041803D0AD77B040341009CC420DC09004180EB51B87F04034100EFC943550A004100B9FC07FD040341009CC420E80A004100D9CEF794050341001283C00A0B004180749318C305034100EE7C3F110C0041807493182007034100894160F90C004180EB51B83D08034100F0A7C6540D0041803BDF4FAE0803410046B6F3DE0D004100D122DB7709034100E92631B20E004180105839830A0341007B14AEC50E0041000AD7A3960A034100EE7C3FDA0E004180749318B20A034100B07268360F004180158CCA280B034100DBF97E8A0F004180749318950B0341002FDD245F10004100FCA9F1A60C034100AC1C5A7D10004180C420B0CB0C0341001F85EB8110004180EB51B8D30C034100894160A410004180749318000D03410091ED7C1211004180C2F528D20D034100BC74934C11004100D34D62380E034100FCA9F18611004180B6F3FD9E0E034100A01A2FA8110041000AD7A3D90E034100C74B37CB1100418095438B320F034100EE7C3F1712004100508D97C30F03410062A1D63512004100CBA145FF0F034100AC1C5A52120041808B6CE73610034100AC1C5A4612004180976E1270100341000B4625281200418013D044A61003410076E01C281200410060764FA61003410000000010120041802BF697C510034100E5F29FA9110041807D1D383311034100933A81E8110041804C158C5111034100713D0AEB110041806ABC745D11034100789CA2EC1100410087A7D76211034100287E0CEF110041008C4AEA671103410035EF38F211004180711B8D6C11034100061214F6110041005305A370110341006FF085FA1100418005A312741103410021B072FF11004180EFA7C6761103410021B0727A1200418014AE47AE1103410096438B7D120041806E1283AF11034100C3D3ABB5120041003B70CEC21103410058A8B5B5120041001895D4C211034100371A40C7120041802BF697CD110341001B0D608E130041806210582A12034100C9E53F93140041806210584212034100B515FBC31400410020D2EF43120341002CD41AC414004180D85FF643120341006ABC745E15004180E3A59B63120341006666665216004100DBF97EB012034100CCEE49E316004100000000F312034100CEAA4FE316004100014D04F312034100C9E5BF7C170041806210586613034100B1506B7D170041001FF4EC5213034100000000B217004180EB51B86D13034100000000BD17004100B4C8765613034100E8D92CBD17004100B29DEF55130341003411B6F217004100B4C876B312034100894160F617004180621058A81203410089416073170041803D0AD7F5100341008941606717004180EB51B868100341009CC4204A170041803D0AD70F0F034100643BDF5217004100D9CEF7D50E0341007F6A3C571700418013D044C90E034100C9E53F5717004100371A40C90E034100C9E5BF8817004180621058860E034100107A36C9170041806132D5420E0341001283C05F180041803D0AD7EA0D034100F5B9DA7C1800418015FB4B660D034100E5F29FA9180041807D1DB8240D034100371A40171900410010E9B7280C0341001B0D60DA180041807D1D38DF0B0341006E3480D4180041809A771CD70B03410052B81EE7180041005EBA49820B034100643BDFE61800410079E926820B0341004F409366180041801C5A643F0B034100371A40531800410010E9B7240B034100E5F29FE5170041807D1DB8280A03410090A0788F17004100AAF1523A090341009CC4209317004100508D9730090341003333339C170041806E128318090341005EBA499D1700418047E17A01090341002FDD24B2170041005A643BDE08034100643BDFD617004180EB51B87C08034100499D00FB17004100D7A3701C08034100F0A746FE17004180EB51B813080341008941601A180041803D0AD7C80703410093180442180041808D976E5F070341006519E2671800418009F9A084060341001D5AE467180041002AA993840603410017B751BD180041800C022BC305034100448B6CD61800418016D9CED405034100DBF97EDA18004100D9CEF7C905034100FB5C6D2619004100F31FD2010603410040A45F56190041002B87162506034100371A40FB1A0041807D1DB87007034100F853632B1B004180BDC1976507034100A245B6371B004180E17A146B07034100643BDF4F1B004180749318660703410041F163CB1B00410076E09C40070341000C022B301E00418023B9FCB2060341009CC420591E0041803D0AD7AC060341006891ED961E004100894160C3060341004C3789A61E0041801C5A64C4060341001283C0AC1E00410013F241C4060341001A51DAB21E00410004560EC3060341009EEFA7B81E004100AAF1D2C00603410079E9263C1F0041800E2DB28006034100C520B03F1F004100A245B67E060341003867C4471F004180A6794774060341002A3A92491F004180FC18F37106034100000000781F00410010E9376706034100C9E53FC7220041004703F8CF05034100000000D42300410010E9379B05034100E5F29F3D240041807D1D38AB05034100371A40832400410010E937D305034100000000C42400410010E9B71406034100E5F29F1D250041802BF6975D060341005AF5391E25004180984C155E06034100CAC3421E250041802C431C5E060341009318044925004100560E2D94060341003108AC49250041806CE7FB9406034100CBA1454E25004100DF4F8D99060341006210588A26004100DBF97E9C07034100000000BC26004180F3FDD4B307034100EE7C3F7B27004100B4C876E207034100643BDFDC27004180EB51B81208034100EE7C3F46280041807493182E08034100840DCF9C28004100C898BB2F08034100371A40C3280041004703F83B0803410055302A6229004100F016486B080341008D976EA82900410052B81E9208034100EE7C3F082A0041002B8716C70803410077BE9F652A004180EB51B8EA08034100894160A72A004180621058F808034100EE7C3F0A2B0041807493181609034100643BDF692B004180C64B37270903410004560E302C004180C876BE5509034100006F814B2C00410032E6AE49090341008A8E64A82C004180F5B9DA7E09034100E5F29FB92C0041807D1DB88809034100C9E53FF32D0041802BF697D109034100EE7C3F592F004180787AA5510A034100894160422F004180EB51B85C0A03410008AC1C3F2F004100CCEE495E0A03410052B81E3B2F004180C64B37600A034100B5157B8A2F004180DD93077B0A034100643BDFBB2F004180EB51B88B0A0341007F6ABCC22F0041006C787A900A0341009CC420DE2F004100FED478A30A034100295C8FEF2F004100894160D50A034100DD24067030004180F3FDD4270B0341001F85EB8B3000418047E17A3A0B034100EE7C3F8D30004100D9CEF73A0B034100022B879030004100295C8F3D0B0341000000001331004180EFA7C6940B034100DBF97E8E31004180EB51B8D00B034100B684FCE0310041006EA301070C034100C9E5BF60320041004703F8970C034100C9E5BF3C330041806210585A0D0341000DE02D8A33004100A4DF3E740D0341007B14AECF3300410004560EBB0D0341005A643BD233004100FA7E6ABD0D034100E92631D633004180F1D24DC00D0341000E2DB23234004100DF4F8DFA0D0341001283C09D34004100D9CEF7280E034100DBF97EDC35004180EB51B8980E034100DCD781C136004180CC5D4BCD0E034100E5F29FC5370041802BF6974D0F0341001B0D60BA3800410010E9B72C100341001B0D607E390041807D1D38AF100341001B0D60563A0041807D1DB82C11034100E5F29F153B0041802BF6977111034100E5F29FDD3B0041004703F8F3110341001B0D60FA3C0041807D1D38D711034100EE5AC2DB3D00418038D645BB11034100B6F3FD2B3F004100D9CEF7141203410077BE9FA53F004100508D975F120341002D439C6440004100B459F5A012034100000000B8400041807D1D38E712034100371A4047410041004703F823130341008A1F63D8410041001136BC3A13034100B3EA734242004100BDE3946C130341001283C04242004180EB51B86C13034100BC9690AB42004100D34DE28F13034100371A40FF420041802BF697B913034100560E2DBD43004180900F7AD513034100894160C543004180621058D71303410089416051440041803D0AD7051403410009F9205F4400410041F16309140341008FE4726544004180B88D060B14034100FED4786544004100022B070B14034100894160C9440041803D0AD72414034100000000D145004100508D9720140341001283C06946004100B4C87614140341001283C031470041803D0AD7E7130341002DB29D9F47004100508D97C3130341008941605148004100D9CEF77A13034100643BDFE5480041807493184513034100894160F848004100FED4783813034100508D970C490041801283C02513034100C9763E10490041003CBDD22113034100CBA14513490041804160651D13034100E3A59B154900418072689118130341004260E52449004100AE47E1F11203410014AE47A949004180703D0A1E12034100B22EEEAB49004180FA5CED1812034100355EBAAD490041806ABC7413120341005A643BAE49004180BA490C111203410017D9CEC8490041000681957111034100C976BECB490041002FDD246F11034100BE9F1AD749004180BE9F1A71110341009CC420DF49004100508D975B110341008D976E1E4A0041808D976E7D1103410052B81E2F4A0041803BDF4F8011034100AA60D4344A0041003D9BD5801103410004E78C3A4A00418069DE718011034100E5D022404A00410079E9267F110341006ABC746B4A004100B072687111034100728A0E6F4A00410061C3D37311034100E5F29F714A0041004703F89F11034100C9E53F574B0041807D1DB8781803410038F8422D4C004100E86AABC8180341002B18152E4C004100849ECDCB180341004469EF424C004180B5157B1B190341001283C0434C0041809999991E190341004547F2474C0041007E8CB91F190341004B5986524C00410097FF902219034100295C8F524C00410073D7922219034100EE7C3F334D0041803D0AD75E19034100560E2D814D0041803F355E831903410083C0CA8C4D004100D9CEF785190341002FDD24904D004100D34D628A19034100E5D022954D004180EB51B88C1903410077BE9F114E004100D9CEF7D4190341007FD93D354E004180BF7D1DFC19034100371AC0304E00410010E9371B1A034100E5F29F514E004180621058DE1A0341001B0D60664E0041802BF697B51B034100371A40A34E00410010E9B7781C0341001B0D603A4F0041004703F8471D0341001B0D6006500041004703F82F1E034100D881F31F50004100B5A6F94E1E03416F00000000CAE53F4AB8FD4080621058FA5C0341000000000BBAFD4080621058C35C034100361AC00ABDFD40807D1D38055D034100CAE53F0BC1FD40807D1DB8075D0341000000006CC5FD4000470378FA5C034100361AC06CC8FD4080621058CC5C0341000000001ECBFD40004703F8755C034100CAE53F2FCFFD4000470378285C034100CAE53F00D2FD40802BF617EA5B034100AED85F41D4FD40004703788B5B0341006E3480B2D5FD4080621058345B03410000000044D7FD40807D1D38A55A034100CAE53FE5D7FD40802BF6972D5A034100E4F21FF7D8FD40807D1D387E590341005227A069D9FD40806210587E58034100AED85F3CD9FD40802BF6175E57034100361AC0CED8FD40807D1DB86D56034100CAE53FB1D7FD40806210D854550341006E348054D6FD40806210D8F353034100CAE53F67D5FD40802BF617D352034100361AC099D4FD4000470378C2510341000000005BD4FD40807D1D383A510341000000000ED4FD40806210D80950034100AED85FC2D3FD40004703F8F74E0341006E3480DBD2FD40807D1D38CE4C034100AED85F18D2FD40004703F8474B0341001C0DE020D4FD40807D1DB834490341006E348042DCFD40004703F841470341005227A07EE0FD4080621058D145034100AED85F30E2FD40807D1D381D460341006E348098E1FD40802BF697D546034100AED85F39DFFD40004703F85048034100361AC087DDFD40004703F8384A0341001C0DE04ADEFD40807D1D387E4B034100361AC0BFE0FD40802BF6170B4C034100AED85F04E7FD40807D1D384C4C034100AED85F19ECFD4000470378454D0341000000002EEEFD40004703F8F74E034100361AC0CDEEFD40807D1DB882510341000000008CEFFD40807D1D384352034100361AC0B9F0FD40802BF6173453034100361AC0C8F1FD40806210D8AC530341006E3480F6F3FD4080621058B6540341001C0DE064F6FD40004703F87755034100AED85F94F8FD4080621058C9550341001C0DE053FAFD40004703781256034100361AC0E2FCFD40802BF617A4560341001C0DE071FFFD40807D1DB82557034100CAE53FE203FE40004703783857034100AED85F0309FE40807D1DB80B57034100CAE53FB40CFE40004703F8E5560341006E34803510FE40802BF6179056034100AED85F7713FE40004703F8F1550341006E34808815FE40807D1D38A355034100E4F21F8916FE40806210D86B55034100CAE53FFB17FE40802BF6979C54034100CAE53F9E19FE40802BF69785530341006E3480011BFE40807D1D384652034100CAE53F551CFE40806210D8CE50034100CAE53FB81CFE40004703F89E4F034100CAE53FD81CFE40004703F8F74E0341005227A0611CFE40004703F8844B0341006E34802D1DFE40802BF6974848034100AED85FDB1DFE40004703781F470341001C0DE0141EFE40802BF697BA410341005227A0E316FE40004703786F3B0341006E3480DA09FE40802BF617A63A034100E4F21F05FBFD40802BF617BD38034100CAE53F13F6FD40802BF6970A360341005227A066F5FD40802BF617832E034100E4F21F19F8FD40807D1D381E290341006E348021F1FD40806210582F25034100CAE53F7DE9FD40802BF61760220341001C0DE051E4FD40802BF6972622034100E4F21FE7DCFD40802BF61763230341001C0DE08DDBFD40802BF69788260341001C0DE042D5FD40004703F8C7280341001C0DE0CFD4FD40806210D8B62C034100E4F21F3DD3FD40802BF697BF2F034100E4F21F2EC4FD40807D1D38F62E034100AED85F9BC2FD4000470378C531034100CAE53F3CBFFD4080621058B43503410092CB7FA9BDFD40807D1D38A339034100AED85F22C0FD4080621058753D034100CAE53F73C8FD4080621058EB3E0341001C0DE0E3D1FD40802BF697B7400341005227A09ECDFD4080621058C34403410092CB7F6DC6FD40802BF6173F48034100E4F21FF7BAFD40004703F8444A0341000000003CB1FD40807D1D38044F0341005227A059B1FD408062105804500341006E348037B1FD4080621058DC5003410000000064B0FD40802BF6173452034100E4F21FE1AFFD40806210D85B530341006E34808FAEFD40802BF617EB53034100CAE53FADADFD40807D1DB8CA540341001C0DE03BACFD40806210D849550341005227A0FAAAFD40802BF617B955034100361AC0F8A9FD40802BF6976856034100AED85F26A9FD40807D1D385857034100E4F21FC3A8FD40802BF617A05803410000000010A9FD4000470378E0590341005227A0FDA9FD400010E937E95A034100CAE53F0AACFD40802BF6975A5C0341006E348098AEFD4080621058345D034100361AC097AFFD40004703F88C5D034100AED85F07B1FD40004703F8C55D0341005227A0D7B2FD40802BF617BF5D03410000000028B5FD4000470378B85D034100E4F21FE9B6FD4000470378615D034100CAE53F4AB8FD4080621058FA5C03414D000000001C0DE0190DFF4000470378D6610341006E34808D0EFF40807D1D3877600341005227A0DF0EFF40806210589F5F034100E4F21FB10EFF40802BF617F75E0341001C0DE0720EFF40806210D8465E034100361AC0140EFF40802BF6977E5D034100000000C70DFF400010E9378E5C0341005227A0F80DFF4080621058EE5B0341006E34807A0EFF40004703782E5B034100AED85F4E10FF4000470378BF59034100361AC00012FF4080621058D8580341001C0DE0C212FF40807D1DB808580341001C0DE0A512FF40802BF697D8560341001C0DE08712FF40806210580856034100CAE53F8A12FF40807D1D3810550341001C0DE0AC12FF40807D1D3800540341006E34801E13FF40806210586853034100361AC04F15FF40807D1DB8F9520341001C0DE09116FF40806210583A520341001C0DE05217FF40806210D8D2510341000000004418FF40806210587351034100E4F21F9619FF40802BF617AC5003410092CB7F631BFF40802BF697874F034100AED85FA51BFF40802BF617554B034100AED85F211CFF40806210D88148034100CAE53F3B19FF40807D1DB82D48034100361AC0E90FFF40806210582D49034100AED85F7B07FF4080621058E648034100AED85F29FFFE4000470378C9490341006E34809DF9FE40806210D8014B03410000000083F4FE4000470378014C034100CAE53F84F1FE40806210D8C74D0341001C0DE020F1FE40004703F8F74E0341006E34807FEFFE40004703F829510341005227A06CEEFE400047037841520341006E34805AEDFE40004703F810530341001C0DE047ECFE40806210580854034100361AC0C5EBFE40807D1D38E054034100361AC0B3EBFE400010E937B055034100AED85FD1EBFE4000470378A856034100000000E0EBFE400047037830570341006E34802EECFE40806210D8D857034100E4F21F3DECFE40806210D860580341006E34802BECFE40004703F808590341001C0DE078EBFE40802BF697085A034100361AC0A6EAFE40807D1D38E05A0341006E3480A4E9FE40807D1DB8B75B03410000000073E8FE40802BF617475C0341005227A050E7FE4000470378265D034100AED85FAEE6FE40807D1D380E5E034100CAE53FECE5FE40806210D8DD5E0341005227A009E6FE40004703F8E55F034100AED85FA7E6FE4000470378DE60034100E4F21F06E7FE40806210D85E610341001C0DE085E8FE40807D1DB88F61034100E4F21FF6EAFE4080621058996103410000000077EDFE40806210D85A61034100AED85FF7EEFE400010E9B7436103410000000087F1FE40806210588D61034100E4F21F56F3FE40802BF697FE610341005227A024F5FE400010E9B7B7620341001C0DE052F6FE40802BF6977863034100AED85F11F7FE40802BF6171964034100361AC0BFF7FE40802BF697C964034100AED85F9EF8FE400010E9376A65034100E4F21F0DFAFE400010E937FB65034100AED85F9CFBFE400010E9375C660341006E3480FBFDFE400010E9B7D566034100AED85F9C01FF40004703F8A766034100E4F21F5D03FF40802BF6177966034100E4F21F8E04FF40807D1DB81966034100E4F21F9F05FF4080621058C265034100361AC05007FF408062105833650341006E34800209FF408062105894640341006E3480640AFF40802BF617DD63034100000000E70BFF40806210D8E5620341001C0DE0190DFF4000470378D6610341010300000004000000F10B000000226C786827014180CE88526649034100E9482E682701410008AC1C66490341006E34005D27014180B437F83C4A03410000000094270141004703F80F4B034100C9E5BF3B2601410010E9B73B4C0341000000001025014180BDC1170E4D03410077BE1F0024014180BDC117204D034100AED8DF2223014100992A18FD4C034100000000F021014180F4DB570A4D0341005BB1BFE320014100A2B4B7234D0341006E3480CB1F014100470378344D034100000000D01E014100A2B4B74A4D03410092CB7FCE1E014100A2B4B74A4D034100E5F29F4D2C014100D9CEF7D8680341006E348048380141806210D8FE7C0341006E3400EC36014100992A986B8E0341000000807340014180F4DBD738980341005227A0FD430141806210581AA9034100AED8DF9B4E01418062105829BC0341005227A0AF4F01410087A75760C3034100000000D0500141807D1DB84BC30341006E3480D651014100470378E9C30341005227204F53014100992A184FC4034100000000F053014100075F988EC40341000000802554014100470378BAC403410000000069540141004703F8EFC40341005227A0B255014180F4DB574DC60341000000001057014180EB51B84EC703410092CB7F7557014180F4DB57AAC7034100371A40E1570141004703F80FC8034100C9E53F555801410010E93755C9034100371AC061580141807D1D38DEC9034100C9E53F83590141004703F82FCB034100894160FD57014100D044581DCC034100000000105701410010E93791CC034100000000DE550141004703F81DCD034100C9E53F5B540141807D1DB8E4CD034100000000F053014100D044D819CE03410092CB7FD153014100D9CE7731CE034100E5F29FAD530141004703F84FCE034100E5F29FBB53014100D0445884CE034100000000F05301410010E9B7C6CE034100C9E5BFF454014180EB51386BD0034100C9E5BF3B550141004703F86FD103410092CBFF3456014100D9CEF74AD2034100000000105701418062105873D3034100E5F29F8657014100D0445819D40341005227A0D2570141004703F88FD4034100C9E53FFB580141807D1DB8C4D5034100000000305A014180621058D0D6034100C9E53FB55A014100A2B4B72AD70341005BB1BF5C5B0141004703F8AFD70341001B0DE0255C014100992A18DAD80341006E3400645C014180B437F8E3D90341008941E0C05C0141004703F8CFDA034100AED85FD45C014100992A984BDB034100000000505D014180621058A8DC034100371AC0E45D0141807D1DB864DB034100371AC0B35E0141004703F8CFDA03410092CBFF945F014100D9CEF7F4D90341000000007060014100D044D8B9D903410092CBFFED61014100D9CEF751D9034100AED8DF0962014180F4DBD749D903410000000090630141807D1D38E5D80341008941E0D564014100D044D8F5D80341008941605D650141802BF69702D9034100000000B066014100D0445812D90341001B0D60176801410047037817D903410000008073680141806210580CD9034100000000D069014100D044D812D90341008941E0CE6A0141802BF617D1D9034100C9E53F4A6C0141004703F8CFDA034100000080B06C014180B437780FDB034100000000F06C014180F4DB5734DB0341006E3480526E0141004703788DDC034100AED8DF786F0141004703F8EFDD0341005227A0C46F0141806210583BDE0341000000001070014100A2B4376EDE0341008941E02E710141802BF617F1DF034100522720C8710141004703F80FE1034100C9E5BF1B72014180EB513824E2034100C9E5BFBE7201410010E9B7BEE3034100AED85FE6720141004703F82FE4034100371AC0EC720141807D1D3873E403410000000030730141802BF6171BE6034100E5F21F5F73014100D044D820E703410092CB7F66730141004703F84FE7034100E5F21F6A730141802BF6178AE7034100371AC08F730141807D1D3810EA0341008941E095730141004703F86FEA034100371A408973014180621058C9EA0341000000003073014100D9CE7738EC034100000080F07201410047037850ED0341006E3400E1720141004703F88FED0341008941E0AE720141802BF61711EE034100E5F21F46720141802BF617C6EF034100C9E53FB1710141004703F8AFF0034100000080ED700141004703788DF1034100000000107001410010E93751F203410077BE9F436F014180BDC19703F3034100C9E53F8D6E0141004703F8CFF30341006E3480396E014180B4377819F50341005BB1BF1C6E01410010E937C3F50341008941E0D96D0141004703F8EFF6034100C9E5BF096E01410010E9B709F8034100522720136E0141806210D8ECF80341006E3480566E0141004703F80FFA034100000080AD6D014100470378CDFA034100000000F06C01410010E93742FB0341006E3400A86B014180B437F8E7FB03410092CBFFD46A014180B4377876FC0341001B0DE0766E0141807D1D3868020441001B0DE02F71014100D044D8C00C044100C9E53FA3730141807D1D3877150441006E340001720141802BF617D11C044100C9E5BF8D6F014100992A988725044100000000176F014180EB5138E4280441006E34002F6F0141004703F8EF28044100C9E53FA66F0141807D1DB859290441000000001070014100D9CE77B52904410077BE9FEA70014180F4DB57352B044100371AC04B710141004703F80F2C044100B30C715F710141805BB13F992C044100000080BB700141802BF697922B04410000000038700141807D1D38E02A044100000000F86E0141806210D8BB29044100000000B86D01410010E9B70729044100000000D96C014100470378C228044100000000386B01410010E9375428044100000000F86901410010E937F027044100371A408469014180621058D227044100000000B868014180621058A32704410000000078670141807D1D386D2704410000000038660141004703F84127044100000000F8640141802BF6172027044100000000B8630141806210D83427044100000000786201410010E9376927044100000000386101410010E9B7A227044100000000F85F0141802BF617AC27044100000000B85E0141004703F8A327044100000000785D0141807D1D389827044100000000385C0141807D1DB87527044100000000F85A01410047037804270441001B0D600A5A01410010E937F225044100E5F21F5A5A01410010E9370225044100000000F85A0141802BF697B124044100000000095C0141802BF6171224044100371A401E5D0141802BF6172223044100E5F29F6E5D0141802BF6173222044100E5F21F3C5D0141004703F841210441001B0DE0365D0141004703F85120044100000000EF5C0141806210D8611F0441001B0D60505C0141806210D8711E044100000000305B0141807D1DB8811D044100000000035A0141807D1DB8911C0441001B0D6010590141807D1DB8A11B04410000000078580141807D1D38001B0441000000003857014180621058EE190441000000001756014100470378D1180441000000803A55014100470378E117044100000000B8540141806210D8FE16044100E5F29F055401418062105801160441001B0D608C530141807D1D3811150441001B0DE0CB520141807D1D3821140441001B0D60425201410010E9373113044100E5F21F33510141802BF6174112044100000000B84F0141802BF6179811044100000000784E0141802BF6976B11044100000080F04D0141802BF6175111044100000000384D0141004703F83411044100000000F84B0141802BF6973711044100000000B84A01410010E9B7111204410000000078490141807D1DB8191304410000000038480141802BF6171C13044100371AC04D470141802BF6174112044100000000F846014100470378EE11044100371A405B460141802BF6175111044100000000B8450141802BF697B1100441000000007844014180621058870F04410000000038430141004703F8890E044100E5F29FF9410141806210D8900D044100000000B8400141004703F8A10C044100000080963F0141807D1DB8B00B044100E5F21F7C3E0141807D1DB8C00A044100E5F29F333D0141802BF697D009044100000080EB3B0141802BF697E008044100000000783A0141807D1D38FF0704410000000038390141004703F81A08044100000000F8370141807D1DB8A3080441001B0DE072370141802BF697D009044100371AC01D370141807D1DB8C00A044100000000B8360141802BF617A80B044100E5F21F49360141807D1DB8A00C04410000000078350141802BF617830D0441000000806A340141806210D8800E04410000000045330141004703F8700F0441001B0D603F320141004703F86010044100000000B8310141802BF6170311044100371A400E310141802BF6174112044100000080023101410010E9373113044100371A40F9300141807D1D382114044100371AC0FF300141807D1D381115044100E5F29F00310141806210580116044100E5F21F1B31014180621058F1160441000000805C31014100470378E1170441001B0DE0B731014100470378D1180441001B0DE056310141802BF697C11904410000000078300141807D1D38501A044100000000382F0141807D1D38C519044100000000F82D01410010E9371C19044100C9E53F652D014100470378D118044100000000B82C0141802BF6178618044100E5F21F8E2B014100470378E117044100000000382A0141807D1D381C17044100000000F82801410047037866160441001B0D6025280141806210580116044100000000B8270141807D1D38CA1504410000000078260141004703781815044100E5F29F81250141807D1D38211404410000000038250141806210584C1304410000000038250141807D1DB8781204410000000032260141802BF6175111044100000000B8270141807D1DB8A9100441001B0DE05C280141004703F86010044100000000F8280141807D1D382910044100000000382A0141802BF617BA0F044100371AC0C82A0141004703F8700F044100000000782B0141807D1DB8130F044100E5F29F702C0141806210D8800E044100E5F29F902D0141806210D8900D0441001B0D60D92D0141807D1DB8A00C044100371AC0BD2E0141807D1DB8B00B044100000000AA2E0141807D1DB8C00A0441001B0D60E92E0141802BF697D0090441001B0DE0902E0141802BF697E008044100000000142E014100470378F007044100000000862D0141004703780007044100371A40B82C0141806210581006044100000000782B0141806210D82F05044100000000382A0141807D1DB8970404410000008038290141806210583004044100000000B82701410010E9B79E030441000000008E260141807D1D384003044100000000382501410010E9B7FB02044100000000F8230141806210D8C302044100000000B8220141807D1D38AB0204410000000078210141806210D89F0204410000000038200141806210589202044100000000F81E0141802BF6978D02044100000000B81D0141004703787B02044100371A40A71C0141807D1D385002044100000000381B0141807D1DB8F501044100000000F8190141807D1D387001044100000000B8180141802BF6172B0104410000000078170141807D1DB81C01044100E5F29F99160141802BF6176001044100000000F8140141806210D83102044100000080EC130141807D1D384003044100371A40F012014180621058300404410000000078120141806210D87E040441001B0D608B11014180621058200504410000000038110141004703F88505044100E5F21FBD1001418062105810060441001B0D605A1001410047037800070441000000801B10014100470378F007044100000000F80F0141806210D87508044100371AC0C40F0141802BF697E008044100371A40070F0141802BF697D009044100000000780D01410010E937270A044100000000380C0141004703F8520A044100000080960B0141807D1DB8C00A044100000000F80A0141806210D8340B044100000080940A0141807D1DB8B00B044100371A40290A0141807D1DB8A00C044100000000B8090141807D1DB8390D04410000000078080141004703787A0D044100371A4021080141807D1DB8A00C044100371AC04B080141807D1DB8B00B0441001B0DE06F080141807D1DB8C00A0441001B0D60A7080141802BF697D009044100E5F21FC7080141802BF697E0080441001B0D60DF08014100470378F007044100E5F21FFD0801410047037800070441000000001109014180621058100604410000000002090141806210582005044100000000C5080141806210583004044100E5F29F7C080141807D1D384003044100371AC043080141807D1D3850020441000000805A080141802BF6176001044100E5F21F01090141802BF6177000044100000000B8090141004703781000044100000000F80A01410047037887FF034100000000380C0141806210588FFF0341001B0D60970C0141004703F88FFE034100000080AD0C0141806210D89FFD034100371AC08D0C0141806210D8AFFC0341001B0DE0470C0141806210D8BFFB034100000000BD0B01410010E9B7CFFA034100000000340B01410010E9B7DFF9034100371AC0480A0141802BF697EFF8034100000000B8090141802BF61768F80341001B0D6093080141802BF697FFF703410000000038070141802BF61740F7034100000000F80501418062105842F7034100000000B8040141802BF617A6F6034100000080A2030141004703781FF603410000000038020141806210585AF5034100000000F8000141004703789EF4034100000000A0000141806210583FF4034100000000B8FF0041004703F861F3034100000080D5FE0041807D1D385FF2034100E5F21F9CFD0041807D1D386FF1034100E5F29FFDFB0041802BF6177FF0034100000000B8FA0041802BF697FEEF03410000000078F90041802BF617D0EF034100371A40CBF80041802BF6177FF0034100C9E5BF55F80041807D1D386FF10341001B0D60DAF70041807D1D385FF20341000000005EF70041806210584FF3034100000000F8F60041807D1D381BF403410000000084F60041004703782FF503410000008069F60041004703781FF6034100000000DDF60041004703780FF703410000000038F800410047037871F70341000000001DF90041802BF697FFF703410000000078F90041807D1D387DF8034100E5F21FB0F90041802BF697EFF8034100E5F21FDFF900410010E9B7DFF9034100E5F29FD4F900410010E9B7CFFA034100E5F21F90F90041806210D8BFFB03410000000078F90041004703783CFC034100E5F29F5FF90041806210D8AFFC034100E5F29F30F90041806210D89FFD034100E5F21FC6F80041004703F88FFE03410000000038F80041802BF69751FF03410000000020F70041802BF61770000441001B0D6038F60041802BF6176001044100000000B8F50041807D1DB8160204410000000078F40041807D1D38B102044100000000DFF30041807D1D38400304410000000038F30041004703F8AE03044100000080C1F200418062105830040441000000804BF20041806210582005044100000000F8F10041806210D8D00504410000000046F100410047037800070441001B0DE021F1004100470378F007044100E5F21F2FF10041802BF697E0080441001B0DE033F10041802BF697D0090441001B0D6013F10041807D1DB8C00A044100000000B8F00041802BF617460B04410000000078EF004100470378250B0441000000000EEF0041807D1DB8C00A044100371AC0A2EE0041802BF697D009044100E5F29F44EE0041802BF697E00804410000000016EE004100470378F007044100000080EFED0041004703780007044100E5F21FF0ED0041806210581006044100371A4027EE004180621058200504410000008087EE0041806210583004044100000080E7EE0041807D1D384003044100E5F29F31EF0041807D1D38500204410000000078EF0041802BF6976201044100371A4064EF0041802BF61770000441001B0DE01DEF0041004703F87FFF03410000000038EE004100470378E9FE03410000000059ED0041004703F88FFE034100000000B8EB0041004703784DFE03410000000078EA0041807D1D3816FE03410000000038E9004180621058BAFD034100000000F8E700410010E9B740FD034100000000B8E60041807D1D38C0FC03410000000078E500418062105848FC03410000008042E40041806210D8BFFB034100000000F8E20041802BF6174AFB034100000000B8E10041806210D8F0FA03410000000078E00041806210D8F6FA03410000000038DF0041802BF6975FFB034100E5F29FA2DE0041806210D8BFFB034100000000F8DD00418062105811FC03410000008045DD0041806210D8AFFC034100000000B8DC0041802BF6171BFD0341000000802BDC0041806210D89FFD03410000000078DB0041004703F848FE03410000008050DA0041004703F87FFF034100371A4091D90041802BF6177000044100000000F8D80041004703F81801044100371AC024D80041807D1D385002044100000000B8D70041807D1DB8D90204410000000003D70041806210583004044100E5F29F7ED6004180621058200504410000008023D600418062105810060441001B0DE0EBD500410047037800070441000000005BD5004100470378F007044100371A40F7D40041802BF697E008044100371AC0F9D40041802BF697D009044100C9E5BFF2D40041807D1DB8C00A0441001B0D60DDD40041807D1DB8B00B0441001B0DE075D40041807D1DB8A00C044100000000F8D30041806210582F0D044100E5F29FB5D30041806210D8900D0441000000802ED30041806210D8800E044100000000B8D2004100470378700F04410000000078D10041802BF6979F0F04410000000038D00041807D1DB89D0F0441000000005ACF0041004703F8700F044100000000B8CD004100470378C40E044100E5F29F99CC0041806210D8800E04410000000038CB0041802BF6970E0F0441001B0D60ECCA0041004703F8700F044100371AC04BCA0041004703F86010044100000000F8C90041806210581711044100E5F29FAAC90041802BF6174112044100371A406AC900410010E93731130441000000002EC90041807D1D3821140441001B0DE0F5C80041807D1D381115044100A7E8C8DFC80041808251C94F150441008FC2F57CB200418037F8C2950F044100371AC097B20041004703F8700F04410000000078B30041004703F8880E0441001B0D6097B30041806210D8900D044100371A402CB30041807D1DB8A00C044100371AC0ACB20041807D1DB8B00B044100E5F29F7AB20041807D1DB8C00A0441001B0D6082B20041802BF697D00904410000000038B20041004703F817090441001B0DE01DB2004100470378F007044100000000F8B00041807D1D382C07044100371AC0DAAF0041806210581006044100000000F8B00041806210D8D405044100C9E5BFB3B1004180621058100604410000000038B20041806210586B0604410000000078B30041004703788806044100000000B8B40041802BF617E906044100000000F8B50041004703F86D0704410000000038B70041802BF617E30704410000000078B80041806210D83D08044100000000A7B90041802BF697E008044100000000F8BA0041004703F888090441001B0DE071BB0041802BF697D00904410000000038BC0041806210584C0A044100000000ECBC0041807D1DB8C00A04410000000078BD0041802BF6971E0B0441001B0DE058BE0041807D1DB8B00B044100E5F21FC6BF0041807D1DB8A00C04410000000038C1004100470378050D04410000000078C20041802BF697A10C0441001B0D6079C30041807D1DB8B00B044100E5F21FC7C40041807D1DB8C00A0441001B0DE099C50041802BF697D00904410000000038C60041004703F8ED08044100E5F29F99C6004100470378F0070441001B0DE0E4C60041004703780007044100E5F21F3FC7004180621058100604410000000078C70041004703788805044100000000E9C70041806210583004044100E5F29FE3C70041807D1D38400304410000000078C70041807D1DB85D0204410000000038C60041806210588201044100000000F8C40041806210582301044100000000B8C3004100470378DD0004410000000078C2004180621058A60004410000000038C10041004703F87100044100000000F8BF0041806210D82500044100000000B8BE0041802BF617EEFF03410000000078BD0041802BF697C3FF03410000000038BC0041802BF617A9FF034100371AC093BB0041004703F87FFF034100000000F8BA0041004703F827FF034100000000B8B900410047037895FE0341001B0DE0DBB80041806210D89FFD03410000000078B800418062105830FD034100E5F21F1CB80041806210D8AFFC034100000080E3B70041806210D8BFFB034100E5F29F09B800410010E9B7CFFA03410000000078B80041802BF6171AFA034100000000A0B90041802BF697EFF8034100000000F8BA0041806210580FF8034100E5F21FF8BB0041004703780FF703410000000078BD0041807D1D382CF6034100000000B8BE0041802BF61704F6034100E5F29F49BF0041004703781FF6034100000000F8BF0041004703784EF603410000000038C10041802BF697A1F603410000000078C20041004703F8C3F6034100000000B8C3004180621058ACF6034100000000F8C4004100470378CBF603410000000038C60041806210D8C9F60341001B0DE010C70041004703780FF7034100000000B8C80041802BF6975DF7034100371A4078C90041004703781FF6034100000000F8C900418062105855F503410000008095CA0041806210583FF4034100371AC0BACA0041806210584FF3034100371AC0C6CA0041807D1D385FF20341001B0DE0C3CA0041807D1D386FF10341001B0D60ABCA0041802BF6177FF0034100000080A2CA0041802BF6178FEF034100000080C8CA0041004703F89EEE03410000000038CB0041802BF697F3ED03410000000078CC0041806210585FED034100000000B8CD00410047037842ED034100000000F8CE00410047037843ED03410000000038D00041004703F851ED03410000000078D10041807D1D387AED034100000000B8D20041802BF617A3ED034100000000F8D30041004703F88FED03410000000038D500418062105855ED03410000000000D60041004703F8BEEC03410000000078D60041802BF6970DEC034100371AC0BAD60041806210D8DEEA034100000080D3D60041807D1DB8EEE903410000008005D70041807D1DB8FEE8034100371AC0A4D70041802BF6970EE8034100000080E4D80041802BF6971EE703410000000038DA0041806210D847E703410000000078DB004180621058A6E7034100371AC00ADC0041802BF6970EE8034100000000B8DC0041807D1DB8F1E8034100E5F29FAFDD0041802BF6970EE8034100000000B8DC0041802BF6976BE703410000000058DC0041802BF6971EE7034100E5F21F00DC0041004703782EE603410000000078DB0041807D1D38B5E50341000000808ADA0041004703783EE5034100000000F8D80041806210D8E5E503410000008099D80041004703783EE5034100E5F29FABD80041004703784EE4034100000000B8D7004180621058CBE3034100E5F29FFDD60041806210585EE303410000000078D60041802BF69732E303410000000038D50041004703F8BEE2034100371A409BD40041806210586EE2034100000000F8D300418062105821E2034100000000B8D20041806210588CE103410000000078D10041802BF617FBE0034100371AC07DD00041807D1D388EE0034100000000F8CE004180621058C2DF034100000000B8CD0041802BF617F3DE0341001B0DE04CCD0041802BF617AEDE03410000000078CC0041806210D8F7DD034100E5F29F72CB0041004703F8CDDC034100371A40C5CA0041004703F8DDDB0341001B0D6029CA0041806210D8EDDA0341001B0D607AC90041806210D8FDD9034100000000B8C80041806210582CD9034100371A40B3C70041807D1DB81DD8034100371AC0D6C60041802BF6972DD703410000000038C600410047037898D60341001B0DE0E9C50041802BF6973DD6034100000000F8C40041807D1D384ED5034100C9E53F12C40041004703785DD4034100000000B8C30041802BF61709D4034100E5F21F09C30041004703786DD303410000000078C20041807D1D38F3D20341001B0D60ECC10041806210587DD203410000000038C10041802BF617E6D10341001B0D60C9C00041806210588DD1034100000000F8BF0041807D1DB8ECD00341001B0DE093BF0041807D1D389DD0034100000000B8BE0041806210D800D0034100E5F21F40BE00410010E937ADCF03410000000078BD0041807D1D383ECF0341000000807CBC00410010E937BDCE034100000000F8BA0041004703780CCE03410000000064BA0041802BF617CDCD034100000000B8B900410047037889CD03410000000078B80041802BF6172FCD03410000000038B700410010E9B77CCD0341001B0D60D1B60041802BF617CDCD034100E5F21F78B600410010E937BDCE034100E5F29F97B600410010E937ADCF034100371A4095B60041807D1D389DD0034100000000F8B50041802BF61752D1034100000000B8B400410047037877D2034100371AC0A9B40041004703786DD3034100371AC0FBB40041004703785DD403410000008071B50041802BF6974DD5034100000000F8B500418062105820D6034100E5F29F40B60041802BF6972DD7034100371AC051B60041807D1DB81DD8034100000000F8B50041802BF617E2D8034100000000B8B40041806210D8CCD803410000000078B30041806210D867D80341000000809EB20041807D1DB81DD8034100000000F8B00041807D1D389BD7034100000000B8AF00410047037836D703410000000078AE004100470378EBD603410000000038AD0041802BF6171DD7034100E5F29F38AC0041807D1DB81DD8034100E5F21F04AC0041807D1DB80DD903410000000002AC0041806210D8FDD9034100000000F8AB0041807D1D38D7DA034100000000B8AA0041807D1D3833DB034100E5F21F90A90041806210D8FDD9034100E5F29F37A90041807D1DB80DD9034100000080DDA80041807D1DB81DD8034100371AC046A80041802BF6972DD7034100E5F21F4CA70041802BF6973DD6034100000000B8A50041802BF61762D5034100E5F21F98A40041802BF6974DD503410000000038A30041807D1D3823D5034100000000F8A10041004703F8FFD4034100000000B8A000410010E937A6D4034100E5F29F3BA00041004703785DD4034100000000D49F0041004703786DD3034100000000B8A0004100470378A9D2034100000000F8A10041802BF6174CD203410000000038A300418062105836D203410000000078A400410047037850D2034100000000B8A50041806210D804D2034100000000A3A60041806210588DD103410000000038A80041802BF697A0D00341001B0D604DA900410010E937ADCF03410000000078A90041802BF6173FCF034100E5F21F8FA900410010E937BDCE034100E5F29F80A90041802BF617CDCD034100E5F21F84A90041802BF617DDCC034100371A40E7A80041004703F8ECCB034100371A4046A80041004703F8FCCA034100000000F8A60041004703F818CA034100000000B8A50041807D1DB882C9034100000080EDA40041806210D81CC903410000000078A400410010E9B7E9C803410000000038A300418062105850C8034100000000F8A10041806210D891C70341001B0D6048A10041807D1DB83CC7034100000000B8A00041004703F8F9C6034100000000789F0041806210589DC6034100000000389E0041807D1DB88EC6034100000000F89C0041802BF61792C6034100000000B89B0041802BF617BCC6034100000000789A0041806210D8C1C603410000000038990041802BF697BFC6034100000000F8970041802BF697B0C6034100000000B89600418062105897C6034100000000789500410047037868C6034100E5F21F89940041807D1DB84CC6034100000000F8920041807D1D384BC6034100000000B8910041004703F8E4C6034100371A4012910041807D1DB82CC8034100000080A9900041806210D81CC903410000000078900041004703F8E4C90341001B0DE04C900041004703F8FCCA0341001B0D60EC8F0041004703F8ECCB034100E5F21F828F0041802BF617DDCC034100000000388F0041806210D87CCD0341001B0D60D98E00410010E937BDCE0341001B0DE0908E00410010E937ADCF034100371A40488E0041807D1D389DD0034100E5F29F3D8E0041806210588DD1034100000000878E0041806210587DD2034100000000388F0041806210D836D303410000000078900041807D1DB869D3034100000000B8910041806210D8E3D3034100371A4035920041004703785DD4034100000000DD910041802BF6974DD503410000000078900041004703783BD6034100000080668F0041802BF6972DD7034100E5F29FDF8E0041807D1DB81DD8034100000080B68E0041807D1DB80DD9034100E5F21FAC8E0041806210D8FDD9034100371A40D58E0041806210D8EDDA034100371AC0158F0041004703F8DDDB034100000000F88D0041802BF6178CDC034100000000B88C0041004703F8E6DB034100000000868B0041806210D8EDDA034100000000388A0041807D1D3816DA034100000000F888004100470378B4D9034100000000B8870041802BF69756D9034100000000E2860041807D1DB80DD903410000000038850041802BF697B0D8034100000000F8830041807D1DB885D8034100000000B8820041806210D8A6D80341001B0D6021820041807D1DB80DD90341001B0D60AC810041806210D8FDD90341001B0D60A4810041806210D8EDDA0341001B0DE01B820041004703F8DDDB034100000080A0820041004703F8CDDC034100E5F21F94830041802BF617BEDD034100000000F8830041802BF69743DE034100371A4049840041802BF617AEDE034100E5F21FB9840041802BF6179EDF034100371AC0D8840041807D1D388EE0034100000000F8830041807D1DB8DFE0034100000000B88200410047037854E103410000000078810041004703F859E103410000000038800041004703F816E1034100000000F87E0041802BF617C1E0034100000080417E0041807D1D388EE0034100000000B87D0041004703F865E0034100000000787C0041807D1D38F9DF034100000000387B004100470378A9DF034100000000F8790041806210D87BDF034100000000B8780041807D1DB876DF03410000000078770041806210D88ADF0341000000003876004100470378F6DF03410000008038750041807D1D388EE00341001B0D60027400410010E9377EE103410000008084720041806210586EE203410000000038710041806210581EE3034100000000F86F0041802BF697D3E3034100000000B86E00410047037840E4034100000000786D0041802BF69772E4034100000000386C00410047037886E4034100000000F86A0041004703F866E4034100371AC06F6A0041004703784EE4034100000000B8690041806210582AE403410000000078680041004703F8E8E3034100000000386700410010E93701E4034100371AC0B1660041004703784EE4034100000000F8650041806210D8F5E4034100371A4092650041004703782EE6034100E5F29F91650041802BF6971EE70341001B0D60AF650041802BF6970EE8034100000080B8650041807D1DB8FEE80341001B0D60D4650041807D1DB8EEE9034100000000F86500410010E9377EEA0341001B0D6048660041806210D8CEEB0341000000809A660041004703F8BEEC034100E5F29F0F670041004703F8AEED034100371A4066670041004703F89EEE034100E5F29FA9670041802BF6178FEF034100E5F21FFD670041802BF6177FF003410000008062680041807D1D386FF10341001B0DE0CF680041807D1D385FF20341001B0DE03B690041806210584FF3034100000000B8690041004703F819F4034100371AC0736A0041004703782FF5034100000000F86A0041004703F8EEF5034100000000E46B0041004703780FF7034100000000F86A004100470378B2F70341000000008F6A0041802BF697FFF7034100000000B8690041802BF69728F8034100000080F8680041802BF697FFF703410000000078680041807D1DB8EDF703410000000038670041807D1DB86AF7034100371A407F660041004703780FF7034100000000F865004180621058D1F6034100000000B86400410047037870F603410000000078630041802BF69723F603410000000038620041004703F8F6F5034100000000F8600041806210D8F0F5034100C9E53F42600041004703781FF6034100000000B85F0041807D1DB833F6034100000000785E0041004703F866F6034100000000385D0041807D1D388CF6034100000000F85B0041806210D8F4F6034100000000B85A0041806210588EF7034100E5F29F005A0041802BF697FFF703410000000078590041806210583BF8034100371AC042580041802BF697EFF8034100569FAB5C57004180923A01A6F90341002AA993A65000410060764F24F8034100371A40B4500041802BF697FFF7034100371AC0D5500041004703780FF7034100371AC0D2500041004703781FF6034100371A407C500041004703782FF5034100371A4003500041806210583FF4034100000000784F0041802BF6179CF30341000000804A4E0041807D1D385FF20341001B0D60934D0041807D1D386FF1034100E5F21F244D0041802BF6177FF00341000000002E4D0041802BF6178FEF034100000080434D0041004703F89EEE034100371AC00F4E0041004703F8AEED034100000000784F0041802BF61722ED034100000000B8500041802BF69703ED034100000000F8510041802BF69704ED03410000000038530041802BF697F7EC0341000000007854004100470378D7EC034100371AC0FE540041004703F8BEEC034100000000B8550041806210588FEC034100000000F8560041806210D8DCEB034100371AC083570041806210D8DEEA034100E5F21FB9570041807D1DB8EEE903410000000095570041807D1DB8FEE803410000000005570041802BF6970EE8034100E5F29F28560041802BF6971EE7034100000000B8550041802BF697BCE6034100371AC021550041004703782EE6034100000000785400410047037892E5034100371A4016540041004703783EE5034100371A404E530041004703784EE40341001B0D6066520041806210585EE3034100E5F21F11520041806210586EE2034100E5F29FA55100410010E9377EE1034100371AC0CF510041807D1D388EE0034100E5F21FC4520041802BF6179EDF03410000000038530041806210D85FDF03410000000078540041807D1DB836DF034100000000B8550041807D1D383ADF034100E5F21FB6560041802BF6179EDF03410000000040570041802BF6179EDF03410000000038580041806210586EDF03410000000078590041004703787FDF034100371AC04B5A0041802BF6179EDF034100000000F85B0041004703F837DF034100E5F21F815C0041802BF617AEDE0341001B0DE0DC5C0041802BF617BEDD034100000080EA5C0041004703F8CDDC034100371A40725C0041004703F8DDDB0341001B0DE0235C0041806210D8EDDA034100000000155B0041806210D8FDD9034100000000B85A0041004703F89DD9034100E5F21F1C5A0041807D1DB80DD903410000000078590041806210D848D80341001B0DE06E580041802BF6972DD70341001B0D6092570041802BF6973DD6034100000000F8560041806210D897D5034100000000B85500410047037870D4034100E5F29FB8540041004703786DD3034100371A40D9530041806210587DD20341001B0D6040530041806210588DD1034100E5F29F2D530041807D1D389DD00341000000802B5400410010E937ADCF034100000000B85500410047037865CF034100000000F85600410047037855CF03410000000038580041004703F80CCF034100000080385900410010E937BDCE034100000080BE5900410010E937BDCE034100000000B85A0041802BF697DECE034100E5F29FC55B00410010E937ADCF0341000000006A5C0041807D1D389DD0034100000000E75C0041806210588DD10341001B0D60B85D0041806210587DD2034100000000785E0041004703F82DD3034100000000B85F00418062105828D4034100E5F21F62600041004703785DD4034100000000F860004180621058A3D4034100000000386200410010E937CCD403410000000078630041004703F8ADD4034100E5F29F04640041004703785DD4034100371AC07A640041004703786DD3034100E5F21F4D640041806210587DD203410000008012640041806210588DD1034100000000EE630041807D1D389DD00341001B0D60166400410010E937ADCF0341000000004D6400410010E937BDCE034100000000B8640041802BF617F7CD034100E5F29FE2640041802BF617DDCC034100E5F21F19650041004703F8ECCB0341001B0D6067650041004703F8FCCA034100371AC0A5650041806210D80CCA034100371AC0B1650041806210D81CC9034100E5F21F83650041807D1DB82CC8034100371AC001650041807D1DB83CC7034100000000B8640041004703F8DDC603410000000070640041807D1DB84CC60341001B0DE0E8630041802BF6975CC5034100000000B0630041802BF6976CC403410000008046640041004703787CC3034100000000B8640041004703F836C3034100000000F8650041802BF6973BC3034100371AC0AA660041004703787CC303410000000038670041807D1DB8B1C303410000000078680041004703781DC4034100E5F29F53690041802BF6976CC4034100000000F86A0041802BF697EFC4034100000000386C0041004703F811C50341001B0DE00F6D0041802BF6975CC5034100000000B86E0041806210D8A4C5034100000000F86F0041802BF69769C50341001B0D6069700041802BF6976CC403410000008086700041004703787CC30341001B0DE097700041004703788CC2034100000000B0700041806210589CC1034100E5F29F2771004180621058ACC003410000000078720041807D1D38D4BF034100000000B8730041806210586DBF034100000000F8740041004703F845BF03410000000038760041807D1D3816BF034100E5F21F70770041807D1D38CCBE0341001B0D60B6780041807D1D38DCBD0341001B0DE0C5780041802BF617ECBC03410000008087780041802BF617FCBB03410000000008780041004703F80BBB03410000000078770041807D1D3841BA034100000000AD760041806210D82BB903410000000038760041802BF697C3B80341001B0DE0BE750041806210D83BB8034100000000F874004100470378AFB70341001B0DE070740041806210D84BB7034100000000B8730041802BF697E0B60341001B0DE0E27200410010E9B75BB6034100000000787200410047037816B60341001B0D608D7100410010E9B76BB50341001B0DE089700041802BF6977BB4034100000000F86F0041807D1DB8E4B3034100371A40A46F0041802BF6978BB3034100000000B86E0041802BF617A5B2034100000000786D0041802BF617CCB1034100000000386C0041802BF69736B1034100E5F21F136B004180621058BBB0034100000000B86900410047037814B00341001B0D60A768004180621058CBAF0341001B0D601568004180621058CBAF03410000000038670041802BF697EAAF034100000000F8650041806210D854B0034100000000B8640041802BF6975FB0034100371A409C63004180621058CBAF034100000080D662004180621058DBAE03410000000038620041802BF61713AE034100E5F21F50610041807D1D38FBAC034100000000F860004180621058A5AC034100371AC053600041802BF6170BAC034100000000B85F0041806210D88AAB034100371AC0DF5E0041802BF6171BAB034100000000785E0041807D1D38DEAA034100000000385D0041802BF61761AA034100E5F29F745C0041004703F82AAA034100000000F85B0041004703F8FFA9034100000000B85A0041802BF697AFA903410000000078590041807D1DB877A903410000000038580041802BF69751A9034100000000A7570041004703F83AA9034100000000F85600410047037821A9034100000000B8550041004703F8DDA8034100E5F29F78540041004703F84AA803410000000038530041004703F8CBA7034100000000F8510041807D1DB8AFA7034100000000B85000418062105894A7034100000000784F0041802BF617AFA7034100000000384E0041806210D81DA8034100000000F84C00410010E9B7AFA8034100371A40234C0041004703F83AA9034100000000B84B0041806210D885A9034100371AC0F64A0041004703F82AAA034100000000784A0041806210D89CAA034100000080F6490041802BF6171BAB03410000000038490041806210D8CFAB034100000000F84700410010E937ADAC03410000008021470041807D1D38FBAC03410000008010460041807D1D38FBAC03410000000078450041807D1DB8CBAC034100371AC006450041802BF6170BAC0341001B0D6028450041802BF6171BAB034100371A4051450041004703F82AAA034100C9E5BF77450041004703F83AA9034100371AC0A6450041004703F84AA803410000000091450041806210D85AA7034100371AC03F450041806210D86AA60341001B0DE0A6440041807D1DB87AA503410000000038440041807D1D38F8A40341001B0DE0BB430041807D1DB88AA4034100000000F8420041807D1D380FA4034100000080FB410041802BF6979AA303410000000078400041802BF69743A3034100000000383F0041806210D826A3034100000000F83D00418062105854A3034100000080F63C0041802BF6979AA3034100000000783B0041806210D8ECA3034100000000383A0041802BF6974DA4034100E5F21F13390041807D1DB88AA4034100000000B8370041802BF697D3A403410000000078360041807D1D3834A5034100E5F29FE9350041807D1DB87AA503410000000038350041806210D831A60341001B0D6095340041806210D85AA70341001B0DE034340041004703F84AA8034100000000F8330041806210D8B3A8034100000000B83200410010E937F3A803410000000078310041806210D860A803410000000038300041807D1D38D2A7034100000000F82E00418062105881A70341001B0DE02A2E0041806210D85AA7034100000000782C0041806210D820A7034100000000382B0041806210D80CA7034100000000F8290041806210D82AA7034100000000B8280041004703F8FEA703410000000055280041004703F84AA8034100E5F29F94270041004703F83AA9034100E5F21F28270041004703F82AAA034100E5F29FE7260041802BF6171BAB034100E5F21FD0260041802BF6170BAC034100E5F29F7D260041807D1D38FBAC03410000000038260041802BF69781AD034100371A406C250041807D1D38EBAD0341000000801C25004180621058DBAE034100E5F21FD324004180621058CBAF034100E5F21F6324004180621058DBAE034100000000B8230041004703F8A5AE03410000000078220041807D1D3861AE034100371A40F3210041807D1D38EBAD03410000000038210041004703F880AD0341001B0DE0AD200041807D1D38FBAC034100000000F81F0041802BF6175CAC034100371A40991F0041802BF6170BAC034100000000B81E0041802BF6976DAB034100E5F21FD71D0041802BF6171BAB034100000000381C004100470378C1AA034100000000F81A0041806210D8C5AA034100000000B819004180621058E8AA034100000000FC180041802BF6171BAB03410000000078180041802BF69744AB034100E5F21FBD170041802BF6170BAC034100C9E5BF7E170041807D1D38FBAC03410000008053170041807D1D38EBAD034100371AC0F316004180621058DBAE0341000000007216004180621058CBAF034100371AC01F16004180621058BBB00341001B0D60BA15004100470378ABB103410000008031150041004703789BB2034100000000B8140041807D1DB812B303410000008024140041802BF6978BB303410000000078130041806210D8F8B303410000000038120041004703F823B4034100000000F81000410047037801B4034100000000B80F004100470378D5B3034100000000780E00410010E937A7B3034100000000380D00418062105895B3034100000000F80B0041807D1DB871B3034100000000B80A0041806210D8CAB30341001B0DE0BF090041802BF6977BB4034100371AC08C0800410010E9B76BB50341000000003808004100470378DBB5034100E5F21F770700410010E9B75BB6034100000000F806004100470378D8B6034100371A400B060041806210D84BB703410000000078040041802BF617EBB703410000000056030041806210D84BB7034100000000F8010041004703F8DDB6034100E5F21F000100410010E9B75BB6034100000000F80100418062105878B50341001B0DE0D3020041802BF6977BB403410000000038030041806210D8DEB30341000000806B040041004703789BB20341000000802805004100470378ABB1034100000000B8050041806210580CB10341000000009306004180621058CBAF034100000000F8060041004703F842AF0341001B0D60BF070041807D1D38EBAD03410000000038080041802BF6975CAD034100371AC001090041807D1D38FBAC03410000000078090041806210D8B9AC034100000000B80A0041806210D891AC034100000000F80B0041806210D887AC034100000000380D0041004703F837AC034100000000780E0041004703F853AB034100371AC0320F0041004703F82AAA0341001B0D60620F0041004703F83AA9034100000000610F0041004703F84AA80341001B0D60450F0041806210D85AA7034100E5F29F020F0041806210D86AA6034100E5F21F9D0E0041807D1DB87AA5034100E5F29F0D0E0041807D1DB88AA4034100371AC00A0E0041802BF6979AA3034100E5F29FC90D0041802BF697AAA2034100371AC0820D004100470378BAA1034100E5F29F660D004100470378CAA0034100000000270C004100470378DA9F0341001B0DE0600B004100470378DA9F034100000000B80A0041802BF69728A0034100E5F21FD009004180621058EA9E034100C9E53F4709004180621058FA9D0341001B0D60900A0041807D1D380A9D034100000000F80B0041806210D8749D034100000000380D0041802BF617E79D034100000000780E0041802BF697849E034100E5F29FB20F004180621058EA9E034100000000F810004100470378849F0341000000001812004100470378DA9F03410000000078130041802BF6174EA0034100371AC0A214004100470378CAA0034100000000F8150041802BF6172DA1034100000000381700410010E9B738A103410000000078180041807D1DB85BA1034100000000B8190041806210D84EA1034100000000F81A0041004703F80CA1034100000000381C004100470378CEA0034100000000781D0041806210D860A0034100E5F29F601E004100470378DA9F034100E5F29F471E004180621058EA9E034100E5F29F3D1E004180621058FA9D034100371A40F61D0041807D1D380A9D034100000000D51D0041807D1D381A9C034100E5F29FCA1D0041802BF6172A9B034100000000781D0041807D1DB8AB9A034100000080561D0041802BF6173A9A034100371A40051D0041802BF6174A99034100000000381C0041004703F896980341001B0D60451B0041004703F86997034100E5F21F651A0041806210D87996034100000000B8190041802BF617E195034100E5F21F55190041806210D889950341000000007818004100470378E794034100E5F21F14180041807D1DB8999403410000000038170041802BF697F8930341001B0D60D7160041807D1DB8A993034100000000F815004180621058F492034100000000B8140041806210D8EF9103410000000078130041807D1D38099103410000000038120041807D1DB8FF8F034100000000F8100041004703F8268F034100000000B80F0041806210D8858E034100000000780E0041802BF697188E034100000000380D0041004703F8B88D034100000000F80B0041802BF617748D034100000000B80A0041806210D8208D03410000000078090041806210D85B8C034100371A40090900410010E937398B034100E5F29F26090041802BF617498A03410000008063090041802BF6175989034100371A40AE090041802BF6176988034100371AC0D0090041004703F878870341001B0D60D3090041004703F88886034100000000A8090041806210D8988503410000000078090041806210D8FD840341001B0DE0DE080041807D1DB8B88303410000000076080041807D1DB8C882034100371A40DB070041807D1DB8D881034100E5F29F23070041802BF697E88003410000008092060041802BF697F87F0341001B0DE0D005004100470378087F034100E5F29F1305004100470378187E03410000000078040041802BF617597D034100371A409903004180621058387C03410000000038030041802BF697B77B0341001B0DE0E5020041807D1D38487B034100E5F21F57020041807D1D38587A034100000000F80100410010E9B79B79034100371A406C010041802BF6177878034100E5F29F09010041802BF6178877034100000000B8000041004703F8FC76034100371A4025000041004703F8A7750341005227A068FFFF40806210D8B7740341005227A040FEFF40806210D8C773034100CAE53F1CFDFF40806210D8D77203410000000070FCFF40802BF6175F72034100AED85FBFFBFF400010E9B7E771034100361AC01BFAFF40807D1DB8F770034100361AC05CF8FF40802BF697077003410000000070F7FF40802BF697A06F0341005227A04AF6FF40802BF697176F034100000000F0F4FF40004703F8866E034100E4F21F3BF4FF4000470378276E034100361AC0ADF2FF4000470378376D0341006E348090F1FF4080621058476C034100000000F6F0FF4080621058576B0341000000000DF1FF4080621058676A034100361AC04EF1FF40807D1D38776903410000000070F2FF40802BF697AE68034100000000F0F4FF40807D1D38A467034100E4F21F8FF6FF40802BF617A7660341005227A04CF7FF40004703F8B665034100CAE53FD5F7FF40004703F8C664034100CAE53F13F8FF40806210D8D663034100E4F21F2CF8FF40806210D8E662034100CAE53F0FF8FF40806210D8F6610341006E3480AAF7FF40807D1DB80661034100361AC0FEF6FF400010E9B71660034100E4F21F05F6FF40802BF697265F034100E4F21FA1F4FF40802BF697365E0341001C0DE03FF4FF4000470378465D03410092CB7FA4F4FF4000470378565C03410000000070F7FF40004703F8545C034100000000F0F9FF4000470378375D03410000000048FBFF40802BF697365E03410000000070FCFF4000470378F25E034100000000F0FEFF400010E937F95F034100000000B8000041004703781360034100000000F8010041802BF697265F0341000000003803004100470378575E03410000000078040041802BF617D45D034100C9E5BF7405004100470378465D034100000000F8060041807D1DB8BE5C034100C9E5BF9607004100470378565C03410000000038080041807D1D38FE5B03410000000078090041807D1DB8915B0341001B0DE0B909004100470378565C0341000000009009004100470378465D0341001B0D60A1080041802BF697365E03410000000038080041802BF617085F0341001B0D60120700410010E9B71660034100C9E53FC0050041807D1DB80661034100000000780400418062105891600341000000003803004180621058F460034100E5F21FC4020041806210D8F66103410000000038030041806210D89062034100E5F29FDB030041806210D8F66103410000000078040041806210D87261034100E5F21FD5040041806210D8F661034100E5F21FCA040041806210D8E6620341000000007804004100470378506303410000000038030041806210D81C63034100371AC099020041806210D8D663034100371AC04A020041004703F8C664034100371AC002020041004703F8B6650341001B0D6002020041802BF617A766034100000000F8010041806210582867034100371AC0CD010041807D1D3887680341001B0D60BF010041807D1D387769034100371AC0BA01004180621058676A034100000080B401004180621058576B034100000000C001004180621058476C034100371A40CC01004100470378376D034100000000F8010041004703F8D06D034100E5F21F63020041802BF697176F034100E5F21FBD020041802BF697077003410000008025030041807D1DB8F770034100371AC0B70300410010E9B7E7710341000000806D040041806210D8D7720341001B0D6085050041806210D8C773034100000000F8060041004703F8807403410000000082070041806210D8B77403410000000038080041802BF617FB7403410000000078090041802BF6176D75034100000000B80A0041806210584776034100000000250B0041004703F89776034100000000F80B00418062105842770341001B0DE00F0D0041802BF6178877034100000000780E0041807D1DB8CF77034100000000B80F0041806210588D77034100000000F810004100470378BB76034100E5F29F18120041004703F8A775034100371AC005130041004703F8A77503410000000078130041807D1DB8F175034100E5F21F86130041004703F897760341001B0DE04E130041802BF6178877034100371AC00D130041802BF61778780341001B0D60D7120041807D1D386879034100E5F21F95120041807D1D38587A03410000000038120041807D1D38FE7A034100E5F21FD4110041807D1D38487B0341000000802311004180621058387C034100371AC00F11004180621058287D0341001B0D605711004100470378187E0341000000809D11004100470378087F034100000000F0110041802BF697F87F03410000000038120041004703788880034100371AC003130041807D1DB8D88103410000000078130041806210D87182034100E5F21FD9130041807D1DB8C8820341001B0D60B7140041807D1DB8B8830341001B0D608D150041806210D8A884034100000000F8150041802BF6174085034100E5F21FEF160041004703F8888603410000000038170041802BF617FB8603410000008076170041004703F87887034100371AC00C180041802BF617698803410000000078180041806210D80489034100000000B8190041004703F8388A034100000000F81A0041802BF697E98A0341000000009C1B00410010E937398B034100000000381C0041807D1D38818B034100000000781D0041802BF697F58B034100371AC0301E0041807D1D38298C034100000000B81E0041807D1DB84E8C034100000000F81F0041004703F8868C03410000000038210041806210D8B48C0341000000007822004100470378E28C034100000000B8230041802BF697138D034100000000F8240041004703F8518D03410000000038260041807D1D38928D03410000000078270041806210D8CF8D034100E5F29FAF28004180621058098E034100000000F829004180621058408E034100000000382B0041004703F8718E034100000000782C0041004703789A8E034100000000B82D0041807D1DB8A08E034100000000F82E0041806210588B8E0341000000003830004100470378728E03410000000078310041802BF697528E034100000000B8320041802BF617478E034100000000F8330041807D1D38428E03410000000038350041004703F84B8E03410000000078360041004703783F8E034100000000B837004100470378388E034100000000F8380041807D1DB8298E034100000000383A004180621058228E034100000000783B0041802BF6974D8E034100000000B83C0041807D1D38AD8E034100E5F21F713D004100470378F98E034100000000F83D0041802BF6172C8F034100000000383F0041802BF697948F0341000000007840004100470378CD8F034100000000B8410041807D1D38CD8F034100000000F8420041806210D8958F034100E5F21F1A44004100470378F98E034100371AC0F644004180621058098E03410000000078450041807D1DB8778D034100E5F29FC545004180621058198D034100E5F21FE3450041807D1D38298C034100000000084600410010E937398B034100000000BD450041802BF617498A0341000000007845004180621058DA8903410000008011450041802BF61759890341000000009B440041802BF617698803410000000038440041807D1D38D787034100371AC05C430041004703F88886034100000000F8420041802BF697F585034100371AC033420041806210D8A884034100000000B841004180621058C383034100371A4037410041807D1DB8C882034100371A40BE400041807D1DB8D88103410000000078400041004703F84081034100E5F21FC83F0041802BF697F87F034100E5F29F393F004100470378087F034100E5F21F933E004100470378187E034100000000FA3D004180621058287D0341001B0DE0353D004180621058387C034100000000B83C0041806210D8CB7B034100000000223C0041807D1D38487B034100000000783B0041806210D8D17A034100371AC0D13A0041807D1D38587A034100000000383A0041802BF617FF790341000000002E390041807D1D386879034100000000B837004100470378AF78034100E5F21F4E370041802BF617787803410000000078360041802BF6970A78034100371A4093350041802BF6178877034100000000F8330041807D1D389A76034100000000B8320041807D1DB8AC7503410000008015320041806210D8B774034100371A40A3320041806210D8C773034100000000F8330041807D1DB802730341000000003835004100470378857203410000000078360041807D1DB89172034100000000B8370041004703F8BA72034100000000F8380041807D1D38FB72034100000000383A0041806210D82E73034100000000783B0041802BF6975A73034100000000B83C0041806210D85C73034100000000F83D00410010E9B7E172034100371A40023E00410010E9B7E771034100371A40043E0041807D1DB8F770034100000000BD3D0041802BF6970770034100371A40C03D0041802BF697176F034100000000423D004100470378276E034100000000B83C0041806210D85E6D0341001B0DE0DF3B004180621058476C034100000000783B0041807D1D38B96B034100E5F21F223B004180621058576B034100E5F21F543A004180621058676A0341001B0D607E390041807D1D387769034100000000F838004100470378D568034100E5F29FDC370041802BF61797670341001B0DE0EC360041802BF617A76603410000000078360041806210D82566034100E5F29FFB350041004703F8B66503410000000038350041802BF69712650341001B0DE015340041806210D8D66303410000000086330041806210D8E662034100E5F29F28330041806210D8F661034100000080D2320041807D1DB80661034100000000B8320041802BF6177F60034100E5F29FB4320041802BF697265F034100371A4079330041802BF697365E034100000000F8330041802BF617C65D034100000000FE34004100470378465D0341000000007836004180621058DE5C034100000000B8370041004703F8385D03410000000098380041802BF697365E034100000000F8380041802BF697035F034100000080CE3900410010E9B71660034100000000383A0041004703F8DE60034100000000783B0041807D1DB84661034100000000B83C0041806210D82261034100000000F83D0041802BF6178B600341001B0DE0E73E00410010E9B71660034100E5F29F3A400041802BF697265F0341000000007840004180621058BC5E0341001B0DE096400041802BF697365E034100E5F21FB340004100470378465D034100371AC0BB40004100470378565C034100371AC0BD40004100470378665B0341001B0DE0AE40004180621058765A034100371AC0884000418062105886590341001B0D6047400041807D1D389658034100000000F53F0041807D1D38A657034100000000B73F0041802BF617B6560341001B0DE07A3F0041802BF617C6550341001B0DE0403F0041004703F8D5540341001B0D602A3F0041004703F8E553034100E5F21F4E3F0041004703F8F552034100E5F21F49400041806210D8055203410000008082410041806210D80552034100000080E3420041004703F8F552034100E5F29FA4430041004703F8E55303410000000038440041004703F8AD54034100371A4012450041802BF617C65503410000000078450041807D1D386F56034100371AC017460041807D1D38A657034100371A408E460041807D1D3896580341001B0D60024700418062105886590341001B0D605547004180621058765A034100E5F21F7647004100470378665B0341001B0D607947004100470378565C034100371AC08647004100470378465D034100000000A4470041802BF697365E0341001B0D60C1470041802BF697265F034100371A40E94700410010E9B71660034100E5F29F1C480041807D1DB806610341001B0D6054480041806210D8F661034100E5F21F7F480041806210D8E662034100E5F29FBB480041806210D8D663034100E5F29FEE480041004703F8C6640341001B0D6014490041004703F8B66503410000000038490041807D1D3841660341001B0D60C1490041802BF61797670341001B0DE05E4A0041807D1D388768034100371AC0EC4A0041807D1D387769034100371A40664B004180621058676A034100000000B84B0041802BF6172F6B0341001B0D609B4C004180621058476C034100E5F29FB04D004100470378376D034100000000384E0041004703F8E26D0341001B0DE0534F0041802BF697176F034100000000B8500041807D1D38D56F034100000000F8510041807D1DB8FE6F03410000000038530041802BF617F96F03410000000078540041807D1D38CB6F034100000000B855004180621058486F034100000000F8560041004703F85A6E0341000000001658004100470378376D034100E5F29F8558004180621058476C034100000080F758004180621058576B03410000000078590041806210D8736A0341001B0D60895A0041807D1D387769034100000000FF5A0041807D1D388768034100000080375B0041802BF6179767034100371A40595B0041802BF617A766034100C9E53F3C5B0041004703F8B665034100E5F21FEA5A0041004703F8C664034100000000B85A0041806210584164034100000080615A0041806210D8E6620341000000801A5A0041806210D8F661034100371AC0CC590041807D1DB8066103410000000078590041806210D81E60034100E5F29F33590041802BF697265F034100371AC0A7580041802BF697365E0341001B0D604958004100470378465D0341000000808657004100470378565C034100000080FD56004100470378665B034100371A40B256004180621058765A0341001B0DE09E5600418062105886590341001B0DE0D2560041807D1D389658034100000000F8560041806210D82558034100E5F21F33570041807D1D38A6570341001B0D60AE570041802BF617B6560341001B0DE0EB570041802BF617C655034100E5F29FC9570041004703F8D55403410000000074570041004703F8E553034100000000F8560041806210584F530341001B0D60D6550041806210D805520341001B0D6027550041806210D815510341001B0D60EA540041807D1DB82550034100C9E5BF95540041807D1DB8354F034100C9E53F3D570041802BF617AA4B034100371AC097580041806210D8A949034100371A401F580041004703F821460341000000806A580041802BF6176D44034100E5F21FC55A0041807D1DB83043034100E5F21FB65B0041802BF617A941034100000000B65A0041004703785D3E034100E5F29F5B5A0041807D1DB8B73C03410052B89EF75A00410022FDF6E33B034100C1CAA1115B00418095438B5A3C0341007B14AE2A5B004180E5D0221A3D034100302AA92B5B00410092CBFF1E3D034100A8354D2D5B004180E848AE233D034100BC05922F5B0041802BF617283D034100448B6C325B00410079E9262C3D034100F6285C925B004100894160A23D034100E92631975B004180C64B37A73D03410054E3A5FB5B004180490C02F93D034100FF217D005C004180F38E53FC3D03410017D9CE055C00418016D9CEFE3D034100DD2406215C0041004E6210D43D034100DBF97E365C004180621058B23D034100DBF97E475C0041800C022BA03D034100787A258C5C00410063EEDA753D034100E5F29F105D0041807D1D38033F0341001B0DE02E5E0041802BF6972142034100371AC0F25F0041802BF6178B42034100371A404D610041807D1D38E540034100C9E5BFA7620041807D1DB88A3E034100371A4002640041004703787B3D034100E5F21FD56400410010E937123E034100000000B7640041807D1DB87B3F034100C9E5BF7A640041802BF617A941034100E5F29F5C640041806210D81245034100E5F29F4D650041004703F8F447034100371AC03E670041802BF617D7490341001B0DE0026A0041802BF6974F4A0341001B0D607B6B0041806210D88B4A034100E5F29F8A6C0041802BF6179B4C0341001B0DE0C56D0041004703F8F74E034100E5F29F796E0041807D1DB82550034100000000F86F0041807D1DB8F350034100E5F21FA0700041806210D8155103410000000038710041802BF697375103410000000042720041806210D81551034100000000B873004100470378AE500341000000804B740041807D1DB82550034100E5F29F35740041807D1DB8354F034100371A409174004100470378C64D0341000000006F760041806210D87E4D0341001B0D60427A0041802BF6174F4D034100371A40B67D0041807D1D38604C0341001B0D6023800041802BF617CA4A034100371AC048820041807D1DB8CD470341001B0D6067830041806210589045034100000080FD840041004703F89342034100E5F21FF386004100470378D83E034100000000B988004180621058193E034100E5F29F18890041807D1DB8673F034100E5F29F41880041802BF6970442034100000080AB860041806210D85944034100371AC07B860041004703F8C646034100E5F29F29880041807D1DB8CD470341001B0D6071880041807D1D38DB49034100E5F21F0E850041004703F8F74E034100000000F8830041802BF617994F0341001B0D6051830041807D1DB82550034100000000B8820041807D1D38A350034100E5F21F51820041806210D81551034100E5F29FE8810041806210D80552034100E5F21FBD810041004703F8F552034100E5F21FCA810041004703F8E553034100E5F29F7E810041004703F8D55403410000000038800041806210583955034100E5F21F707F0041004703F8D554034100000000F87E0041802BF697B654034100E5F21F357E0041004703F8D554034100000000B87D0041807D1D38ED540341001B0DE09B7C0041802BF617C655034100E5F29F927B0041802BF617B656034100000000387B0041807D1D380557034100E5F21F8B7A0041807D1D38A657034100000000F8790041802BF69749580341001B0D60DA780041806210588659034100000000B8780041004703F8045A034100000000B8780041802BF617235B034100C9E5BFDB78004100470378565C034100C9E5BF3479004100470378465D0341001B0D60E6790041802BF697365E034100371AC07A7A0041802BF697265F034100E5F21FF07A00410010E9B71660034100000000387B0041807D1D389460034100000000957B0041807D1DB80661034100000000387B0041802BF617F661034100000000467B0041806210D8E662034100E5F29F557B0041806210D8D6630341001B0D60597B0041004703F8C664034100000000F8790041004703780D65034100000000B8780041802BF6970A640341001B0DE04A780041004703F8C664034100000080A9780041004703F8B6650341001B0DE09A790041802BF617A766034100000000387B0041807D1DB81167034100000000787C004180621058AA66034100000000B87D0041004703F8C865034100371A404A7E0041004703F8C664034100E5F29F0F7F0041806210D8D663034100000080647F0041806210D8E66203410000000038800041807D1DB829620341000000006E810041807D1DB80661034100000000B8820041806210587F60034100000000F8830041802BF697C8600341000000005F840041807D1DB8066103410000000006850041806210D8F6610341001B0D6034850041806210D8E6620341001B0DE020850041806210D8D663034100E5F21FCB840041004703F8C664034100E5F29F63840041004703F8B665034100000080FA830041802BF617A766034100000000A1830041802BF617976703410000008057830041807D1D388768034100371A4039830041807D1D387769034100E5F29F2283004180621058676A034100E5F21F2683004180621058576B0341000000803D83004180621058476C034100371AC05883004100470378376D034100E5F21F7D83004100470378276E034100C9E5BFB5830041802BF697176F034100000000F8830041802BF697BF6F0341001B0D6077840041807D1DB8F770034100E5F29FC68400410010E9B7E77103410000000038850041004703F86972034100E5F29FA2850041806210D8D772034100E5F29FFB850041806210D8C773034100371AC0AF850041806210D8B77403410000000038850041004703F8ED74034100000000F8830041802BF617EF74034100000000B882004100470378F57403410000000078810041806210D877750341001B0DE0AC800041004703F8977603410000000079800041802BF617887703410000000045800041802BF6177878034100000080DA7F0041807D1D386879034100000000F87E004180621058E779034100000000B87D0041004703F8277A034100000080A97C0041807D1D38587A034100000000387B004180621058B97A0341001B0D603F7A0041807D1D38487B0341000000009779004180621058387C0341001B0D602679004180621058287D0341001B0D600179004100470378187E0341001B0D605279004100470378087F034100000080E6790041802BF697F87F0341000000004F7A0041802BF697E880034100000000A87A0041807D1DB8D8810341001B0DE0047B0041807D1DB8C882034100000000387B0041004703F89B83034100E5F29F487B0041806210D8A8840341001B0D603B7B0041806210D89885034100E5F29F207B0041004703F88886034100E5F29FE67A0041004703F87887034100000000B97A0041802BF6176988034100371A404E7A0041802BF6175989034100000000B8780041806210D88F89034100371A4047780041802BF617598903410000000078770041802BF697028903410000000038760041802BF6177C88034100000000F8740041802BF617E48703410000000005740041004703F878870341000000007872004180621058EE8603410000000038710041802BF617C786034100000000F86F0041802BF697DB86034100000000B86E0041802BF6975687034100E5F29FC76D0041802BF6176988034100E5F21F816D0041802BF6175989034100E5F21F966D0041802BF617498A034100000000E66D00410010E937398B0341001B0DE0B66E0041807D1D38298C034100371A40F56F004180621058198D0341000000000571004180621058098E034100371A40F471004100470378F98E03410000000078720041004703F8B78F034100C9E5BF5B730041802BF697D990034100000000B8730041802BF6177B910341001B0D6049740041802BF697B992034100E5F29F91740041807D1DB8A9930341001B0D60E5740041807D1DB89994034100371A403F750041806210D8899503410000000097750041806210D87996034100C9E5BFE8750041004703F8699703410000000038760041807D1DB82298034100371AC0A8760041802BF6174A99034100371A4000770041802BF6173A9A034100371A4052770041802BF6172A9B034100000000AC770041807D1D381A9C03410000000006780041807D1D380A9D034100E5F21F4178004180621058FA9D0341001B0DE0AA78004180621058EA9E034100371A40FC78004100470378DA9F0341001B0D602379004100470378CAA00341000000009379004100470378BAA1034100371A40C4790041802BF697AAA2034100000000F8790041806210D829A3034100000080357A0041802BF6979AA3034100371A40BD7A0041807D1DB88AA40341001B0DE0FD7A0041807D1DB87AA5034100000000387B00418062105809A6034100E5F21F427C0041806210D85AA7034100000000B87D00418062105842A8034100000000F87E004100470378ACA803410000000038800041802BF617DDA803410000000078810041004703F801A9034100E5F21F8B820041004703F83AA9034100000000F8830041807D1D38D3A8034100E5F21FF0840041004703F84AA8034100371AC0BF850041806210D85AA703410000000004860041806210D86AA603410000008029860041807D1DB87AA50341001B0DE049860041807D1DB88AA403410000008031860041802BF6979AA3034100371A4014860041802BF697AAA2034100E5F21FF885004100470378BAA1034100000080D285004100470378CAA0034100E5F29FB585004100470378DA9F034100E5F21F9585004180621058EA9E034100371A408885004180621058FA9D034100371A408E850041807D1D380A9D034100E5F29F89850041807D1D381A9C034100E5F29F69850041802BF6172A9B03410000008040850041802BF6173A9A03410000000042860041802BF6174A99034100000000B8870041802BF617EC990341001B0DE048880041802BF6173A9A034100000000F888004100470378B79A03410000000030890041802BF6172A9B0341001B0DE0088A0041807D1D381A9C0341001B0D60BB8A0041807D1D380A9D034100000000788B0041806210D8E29D034100000000378C004180621058EA9E034100000000B88C0041807D1DB87D9F034100000080058D004100470378DA9F034100371A40C78D004100470378CAA0034100E5F21F718E004100470378BAA1034100000000018F0041802BF697AAA2034100000080738F0041802BF6979AA303410000008000900041807D1DB88AA403410000000078900041802BF6175DA5034100371A403A910041806210D86AA6034100000000B8910041802BF69721A70341001B0DE0B6920041004703F84AA8034100E5F29FC5930041004703F83AA90341000000003894004100470378CCA9034100E5F21FBC940041004703F82AAA03410000000078950041802BF697B8AA0341000000002C960041802BF6171BAB034100000000B8960041004703F862AB034100000000F8970041806210D8DAAB0341000000807F980041802BF6170BAC034100000000389900418062105847AC034100000000789A0041004703F85AAC034100000000B89B00410010E9B785AC034100000000F89C0041004703F861AC034100000000389E0041004703F837AC034100000000789F0041802BF61712AC034100000000B8A00041802BF697F6AB034100000080ECA10041802BF6170BAC03410000000038A30041806210D848AC034100371A40D1A30041802BF6170BAC03410000000078A40041807D1D38D8AB03410000000091A50041802BF6171BAB034100000000B8A50041004703F8AAAA034100000000DCA50041004703F82AAA034100371A401AA60041004703F83AA90341001B0DE0E1A50041004703F84AA8034100C9E5BF3EA50041806210D85AA7034100371A408EA40041806210D86AA60341001B0DE0B8A30041807D1DB87AA503410000000038A30041807D1D38F1A4034100E5F29FCEA20041807D1DB88AA4034100000000F8A10041802BF617E3A30341001B0DE084A10041802BF6979AA3034100000000B8A00041806210D8F9A2034100371A4055A00041802BF697AAA2034100000000789F0041802BF697F6A1034100000000389E004100470378E8A0034100000000F89C0041806210D8DC9F034100E5F29FD09B004180621058EA9E034100E5F21FB89A004180621058FA9D034100371AC099990041807D1D380A9D0341000000003899004100470378BD9C034100371A4042980041807D1D381A9C034100371A401C970041802BF6172A9B034100000000B8960041802BF697CB9A03410000000029960041802BF6173A9A03410000000078950041806210585C99034100371A40E1940041004703F85998034100371AC072940041004703F8699703410000000038940041802BF6178996034100E5F29F00940041806210D88995034100000000F1930041807D1DB8999403410000000038940041802BF617C593034100C9E53F2F950041802BF697B992034100000000B8960041807D1D384192034100000000F897004100470378469203410000000038990041004703786592034100000000789A004100470378B892034100000000B89B0041806210581693034100000000F89C0041806210586E93034100000000389E0041806210D88593034100000000789F0041802BF697E892034100000000ACA00041802BF697C99103410000000015A10041802BF697D99003410000008052A1004100470378E98F0341000000803CA1004100470378F98E03410000008013A1004180621058098E034100000080E3A0004180621058198D034100000000B8A00041802BF617888C0341001B0DE05EA000410010E937398B034100371A4026A00041802BF617498A0341001B0D600CA00041802BF61759890341001B0DE011A00041802BF6176988034100371A4036A00041004703F878870341001B0DE057A00041004703F88886034100371AC084A00041806210D898850341001B0DE0A0A00041806210D8A884034100371A40B5A00041807D1DB8B8830341001B0DE0A4A00041807D1DB8C882034100371AC077A00041807D1DB8D8810341001B0DE031A00041802BF697E8800341001B0DE0C99F0041802BF697F87F034100000000789F0041802BF6174B7F034100000000E29E004100470378187E034100E5F21F4F9E004180621058287D034100000000AD9D004180621058387C034100371AC0259D0041807D1D38487B034100000000F89C0041802BF617937A034100000000389E0041802BF697A379034100000000789F0041807D1DB8E979034100000000B8A0004100470378427A034100000000F8A10041806210D8917A03410000000038A30041802BF697FF7A034100E5F21FBBA30041807D1D38487B03410000000078A40041004703F8987B034100000000B8A50041802BF697F27B034100000000F8A6004180621058CE7B03410000000038A8004180621058767B034100371AC0D3A80041807D1D38487B0341000000005EA90041807D1D38587A034100000000B8AA0041802BF617E479034100000080CDAB0041807D1D3868790341001B0DE0FAAB0041807D1D38587A03410000000038AD00410010E937697A03410000000078AE0041807D1D38737A034100000000B8AF0041004703F8B07A034100000000F8B00041004703F8BB7A03410000000038B2004180621058A77A03410000000078B30041806210D8867A034100000000B8B40041802BF6978B7A034100000000F8B50041806210D8CB7A03410000000038B70041802BF697197B03410000000078B80041802BF697FA7B034100E5F21F3BB9004180621058287D034100E5F21F5DB9004100470378187E0341001B0D6009B9004100470378087F03410000000078B80041802BF617C37F0341000000003BB70041802BF697E880034100E5F29F1CB60041807D1DB8D88103410000008028B50041807D1DB8C882034100000000B8B40041802BF6975B83034100000000BDB30041806210D8A884034100E5F29F2AB30041806210D89885034100E5F29FB5B20041004703F888860341001B0D605EB20041004703F878870341001B0DE042B20041802BF61769880341001B0DE012B20041802BF6175989034100E5F21F0CB20041802BF617498A034100C9E53F1DB20041807D1D38398B03410000000038B20041802BF617EF8B0341000000008DB2004180621058198D0341001B0DE004B3004180621058098E03410000000078B30041802BF697AC8E034100E5F29FD1B3004100470378F98E034100000000B8B40041807D1DB8AA8F034100C9E53F36B5004100470378E98F034100000000F8B50041806210585B900341001B0D60C3B60041802BF697D99003410000000038B70041004703F815910341001B0D6045B80041802BF697C9910341000000805AB90041802BF697B992034100000000B8B90041802BF61709930341001B0D60A8BA0041807D1DB8A99303410000000038BC0041806210D8849403410000000078BD0041802BF6177D95034100000000B8BE0041004703F86196034100000000E4BF0041004703F8699703410000000038C10041802BF697559803410000000078C200410047037831990341001B0D602AC30041802BF6174A99034100000000B8C30041807D1DB85799034100371AC054C40041802BF6174A99034100000000F8C4004180621058349903410000000038C60041807D1DB86898034100E5F21FB4C60041004703F869970341001B0D60ACC60041806210D879960341001B0D6092C60041806210D88995034100E5F29F77C60041807D1DB89994034100E5F29F49C60041807D1DB8A993034100371AC01DC60041802BF697B9920341001B0D600EC60041802BF697C9910341001B0D60ABC60041802BF697D99003410000000078C700410010E9B74790034100000000B8C80041004703F8F68F034100000000F8C90041004703F80C8F0341000000001CCA004180621058098E034100000000FEC9004180621058198D034100371AC071C90041807D1D38298C034100000000B8C80041802BF6178F8B034100E5F29F5DC80041807D1D38398B03410000000078C70041802BF697748A034100000080D2C60041802BF61759890341001B0DE0EDC60041802BF617698803410000000078C70041806210D8E087034100000000B8C80041802BF6178587034100000000F8C90041802BF697988703410000000038CB0041004703789B87034100000000EACB0041004703F8788703410000000078CC0041806210585987034100E5F21F67CD0041004703F88886034100000000B8CD0041802BF617F185034100371A4003CE0041806210D8A8840341001B0DE042CE0041807D1DB8B883034100000000F8CE0041807D1DB8C88203410000000038D00041004703F85F8203410000000078D10041802BF697C082034100000000B8D20041802BF6177A83034100E5F21FEBD30041806210D8A884034100E5F29FD0D40041806210D8988503410000000038D50041807D1DB8D48503410000000078D60041004703F86A86034100E5F21FF8D60041004703F88886034100000000B8D7004180621058B186034100000000F8D800410010E9B7BB86034100E5F29FC8D90041004703F8888603410000000078DB00410010E9B7C185034100E5F29FB6DC0041806210D8A884034100E5F21F72DD0041807D1DB8B883034100000000F8DD0041004703F815830341000000809CDE0041807D1DB8D881034100E5F29FD8DE0041802BF697E8800341001B0DE00CDF0041802BF697F87F03410000000038DF00410010E937377F034100371A4055DF004100470378187E0341001B0D6056DF004180621058287D034100371AC042DF004180621058387C034100371AC0B4DE0041807D1D38487B0341001B0D6078DE0041807D1D38587A034100371AC045DE0041807D1D3868790341001B0D6017DE0041802BF61778780341001B0DE081DE0041802BF617887703410000000038DF0041802BF697B47603410000000078E00041004703787A76034100C9E53F52E10041004703F8977603410000000099E20041802BF6178877034100000000F8E2004100470378F97703410000000041E30041802BF6177878034100000000A7E30041807D1D386879034100C9E5BF01E40041807D1D38587A03410000000038E40041802BF697C27A0341001B0DE076E40041807D1D38487B034100E5F29FE2E4004180621058387C03410000008067E5004180621058287D034100E5F21FE9E5004100470378187E034100E5F29F79E6004100470378087F034100000000B8E60041802BF617767F0341001B0DE0F7E60041802BF697F87F03410000000058E70041802BF697E880034100000080AFE70041807D1DB8D8810341001B0DE0BAE70041807D1DB8C8820341001B0DE0AAE70041807D1DB8B8830341001B0D6098E70041806210D8A8840341001B0DE06FE70041806210D89885034100E5F29F4CE70041004703F88886034100C9E53FF3E60041004703F878870341001B0DE08FE70041802BF6176988034100000000F8E70041004703783D89034100000000F8E70041806210D8E089034100E5F21F3DE70041807D1D38398B034100000000B8E60041807D1D38698B03410000000078E50041802BF617D78B034100E5F21FF8E40041807D1D38298C03410000000038E40041807D1DB8AC8C034100E5F29FCEE3004180621058198D034100000000F8E2004100470378F48D0341001B0DE068E2004100470378F98E034100000000B8E1004180621058E98F0341001B0D60A8E00041802BF697D990034100371A40BBDF0041802BF697C99103410000000038DF0041004703F82C92034100E5F29F89DE0041802BF697B992034100000000F8DD00410010E9371393034100371A402EDD0041807D1DB8A993034100000000B8DC0041806210582D9403410000000074DC0041807D1DB899940341000000805EDC0041806210D8899503410000008080DC0041806210D87996034100000000B8DC0041004703F81897034100371AC054DD0041004703F85998034100371AC0D4DD0041802BF6174A9903410000008098DE0041802BF6173A9A034100E5F21FFEDE0041802BF6172A9B03410000000038DF0041004703F8F69B0341000000008DDF0041807D1D380A9D03410000000038DF0041004703F8979D034100000000F8DD004100470378A79D034100000000B8DC0041806210D85B9D034100371A4015DC0041807D1D380A9D03410000000078DB0041802BF697DB9C03410000000038DA004100470378879C034100000000F8D80041802BF697819C034100000000B8D700410010E937DF9C034100E5F29F37D70041807D1D380A9D03410000000078D60041802BF617679D03410000000094D5004180621058FA9D03410000000077D4004180621058EA9E034100000000F8D30041807D1DB83E9F0341000000006CD3004100470378DA9F034100E5F29F18D3004100470378CAA00341001B0D606BD3004100470378BAA1034100000000B7D30041802BF697AAA2034100371A40EAD30041802BF6979AA3034100371AC020D40041807D1DB88AA4034100E5F29F6DD40041807D1DB87AA50341001B0D609CD40041806210D86AA6034100000080F4D40041806210D85AA70341001B0DE035D50041004703F84AA8034100371AC07FD50041004703F83AA9034100000000C5D50041004703F82AAA034100E5F29F39D60041802BF6171BAB03410000000078D60041806210D888AB034100000080BED60041802BF6170BAC0341001B0D608ED70041807D1D38FBAC034100000000F8D80041807D1DB8DAAD03410000000038DA004180621058BEAE03410000000078DB0041807D1D3826AE0341001B0D6021DC0041807D1D38FBAC034100000000B8DC00418062105864AC03410000000089DD0041802BF6171BAB034100000000F8DD00418062105896AA0341000000803EDE0041004703F82AAA0341001B0D60D0DE0041004703F83AA903410000000038DF004180621058BAA8034100E5F29FEDDF0041004703F84AA803410000000078E00041807D1DB805A80341001B0D605FE10041004703F84AA80341000000003EE20041004703F83AA90341000000805FE20041004703F82AAA034100E5F21F5EE20041802BF6171BAB0341000000003EE20041802BF6170BAC034100371AC00CE20041807D1D38FBAC034100000080E5E10041807D1D38EBAD034100000000B8E1004100470378A7AE0341001B0D604BE1004180621058CBAF034100E5F21F0FE1004180621058BBB0034100E5F29FCFE0004100470378ABB103410000000093E00041004703789BB2034100E5F29F60E00041802BF6978BB3034100371AC034E00041802BF6977BB4034100C9E5BF09E000410010E9B76BB5034100C9E5BFE3DF00410010E9B75BB6034100E5F29FBCDF0041806210D84BB7034100E5F29F91DF0041806210D83BB8034100E5F21F61DF0041806210D82BB9034100E5F21F47DF0041004703F81BBA0341001B0D605BDF0041004703F80BBB034100E5F29F78DF0041802BF617FCBB03410000008094DF0041802BF617ECBC0341001B0D60B5DF0041807D1D38DCBD034100E5F21FE4DF0041807D1D38CCBE03410000008040E0004180621058BCBF03410000000078E000410047037840C00341001B0D60BDE0004180621058ACC0034100E5F21F6DE10041806210589CC10341000000009CE20041004703788CC203410000000038E40041802BF6176CC303410000000078E500410010E937D6C3034100000000B8E60041004703F838C4034100000000F8E70041807D1D3856C403410000000038E90041004703F85DC403410000000078EA0041802BF61702C403410000008086EB0041004703787CC3034100000000F8EC0041807D1DB894C203410000000038EE0041807D1DB8AAC1034100371AC0DAEE004180621058ACC00341000000805AEF004180621058BCBF034100E5F21FABEF0041807D1D38CCBE034100000000FCEF0041807D1D38DCBD034100E5F29F33F00041802BF617ECBC0341001B0D6068F00041802BF617FCBB034100371AC09FF00041004703F80BBB034100E5F21FDAF00041004703F81BBA034100371AC01DF10041806210D82BB903410000008077F10041806210D83BB8034100000000F8F10041802BF69768B7034100000000B9F200410010E9B75BB603410000000038F30041806210588AB5034100371A40ABF30041802BF6977BB4034100E5F21FFBF30041802BF6978BB3034100371AC03FF40041004703789BB203410000000078F40041802BF697F0B1034100000080E7F4004180621058BBB0034100E5F29F6EF5004180621058CBAF034100000000B8F50041806210D82DAF034100000000F8F60041802BF697B5AF0341000000008CF7004180621058BBB0034100E5F21F17F8004100470378ABB10341001B0D607CF80041004703789BB2034100371AC0D4F80041802BF6978BB30341001B0D6029F90041802BF6977BB403410000008069F900410010E9B76BB503410000000078F90041004703F82AB60341001B0D6059F90041806210D84BB7034100371A401AF90041806210D83BB8034100371AC0B3F80041806210D82BB903410000000038F80041802BF69703BA034100E5F21FBDF70041004703F80BBB034100E5F29F5FF70041802BF617FCBB03410000000012F70041802BF617ECBC034100000000F8F60041807D1D3869BD0341001B0DE0C1F60041807D1D38CCBE034100E5F29F8BF6004180621058BCBF034100E5F21F77F6004180621058ACC00341001B0DE03FF60041806210589CC1034100000000FBF50041004703788CC2034100000000B8F50041802BF6177AC30341001B0DE057F50041802BF6976CC4034100000080E2F40041802BF6975CC503410000000078F40041806210D8F7C5034100000080ADF30041807D1DB83CC703410000000038F3004180621058F9C70341001B0DE08DF20041806210D81CC9034100C9E53F8AF20041806210D80CCA03410000000038F30041807D1DB850CA0341001B0DE054F40041004703F8FCCA0341001B0DE04EF50041004703F8ECCB034100000000B8F50041004703F845CC034100000080DDF60041802BF617DDCC03410000000038F80041807D1DB8C0CD034100E5F29F17F900410010E937BDCE03410000000078F90041806210D84ACF0341001B0D60BAF900410010E937ADCF0341001B0DE057FA0041807D1D389DD0034100000000B8FA0041004703F830D1034100371AC00BFB0041806210588DD1034100000000F8FB0041004703785CD203410000000038FD0041806210D8D1D203410000000078FE00410010E937F9D2034100000000B8FF0041802BF697C0D2034100371AC0E3000141806210587DD2034100E5F21F67010141806210587DD203410000000038020141806210D885D20341000000007803014180621058E3D203410000000063040141004703786DD3034100000000F8050141802BF61753D40341000000003807014180621058FDD40341000000007808014100470378A3D4034100000000B1090141004703786DD3034100371A400A0A0141806210587DD20341000000800F0A0141806210588DD1034100000080F8090141807D1D389DD00341001B0DE0D10901410010E937ADCF0341000000808C0901410010E937BDCE0341001B0DE07B090141802BF617CDCD03410000008032090141802BF617DDCC034100371A40F3080141004703F8ECCB0341001B0D60BC080141004703F8FCCA03410000000081080141806210D80CCA034100371A403E080141806210D81CC90341001B0D60F3070141807D1DB82CC8034100E5F21F93070141807D1DB83CC70341000000003807014180621058A3C60341001B0DE0BC060141807D1DB84CC6034100000000F805014100470378A6C50341000000005A050141802BF6976CC4034100000000C9040141004703787CC30341001B0DE064040141004703788CC20341000000801D040141806210589CC10341001B0DE01C04014180621058ACC0034100000000B80401410047037808C0034100E5F29FE205014180621058BCBF03410000000038070141807D1D3803C0034100000000780801410047037846C0034100000000B8090141802BF6176DC0034100000000F80A0141802BF6178FC0034100000000380C0141802BF69765C0034100000000780D0141806210D886C0034100E5F29F5E0E0141806210589CC1034100E5F21F780E0141004703788CC2034100371A406D0E0141004703787CC3034100371AC0610E0141802BF6976CC4034100371AC0A30E0141802BF6975CC5034100E5F21FD90E0141807D1DB84CC6034100000080AC0F0141807D1DB83CC703410000000038110141004703F8FEC70341001B0D6063120141806210D81CC903410000000005130141806210D80CCA034100000000B813014100470378B1CA03410000000069140141004703F8ECCB0341001B0D60E8140141802BF617DDCC034100371A4024150141802BF617CDCD0341001B0DE0281501410010E937BDCE034100E5F29F971501410010E937ADCF034100E5F29FE7150141807D1D389DD003410000000038160141806210D839D10341001B0DE0D3160141806210587DD2034100E5F21F34170141004703786DD303410000000078170141004703F819D4034100000000D2170141802BF6974DD5034100371A400E180141802BF6973DD6034100371AC047180141802BF6972DD7034100371AC075180141807D1DB81DD8034100000000B8180141807D1D3887D80341001B0D60FE180141807D1DB80DD9034100000000B8180141806210D8BED9034100000000B8180141807D1DB84ADA034100371A4022190141806210D8EDDA034100000000F8190141802BF617B8DB034100000000381B0141807D1DB8A7DC034100000000781C0141004703F8CDDC034100000000B81D0141802BF6977FDD0341001B0D60201E0141802BF617BEDD034100000000F81E0141806210584DDE034100371A406B1F0141802BF617AEDE03410000000038200141806210D85ADF034100371AC074210141807D1D388EE0034100E5F29F492201410010E9377EE1034100000000B8220141802BF617E4E103410000000069230141806210586EE2034100000000F82301410010E937C5E203410000000038250141807D1DB82CE303410000000078260141807D1D3859E30341001B0DE0F4260141806210585EE3034100000000B82701410047037865E3034100000000F8280141806210D866E3034100000000382A01410010E9B785E3034100000000782B0141004703F8A9E3034100000000B82C0141807D1D38C8E3034100000000F82D0141806210D887E30341001B0D60F42E0141806210586EE2034100000000382F0141004703F8A5E1034100371A40C92F0141807D1D388EE0034100371AC05E2F0141802BF6179EDF0341001B0DE0E72E0141004703F8F0DE034100000000F82D0141806210D8BFDD034100371AC0602D0141004703F8CDDC034100000000D62C0141004703F8DDDB034100000080642C0141806210D8EDDA034100000000352C0141806210D8FDD90341000000006B2C0141807D1DB80DD9034100000000B82C01410047037888D8034100C9E5BF082D0141807D1DB81DD8034100000000F82D0141802BF61768D7034100E5F21F6B2E0141802BF6972DD7034100000000382F014100470378CAD60341001B0D604E300141802BF6973DD60341001B0D605B310141802BF6974DD5034100000000B8310141004703789FD40341001B0D6023320141004703786DD3034100C9E53F4C320141806210587DD20341001B0DE071320141806210588DD10341001B0DE0CE320141807D1D389DD0034100000080BA3301410010E937ADCF03410000000038340141802BF69765CF03410000000078350141806210D82ACF034100000000B8360141004703786ECF034100371A40373701410010E937ADCF034100000000F83701410047037818D0034100371A40AE380141807D1D389DD003410000000038390141004703781BD1034100E5F21FA9390141806210588DD1034100371AC06F3A0141806210587DD2034100E5F29F1F3B0141004703786DD30341001B0DE0A73B0141004703785DD4034100371AC0493C0141802BF6974DD5034100371A407F3C0141802BF6973DD6034100E5F21F9F3C0141802BF6972DD7034100000000BC3C0141807D1DB81DD8034100E5F21FD93C0141807D1DB80DD9034100000000DF3C0141806210D8FDD9034100371A40293D0141806210D8EDDA034100E5F29F843D0141004703F8DDDB0341001B0D60053E0141004703F8CDDC034100C9E5BF143F0141802BF617BEDD034100000000783F0141806210D81BDE034100000000B84001410047037858DE034100E5F29F22410141802BF617BEDD03410000000050410141004703F8CDDC034100371AC009410141004703F8DDDB034100000000B8400141802BF6174BDB034100E5F29F4E400141806210D8EDDA034100E5F29FDF3F0141806210D8FDD9034100E5F29FD23F0141807D1DB80DD9034100371A4063400141807D1DB81DD8034100E5F29FFC400141807D1DB81DD8034100000000F8410141802BF69751D803410000000023430141807D1DB80DD903410000000078440141004703F832D9034100000000B8450141802BF6174ED8034100E5F29F6B460141802BF6972DD7034100000000F8460141802BF69782D603410000000038480141806210D863D503410000000078490141802BF617BFD4034100000000B84A014100470378E1D4034100C9E53FCD4B0141802BF6974DD5034100000000D84C0141802BF6973DD6034100000000384D0141806210D8B7D6034100E5F21FD74D0141802BF6972DD7034100000080364E0141807D1DB81DD80341001B0DE0424E0141807D1DB80DD9034100371AC0234E0141806210D8FDD9034100C9E53FE24D0141806210D8EDDA034100000000434D0141004703F8DDDB034100E5F21F294C0141004703F8CDDC034100E5F29F884B0141802BF617BEDD034100371AC0384B0141802BF617AEDE034100000080144B0141802BF6179EDF034100000000384B0141807D1D388EE00341001B0D60744B01410010E9377EE1034100E5F29FAF4B0141806210586EE2034100000000F84B01410010E93730E3034100E5F21FAF4C0141004703784EE4034100371A40094D0141004703783EE5034100000000CF4D0141004703782EE6034100000000784E0141807D1D38ABE6034100E5F29F104F0141802BF6971EE7034100000000B84F0141802BF617B2E7034100371AC031500141807D1DB8FEE8034100371AC05B500141807D1DB8EEE9034100371A4094500141806210D8DEEA034100371AC0DC500141806210D8CEEB03410000008029510141004703F8BEEC0341001B0D607D510141004703F8AEED034100000000CD510141004703F89EEE0341000000002F520141802BF6178FEF0341001B0DE089520141802BF6177FF0034100371AC0DD520141807D1D386FF10341001B0DE04C530141807D1D385FF2034100000000B3530141806210584FF3034100E5F21FF6530141806210583FF4034100371AC06C540141004703782FF5034100000000B8540141004703F8B7F5034100000000F8540141004703781FF60341000000008B550141004703780FF7034100000000F8550141004703F89CF703410000000064560141802BF697FFF703410000000038570141802BF697D0F803410000000078580141806210D82BF9034100E5F21FE5580141802BF697EFF8034100000000B85901418062105852F8034100000000405A0141004703780FF7034100000000F85A0141802BF69730F60341001B0DE0D15B0141004703781FF6034100000000785D0141807D1DB8ABF6034100371AC0725E0141004703780FF7034100000000F85F014180621058D5F70341001B0D6066600141802BF697FFF703410000000038610141806210D85CF803410000000078620141806210D868F8034100000000B8630141807D1DB88FF8034100000000F8640141802BF69714F80341001B0D6006660141004703780FF7034100371A40B7660141004703781FF6034100E5F21F1A670141004703782FF50341001B0DE042670141806210583FF4034100E5F29F3F670141806210584FF30341001B0DE040670141807D1D385FF203410000008043670141807D1D386FF1034100E5F21F4A670141802BF6177FF00341001B0DE070670141802BF6178FEF0341001B0DE0C6670141004703F89EEE0341001B0DE046680141004703F8AEED034100000000B86801418062105828ED03410000000056690141004703F8BEEC034100000000F86901410047037860EC034100000000386B0141807D1DB8EDEB034100000000786C01410010E93766EB034100E5F21F596D0141806210D8DEEA034100000000B86D01410047037871EA034100000000066E0141807D1DB8EEE9034100000000226E0141807D1DB8FEE80341001B0DE0186E0141802BF6970EE8034100E5F21FCA6D0141802BF6971EE70341001B0D60516D0141004703782EE6034100371AC09C6C0141004703783EE50341001B0DE08F6B0141004703784EE4034100000000366A0141806210585EE3034100000000E0680141806210586EE2034100000080C96701410010E9377EE1034100E5F21F96660141807D1D388EE00341001B0D6026650141802BF6179EDF034100000000B86301410047037800DF03410000000078620141807D1DB8F9DE034100000000386101418062105863DF034100371AC01B600141807D1D388EE0034100000080475F01410010E9377EE1034100000000B85E0141806210D836E2034100000000785D01410047037832E3034100000000385C0141806210D8CAE30341000000006C5B0141004703784EE4034100000000F85A0141802BF6178BE4034100000000B8590141004703F8DEE40341000000007858014100470378FAE403410000000038570141802BF617B1E40341001B0D60B8560141004703784EE4034100000000F8550141807D1DB8D0E3034100000080B2550141806210585EE3034100E5F29F42550141806210586EE20341001B0DE0F55401410010E9377EE10341001B0DE005550141807D1D388EE0034100000000A1550141802BF6179EDF034100371A40EE560141802BF617F0DE034100371AC07E570141802BF617AEDE03410000000078580141802BF617D4DD034100E5F29FA4580141004703F8CDDC0341001B0D6022580141004703F8DDDB034100371AC095570141806210D8EDDA03410000000038570141004703F88CDA034100371AC067560141806210D8FDD9034100000000F8550141004703F8B3D903410000000004550141807D1DB80DD90341001B0D60EE530141807D1DB81DD803410000000078530141004703F8C1D70341001B0D60BF520141802BF6972DD703410000000038520141004703F8A1D6034100C9E53FAE510141802BF6973DD6034100000000F8500141004703F8B2D503410000000063500141802BF6974DD5034100000000B84F0141802BF69703D5034100000000784E0141004703F8A5D4034100371A40134E0141004703785DD4034100000000384D0141806210D8E2D30341001B0DE0AB4C0141004703786DD3034100000000F84B0141807D1D389AD2034100371A40734B0141806210588DD1034100E5F21F5E4B0141807D1D389DD0034100E5F29F5E4B01410010E937ADCF0341001B0D60484B01410010E937BDCE034100000000284B0141802BF617CDCD034100000000B84A0141807D1D38F9CC0341001B0D60CF490141004703F8ECCB0341000000005B480141004703F8FCCA034100000000F8460141802BF6973FCA034100000000B845014180621058A6C903410000000078440141802BF6172BC903410000000038430141004703F8C7C8034100000000F8410141802BF61764C8034100E5F21F75410141807D1DB82CC8034100000000B8400141806210D8DFC7034100371AC0B43F0141807D1DB83CC70341001B0D60853E0141807D1DB84CC6034100E5F29F693D0141802BF6975CC5034100000000F83C01410047037803C5034100C9E5BFF83B0141802BF6976CC4034100000000783A014100470378A3C30341000000003839014180621058F3C2034100371A4093380141004703788CC2034100000000F8370141807D1DB833C2034100E5F21FD2360141806210589CC103410000000078350141807D1DB81AC103410000000038340141806210D8D6C0034100371A400E33014180621058ACC0034100000000B8310141802BF61784C003410000000078300141004703F869C0034100000000382F0141806210D849C0034100000000F82D01410047037826C0034100000000B82C014180621058FABF034100000000782B0141802BF697E5BF034100000000382A014180621058E7BF034100000000F8280141802BF697CCBF034100000000B8270141807D1DB80ABF034100000080F5260141807D1D38DCBD03410000000078260141807D1D3849BD034100C9E53F0F260141802BF617ECBC03410000000038250141807D1D3862BC034100371A407F240141802BF617FCBB034100000000F8230141802BF697BABB034100000000B8220141004703783EBB034100E5F29F7F210141004703F81BBA034100371A401D210141806210D82BB90341000000000C210141806210D83BB80341001B0D6046210141806210D84BB7034100000000782101410010E9B7DDB6034100E5F29FC12101410010E9B75BB60341001B0DE0382201410010E9B76BB5034100000000B8220141004703F899B4034100000080CF230141802BF6978BB3034100371AC0A5240141004703789BB20341001B0DE01625014100470378ABB1034100E5F29F6225014180621058BBB0034100E5F21FB525014180621058CBAF0341001B0DE0AB25014180621058DBAE03410000000095250141807D1D38EBAD0341001B0D6058250141807D1D38FBAC0341001B0DE01B250141802BF6170BAC0341001B0D60DA240141802BF6171BAB034100E5F21F95240141004703F82AAA0341000000003D240141004703F83AA9034100000000F8230141807D1D3885A8034100371A4071230141806210D85AA7034100E5F29FFF220141806210D86AA6034100000000B822014180621058F2A5034100C9E5BF44220141807D1DB87AA5034100371AC07A210141807D1DB88AA4034100371AC090200141802BF6979AA303410000000038200141806210583AA30341001B0DE06D1F0141802BF697AAA2034100000000F81E0141806210D880A2034100000000B81D0141806210588EA20341001B0DE0231D0141802BF6979AA3034100371A40E81C0141807D1DB88AA4034100000000091D0141807D1DB87AA5034100000000341D0141806210D86AA6034100371AC0701D0141806210D85AA7034100E5F21F8F1D0141004703F84AA80341000000805E1D0141004703F83AA9034100371AC0C31C0141004703F82AAA034100000000381B0141802BF617DAAA034100000000F8190141807D1DB8F1AA034100000000B8180141807D1D38F3AA03410000000078170141807D1DB8E2AA0341000000003816014100470378C0AA034100000000F8140141802BF61799AA034100000000B81301418062105870AA03410000000078120141807D1DB834AA0341000000003811014100470378F1A9034100000000F80F0141802BF697A4A9034100000000B80E0141004703F84EA9034100000000780D0141802BF617EAA8034100000000380C0141807D1DB884A8034100371AC0AB0B0141004703F84AA8034100000000F80A0141807D1D38CCA70341000000801A0A0141806210D85AA7034100000000D2090141806210D86AA6034100E5F21F0C0A0141807D1DB87AA5034100000000F80A0141807D1D38E9A40341001B0DE06D0B0141807D1DB88AA4034100000000380C0141802BF617FAA30341001B0D60A60C0141802BF6979AA30341001B0DE0530D0141802BF697AAA20341001B0D60A70D014100470378BAA1034100371A40F20D014100470378CAA00341001B0DE02F0E014100470378DA9F034100000080670E014180621058EA9E034100E5F29F8C0E014180621058FA9D034100371AC09F0E0141807D1D380A9D034100E5F29FA40E0141807D1D381A9C034100000000980E0141802BF6172A9B034100E5F21F720E0141802BF6173A9A034100E5F29F3B0E0141802BF6174A99034100C9E5BFD90D0141004703F85998034100000080790D0141004703F86997034100000000170D0141806210D879960341001B0DE0B20C0141806210D88995034100E5F29F4B0C0141807D1DB899940341001B0DE0E80B0141807D1DB8A993034100000080790B0141802BF697B992034100371A401E0B0141802BF697C991034100371AC0520A0141802BF697D990034100000080280A014100470378E98F034100000000B8090141806210589C8F03410000000078080141004703F8B4900341001B0DE0EF070141802BF697D9900341000000003807014180621058F290034100000000F8050141004703789E91034100000000B8040141004703F8729203410000000078030141802BF69750930341001B0D601E030141807D1DB8A99303410000000038020141802BF6974394034100371AC0E7010141807D1DB89994034100000000F8000141802BF6975B95034100000000C7FF0041806210D87996034100E5F21FE1FE0041004703F8699703410000000078FE00410010E937D9970341001B0DE09CFD0041004703F8599803410000000038FD0041807D1D38A498034100000000F8FB0041802BF617E39803410000000044FB0041004703F85998034100E5F21FECFA0041004703F86997034100000000DBFA0041806210D87996034100371AC0DCFA0041806210D88995034100000000B8FA004100470378B39403410000000078F9004100470378EA930341001B0DE085F80041807D1DB8A993034100000000F8F60041802BF6972F93034100000000B8F5004180621058BA92034100E5F21F03F50041802BF697C991034100E5F29FB9F40041802BF697D990034100000000B8F50041802BF6172490034100000000F8F60041004703F8469003410000000038F80041802BF6177D9003410000000078F9004100470378569003410000008012FA004100470378E98F034100000000B8FA004100470378518F034100E5F21F80FB004180621058098E0341001B0DE0CDFB004180621058198D0341001B0D6078FB0041807D1D38298C034100000000B8FA0041807D1DB8578B0341000000808CF90041802BF617498A034100371A4005F90041802BF6175989034100E5F21F8EF80041802BF617698803410000000038F80041806210D88D87034100371A40EFF70041004703F88886034100E5F29F9DF70041806210D898850341001B0D6068F70041806210D8A88403410000000046F70041807D1DB8B883034100E5F29F4EF70041807D1DB8C8820341000000808DF70041807D1DB8D881034100000000D5F70041802BF697E880034100000000FBF70041802BF697F87F0341001B0DE0F9F7004100470378087F034100E5F21FEDF7004100470378187E034100E5F29FB2F7004180621058287D034100E5F29F85F7004180621058387C0341001B0D6069F70041807D1D38487B03410000008063F70041807D1D38587A034100371AC096F70041807D1D3868790341001B0D6004F80041802BF6177878034100C9E5BFD2F80041802BF617887703410000000078F90041004703F81077034100000000B8FA0041806210589B76034100000000F8FB0041802BF6975F7603410000000038FD0041806210583D7603410000000078FE0041802BF6972776034100000000B8FF0041806210581F76034100000000F8000141004703F8117603410000000038020141806210D8DD75034100E5F29FC5020141004703F8A77503410000000078030141807D1DB861750341001B0DE0AB040141806210D8B774034100000000F805014180621058F97303410000000038070141806210D8707303410000000078080141807D1D382173034100000000B8090141806210D8D772034100000000F80A0141807D1DB86772034100000080C80B01410010E9B7E771034100000000380C01410010E9378D71034100E5F21FD30C0141807D1DB8F770034100000000780D0141807D1D381F70034100371A40040E0141802BF697176F034100000000370E014100470378276E034100E5F29F5D0E014100470378376D034100E5F21F830E014180621058476C0341000000005F0E014180621058576B034100E5F21F140E014180621058676A034100E5F29FA30D0141807D1D3877690341001B0DE0290D0141807D1D388768034100371A40880C0141802BF6179767034100000000380C0141802BF6173867034100E5F29F8B0B0141802BF617A766034100000000F80A0141004703782F66034100000080050A0141004703F8B66503410000000078080141807D1DB82E6503410000000038070141802BF697F164034100000000F80501410010E9B70765034100000000B8040141802BF61733650341000000007803014180621058A06503410000000038020141004703F84466034100371AC067010141802BF617A766034100000000F8000141802BF697DC66034100000000B8FF0041802BF6174C6703410000000078FE0041802BF697766703410000000038FD0041807D1D386B67034100000000F8FB0041807D1DB86D67034100000000B8FA0041802BF6977767034100371AC01DFA0041802BF617976703410000000078F90041802BF697B26703410000000038F80041802BF617F567034100000000F8F60041802BF6176368034100000000B8F500410010E9B7016903410000000078F4004180621058756903410000000038F300410010E9379769034100E5F29FBFF20041807D1D387769034100000000F8F10041807D1DB83D690341000000002BF10041807D1D388768034100000000B8F000410047037808680341001B0DE072F00041802BF61797670341001B0D60FAEF0041802BF617A76603410000000078EF0041004703F8D665034100371AC0EAEE0041004703F8C6640341000000804CEE0041806210D8D6630341001B0DE0A7ED0041806210D8E662034100000000F8EC0041802BF6970662034100000080C2EB0041807D1DB8066103410000000078EA0041802BF6973C60034100371AC0F8E900410010E9B7166003410000000038E90041806210D8EF5F034100000000F8E70041004703F8E95F0341001B0DE02CE700410010E9B71660034100000000B8E60041806210583F60034100E5F21FACE50041807D1DB80661034100E5F21F36E50041806210D8F66103410000008007E50041806210D8E662034100371AC0FCE40041806210D8D663034100E5F29F24E50041004703F8C66403410000000069E50041004703F8B6650341001B0D60CDE50041802BF617A766034100371AC066E60041802BF6179767034100000000B8E60041004703F80E68034100E5F29F0EE70041807D1D388768034100000080A1E70041807D1D387769034100000000F8E7004180621058326A0341001B0DE027E8004180621058576B034100000000F8E70041807D1DB8C86B034100000000A2E7004180621058476C034100000000B8E60041802BF617CB6C03410000000078E5004100470378D36C03410000000038E40041802BF697866C03410000000055E3004180621058476C034100000000B8E10041802BF6970C6C03410000000078E00041802BF617336C03410000000038DF0041806210D8936C034100000000F8DD0041004703781C6D034100000000B8DC0041802BF697BD6D0341001B0D60FFDB004100470378276E03410000000078DB0041807D1DB8856E034100E5F29F78DA0041802BF697176F034100E5F29F08D90041802BF6970770034100000000B8D70041004703F8A070034100C9E5BF2DD700410010E9B7E77103410000000078D60041802BF617977203410000000055D60041806210D8C773034100C9E53F9FD50041806210D8B77403410000000038D50041004703F802750341001B0D60DDD40041004703F8A77503410000008082D40041004703F89776034100000000F8D30041806210D82D77034100C9E5BF9CD30041802BF6178877034100000000B8D20041802BF617007803410000000078D1004100470378667803410000000038D00041807D1DB86278034100000000F8CE0041807D1DB8B377034100371A4021CE0041004703F89776034100000000B8CD0041802BF6170276034100000000BACC0041806210D8B77403410000000078CC004180621058497403410000000038CB0041807D1DB8C873034100000000F8C90041802BF697B673034100000000B8C80041807D1D38C07303410000000078C70041004703F82C73034100000080FFC60041806210D8D77203410000000038C60041807D1D386872034100000000F8C40041806210580372034100000000B8C30041802BF617A17103410000000078C20041807D1D38037103410000000038C10041806210D85B700341001B0D608AC00041802BF6970770034100000000F8BF0041004703F8C06F034100000000B8BE0041807D1D383F6F03410000000078BD0041802BF617BA6E03410000000038BC0041807D1D38316E034100000000F8BA0041004703F84A6D03410000000072BA004180621058476C0341001B0DE05FBA004180621058576B034100371A40CDBA004180621058676A03410000000038BC0041802BF6177F6903410000000078BD00410010E9B71369034100000000B8BE0041004703F8C768034100000000F8BF0041806210D8896803410000000038C100410010E9370968034100371A40F4C10041802BF617976703410000000078C20041806210D80C67034100000000D1C20041802BF617A7660341001B0D6023C30041004703F8B665034100E5F21F2FC30041004703F8C664034100E5F21F2AC30041806210D8D663034100000080D3C20041806210D8E66203410000000078C20041802BF6975162034100371AC0BDC10041807D1DB8066103410000000038C10041802BF6976D60034100C9E53F20C00041802BF697265F0341000000004FBF0041802BF697365E034100000000B8BE004180621058AC5D034100371AC064BE004100470378465D0341000000007DBD004100470378565C03410000000038BC0041807D1D386F5B034100000000F8BA0041807D1D38E55A034100000000B8B90041807D1DB8D75A034100371A40EDB8004100470378665B03410000000078B80041004703F8D15B03410000000038B700410010E937FA5B034100000000F8B50041802BF6172C5C03410000000002B5004100470378565C034100000000AEB3004100470378565C03410000000038B2004180621058165C034100000000F8B00041802BF697E45B034100E5F21F3CB0004100470378565C034100000000B8AF004100470378985C034100E5F29F93AE004100470378465D03410000000047AD0041802BF697365E034100371AC0BEAC0041802BF697265F034100E5F29F52AC00410010E9B71660034100000000F8AB0041806210D8A460034100E5F21F96AB0041807D1DB80661034100000000B8AA0041802BF617C4610341001B0DE0D6A90041806210D8F66103410000000038A80041806210584862034100000000F8A60041806210D85862034100000000B8A50041806210D84B6203410000000078A40041004703F80B6203410000000038A30041806210D8CC61034100000000F8A10041806210589561034100000000B8A000410010E9379661034100000000789F0041004703789561034100000000389E0041807D1D388F61034100E5F29F339D0041806210D8F661034100000000B89B0041802BF6175C620341001B0D60D69A0041806210D8E66203410000000038990041806210D82F630341001B0D60DB980041806210D8E6620341001B0DE010990041806210D8F661034100E5F21FBB990041807D1DB80661034100E5F29F4B9A00410010E9B71660034100371A40329A0041802BF697265F0341001B0DE05E9A0041802BF697365E034100000000289A004100470378465D0341001B0DE0BF99004100470378565C0341000000004399004100470378665B0341000000002C99004180621058765A03410000000038990041004703F8B159034100000000789A0041806210D8BC58034100000000B89B0041802BF697AC58034100000000F89C0041807D1D386058034100371A40A49D0041807D1D38A6570341001B0D60199E0041802BF617B656034100000000389E00410010E937E855034100371A406F9E0041004703F8D554034100E5F29FC59E0041004703F8E553034100000000389E0041802BF6175A530341001B0DE0E69D0041004703F8F552034100000000F89C0041004703783652034100000000B89B0041807D1DB826510341001B0D60AA9A0041807D1DB82550034100E5F21FCA990041004703F8F74E0341000000006E980041807D1D384A4C034100000080019A00410010E9B7DA49034100E5F29F729E0041807D1D382349034100E5F29F2BA10041807D1D38B5480341000000804FA00041802BF697D745034100C9E5BF279C0041806210D81D42034100C9E5BF279C0041807D1D38403F0341001B0DE02AA000410010E9B7D3400341001B0D60BEA10041806210D8643F034100371A4050A10041004703F8193D034100371A4009A4004100470378AD3E034100E5F21FE6A50041802BF697F73F034100000000C3A7004180621058633D034100371AC0EAAB0041806210D8883E034100E5F21FEDAE004100470378664103410000000086B70041802BF617B2440341001B0DE01BBC0041807D1D38FC450341001B0D60AFBD0041004703F8D5430341001B0DE01EC00041802BF6171D410341001B0D60B2C10041807D1D38403F0341001B0D60AFBD0041807D1DB8853A034100E5F21F65BC0041807D1D38803303410000000069C10041802BF6973430034100E5F21F6CC50041004703F87D30034100C9E53F6FC90041806210D8C832034100C9E5BF70CB0041806210D83633034100E5F29FDFCC0041802BF6973B390341001B0D6072CD0041004703F8873D0341001B0DE005CF0041802BF61744440341001B0DE09AD2004180621058D8460341001B0D6055D700410010E9378E45034100371A4032D90041802BF697B042034100371A407DDB0041004703F8AE400341001B0D6086DC0041806210D8F93C034100E5F29FE9DB0041806210580E39034100C9E53F28DC004180621058AA33034100E5F21FC5DC0041807D1DB8B930034100371A4081DD0041802BF617852E0341001B0D6038DF0041004703784B2D034100371AC00EE100410010E9B7602F0341001B0D6043E30041802BF697B431034100000000F6EB0041807D1DB8173103410000008073EC0041806210D8C32E0341001B0DE0EBED004100470378942B034100371AC0F1F2004100470378942B0341001B0D60DDF60041802BF617DE2F034100E5F29F55F80041802BF6976B33034100371AC0C3FB0041806210585C360341001B0DE02C000141802BF6977637034100E5F29F7B030141802BF617A0350341000000801D03014180621058F830034100000000D9FD0041807D1D388A2D034100E5F21F85FB0041807D1DB8FC29034100371AC00CFA0041802BF617B325034100371A40E8FA0041806210581B240341001B0DE02C00014100470378C72103410000008028070141807D1D380B210341001B0DE0520B0141802BF6971A1E034100E5F29FEF0B0141004703F82E1A0341001B0D607C090141807D1DB86216034100E5F29FD404014180621058EA1403410000000032FF004100470378A116034100C9E53F41FC004100470378FA17034100371A40D8F70041806210584E1A034100E5F21F11F30041004703F83E1D0341001B0DE044EF0041802BF6179D1D034100371A4000EA0041802BF6971A1E0341001B0DE02EE70041806210D8DB1D034100371A404EE70041802BF6178D1A034100371AC019E50041004703F87C1703410000000024E30041806210D80E14034100E5F29F52E00041802BF6979B11034100C9E5BFFEDD0041802BF617C010034100000000C0DD00410010E9B7EE0D034100E5F29F42DD004180621058180C034100000000B0DA0041802BF697370C034100000080D9D80041004703F8FD0A034100371A40F1D50041004703F82507034100000080D6D40041802BF6978205034100C9E5BF76D30041802BF697ED0603410000000041D10041004703F8ED07034100371A4089C90041004703787007034100371AC0FBC5004180621058B406034100E5F21FDDCB0041806210584B02034100E5F29F21D10041802BF697B300034100000000FED70041802BF617DDFE024100E5F29F90DA0041802BF617D8FF024100C9E5BFFEDD0041807D1D3879F90241001361C373DF004180DE710A0AF6024100250681A1DF004100508D970DF602410077BE9F7BE0004100B4C8761EF6024100643BDFFBE0004100D9CEF72FF6024100643BDF3DE10041801058393BF60241009A99996EE3004100D34D62F5F6024100BE9F1A75E50041808B6CE76DF702410005A3127BE500418038D6C56EF7024100E3361A81E50041802EFFA16EF7024100DD240687E500418091ED7C6DF702410061C35392E70041808B6CE7D9F60241000000800AE80041004703F805F702410059173725E8004180FE437AB0F6024100EE7C3F4FE800410077BE9FA4F6024100DBF97E8CE80041806210588EF60241002DB29D0EE9004100D9CEF768F60241009CC4204CEA0041803D0AD713F6024100643BDF36EB004180EB51B8DEF5024100894160D2EB004100508D97BBF50241000000007AEC0041803D0AD791F50241003867C4F6EC00418045B6736DF50241008885DAF6EC0041801F636E6DF50241001283C00AEE004100D9CEF71CF502410066666694EE00410052B81EE9F402410014AE47D0EE004100A4703DD9F402410073689117EF004100333333B3F402410089416022EF004180EB51B8AEF40241009CC42073F0004100B4C87612F402410033333305F1004100FFB2FBC6F30241009CC420D9F1004100FED47859F3024100FED478B4F200418016D9CEE0F2024100894160CCF30041803D0AD746F2024100000000E9F4004100508D97ACF10241009CC42033F5004100508D9781F10241001283C063F5004100FED47847F1024100ECC039E3F500410073D712DDF0024100819543E3F500410095D409DDF0024100894160ADF600418062105834F00241001283C0F6F6004180749318F7EF024100643BDF6EF7004100508D9796EF024100643BDF2FF8004180EB51B8EBEE0241001283C0ACF8004100B684FC7CEE024100894160A0FA0041808FC2F5C1EC024100EE7C3F87FA004180621058A3EC024100A1F83192FA004100FA7E6A99EC024100F853E3CEFA00418096908FCEEC024100AB3E5765FB004180F1D24D52ED024100FA7EEA93FB004100FFB2FB8EED024100FC18F397FB00418011A53D94ED024100DAACFA97FB004180A6794794ED024100F697DD98FB00410069006F95ED02410026E40311FC004180D85FF631EE0241001B0D60B2FF0041807D1DB8ECF2024100D9CEF7B4FF004100D34D622BF30241005227A0B9FF004100D9CEF79BF302410089416092000141806B9AF76BF5024100894160A601014180EB5138FBF8024100A54EC05C0201410087A75752FD024100371A409F02014100C7BAB808FE024100371A40370701410010E9375BFF02410058CAB2D10A01410086C9D49B000341005227A0E10B014180F4DB57FA00034100871659AC0C01410080483FD601034100A54EC09C0E0141807D1DB8F0030341008941602E0F014100A2B43763040341001283C0FC0F0141802BF697B904034100E5F29FB51001410090A0F88B040341008126C2041101410060764FDF0303410089416012110141802BF697C1030341009FCDAA4B11014100F8C2E48C02034100C9E53F4F110141802BF69779020341005227A09D11014100A2B437E301034100D509E8C31101410012143FD001034100A54EC0E411014100D9CEF7BF01034100C9E53F17130141807493981102034100E5F29FED140141805986384703034100A54EC09C1501410010E9B77C030341000612949916014180A679C732030341005227A0A917014100A2B437E3020341000000004C19014100A2B4372302034100819543031A0141801E85EBF901034100C9E53F0F1A014180348037F701034100F775609E1A0141807D1DB808020341002BF697F11A01410030BBA74702034100371AC0101B014100A2B4375F02034100C9E53FAB1B014180BDC1977D03034100894160A21C014100C7BAB8800703410029CB90901C014180984C952408034100F775607A1C0141807D1DB8F0080341001B0D60321C014180749398ED090341009FCDAA031C0141803C2C544B0B0341001B0D60EA1B0141807D1DB8080C034100B22EEEEC1B014180779C22BE0C0341005227A0F11B0141803480370B0E034100A54EC0541C01410010E9B70410034100894160E21C014100508D97B111034100764F9E7B1D014180D3BCE3921203410035EFB8891D01410077BE9FA712034100C9E53F271E014180EB51388F1303410077BE9FAD1E014180EB5138C7140341002B1895701F01418029CB1099150341008941601220014100A2B4374716034100C9E53F7720014180749398ED16034100C9E53FAF200141004703F88317034100000000C4200141004703F803190341009D11A5C420014100AAF1D2541A034100894160C6200141807D1DB8DC1D034100C9E53F6320014180749398491F034100E5F29FB51F014100D9CEF7232003410020D26FE91E0141804BC8877820034100894160121E014100075F98D120034100E5F29FA11D01410090A0F84B2103410033C431A71D014180BF7D9DA0210341006E3400B41D01410010E9376322034100894160521E014100075F985922034100F08549761E0141000534916622034100C9E53FD71E0141807493988922034100F77560021F014100C7BAB8C822034100E5F29F091F0141807D1DB85423034100894160F61E014100A2B437E323034100643BDFE41E014180567D2EFE23034100000000B81D014100508D97CD25034100D6C56DDC1D0141807AA52C8E26034100000000681E01410090A0F86F290341005227A0291E0141807D1DB8D42A034100234A7B061E0141002C6519F92A034100371AC0D01D0141807D1DB8302B034100371A40EF1C01410090A0F8AF2B034100A54EC0881C014180F4DB578A2C034100A54EC0A41D014180348037E72E0341006E3400481F014100508D978D31034100D3DEE0471F014180B1BF6C9C310341001B0DE0361F0141004703F80F320341005BB13F2E1F014100A2B4376E32034100000000D01E014100992A988034034100522720A41E014100992A1804350341006E3400931E0141004703F82F350341008941E07C1E0141802BF6178335034100000000D01E014100470378BA350341005BB13F471F01410010E9B7D837034100E5F21F541F0141004703F84F3803410092CBFF691F0141802BF617EA380341006ADE71891F014100CDCC4C313A034100A167B3881F0141803E575B333A0341008FE4F26E1F0141806B9AF73A3A0341006E3400881E01410010E9377F3A034100D04458601E014180AD69DEB53A0341006E3400601E014180621058B63A0341008C4A6A351E0141801C5AE48C3B034100371AC0101E014180BDC197453C034100C9E53F171E014180EB5138973D0341008941604A1E0141004703F8DF3D034100C9E53F7F1F014180621058623E034100D578697A20014180F3FDD49C3E03410068916D7720014180FF90FE9C3E0341008941603920014180EB5138F93F034100000000D01E0141004703F8B04003410092CBFF0A1E014180B437F8EA40034100C9E5BFDE1D0141004703F8AF410341006E34803D1E0141004703784242034100000000D01E0141806210D8E442034100C9E5BFD01F01410010E9B7B042034100AED8DF9021014100992A180F42034100000000F021014180B437F8F54103410092CBFF1B23014180B437F8A343034100E5F29F13230141004703F8CF440341001B0D60652301418062105845460341006E348084220141004703785B4703410058CAB294230141003CBDD2B1460341008941608A230141807493981147034100E5F29FC9230141806210585A47034100371A401F24014180BDC1976547034100C1CA21D424014100A5BDC135470341005BD33CD7240141804F1E9636470341000000001025014180EB51B85547034100C9E5BF8625014180EB51387947034100C8983BCA26014100AF2564E147034100A52C43CA26014180AF9465E14703410040A4DFEB260141806FF0053448034100226C786827014180CE8852664903414300000000E5F29FC1D90041806210D84BB7034100E5F29F36DA00410010E9B75BB60341001B0DE092DA00410010E9B76BB5034100000000E4DA0041802BF6977BB403410000000038DA0041802BF617E1B3034100000000F8D80041802BF697E9B3034100371AC0FDD70041802BF6978BB303410000000078D60041802BF617F1B2034100E5F29FEBD50041004703789BB203410000000038D50041807D1D3834B2034100E5F29F84D4004100470378ABB1034100E5F29F58D4004180621058BBB0034100000000F8D30041806210D856B0034100E5F29F68D3004180621058CBAF034100000000B8D20041802BF61731AF034100E5F21F11D2004180621058DBAE03410000000078D10041802BF69797AE034100E5F21FBFD00041807D1D38EBAD03410000000038D00041807D1DB867AD0341001B0D60E3CF0041807D1D38FBAC034100371A4042CF0041802BF6170BAC034100C9E53F5DCE0041802BF6171BAB034100E5F29FD3CD0041004703F82AAA034100000000EDCC0041004703F83AA903410000000078CC0041806210D8F2A80341001B0DE046CB0041004703F84AA8034100000000F8C900418062105830A8034100000000B8C800410010E9B7BEA8034100371A405AC80041004703F83AA90341001B0DE0ABC70041004703F82AAA03410000000078C70041004703F8BAAA0341000000002FC70041802BF6170BAC0341000000801EC70041807D1D38FBAC0341001B0DE0CDC60041807D1D38EBAD034100E5F29F66C6004180621058DBAE0341001B0D60F5C5004180621058CBAF034100E5F29F94C5004180621058BBB0034100000000F8C40041004703F870B10341000000008BC40041004703789BB20341001B0DE034C40041802BF6978BB3034100371A4051C40041802BF6977BB4034100000000F8C40041807D1DB843B503410000000038C600410010E937C1B503410000000078C70041802BF6173BB6034100000000B8C80041004703F8DAB6034100E5F21FC1C90041806210D84BB703410000000038CB0041806210D8ABB7034100000080A4CB0041806210D84BB703410000000078CC0041802BF697ACB6034100000000B8CD004180621058AAB6034100000000F8CE0041806210D80AB703410000000038D00041806210D8E8B7034100E5F21FA8D00041806210D83BB803410000000078D10041806210D8DDB8034100371A40EBD10041806210D82BB9034100000000B8D20041802BF697C1B90341001B0D6062D30041004703F81BBA0341001B0D60B0D30041004703F80BBB03410000000038D50041807D1DB837BB034100371AC0F5D50041802BF617FCBB03410000000078D60041004703F87CBC034100000000B8D70041806210584BBC0341000000804DD80041004703F80BBB034100000000F8D80041807D1D3865BA034100371AC02DD90041806210D82BB90341001B0D6069D90041806210D83BB8034100E5F29FC1D90041806210D84BB703410C00000000000000F80A0141004703F8B189034100000000F80A0141806210D83689034100000000B8090141806210D84F89034100371A405D090141802BF617498A03410000008035090141807D1D38398B034100E5F21F2F090141807D1D38298C034100000000B8090141802BF617B08C034100000000F80A0141802BF697AF8C0341001B0D60680B0141807D1D38298C034100E5F29F840B01410010E937398B034100000000F80A0141807D1D38868A034100000000F80A0141004703F8B18903410A0000000000000069E10041807D1DB8EEE903410000000078E00041806210D828E903410000000038DF0041806210D85FE9034100000000F8DD0041802BF69775E9034100000000D6DD0041807D1DB8EEE9034100000000E0DE0041806210D8DEEA03410000000038DF0041807D1D38A0EB03410000000078E00041807D1D3837EB03410000000066E10041806210D8DEEA03410000000069E10041807D1DB8EEE90341010300000001000000790400000076029AD243FC4080BB96108CDD0341007CF2B0C4E3FB400020D2EFD2E40341004AEA04B8CCFB408003780B0FE60341005227A0CCCBFB40004703F88AE50341006E34801FCBFB40806210585AE4034100CAE53F82CCFB40802BF6174BE30341001C0DE074CEFB40807D1D3854E20341001C0DE056D1FB40806210D8ADE10341005227A0F7D6FB4080621058B1E1034100E4F21F28DAFB40806210589BE1034100361AC058DBFB40802BF61774E10341001C0DE009DDFB40802BF61715E1034100AED85FFCDFFB40806210D83EE00341000000005EE2FB40807D1D38B0DF0341006E34805FE5FB40004703F841DF0341006E348060E7FB40807D1D38F3DE034100CAE53FC2E8FB40004703F84BDE03410092CB7F85E9FB408062105804DD034100E4F21F67E9FB400010E93764DC0341005227A018E9FB40806210D8BBDB0341001C0DE04BE7FB40802BF6975ADA0341005227A0BDE4FB40806210D888D903410092CB7FFEE2FB40807D1DB80FD9034100000000FFE1FB40802BF617CFD8034100000000E1DFFB40807D1DB8EDD7034100CAE53F93DEFB400010E9B7F4D60341006E348046DDFB40807D1DB893D50341005227A098DBFB40802BF697A2D4034100000000ACD8FB40802BF69720D30341000000001ED6FB40806210D82ED20341000000004FD2FB400047037894D10341001C0DE09ECEFB40802BF6176AD1034100361AC01ECCFB40802BF69760D1034100AED85F0ECAFB408062105867D1034100CAE53F6DC6FB40802BF617A5D10341000000002CC4FB40807D1DB80BD203410092CB7FBAC1FB40807D1D3882D2034100E4F21F29BFFB400010E9B7F0D20341006E348008BEFB40802BF61728D303410092CB7FC6BBFB40807D1DB8CED3034100E4F21FC4B9FB40802BF697B5D4034100AED85F22B8FB400010E9B754D503410000000030B7FB400010E93734D60341001C0DE06CB5FB40807D1D3863D70341005227A02AB4FB40802BF69732D80341006E3480FAB1FB40807D1D3829D80341001C0DE09AAFFB40807D1DB8E7D7034100CAE53F7AADFB408062105806D803410092CB7F19ABFB40004703F834D8034100000000C9A9FB40802BF6175CD8034100361AC048A8FB40807D1D3863D8034100CAE53F78A6FB40802BF6177AD803410092CB7FF8A4FB400010E93749D8034100CAE53F49A4FB40807D1DB8F0D7034100E4F21FBBA3FB40807D1D3830D70341005227A04EA4FB4000470378C8D5034100CAE53F61A5FB40004703F8C8D40341001C0DE054A6FB408062105861D3034100AED85F37A7FB40806210D869D2034100CAE53FE9A7FB40802BF617B2D10341001C0DE07AA8FB400047037812D10341005227A06CA8FB40806210585AD0034100E4F21FEFA7FB40806210D851CF0341006E348091A7FB400047037859CE034100361AC004A6FB4080621058F8CC034100CAE53F88A3FB40802BF6976ECB0341006E3480AAA1FB40806210586DCA0341005227A04D9EFB40802BF61703C9034100AED85F309BFB40004703F8B8C7034100361AC07298FB40802BF617A7C6034100361AC0B495FB4080621058ADC5034100CAE53F0594FB400010E93764C50341001C0DE0A592FB408062105813C503410092CB7F6590FB40004703F819C50341006E3480558DFB40004703F8EFC40341001C0DE04689FB40806210582DC40341005227A0C885FB40802BF6174BC3034100000000C983FB40806210D809C30341000000004880FB40807D1DB83FC30341006E3480F77DFB400010E9375EC3034100000000977BFB40806210D86CC30341006E3480A679FB40802BF69783C3034100000000F576FB40004703F801C40341001C0DE00375FB40806210D858C40341005227A0A273FB40802BF617C8C4034100AED85F0173FB40807D1DB847C5034100E4F21FAE71FB40802BF6177FC60341006E34803C71FB40806210D826C7034100E4F21F0A70FB40807D1D3806C8034100361AC0076EFB40802BF617E5C8034100361AC0876BFB4000470378C3C80341001C0DE06869FB40802BF6172AC8034100CAE53F8968FB40802BF697F9C70341006E3480B866FB400047037830C8034100E4F21F8663FB40802BF697FEC8034100E4F21FD561FB40802BF6974DC9034100E4F21FE460FB40004703F8A4C9034100CAE53F125FFB40004703F84BCA034100E4F21FA15CFB4000470378AACA034100CAE53FE059FB40806210D8E0CA034100361AC01E58FB40806210D85FCB0341006E34808C56FB40004703F836CC034100AED85FC955FB40802BF6976ECD034100361AC07554FB40004703F8CDCE034100361AC0E353FB40807D1DB895CF034100E4F21F5352FB40807D1DB8BCCF034100E4F21FD34FFB40807D1D38A3CF034100CAE53F134EFB40802BF61782CF0341001C0DE0C24AFB40004703F877CF034100361AC03248FB40004703785ECF034100E4F21F0346FB40802BF6171DCF0341005227A0C242FB40004703F822CF034100E4F21F923EFB400047037828CF034100CAE53F813CFB40807D1D385FCF0341005227A08F3AFB40802BF617F6CF0341001C0DE02C38FB40807D1DB8ECD00341000000000A36FB4000470378FBD1034100361AC00734FB4080621058C2D2034100AED85F4633FB40004703F849D3034100361AC04331FB40806210D838D4034100000000312FFB40807D1DB83FD50341001C0DE0ED2CFB408062105866D60341000000000B2BFB40806210586DD70341000000004929FB40806210582CD80341001C0DE08727FB40806210588BD80341005227A05725FB40004703F881D80341000000003824FB400010E93749D8034100E4F21FDB23FB40806210D800D7034100361AC0CD24FB408062105801D6034100CAE53F0F25FB408062105871D5034100361AC0C025FB40807D1DB8D9D40341001C0DE07227FB40807D1DB812D40341005227A01429FB40802BF6977BD3034100CAE53F072BFB40807D1DB884D20341006E34809A2CFB40802BF69745D10341005227A06D2DFB40806210D815D00341006E3480EF2DFB40802BF61756CF0341005227A0922EFB40806210581ECE034100CAE53FF52EFB400047037816CD0341006E3480982EFB40802BF617C6CB034100AED85FEC2CFB40806210D81CCA034100CAE53F4F2BFB40807D1DB8DBC8034100361AC04129FB40807D1D38BAC70341006E34804326FB400010E937E0C6034100AED85F4423FB408062105866C60341001C0DE00321FB40004703F874C60341005227A0521DFB400010E9B7CAC6034100000000E21AFB40807D1D38E9C6034100AED85FA117FB400010E937F7C6034100E4F21F8115FB40806210D8FDC6034100E4F21F1112FB40807D1DB8D3C60341006E3480C20EFB40802BF69719C6034100361AC0330CFB40004703F86FC50341001C0DE09608FB4000470378FDC30341005227A04906FB40806210D8C3C20341005227A0ED02FB40802BF697F9C0034100E4F21F5100FB400010E9B76FBF0341001C0DE002FFFA40806210D8AEBE034100E4F21F64FEFA40806210581EBE0341006E348075FDFA400010E9B78DBD03410000000057FBFA4080621058D4BC0341005227A047F8FA40806210586ABC034100361AC067F6FA40807D1D3849BC0341005227A086F2FA40806210D886BC034100CAE53FE6EFFA40807D1D3885BC034100CAE53FA4EBFA40802BF6971ABD034100361AC071E7FA40802BF617E8BD034100E4F21F50E5FA40806210D86EBE0341000000007EE2FA40807D1D3825BF034100E4F21F0CE0FA40806210D8CBBF0341000000005ADEFA40806210D892C0034100E4F21FB8DDFA40004703784AC10341006E3480D4DBFA4000470378A9C20341005227A071DAFA40806210D8C0C30341006E348030D9FA40802BF61720C403410000000060D7FA40004703F83EC403410000000031D5FA40802BF697BDC3034100CAE53FE3D3FA40802BF697BCC2034100CAE53FC5D2FA40806210D8E3C10341001C0DE026D1FA40807D1DB822C1034100361AC047D0FA40802BF617BAC0034100361AC058CDFA400010E93730C00341006E3480E8CAFA40807D1DB82EC00341006E348067C8FA400010E93775C00341006E348046C5FA40807D1D38B3C00341005227A095C2FA40802BF697E1C00341001C0DE093BEFA40807D1D385FC10341005227A012BCFA40807D1DB8C5C1034100CAE53F02BAFA4000470378CCC1034100AED85FA2B7FA40004703F8A2C103410092CB7F62B4FA40806210D870C103410000000052B2FA40802BF69787C1034100361AC081B0FA400047037886C1034100361AC0E0ADFA40806210D8CCC10341001C0DE04FABFA408062105803C20341001C0DE08EAAFA40806210D85AC2034100361AC0CDA8FA40806210D8C1C2034100AED85FBCA6FA40802BF69730C3034100CAE53F4BA5FA40806210D88FC3034100CAE53F3BA4FA40802BF61787C30341001C0DE00BA3FA408062105836C3034100000000FEA1FA40802BF69745C20341006E348070A1FA40802BF61745C1034100CAE53F83A0FA408062105814C00341005227A0D49FFA40806210D87BBF034100361AC0369CFA400047037871BE034100AED85F6799FA400010E9B70FBE0341006E34806797FA4000470378E6BD0341006E34806594FA400010E9B794BE034100AED85F9491FA40004703F8E2BE0341006E3480038FFA408062105811BF0341006E3480F48CFA40802BF61798BE0341001C0DE0158CFA4000470378F7BD034100CAE53FC78BFA40807D1D3867BD0341005227A0A98BFA40004703F86EBC034100AED85F0B8CFA40802BF617BFBB0341000000009F8DFA40004703F85FBA034100E4F21F018FFA400010E9B7A0B90341005227A08390FA4000470378B1B80341006E34803693FA40802BF617ABB7034100E4F21FA895FA400047037824B7034100361AC0FA97FA40806210D835B6034100361AC00E9CFA40807D1D38D0B4034100361AC0409FFA40802BF61722B4034100CAE53F62A1FA4080621058A3B30341001C0DE0D3A4FA40806210582DB3034100E4F21FE4A7FA408062105837B3034100CAE53FB4AAFA40802BF61751B3034100E4F21FC5ADFA40004703F822B303410000000097AFFA40004703F87BB2034100E4F21F98B1FA40807D1D381DB20341000000002AB3FA40802BF61776B10341006E34800CB4FA40004703787EB0034100AED85F6FB4FA40802BF69756AF0341006E348082B4FA40004703781EAE0341005227A0F4B4FA40807D1DB846AD0341005227A066B5FA40806210D87EAC034100361AC0C8B6FA40802BF697B7AB034100361AC0A8B8FA400010E9B7D0AB03410000000015BEFA408062105894AD0341001C0DE0B1C0FA40807D1D38F6AE034100E4F21F10C4FA4080621058E0AF034100AED85F3FC6FA40807D1DB841B003410000000071C9FA40807D1DB8C3AF034100CAE53F52CBFA40806210D864AF034100CAE53FC4CDFA400010E937AEAE034100AED85F67CFFA40802BF6177FAD0341006E3480F9CFFA4080621058AFAC0341005227A0EDD0FA40807D1DB817AB034100AED85F00D2FA40807D1D3808AA034100AED85F63D3FA40004703F8E0A8034100361AC036D5FA40004703F8A1A7034100361AC039D7FA40802BF61783A6034100CAE53FEBD8FA40802BF61704A603410092CB7FACDBFA400010E9B7A5A5034100CAE53F3DDEFA40806210587FA5034100000000FCE1FA40807D1DB829A6034100CAE53F6BE5FA40806210D8ABA6034100E4F21FABE7FA4080621058CDA6034100000000EBE9FA40807D1DB8F6A60341006E34804AEFFA40802BF61772A7034100000000FAF3FA40802BF617EDA70341001C0DE0D9F5FA40807D1D380EA8034100CAE53F59F8FA40806210D86FA80341005227A0B8FBFA40004703F8D9A8034100AED85F58FEFA40807D1DB813A9034100AED85FD8FFFA40802BF6972CA90341005227A01801FB408062105825A9034100CAE53FD902FB4000470378F6A80341005227A0F903FB40802BF617E7A80341006E3480DA04FB40802BF6978FA80341001C0DE07B05FB40004703F80FA8034100000000CE05FB40802BF61738A70341001C0DE0D005FB40004703F807A60341005227A05205FB40802BF69757A5034100CAE53F3404FB400010E9B79EA40341000000004602FB4000470378CDA3034100361AC0A6FFFA40807D1DB85BA3034100361AC067FCFA40807D1DB8D1A2034100E4F21FDAF6FA40802BF6178EA1034100E4F21FDBF4FA40807D1DB80CA10341000000003CF3FA40807D1DB89BA00341001C0DE08CF1FA40802BF69732A003410092CB7F9DEFFA4080621058D19F0341005227A02DEEFA4000470378B89F034100E4F21F5DEBFA40807D1DB8BE9F034100CAE53FBDE8FA40802BF617959F0341001C0DE0EDE4FA40802BF6971A9F0341001C0DE0BEE1FA40802BF697909E034100361AC05FDFFA40802BF6170F9E034100361AC0F0DCFA40004703788D9D034100E4F21F62D9FA40807D1D38CB9C034100E4F21F53D7FA40806210D8519C034100AED85FC4D4FA40807D1D38B09B0341005227A084D3FA4080621058879B034100AED85F95D0FA40004703780D9B034100E4F21F96CDFA40802BF6979B9A034100000000A7CBFA40806210582A9A0341005227A008C8FA40004703F847990341005227A05AC4FA40802BF6974D98034100AED85F2BC2FA40802BF617E4970341005227A01CBFFA40802BF6173A97034100361AC01DBCFA40807D1D38A096034100AED85FCFB8FA40802BF617CE95034100361AC0AFB6FA40807D1DB88C950341001C0DE0AFB5FA40802BF6176C950341001C0DE05EB4FA4080621058C395034100AED85F3DB2FA40004703F84196034100CAE53F3CB0FA40806210D8A0960341001C0DE05BAEFA40807D1DB8A7960341005227A0CDABFA40004703F8D5950341005227A05DAAFA40802BF617C595034100E4F21F6DA7FA40807D1D38D395034100361AC03BA4FA408062105831960341006E34803BA2FA40802BF61738960341005227A09A9FFA40004703786E96034100361AC0799DFA40807D1D38AD96034100000000799BFA40004703F8DB96034100AED85F489AFA40807D1D380B970341001C0DE01799FA40802BF6972A97034100E4F21FD897FA400010E9B701970341001C0DE05895FA40802BF61798960341001C0DE0C893FA40807D1D388796034100AED85F1992FA40802BF6173E96034100E4F21F3990FA40004703F834960341006E3480088EFA40802BF6975B960341001C0DE0488CFA40004703782296034100000000AA8BFA40004703F8A195034100000000AC8BFA40004703F8D994034100AED85F8F8DFA40004703F8929303410092CB7F718FFA40004703F8D392034100AED85F3492FA40802BF697CD91034100E4F21F8896FA40802BF61788900341005227A07999FA40806210D81190034100361AC06B9DFA40807D1D386C8F034100361AC0DDA1FA40004703F8CE8E034100CAE53F5FA5FA40802BF617698E03410000000010A8FA40807D1DB8428E0341001C0DE0A1AAFA40807D1D389C8D0341001C0DE0F3AEFA40806210D80E8D034100000000A5B2FA40802BF617C18C034100361AC085B5FA40806210D89A8C0341005227A0A6B8FA400010E9B76C8C03410000000008BDFA4000470378178C034100CAE53F09BFFA40802BF697B08B034100CAE53FDAC1FA4080621058728B034100AED85F0BC5FA40806210582C8B034100E4F21F1CC8FA400010E937FE8A034100CAE53FECCAFA40004703F81F8B034100CAE53F6BCDFA40802BF697A18B034100E4F21FBACFFA40802BF6173B8C034100CAE53F69D1FA40802BF617AC8C0341001C0DE057D4FA40802BF617568D0341005227A0D5D6FA40807D1DB85F8E034100361AC0D2D8FA40802BF617A98F034100361AC0F0DBFA40807D1D38A390034100AED85F00DDFA40806210D8CB90034100CAE53FC0DFFA40802BF69705910341006E348060E2FA40807D1D380791034100361AC070E3FA40806210D8FF90034100361AC0D1E5FA4080621058B990034100AED85F63E7FA40807D1D382290034100AED85F64E9FA4000470378D38F034100000000B5EAFA40807D1D38AC8F034100AED85FA5EBFA40806210D88C8F0341001C0DE0D5EDFA40802BF617768F034100AED85F36F0FA40802BF697678F034100AED85FE5F3FA40004703F8F98F0341001C0DE0A4F7FA40806210586490034100361AC0B4F9FA40807D1DB88D900341005227A0E4FAFA4000470378A690034100AED85F75FFFA40807D1D389990034100E4F21F9603FB40806210D883900341000000009606FB400010E9B7B5900341006E34805509FB40004703780F91034100AED85F740BFB40806210D8A0910341005227A0420DFB40802BF6176292034100000000310FFB40806210582B930341005227A09F10FB4080621058C4930341001C0DE0BE13FB40806210583E940341001C0DE0BE15FB40802BF6975F94034100AED85F5F17FB40802BF69738940341005227A03F1BFB40004703F852940341001C0DE09E1EFB40802BF617D594034100CAE53FFD21FB4080621058A795034100361AC00926FB40802BF6174297034100AED85F3729FB40807D1D386498034100E4F21F742DFB40004703F8E69903410092CB7F3231FB4000470378C19A0341005227A0E035FB4000470378BC9B0341005227A0BF39FB40004703F85E9C034100E4F21F2F3CFB4000470378A89C034100CAE53FAF3DFB4000470378B99C034100E4F21F9F3FFB40802BF697DA9C034100E4F21F1F41FB40802BF697EB9C034100CAE53F8F43FB40802BF617059D0341006E3480BF46FB40802BF617179D0341001C0DE02F48FB40004703F8F79C0341005227A09049FB40806210D8C09C0341001C0DE0714AFB4080621058519C034100E4F21F154BFB4000470378099B0341005227A0A84AFB40802BF6179999034100361AC0EA49FB4000470378B098034100CAE53F7E47FB40806210D82E97034100E4F21FC241FB40004703F85295034100361AC0D33BFB400010E93757940341000000008636FB400010E9B72B930341006E34802832FB40806210D8F0910341006E3480D92EFB40807D1DB85E9103410092CB7FFA2AFB40807D1D38C4900341001C0DE06B27FB40004703F809900341005227A03E23FB40807D1D38B78E034100CAE53F0220FB40004703F8148D0341006E3480D31CFB40004703F8628C03410026C2867518FB4080CE88D2B08B0341006666663C19FB4080438B6CEF8A034100E6D0223D19FB40005839B4EE8A0341009CC420F419FB40806ABC74328A034100703D0AF719FB4000DD24062F8A034100BE9F1AF919FB40800E2DB22B8A03410052B81E271AFB4000FA7E6AD2890341009CC4204A1AFB40806210586689034100D022DBC11AFB4080BE9F1ABD87034100726891C11AFB40000AD7A3B387034100C64B376B1BFB4080EB51B82385034100726891031CFB4080BA490C1383034100B6F3FD3C1CFB4000508D973B82034100000000521CFB4000000000F681034100A4703D881CFB40007B14AEF380034100986E12EF1CFB4000AAF1D2807F03410090C2F53C1DFB408095438BB57E034100829543891DFB4080490C02287E034100DACEF73F1EFB4080BC7493557D034100000000E61EFB4080F3FDD4A57C034100621058A11FFB4000560E2D1B7C0341007493188A20FB4000AC1C5A8E7B034100F0A7C67721FB4000AC1C5A137B03410090C2F57821FB40007F6ABC127B034100AAF1D27B22FB40005A643B867A034100C0CAA18122FB400054E3A5827A0341002406814323FB408016D9CEF979034100986E121F24FB40007D3F356A79034100E04F8D2124FB408047E17A6879034100105839F024FB40803F355ECF7803410052B81E6D25FB40808FC2F581780341001C5A64A325FB4000371AC04478034100AED85FD625FB40807D1DB84C780341005227A05126FB40802BF69765780341005227A0A926FB40802BF6976978034100B27BF2BC26FB4000BB278F6B78034100AC1C5AA227FB4080F1D24DA078034100C876BE8528FB4000508D97D47803410052B81EF328FB408039B4C8EA780341004C37892529FB4080E17A14EF78034100C0CAA13129FB4000508D97EF7803410022DBF93429FB4000DF4F8DEF780341007C3F35C829FB40809BC420EC780341000681953D2AFB4080EB51B8EE7803410040355E1E2DFB4000FED478B17803410096438B7A2DFB40806CE77BA9780341005227A0412EFB40004703F89F780341000000003033FB40807D1D381F780341005227A02139FB400010E9B75877034100081B9EFC39FB4000ECC0393D77034100DCF97E623AFB4000B4C8763C77034100DCF97EA43AFB40807493183C770341002C431CCE3AFB408026C2863777034100A2B437CE3AFB40004A7B83377703410094A982B43CFB4000780B240277034100C264AAB43CFB4000C05B2002770341007E6ABCB43CFB4000E4831E02770341008816D9443DFB4000166A4DF276034100DCF97E963DFB4080621058E9760341004EF38EB93DFB408048BFFDE17603410052499DB93DFB408047E1FAE1760341009ECDAAB93DFB4080D93DF9E176034100EE7C3F7D3EFB4000D9CEF7B8760341007E6ABCEC3EFB40004B59869E76034100AED85FA63FFB40807D1D3887760341005227A0F143FB40004703F88F750341000000004847FB400010E9B79874034100A62C43A847FB4000B29DEF73740341008A4160FD48FB4000D9CEF70B74034100C0CAA1EF4CFB4080C64B37AD72034100C876BED34DFB4000FED4784572034100DACEF7E14DFB40008716D94472034100E27A14E24DFB4000A69BC44472034100263108E64DFB4080F1D24D4172034100D478E9704EFB4080F1D24DB671034100C2F528744EFB4080B81E85B2710341000E2DB2CB4EFB400079E9263A710341006CE7FB454FFB40803BDF4FB770034100D022DB994FFB40808D976E5B700341003433331951FB40808B6CE7B37003410074B5152151FB4000B37B72B5700341005EDC462951FB4080931884B670034100E8FBA93151FB4080749318B770034100B6F3FDF451FB4000AC1C5AB66F034100D022DB2352FB400077BE9FBC6F034100A01A2F3752FB400023DBF9AA6F034100703D0A4D52FB40800E2DB2AC6F034100643BDFC352FB4000CBA145BD6F034100BE9F1A9D53FB4000CDCCCCE46F0341007E6ABC1454FB40809BC420F96F0341003CDF4FE154FB400021B0722470034100DE24066D55FB4000AAF1D24B70034100CAA1457855FB4000D34D624E700341000AD7A3FC57FB4000F853E3C0700341003889410858FB4000FA7E6AC27003410014AE470B58FB40809DEFA7C270034100A245B6CB58FB4000FA7E6ACF7003410022DBF9C059FB400085EB51FA70034100EC51B8C459FB40801E85EBFA70034100749318045AFB40807493180471034100388941825AFB40807493180D71034100388941185BFB4000355EBA277103410086EB51405BFB4000AE47E13171034100749318B65DFB4080BA490C8D710341007A14AEB75DFB40008195438D71034100560E2DBA5EFB40806E1283AF71034100FCA9F1BE5EFB400004560EB071034100E4A59B3661FB4000560E2DED71034100EA26312663FB408016D9CE1D720341004E62102464FB4080BA490C3672034100CAA1451A66FB4000A4703D68720341003A92CBC867FB40805986B89C720341004AEA045E69FB40004B5906D6720341007A14AE696AFB408066666604730341008C6CE7976BFB40002506813773034100A4703D186CFB4000D34D625B73034100B22E6E226CFB4080EC2FBB5D73034100ECC0392D6CFB4080EE5A425F7303410088635D386CFB400044FAED5F7303410098DD93436CFB408059F5B95F730341007424974E6CFB400030BBA75E73034100C03923596CFB40807FD9BD5C730341006A2BF6626CFB40002731085A73034100AAF1D26B6CFB4000508D975673034100AE69DEE673FB400067D567EF6F034100AED85FC67BFB40004703F8C36C034100AED85FFE7BFB40807D1DB8E86C034100000000587CFB40802BF697216D0341005227A0F17CFB400010E937776D034100AED85FFE7DFB40802BF697F96D0341005227A0B97EFB40807D1D38636E034100361AC0B47FFB40807D1DB8B06E034100925CFE5D80FB4000CDCC4C036F034100D8A3705F80FB4080976E12086F034100F4FDD46E80FB40008195431D6F034100D022DB7380FB40007D3FB5226F034100081B9E7A80FB4080302AA9276F0341006891ED8280FB40806CE7FB2B6F034100CCCCCC8E80FB40007D3F35316F0341001C5A64A180FB4080C64B37376F0341005C6DC5D380FB40809DEFA7376F034100361AC09C81FB40807D1DB8746F034100000000A882FB400010E937FF6F034100F697DDEE83FB40804BC8879B70034100E6D0220184FB4080F1D24DAF700341005A643B0784FB40809DEFA7B470034100A4703D3684FB40007F6ABCD6700341006666665484FB40803F355EF5700341000AD7A35684FB4000B4C876F7700341000E2DB25B84FB4000819543FB70034100AAF1D28784FB4000819543187103410098FF908E84FB4080E2C7181C71034100900F7A9284FB40808104C51D710341005227A03185FB40806210589E71034100361AC04486FB40807D1DB814720341000000001087FB40004703F8EF72034100CAE53FEB87FB40802BF6974573034100361AC08C88FB40004703F8C773034100B6627F8D89FB40005D6DC55B740341005C8FC28D89FB40000AD7A36B74034100423EE88E89FB40803789C171740341009AE61D9289FB40800D4FAF7774034100A8C64B9789FB40803789417D74034100E4A59BB089FB4000A8C64B937403410076BE9FB689FB4080E3A59B9774034100E27A14C089FB408016D9CE9C7403410060E5D0CA89FB40007D3F35A274034100388941248AFB4080749318B7740341002406817B8AFB4080C64B37F3740341008E976EAE8AFB40004E6210247503410040355EB08AFB40806CE7FB24750341003E0AD7B38AFB408095438B2675034100B81E85BB8AFB4000AC1C5A297503410092ED7CC78AFB400079E9262D75034100AC1C5A028BFB4000D9CEF74F75034100F2D24D208BFB4000F6285C6375034100FA7E6A328BFB4000560E2D7175034100EA2631448BFB40004C37897E75034100508D974A8BFB40807268918475034100BC7493588BFB400079E92693750341000000005A8BFB4000295C8F94750341002406816B8BFB40806891EDA4750341000AD7A36C8BFB4000D9CEF7A5750341008816D9728BFB4000250681AA750341006210587F8BFB408014AE47B275034100C876BEB38BFB40007F6ABCD875034100C64B37B58BFB4080EFA7C6D9750341004260E5D28BFB400023DBF9ED75034100285C8F0A8CFB40800C022B1476034100D0B359108CFB40809DEFA717760341006C09F9168CFB40801283C01A7603410086EB511E8CFB4000B072681D76034100666666308CFB400008AC1C23760341006210583D8CFB408047E17A2C76034100CEF753478CFB4080ED7C3F3A760341000AD7A34C8CFB4000D9CEF73F76034100EE7C3F518CFB4000B4C87643760341004A0C02858CFB4080C64B37667603410012A5BD8B8CFB400052B81E6A76034100B81E85938CFB40002506816D760341003CDF4FB38CFB4000D7A3707976034100FED478CD8CFB40007D3F358876034100A8C64BD58CFB4080490C028C76034100343333ED8CFB408045B6F395760341005839B4068DFB4000F6285CA976034100A245B6098DFB408047E17AAB76034100E27A145C8DFB4000A4703DE176034100AAF1D2758DFB40005839B4F476034100560E2D788DFB4000894160F67603410076BE9FA48DFB40002FDD24147703410014AE47B18DFB400077BE9F28770341009A9999F98DFB40809F1A2F8977034100343333038EFB40801804568C77034100986E121B8EFB4080F3FDD49277034100240681258EFB400079E92695770341003E0AD7318EFB4000B072689777034100BC7493488EFB40007F6ABCA77703410078E9264D8EFB4000333333AE770341009CC420548EFB4000D9CEF7B077034100388941848EFB4080749318E1770341000E2DB26B8EFB4080BE9F1A38780341002C8716718EFB4080DE710A3A7803410060E5D0768EFB4080EB51B83B78034100285C8F9A8EFB4080749318457803410038F8C2A58EFB4080FC87744778034100E2C798B18EFB400062A1D64878034100365EBABD8EFB40007D3F354978034100C876BEC18EFB4080C2F528497803410040355EFC8EFB4000508D972A7803410040355E208FFB40807493182778034100C876BE978FFB4000508D975A78034100C876BEB38FFB4080621058787803410052B81EAB8FFB4000B4C87698780341003889413690FB4080C64B37C678034100C876BEAF91FB4080EB51B88F7903410048E17AD091FB400004560EBA790341007A14AEED91FB4080BC7493D479034100726891F391FB40002B8716D979034100C420B01C92FB4000D9CEF7F379034100DE02092392FB4080984C95F7790341005EBA492A92FB4080C876BEFA79034100287E8CA792FB4080F7E4612A7A034100CAE53F2B93FB40807D1D38837A03410040355E5C93FB4080D1915C947A0341001283C07293FB4000DF4F8DC17A0341007E1D387993FB4080E7FB29C57A034100E4A59B8093FB4080F1D24DC87A0341006CE7FBB593FB4000000000DC7A034100C0CAA1ED93FB4000022B87F47A0341000000005694FB4000D9CEF7087B0341009A99996F94FB4000D122DB147B034100AC1C5A9A94FB400004560E1D7B034100643BDF9395FB40803D0AD75B7B0341003AB4C89A95FB4000ABCFD55C7B034100AED85F1696FB40004703F8877B0341005227A01997FB40807D1DB8B07B034100780B24AA97FB4000A7E848AB7B034100180456EE97FB400083C0CAC07B034100FA7E6AF797FB400010E937C37B0341009A2A180198FB4080D93DF9C47B0341007AA52C0B98FB4000265305C67B034100D881731598FB400062A156C67B034100A245B61F98FB40801E85EBC57B03410022DBF96A98FB400052B81EC07B034100A69BC4A298FB400023DBF9C27B03410076BE9FBA98FB4080C64B37BC7B0341009CC420C099FB4000D9CEF7C37B034100941804149AFB4000B4C876E07B0341004A0C02D39AFB4080749318FF7B034100240681199BFB4080C64B37F27B034100941804659BFB4080E5D022C27B034100EE5A428E9BFB4080D93DF9BA7B0341004A0C02A39BFB4080EB51B8B97B0341004A0C02779CFB4080C64B37E27B03410040355EAC9CFB408039B4C8E67B034100A69BC4BA9CFB4080E92631E47B034100F0A7C6BD9CFB4000DBF97EE37B034100643BDF4F9DFB4000508D97A57B0341005AF5B9629DFB4080A1D6B4A67B0341005E4BC8629DFB4000A245B6A67B034100AE47E1889DFB4000D9CEF7A87B0341000E2DB2999DFB408038D6C5B67B034100388941E09DFB4080EB51B8F07B03410076BE9FD09EFB40803D0AD7FA7B03410052B81E1B9FFB4080749318DD7B034100C876BE579FFB4080621058E37B03410076BE9FB89FFB40803D0AD7FA7B0341000AD7A3F89FFB40809BC420F67B034100D022DBFF9FFB4000CFF753F47B034100703D0A1EA0FB4080AD69DEDB7B03410000000020A0FB40004703F8DB7B034100FEB27BC2A0FB40005A64BBDF7B034100388941E2A0FB4080621058F77B03410038894100A1FB4080C64B37F67B034100E6AE2533A1FB4000AB3E57E27B034100AC3E57A6A2FB40808D06F0EA7B034100643BDF2DA3FB4000D9CEF70D7C034100782D2132A3FB400071AC0B0E7C034100EE7C3F6FA3FB4080C64B370F7C034100006F81A3A3FB408079C7291D7C0341008A4160CDA3FB4080621058287C034100384547EBA3FB4080499D002A7C0341003E7958EBA3FB40006EA3012A7C0341002406813BA4FB4000B4C8762E7C034100388941ACA4FB4000D9CEF7587C0341001283C00EA5FB4080999999647C03410000000066A5FB4000B4C876627C034100F6285CA1A5FB40803D0AD76A7C03410076BE9F72A6FB4080C64B379F7C034100AE47E1DAA6FB4000D9CEF7BE7C0341004CC80744A7FB40806B9AF7D67C0341003E79588DA7FB400062A1D6037D0341005A423E8DA7FB40009A081B047D03410076BE9F84A7FB4000508D971A7D034100C0CAA149A7FB40808FC2F5427D0341001283C038A7FB4000D9CEF7627D0341005A643B4BA7FB4000508D97767D034100C876BEB3A7FB4000D9CEF7AF7D03410040355E34A8FB4080749318DE7D0341009A081B4BA8FB408047E17AF17D03410052B81E63A8FB40808FC2F5057E034100B003E76FA8FB4000B29D6F1F7E034100643BDF73A8FB4080621058277E034100AE47E110A9FB4000508D978D7E034100240681C9A9FB4080C64B37E97E03410024068129AAFB4080EB51B81A7F034100643BDF43ABFB4000D9CEF7B27F03410048BF7DA5ABFB4000068195C77F034100E8482E75ACFB400060764F2F80034100CEF75375ACFB40003108AC2F800341005C6DC577ACFB400054E325348003410086A7577BACFB4000B22E6E38800341006CE7FB7FACFB400021B0723C80034100F6285CA5ACFB400021B0725880034100749318AAACFB4080E3A59B5B800341008C6CE7AFACFB40007B14AE5E80034100FED4781DADFB4080ED7C3F92800341001283C02AADFB4080EB51B892800341003889416AADFB4080621058B4800341003208AC6CADFB4000022B87B7800341009418049CADFB408016D9CECD8003410054E3A5B5ADFB4080976E12E280034100240681BBADFB4080E17A14E680034100022B8706AEFB40806ABC74128103410040355E56AEFB40803D0AD72781034100F4FDD484AEFB4000560E2D5D810341003AB4C89EAEFB4000022B876C81034100448B6CC7AEFB4080E926318681034100F6285CC9AEFB4000AC1C5A8781034100CAA145F0AEFB40808B6CE79D81034100AE47E136AFFB4000D9CEF7A68103410040355E7EAFFB4080438B6CE0810341000E2DB2AFAFFB40804160E5F68103410022DBF9D8AFFB40008195430D820341006891ED10B0FB4000022B872B820341009CC42068B0FB40807493185382034100C2F52868B0FB40806E12835D82034100D478E93CB1FB4080C64B37DB8203410068446945B1FB40006B2B76DF82034100C66D344FB1FB4080B20CF1E2820341007081045AB1FB4000BB278FE582034100E04F8D65B1FB40807FD93DE782034100DA1B7C71B1FB400045D8F0E782034100DA1B7C7DB1FB40809CA2A3E782034100C64B3789B1FB4080621058E682034100A401BCB2B1FB4000D49AE6DF82034100361AC01CB2FB408062105812830341000AF9A087B2FB4000508D9733830341009EEFA7A8B2FB40005C8FC24383034100666666B4B2FB40803789415383034100C876BE05B3FB4000508D977183034100DCF97E4AB3FB4080621058928303410072689147B3FB40800E2DB2AE830341009418045EB3FB4000508D97C183034100D24D625EB3FB4000AC1C5ADA83034100FED4785FB3FB4080621058E083034100C2F52862B3FB4080999999E583034100CAA1457AB3FB4000AC1C5A09840341001058397FB3FB4080B88D060F84034100BA490C86B3FB4080E9263114840341005C8FC28DB3FB40004E62101984034100A4703DE8B3FB4000B4C8762A840341006619E216B4FB408068916D228403410052B81E45B4FB4000B4C8761A84034100E4839E70B4FB40803789C121840341002041F17BB4FB4000CE19512884034100C876BE99B4FB40807493183F840341007C832F9BB4FB4000CD3BCE41840341004A0C02CBB4FB4000508D979B84034100FEB27BD6B5FB4080C3D32BD284034100643BDF23B6FB4000D9CEF7E18403410038894130B6FB4080749318F68403410040A4DF2FB6FB40003D9B55FB8403410076BE9F2EB6FB4080EB51B80C8503410038894102B6FB40806210585985034100DCF97E1AB6FB4080C64B376985034100C0CAA16BB6FB4000508D977A850341001C0DE087B6FB40003411B68F8503410052B81EA1B6FB4000508D97A285034100FC1873A1B6FB40008F5374B785034100240681A1B6FB4080C64B37BB850341005839B470B6FB4000295C8FE1850341004260E58CB6FB4000AE47E10D86034100B81E8591B6FB408047E17A1386034100A245B6F1B6FB40001FF4EC1086034100C876BE3BB7FB4000D9CEF70E86034100703D0A75B7FB4080ED7C3FDE850341007C3F358AB7FB4080EB51B8D585034100643BDFCDB7FB4080EB51B8C98503410040355E0CB8FB4000508D97B385034100EE7C3F23B8FB40000AD7A394850341007493183AB8FB4000295C8F8785034100423EE83DB8FB400033C4B18585034100CAE53F53B8FB40806210588A850341003EE8D98AB8FB4080270F8B6E85034100B81E85A3B8FB40806210586785034100C286A7AAB8FB4080C117A66585034100EE7C3FB2B8FB408004C58F6485034100749318BAB8FB400008AC1C64850341006CE7FBC1B8FB40803BDF4F6485034100FED478F7B8FB40804160E567850341006ABC7421B9FB4000D578E982850341004A0C0233B9FB4080C64B378E85034100AC1C5A60B9FB4080E4F21F87850341008A416069B9FB4080EB51B8858503410076BE9F8AB9FB408074931863850341001A2FDDAEB9FB4080B6F3FD4685034100E8FBA9BBB9FB408036AB3E4285034100361AC084BAFB40802BF69759850341006C567D91BAFB400020D26F568503410024068199BAFB4080EB51B85785034100083D9BADBAFB4080B437784F85034100CAE53FDBBBFB40807D1DB8048503410094F606F5BBFB4000E56121F58403410010583982BCFB4080E7FBA9CC84034100FCA9F1C4BCFB4000295C8FB584034100941804E2BCFB4000B072689D84034100B6F3FDECBCFB4000508D979B84034100388941EEBCFB4000A4703D938403410060E5D034BDFB40000AD7A358840341005C8FC243BDFB40806CE7FB2E840341008C6CE73FBDFB40005A643B2B8403410082734664BDFB4080764F1E1A840341002A3A928FBDFB4080A44EC005840341008816D9EABDFB408016D9CEDA830341001C5A6421BEFB40005A643BD4830341006ABC7463BEFB4000D122DBB5830341000A46257CBEFB4000C58FB1A683034100F6B9DAA1BEFB4080EA73B594830341001461C3A7BEFB40805BB1BF9383034100448B6CAFBEFB4080C2F5289383034100365EBADDBEFB4000DBF97E918303410038894128BFFB40806210589683034100EE7C3F63BFFB40803D0AD794830341008A4160BFBFFB40801804569883034100D0B359DCBFFB40803EE85987830341008A4160F9BFFB4080621058768303410014D044F6BFFB400097FF907383034100B81E85DDBFFB4080438B6C5D8303410040355EFEBFFB40002B87164B8303410082954303C0FB4080EB51B84883034100DCF97E0AC0FB4080C64B37468303410074931866C0FB40803D0AD72C830341001283C0AAC0FB4000B4C8762883034100643BDFC7C0FB4000508D970C83034100240681D3C0FB4000508D97EF820341002C8716D3C0FB40806891EDED8203410090C2F5D4C0FB40005839B4EB82034100240681D5C0FB4080931804EB82034100068195E7C0FB4000068195D382034100EEEBC0EAC0FB4080EE5A42D0820341008C6CE7EEC0FB4080348037CD82034100B29DEFF3C0FB4000022B87CA820341000E2DB21DC1FB4000B07268B78203410080B74023C1FB4080C64B37B5820341001C5A6429C1FB40806ABC74B3820341006CE7FB2FC1FB40800C022BB282034100365EBA4DC1FB4000068195AD8203410052B81EC9C1FB4080C64B37AE82034100240681E7C1FB4000508D978B820341002C8716E9C1FB400085EB517D82034100726891EDC1FB40008716D97A82034100388941F2C1FB4000508D977882034100285C8FF4C1FB4080C420B077820341009418041AC2FB4000AAF1D26982034100643BDFADC2FB408060E5D0258203410042F163B2C2FB400023DBF92382034100605452B7C2FB4080B1BF6C2282034100508D97BCC2FB40809F1A2F2182034100A245B613C3FB408016D9CE0F82034100EE7C3F19C3FB40804160E50E820341009A999971C3FB4000A4703D0382034100DACEF79DC3FB40003108ACFB81034100C0CAA1ABC3FB4000FED478FD810341008A4160F7C3FB4080621058F981034100D24D6228C4FB4080E17A14D881034100E04F8D8BC4FB4000B07268BA810341006ABC7493C4FB4000DF4F8DB881034100F6285C9BC4FB408047E17AB781034100D24D62E6C4FB4000355EBAB08103410052B81E3DC5FB408045B6F3A8810341006210589BC5FB408066666682810341008048BFA0C5FB4080DA1B7C80810341002E90A0A6C5FB4000FDF6F57E81034100643BDFACC5FB4000D122DB7D81034100F6285CB3C5FB4080E926317D81034100F2D24D1CC6FB40002FDD247681034100E27A1454C6FB40806CE7FB708103410076BE9F5CC6FB40809999997081034100BC749364C6FB4000B29DEF708103410014AE4785C6FB40007B14AE7381034100240681FBC6FB400085EB5167810341000E9C3303C7FB4080AF9465428103410000000070C7FB40004703F83F810341005227A0D9C7FB40004703F83F810341008AB0E117C8FB4080E61D273B8103410040355E3CC8FB40000000007C8103410062C3D342C8FB4000B22EEE8181034100A8A44E35C8FB40809F1AAFA7810341002041F109C8FB4000CB10C72082034100CEAACF68C7FB4080C2F5280C830341005817B768C7FB4000849E4D0C8303410046477268C7FB408059F5B90C830341002ACB1019C7FB408048BF7D9083034100363CBD18C7FB40002A3A1291830341007CD0B318C7FB40002FDD24918303410092ED7CE8C6FB400023DB79EF83034100D656EC15C7FB4080635D5CF583034100CAE53FE3C6FB40807D1DB83C84034100567DAEEAC6FB4000EF38C57884034100DA8AFDE4C6FB4080423EE8858403410006F016CCC6FB4080711B0DEB84034100D8A370D5C6FB4000287E8C5685034100F675E0F4C6FB4080469476CC8503410064EE5A1AC7FB4000AB3ED72E86034100506B9A3DC7FB40805374A46086034100E2E9955CC7FB4080D5E76A8C860341005EBA49B6C7FB4000AAF1D2DF86034100643BDFE1C7FB40801283C0FC860341005A423EEAC7FB4000D49A660187034100749318F4C7FB40803789410587034100C286A7B4C8FB4080B20CF14387034100000000D0C8FB40806210585687034100F48E53DEC8FB4000DCD78151870341008816D91EC9FB40806E128366870341002E6EA325C9FB4000B4C876688703410060E5D02CC9FB40002731086A8703410046B6F33DC9FB40000AD7A36287034100FED4785DC9FB40806E128357870341002EB29DB7C9FB4080C420B037870341001A2FDDE0C9FB4080BE9F1A0D8703410094F606FCC9FB40004E6210F186034100361AC014CAFB40807D1DB8E886034100361AC074CBFB40004703F8BF85034100CAE53FA3CCFB40807D1DB8E0830341007A58A88DCCFB40004FAF146A820341006CE7FB93CCFB40002B87166482034100923A01BECCFB40808D97EE0C820341009CC420EDCCFB4080F606DFC2810341008CB96BFBCCFB40800E9CB3B781034100FCA9F108CDFB4080E5D022AD8103410054E3A515CDFB4000333333A381034100DE24065FCDFB4000CFF7537D81034100A01A2F45CEFB408062105837810341006054524BCEFB408013D0C435810341008816D951CEFB40802F4CA634810341002E90A058CEFB400093A9023481034100B81E855FCEFB40801A2FDD33810341000E2DB201D0FB40806CE7FB3981034100D2915C70D0FB408050FC985381034100EE7C3F31D1FB408037894180810341004E621088D1FB4000D9CEF79D810341002063EEC1D2FB40804F1E16DD81034100E63FA414D3FB4000B7D1800282034100A69BC462D3FB4000CDCCCC378203410040355E18D4FB4000A4703DB7820341005A643B95D4FB4000250681B583034100EE7C3FA7D4FB40000AD7A3BD830341009C33A2D4D4FB40009D1125D28303410070CE88DCD4FB408009F9200884034100BC7493B0D4FB4000F6285C0B840341004260E5ACD4FB4000022B877084034100C2F528BED3FB40807268912C85034100B6F3FDE6D1FB408014AE4792870341005EBA490ED0FB4080703D0A128A0341008A4160B3CFFB4000DBF97E728A0341007C3F35B0CFFB4000D7A370768A0341002C871697CFFB40004E62109C8A034100B4C87694CFFB4080C64B37A18A0341006ABC7493CFFB4000AAF1D2A58A034100FCA9F14ECFFB4080BC7493938D034100D8A3704FCFFB40007B14AE998D03410074B51565CFFB40000F0B35F08D034100CAE53F43CFFB4080621058968E034100361AC0DCCFFB40004703F8DB8F034100CAE53F7BD1FB40807D1DB80090034100BA490C42D2FB4000E3369A878F034100285C8F5AD2FB4000333333808F034100E4A59B98D3FB4000819543EC8E034100F2D24D9AD3FB4000022B87E98E034100365EBAC1D3FB40005839B49D8E034100F0A7C637D4FB4000DF4F8D5C8E03410074931856D4FB4000B29DEF4B8E034100E04F8D61D4FB4000A8C64B3C8E0341009C559F7FD4FB4080C807BDFF8D0341001E85EB97D4FB408016D9CECE8D03410048E17AD0D4FB400083C0CAAF8D034100C420B038D5FB40005A643B0D8D034100E04F8D53D5FB4080EB51B8BA8C0341008C6CE755D5FB4000CFF753B68C03410086EB5172D5FB4000819543918C03410012A5BDB8D5FB4080AD695E238C034100AED85FEED5FB40802BF697F58B03410000000008D7FB40807D1DB87C8A0341005227A021D7FB40807D1D380B8A03410088635D23D7FB40005F984C058A0341001283C088D7FB408060E5D06E890341002EB29D89D7FB40000AD7A36D8903410022DBF9A2D7FB40809999994D890341004C3789BDD7FB4000AC1C5A2189034100AE47E12AD8FB4080438B6C7788034100A8C64B2BD8FB40801283C076880341008295433DD8FB40007B14AE5788034100A4703D3ED8FB40800C022B5688034100DACEF75ED8FB4000DE020929880341005227A071D8FB40807D1D381B88034100CAE53FF3D8FB4080621058EE87034100361AC04CD9FB4080621058DE870341009C33A284D9FB4080D191DCD6870341001E3867B4D9FB4080726811D48703410000000008DAFB4080621058DE870341005227A099DAFB40004703F80388034100CAE53F03DBFB40807D1DB83C88034100361AC084DBFB40802BF6976188034100AED85FBEDBFB4080621058A688034100AED85F06DCFB40807D1D38CB88034100AED85F2EDCFB400010E9B7108903410000000078DCFB40802BF6975D89034100C07D1D8EDCFB40005D6D457489034100C420B090DCFB40002FDD24998903410078E92693DCFB40002FDD24B189034100CCCCCC8EDCFB40000AD7A3C38903410078E92693DCFB4080EFA7C6DC89034100FA7E6AEEDCFB4080B81E85718A0341003AB4C8F0DCFB408016D9CE748A034100BA490C06DDFB40803BDF4F8E8A034100285C8F30DDFB4000000000D28A0341005C8FC231DDFB40801283C0D38A034100107A363ADDFB4080522720DF8A03410000000050DDFB40802BF697258B03410000000050DDFB40806210586A8B03410000000058DDFB40004703F8BB8B034100000000A8DDFB40802BF697098C034100CAE53F33DEFB40806210584A8C034100CAE53F8BDEFB40004703F8878C034100361AC004DFFB40807D1DB8BC8C034100AED85F86DFFB40802BF697DD8C0341005227A031E0FB4080621058128D034100CAE53FB3E0FB4080621058168D0341004850FCD4E0FB408013D044F98C034100DCF97E44E1FB408016D9CE148D034100A835CD5CE1FB4000454772FD8C0341002631084AE2FB4000B07268198C0341002E6EA318E3FB4080F2B0D0528B03410076711B03E4FB4000D7A370718A0341005227A0D9E4FB4080621058D289034100AED85FE6E5FB40004703F82F89034100CAE53FEBE6FB4080621058B6880341005227A041E8FB40004703F81788034100361AC0E4EAFB40807D1DB83087034100AAF1D2A6F5FB40003333335F830341005227A0DEF6FB40806210588084034100361AC0D9F9FB4000470378A2860341001C0DE0E4FBFB40802BF617B4880341001C0DE080FDFB400010E937658A034100E4F21FBBFFFB40004703F8CE8C0341000000007701FC40807D1D38908E0341001C0DE0A202FC400010E9374990034100E4F21F2004FC408062105872910341005227A08C06FC40004703F803930341005227A09709FC400010E93726950341006E3480630BFC4000470378E796034100E4F21FDF0CFC40807D1DB8C898034100E4F21F8A0EFC40004703F8D99A0341001C0DE00510FC40802BF617AB9C0341001C0DE09111FC40806210585C9E0341000000007E13FC40807D1DB805A0034100E4F21F5C15FC40004703F8DEA0034100E4F21FFB16FC40004703F857A10341000000008B18FC40004703F878A1034100CAE53F0A1CFC40807D1D38F3A1034100AED85F591FFC40806210587DA2034100CAE53F3822FC400010E9370FA3034100CAE53FD724FC40004703F898A3034100AED85F6427FC40807D1DB8EAA40341001C0DE05128FC4080621058F3A5034100E4F21FCF27FC40807D1D380BA7034100E4F21F9C26FC40802BF69732A8034100AED85F2A26FC4000470378E2A803410092CB7FA726FC40806210D80AAA034100E4F21F8427FC40802BF69773AB0341005227A09028FC4000470378ECAC0341001C0DE0BD29FC408062105815AE0341001C0DE0CC2BFC40802BF69796AE0341005227A05B2FFC40004703F848AF034100361AC05B30FC40802BF69749AF034100AED85F7931FC40806210584AB0034100361AC06632FC40802BF61763B10341000000008333FC40004703F8F3B2034100CAE53F9F34FC40806210D884B40341000000002C36FC40004703F8EDB50341005227A07738FC40807D1DB8CFB7034100361AC0343AFC40004703F808B90341001C0DE0823BFC40806210D8E1B9034100AED85F2D3FFC40004703784CBC0341001C0DE02940FC40807D1D38BDBD0341001C0DE0C540FC40806210D85DBF034100E4F21FEF41FC40806210D82EC20341005227A0C842FC40807D1DB8D7C4034100CAE53F2443FC40807D1D3898C6034100AED85F6F43FC40802BF69790C803410092CB7FE943FC40807D1D38F9CA0341001C0DE02444FC40802BF697D9CC0341001C0DE03044FC40806210D879CE0341000000000B44FC40802BF617D2D00341001C0DE01544FC4080621058E2D2034100E4F21F7044FC40806210D83AD50341001C0DE04944FC40802BF617BBD7034100E4F21F2444FC40806210580BDA0341006E34800E44FC40802BF6974BDC03410076029AD243FC4080BB96108CDD034101030000000100000033000000005A64BB067201410052B81E7538044100DE9307D2720141803BDF4F1F43044100BE9F9A8768014100BADA0A6A400441001B0D605C680141806210D833400441001B0D608B6701410010E9B7433F044100371AC0846601410010E9B7533E044100E5F21F26650141802BF697633D044100000000B8630141806210D8C13C04410000000078620141807D1D38843C04410000000038610141806210D87F3C044100000000F85F0141806210D8A03C044100000000B85E0141807D1D38FD3C044100000080165E0141802BF697633D04410030BBA7B25D01418017B751903D044100C66DB433590141808D0670613C044100000000B8590141802BF6979A3B0441001B0D607C5A014100470378933A0441001B0DE0EC5A014100470378A339044100371A402D5B014180621058B338044100371A40395B014180621058C337044100000000F85A0141004703781637044100000080465A0141807D1D38E335044100000000B8590141807D1D381B3504410000008047590141802BF617033404410000000081590141004703F81233044100000000F85A0141802BF6173F32044100000000385C0141004703784532044100000000785D0141004703F87232044100000000B85E0141806210D8D4320441001B0DE07C5F0141004703F81233044100000000F85F0141807D1DB83F3304410000000038610141802BF697913304410000000078620141004703F8BF33044100000000B863014180621058F533044100000000F8640141802BF6973E3404410000000038660141807D1D38A834044100000000DC660141802BF617F33404410000000078670141806210D83935044100E5F21F97680141807D1D38E335044100000080E1690141807D1D38D336044100E5F21FD26A014180621058C337044100000000386B0141802BF6973B38044100E5F29F946B014180621058B338044100000000786C0141802BF6979F39044100000000B86D0141004703785C3A044100371A407A6E014100470378933A044100000000F86E0141807D1D38AE3A0441001B0DE03370014100470378933A04410000000078710141806210D8B639044100E5F29FE871014180621058B3380441005A64BB067201410052B81E753804410103000000010000003800000000986E12C672004180764F1E653B034100000080B7700041802BF697813B034100E5F21F8D6C0041802BF697813B034100000080536B0041806210D89038034100000000886D0041004703786135034100371AC078700041806210585132034100ED0D3E39710041808B6C677730034100894160FC720041800C022B5A2F0341001D5A64FF7200418039B4C8702F0341006A4DF3207300410008AC1C5C2F0341009F3CAC2673004100068195582F034100E5F21F4873004180863856CC2F0341000AD7234873004180D3BC63CC2F03410046B6F3A673004100D9CEF79232034100C1CAA1AD73004100AE47E1A13203410083C0CAB8730041007D3F359F320341004C3789BE73004100250681B632034100C3F528C7730041806ABC74D932034100986E120074004180F3FDD4CB320341004A0C020D7400418091ED7CEA32034100EE7C3F2E74004100FED478DF32034100E2E9152F740041802EFFA1DE32034100666666417400410085EB51CC320341000681955174004180E7FBA9F232034100832FCC52740041805BB13F333303410077BE9F53740041806210585F33034100083D1B7874004180D0B3598333034100621058B374004100AAF1D2BD33034100DD2406B5740041808B6CE7C0330341006ABC742175004100508D9731340341002B87165B7500410052B81E613403410077BE9FCB75004100D9CEF7CB34034100C520B0F476004100AE47E1CE35034100B29DEF96770041806210586A360341001F85EBB0770041002FDD246E360341001283C0E777004180C64B379A360341006210585A780041005E4B48F63603410000000073780041807493180A37034100DBF97E1079004180EB51B80C370341009CC420F679004100AC1C5A10370341002BF6977F7A00418007CE991237034100371AC0B47A0041807D1DB82037034100348037D07A0041008D28ED28370341009A771CCD7A00418058A8352F370341001B2FDDCC7A004180EB51B82F37034100C74B37CF7A004180E17A1430370341005DDCC6D97A004100EBE2B631370341001B0DE0017B0041004703F89537034100DBF9FECA79004180348037F938034100E5F29F11790041807D1D38AF38034100448B6CCF780041001973D76A390341009CC420E677004100355EBA5039034100E0BE8E42770041805CFE433F3903410001DE8242770041803789413F39034100E5F29FFD730041806210589A38034100986E12C672004180764F1E653B03410103000000010000000500000000BC74930A70FA40000BB5A668830341008204C5A570FA4000B072E80E8303410030BB271D76FA4080703D8AEC84034100361AC09E70FA40004703F8B683034100BC74930A70FA40000BB5A6688303410103000000010000000F00000000C58F312772014100098A1F2731044100371AC032720141807D1DB852310441001B0DE061720141004703F84F320441006E34008C720141004703F8F33204410000000030730141807D1D389E34044100C9E5BF5B73014180EB513844350441008941606F730141004703F86F350441005BB1BF5573014100A2B4B79535044100B1E1E94A730141804DF38EA135044100E5F29FFE720141802BF617F334044100000000B8720141806210D83834044100371A4074720141004703F812330441001B0D6045720141004703F822320441001B0DE024720141004703F83231044100C58F312772014100098A1F2731044101030000000100000008000000001B0DE09EA600418068916D61120341001B0DE09EA60041806210583313034100CC7FC804A50041801B0DE00E130341001B0D6006A50041807D1D380B13034100371A402FA50041807D1D38FF1203410000000070A5004180621058D6120341001B0D60E6A50041802BF697B9120341001B0DE09EA600418068916D61120341010300000001000000090000000096B20C2FE8FA40803D0AD7808B034100CAE53FFEE6FA4000470378B98B034100CAE53FCEE5FA40807D1DB8B08B034100361AC07EE3FA400010E9375F8B034100A679C73FE2FA40005C2041428B0341003208AC70E2FA4000CB10C73F8B03410052B81E37E5FA4000FED478698B034100C0CAA1ABE7FA40803D0AD7808B03410096B20C2FE8FA40803D0AD7808B034101030000000100000009000000006C567DD40CFB4080832F4C5A8A034100A69BC4E80CFB4000819543578A034100C876BE7D0DFB4000022B87498A034100000000820EFB4080749318238A0341006EA3018B0EFB4080CF66D51F8A034100CAE53FAB0EFB40802BF697258A03410046B6F3AB0FFB400086C9548E8A0341006E3480E60EFB40802BF6177A8A0341006C567DD40CFB4080832F4C5A8A0341');
INSERT INTO grondwaterlichamen_new VALUES (17, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                                <BR> SS_1000_GWL_2 Paleoceen Aquifersysteem, Gespannen<BR> SS_1300_GWL_1 Kolenkalk, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (18, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                              <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter        <BR> SS_1300_GWL_4 Sokkel + Krijt Aquifersysteem, Gespannen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
INSERT INTO grondwaterlichamen_new VALUES (40, '     <BR> BLKS_0400_GWL_2M Oligoceen Aquifersysteem, Gespannen                <BR> BLKS_1100_GWL_2M Krijt Aquifersysteem, Gespannen                                    <BR> MS_0100_GWL_1 Quartaire Aquifersystemen, Freatisch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
INSERT INTO grondwaterlichamen_new VALUES (19, '                                   <BR> CVS_0100_GWL_1 Dun Quartair dek boven op Paleogeen klei, Freatisch                              <BR> SS_1000_GWL_1 Paleoceen Aquifersysteem, Gespannen, Depressietrechter      <BR> SS_1300_GWL_3 Sokkel + Krijt Aquifersysteem, Gespannen, Depressietrechter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
INSERT INTO grondwaterlichamen_new VALUES (21, '                         <BR> BLKS_1100_GWL_2S Krijt Aquifersysteem, Gespannen                                  <BR> MS_0100_GWL_1 Quartaire Aquifersystemen, Freatisch<BR> MS_0200_GWL_1 Kempens Aquifersysteem, Freatisch, Plaatselijk Semi-Freatisch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
INSERT INTO grondwaterlichamen_new VALUES (22, '                           <BR> CKS_0200_GWL_1 Centrale zanden van de Kempen, Freatisch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