#!/usr/bin/env Rscript

files.sources <- list.files('/home/facet/R/functions/', full.names = TRUE)
invisible(sapply(files.sources, source))

library(argparser, quietly=TRUE)
suppressMessages(library(dplyr, quietly=TRUE))
library(logger, quietly=TRUE)
library(raster, quietly=TRUE)
library(sf, quietly=TRUE)

# Build up our command line argument parser
p <- arg_parser("Calculate distance from points as static gridded data")

p <- add_argument(p, "input_file", help="Path to a text file containing the point data as csv. Requires columns named lon and lat.")
p <- add_argument(p, "--mask_shp", help="Path to a shapefile used to mask the resulting distance grid.", default=NA)
p <- add_argument(p, "--lon_col", help="Character; variable name containing longitude info.", default='lon')
p <- add_argument(p, "--lat_col", help="Character; variable name containing latitude info.", default='lat')
p <- add_argument(p, "--x_min", help="Numeric; indicating desired minimum longitude of output grid.", default=-180)
p <- add_argument(p, "--x_max", help="Numeric; indicating desired maximum longitude of output grid.", default=180)
p <- add_argument(p, "--y_min", help="Numeric; indicating desired minimum latitude of output grid.", default=-90)
p <- add_argument(p, "--y_max", help="Numeric; indicating desired maximum latitude of output grid.", default=90)
p <- add_argument(p, "--res", help="Numeric; indicating desired spatial resolution (in decimal degrees) of output grid.", default=0.08)
p <- add_argument(p, "output_file", help="Path to output netCDF file", default='output.nc')

## parse the arguments
args <- parse_args(p)
for (i in 1:length(args)){
  logger::log_info(paste0('Args include ', names(args)[i],': ', args[[i]]))
}

## detect input_file extension and read it
ext <- tools::file_ext(args$input_file)
if (ext == 'csv'){
  df <- data.table::fread(args$input_file)
} else if (ext == 'shp'){
  df <- sf::st_read(args$input_file) %>% as.data.frame()  
} else{
  logger::log_error('Input file type must be either csv or shp.')
}

## change df names to standard
names(df)[which(names(df) == args$lat_col)] <- 'lat'
names(df)[which(names(df) == args$lon_col)] <- 'lon'

## do the calculation
dist_raster <- distance_from_points(df, args$mask_shp, x_min = args$x_min, x_max = args$x_max,
                                    y_min = args$y_min, y_max = args$y_max, res = args$res)

## save as netCDF
writeRaster(dist_raster, args$output_file, format='CDF')
log_info('Distance grid saved to {args$output_file}')

