#!/usr/bin/env python

import argparse
import itertools
import pathlib

import pandas as pd

from facet_shared.logger import logger
import hycom_common as hycom


class ParsedArguments(argparse.Namespace):
    input_csv: pathlib.Path
    output_csv: pathlib.Path


def main():
    args = parse_arguments()

    df = pd.read_csv(args.input_csv, parse_dates=[args.date_col])

    date = str(df[args.date_col].dt.date.iloc[0])

    hycom_ds = hycom.hycom_for_date(date, chunk=False, logger=logger)

    output_df = pd.concat(
        [df, hycom.extract_by_apply(hycom_ds, df, logger=logger, lat_col=args.lat_col, lon_col=args.lon_col)], axis=1
    )

    output_df.to_csv(args.output_csv, index=False)


def parse_arguments() -> ParsedArguments:
    parser = argparse.ArgumentParser(
        description="Enhance a given CSV file with HYCOM data"
    )
    parser.add_argument("input_csv", type=pathlib.Path, help="CSV for a single date")
    parser.add_argument(
        "output_csv", type=pathlib.Path, help="CSV to output enhanced data to"
    )
    parser.add_argument(
        "--date_col",
        type=str,
        default="date",
        help="String representing column name containing date",
    )
    parser.add_argument(
        "--lat_col",
        type=str,
        default="lat",
        help="String representing column name containing latitude data",
    )
    parser.add_argument(
        "--lon_col",
        type=str,
        default="lon",
        help="String representing column name containing longitude data",
    )
    return parser.parse_args()


if __name__ == "__main__":
    main()
