#!/usr/bin/env python
"""
Given a date (YYYY-MM-DD) and an output .nc, download and export HYCOM data
"""
import argparse
import sys
import pathlib
from tempfile import TemporaryDirectory

from dask.diagnostics import ProgressBar
import xarray as xr

import hycom_common as common
from facet_shared.logger import logger


class Arguments(argparse.Namespace):
    date: str
    output_nc: pathlib.Path
    quiet_missing: bool


def main(args: Arguments):
    logger.info(f"Arguments: {args}")

    try:
        ds = common.hycom_for_date(args.date, logger=logger)
    except common.MissingDateError as e:
        if args.quiet_missing:
            logger.warning(f"{args.date} was missing from dataset, exiting quietly")
            sys.exit(0)

        else:
            raise e
    # with TemporaryDirectory() as temp_dir:
    # temp_dir_path = pathlib.Path(temp_dir)
    # temp_netcdf_path = temp_dir_path / "temp.nc"

    # temp_var_files = []

    # for var in ds.variables:
    #     if var not in ds.coords:
    #         var_path = temp_dir_path / f"{var}.nc"
    #         temp_var_files.append(var_path)

    #         logger.info(
    #             f"Downloading variable {var} before calculating derived data"
    #         )

    #         var_file = ds[var].to_netcdf(var_path, compute=False)
    #         with ProgressBar():
    #             var_file.compute()

    # # temp_netcdf = ds.to_netcdf(temp_netcdf_path, compute=False)

    # # logger.info(
    # #     f"Downloading HYCOM day to {temp_netcdf_path} before calculating derived data"
    # # )

    # # with ProgressBar():
    # #     temp_netcdf.compute()

    # # ds = xr.open_dataset(temp_netcdf_path)
    # ds = xr.open_mfdataset(temp_var_files)
    ds = common.hycom_vars(ds, logger=logger)

    logger.info(f"Outputting to {args.output_nc}")

    netcdf = ds.to_netcdf(args.output_nc, compute=False)

    with ProgressBar():
        netcdf.compute()


def parse_arguments() -> Arguments:
    parser = argparse.ArgumentParser(description="Download HYCOM NetCDF")
    parser.add_argument("date", help="Date in YYYY-MM-DD to download", type=str)
    parser.add_argument("output_nc", help="Path to output data to", type=pathlib.Path)
    parser.add_argument(
        "-q",
        "--quiet_missing",
        help="Quiet output when there is missing data",
        action="store_true",
    )

    return parser.parse_args()


if __name__ == "__main__":
    args = parse_arguments()
    main(args)
