#!/usr/bin/env Rscript

## we need some custom functions we've written so have to figure out how to load those in rocker
files.sources <- list.files('/home/facet/R/functions/', full.names = TRUE)
invisible(sapply(files.sources, source))

library(argparser, quietly=TRUE)
library(logger, quietly=TRUE)
suppressMessages(library(raster, quietly=TRUE))
suppressMessages(library(ncdf4, quietly=TRUE))
library(oce, quietly=TRUE)
library(snow, quietly=TRUE)
library(stringr, quietly=TRUE)

# Build up our command line argument parser
p <- arg_parser("Calculate isothermal layer depth and buoyancy frequency from input NetCDF")
p <- add_argument(p, "in_dir", help="path containing folders of dates (YYYY-mm-dd) generated by chron")
p <- add_argument(p, "--test_bounds", help="Logical, use a much smaller set of spatial bounds for testing", flag = TRUE, default = FALSE)
p <- add_argument(p, "--deltaT", help="Numeric, desired temperature difference to be used in the isothermal calculation", default = 0.5)
p <- add_argument(p, "--parallel", help="Logical, if TRUE the calculations will attempt to proceed in parallel by automatically using whatever cores are available. If FALSE (default), only 1 node will be used per calculation.", flag = TRUE, default = FALSE)
p <- add_argument(p, "out_ild_dir", help="path to output resulting isothermal layer depth (ILD) NetCDF")
p <- add_argument(p, "out_n2_dir", help="path to output resulting buoyancy frequency (N2) NetCDF")

# parse the arguments
args <- parse_args(p)
for (i in 1:length(args)){
  log_info(paste0('Args include ', names(args)[i],': ', args[[i]]))
}

original_dir <- getwd()

## test with smaller spatial bounds?
if (args$test_bounds){
  args$xmin <- -70
  args$xmax <- -65
  args$ymin <- 25
  args$ymax <- 30
} else{
  ## full spatial bounds
  args$xmin <- -180
  args$xmax <- 180
  args$ymin <- -70
  args$ymax <- 70
}

## identify input file
files_in <- list.files(args$in_dir, full.names = F, recursive = TRUE)
## have to use list.dirs as cron files are created with linux "touch" command generating an empty file which R thinks is a directory
#files_in <- list.dirs(args$in_dir, full.names = F, recursive = F)
log_info('Input cron is {files_in}')

for (i in 1:length(files_in)){
  
  args$date <- files_in[i]
  
  ## use date and optional spatial bounds to get hycom temp & salinity
  ## GET TEMPERATURE
  tdir <- tempdir()
  fileout <- 'temp.nc'
  capture.output(myURL <- try(get.hycom(limits=as.numeric(c(args$xmin, args$xmax, args$ymin, args$ymax)), time=as.Date(args$date), download.file=T, dir=tdir,
                                  vars=c('water_temp')), silent = TRUE))
  log_info('For {files_in[i]} temperature the URL is {print(myURL)}')
  
  ## download the file, try a few times in case that helps
  for (bb in 1:5){
    d.try <- try(download.file(myURL, fileout, quiet=FALSE, method = 'curl'), TRUE)
    
    if(class(d.try) != 'try-error'){
      break
    } else if (class(d.try) == 'try-error' & bb == 5){
      ## remove the tempdir we created and all temp files it contains
      unlink(tdir, recursive = T)
      log_error(paste0('Getting HYCOM data from ', myURL, ' failed after 5 attempts.'))
      #stop()
    }
  }
  
  fileout2 <- 'salinity.nc'
  capture.output(myURL2 <- try(get.hycom(limits=as.numeric(c(args$xmin, args$xmax, args$ymin, args$ymax)), time=as.Date(args$date), download.file=T, dir=tdir,
                                   vars=c('salinity')), silent = TRUE))
  
  log_info('For {files_in[i]} salinity the URL is {print(myURL2)}')
  
  ## download the file, try a few times in case that helps
  for (bb in 1:5){
    d.try <- try(download.file(myURL2, fileout2, quiet=FALSE, method = 'curl'), TRUE)
    
    if(class(d.try) != 'try-error'){
      break
    } else if (class(d.try) == 'try-error' & bb == 5){
      ## remove the tempdir we created and all temp files it contains
      unlink(tdir, recursive = T)
      log_error(paste0('Getting HYCOM data from ', myURL2, ' failed after 5 attempts.'))
      #stop()
    }
  }
  
  # get var indices
  nc <- try(RNetCDF::open.nc(fileout), TRUE)
  ncnames = NULL
  nmax <- RNetCDF::file.inq.nc(nc)$nvars - 1
  for(ii in 0:nmax) ncnames[ii + 1] <- RNetCDF::var.inq.nc(nc, ii)$name
  dep.idx <- grep('dep', ncnames, ignore.case=TRUE) - 1
  dep <- as.numeric(RNetCDF::var.get.nc(nc, dep.idx))
  # close netcdf file
  RNetCDF::close.nc(nc)
  
  ## get data
  wtemp <- raster::brick(fileout, varname = 'water_temp', lvar=4) #* scale + offset
  sal <- raster::brick(fileout2, varname = 'salinity', lvar=4) #* scale + offset
  
  ## ISOTHERMAL LAYER DEPTH
  
  ## define a function to get ILD - 0.5 deg C
  fun <- function(x) {
    if (all(is.na(x))){
      NA
    } else{
      dep[which.min(abs(x[1] - x[2:40] - args$deltaT))]
    }
  } ## index of depth level closest to -0.5
  if (args$parallel){
    t1 <- Sys.time()
    beginCluster()
    ild <- calc(wtemp, fun)
    endCluster()
    t2 <- Sys.time()
    log_info(paste0('Parallel ILD calculation took ', round(difftime(t2,t1,units='mins'), 1), ' mins'))
  } else{
    t1 <- Sys.time()
    ild <- calc(wtemp, fun)
    t2 <- Sys.time()
    log_info(paste0('ILD calculation took ', round(difftime(t2,t1,units='mins'), 1), ' mins'))
  }
  
  
  ## BUOYANCY FREQUENCY
  
  ## define a function to approx N2 for top 200m (e.g. index 1:23) using temp and salinity
  s <- raster::stack(wtemp[[1:23]], sal[[1:23]])
  fun <- function(x) {
    if (all(is.na(x))){
      rep(NA, length.out=length(dep[1:23]))
    } else if(any(is.na(x[1:10]))){ ## if HYCOM is NA in top 25 m, not worth trying (and failing) to calc N2
      rep(NA, length.out=length(dep[1:23]))
    } else{
      oce::swN2(pressure = dep[1:23], sigmaTheta = oce::swSigmaTheta(x[24:46], x[1:23], pressure = dep[1:23]))#, referencePressure = median(depth[1:23], na.rm = TRUE)))
    }
  }
  
  if (args$parallel){
    t1 <- Sys.time()
    beginCluster()
    n2 <- calc(calc(s, fun), fun = function(x) {mean(x, na.rm=T)}, progress = 'text')
    endCluster()
    t2 <- Sys.time()
    log_info(paste0('Parallel N2 calculation took ', round(difftime(t2, t1, units='mins'), 1), ' mins'))
  } else{
    t1 <- Sys.time()
    n2 <- calc(calc(s, fun, progress = 'text'), fun = function(x) {mean(x, na.rm=T)})
    t2 <- Sys.time()
    log_info(paste0('N2 calculation took ', round(difftime(t2, t1, units='mins'), 1), ' mins'))
  }
  
  setwd(original_dir)
  
  ## build output name
  out_ild_dir <- paste0(args$out_ild_dir, '/', args$date, '/')
  out_n2_dir <- paste0(args$out_n2_dir, '/', args$date, '/')
  if (!dir.exists(out_ild_dir)) dir.create(out_ild_dir, recursive = TRUE)
  if (!dir.exists(out_n2_dir)) dir.create(out_n2_dir, recursive = TRUE)
  
  out_ild <- paste0(out_ild_dir, args$date, '_ild.nc')
  out_n2 <- paste0(out_n2_dir, args$date, '_n2.nc')
  log_info('ILD file output to {print(out_ild)}')
  log_info('N2 file output to {print(out_n2)}')
  
  ## write to CSV
  writeRaster(ild, file = out_ild, format='CDF', varname = paste0('ild_', args$deltaT), overwrite=TRUE)
  writeRaster(n2, file = out_n2, format='CDF', varname = 'n2', overwrite=TRUE)

}

