#!/usr/bin/env Rscript

## we need some custom functions we've written so have to figure out how to load those in rocker
#files.sources <- list.files('/home/facet/R/', full.names = TRUE)
#sapply(files.sources, source)

library(argparser, quietly=TRUE)
library(stringr, quietly=TRUE)
library(raster, quietly=TRUE)
#library(snow, quietly=TRUE)

# Build up our command line argument parser
p <- arg_parser("Calculate sd from input NetCDF")
p <- add_argument(p, "in_dir", help="path to NetCDF containing source data")
p <- add_argument(p, "--var_name", help="Character indicating variable name of interest in source file.", default = NA)
p <- add_argument(p, "--focal_dim", help="Odd integer used in raster::focal() to calculate the sd.", default = 3)
p <- add_argument(p, "out_dir", help="path to output resulting NetCDF")

# parse the arguments
args <- parse_args(p)
print(args)

## identify input file
files_in <- list.files(args$in_dir, full.names = TRUE, recursive = TRUE)
files_in <- files_in[grep('nc', files_in)]
print(files_in)

names_in <- list.files(args$in_dir, full.names = FALSE, recursive = TRUE)
names_in <- names_in[grep('nc', names_in)]
  
for (i in 1:length(files_in)){
  
  ## read in data
  if (is.na(args$var_name)){
    dat <- raster::raster(files_in[i])
  } else{
    dat <- raster::raster(files_in[i], varname = args$var_name)
  }
  
  ## 3 x 3 moving window SD calc
  if (dim(dat)[1] < 3 | dim(dat)[2] < 3) warning('dimensions of input data are smaller than the focal window used for standard deviation calculations. these calcs will likely result in NAs. Expand the spatial extent of your input data.')
  
  #beginCluster()
  #focal(r, polys)
  t1 <- Sys.time()
  dat_sd <- raster::focal(dat[[1]], w = matrix(1, nrow = args$focal_dim, ncol = args$focal_dim), fun = function(x) stats::sd(x, na.rm = T), progress = 'text')
  t2 <- Sys.time()
  #endCluster()
  cat(paste0('SD calc took ', t2-t1, '.'))
  
  ## build output name
  out_name <- substr(names_in[i], 1,
                     stringr::str_locate_all(names_in[i], '.nc')[[1]][1] - 1)
  if (nrow(stringr::str_locate_all(out_name, '/')[[1]]) != 0){
    out_dir <- paste0(args$out_dir,
                    substr(out_name, 1,
                           stringr::str_locate_all(out_name, '/')[[1]][1]))
  }
  
  if (!dir.exists(out_dir)) dir.create(out_dir, recursive = TRUE)

  out_name <- substr(out_name, stringr::str_locate_all(names_in[i], '/')[[1]][1] + 1,
                     nchar(out_name))

  out_name <- paste0(out_dir, '/', out_name, '_sd.nc')
  print(out_name)
  
  ## write
  writeRaster(dat_sd, file = out_name, format='CDF', overwrite=TRUE)
  print(paste0('File written to ', out_name, '.'))
  
}