#!/usr/bin/env python
"""
Given a NetCDF with bathemetry, output a NetCDF with bathemetry and rugosity
"""
import argparse
import pathlib

from dask.diagnostics import ProgressBar
import xarray as xr

from facet_shared.logger import logger


class Arguments(argparse.Namespace):
    input_nc: pathlib.Path
    output_nc: pathlib.Path
    bathemetry_var: str
    rugosity_window: int
    latitude_var: str
    longitude_var: str


def main(args: Arguments):
    logger.info(f"Arguments: {args}")

    ds = xr.open_dataset(args.input_nc)

    logger.info(f"Opened bathemetry dataset: {ds}")

    bathemetry_da = ds[args.bathemetry_var]

    output_ds = xr.Dataset(
        {
            "bathemetry": bathemetry_da,
            "rugosity": std(
                bathemetry_da,
                args.rugosity_window,
                latitude_var=args.latitude_var,
                longitude_var=args.longitude_var,
            ),
        }
    )

    args.output_nc.parent.mkdir(parents=True, exist_ok=True)

    netcdf = output_ds.to_netcdf(args.output_nc, compute=False)

    with ProgressBar():
        netcdf.compute()


def std(
    da: xr.DataArray,
    window: int = 3,
    latitude_var: str = "latitude",
    longitude_var: str = "longitude",
) -> xr.DataArray:
    """ Calculate the standard deviation of a given rolling window """
    window_kwargs = {latitude_var: window, longitude_var: window, "center": True}

    r = da.rolling(**window_kwargs)
    std_da = r.std()

    return std_da


def parse_arguments() -> Arguments:
    parser = argparse.ArgumentParser(description="Generate rugosity from bathemetry")
    parser.add_argument(
        "input_nc", help="Source NetCDF with bathemetry", type=pathlib.Path
    )
    parser.add_argument(
        "output_nc",
        help="Path to output NetCDF with bathemetry and rugosity to",
        type=pathlib.Path,
    )
    parser.add_argument("--bathemetry_var", default="z", help="Bathemetry variable")
    parser.add_argument(
        "--rugosity_window",
        default=3,
        help="Size of window that rugosity should be calculated over",
    )
    parser.add_argument(
        "--latitude_var",
        default="latitude",
        help="Name of latitude variable (default: `latitude`)",
    )
    parser.add_argument(
        "--longitude_var",
        default="longitude",
        help="Name of longitude variable (default: `longitude`)",
    )

    return parser.parse_args()


if __name__ == "__main__":
    args = parse_arguments()
    main(args)
