#!/usr/bin/env python
"""
Enhance CSVs (specifically ICCAT, but may work with other sources) with environmental data
"""
import argparse
from functools import lru_cache
import pathlib

from dask.diagnostics import ProgressBar
import pandas as pd
import timeout_decorator
import xarray as xr

import hycom_common as hycom
from facet_shared.logger import logger

import methods


class Arguments(argparse.Namespace):
    input_csv: pathlib.Path
    output_csv: pathlib.Path
    bathymetry_nc: pathlib.Path
    date_col: str
    latitude_col: str
    longitude_col: str


def main(args: Arguments):
    logger.info(f"Arguments: {args}")

    logger.info(f"Opening CSV from {args.input_csv}")
    df = pd.read_csv(args.input_csv)
    logger.info(f"Opened CSV: \n{df.head()}")

    df = enhance_bathymetry(df, args)
    #df = enhance_hycom(df, args)

    logger.info(f"Writing CSV to {args.output_csv}")
    args.output_csv.parent.mkdir(exist_ok=True, parents=True)
    df.to_csv(args.output_csv, index=False)


def enhance_bathymetry(df: pd.DataFrame, args: Arguments) -> pd.DataFrame:
    """ Load a pre-generated batheymetry dataset to enhance points with """
    logger.info(f"Opening bathymetry from {args.bathymetry_nc}")
    ds = xr.open_dataset(args.bathymetry_nc)
    if 180 < ds["longitude"].max():
        logger.info("Longitude was 0-360, converting to -180 to 180")
        ds = ds.assign_coords(longitude=(((ds["longitude"] + 180) % 360) - 180))

        ds = ds.sortby(["latitude", "longitude"])

    logger.info(f"Opened bathymetry: \n{ds}")

    bathymetry_enhancer = methods.EnhancePointwise(
        ds, {args.latitude_col: "latitude", args.longitude_col: "longitude"}
    )
    enhanced_df = bathymetry_enhancer.enhance(df)

    logger.info(f"Bathymetry enhanced CSV: \n{enhanced_df.head()}")

    return enhanced_df


def enhance_hycom(df: pd.DataFrame, args: Arguments) -> pd.DataFrame:
    """ Dynamically load HYCOM data from THREDDS server """
    logger.info("Enhancing with HYCOM")

    hycom_enhancer = methods.EnhanceGroupByApply(
        hycom_window,
        {args.date_col: "time", args.latitude_col: "lat", args.longitude_col: "lon"},
        hycom.hycom_vars,
    )
    enhanced_df = hycom_enhancer.enhance(df)

    logger.info(f"Enhanced with HYCOM: \n{enhanced_df.head()}")

    return enhanced_df


@lru_cache
def hycom_for_date_cached(time: str) -> xr.Dataset:
    """ Load and cache a single days HYCOM Dataset """
    time = time.split(" ")[0]
    return hycom.hycom_for_date(time)


@timeout_decorator.timeout(120)
def hycom_window(time: str, lat: float, lon: float) -> xr.Dataset:
    """ Load a small window of HYCOM data """
    ds = hycom_for_date_cached(time)

    lat_min = max([lat - 1, ds["lat"].min()])
    lat_max = min([lat + 1, ds["lat"].max()])
    lat_slice = slice(lat_min, lat_max)

    lon_min = max([lon - 1, ds["lon"].min()])
    lon_max = min([lon + 1, ds["lon"].max()])
    lon_slice = slice(lon_min, lon_max)

    subset = ds.sel(lat=lat_slice, lon=lon_slice)

    subset.load()

    return subset


def parse_arguments() -> Arguments:
    parser = argparse.ArgumentParser(description="Enhance CSV with environmental data")
    parser.add_argument("input_csv", help="Source CSV path", type=pathlib.Path)
    parser.add_argument("output_csv", help="Output CSV path", type=pathlib.Path)
    parser.add_argument("bathymetry_nc", help="Bathemetry NetCDF", type=pathlib.Path)
    parser.add_argument(
        "--date_col", default="date", help="Column countain date values"
    )
    parser.add_argument(
        "--latitude_col",
        default="lat",
        help="Column containing latitude values (-90 - 90) (default: `lat`)",
    )
    parser.add_argument(
        "--longitude_col",
        default="lon",
        help="Column containing longitude values (-180 - 180) (default: `lon`)",
    )

    return parser.parse_args()


if __name__ == "__main__":
    args = parse_arguments()
    main(args)
