## Get CMEMS grids for enhancing and predicting

bathy <- raster::raster('~/work/RCode/NASA-FaCeT/data/bathy/global_bathy_0.01.nc')
bathy <- raster::rotate(bathy)
rugosity <- raster::raster('~/work/RCode/NASA-FaCeT/data/enhance/bathy_rugosity.nc', varname = 'rugosity')
rugosity <- raster::rotate(rugosity)
atl <- rgdal::readOGR(dsn='./data/shapefiles/',layer='atlantic') ## prevents great lakes and others that bathymetry mask doesn't catch

tVec <- seq.POSIXt(as.POSIXct('1993-01-15', tz='UTC'), as.POSIXct('2019-12-15', tz='UTC'), by='month')
for (i in 1:length(tVec)){
  
  #base_url <- 'https://nrt.cmems-du.eu/motu-web/Motu'
  base_url <- 'https://my.cmems-du.eu/motu-web/Motu'
  
  #temp_service_id <- 'GLOBAL_MULTIYEAR_PHY_001_030-TDS'
  temp_service_id <- 'GLOBAL_REANALYSIS_PHY_001_031-TDS'
  
  #temp_product_id <- 'cmems_mod_glo_phy_my_0.083_P1D-m'
  temp_product_id <- 'global-reanalysis-phy-001-031-grepv2-mnstd-monthly'
  
  #vars <- c('thetao', 'so', 'uo', 'vo','zos','mlotst')#,'bottomT')
  vars <- c('thetao_mean', 'so_mean', 'uo_mean', 'vo_mean','zos_mean','mlotst_mean')#,'bottomT')
  
  xl <- c(-100, -5); yl <- c(10, 55)
  
  write_dir <- paste0('~/work/EnvData/glorys_monthly/', lubridate::year(tVec[i]), '/')
  out_name <- paste0(write_dir, temp_product_id, '_', format(as.Date(tVec[i]), '%Y%m'), '.nc')
  raster_out <- paste0(write_dir, temp_product_id, '_', format(as.Date(tVec[i]), '%Y-%m'), '.grd')
  if (!dir.exists(write_dir)) dir.create(write_dir, recursive=TRUE)
  if (file.exists(raster_out)) next
  #out_dir <- tempdir()
  
  ## Request CMEMS env data via motu
  if (!file.exists(out_name)){
    dstatus <- try(system(paste0('python3 -m motuclient --motu ', base_url,
                ' --service-id ', temp_service_id,
                ' --product-id ', temp_product_id,
                ' --longitude-min ', xl[1], ' --longitude-max ', xl[2],
                ' --latitude-min ', yl[1], ' --latitude-max ', yl[2],
                ' --date-min ', format(as.Date(tVec[i]) - 3, '%Y-%m-%d'), ' --date-max ', format(as.Date(tVec[i]) + 3, '%Y-%m-%d'), 
                ' --depth-min 0 --depth-max 1', 
                ' --variable ', vars[1],
                ' --variable ', vars[2],
                ' --variable ', vars[3],
                ' --variable ', vars[4],
                ' --variable ', vars[5],
                ' --variable ', vars[6],
                #' --variable ', vars[7],
                ' --out-dir ', write_dir,
                ' --out-name ', out_name,
                ' --user cbraun --pwd "A37x&sk0a6Gq"')), TRUE)
  } else{
    dstatus <- NA
  }
  
  if (file.exists(out_name) & class(dstatus) != 'try-error'){
    logger::log_info(paste0('Download complete for ', tVec[i]))
  } else{
    logger::log_warn(paste0('Download failed for ', tVec[i], '. Skipping.'))
    next
  }
  
  ## iterate through vars and manipulate/write out as needed
  for (tt in 1:length(vars)){
    if (tt == 1){
      r <- raster::raster(out_name, varname=vars[tt])
    } else{
      r <- raster::stack(r, raster::raster(out_name, varname=vars[tt]))
    }
  } 
  # r is now a stack of the following variables:
  #> names(r)
  #[1] "Temperature"                        
  #[2] "Salinity"                            
  #[3]"Eastward.velocity"                  
  #[4] "Northward.velocity"                  
  #[5] "Sea.surface.height"                  
  #[6] "Density.ocean.mixed.layer.thickness"
  names(r) <- c('sst','sss','u','v','ssh','mld')
  ## calc derived vars
  ## eddy kinetic energy
  log_eke <- log((r$u ^ 2 + r$v ^ 2) / 2)
  names(log_eke) <- 'log_eke'
  
  ## 3 x 3 moving window SD calc on SST, SSH, SSS
  sst_sd <- raster::focal(r$sst, w = matrix(1, nrow = 3, ncol = 3), fun = function(x) stats::sd(x, na.rm = T))
  ssh_sd <- raster::focal(r$ssh, w = matrix(1, nrow = 3, ncol = 3), fun = function(x) stats::sd(x, na.rm = T))
  sss_sd <- raster::focal(r$sss, w = matrix(1, nrow = 3, ncol = 3), fun = function(x) stats::sd(x, na.rm = T))
  
  # drop u and v from original raster
  r <- r[[c('sst','sss','ssh','mld')]]
  
  # get bathy and rugosity
  if (!exists('bathy_r')){
    bathy_r <- raster::resample(bathy, log_eke)
    rug_r <- raster::resample(rugosity, log_eke)
    rm(bathy); rm(rugosity)
  }
  
  r <- raster::stack(r, log_eke, sst_sd, ssh_sd, sss_sd, bathy_r, rug_r)
  names(r) <- c('sst','sss','ssh','mld', 'log_eke','sst_sd','ssh_sd','sss_sd','bathy','rugosity')
  r <- raster::mask(r, atl)
  
  writeRaster(r, filename=raster_out, format='raster', overwrite=T)
  logger::log_info(paste0('Raster operations complete and written to ', raster_out))
  
  file.remove(out_name)
  rm(r); rm(ssh_sd); rm(sss_sd); rm(sst_sd); rm(log_eke)
  
}



