#' @param x is input object of class Extent (from raster) or a data.frame with cols lon and lat
#' @param pres is input object of class data.frame with cols lon, lat and date (POSIX). These are used to prevent pseudoabs from being generated in these presence locations and times.
#' @param n is integer indicating number of samples desired
#' @param use_bathy is logical indicating whether to use bathymetry to avoid sampling locations on land. Default is FALSE for simplicity but if TRUE then bathy (as raster) must be provided with land as NA mask.
#' @example {
#' system.time(
#' df <- sp_random(data.frame(lon = ctag$LonX, lat = ctag$LatY), n=10000, bathy = F)
#' )}
#' 
sp_random_presCheck <- function(x, pres, n, use_bathy = FALSE,...){
  
  args <- list(...)
  
  if (use_bathy == TRUE & 'bathy' %in% names(args)){
    bathy <- args$bathy
  } else if (use_bathy == TRUE & !('bathy' %in% names(args))){
    stop('use_bathy = TRUE but no bathy object was provided as input.')
  } else{
    # no bathy
  }
    
  ## take whatever the input is and convert to spatialpoly
  if (class(x) == 'Extent'){
    sp  <- as(x, 'SpatialPolygons')
  } else if (class(x) == "data.frame"){
    if (all(c('lon','lat') %in% names(x))){
      sp <- as(raster::extent(min(x$lon, na.rm=T),
                              max(x$lon, na.rm=T),
                              min(x$lat, na.rm=T),
                              max(x$lat, na.rm=T)), 'SpatialPolygons')
    } else if(class(x) == 'SpatialPolygons'){
      sp <- x
    } else{
      stop('Input does not contain cols named lat/lon.')
    }
  } else if (class(x) == 'SpatialPolygons'){
    sp <- x
  } else{
    stop('Function currently only accepts input of class Extent or data.frame.')
  }
  
  ## this would work but will not account for points on land
  #out <- spsample(sp, n = n, "random")
  
  ## sample from raster that indicates land vs water
  ## could eventually fancy-ify this to allow for adding a bathy constraint other than 0
  
  ## sample from bdf raster masked by sp
  if (use_bathy){
    print('using bathy')
    ## crop bathy to extent of sp
    bathy_sub <- raster::crop(bathy, sp)
    
    n_total <- n
    if (raster::ncell(bathy_sub) < n) n <- raster::ncell(bathy_sub)
    n_now <- 0
    while(n_now < n_total){
      samp <- as(raster::sampleRandom(bathy_sub, size = n, cells=FALSE, sp=TRUE), 'data.frame')
      if (n_now == 0){
        all_samp <- samp
      } else{
        all_samp <- rbind(all_samp, samp)
      }
      n_now <- n_now + nrow(samp)
    }
    samp <- all_samp[,c('x','y')]
    
  } else{
    samp <- as(sp::spsample(sp, n, "random"), 'data.frame')
  }
  
  ## remove bg sampled pseudoabsences based on spatial/temporal criteria
  ## here digits=1 means we dont allow duplicates within .1deg grid cells
  ## and we use month or yearmonth to control temporal duplicates
  
  samp$date <- sample(pres$date, size=nrow(samp), replace = TRUE)
  names(samp) <- c('lon','lat','date')
  samp$LatBin <- round(samp$lat, digits = 1)
  samp$LonBin <- round(samp$lon, digits = 1)
  samp$YearMonth <- paste0(lubridate::year(samp$date), lubridate::month(samp$date))
  samp$month <- lubridate::month(samp$date)
  
  pres$LatBin <- round(pres$lat, digits = 1)
  pres$LonBin <- round(pres$lon, digits = 1)
  pres$YearMonth <- paste0(lubridate::year(pres$date), lubridate::month(pres$date))
  pres$month <- lubridate::month(pres$date)
  
  samp <- anti_join(samp, pres, by=c("month", "LatBin", "LonBin"))
  
  return(samp)
    
}

