
## input df needs cols lon, lat, xdist, ydist, dt
#' @param xpos are x coordinates (lon; -180 to 180)
#' @param ypos are y coordinates (lat)
#' @param tpos is date variable of class Date
#' @param xlen, ylen is the desired error in your x coordinates, default is 0.08 which is 1 hycom grid cell
#' @param ylen is the desired error in your y coordinates, default is 0.08 which is 1 hycom grid cell
#' @param varName is character indicating which variable (or multiple) you want from HYCOM. One or multiple of the following can be used: 'water_temp', 'water_u', 'water_v','surf_el', 'salinity'

facet_hycom <- function(xpos, ypos, tpos, xlen = 0.08, ylen = 0.08, varName = c('water_temp', 'water_u', 'water_v','surf_el','salinity'), verbose = TRUE){


  lengthXpos <- length(xpos)
  lengthYpos <- length(ypos)
  lengthTpos <- length(tpos)
  lengthtest <- identical(lengthXpos, lengthYpos) & identical(lengthXpos, lengthTpos)
  if (!lengthtest){
    print('input vectors are not of the same length')
    print(paste0('length of xpos: ', lengthXpos))
    print(paste0('length of ypos: ', lengthYpos))
    print(paste0('length of tpos: ', lengthTpos))
    return()
  }

  # RECONCILE LON GRIDS
  xpos1 <- xpos

  # DEAL WITH XLEN = CONSTANT V. VECTOR
  if (length(xlen) == 1) {
    xrad <- c(rep(xlen, length(xpos)))
    yrad <- c(rep(ylen, length(ypos)))
  } else {
    xrad <- xlen
    yrad <- ylen
  }

  # SET LIMITS BASED ON INPUT DATA
  udtpos <- as.Date(tpos, origin='1970-01-01',tz= "GMT")
  xposLim <- c(min(xpos1 - (xrad / 2)), max(xpos1 + (xrad / 2)))
  yposLim <- c(min(ypos - (yrad / 2)), max(ypos + (yrad / 2)))
  tposLim <- c(min(udtpos), max(udtpos))

  #create structures to store last request in case it is the same
  out <- base::list()

  #dimnames(out.dataframe)[[2]] <- c('mean', 'stdev', 'n', 'satellite date', 'requested lon min', 'requested lon max', 'requested lat min', 'requested lat max', 'requested date', 'median', 'mad')
  newLonIndex <-  rep(NA_integer_, 2)
  newLatIndex <-  rep(NA_integer_, 2)
  newTimeIndex <-  rep(NA_integer_, 2)
  
  for (i in 1:length(xpos1)) {
    print(paste('Starting ', i, ' of ', length(xpos1), sep=''))

    # define bounding box
    xmax <- xpos1[i] + (xrad[i] / 2)
    xmin <- xpos1[i] - (xrad[i] / 2)
    ymax <- ypos[i] + (yrad[i] / 2)
    ymin <- ypos[i] - (yrad[i] / 2)
    xy <- cbind(xpos1[i], ypos[i])
    
    ## establish out_na for smooth error handling
    out_na <- base::list(x = xy[1], y = xy[2],
                         xmin=xmin, xmax=xmax, 
                         ymin=ymin, ymax=ymax, 
                         tpos = tpos[i],
                         sst = NA,#raster::extract(wtemp[[1]], xy), 
                         sst_sd = NA,#raster::extract(sst_sd, xy),
                         ssh = NA, #raster::extract(surf_el, xy), 
                         ssh_sd = NA, #raster::extract(ssh_sd, xy),
                         water_u = NA, #raster::extract(water_u, xy), 
                         water_v = NA, #raster::extract(water_v, xy),
                         eke = NA,#log10(raster::extract(eke, xy)), 
                         ild.5 = NA, #raster::extract(ild.5, xy),
                         n2 = NA) #raster::extract(n2, xy))
    
    tdir <- tempdir()
    fileout <- 'temp.nc'
    myURL <- try(HMMoce:::get.hycom(limits=c(xmin, xmax, ymin, ymax), time=tpos[i], download.file=F, dir=tdir,
                                vars=varName), TRUE)
    
    if(class(myURL) == 'try-error'){
      print('Invalid URL, assigning NA values to this iteration.')
      
      out <- out_na
      
      return(out)
    }
      

    #d.try <- try(curl::curl_download(myURL, fileout, quiet=!verbose), TRUE)
    d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
    if(class(d.try) == 'try-error'){
      print('First download filed. Try 2 of 5.')
      d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
      if(class(d.try) == 'try-error'){
        print('First download filed. Try 3 of 5.')
        d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
        if(class(d.try) == 'try-error'){
          print('First download filed. Try 4 of 5.')
          d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
          if(class(d.try) == 'try-error'){
            print('First download filed. Try 5 of 5.')
            d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
            if(class(d.try) == 'try-error'){
              print('Download filed after re-try, leaving this iteration blank.')
              
              out <- out_na
              
              ## remove the tempdir we created and all temp files it contains
              unlink(tdir, recursive = T)
              
              return(out)

            }
          }
        }
      }
    }

    ## some weird errors suggesting the file tried to download but then wasn't able to open
    if (!file.exists(fileout)){
      out <- out_na
      
      ## remove the tempdir we created and all temp files it contains
      unlink(tdir, recursive = T)
      
      return(out)
    }
    
    ## get data
    nc <- try(RNetCDF::open.nc(fileout), TRUE)
    
    ## if download worked but nc may be corrupt, re-try download once more
    if (class(nc) == 'try-error'){
      d.try <- try(download.file(myURL, fileout, quiet=!verbose, method = 'curl'), TRUE)
      if (class(d.try) == 'try-error') return(out_na)
      nc <- try(RNetCDF::open.nc(fileout), TRUE)
      if (class(nc) == 'try-error') return(out_na)
    }
    # get var indices
    ncnames = NULL
    nmax <- RNetCDF::file.inq.nc(nc)$nvars - 1
    for(ii in 0:nmax) ncnames[ii + 1] <- RNetCDF::var.inq.nc(nc, ii)$name
    dep.idx <- grep('dep', ncnames, ignore.case=TRUE) - 1
    dep <- as.numeric(RNetCDF::var.get.nc(nc, dep.idx))
    # close netcdf file
    RNetCDF::close.nc(nc)
    
    ## get other vars
    if ('water_temp' %in% varName) wtemp <- raster::brick(fileout, varname = 'water_temp') #* scale + offset
    if ('salinity' %in% varName) sal <- raster::brick(fileout, varname = 'salinity') #* scale + offset
    if ('surf_el' %in% varName) surf_el <- raster(fileout, varname = 'surf_el') #* scale + offset
    if ('water_u' %in% varName) water_u <- raster::brick(fileout, varname = 'water_u') #* scale + offset
    if ('water_v' %in% varName) water_v <- raster::brick(fileout, varname = 'water_v') #* scale + offset
    

    ## so far, raster::extracted 3D temperature (wtemp), 3D salinity (sal),
    ## 2D surface u and v (water_u and water_v)
    ## 2D ssh (surf_el) and vector of depths for wtemp/sal

    ####====================

    ## MLD, ILD and BBV (N2) calculations
   
    ## define a function to get ILD - 0.5 deg C
    fun <- function(x) {
      if (all(is.na(x))){
        NA
      } else{
        dep[which.min(abs(x[1] - x[2:40] - 0.5))]
      }
    } ## index of depth level closest to -0.5
    ild.5 <- calc(wtemp, fun)

    ## define a function to approx N2 for top 200m (e.g. index 1:23) using temp and salinity
    s <- raster::stack(wtemp[[1:23]], sal[[1:23]])
    fun <- function(x) {
      if (all(is.na(x))){
        rep(NA, length.out=length(dep[1:23]))
      } else if(any(is.na(x[1:10]))){ ## if HYCOM is NA in top 25 m, not worth trying (and failing) to calc N2
        rep(NA, length.out=length(dep[1:23]))
      } else{
        oce::swN2(pressure = dep[1:23], sigmaTheta = oce::swSigmaTheta(x[24:46], x[1:23], pressure = dep[1:23]))#, referencePressure = median(depth[1:23], na.rm = TRUE)))
      }
    }
    
    n2 <- calc(calc(s, fun), fun = function(x) {mean(x, na.rm=T)})
    
    
    ####====================
    ## eddy kinetic energy
    eke <- (water_u[[1]] ^ 2 + water_v[[1]] ^ 2) / 2
    
    ####====================
    ## 3 x 3 moving window SD calc on SST
    if (dim(wtemp)[1] < 3 | dim(wtemp)[2] < 3) warning('dimensions of desired hycom product are smaller than the focal window used for standard deviation calculations (e.g. SST SD). these calcs will likely result in NAs. expand your grid using xlen/ylen inputs.')
    sst_sd <- raster::focal(wtemp[[1]], w = matrix(1, nrow = 3, ncol = 3), fun = function(x) stats::sd(x, na.rm = T))
    
    if (dim(surf_el)[1] < 3 | dim(surf_el)[2] < 3) warning('dimensions of desired hycom product are smaller than the focal window used for standard deviation calculations (e.g. SST SD). these calcs will likely result in NAs. expand your grid using xlen/ylen inputs.')
    ssh_sd <- raster::focal(surf_el[[1]], w = matrix(1, nrow = 3, ncol = 3), fun = function(x) stats::sd(x, na.rm = T))
    
    print('output calculations')

    if (!exists('out')) out <- list()
    out[[i]] <- base::list(x = xy[1], y = xy[2],
                           xmin=xmin, xmax=xmax, 
                           ymin=ymin, ymax=ymax, 
                           tpos = tpos[i],
                           sst = NA,#raster::extract(wtemp[[1]], xy), 
                           sst_sd = NA,#raster::extract(sst_sd, xy),
                           ssh = NA, #raster::extract(surf_el, xy), 
                           ssh_sd = NA, #raster::extract(ssh_sd, xy),
                           water_u = NA, #raster::extract(water_u, xy), 
                           water_v = NA, #raster::extract(water_v, xy),
                           eke = NA,#log10(raster::extract(eke, xy)), 
                           ild.5 = NA, #raster::extract(ild.5, xy),
                           n2 = NA) #raster::extract(n2, xy)) 

    if (xy[1,1] < 0 & extent(wtemp)@xmax > 180){
      # if xy is 180 and raster is 360
      xy[1,1] <- make360(xy[1,1])
    } else if(xy[1,1] > 180 & extent(wtemp)@xmin < 0){
      # if xy is 360 and raster is 180
      xy[1,1] <- make180(xy[1,1])
      
    }
    
    if (exists('wtemp')) out[[i]]$sst <- raster::extract(wtemp[[1]], xy)
    if (exists('sst_sd')) out[[i]]$sst_sd <- raster::extract(sst_sd, xy)
    if (exists('surf_el')) out[[i]]$ssh <- raster::extract(surf_el, xy)
    if (exists('ssh_sd')) out[[i]]$ssh_sd <- raster::extract(ssh_sd, xy)
    if (exists('water_u')) out[[i]]$water_u <- raster::extract(water_u[[1]], xy)
    if (exists('water_v')) out[[i]]$water_v <- raster::extract(water_v[[1]], xy)
    if (exists('eke')) out[[i]]$eke <- log10(raster::extract(eke, xy))
    if (exists('ild.5')) out[[i]]$ild.5 <- raster::extract(ild.5, xy)
    if (exists('n2')) out[[i]]$n2 <- raster::extract(n2, xy)
    
    
    
    print('--------------------')

    #if(i %% 5000 == 0){
    #  save(out, file=paste('~/work/Data/all_hycom_',i,'.rda', sep=''))
    #}
    
  }
  
  ## remove the tempdir we created and all temp files it contains
  unlink(tdir, recursive = T)
  
  return(out)

}
