#' Function calculates distance from points on a grid
#'
#' @param input_df is path to csv file that can be read with data.table::fread. Must contains cols lon and lat.
#' @param mask_shp is optional (default is NULL) path to shapefile used to mask the output distance grid. If no mask is provided, output is global.
#' @param x_min Numeric; indicating desired minimum longitude of output grid.
#' @param x_max Numeric; indicating desired maximum longitude of output grid.
#' @param y_min Numeric; indicating desired minimum latitude of output grid.
#' @param y_max Numeric; indicating desired maximum latitude of output grid.
#' @param res Numeric; indicating desired spatial resolution (in decimal degrees) of output grid.
#' @return a raster of distance from input points in kilometers

distance_from_points <- function(input_df, mask_shp=NA, x_min, x_max,
                                 y_min, y_max, res){
  ###-----------making a distance from points raster (such as seamounts or ports)------#####
  
  ## check input coords
  if (length(which(input_df$lon > 180)) > 0) input_df$lon <- make180(input_df$lon)
  
  ## creating a spatial object
  port.selection_shp <- input_df
  coordinates(port.selection_shp) <- ~lon + lat

  ## making raster template
  template <- raster(xmn = x_min, ymn = y_min, xmx = x_max, ymx = y_max, res = res)
  port.selection_shp <- raster::crop(port.selection_shp, raster::extent(template))
  
  ## if mask provided, use it for example to isolate just the atlantic ocean
  if (!is.na(mask_shp)){
    mask_ocean <- sf::st_read(mask_shp)
    port.selection_shp <- raster::crop(port.selection_shp, raster::extent(mask_ocean))
  }
  
  ## setting the coordinate the same as port.selection_shp
  proj4string(port.selection_shp) <- CRS(proj4string(template))
  dist_port <- distanceFromPoints(template, port.selection_shp)
  if (!is.na(mask_shp)) dist_port <- raster::mask(dist_port, mask_ocean)
  dist_port <- dist_port * 0.001 #converting to km

  return(dist_port)
}
