/**
 * This file is part of Princess, a theorem prover for Presburger
 * arithmetic with uninterpreted predicates.
 * <http://www.philipp.ruemmer.org/princess.shtml>
 *
 * Copyright (C) 2017-2018 Philipp Ruemmer <ph_r@gmx.net>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * * Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 * 
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 * 
 * * Neither the name of the authors nor the names of their
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package ap.terfor.conjunctions;

import ap.terfor._
import ap.terfor.preds.{PredConj, Atom, Predicate}
import ap.terfor.arithconj.ArithConj
import ap.util.{Seqs, Debug}

import scala.collection.mutable.{HashSet => MHashSet}

object ReducerPlugin {

  private val AC = Debug.AC_PROPAGATION

  sealed abstract class ReductionResult {
    def orElse(otherResult : => ReductionResult) : ReductionResult
  }
  
  case object UnchangedResult
              extends ReductionResult {
    def orElse(otherResult : => ReductionResult) : ReductionResult =
      otherResult
  }
  
  case object FalseResult
              extends ReductionResult {
    def orElse(otherResult : => ReductionResult) : ReductionResult = this
  }
  
  case class  ChangedConjResult(arithConj : ArithConj,
                                predConj : PredConj,
                                negConjs : NegatedConjunctions)
              extends ReductionResult {
    def orElse(otherResult : => ReductionResult) : ReductionResult = this
  }

  object ReductionMode extends Enumeration {
    val Simple, Contextual = Value
  }

  /**
   * Rewrite occurrences of the predicates in the set
   * <code>rewrittenPreds</code> using a given rewriting function.
   *
   * As a convention, if the given <code>logger</code> is logging,
   * only positive occurrences of predicates will be rewritten
   * (otherwise the axioms generated by the logger might not make sense).
   */
  def rewritePreds(predConj : PredConj,
                   rewrittenPreds : Iterable[Predicate],
                   order : TermOrder,
                   logger : ComputationLogger = ComputationLogger.NonLogger)
                  (rewrite : Atom => Formula) : ReductionResult =
    if (predConj.isTrue ||
        Seqs.disjointSeq(predConj.predicates, rewrittenPreds)) {
      UnchangedResult
    } else {
      val removedAtoms = new MHashSet[Atom]
      var addedPosFormulas, addedNegFormulas = List[Formula]()

      val predIt = rewrittenPreds.iterator
      while (predIt.hasNext) {
        val pred = predIt.next

        {
          val atomIt = (predConj positiveLitsWithPred pred).iterator
          while (atomIt.hasNext) {
            val a = atomIt.next
            val f = rewrite(a)

            if (!(f eq a)) {
              if (f.isFalse)
                return FalseResult
              removedAtoms += a
              addedPosFormulas = f :: addedPosFormulas
            }
          }
        }

        if (!logger.isLogging) {
          val atomIt = (predConj negativeLitsWithPred pred).iterator

          while (atomIt.hasNext) {
            val a = atomIt.next
            val f = rewrite(a)

            if (!(f eq a)) {
              if (f.isTrue)
                return FalseResult
              removedAtoms += a
              addedNegFormulas = f :: addedNegFormulas
            }
          }
        }
      }

      if (removedAtoms.isEmpty) {
        UnchangedResult
      } else {
        val remainingPredConj = predConj.updateLitsSubset(
                                  predConj.positiveLits filterNot removedAtoms,
                                  predConj.negativeLits filterNot removedAtoms,
                                  order)

        val conj = Conjunction(List(),
                               addedPosFormulas.iterator ++
                               (for (f <- addedNegFormulas.iterator)
                                yield Conjunction.negate(f, order)) ++
                               (Iterator single remainingPredConj),
                               logger,
                               order)
                               
        //-BEGIN-ASSERTION-/////////////////////////////////////////////////////
        Debug.assertPre(AC, conj.quans.isEmpty)
        //-END-ASSERTION-///////////////////////////////////////////////////////
        
        if (conj.isFalse)
          FalseResult
        else
          ChangedConjResult(conj.arithConj, conj.predConj, conj.negatedConjs)
      }
    }

}

////////////////////////////////////////////////////////////////////////////////

/**
 * Interface for plugins that can be added to the
 * <code>ReduceWithConjunction</code> class.
 */
abstract class ReducerPlugin {

  import ReducerPlugin._

  /**
   * Factory to generate further instances of this plugin.
   */
  val factory : ReducerPluginFactory

  /**
   * Construct a new reducer to work underneath <code>num</code>
   * quantifiers.
   */
  def passQuantifiers(num : Int) : ReducerPlugin

  /**
   * Add further assumptions to this reducer.
   */
  def addAssumptions(arithConj : ArithConj,
                     mode : ReductionMode.Value) : ReducerPlugin

  /**
   * Add further assumptions to this reducer.
   */
  def addAssumptions(predConj : PredConj,
                     mode : ReductionMode.Value) : ReducerPlugin

  /**
   * Reduce the given predicate formulas. The result indicates whether
   * nothing changed, or whether the formulas were updated, possibly
   * generating additional arithmetic constraints or negated
   * constraints. Reduction is not required to be idempotent.
   */
  def reduce(predConj : PredConj,
             baseReducer : ReduceWithConjunction,
             logger : ComputationLogger,
             mode : ReductionMode.Value) : ReductionResult

  /**
   * Do a final check whether a complete conjunction can be reduced.
   * All sub-formulas are already fully reduced at this point.
   */
  def finalReduce(conj : Conjunction) : Conjunction

}

////////////////////////////////////////////////////////////////////////////////

/**
 * Factory to construct new reducer plugins.
 */
abstract class ReducerPluginFactory {
  def apply(conj : Conjunction, order : TermOrder) : ReducerPlugin
}

////////////////////////////////////////////////////////////////////////////////

object IdentityReducerPluginFactory extends ReducerPluginFactory {
  def apply(conj : Conjunction, order : TermOrder) = IdentityReducerPlugin
}

/**
 * Reducer plugin that always just returns unchanged formulas.
 */
object IdentityReducerPlugin extends ReducerPlugin {

  import ReducerPlugin._

  val factory = IdentityReducerPluginFactory

  def passQuantifiers(num : Int) : ReducerPlugin = this

  def addAssumptions(arithConj : ArithConj,
                     mode : ReductionMode.Value) : ReducerPlugin = this

  def addAssumptions(predConj : PredConj,
                     mode : ReductionMode.Value) : ReducerPlugin = this

  def reduce(predConj : PredConj,
             baseReducer : ReduceWithConjunction,
             logger : ComputationLogger,
             mode : ReductionMode.Value) : ReductionResult = {
//println("Reducing ... " + predConj)
    UnchangedResult
  }

  def finalReduce(conj : Conjunction) : Conjunction = conj

}

////////////////////////////////////////////////////////////////////////////////

object SeqReducerPluginFactory {
  def apply(plugins : Seq[ReducerPluginFactory]) : ReducerPluginFactory = {
    val allPlugins =
      for (p <- plugins.iterator;
           q <- p match {
             case IdentityReducerPluginFactory => Iterator.empty
             case p : SeqReducerPluginFactory => p.plugins.iterator
             case p => Iterator single p
           })
      yield q
    if (allPlugins.hasNext)
      new SeqReducerPluginFactory(allPlugins.toIndexedSeq)
    else
      IdentityReducerPluginFactory
  }
}

class SeqReducerPluginFactory private
      (val plugins : IndexedSeq[ReducerPluginFactory])
      extends ReducerPluginFactory {
  def apply(conj : Conjunction, order : TermOrder) : SeqReducerPlugin =
    new SeqReducerPlugin(this, for (p <- plugins) yield p(conj, order))
}

////////////////////////////////////////////////////////////////////////////////

/**
 * Reducer plugin that sequentially applies a list of plugins.
 */
class SeqReducerPlugin(val factory : SeqReducerPluginFactory,
                       plugins : IndexedSeq[ReducerPlugin])
      extends ReducerPlugin {

  import ReducerPlugin._

  private val N = plugins.size

  def passQuantifiers(num : Int) : ReducerPlugin = {
    var newPlugins : Array[ReducerPlugin] = null
    var i = 0
    
    while (i < N) {
      val p = plugins(i)
      val q = p passQuantifiers num
      if (newPlugins != null) {
        newPlugins(i) = q
      } else if (!(p eq q)) {
        newPlugins = new Array(N)
        var j = 0
        while (j < i) {
          newPlugins(j) = plugins(j)
          j = j + 1
        }
        newPlugins(i) = q
      }
      i = i + 1
    }
    
    if (newPlugins == null)
      this
    else
      new SeqReducerPlugin(factory, newPlugins)
  }

  def addAssumptions(arithConj : ArithConj,
                     mode : ReductionMode.Value) : ReducerPlugin = {
    var newPlugins : Array[ReducerPlugin] = null
    var i = 0
    
    while (i < N) {
      val p = plugins(i)
      val q = p.addAssumptions(arithConj, mode)
      if (newPlugins != null) {
        newPlugins(i) = q
      } else if (!(p eq q)) {
        newPlugins = new Array(N)
        var j = 0
        while (j < i) {
          newPlugins(j) = plugins(j)
          j = j + 1
        }
        newPlugins(i) = q
      }
      i = i + 1
    }
    
    if (newPlugins == null)
      this
    else
      new SeqReducerPlugin(factory, newPlugins)
  }

  def addAssumptions(predConj : PredConj,
                     mode : ReductionMode.Value) : ReducerPlugin = {
    var newPlugins : Array[ReducerPlugin] = null
    var i = 0
    
    while (i < N) {
      val p = plugins(i)
      val q = p.addAssumptions(predConj, mode)
      if (newPlugins != null) {
        newPlugins(i) = q
      } else if (!(p eq q)) {
        newPlugins = new Array(N)
        var j = 0
        while (j < i) {
          newPlugins(j) = plugins(j)
          j = j + 1
        }
        newPlugins(i) = q
      }
      i = i + 1
    }
    
    if (newPlugins == null)
      this
    else
      new SeqReducerPlugin(factory, newPlugins)
  }

  def reduce(predConj : PredConj,
             baseReducer : ReduceWithConjunction,
             logger : ComputationLogger,
             mode : ReductionMode.Value) : ReductionResult = {
    var i = 0

    while (i < N) {
      val res = plugins(i).reduce(predConj, baseReducer, logger, mode)
      if (res != UnchangedResult)
        return res
      i = i + 1
    }

    UnchangedResult
  }

  def finalReduce(conj : Conjunction) : Conjunction = {
    var res = conj
    var i = 0

    while (i < N) {
      res = plugins(i) finalReduce res
      i = i + 1
    }

    res
  }

}
