# Written by D. Ibarra, Jan. 2021 
# Analysis of Tibet Data for Figure 2 
library(IsoplotR)
setwd("~/Desktop/Tibet")
getwd()

# Function for curve Intersect
curve_intersect <- function(curve1, curve2, empirical=TRUE, domain=NULL) {
  if (!empirical & missing(domain)) {
    stop("'domain' must be provided with non-empirical curves")
  }
  
  if (!empirical & (length(domain) != 2 | !is.numeric(domain))) {
    stop("'domain' must be a two-value numeric vector, like c(0, 10)")
  }
  
  if (empirical) {
    # Approximate the functional form of both curves
    curve1_f <- approxfun(curve1$x, curve1$y, rule = 2)
    curve2_f <- approxfun(curve2$x, curve2$y, rule = 2)
    
    # Calculate the intersection of curve 1 and curve 2 along the x-axis
    point_x <- uniroot(function(x) curve1_f(x) - curve2_f(x),
                       c(min(curve1$x), max(curve1$x)))$root
    
    # Find where point_x is in curve 2
    point_y <- curve2_f(point_x)
  } else {
    # Calculate the intersection of curve 1 and curve 2 along the x-axis
    # within the given domain
    point_x <- uniroot(function(x) curve1(x) - curve2(x), domain)$root
    
    # Find where point_x is in curve 2
    point_y <- curve2(point_x)
  }
  
  return(list(x = point_x, y = point_y))
}



# Meteoric water of Aron et al. (2020)
lambda = 0.5282287
gamma = 0.0364796
d18O.prime = seq(-70,-8,0.05)
d17O.prime = lambda*d18O.prime+gamma
cap17O.prime = d17O.prime - 0.528*d18O.prime

d17O = 1000*(exp(d17O.prime/1000)-1)
d18O = 1000*(exp(d18O.prime /1000)-1)
cap17O.curved = d17O - (0.528*d18O)

Temp = 273.15+240
d18O.alpha = exp((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))/1000)
d18O.Si =  d18O.alpha*(d18O+1000)-1000
d17O.alpha = exp(((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))*((-1.85/Temp) + 0.5305))/1000)
d17O.Si = d17O.alpha*(d17O+1000)-1000
cap17O.curved.Si = d17O.Si - (0.528*d18O.Si)
#lines(d18O.Si,cap17O.curved.Si)

d18O.Si.lin = 1000*log(d18O.Si/1000+1)
cap17O.lin.Si = (1000*log(d17O.Si/1000+1)) - (0.528*(1000*log(d18O.Si/1000+1)))


# Read in data
Data = read.csv('TibetData 3.csv')
Data$D17Oprime
d18O.points = Data$d18O
D17O.points = Data$d17O-0.528*Data$d18O
points(d18O.points,D17O.points ,bg="orange",pch=21,cex=1.5)


water = read.csv("Waters.csv")
water.d18O = 1000*(-1+exp(water$d18Oprime/1000))
water.d17Oprime = water$d18Oprime*0.528+water$D17O
water.d17O = 1000*(-1+exp(water.d17Oprime/1000))
water.D17O.points = water.d17O-0.528*water.d18O

points(water.d18O,water.D17O.points,pch=23,bg="lightblue",cex=1.5)

lines(d18O.Si,cap17O.curved.Si)



All.data.regress = cbind(Data$d18O,rep(0.14,length(d18O.points)),Data$D17Oprime,Data$D17Oprime_SE,rep(0,length(Data$D17Oprime_SE)))
All.york = york(All.data.regress)
abline(a=All.york$a,b=All.york$b,lwd=2)
abline(a=All.york$a+2*All.york$a[2],b=All.york$b-2*All.york$b[2])
abline(a=All.york$a-2*All.york$a[2],b=All.york$b+2*All.york$b[2])
abline(a=All.york$a+2*All.york$a[2],b=All.york$b+2*All.york$b[2])
abline(a=All.york$a-2*All.york$a[2],b=All.york$b-2*All.york$b[2])



all.d18O.reg = seq(-30,20,0.05)
all.cap17O.reg = all.d18O.reg*All.york$b[1] + All.york$a[1]
all.d17O.reg = all.cap17O.reg+all.d18O.reg *0.528
#lines(all.d18O.reg,all.cap17O.reg,lwd=2)
#cap17O.test = all.cap17O.reg+all.d18O.reg*0.528

all.cap17O.reg1 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg1 = all.cap17O.reg1+all.d18O.reg *0.528
all.cap17O.reg2 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg2 = all.cap17O.reg2+all.d18O.reg *0.528
all.cap17O.reg3 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg3 = all.cap17O.reg3+all.d18O.reg *0.528
all.cap17O.reg4 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg4 = all.cap17O.reg4+all.d18O.reg *0.528



Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Siend.d18Oprime = 1000*log(Meteoric$x/1000 + 1)
Siend.d17O = Meteoric$y+0.528*Meteoric$x
Siend.d17Oprime = 1000*log(Siend.d17O/1000 + 1)
Siend.D17Oprime = Siend.d17Oprime - 0.528*Siend.d18Oprime



Meteoric.d18O = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O; Meteoric.D17O


points(Meteoric.d18O,Meteoric.D17O,pch=23,bg="blue",cex=1.5)

WaterSiend.d18Oprime = 1000*log(Meteoric.d18O/1000 + 1)
WaterSiend.d17O = Meteoric.D17O+0.528*Meteoric.d18O
WaterSiend.d17Oprime = 1000*log(WaterSiend.d17O/1000 + 1)
WaterSiend.D17Oprime = WaterSiend.d17Oprime - 0.528*WaterSiend.d18Oprime



d18O.reg.prime = 1000*log(all.d18O.reg/1000 + 1)
d17O.reg.prime = 1000*log(all.d17O.reg/1000 + 1)
cap.pri.17O.reg = d17O.reg.prime - 0.528*d18O.reg.prime


all.cap17O.prime.reg1 = 1000*log(all.d17O.reg1/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg2 = 1000*log(all.d17O.reg2/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg3 = 1000*log(all.d17O.reg3/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg4 = 1000*log(all.d17O.reg4/1000+1) - 0.528*d18O.reg.prime


lines(c(Meteoric.d18O,Meteoric$x),c(Meteoric.D17O,Meteoric$y),lwd=2)
Meteoric.d18O
Meteoric.D17O
# 2 sigma error
Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg1),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.1 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.1 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.1 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.1; Meteoric.D17O.1
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg2),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.2 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.2 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.2 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.2; Meteoric.D17O.2
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


par(mfrow=c(1,2))
# Linearized plot with curved mixing line
plot(d18O.prime,cap17O.prime,type="l",xlim=c(-25,12),ylim=c(-0.11,0.05),col="black",lwd=0,
     xlab="d'18O",ylab="cap'17O (theta=0.528)",las=1)
lines(d18O.reg.prime,cap.pri.17O.reg,col="black",lty=2)

Data2 = read.csv('TibetData 2.csv')


waters2 = read.csv('Waters2.csv')
points(waters2$..d18Oprime,waters2$D17O,pch=23,bg="lightblue",cex=1)

d17O.prime.waters2 = waters2$D17O+0.528*waters2$..d18Oprime
d17O.prime.waters = water$D17O+0.528*water$..d18Oprime

summary(lm(d17O.prime.waters~water$..d18Oprime))
summary(lm(d17O.prime.waters2~waters2$..d18Oprime))


# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters2~waters2$..d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters2~waters2$..d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)




# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters~water$..d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters~water$..d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

#lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)

lines(d18O.reg.prime,all.cap17O.prime.reg1)
lines(d18O.reg.prime,all.cap17O.prime.reg2)
lines(d18O.reg.prime,all.cap17O.prime.reg3)
lines(d18O.reg.prime,all.cap17O.prime.reg4)


points(Data2$d18Oprime,Data2$D17Oprime,bg="orange",pch=21,cex=1.5)
points(Data2$d18Oprime[15:17],Data2$D17Oprime[15:17],bg="red",pch=21,cex=1.5)

for(i in 1:length(water$..d18Oprime)){
  lines(rep(water$..d18Oprime[i],2),c(water$D17O[i]+water$SE[i],water$D17O[i]-water$SE[i]),lwd=1.5)
}
points(water$..d18Oprime,water$D17O,pch=23,bg="lightblue",cex=1.5)


for(i in 1:length(Data$d18O)){
  lines(rep(Data$d18Oprime[i],2),c(Data$D17Oprime[i]+Data$D17Oprime_SE[i],Data$D17Oprime[i]-Data$D17Oprime_SE[i]))
}
points(Data$d18Oprime,Data$D17Oprime,bg="orange",pch=21,cex=2)


points(Siend.d18Oprime,Siend.D17Oprime,pch=24,bg="grey",cex=2.5)
points(WaterSiend.d18Oprime,WaterSiend.D17Oprime,pch=24,bg="blue",cex=2.5)
Siend.d18Oprime;Siend.D17Oprime
WaterSiend.d18Oprime;WaterSiend.D17Oprime

Meteoric.d18O
Meteoric.d18O.1 - Meteoric.d18O
Meteoric.d18O.2 - Meteoric.d18O

legend("bottomleft",legend=c("Tibet Stream Data","Tibet Tap Water","Quartz","Whole Rock/Quartz (not in regression)","W/R = 100","Paleo-Meteoric Water","TibetMWL"),
       pch=c(23,23,21,21,24,24,NA),pt.bg=c("lightblue","lightblue","Orange","Red","Grey","Blue",NA),lwd=c(NA,NA,NA,NA,NA,NA,2),pt.cex=c(2,1.5,2,1.5,2,2,NA),
       cex=0.85,col=c("black","black","black","black","black","black","blue"))
       


capd18O = seq(-17,0,0.05)
meanelev = -0.0129*capd18O^4-1.121*capd18O^3-38.213*capd18O^2-715.22*capd18O
sigmaplus = 0.015*(capd18O^4)+0.738*(capd18O^3)+9.031*(capd18O^2)-47.186*capd18O
sigmaminus = -0.0126*(capd18O^4)-0.58*(capd18O^3)-5.262*(capd18O^2)+89.212*capd18O

water.comp = read.csv("Tibet_comp.csv")
highlight = water.comp[which(water.comp$study != 2),]
highlight = water.comp[which(water.comp$study != 2),]

water.comp=water.comp[which(water.comp$lat<31),]

plot(capd18O,meanelev,type="l",las=1,ylab="Elevation (m)",ylim=c(0,5200),xlim=c(-16,0),lwd=0)
#points(water.comp$d18O+5.8,water.comp$Elev,cex=0.5,col="grey",pch=16)


stream = water.comp[which(water.comp$Type != "Precipitation"),]
points(stream$d18O+5.8,stream$..Elev,cex=0.5,bg="lightblue",pch=21,lwd=0.4)

points(highlight$d18O+5.8,highlight$..Elev,pch=23,bg="lightblue",cex=1.5)

lines(capd18O,meanelev,lwd=2)
lines(capd18O,meanelev+sigmaplus,lty=2,lwd=1.5)
lines(capd18O,meanelev+sigmaminus,lty=2,lwd=1.5)



high = -7.2+1.5
meanelevhigh = -0.0129*high^4-1.121*high^3-38.213*high^2-715.22*high
sigmaminusval = -0.0126*(high^4)-0.58*(high^3)-5.262*(high^2)+89.212*high
meanelevhigh+sigmaminusval
#points(-6.2+1.2,meanelevhigh+sigmaminusval)
meanelevhigh+sigmaminusval-3552


low = -7.2-2.2
meanelevlow = -0.0129*low ^4-1.121*low ^3-38.213*low ^2-715.22*low 
sigmaminusval = 0.015*(low^4)+0.738*(low^3)+9.031*(low^2)-47.186*low
meanelevlow+sigmaminusval
#points(-2.2-1.7,meanelevlow+sigmaminusval)
meanelevlow+sigmaminusval-3552

#lines(rep(-7.2,2),c(2820,4125),lwd=3,col="black")

points(-7.2,3552,pch=24,bg="blue",cex=2.5)

points(-8.5,3940,cex=1.5,pch=21,bg="white")
lines(c(-8.5,-8.5+2),c(5000,5000))
lines(c(-8.5+2,-7.5+2-0.7),c(4500,4500))

legend("bottomleft",legend=c("Stream/Tap Water with D'17O", "Tibet Surface Water", "Tibet Precipitation"),
       pch=c(24,21,21),pt.cex=c(1.5,0.5,0.5),pt.bg=c("lightblue","lightblue",NA),col=c("black","black","grey"),pt.lwd=c(1,0.4,NA) ,cex=0.85)





#### Figure 
# Meteoric water of Aron et al. (2020)
lambda = 0.5268
gamma = 0.015
d18O.prime = seq(-70,-8,0.05)
d17O.prime = lambda*d18O.prime+gamma
cap17O.prime = d17O.prime - 0.528*d18O.prime

d17O = 1000*(exp(d17O.prime/1000)-1)
d18O = 1000*(exp(d18O.prime /1000)-1)
cap17O.curved = d17O - (0.528*d18O)

Temp = 273.15+240
d18O.alpha = exp((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))/1000)
d18O.Si =  d18O.alpha*(d18O+1000)-1000
d17O.alpha = exp(((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))*((-1.85/Temp) + 0.5305))/1000)
d17O.Si = d17O.alpha*(d17O+1000)-1000
cap17O.curved.Si = d17O.Si - (0.528*d18O.Si)
#lines(d18O.Si,cap17O.curved.Si)

d18O.Si.lin = 1000*log(d18O.Si/1000+1)
cap17O.lin.Si = (1000*log(d17O.Si/1000+1)) - (0.528*(1000*log(d18O.Si/1000+1)))


# Read in data
Data = read.csv('TibetData 3.csv')
Data$D17Oprime
d18O.points = Data$d18O
D17O.points = Data$d17O-0.528*Data$d18O
#points(d18O.points,D17O.points ,bg="orange",pch=21,cex=1.5)


water = read.csv("Waters.csv")
water.d18O = 1000*(-1+exp(water$d18Oprime/1000))
water.d17Oprime = water$d18Oprime*0.528+water$D17O
water.d17O = 1000*(-1+exp(water.d17Oprime/1000))
water.D17O.points = water.d17O-0.528*water.d18O

#points(water.d18O,water.D17O.points,pch=23,bg="lightblue",cex=1.5)

#lines(d18O.Si,cap17O.curved.Si)



All.data.regress = cbind(Data$d18O,rep(0.14,length(d18O.points)),Data$D17Oprime,Data$D17Oprime_SE,rep(0,length(Data$D17Oprime_SE)))
All.york = york(All.data.regress)
# abline(a=All.york$a,b=All.york$b,lwd=2)
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b-2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b-2*All.york$b[2])



all.d18O.reg = seq(-30,20,0.05)
all.cap17O.reg = all.d18O.reg*All.york$b[1] + All.york$a[1]
all.d17O.reg = all.cap17O.reg+all.d18O.reg *0.528
#lines(all.d18O.reg,all.cap17O.reg,lwd=2)
#cap17O.test = all.cap17O.reg+all.d18O.reg*0.528

all.cap17O.reg1 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg1 = all.cap17O.reg1+all.d18O.reg *0.528
all.cap17O.reg2 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg2 = all.cap17O.reg2+all.d18O.reg *0.528
all.cap17O.reg3 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg3 = all.cap17O.reg3+all.d18O.reg *0.528
all.cap17O.reg4 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg4 = all.cap17O.reg4+all.d18O.reg *0.528



Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
#points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Siend.d18Oprime = 1000*log(Meteoric$x/1000 + 1)
Siend.d17O = Meteoric$y+0.528*Meteoric$x
Siend.d17Oprime = 1000*log(Siend.d17O/1000 + 1)
Siend.D17Oprime = Siend.d17Oprime - 0.528*Siend.d18Oprime



Meteoric.d18O = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O; Meteoric.D17O


#points(Meteoric.d18O,Meteoric.D17O,pch=23,bg="blue",cex=1.5)

WaterSiend.d18Oprime = 1000*log(Meteoric.d18O/1000 + 1)
WaterSiend.d17O = Meteoric.D17O+0.528*Meteoric.d18O
WaterSiend.d17Oprime = 1000*log(WaterSiend.d17O/1000 + 1)
WaterSiend.D17Oprime = WaterSiend.d17Oprime - 0.528*WaterSiend.d18Oprime



d18O.reg.prime = 1000*log(all.d18O.reg/1000 + 1)
d17O.reg.prime = 1000*log(all.d17O.reg/1000 + 1)
cap.pri.17O.reg = d17O.reg.prime - 0.528*d18O.reg.prime


all.cap17O.prime.reg1 = 1000*log(all.d17O.reg1/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg2 = 1000*log(all.d17O.reg2/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg3 = 1000*log(all.d17O.reg3/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg4 = 1000*log(all.d17O.reg4/1000+1) - 0.528*d18O.reg.prime


#lines(c(Meteoric.d18O,Meteoric$x),c(Meteoric.D17O,Meteoric$y),lwd=2)
Meteoric.d18O
Meteoric.D17O
# 2 sigma error
Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg1),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.1 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.1 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.1 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.1; Meteoric.D17O.1
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg2),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.2 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.2 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.2 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.2; Meteoric.D17O.2
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


#par(mfrow=c(1,2))
# Linearized plot with curved mixing line
plot(d18O.prime,cap17O.prime,type="l",xlim=c(-25,12),ylim=c(-0.11,0.05),col="black",lwd=2,
     xlab="d'18O",ylab="cap'17O (theta=0.528)",las=1)
lines(d18O.reg.prime,cap.pri.17O.reg,col="black",lty=2)

Data2 = read.csv('TibetData 2.csv')


waters2 = read.csv('Waters2.csv')
#points(waters2$d18Oprime,waters2$D17O,pch=23,bg="lightblue",cex=1)

d17O.prime.waters2 = waters2$D17O+0.528*waters2$d18Oprime
d17O.prime.waters = water$D17O+0.528*water$d18Oprime

summary(lm(d17O.prime.waters~water$d18Oprime))
summary(lm(d17O.prime.waters2~waters2$d18Oprime))


# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)




# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

#lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)

lines(d18O.reg.prime,all.cap17O.prime.reg1)
lines(d18O.reg.prime,all.cap17O.prime.reg2)
lines(d18O.reg.prime,all.cap17O.prime.reg3)
lines(d18O.reg.prime,all.cap17O.prime.reg4)


points(Data2$d18Oprime,Data2$D17Oprime,bg="orange",pch=21,cex=1.5)
points(Data2$d18Oprime[15:17],Data2$D17Oprime[15:17],bg="red",pch=21,cex=1.5)
# 
# for(i in 1:length(water$d18Oprime)){
#   lines(rep(water$d18Oprime[i],2),c(water$D17O[i]+water$SE[i],water$D17O[i]-water$SE[i]),lwd=1.5)
# }
# points(water$d18Oprime,water$D17O,pch=23,bg="lightblue",cex=1.5)


for(i in 1:length(Data$d18O)){
  lines(rep(Data$d18Oprime[i],2),c(Data$D17Oprime[i]+Data$D17Oprime_SE[i],Data$D17Oprime[i]-Data$D17Oprime_SE[i]))
}
points(Data$d18Oprime,Data$D17Oprime,bg="orange",pch=21,cex=2)


points(Siend.d18Oprime,Siend.D17Oprime,pch=24,bg="grey",cex=2.5)
points(WaterSiend.d18Oprime,WaterSiend.D17Oprime,pch=24,bg="blue",cex=2.5)
Siend.d18Oprime;Siend.D17Oprime
WaterSiend.d18Oprime;WaterSiend.D17Oprime

Meteoric.d18O
Meteoric.d18O.1 - Meteoric.d18O
Meteoric.d18O.2 - Meteoric.d18O

legend("bottomleft",legend=c("Tibet Stream Data","Tibet Tap Water","Quartz","Whole Rock/Quartz (not in regression)","W/R = 100","Paleo-Meteoric Water","TibetMWL"),
       pch=c(23,23,21,21,24,24,NA),pt.bg=c("lightblue","lightblue","Orange","Red","Grey","Blue",NA),lwd=c(NA,NA,NA,NA,NA,NA,2),pt.cex=c(2,1.5,2,1.5,2,2,NA),
       cex=0.85,col=c("black","black","black","black","black","black","blue"))



# Meteoric water of Luz and Barkan (2010)
lambda = 0.528
gamma = 0.033
d18O.prime = seq(-70,-8,0.05)
d17O.prime = lambda*d18O.prime+gamma
cap17O.prime = d17O.prime - 0.528*d18O.prime

d17O = 1000*(exp(d17O.prime/1000)-1)
d18O = 1000*(exp(d18O.prime /1000)-1)
cap17O.curved = d17O - (0.528*d18O)

Temp = 273.15+240
d18O.alpha = exp((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))/1000)
d18O.Si =  d18O.alpha*(d18O+1000)-1000
d17O.alpha = exp(((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))*((-1.85/Temp) + 0.5305))/1000)
d17O.Si = d17O.alpha*(d17O+1000)-1000
cap17O.curved.Si = d17O.Si - (0.528*d18O.Si)
#lines(d18O.Si,cap17O.curved.Si)

d18O.Si.lin = 1000*log(d18O.Si/1000+1)
cap17O.lin.Si = (1000*log(d17O.Si/1000+1)) - (0.528*(1000*log(d18O.Si/1000+1)))


# Read in data
Data = read.csv('TibetData 3.csv')
Data$D17Oprime
d18O.points = Data$d18O
D17O.points = Data$d17O-0.528*Data$d18O
#points(d18O.points,D17O.points ,bg="orange",pch=21,cex=1.5)


water = read.csv("Waters.csv")
water.d18O = 1000*(-1+exp(water$d18Oprime/1000))
water.d17Oprime = water$d18Oprime*0.528+water$D17O
water.d17O = 1000*(-1+exp(water.d17Oprime/1000))
water.D17O.points = water.d17O-0.528*water.d18O

#points(water.d18O,water.D17O.points,pch=23,bg="lightblue",cex=1.5)

#lines(d18O.Si,cap17O.curved.Si)



All.data.regress = cbind(Data$d18O,rep(0.14,length(d18O.points)),Data$D17Oprime,Data$D17Oprime_SE,rep(0,length(Data$D17Oprime_SE)))
All.york = york(All.data.regress)
# abline(a=All.york$a,b=All.york$b,lwd=2)
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b-2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b-2*All.york$b[2])



all.d18O.reg = seq(-30,20,0.05)
all.cap17O.reg = all.d18O.reg*All.york$b[1] + All.york$a[1]
all.d17O.reg = all.cap17O.reg+all.d18O.reg *0.528
#lines(all.d18O.reg,all.cap17O.reg,lwd=2)
#cap17O.test = all.cap17O.reg+all.d18O.reg*0.528

all.cap17O.reg1 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg1 = all.cap17O.reg1+all.d18O.reg *0.528
all.cap17O.reg2 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg2 = all.cap17O.reg2+all.d18O.reg *0.528
all.cap17O.reg3 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg3 = all.cap17O.reg3+all.d18O.reg *0.528
all.cap17O.reg4 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg4 = all.cap17O.reg4+all.d18O.reg *0.528



Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
#points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Siend.d18Oprime = 1000*log(Meteoric$x/1000 + 1)
Siend.d17O = Meteoric$y+0.528*Meteoric$x
Siend.d17Oprime = 1000*log(Siend.d17O/1000 + 1)
Siend.D17Oprime = Siend.d17Oprime - 0.528*Siend.d18Oprime



Meteoric.d18O = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O; Meteoric.D17O


#points(Meteoric.d18O,Meteoric.D17O,pch=23,bg="blue",cex=1.5)

WaterSiend.d18Oprime = 1000*log(Meteoric.d18O/1000 + 1)
WaterSiend.d17O = Meteoric.D17O+0.528*Meteoric.d18O
WaterSiend.d17Oprime = 1000*log(WaterSiend.d17O/1000 + 1)
WaterSiend.D17Oprime = WaterSiend.d17Oprime - 0.528*WaterSiend.d18Oprime



d18O.reg.prime = 1000*log(all.d18O.reg/1000 + 1)
d17O.reg.prime = 1000*log(all.d17O.reg/1000 + 1)
cap.pri.17O.reg = d17O.reg.prime - 0.528*d18O.reg.prime


all.cap17O.prime.reg1 = 1000*log(all.d17O.reg1/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg2 = 1000*log(all.d17O.reg2/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg3 = 1000*log(all.d17O.reg3/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg4 = 1000*log(all.d17O.reg4/1000+1) - 0.528*d18O.reg.prime


#lines(c(Meteoric.d18O,Meteoric$x),c(Meteoric.D17O,Meteoric$y),lwd=2)
Meteoric.d18O
Meteoric.D17O
# 2 sigma error
Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg1),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.1 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.1 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.1 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.1; Meteoric.D17O.1
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg2),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.2 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.2 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.2 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.2; Meteoric.D17O.2
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


#par(mfrow=c(1,2))
# Linearized plot with curved mixing line
plot(d18O.prime,cap17O.prime,type="l",xlim=c(-25,12),ylim=c(-0.11,0.05),col="black",lwd=2,
     xlab="d'18O",ylab="cap'17O (theta=0.528)",las=1)
lines(d18O.reg.prime,cap.pri.17O.reg,col="black",lty=2)

Data2 = read.csv('TibetData 2.csv')


waters2 = read.csv('Waters2.csv')
#points(waters2$d18Oprime,waters2$D17O,pch=23,bg="lightblue",cex=1)

d17O.prime.waters2 = waters2$D17O+0.528*waters2$d18Oprime
d17O.prime.waters = water$D17O+0.528*water$d18Oprime

summary(lm(d17O.prime.waters~water$d18Oprime))
summary(lm(d17O.prime.waters2~waters2$d18Oprime))


# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)




# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

#lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)

lines(d18O.reg.prime,all.cap17O.prime.reg1)
lines(d18O.reg.prime,all.cap17O.prime.reg2)
lines(d18O.reg.prime,all.cap17O.prime.reg3)
lines(d18O.reg.prime,all.cap17O.prime.reg4)


points(Data2$d18Oprime,Data2$D17Oprime,bg="orange",pch=21,cex=1.5)
points(Data2$d18Oprime[15:17],Data2$D17Oprime[15:17],bg="red",pch=21,cex=1.5)
# 
# for(i in 1:length(water$d18Oprime)){
#   lines(rep(water$d18Oprime[i],2),c(water$D17O[i]+water$SE[i],water$D17O[i]-water$SE[i]),lwd=1.5)
# }
# points(water$d18Oprime,water$D17O,pch=23,bg="lightblue",cex=1.5)


for(i in 1:length(Data$d18O)){
  lines(rep(Data$d18Oprime[i],2),c(Data$D17Oprime[i]+Data$D17Oprime_SE[i],Data$D17Oprime[i]-Data$D17Oprime_SE[i]))
}
points(Data$d18Oprime,Data$D17Oprime,bg="orange",pch=21,cex=2)


points(Siend.d18Oprime,Siend.D17Oprime,pch=24,bg="grey",cex=2.5)
points(WaterSiend.d18Oprime,WaterSiend.D17Oprime,pch=24,bg="blue",cex=2.5)
Siend.d18Oprime;Siend.D17Oprime
WaterSiend.d18Oprime;WaterSiend.D17Oprime

Meteoric.d18O
Meteoric.d18O.1 - Meteoric.d18O
Meteoric.d18O.2 - Meteoric.d18O

legend("bottomleft",legend=c("Tibet Stream Data","Tibet Tap Water","Quartz","Whole Rock/Quartz (not in regression)","W/R = 100","Paleo-Meteoric Water","TibetMWL"),
       pch=c(23,23,21,21,24,24,NA),pt.bg=c("lightblue","lightblue","Orange","Red","Grey","Blue",NA),lwd=c(NA,NA,NA,NA,NA,NA,2),pt.cex=c(2,1.5,2,1.5,2,2,NA),
       cex=0.85,col=c("black","black","black","black","black","black","blue"))




# Meteoric water of tibet 210 degC
lambda = 0.5282287
gamma = 0.0364796

d18O.prime = seq(-70,-8,0.05)
d17O.prime = lambda*d18O.prime+gamma
cap17O.prime = d17O.prime - 0.528*d18O.prime

d17O = 1000*(exp(d17O.prime/1000)-1)
d18O = 1000*(exp(d18O.prime /1000)-1)
cap17O.curved = d17O - (0.528*d18O)

Temp = 273.15+210
d18O.alpha = exp((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))/1000)
d18O.Si =  d18O.alpha*(d18O+1000)-1000
d17O.alpha = exp(((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))*((-1.85/Temp) + 0.5305))/1000)
d17O.Si = d17O.alpha*(d17O+1000)-1000
cap17O.curved.Si = d17O.Si - (0.528*d18O.Si)
#lines(d18O.Si,cap17O.curved.Si)

d18O.Si.lin = 1000*log(d18O.Si/1000+1)
cap17O.lin.Si = (1000*log(d17O.Si/1000+1)) - (0.528*(1000*log(d18O.Si/1000+1)))


# Read in data
Data = read.csv('TibetData 3.csv')
Data$D17Oprime
d18O.points = Data$d18O
D17O.points = Data$d17O-0.528*Data$d18O
#points(d18O.points,D17O.points ,bg="orange",pch=21,cex=1.5)


water = read.csv("Waters.csv")
water.d18O = 1000*(-1+exp(water$d18Oprime/1000))
water.d17Oprime = water$d18Oprime*0.528+water$D17O
water.d17O = 1000*(-1+exp(water.d17Oprime/1000))
water.D17O.points = water.d17O-0.528*water.d18O

#points(water.d18O,water.D17O.points,pch=23,bg="lightblue",cex=1.5)

#lines(d18O.Si,cap17O.curved.Si)



All.data.regress = cbind(Data$d18O,rep(0.14,length(d18O.points)),Data$D17Oprime,Data$D17Oprime_SE,rep(0,length(Data$D17Oprime_SE)))
All.york = york(All.data.regress)
# abline(a=All.york$a,b=All.york$b,lwd=2)
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b-2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b-2*All.york$b[2])



all.d18O.reg = seq(-30,20,0.05)
all.cap17O.reg = all.d18O.reg*All.york$b[1] + All.york$a[1]
all.d17O.reg = all.cap17O.reg+all.d18O.reg *0.528
#lines(all.d18O.reg,all.cap17O.reg,lwd=2)
#cap17O.test = all.cap17O.reg+all.d18O.reg*0.528

all.cap17O.reg1 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg1 = all.cap17O.reg1+all.d18O.reg *0.528
all.cap17O.reg2 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg2 = all.cap17O.reg2+all.d18O.reg *0.528
all.cap17O.reg3 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg3 = all.cap17O.reg3+all.d18O.reg *0.528
all.cap17O.reg4 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg4 = all.cap17O.reg4+all.d18O.reg *0.528



Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
#points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Siend.d18Oprime = 1000*log(Meteoric$x/1000 + 1)
Siend.d17O = Meteoric$y+0.528*Meteoric$x
Siend.d17Oprime = 1000*log(Siend.d17O/1000 + 1)
Siend.D17Oprime = Siend.d17Oprime - 0.528*Siend.d18Oprime



Meteoric.d18O = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O; Meteoric.D17O


#points(Meteoric.d18O,Meteoric.D17O,pch=23,bg="blue",cex=1.5)

WaterSiend.d18Oprime = 1000*log(Meteoric.d18O/1000 + 1)
WaterSiend.d17O = Meteoric.D17O+0.528*Meteoric.d18O
WaterSiend.d17Oprime = 1000*log(WaterSiend.d17O/1000 + 1)
WaterSiend.D17Oprime = WaterSiend.d17Oprime - 0.528*WaterSiend.d18Oprime



d18O.reg.prime = 1000*log(all.d18O.reg/1000 + 1)
d17O.reg.prime = 1000*log(all.d17O.reg/1000 + 1)
cap.pri.17O.reg = d17O.reg.prime - 0.528*d18O.reg.prime


all.cap17O.prime.reg1 = 1000*log(all.d17O.reg1/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg2 = 1000*log(all.d17O.reg2/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg3 = 1000*log(all.d17O.reg3/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg4 = 1000*log(all.d17O.reg4/1000+1) - 0.528*d18O.reg.prime


#lines(c(Meteoric.d18O,Meteoric$x),c(Meteoric.D17O,Meteoric$y),lwd=2)
Meteoric.d18O
Meteoric.D17O
# 2 sigma error
Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg1),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.1 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.1 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.1 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.1; Meteoric.D17O.1
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg2),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.2 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.2 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.2 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.2; Meteoric.D17O.2
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


#par(mfrow=c(1,2))
# Linearized plot with curved mixing line
plot(d18O.prime,cap17O.prime,type="l",xlim=c(-25,12),ylim=c(-0.11,0.05),col="black",lwd=2,
     xlab="d'18O",ylab="cap'17O (theta=0.528)",las=1)
lines(d18O.reg.prime,cap.pri.17O.reg,col="black",lty=2)

Data2 = read.csv('TibetData 2.csv')


waters2 = read.csv('Waters2.csv')
points(waters2$d18Oprime,waters2$D17O,pch=23,bg="lightblue",cex=1)

d17O.prime.waters2 = waters2$D17O+0.528*waters2$d18Oprime
d17O.prime.waters = water$D17O+0.528*water$d18Oprime

summary(lm(d17O.prime.waters~water$d18Oprime))
summary(lm(d17O.prime.waters2~waters2$d18Oprime))


# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)




# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

#lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)

lines(d18O.reg.prime,all.cap17O.prime.reg1)
lines(d18O.reg.prime,all.cap17O.prime.reg2)
lines(d18O.reg.prime,all.cap17O.prime.reg3)
lines(d18O.reg.prime,all.cap17O.prime.reg4)


points(Data2$d18Oprime,Data2$D17Oprime,bg="orange",pch=21,cex=1.5)
points(Data2$d18Oprime[15:17],Data2$D17Oprime[15:17],bg="red",pch=21,cex=1.5)

for(i in 1:length(water$d18Oprime)){
  lines(rep(water$d18Oprime[i],2),c(water$D17O[i]+water$SE[i],water$D17O[i]-water$SE[i]),lwd=1.5)
}
points(water$d18Oprime,water$D17O,pch=23,bg="lightblue",cex=1.5)


for(i in 1:length(Data$d18O)){
  lines(rep(Data$d18Oprime[i],2),c(Data$D17Oprime[i]+Data$D17Oprime_SE[i],Data$D17Oprime[i]-Data$D17Oprime_SE[i]))
}
points(Data$d18Oprime,Data$D17Oprime,bg="orange",pch=21,cex=2)


points(Siend.d18Oprime,Siend.D17Oprime,pch=24,bg="grey",cex=2.5)
points(WaterSiend.d18Oprime,WaterSiend.D17Oprime,pch=24,bg="blue",cex=2.5)
Siend.d18Oprime;Siend.D17Oprime
WaterSiend.d18Oprime;WaterSiend.D17Oprime

Meteoric.d18O
Meteoric.d18O.1 - Meteoric.d18O
Meteoric.d18O.2 - Meteoric.d18O

legend("bottomleft",legend=c("Tibet Stream Data","Tibet Tap Water","Quartz","Whole Rock/Quartz (not in regression)","W/R = 100","Paleo-Meteoric Water","TibetMWL"),
       pch=c(23,23,21,21,24,24,NA),pt.bg=c("lightblue","lightblue","Orange","Red","Grey","Blue",NA),lwd=c(NA,NA,NA,NA,NA,NA,2),pt.cex=c(2,1.5,2,1.5,2,2,NA),
       cex=0.85,col=c("black","black","black","black","black","black","blue"))



# Meteoric water ofTibet with All Data ####
lambda = 0.5282287
gamma = 0.0364796

d18O.prime = seq(-70,-8,0.05)
d17O.prime = lambda*d18O.prime+gamma
cap17O.prime = d17O.prime - 0.528*d18O.prime

d17O = 1000*(exp(d17O.prime/1000)-1)
d18O = 1000*(exp(d18O.prime /1000)-1)
cap17O.curved = d17O - (0.528*d18O)

Temp = 273.15+240
d18O.alpha = exp((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))/1000)
d18O.Si =  d18O.alpha*(d18O+1000)-1000
d17O.alpha = exp(((((4.2*10^6)/(Temp^2))-((3.3*1000)/Temp))*((-1.85/Temp) + 0.5305))/1000)
d17O.Si = d17O.alpha*(d17O+1000)-1000
cap17O.curved.Si = d17O.Si - (0.528*d18O.Si)
#lines(d18O.Si,cap17O.curved.Si)

d18O.Si.lin = 1000*log(d18O.Si/1000+1)
cap17O.lin.Si = (1000*log(d17O.Si/1000+1)) - (0.528*(1000*log(d18O.Si/1000+1)))


# Read in data
Data = read.csv('TibetData ALL.csv')
Data$D17Oprime
d18O.points = Data$d18O
D17O.points = Data$d17O-0.528*Data$d18O
#points(d18O.points,D17O.points ,bg="orange",pch=21,cex=1.5)


water = read.csv("Waters.csv")
water.d18O = 1000*(-1+exp(water$d18Oprime/1000))
water.d17Oprime = water$d18Oprime*0.528+water$D17O
water.d17O = 1000*(-1+exp(water.d17Oprime/1000))
water.D17O.points = water.d17O-0.528*water.d18O

#points(water.d18O,water.D17O.points,pch=23,bg="lightblue",cex=1.5)

#lines(d18O.Si,cap17O.curved.Si)



All.data.regress = cbind(Data$d18O,rep(0.14,length(d18O.points)),Data$D17Oprime,Data$D17Oprime_SE,rep(0,length(Data$D17Oprime_SE)))
All.york = york(All.data.regress)
# abline(a=All.york$a,b=All.york$b,lwd=2)
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b-2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a+2*All.york$a[2],b=All.york$b+2*All.york$b[2])
# abline(a=All.york$a-2*All.york$a[2],b=All.york$b-2*All.york$b[2])



all.d18O.reg = seq(-30,20,0.05)
all.cap17O.reg = all.d18O.reg*All.york$b[1] + All.york$a[1]
all.d17O.reg = all.cap17O.reg+all.d18O.reg *0.528
#lines(all.d18O.reg,all.cap17O.reg,lwd=2)
#cap17O.test = all.cap17O.reg+all.d18O.reg*0.528

all.cap17O.reg1 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg1 = all.cap17O.reg1+all.d18O.reg *0.528
all.cap17O.reg2 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg2 = all.cap17O.reg2+all.d18O.reg *0.528
all.cap17O.reg3 = all.d18O.reg*(All.york$b[1]+2*All.york$b[2]) + (All.york$a[1]+2*All.york$a[2])
all.d17O.reg3 = all.cap17O.reg3+all.d18O.reg *0.528
all.cap17O.reg4 = all.d18O.reg*(All.york$b[1]-2*All.york$b[2]) + (All.york$a[1]-2*All.york$a[2])
all.d17O.reg4 = all.cap17O.reg4+all.d18O.reg *0.528



Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
#points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Siend.d18Oprime = 1000*log(Meteoric$x/1000 + 1)
Siend.d17O = Meteoric$y+0.528*Meteoric$x
Siend.d17Oprime = 1000*log(Siend.d17O/1000 + 1)
Siend.D17Oprime = Siend.d17Oprime - 0.528*Siend.d18Oprime



Meteoric.d18O = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O; Meteoric.D17O


#points(Meteoric.d18O,Meteoric.D17O,pch=23,bg="blue",cex=1.5)

WaterSiend.d18Oprime = 1000*log(Meteoric.d18O/1000 + 1)
WaterSiend.d17O = Meteoric.D17O+0.528*Meteoric.d18O
WaterSiend.d17Oprime = 1000*log(WaterSiend.d17O/1000 + 1)
WaterSiend.D17Oprime = WaterSiend.d17Oprime - 0.528*WaterSiend.d18Oprime



d18O.reg.prime = 1000*log(all.d18O.reg/1000 + 1)
d17O.reg.prime = 1000*log(all.d17O.reg/1000 + 1)
cap.pri.17O.reg = d17O.reg.prime - 0.528*d18O.reg.prime


all.cap17O.prime.reg1 = 1000*log(all.d17O.reg1/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg2 = 1000*log(all.d17O.reg2/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg3 = 1000*log(all.d17O.reg3/1000+1) - 0.528*d18O.reg.prime
all.cap17O.prime.reg4 = 1000*log(all.d17O.reg4/1000+1) - 0.528*d18O.reg.prime


#lines(c(Meteoric.d18O,Meteoric$x),c(Meteoric.D17O,Meteoric$y),lwd=2)
Meteoric.d18O
Meteoric.D17O
# 2 sigma error
Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg1),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.1 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.1 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.1 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.1; Meteoric.D17O.1
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


Meteoric = curve_intersect(data.frame(x=all.d18O.reg,y=all.cap17O.reg2),data.frame(x=d18O.Si,y=cap17O.curved.Si),domain=c(-30,-10))
# points(Meteoric$x,Meteoric$y,pch=23,cex=1.5,bg="grey")
Meteoric

Meteoric.d18O.2 = (Meteoric$x + 1000)/d18O.alpha-1000
Meteoric.d17O.2 = ((Meteoric$y + Meteoric$x *0.528) + 1000)/d17O.alpha-1000
Meteoric.D17O.2 = Meteoric.d17O - 0.528*Meteoric.d18O
Meteoric.d18O.2; Meteoric.D17O.2
# points(Meteoric.d18O,Meteoric.D17O,pch=3,cex=1.5,col="blue")


#par(mfrow=c(1,2))
# Linearized plot with curved mixing line
plot(d18O.prime,cap17O.prime,type="l",xlim=c(-25,12),ylim=c(-0.11,0.05),col="black",lwd=2,
     xlab="d'18O",ylab="cap'17O (theta=0.528)",las=1)
lines(d18O.reg.prime,cap.pri.17O.reg,col="black",lty=2)

Data2 = read.csv('TibetData 2.csv')


waters2 = read.csv('Waters2.csv')
points(waters2$d18Oprime,waters2$D17O,pch=23,bg="lightblue",cex=1)

d17O.prime.waters2 = waters2$D17O+0.528*waters2$d18Oprime
d17O.prime.waters = water$D17O+0.528*water$d18Oprime

summary(lm(d17O.prime.waters~water$d18Oprime))
summary(lm(d17O.prime.waters2~waters2$d18Oprime))


# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters2~waters2$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)




# Tibet Meteoric Water Line
lambda.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[2]
gamma.tibet = coef(lm(d17O.prime.waters~water$d18Oprime))[1]
d18O.prime.tibet = seq(-26,-10,0.05)
d17O.prime.tibet = lambda.tibet*d18O.prime.tibet+gamma.tibet
cap17O.prime.tibet = d17O.prime.tibet - 0.528*d18O.prime.tibet

#lines(d18O.prime.tibet,cap17O.prime.tibet,col="blue",lwd=2)

lines(d18O.reg.prime,all.cap17O.prime.reg1)
lines(d18O.reg.prime,all.cap17O.prime.reg2)
lines(d18O.reg.prime,all.cap17O.prime.reg3)
lines(d18O.reg.prime,all.cap17O.prime.reg4)

points(Data2$d18Oprime,Data2$D17Oprime,bg="orange",pch=21,cex=1.5)
points(Data2$d18Oprime[15:17],Data2$D17Oprime[15:17],bg="red",pch=21,cex=1.5)

for(i in 1:length(water$d18Oprime)){
  lines(rep(water$d18Oprime[i],2),c(water$D17O[i]+water$SE[i],water$D17O[i]-water$SE[i]),lwd=1.5)
}
points(water$d18Oprime,water$D17O,pch=23,bg="lightblue",cex=1.5)


for(i in 1:length(Data$d18O)){
  lines(rep(Data$d18Oprime[i],2),c(Data$D17Oprime[i]+Data$D17Oprime_SE[i],Data$D17Oprime[i]-Data$D17Oprime_SE[i]))
}
points(Data$d18Oprime,Data$D17Oprime,bg="darkgreen",pch=21,cex=2)
points(Data2$d18Oprime[1:14],Data2$D17Oprime[1:14],bg="orange",pch=21,cex=2)


points(Siend.d18Oprime,Siend.D17Oprime,pch=24,bg="grey",cex=2.5)
points(WaterSiend.d18Oprime,WaterSiend.D17Oprime,pch=24,bg="blue",cex=2.5)
Siend.d18Oprime;Siend.D17Oprime
WaterSiend.d18Oprime;WaterSiend.D17Oprime

Meteoric.d18O
Meteoric.d18O.1 - Meteoric.d18O
Meteoric.d18O.2 - Meteoric.d18O

legend("bottomleft",legend=c("Tibet Stream Data","Tibet Tap Water","Quartz","Whole Rock/Quartz (not in regression)","W/R = 100","Paleo-Meteoric Water","TibetMWL"),
       pch=c(23,23,21,21,24,24,NA),pt.bg=c("lightblue","lightblue","Orange","Red","Grey","Blue",NA),lwd=c(NA,NA,NA,NA,NA,NA,2),pt.cex=c(2,1.5,2,1.5,2,2,NA),
       cex=0.85,col=c("black","black","black","black","black","black","blue"))



