##load libraries and function

library(mclust)
library(tidyverse)
library(respR)
library(imputeTS)
library(respirometry)

calcSMR = function(Y, q=c(0.1,0.15,0.2,0.25,0.3), G=1:4){
  u = sort(Y)
  the.Mclust <- Mclust(Y,  G=G)
  cl <- the.Mclust$classification
  # sometimes, the class containing SMR is not called 1
  # the following presumes that when class 1 contains > 10% of cases, 
  # it contains SMR, otherwise we take class 2
  cl2 <- as.data.frame(table(cl))
  cl2$cl <- as.numeric(levels(cl2$cl))
  valid <- cl2$Freq>=0.1*length(time)  
  the.cl <- min(cl2$cl[valid])	
  left.distr <- Y[the.Mclust$classification==the.cl]
  mlnd = the.Mclust$parameters$mean[the.cl]
  CVmlnd = sd(left.distr)/mlnd * 100
  quant=quantile(Y, q)
  low10=mean(u[1:10])
  low10pc = mean(u[6:(5 + round(0.1*(length(u)-5)))])
  # remove 5 outliers, keep lowest 10% of the rest, average
  # Herrmann & Enders 2000
  return(list(mlnd=mlnd, quant=quant, low10=low10, low10pc=low10pc,
              cl=cl, CVmlnd=CVmlnd))
}

calcO2crit <- function(Data, SMR, lowestMO2=NA, gapLimit = 4, 
                       max.nb.MO2.for.reg = 20) 
{
  # programmed by Denis Chabot, Institut Maurice-Lamontagne, DFO, Canada
  # first version written in June 2009
  # last updated in January 2015
  
  method = "LS_reg"  # will become "through_origin" if intercept is > 0
  if(is.na(lowestMO2)) lowestMO2 = quantile(Data$MO2[Data$DO >= 70], p=0.05)
  
  # Step 1: identify points where MO2 is proportional to DO
  geqSMR = Data$MO2 >= lowestMO2
  pivotDO = min(Data$DO[geqSMR])
  lethal = Data$DO < pivotDO
  N_under_SMR = sum(lethal)  	# points available for regression?
  final_N_under_SMR = lethal  # some points may be removed at Step 4
  lastMO2reg = Data$MO2[Data$DO == pivotDO] # last MO2 when regulating
  if(N_under_SMR > 1) theMod = lm(MO2~DO, data=Data[lethal,])
  
  # Step 2, add one or more point at or above SMR
  # 2A, when there are fewer than 3 valid points to calculate a regression
  if(N_under_SMR < 3){
    missing = 3 - sum(lethal)
    not.lethal = Data$DO[geqSMR]
    DOlimit = max(sort(not.lethal)[1:missing])  # highest DO acceptable 
    #  to reach a N of 3
    addedPoints = Data$DO <= DOlimit
    lethal = lethal | addedPoints
    theMod = lm(MO2~DO, data=Data[lethal,])
  }  
  
  # 2B, add pivotDO to the fit when Step 1 yielded 3 or more values?
  if(N_under_SMR >= 3){
    lethalB = Data$DO <= pivotDO # has one more value than "lethal"
    regA = theMod
    regB = lm(MO2~DO, data=Data[lethalB,])
    large_slope_drop = (coef(regA)[2]/coef(regB)[2]) > 1.1 # arbitrary
    large_DO_gap = (max(Data$DO[lethalB]) - max(Data$DO[lethal])) > gapLimit  
    tooSmallMO2 = lastMO2reg < SMR
    if(!large_slope_drop & !large_DO_gap & !tooSmallMO2) {
      lethal = lethalB
      theMod = regB
    } # otherwise we do not accept the additional point
  } 
  
  # Step 3
  # if the user wants to limit the number of points in the regression
  if(!is.na(max.nb.MO2.for.reg) & sum(lethal)>max.nb.MO2.for.reg){
    Ranks = rank(Data$DO) 
    lethal = Ranks <= max.nb.MO2.for.reg
    theMod = lm(MO2~DO, data=Data[lethal,]) 
    final_N_under_SMR = max.nb.MO2.for.reg
  }
  
  # Step 4
  predMO2 = as.numeric(predict(theMod, data.frame(DO=Data$DO)))
  Data$delta = (Data$MO2-predMO2)/predMO2 * 100 # residuals set to zero  
  # when below pivotDO
  Data$delta[Data$DO < pivotDO | lethal] = 0
  tol = 0 # any positive residual is unacceptable
  HighValues = Data$delta > tol
  Ranks = rank(-1*Data$delta) 
  HighMO2 = HighValues & Ranks == min(Ranks)    # keep largest residual
  if (sum(HighValues) > 0) {
    nblethal = sum(lethal)	
    Data$W = NA
    Data$W[lethal]=1/nblethal
    Data$W[HighMO2] = 1
    theMod = lm(MO2~DO, weight=W, data=Data[lethal | HighMO2,])
    # This new regression is always an improvement, but there can still 
    # be points above the line, so we repeat
    predMO2_2 = as.numeric(predict(theMod, data.frame(DO=Data$DO)))
    Data$delta2 = (Data$MO2-predMO2_2)/predMO2_2 * 100
    Data$delta2[Data$DO < pivotDO] = 0
    tol = Data$delta2[HighMO2]
    HighValues2 = Data$delta2 > tol  
    if(sum(HighValues2)>0){
      Ranks2 = rank(-1*Data$delta2) 
      HighMO2_2 = HighValues2 & Ranks2 == 1  # keep the largest residual
      nblethal = sum(lethal)	
      Data$W = NA
      Data$W[lethal]=1/nblethal
      Data$W[HighMO2_2] = 1
      theMod2 = lm(MO2~DO, weight=W, data=Data[lethal | HighMO2_2,])
      # is new slope steeper than the old one? 
      if(theMod2$coef[2] > theMod$coef[2]) {
        theMod = theMod2
        HighMO2 = HighMO2_2
      }
    } # end second search for high value
  } # end first search for high value
  
  Coef = coefficients(theMod)
  
  #Step 5, check for positive intercept
  AboveOrigin = Coef[1] > 0
  # if it is, we use a regression that goes through the origin
  if (AboveOrigin){
    theMod = lm(MO2~DO -1, data=Data[lethal,])
    Coef = c(0, coefficients(theMod)) # need to add the intercept (0)
    #  manually to have a pair of coefficients
    method = "through_origin"
    HighMO2 = rep(FALSE, nrow(Data)) # did not use the additional value
    # from Step 4
  }
  
  po2crit = as.numeric(round((SMR - Coef[1]) / Coef[2], 1))
  sum_mod = summary(theMod)
  anov_mod = anova(theMod)
  O2CRIT = list(o2crit=po2crit, SMR=SMR, Nb_MO2_conforming = N_under_SMR, 
                Nb_MO2_conf_used = final_N_under_SMR,
                High_MO2_required = sum(HighMO2) == 1, origData=Data,
                Method=method, mod=theMod, r2 = sum_mod$r.squared, 
                P = anov_mod$"Pr(>F)", lethalPoints = which(lethal), 
                AddedPoints = which(HighMO2))
} # end function 

plotO2crit <- function(o2critobj, plotID="", Xlab="Dissolved oxygen (% sat.)",
                       Ylab="dotitalumol", smr.cex=0.9, o2crit.cex=0.9, plotID.cex=1.2, Transparency=T,...) 
{    
  # programmed by Denis Chabot, Institut Maurice-Lamontagne, DFO, Canada
  # first version written in June 2009
  # last updated in January 2015
  # for R plotting devices that do not support transparency (e.g., postscript), 
  #     set Transparency to FALSE
  
  smr = o2critobj$SMR
  if(Ylab %in% c("dotitalumol", "italumol", "dotumol", "umol",
                 "dotitalmg", "italmg", "dotmg", "mg")) {
    switch(Ylab,
           dotitalumol = {
             mo2.lab = expression(paste(italic(dot(M))[O[2]], " (",mu,"mol ", O[2],
                                        " ", min^-1, " ", kg^-1, ")"))
           }, 
           italumol = {
             mo2.lab = expression(paste(italic(M)[O[2]], " (",mu,"mol ", O[2], " ", 
                                        min^-1, " ", kg^-1, ")"))
           },
           dotumol = {
             mo2.lab = expression(paste(dot(M)[O[2]], " (",mu,"mol ", O[2], " ", 
                                        min^-1, " ", kg^-1, ")"))
           },
           umol = {
             mo2.lab = expression(paste(M[O[2]], " (",mu,"mol ", O[2], " ", min^-1, 
                                        " ", kg^-1, ")"))
           },
           dotitalmg = {
             mo2.lab = expression(paste(italic(dot(M))[O[2]], " (mg ", O[2], " ", 
                                        h^-1, " ", kg^-1, ")"))
           },
           italmg = {
             mo2.lab = expression(paste(italic(M)[O[2]], " (mg ", O[2], " ", 
                                        h^-1, " ", kg^-1, ")"))
           },
           dotmg = {
             mo2.lab = expression(paste(dot(M)[O[2]], " (mg ", O[2], " ", h^-1, " ", 
                                        kg^-1, ")"))
           },
           mg = {
             mo2.lab = expression(paste(M[O[2]], " (mg ", O[2], " ", h^-1, " ", 
                                        kg^-1, ")"))
           } 
    )
  } else mo2.lab=Ylab
  
  if(Transparency) Col=c(rgb(0,0,0,0.7), "red", "orange") else Col=c(grey(0.3), "red", "orange")
  Data=o2critobj$origData
  Data$Color = Col[1]
  Data$Color[o2critobj$lethalPoints] = Col[2]
  Data$Color[o2critobj$AddedPoints] = Col[3]
  # ordinary LS regression, without added points: blue line, red symbols
  # ordinary LS regression, with added points: blue line, red & orange symbls
  # regression through origin, green dotted line, red symbols
  line.color = ifelse(o2critobj$Method=="LS_reg", "blue", "darkgreen")
  line.type = ifelse(o2critobj$Method=="LS_reg", 1, 3)
  limX = c(0, max(Data$DO))
  limY = c(0, max(Data$MO2))
  plot(MO2~DO, data=Data, xlim=limX, ylim=limY, col=Data$Color, xlab=Xlab, 
       ylab=mo2.lab, ...)
  coord <- par("usr")
  if(plotID != ""){
    text(0, coord[4], plotID, cex=plotID.cex, adj=c(0,1.2))    
  }
  
  abline(h=smr, col="orange")
  text(coord[1], smr, "SMR", adj=c(-0.1,1.3), cex=smr.cex)
  text(coord[1], smr, round(smr,1), adj=c(-0.1,-0.3), cex=smr.cex)
  if(!is.na(o2critobj$o2crit)) {
    abline(o2critobj$mod, col=line.color, lty=line.type)
    segments(o2critobj$o2crit, smr, o2critobj$o2crit, coord[3], 
             col=line.color, lwd=1)
    text(x=o2critobj$o2crit, y=0, o2critobj$o2crit, col=line.color, 
         cex=o2crit.cex, adj=c(-0.1,0.5))
  } 
} # end of function


##------------------------##
## estimate smr and pcrit ##
##------------------------##

mass = 29.2611
vol = 275
temp = 7
pcrit_start = 87000
pcrit_end = 122000

a <- seq(60, 87000, 900) #define measurement periods and check a & b
b <- seq(240,87000,900)

## get the data formatted

data_1 <- read.csv("jumars_c1.csv", header = TRUE) #load resp data
blank <- read.csv("blank.csv", header = TRUE) #load blank
date_time <- read.csv("date_time.csv", header = TRUE) #load a date_time dataframe per second

data <- left_join(date_time, data_1, by = c("date", "time")) #combine into one dataframe
data <- left_join(data, blank, by = c("date", "time"))

data <- data[,c(1,2,4,7)] #remove unecessary columns

data$seq <- seq(1,nrow(data),1) #add a sequence of numbers representing seconds

names(data)[3] <- "oxy" #rename columns
names(data)[4] <- "blank"

## Calculate SMR from intermittent flow data 

smr_data <- subset(data, seq < pcrit_start) #subset the data to exclude Pcrit drawdown measurement

meas_data <- data.frame(time = smr_data$seq, c = smr_data$oxy) # get measurement channel

meas_data$c <- na_interpolation(meas_data$c, option = 'linear') #interpolate missing values
inspect(meas_data) #check

c <- calc_rate(meas_data, from = c(a), 
               to = c(b), by = 'time') #calculate slope of each measurement period

plot(c, pos = 33, panel = 1, legend = FALSE)


slope_data <- data.frame(slope = c$summary$rate_b1, 
                         r2 = c$summary$rsq,
                         oxy_start = c$summary$oxy, 
                         oxy_end = c$summary$endoxy) #create a new dataframe of measurements

blank_data <- data.frame(time = smr_data$seq, c = smr_data$blank) #get blank from firesting

blank_data$c <- na_interpolation(blank_data$c, option = 'linear') #interpolate missing values

d <- calc_rate(blank_data, from = c(a), 
               to = c(b), by = 'time') #get blank slope, ignore r2

slope_data$blank <- d$summary$rate_b1 #add to dataframe

slope_data$mr <- (((vol - mass)/mass)*(slope_data$slope*-60)) - ((((vol - mass)/mass)*(slope_data$blank*-60))*(vol/(vol - mass))) #estimate metabolic rate

slope_data$oxy <- (slope_data$oxy_start + slope_data$oxy_end)/2 #calculate average oxygen in chamber during measurement

slope_data_all <- slope_data #rename and store for later calculation

slope_data <- subset(slope_data, r2 > 0.95) #exclude bad measurements
slope_data <- subset(slope_data, blank  > quantile(slope_data$blank, 0.05)) #exclude the far out blanks as they likely occurred during a temp change
slope_data <- subset(slope_data, blank  < quantile(slope_data$blank, 0.95))

hist(slope_data$mr) #check the histogram

e <- calcSMR(slope_data$mr) #calculate SMR
smr <- as.numeric(e$quant[1]) #take the 0.1% as SMR

## Calculate metabolic rates during draw down

pcrit_data <- subset(data, seq > pcrit_start) #subset the data to include Pcrit measurement only
plot(pcrit_data$oxy ~ pcrit_data$seq) #plot to check

meas_data_2 <- data.frame(time = pcrit_data$seq, c = pcrit_data$oxy) #get measurement data
meas_data_2$c <- na_interpolation(meas_data_2$c, option = 'linear') #interpolate missing values


meas_data_2 <- subset(meas_data_2, time < pcrit_end) # cut tail shorter
plot(meas_data_2$c ~ meas_data_2$time) # check plot to check

inspect(meas_data_2) #check

f <- seq(min(meas_data_2$time), max(meas_data_2$time)-299, 300) #define measurement areas for every 5 mins
g <- seq(min(meas_data_2$time)+299,max(meas_data_2$time),300)

h <- calc_rate(meas_data_2, from = c(f), 
               to = c(g), by = 'time') #calculate slope of each measurement period

slope_data_pcrit <- data.frame(slope = h$summary$rate_b1, 
                               r2 = h$summary$rsq, 
                               oxy_start = h$summary$oxy, 
                               oxy_end = h$summary$endoxy) #create a dataframe

slope_data_pcrit$mr <- (((vol - mass)/mass)*(slope_data_pcrit$slope*-60)) - ((((vol - mass)/mass)*(mean(slope_data$blank)*-60))*(vol/(vol - mass))) #estimate metabolic rate

slope_data_pcrit$oxy <- (slope_data_pcrit$oxy_start + slope_data_pcrit$oxy_end)/2 #estimate corresponding oxygen

## Calculate Pcrit

final_data <- dplyr::bind_rows(slope_data, slope_data_pcrit) #create a combined dataframe
final_data <- dplyr::select(final_data, mr, oxy)
final_data <- data.frame(MO2 = final_data$mr, DO = final_data$oxy) #rename for function
final_data$DO <- conv_o2(o2 = final_data$DO, from = "mg_per_l", to = "percent_a.s.", temp = temp, sal = 35, atm_pres = 1013.25)
final_data <- subset(final_data, MO2 > 0)

j <- calcO2crit(Data = final_data, SMR = smr, max.nb.MO2.for.reg = 5)
j$r2
j$o2crit
plotO2crit(o2critobj = j)

calc_pcrit_data <- subset(final_data, DO < 90)
respirometry::plot_pcrit(calc_pcrit_data$DO, calc_pcrit_data$MO2, MR = smr)

## ------- ##
## results ##
## ------- ##

smr # standard metabolic rate
j$o2crit #po2crit
respirometry::calc_pcrit(calc_pcrit_data$DO, calc_pcrit_data$MO2, MR = smr) #po2crit alpha
nrow(slope_data_all) #total number of measurment periods for smr
nrow(slope_data) #number of slopes used after filtering bad measuremnts
mean(slope_data$blank)/mean(slope_data$slope)*100 # blank as proportion of measurement
conv_o2(o2 = min(na.omit(smr_data$oxy)), from = "mg_per_l", to = "percent_a.s.", 
        temp = temp, sal = 35, atm_pres = 1013.25) #lowest oxy value during intermittent flow




