data <- read.csv("data.csv", header = TRUE)

library(tidyverse)
library(respirometry)
library(viridis)
library(ggpubr)
library(gridExtra)

data$pcrit_kpa <- conv_o2(data$pcrit, from = "percent_a.s.", to = "kPa", temp = data$temp, sal = 35, atm_pres = 1013.25) # convert oxy units to kpa

kelvin <- 273.15 # kelvin constant
kb <- 8.6173324e-5 # Boltzmann constant

data$inv_temp_arr <- 1/((data$temp + kelvin) * kb) # arrenhius temp column

urchin_data <- subset(data, species == "purple urchin") # split into data frames
abalone_data <- subset(data, species == "red abalone")

## supp figure of pcrit comparisons ##

pcrit_comp_urchin <- ggplot(data = subset(data, species == "purple urchin"), aes(x = pcrit, y = pcrit_alpha))+
  geom_point(colour = 'purple', size = 4, alpha = 0.3)+
  geom_smooth(method = 'lm', colour = 'black', fill = NA)+
  labs(x=expression(paste("Pcrit"))) +
  labs(y=expression(paste("Pcrit alpha")))+
  scale_x_continuous(breaks = seq(0,40,5), limits = c(0,40)) +
  scale_y_continuous(breaks = seq(0,40,5), limits = c(0,40)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  stat_regline_equation(label.y = 40, label.x = 2.5, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 35,label.x = 2.5, aes(label = ..rr.label..)) +
  annotate('text', x = 35, y = 40, label = 'purple urchin')+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=0, y=40, label="(a)", fontface = "bold")

pcrit_comp_abalone <- ggplot(data = subset(data, species == "red abalone"), aes(x = pcrit, y = pcrit_alpha))+
  geom_point(colour = 'firebrick', size = 4, alpha = 0.3)+
  geom_smooth(method = 'lm', colour = 'black', fill = NA)+
  labs(x=expression(paste("Pcrit"))) +
  labs(y=expression(paste("Pcrit alpha")))+
  scale_x_continuous(breaks = seq(0,55,5), limits = c(0,55)) +
  scale_y_continuous(breaks = seq(0,55,5), limits = c(0,55)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  stat_regline_equation(label.y = 55, label.x = 2.5,aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 50, label.x = 2.5,aes(label = ..rr.label..)) +
  annotate('text', x = 45, y = 55, label = 'red abalone')+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=0, y=55, label="(b)", fontface = "bold")

fig_s4 <- grid.arrange(pcrit_comp_urchin , pcrit_comp_abalone , ncol = 2)
ggsave("fig_s4.pdf", plot = fig_s4, width = 8, height = 3)

##-----------##
##  urchins  ##
##-----------##

##-------##
##  SMR  ##
##-------##

ggplot(data = urchin_data, aes(x = inv_temp_arr, y = log(smr)))+ # check arrhenius data
  geom_point()+
  geom_smooth(method = 'lm')

smr_arr_mod_urchin <- lm(log(smr) ~ inv_temp_arr, data = urchin_data) # model temp relationship

smr_arr_mod_urchin$coefficients[2] # check slope

urchin_data$temp_norm_smr <- urchin_data$smr*(exp((smr_arr_mod_urchin$coefficients[2]*-1)/((urchin_data$temp + kelvin)*kb))) #temp normalize smr data

ggplot(data = urchin_data, aes(x = temp, y = temp_norm_smr))+ # check temp normalized smr data
  geom_point()+
  geom_smooth(method = 'lm')

mass_scaler_mod_urchin <- lm(log(temp_norm_smr) ~ log(wet_mass), data = urchin_data) # model temp normalized smr vs mass relationship

summary(mass_scaler_mod_urchin)

mass_scaler_urchin <- as.numeric(mass_scaler_mod_urchin$coefficients[2]) 

ggplot(data = urchin_data, aes(x = log(wet_mass), y = log(temp_norm_smr)))+ # check temp normalized smr data
  geom_point(colour = 'purple', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Ln(wet mass) (g)"))) +
  labs(y=expression(paste("Ln(SMR"[temp_normalized],") (O" [2], ".min"^{-1},".g"^{-1},")")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 24.9, label.x = 3, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 24.8, label.x = 3, aes(label = ..rr.label..)) +
  annotate('text', x = 4.5, y = 24.9, label = 'purple urchin')+
  annotate(geom="text", x=2.7, y=24.9, label="(a)", fontface = "bold")+
  ggsave('fig_s1a.pdf', height = 3, width = 4)

mass_scaler_urchin <- as.numeric(mass_scaler_mod_urchin$coefficients[2]) # get smr mass scaling relationship for urchin

urchin_data$mass_norm_smr <- urchin_data$smr/(urchin_data$wet_mass^mass_scaler_urchin)

ggplot(data = urchin_data, aes(x = temp, y = mass_norm_smr))+ # check mass normalized smr data
  geom_point()

ggplot(data = urchin_data, aes(x = inv_temp_arr, y = log(mass_norm_smr)))+ # check mass normalized smr data
  geom_point()+
  geom_smooth(method = 'lm')

ggplot(data = urchin_data, aes(x = inv_temp_arr, y = log(mass_norm_smr)))+ # check temp normalized smr data
  geom_point(colour = 'purple', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Temperature (1/(Kb.T)"))) +
  labs(y=expression(paste("Ln(SMR.B"^{0.14},") (O" [2], ".min"^{-1},".g"^{-1},")")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 0.5, label.x = 39.6, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 0.35, label.x = 39.6, aes(label = ..rr.label..)) +
  annotate('text', x = 41.2, y = 0.5, label = 'purple urchin')+
  annotate(geom="text", x=39.3, y=0.5, label="(b)", fontface = "bold")+
  ggsave('fig_s1b.pdf', height = 3, width = 4)

smr_arr_mod_final_urchin <- lm(log(mass_norm_smr) ~ inv_temp_arr, data = urchin_data) # model arrhenius temperature relationship
summary(smr_arr_mod_final_urchin)
smr_slope_urchin <- as.numeric(smr_arr_mod_final_urchin$coefficients[2]) # get arrhenius model parameters
smr_intercept_urchin <- exp(as.numeric(smr_arr_mod_final_urchin$coefficients[1]))

min(urchin_data$wet_mass)
max(urchin_data$wet_mass)

smr_urchin_predict_df <- data.frame(temp = seq(0,35,1)) # get smr predictions for different masses with calibrated eqn
smr_urchin_predict_df$smr_10g <- smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(smr_urchin_predict_df$temp+kelvin))))*(15^mass_scaler_urchin)
smr_urchin_predict_df$smr_150g <- smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(smr_urchin_predict_df$temp+kelvin))))*(150^mass_scaler_urchin)

smr_urchin_fig <- ggplot() + # make figure of raw experimental data and arrhenius models for two sizes of urchin
  geom_point(data = urchin_data, aes(x = temp, y = smr),colour = 'purple', size = 4, alpha = 0.3) +
  geom_line(data = smr_urchin_predict_df, aes(x = temp, y = smr_10g)) +
  geom_line(data = smr_urchin_predict_df, aes(x = temp, y = smr_150g)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y=expression(paste("SMR (O" [2], ".min"^{-1},".g"^{-1},")")))+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35)) +
  scale_y_continuous(breaks = seq(0,1.5,0.5), limits = c(0,1.5)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  annotate('text', x = 27, y = 0.6, label = '150g')+
  annotate('text', x = 23, y = 1, label = '15g')+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=0, y=1.5, label="(a)", fontface = "bold")

##---------##
##  pcrit  ##
##---------##

ggplot(data = urchin_data, aes(x = inv_temp_arr, y = log(pcrit_kpa)))+ # check arrhenius for pcrit and work only with RHS
  geom_point()+
  geom_smooth(method = 'lm')

ggplot(data = subset(urchin_data, temp > 7), aes(x = inv_temp_arr, y = log(pcrit_kpa)))+ # check arrhenius for pcrit and work only with RHS
  geom_point()+
  geom_smooth(method = 'lm')

pcrit_arr_mod_urchin <- lm(log(pcrit_kpa) ~ inv_temp_arr, data = subset(urchin_data, temp > 7))
pcrit_arr_mod_urchin$coefficients[2] # check slope

urchin_data$temp_norm_pcrit <- urchin_data$pcrit_kpa*(exp((pcrit_arr_mod_urchin$coefficients[2]*-1)/((urchin_data$temp + kelvin)*kb))) #temp normalise pcrit data

ggplot(data = subset(urchin_data, temp > 7), aes(x = inv_temp_arr, y = temp_norm_pcrit))+ # check arrhenius for pcrit
  geom_point()+
  geom_smooth(method = 'lm')

mass_scaler_pcrit_mod_urchin <- lm(log(temp_norm_pcrit) ~ log(wet_mass), data = subset(urchin_data, temp > 7)) # model temp normalized pcrit vs mass relationship

summary(mass_scaler_pcrit_mod_urchin)

mass_scaler_pcrit_urchin <- as.numeric(mass_scaler_pcrit_mod_urchin$coefficients[2]) # get pcrit mass scaling relationship for urchin

ggplot(data = subset(urchin_data, temp > 7), aes(x = log(wet_mass), y = log(temp_norm_pcrit)))+ # check temp normalized smr data
  geom_point(colour = 'purple', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Ln(wet mass) (g)"))) +
  labs(y=expression(paste("Ln(Pcrit"[temp_normalized],") (kPa)")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 27.7, label.x = 2.7, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 27.6, label.x = 2.7,  aes(label = ..rr.label..)) +
  annotate('text', x = 4.5, y = 27.7, label = 'purple urchin')+
  annotate(geom="text", x=2.4, y=27.7, label="(c)", fontface = "bold")+
  ggsave('fig_s1c.pdf', height = 3, width = 4)


urchin_data$mass_norm_pcrit <- urchin_data$pcrit_kpa/(urchin_data$wet_mass^mass_scaler_pcrit_urchin) # get mass normalised pcrit

ggplot(data = urchin_data, aes(x = temp, y = mass_norm_pcrit))+ # check mass normalised smr data
  geom_point()

pcrit_mod <- lm(mass_norm_pcrit ~ poly(temp, 2, raw = TRUE), data = urchin_data) # model pcrit data with a polynomial in lieu of MI 2.0
p <- pcrit_mod$coefficients[1]
x <- pcrit_mod$coefficients[2]
x2 <- pcrit_mod$coefficients[3]

summary(pcrit_mod)

formula <- y ~ poly(x, 2, raw = TRUE)

ggplot(data = urchin_data, aes(x = temp, y = mass_norm_pcrit))+ # check temp normalized smr data
  geom_point(colour = 'purple', size = 4, alpha = 0.3) +
  geom_smooth(method='lm', formula = formula, colour = "black")+
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y=expression(paste("Pcrit.B"^{-0.08}," (kPa)"))) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 5.7, label.x = 6, aes(label =  paste(..eq.label..)), formula = formula) +
  stat_regline_equation(label.y = 5.2, label.x = 6, aes(label = paste(..rr.label..))) +
  annotate('text', x = 19, y = 5.7, label = 'purple urchin')+
  annotate(geom="text", x=5, y=5.7, label="(d)", fontface = "bold")+
  ggsave('fig_s1d.pdf', height = 3, width = 4)

pcrit_urchin_predict_df <- data.frame(temp = seq(0,35,1)) # create prediction dataframe

pcrit_urchin_predict_df$pcrit_10g <- ((pcrit_urchin_predict_df$temp^2*x2) + (pcrit_urchin_predict_df$temp*x) + p)*(15^mass_scaler_pcrit_urchin) # predict for different sized organism
pcrit_urchin_predict_df$pcrit_150g <- ((pcrit_urchin_predict_df$temp^2*x2) + (pcrit_urchin_predict_df$temp*x) + p)*(150^mass_scaler_pcrit_urchin)

pcrit_urchin_fig <- ggplot() + # make figure of raw experimental data and pcrit models for two sizes of urchin
  geom_point(data = urchin_data, aes(x = temp, y = pcrit_kpa),colour = 'purple', size = 4, alpha =0.3) +
  geom_line(data = pcrit_urchin_predict_df, aes(x = temp, y = pcrit_10g)) +
  geom_line(data = pcrit_urchin_predict_df, aes(x = temp, y = pcrit_150g)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y = expression(paste(italic("PO")["2crit"])))+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35)) +
  scale_y_continuous(breaks = seq(0,18,3), limits = c(0,18)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  annotate('text', x = 27, y = 6.5, label = '15g')+
  annotate('text', x = 23, y = 9.5, label = '150g')+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=0, y=18, label="(c)", fontface = "bold")



##-----------##
##  abalone  ##
##-----------##

##-------##
##  SMR  ##
##-------##

ggplot(data = abalone_data, aes(x = inv_temp_arr, y = log(smr)))+ # check arrhenius data
  geom_point()+
  geom_smooth(method = 'lm')

smr_arr_mod_abalone <- lm(log(smr) ~ inv_temp_arr, data = abalone_data) # model temp relationship

smr_arr_mod_abalone$coefficients[2] # check slope

abalone_data$temp_norm_smr <- abalone_data$smr*(exp((smr_arr_mod_abalone$coefficients[2]*-1)/((abalone_data$temp + kelvin)*kb))) #temp normalize smr data

ggplot(data = abalone_data, aes(x = temp, y = temp_norm_smr))+ # check temp normalized smr data
  geom_point()+
  geom_smooth(method = 'lm')

mass_scaler_mod_abalone <- lm(log(temp_norm_smr) ~ log(wet_mass), data = abalone_data) # model temp normalized smr vs mass relationship

summary(mass_scaler_mod_abalone)

mass_scaler_abalone <- as.numeric(mass_scaler_mod_abalone$coefficients[2]) # get smr mass scaling relationship for abalone

ggplot(data = abalone_data, aes(x = log(wet_mass), y = log(temp_norm_smr)))+ # check temp normalized smr data
  geom_point(colour = 'firebrick', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Ln(wet mass) (g)"))) +
  labs(y=expression(paste("Ln(SMR"[temp_normalized],") (O" [2], ".min"^{-1},".g"^{-1},")")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 29.5, label.x = 3.4, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 29.4, label.x = 3.4, aes(label = ..rr.label..)) +
  annotate('text', x = 3.8, y = 29.5, label = 'red abalone')+
  annotate(geom="text", x=3.3, y=29.5, label="(a)", fontface = "bold")+
  ggsave('fig_s2a.pdf', height = 3, width = 4)

abalone_data$mass_norm_smr <- abalone_data$smr/(abalone_data$wet_mass^mass_scaler_abalone)

ggplot(data = abalone_data, aes(x = temp, y = mass_norm_smr))+ # check mass normalized smr data
  geom_point()

ggplot(data = abalone_data, aes(x = inv_temp_arr, y = log(mass_norm_smr)))+ # check mass normalized smr data
  geom_point()+
  geom_smooth(method = 'lm')

smr_arr_mod_final_abalone <- lm(log(mass_norm_smr) ~ inv_temp_arr, data = abalone_data) # model arrhenius temperature relationship

summary(smr_arr_mod_final_abalone)

smr_slope_abalone <- as.numeric(smr_arr_mod_final_abalone$coefficients[2]) # get arrhenius model parameters
smr_intercept_abalone <- exp(as.numeric(smr_arr_mod_final_abalone$coefficients[1]))


ggplot(data = abalone_data, aes(x = inv_temp_arr, y = log(mass_norm_smr)))+ # check temp normalized smr data
  geom_point(colour = 'firebrick', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Temperature (1/(Kb.T)"))) +
  labs(y=expression(paste("Ln(SMR.B"^{0.4},") (O" [2], ".min"^{-1},".g"^{-1},")")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 1.95, label.x = 39.5, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 1.8, label.x = 39.5, aes(label = ..rr.label..)) +
  annotate('text', x = 41, y = 1.95, label = 'red abalone')+
  annotate(geom="text", x=39.3, y=1.95, label="(b)", fontface = "bold")+
  ggsave('fig_s2b.pdf', height = 3, width = 4)

min(abalone_data$wet_mass)
max(abalone_data$wet_mass)

smr_abalone_predict_df <- data.frame(temp = seq(0,35,1)) # get smr predictions for different masses with calibrated eqn
smr_abalone_predict_df$smr_10g <- smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(smr_abalone_predict_df$temp+kelvin))))*(25^mass_scaler_abalone)
smr_abalone_predict_df$smr_150g <- smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(smr_abalone_predict_df$temp+kelvin))))*(50^mass_scaler_abalone)

smr_abalone_fig <- ggplot() + # make figure of raw experimental data and arrhenius models for two sizes of abalone
  geom_point(data = abalone_data, aes(x = temp, y = smr),colour = 'firebrick', size = 4, alpha = 0.3) +
  geom_line(data = smr_abalone_predict_df, aes(x = temp, y = smr_10g)) +
  geom_line(data = smr_abalone_predict_df, aes(x = temp, y = smr_150g)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y=expression(paste("SMR (O" [2], ".min"^{-1},".g"^{-1},")")))+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35)) +
  scale_y_continuous(breaks = seq(0,2.5,0.5), limits = c(0,2.6)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  annotate('text', x = 26, y = 1.1, label = '50g')+
  annotate('text', x = 23, y = 1.7, label = '25g')+
  annotate(geom="text", x=0, y=2.6, label="(b)", fontface = "bold")+
  theme(text = element_text(size=15))

##---------##
##  pcrit  ##
##---------##

ggplot(data = abalone_data, aes(x = inv_temp_arr, y = log(pcrit_kpa)))+ # check arrhenius for pcrit and work only with RHS
  geom_point()+
  geom_smooth(method = 'lm')

ggplot(data = subset(abalone_data, temp > 7), aes(x = inv_temp_arr, y = log(pcrit_kpa)))+ # check arrhenius for pcrit and work only with RHS
  geom_point()+
  geom_smooth(method = 'lm')

pcrit_arr_mod_abalone <- lm(log(pcrit_kpa) ~ inv_temp_arr, data = subset(abalone_data, temp > 7))
pcrit_arr_mod_abalone$coefficients[2] # check slope

abalone_data$temp_norm_pcrit <- abalone_data$pcrit_kpa*(exp((pcrit_arr_mod_abalone$coefficients[2]*-1)/((abalone_data$temp + kelvin)*kb))) #temp normalise pcrit data

ggplot(data = subset(abalone_data, temp > 7), aes(x = inv_temp_arr, y = temp_norm_pcrit))+ # check arrhenius for pcrit
  geom_point()+
  geom_smooth(method = 'lm')

mass_scaler_pcrit_mod_abalone <- lm(log(temp_norm_pcrit) ~ log(wet_mass), data = subset(abalone_data, temp > 7)) # model temp normalized pcrit vs mass relationship

summary(mass_scaler_pcrit_mod_abalone)

mass_scaler_pcrit_abalone <- as.numeric(mass_scaler_pcrit_mod_abalone$coefficients[2]) # get pcrit mass scaling relationship for abalone


ggplot(data = subset(abalone_data, temp > 7), aes(x = log(wet_mass), y = log(temp_norm_pcrit)))+ # check temp normalized smr data
  geom_point(colour = 'firebrick', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Ln(wet mass) (g)"))) +
  labs(y=expression(paste("Ln(Pcrit"[temp_normalized],") (kPa)")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 19, label.x = 3.35, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 18.95, label.x = 3.35, aes(label = ..rr.label..)) +
  annotate('text', x = 3.8, y = 19, label = 'red abalone')+
  annotate(geom="text", x=3.3, y=19, label="(c)", fontface = "bold")+
  ggsave('fig_s2c.pdf', height = 3, width = 4)

abalone_data$mass_norm_pcrit <- abalone_data$pcrit_kpa/(abalone_data$wet_mass^mass_scaler_pcrit_abalone) # get mass normalised pcrit

ggplot(data = abalone_data, aes(x = temp, y = mass_norm_pcrit))+ # check mass normalised smr data
  geom_point()


pcrit_mod_abalone <- lm(log(mass_norm_pcrit) ~ temp, data = abalone_data) # model pcrit data with an exponential in lieu of MI 2.0

summary(pcrit_mod_abalone)

m <- exp(pcrit_mod_abalone$coefficients[1])
n <- pcrit_mod_abalone$coefficients[2]

ggplot(data = subset(abalone_data), aes(x = temp, y = log(mass_norm_pcrit)))+ # check temp normalized smr data
  geom_point(colour = 'firebrick', size = 4, alpha = 0.3) +
  geom_smooth(method = "lm", colour = "black")+
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y=expression(paste("Ln(Pcrit.B"^{-0.07},") (kPa)")))+
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=11))+
  stat_regline_equation(label.y = 2.2, label.x = 8.5, aes(label = ..eq.label..)) +
  stat_regline_equation(label.y = 2.1, label.x = 8.5, aes(label = ..rr.label..)) +
  annotate('text', x = 18, y = 2.2, label = 'red abalone')+
  annotate(geom="text", x=7, y=2.2, label="(d)", fontface = "bold")+
  ggsave('fig_s2d.pdf', height = 3, width = 4)

pcrit_abalone_predict_df <- data.frame(temp = seq(0,35,1)) # create prediction dataframe

pcrit_abalone_predict_df$pcrit_25g <- (m*(exp(pcrit_abalone_predict_df$temp*n)))*(25^mass_scaler_pcrit_abalone) # predict for different sized organism
pcrit_abalone_predict_df$pcrit_50g <- (m*(exp(pcrit_abalone_predict_df$temp*n)))*(50^mass_scaler_pcrit_abalone)

pcrit_abalone_fig <- ggplot() + # make figure of raw experimental data and pcrit models for two sizes of abalone
  geom_point(data = abalone_data, aes(x = temp, y = pcrit_kpa),colour = 'firebrick', size = 4, alpha =0.3) +
  geom_line(data = pcrit_abalone_predict_df, aes(x = temp, y = pcrit_25g)) +
  geom_line(data = pcrit_abalone_predict_df, aes(x = temp, y = pcrit_50g)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)"))) +
  labs(y = expression(paste(italic("PO")["2crit"])))+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35)) +
  scale_y_continuous(breaks = seq(0,18,3), limits = c(0,18)) +
  guides(color=FALSE) +
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  annotate('text', x = 27, y = 11, label = '25g')+
  annotate('text', x = 22.5, y = 13, label = '50g')+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=0, y=18, label="(d)", fontface = "bold")


fig_3 <- grid.arrange(smr_urchin_fig, smr_abalone_fig, pcrit_urchin_fig, pcrit_abalone_fig, ncol = 2)
ggsave("fig_3.pdf", plot = fig_3, width = 8, height = 6)



##-------##
##  AMI  ##
##-------##

#############
## urchins ##
#############

## create data frame for plots

oxygen <- rep(1:23,each = 41)
temp <- rep(seq(0,40,1),23)
ami_df_urchin <- data.frame(oxygen, temp)

mass_urchin <- 67.5 # mid point of urchins tested

ami_df_urchin$ami <- (ami_df_urchin$oxygen*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(ami_df_urchin$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                                (((ami_df_urchin$temp^2*x2) + (ami_df_urchin$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(ami_df_urchin$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))

ami_df_urchin$ami <- ifelse(ami_df_urchin$ami <= 0, NA, ami_df_urchin$ami)

##------------------------##
##  occurrence data data  ##
##------------------------##

library(robis)
library(raster)
library(oce)
library(rgbif)
library(marmap)

# OBIS
occ_obis_urchin <- occurrence("Strongylocentrotus purpuratus") # pull occurence data from OBIS
occ_df_obis_urchin <- data.frame(lon = occ_obis_urchin$decimalLongitude, lat = occ_obis_urchin$decimalLatitude) # create a data frame
occ_df_obis_urchin <- dplyr::distinct(occ_df_obis_urchin, lon, lat) #remove duplicated co ordinates

occ_df_urchin <- occ_df_obis_urchin

load("../ocean_model/current_bottom_temp_monthly.RDA") # load environmental dataframes
temp_df <- data.frame(df)

load("../ocean_model/current_bottom_oxy_monthly.RDA")
oxy_df <- data.frame(df)

load("../ocean_model/current_bottom_salt_monthly.RDA")
salt_df <- data.frame(df)

remove(df)

bath <- raster("../ocean_model/bath_grid.tif") #load bath from different folder
bath <- aggregate(bath, 8, fun=mean) #aggregate further to reduce size

depth <- resample(griddify(dplyr::select(temp_df,1,2,3), nlon = 322, nlat=450), bath, method='bilinear') #resample to create rasters for each month
temp <- resample(griddify(dplyr::select(temp_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 

jan <- stack(depth,temp,oxy,salt)
names(jan) <- c('depth','temp','oxy','salt')
jan_df_urchin <- raster::extract(jan, occ_df_urchin)
jan_df_urchin <- cbind(occ_df_urchin, jan_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear')

feb <- stack(depth,temp,oxy,salt)
names(feb) <- c('depth','temp','oxy','salt')
feb_df_urchin <- raster::extract(feb, occ_df_urchin)
feb_df_urchin <- cbind(occ_df_urchin, feb_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear')

mar <- stack(depth,temp,oxy,salt)
names(mar) <- c('depth','temp','oxy','salt')
mar_df_urchin <- raster::extract(mar, occ_df_urchin)
mar_df_urchin <- cbind(occ_df_urchin, mar_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear')

apr <- stack(depth,temp,oxy,salt)
names(apr) <- c('depth','temp','oxy','salt')
apr_df_urchin <- raster::extract(apr, occ_df_urchin)
apr_df_urchin <- cbind(occ_df_urchin, apr_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear')

may <- stack(depth,temp,oxy,salt)
names(may) <- c('depth','temp','oxy','salt')
may_df_urchin <- raster::extract(may, occ_df_urchin)
may_df_urchin <- cbind(occ_df_urchin, may_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear')

jun <- stack(depth,temp,oxy,salt)
names(jun) <- c('depth','temp','oxy','salt')
jun_df_urchin <- raster::extract(jun, occ_df_urchin)
jun_df_urchin <- cbind(occ_df_urchin, jun_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear')

jul <- stack(depth,temp,oxy,salt)
names(jul) <- c('depth','temp','oxy','salt')
jul_df_urchin <- raster::extract(jul, occ_df_urchin)
jul_df_urchin <- cbind(occ_df_urchin, jul_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear')

aug <- stack(depth,temp,oxy,salt)
names(aug) <- c('depth','temp','oxy','salt')
aug_df_urchin <- raster::extract(aug, occ_df_urchin)
aug_df_urchin <- cbind(occ_df_urchin, aug_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear')

sep <- stack(depth,temp,oxy,salt)
names(sep) <- c('depth','temp','oxy','salt')
sep_df_urchin <- raster::extract(sep, occ_df_urchin)
sep_df_urchin <- cbind(occ_df_urchin, sep_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

oct <- stack(depth,temp,oxy,salt)
names(oct) <- c('depth','temp','oxy','salt')
oct_df_urchin <- raster::extract(oct, occ_df_urchin)
oct_df_urchin <- cbind(occ_df_urchin, oct_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear')

nov <- stack(depth,temp,oxy,salt)
names(nov) <- c('depth','temp','oxy','salt')
nov_df_urchin <- raster::extract(nov, occ_df_urchin)
nov_df_urchin <- cbind(occ_df_urchin, nov_df_urchin)

temp <- resample(griddify(dplyr::select(temp_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

dec <- stack(depth,temp,oxy,salt)
names(dec) <- c('depth','temp','oxy','salt')
dec_df_urchin <- raster::extract(dec, occ_df_urchin)
dec_df_urchin <- cbind(occ_df_urchin, dec_df_urchin)

test_urchin <- data.frame(rbind(jan_df_urchin,feb_df_urchin,mar_df_urchin,
                                apr_df_urchin,may_df_urchin,jun_df_urchin,
                                jul_df_urchin,aug_df_urchin,sep_df_urchin,
                                oct_df_urchin,nov_df_urchin,dec_df_urchin))

test_urchin <- drop_na(test_urchin)
test_urchin$depth_pressure <- swPressure((test_urchin$depth*-1), test_urchin$lat, eos="unesco")*100 #calculate pressure in mbar
test_urchin$depth_pressure <- ifelse(test_urchin$depth_pressure < 0, 0, test_urchin$depth_pressure) #make anything above sea level = 0
test_urchin$depth_pressure <- test_urchin$depth_pressure + 1013.25 #add air pressure at surface

test_urchin$kpa <- conv_o2(o2 = test_urchin$oxy, from="umol_per_l", to = "kPa", 
                    temp = test_urchin$temp, 
                    sal = test_urchin$salt ,
                    atm_pres = test_urchin$depth_pressure)

ggplot()+
  geom_point(data = test_urchin, aes(x = temp, y = kpa))

test_urchin <- dplyr::distinct(test_urchin, temp, kpa) #remove duplicated co ordinates

ami_urchin_fig <- ggplot()+
  geom_tile(data = ami_df_urchin, aes(x = temp, y = oxygen, fill = ami))+
  geom_contour(data = ami_df_urchin, aes(x = temp, y = oxygen, z = ami), binwidth = 0.5, colour = "white", na.rm = TRUE)+
  scale_fill_viridis(name = expression(paste(~phi[A])), limits = c(0,max(ami_df_urchin$ami)), na.value="transparent")+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  #geom_point(data = test_urchin, aes(x = temp, y = kpa), alpha = 0.1)+
  geom_point(data = urchin_data, aes(y = pcrit_kpa, x = temp), colour = 'red')+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=1.5, y=22, label="(a)", fontface = "bold")

## plot the normalized ami per oxygen bin

head(ami_df_urchin)

ami_df_urchin$ami <- ifelse(ami_df_urchin$ami < 0, 0, ami_df_urchin$ami)

norm_urchin <- data.frame(temp = ami_df_urchin$temp, oxygen = ami_df_urchin$oxygen, ami = ami_df_urchin$ami)
a_urchin <- spread(norm_urchin, oxygen, ami)
a_urchin[is.na(a_urchin)] <- 0

normalize <- function(x) {
  return ((x - min(x)) / (max(x) - min(x)))
}


b_urchin <- data.frame(kpa_1 = data.frame(normalize(a_urchin[,2])),
                kpa_2 = data.frame(normalize(a_urchin[,3])),
                kpa_3 = data.frame(normalize(a_urchin[,4])),
                kpa_4 = data.frame(normalize(a_urchin[,5])),
                kpa_5 = data.frame(normalize(a_urchin[,6])),
                kpa_6 = data.frame(normalize(a_urchin[,7])),
                kpa_7 = data.frame(normalize(a_urchin[,8])),
                kpa_8 = data.frame(normalize(a_urchin[,9])),
                kpa_9 = data.frame(normalize(a_urchin[,10])),
                kpa_10 = data.frame(normalize(a_urchin[,11])),
                kpa_11 = data.frame(normalize(a_urchin[,12])),
                kpa_12 = data.frame(normalize(a_urchin[,13])),
                kpa_13 = data.frame(normalize(a_urchin[,14])),
                kpa_14 = data.frame(normalize(a_urchin[,15])),
                kpa_15 = data.frame(normalize(a_urchin[,16])),
                kpa_16 = data.frame(normalize(a_urchin[,17])),
                kpa_17 = data.frame(normalize(a_urchin[,18])),
                kpa_18 = data.frame(normalize(a_urchin[,19])),
                kpa_19 = data.frame(normalize(a_urchin[,20])),
                kpa_20 = data.frame(normalize(a_urchin[,21])),
                kpa_21 = data.frame(normalize(a_urchin[,22])),
                kpa_22 = data.frame(normalize(a_urchin[,23])),
                kpa_23 = data.frame(normalize(a_urchin[,24])))

b_urchin[is.na(b_urchin)] <- 0

library(reshape)

c_urchin <- melt(b_urchin)

head(c_urchin)

d_urchin <- data.frame(ami = c_urchin[,2], temp = rep(seq(0,40,1),23), oxygen = rep(1:23,each = 41))

ami_norm_urchin_fig <- ggplot()+
  geom_tile(data = d_urchin, aes(x = temp, y = oxygen, fill = ami))+
  scale_fill_viridis(name = expression(paste(~phi[A]^{"'"})), option = "viridis", limits = c(0.1,max(d_urchin$ami)), na.value="transparent")+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  geom_point(data = test_urchin, aes(x = temp, y = kpa), alpha = 0.1)+
  #geom_point(data = urchin_data, aes(y = pcrit_kpa, x = temp), colour = 'red')+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=1.5, y=22, label="(a)", fontface = "bold")


#############
## abalone ##
#############


## create data frame for plots

oxygen <- rep(1:23,each = 41)
temp <- rep(seq(0,40,1),23)
ami_df_abalone <- data.frame(oxygen, temp)

mass_abalone <- 37.5 # mid point of urchins tested

ami_df_abalone$ami <- (ami_df_abalone$oxygen*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(ami_df_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                                                ((m*(exp(ami_df_abalone$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(ami_df_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))

ami_df_abalone$ami <- ifelse(ami_df_abalone$ami <= 0, NA, ami_df_abalone$ami)

##------------------------##
##  occurrence data data  ##
##------------------------##

# OBIS
occ_obis_abalone <- occurrence("Haliotis rufescens") # pull occurence data from OBIS
occ_df_obis_abalone <- data.frame(lon = occ_obis_abalone$decimalLongitude, lat = occ_obis_abalone$decimalLatitude) # create a data frame
occ_df_obis_abalone <- dplyr::distinct(occ_df_obis_abalone, lon, lat) #remove duplicated co ordinates

occ_df_abalone <- occ_df_obis_abalone

jan_df_abalone <- raster::extract(jan, occ_df_abalone)
jan_df_abalone <- cbind(occ_df_abalone, jan_df_abalone)

feb_df_abalone <- raster::extract(feb, occ_df_abalone)
feb_df_abalone <- cbind(occ_df_abalone, feb_df_abalone)

mar_df_abalone <- raster::extract(mar, occ_df_abalone)
mar_df_abalone <- cbind(occ_df_abalone, mar_df_abalone)

apr_df_abalone <- raster::extract(apr, occ_df_abalone)
apr_df_abalone <- cbind(occ_df_abalone, apr_df_abalone)

may_df_abalone <- raster::extract(may, occ_df_abalone)
may_df_abalone <- cbind(occ_df_abalone, may_df_abalone)

jun_df_abalone <- raster::extract(jun, occ_df_abalone)
jun_df_abalone <- cbind(occ_df_abalone, jun_df_abalone)

jul_df_abalone <- raster::extract(jul, occ_df_abalone)
jul_df_abalone <- cbind(occ_df_abalone, jul_df_abalone)

aug_df_abalone <- raster::extract(aug, occ_df_abalone)
aug_df_abalone <- cbind(occ_df_abalone, aug_df_abalone)

sep_df_abalone <- raster::extract(sep, occ_df_abalone)
sep_df_abalone <- cbind(occ_df_abalone, sep_df_abalone)

oct_df_abalone <- raster::extract(oct, occ_df_abalone)
oct_df_abalone <- cbind(occ_df_abalone, oct_df_abalone)

nov_df_abalone <- raster::extract(nov, occ_df_abalone)
nov_df_abalone <- cbind(occ_df_abalone, nov_df_abalone)

dec_df_abalone <- raster::extract(dec, occ_df_abalone)
dec_df_abalone <- cbind(occ_df_abalone, dec_df_abalone)

test_abalone <- data.frame(rbind(jan_df_abalone,feb_df_abalone,mar_df_abalone,
                                 apr_df_abalone,may_df_abalone,jun_df_abalone,
                                 jul_df_abalone,aug_df_abalone,sep_df_abalone,
                                 oct_df_abalone,nov_df_abalone,dec_df_abalone))

test_abalone <- drop_na(test_abalone)
test_abalone$depth_pressure <- swPressure((test_abalone$depth*-1), test_abalone$lat, eos="unesco")*100 #calculate pressure in mbar
test_abalone$depth_pressure <- ifelse(test_abalone$depth_pressure < 0, 0, test_abalone$depth_pressure) #make anything above sea level = 0
test_abalone$depth_pressure <- test_abalone$depth_pressure + 1013.25 #add air pressure at surface

test_abalone$kpa <- conv_o2(o2 = test_abalone$oxy, from="umol_per_l", to = "kPa", 
                            temp = test_abalone$temp, 
                            sal = test_abalone$salt ,
                            atm_pres = test_abalone$depth_pressure)

ggplot()+
  geom_point(data = test_abalone, aes(x = temp, y = kpa))

test_abalone <- dplyr::distinct(test_abalone, temp, kpa) #remove duplicated co ordinates

ami_abalone_fig <- ggplot()+
  geom_tile(data = ami_df_abalone, aes(x = temp, y = oxygen, fill = ami))+
  geom_contour(data = ami_df_abalone, aes(x = temp, y = oxygen, z = ami), binwidth = 0.25, colour = "white", na.rm = TRUE)+
  scale_fill_viridis(name = expression(paste(~phi[A])), option = "D", limits = c(0,max(ami_df_abalone$ami)), na.value="transparent")+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  #geom_point(data = test_abalone, aes(x = temp, y = kpa), alpha = 0.1)+
  geom_point(data = abalone_data, aes(y = pcrit_kpa, x = temp), colour = 'red')+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=1.5, y=22, label="(b)", fontface = "bold")

fig_4 <- grid.arrange(ami_urchin_fig, ami_abalone_fig, ncol = 2)
ggsave("fig_4.pdf", plot = fig_4, width = 8, height = 3)


## plot the normalized ami per oxygen bin

head(ami_df_abalone)

ami_df_abalone$ami <- ifelse(ami_df_abalone$ami < 0, 0, ami_df_abalone$ami)

norm_abalone <- data.frame(temp = ami_df_abalone$temp, oxygen = ami_df_abalone$oxygen, ami = ami_df_abalone$ami)
a_abalone <- spread(norm_abalone, oxygen, ami)
a_abalone[is.na(a_abalone)] <- 0

b_abalone <- data.frame(kpa_1 = data.frame(normalize(a_abalone[,2])),
                        kpa_2 = data.frame(normalize(a_abalone[,3])),
                        kpa_3 = data.frame(normalize(a_abalone[,4])),
                        kpa_4 = data.frame(normalize(a_abalone[,5])),
                        kpa_5 = data.frame(normalize(a_abalone[,6])),
                        kpa_6 = data.frame(normalize(a_abalone[,7])),
                        kpa_7 = data.frame(normalize(a_abalone[,8])),
                        kpa_8 = data.frame(normalize(a_abalone[,9])),
                        kpa_9 = data.frame(normalize(a_abalone[,10])),
                        kpa_10 = data.frame(normalize(a_abalone[,11])),
                        kpa_11 = data.frame(normalize(a_abalone[,12])),
                        kpa_12 = data.frame(normalize(a_abalone[,13])),
                        kpa_13 = data.frame(normalize(a_abalone[,14])),
                        kpa_14 = data.frame(normalize(a_abalone[,15])),
                        kpa_15 = data.frame(normalize(a_abalone[,16])),
                        kpa_16 = data.frame(normalize(a_abalone[,17])),
                        kpa_17 = data.frame(normalize(a_abalone[,18])),
                        kpa_18 = data.frame(normalize(a_abalone[,19])),
                        kpa_19 = data.frame(normalize(a_abalone[,20])),
                        kpa_20 = data.frame(normalize(a_abalone[,21])),
                        kpa_21 = data.frame(normalize(a_abalone[,22])),
                        kpa_22 = data.frame(normalize(a_abalone[,23])),
                        kpa_23 = data.frame(normalize(a_abalone[,24])))

b_abalone[is.na(b_abalone)] <- 0

c_abalone <- melt(b_abalone)

head(c_abalone)

d_abalone <- data.frame(ami = c_abalone[,2], temp = rep(seq(0,40,1),23), oxygen = rep(1:23,each = 41))

ami_norm_abalone_fig <- ggplot()+
  geom_tile(data = d_abalone, aes(x = temp, y = oxygen, fill = ami))+
  scale_fill_viridis(name = expression(paste(~phi[A]^{"'"})), option = "viridis", limits = c(0.1,max(d_abalone$ami)), na.value="transparent")+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  geom_point(data = test_abalone, aes(x = temp, y = kpa), alpha = 0.1)+
  #geom_point(data = abalone_data, aes(y = pcrit_kpa, x = temp), colour = 'red')+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(text = element_text(size=15))+
  annotate(geom="text", x=1.5, y=22, label="(b)", fontface = "bold")

fig_6 <- grid.arrange(ami_norm_urchin_fig, ami_norm_abalone_fig, ncol = 2)
ggsave("fig_6.pdf", plot = fig_6, width = 8, height = 3)


##-----------------------------------------------------
## effect of oxygen availability on optimum temperature
##-----------------------------------------------------

##--------##
## urchin ##
##--------##

oxy_100_urchin <- data.frame(temp = seq(0,35,1),
                      kpa = rep(21,36))

oxy_25_urchin <- data.frame(temp = seq(0,35,1),
                     kpa = rep(10.5,36))

oxy_100_urchin$ami <- (oxy_100_urchin$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oxy_100_urchin$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                               (((oxy_100_urchin$temp^2*x2) + (oxy_100_urchin$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oxy_100_urchin$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))

oxy_25_urchin$ami <- (oxy_25_urchin$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oxy_25_urchin$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                             (((oxy_25_urchin$temp^2*x2) + (oxy_25_urchin$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oxy_25_urchin$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))

oxy_100_urchin$ami <- ifelse(oxy_100_urchin$ami < 0, 0, oxy_100_urchin$ami)
oxy_25_urchin$ami <- ifelse(oxy_25_urchin$ami < 0, 0, oxy_25_urchin$ami)

oxy_100_urchin$ami_norm <- normalize(oxy_100_urchin[,3])
oxy_25_urchin$ami_norm <- normalize(oxy_25_urchin[,3])

urchin_topt_oxy <- ggplot()+
  geom_line(data = subset(oxy_100_urchin, ami_norm > 0), aes(x = temp, y = ami_norm, colour = 'a'), size = 2)+
  geom_line(data = subset(oxy_25_urchin, ami_norm > 0), aes(x = temp, y = ami_norm, colour = 'b'), size = 2)+
  scale_colour_manual(values = c('purple','plum'),
                      labels = c('21 kPa','10.5 kPa'),
                      name = 'Oxygen')+
  geom_segment(aes(x = 11.5, y = 0, xend = 11.5, yend = 1), colour = "purple", linetype = 'dashed', size = 1)+
  geom_segment(aes(x = 10, y = 0, xend = 10, yend = 1), colour = "plum", linetype = 'dashed', size = 1)+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0.01,0.01)) +
  scale_y_continuous(breaks = seq(0,1.1,0.2), limits = c(0,1.1), expand = c(0.01,0.01)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("normalized AMI (", ~phi[A]^{"'"}, ")")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(legend.position = c (0.8,0.8),
        legend.background = element_rect(colour = NA, fill = "transparent"))+
  annotate(geom="text", x=1.5, y=1.05, label="(a)", fontface = "bold")+
  theme(text = element_text(size=11))+
  annotate('text', x = 15, y = 0.4, label = '11.5 °C', size = 4)+
  annotate('text', x = 7, y = 0.4, label = '10 °C', size = 4)

##----_----##
## abalone ##
##------_--##

oxy_100_abalone <- data.frame(temp = seq(0,35,1),
                              kpa = rep(21,36))

oxy_50_abalone <- data.frame(temp = seq(0,35,1),
                             kpa = rep(10.5,36))

oxy_100_abalone$ami <- (oxy_100_abalone$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oxy_100_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                                               ((m*(exp(oxy_100_abalone$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oxy_100_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))

oxy_50_abalone$ami <- (oxy_50_abalone$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oxy_50_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                                             ((m*(exp(oxy_50_abalone$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oxy_50_abalone$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))

oxy_100_abalone$ami <- ifelse(oxy_100_abalone$ami < 0, 0, oxy_100_abalone$ami)
oxy_50_abalone$ami <- ifelse(oxy_50_abalone$ami < 0, 0, oxy_50_abalone$ami)


oxy_100_abalone$ami_norm <- normalize(oxy_100_abalone[,3])
oxy_50_abalone$ami_norm <- normalize(oxy_50_abalone[,3])

abalone_topt_oxy <- ggplot()+
  geom_line(data = subset(oxy_50_abalone, ami_norm > 0), aes(x = temp, y = ami_norm, colour = "b"), size = 2)+
  geom_line(data = subset(oxy_100_abalone, ami_norm > 0), aes(x = temp, y = ami_norm, colour = "a"), size = 2)+
  scale_colour_manual(values = c('firebrick','tomato'),
                      labels = c('21 kPa', '10.5 kPa'),
                      name = 'Oxygen')+
  geom_segment(aes(x = 18, y = 0, xend = 18, yend = 1), colour = "firebrick", linetype = 'dashed', size = 1)+
  geom_segment(aes(x = 9, y = 0, xend = 9, yend = 1), colour = "tomato", linetype = 'dashed', size = 1)+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0.01,0.01)) +
  scale_y_continuous(breaks = seq(0,1.1,0.2), limits = c(0,1.1), expand = c(0.01,0.01)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("normalized AMI (", ~phi[A]^{"'"}, ")")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(legend.position = c (0.875,0.85),
        legend.background = element_rect(colour = NA, fill = "transparent"))+
  annotate(geom="text", x=1.5, y=1.05, label="(b)", fontface = "bold")+
  theme(text = element_text(size=11))+
  annotate('text', x = 22, y = 0.7, label = '18 °C', size = 4)+
  annotate('text', x = 12, y = 0.7, label = '9 °C', size = 4)


##----------------------------------------
## effect of mass on optimum temperature
##----------------------------------------

##--------##
## urchin ##
##--------##

mass_150_urchin <- data.frame(temp = seq(0,35,1),
                       kpa = rep(21,36),
                       mass = rep(250, 36))

mass_15_urchin <- data.frame(temp = seq(0,35,1),
                      kpa = rep(21,36),
                      mass = rep(15, 36))

mass_150_urchin$ami <- (mass_150_urchin$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mass_150_urchin$temp+kelvin)))))*(mass_150_urchin$mass^mass_scaler_urchin))/
                                 (((mass_150_urchin$temp^2*x2) + (mass_150_urchin$temp*x) + p)*(mass_150_urchin$mass^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mass_150_urchin$temp+kelvin))))*(mass_150_urchin$mass^mass_scaler_urchin))

mass_15_urchin$ami <- (mass_15_urchin$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mass_15_urchin$temp+kelvin)))))*(mass_15_urchin$mass^mass_scaler_urchin))/
                               (((mass_15_urchin$temp^2*x2) + (mass_15_urchin$temp*x) + p)*(mass_15_urchin$mass^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mass_15_urchin$temp+kelvin))))*(mass_15_urchin$mass^mass_scaler_urchin))

mass_150_urchin$ami <- ifelse(mass_150_urchin$ami < 0, 0, mass_150_urchin$ami)
mass_15_urchin$ami <- ifelse(mass_15_urchin$ami < 0, 0, mass_15_urchin$ami)

mass_150_urchin$ami_norm <- normalize(mass_150_urchin[,4])
mass_15_urchin$ami_norm <- normalize(mass_15_urchin[,4])

urchin_topt_mass <- ggplot()+
  geom_line(data = subset(mass_150_urchin, ami > 0), aes(x = temp, y = ami_norm, colour = "a"), size = 2)+
  geom_line(data = subset(mass_15_urchin, ami > 0), aes(x = temp, y = ami_norm, colour = "b"), size = 2)+
  scale_colour_manual(values = c('purple','plum'),
                      labels = c('250 g','15 g'),
                      name = 'Mass')+
  geom_segment(aes(x = 11, y = 0, xend = 11, yend = 1), colour = "purple", linetype = 'dashed', size = 1)+
  geom_segment(aes(x = 12, y = 0, xend = 12, yend = 1), colour = "plum", linetype = 'dashed', size = 1)+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0.01,0.01)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("normalized AMI (", ~phi[A]^{"'"}, ")")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(legend.position = c (0.8,0.8),
                legend.background = element_rect(colour = NA, fill = "transparent"))+
  annotate(geom="text", x=1.5, y=1.05, label="(c)", fontface = "bold")+
  annotate('text', x = 15, y = 0.4, label = '12 °C', size = 4)+
  annotate('text', x = 8, y = 0.4, label = '11 °C', size = 4)+
  theme(text = element_text(size=11))


##---------##
## abalone ##
##---------##



mass_150_abalone <- data.frame(temp = seq(0,35,1),
                               kpa = rep(21,36),
                               mass = rep(1000, 36))

mass_15_abalone <- data.frame(temp = seq(0,35,1),
                              kpa = rep(21,36),
                              mass = rep(25, 36))

mass_150_abalone$ami <- (mass_150_abalone$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mass_150_abalone$temp+kelvin))))*(mass_150_abalone$mass^mass_scaler_abalone)))/
                                                 ((m*(exp(mass_150_abalone$temp*n)))*(mass_150_abalone$mass^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mass_150_abalone$temp+kelvin))))*(mass_150_abalone$mass^mass_scaler_abalone))

mass_15_abalone$ami <- (mass_15_abalone$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mass_15_abalone$temp+kelvin))))*(mass_15_abalone$mass^mass_scaler_abalone)))/
                                               ((m*(exp(mass_15_abalone$temp*n)))*(mass_15_abalone$mass^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mass_15_abalone$temp+kelvin))))*(mass_15_abalone$mass^mass_scaler_abalone))

mass_150_abalone$ami <- ifelse(mass_150_abalone$ami < 0, 0, mass_150_abalone$ami)
mass_15_abalone$ami <- ifelse(mass_15_abalone$ami < 0, 0, mass_15_abalone$ami)

mass_150_abalone$ami_norm <- normalize(mass_150_abalone[,4])
mass_15_abalone$ami_norm <- normalize(mass_15_abalone[,4])

abalone_topt_mass <- ggplot()+
  geom_line(data = subset(mass_150_abalone, ami > 0), aes(x = temp, y = ami_norm, colour = "a"), size = 2)+
  geom_line(data = subset(mass_15_abalone, ami > 0), aes(x = temp, y = ami_norm, colour = "b"), size = 2)+
  scale_colour_manual(values = c('firebrick','tomato'),
                      labels = c('1000 g','25 g'),
                      name = 'Mass')+
  geom_segment(aes(x = 15, y = 0, xend = 15, yend = 1), colour = "firebrick", linetype = 'dashed', size = 1)+
  geom_segment(aes(x = 18, y = 0, xend = 18, yend = 1), colour = "tomato", linetype = 'dashed', size = 1)+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0.01,0.01)) +
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("normalized AMI (", ~phi[A]^{"'"}, ")")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.line=element_line(colour="black", size=0.3))+
  theme(legend.position = c (0.875,0.85),
        legend.background = element_rect(colour = NA, fill = "transparent"))+
  theme(text = element_text(size=11))+
  annotate(geom="text", x=1.5, y=1.05, label="(d)", fontface = "bold")+
  annotate('text', x = 21, y = 0.4, label = '18 °C', size = 4)+
  annotate('text', x = 10, y = 0.4, label = '15 °C', size = 4)


fig_5 <- grid.arrange(urchin_topt_oxy,abalone_topt_oxy,urchin_topt_mass, abalone_topt_mass, ncol = 2)
ggsave("fig_5.pdf", plot = fig_5, width = 8, height = 6)


##--------------------##
##                    ##
##  spatial modeling  ##
##                    ##
##--------------------##

##---------##
## urchin ##
##---------##

## jan
jan <- as.data.frame(jan, xy = TRUE)
jan <- drop_na(jan)
jan$depth_pressure <- swPressure((jan$depth*-1), jan$y, eos="unesco")*100 #calculate pressure in mbar
jan$depth_pressure <- ifelse(jan$depth_pressure < 0, 0, jan$depth_pressure) #make anything above sea level = 0
jan$depth_pressure <- jan$depth_pressure + 1013.25 #add air pressure at surface
jan$kpa <- conv_o2(o2 = jan$oxy, from="umol_per_l", to = "kPa", 
                   temp = jan$temp, 
                   sal = jan$salt ,
                   atm_pres = jan$depth_pressure)
jan <- subset(jan, depth > -150 & depth < 50)

## feb
feb <- as.data.frame(feb, xy = TRUE)
feb <- drop_na(feb)
feb$depth_pressure <- swPressure((feb$depth*-1), feb$y, eos="unesco")*100 #calculate pressure in mbar
feb$depth_pressure <- ifelse(feb$depth_pressure < 0, 0, feb$depth_pressure) #make anything above sea level = 0
feb$depth_pressure <- feb$depth_pressure + 1013.25 #add air pressure at surface
feb$kpa <- conv_o2(o2 = feb$oxy, from="umol_per_l", to = "kPa", 
                   temp = feb$temp, 
                   sal = feb$salt ,
                   atm_pres = feb$depth_pressure)
feb <- subset(feb, depth > -150 & depth < 50)

## mar
mar <- as.data.frame(mar, xy = TRUE)
mar <- drop_na(mar)
mar$depth_pressure <- swPressure((mar$depth*-1), mar$y, eos="unesco")*100 #calculate pressure in mbar
mar$depth_pressure <- ifelse(mar$depth_pressure < 0, 0, mar$depth_pressure) #make anything above sea level = 0
mar$depth_pressure <- mar$depth_pressure + 1013.25 #add air pressure at surface
mar$kpa <- conv_o2(o2 = mar$oxy, from="umol_per_l", to = "kPa", 
                   temp = mar$temp, 
                   sal = mar$salt ,
                   atm_pres = mar$depth_pressure)
mar <- subset(mar, depth > -150 & depth < 50)

## apr
apr <- as.data.frame(apr, xy = TRUE)
apr <- drop_na(apr)
apr$depth_pressure <- swPressure((apr$depth*-1), apr$y, eos="unesco")*100 #calculate pressure in mbar
apr$depth_pressure <- ifelse(apr$depth_pressure < 0, 0, apr$depth_pressure) #make anything above sea level = 0
apr$depth_pressure <- apr$depth_pressure + 1013.25 #add air pressure at surface
apr$kpa <- conv_o2(o2 = apr$oxy, from="umol_per_l", to = "kPa", 
                   temp = apr$temp, 
                   sal = apr$salt ,
                   atm_pres = apr$depth_pressure)
apr <- subset(apr, depth > -150 & depth < 50)

## may
may <- as.data.frame(may, xy = TRUE)
may <- drop_na(may)
may$depth_pressure <- swPressure((may$depth*-1), may$y, eos="unesco")*100 #calculate pressure in mbar
may$depth_pressure <- ifelse(may$depth_pressure < 0, 0, may$depth_pressure) #make anything above sea level = 0
may$depth_pressure <- may$depth_pressure + 1013.25 #add air pressure at surface
may$kpa <- conv_o2(o2 = may$oxy, from="umol_per_l", to = "kPa", 
                   temp = may$temp, 
                   sal = may$salt ,
                   atm_pres = may$depth_pressure)
may <- subset(may, depth > -150 & depth < 50)

## jun
jun <- as.data.frame(jun, xy = TRUE)
jun <- drop_na(jun)
jun$depth_pressure <- swPressure((jun$depth*-1), jun$y, eos="unesco")*100 #calculate pressure in mbar
jun$depth_pressure <- ifelse(jun$depth_pressure < 0, 0, jun$depth_pressure) #make anything above sea level = 0
jun$depth_pressure <- jun$depth_pressure + 1013.25 #add air pressure at surface
jun$kpa <- conv_o2(o2 = jun$oxy, from="umol_per_l", to = "kPa", 
                   temp = jun$temp, 
                   sal = jun$salt ,
                   atm_pres = jun$depth_pressure)
jun <- subset(jun, depth > -150 & depth < 50)

## jul
jul <- as.data.frame(jul, xy = TRUE)
jul <- drop_na(jul)
jul$depth_pressure <- swPressure((jul$depth*-1), jul$y, eos="unesco")*100 #calculate pressure in mbar
jul$depth_pressure <- ifelse(jul$depth_pressure < 0, 0, jul$depth_pressure) #make anything above sea level = 0
jul$depth_pressure <- jul$depth_pressure + 1013.25 #add air pressure at surface
jul$kpa <- conv_o2(o2 = jul$oxy, from="umol_per_l", to = "kPa", 
                   temp = jul$temp, 
                   sal = jul$salt ,
                   atm_pres = jul$depth_pressure)
jul <- subset(jul, depth > -150 & depth < 50)

## aug
aug <- as.data.frame(aug, xy = TRUE)
aug <- drop_na(aug)
aug$depth_pressure <- swPressure((aug$depth*-1), aug$y, eos="unesco")*100 #calculate pressure in mbar
aug$depth_pressure <- ifelse(aug$depth_pressure < 0, 0, aug$depth_pressure) #make anything above sea level = 0
aug$depth_pressure <- aug$depth_pressure + 1013.25 #add air pressure at surface
aug$kpa <- conv_o2(o2 = aug$oxy, from="umol_per_l", to = "kPa", 
                   temp = aug$temp, 
                   sal = aug$salt ,
                   atm_pres = aug$depth_pressure)
aug <- subset(aug, depth > -150 & depth < 50)

## sep
sep <- as.data.frame(sep, xy = TRUE)
sep <- drop_na(sep)
sep$depth_pressure <- swPressure((sep$depth*-1), sep$y, eos="unesco")*100 #calculate pressure in mbar
sep$depth_pressure <- ifelse(sep$depth_pressure < 0, 0, sep$depth_pressure) #make anything above sea level = 0
sep$depth_pressure <- sep$depth_pressure + 1013.25 #add air pressure at surface
sep$kpa <- conv_o2(o2 = sep$oxy, from="umol_per_l", to = "kPa", 
                   temp = sep$temp, 
                   sal = sep$salt ,
                   atm_pres = sep$depth_pressure)
sep <- subset(sep, depth > -150 & depth < 50)

## oct
oct <- as.data.frame(oct, xy = TRUE)
oct <- drop_na(oct)
oct$depth_pressure <- swPressure((oct$depth*-1), oct$y, eos="unesco")*100 #calculate pressure in mbar
oct$depth_pressure <- ifelse(oct$depth_pressure < 0, 0, oct$depth_pressure) #make anything above sea level = 0
oct$depth_pressure <- oct$depth_pressure + 1013.25 #add air pressure at surface
oct$kpa <- conv_o2(o2 = oct$oxy, from="umol_per_l", to = "kPa", 
                   temp = oct$temp, 
                   sal = oct$salt ,
                   atm_pres = oct$depth_pressure)
oct <- subset(oct, depth > -150 & depth < 50)

## nov
nov <- as.data.frame(nov, xy = TRUE)
nov <- drop_na(nov)
nov$depth_pressure <- swPressure((nov$depth*-1), nov$y, eos="unesco")*100 #calculate pressure in mbar
nov$depth_pressure <- ifelse(nov$depth_pressure < 0, 0, nov$depth_pressure) #make anything above sea level = 0
nov$depth_pressure <- nov$depth_pressure + 1013.25 #add air pressure at surface
nov$kpa <- conv_o2(o2 = nov$oxy, from="umol_per_l", to = "kPa", 
                   temp = nov$temp, 
                   sal = nov$salt ,
                   atm_pres = nov$depth_pressure)
nov <- subset(nov, depth > -150 & depth < 50)

## dec
dec <- as.data.frame(dec, xy = TRUE)
dec <- drop_na(dec)
dec$depth_pressure <- swPressure((dec$depth*-1), dec$y, eos="unesco")*100 #calculate pressure in mbar
dec$depth_pressure <- ifelse(dec$depth_pressure < 0, 0, dec$depth_pressure) #make anything above sea level = 0
dec$depth_pressure <- dec$depth_pressure + 1013.25 #add air pressure at surface
dec$kpa <- conv_o2(o2 = dec$oxy, from="umol_per_l", to = "kPa", 
                   temp = dec$temp, 
                   sal = dec$salt ,
                   atm_pres = dec$depth_pressure)
dec <- subset(dec, depth > -150 & depth < 50)


jan$ami_urchin <- (jan$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jan$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jan$temp^2*x2) + (jan$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jan$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jan$ami_urchin <- ifelse(jan$ami_urchin <= 0, NA, jan$ami_urchin)

feb$ami_urchin <- (feb$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(feb$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((feb$temp^2*x2) + (feb$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(feb$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
feb$ami_urchin <- ifelse(feb$ami_urchin <= 0, NA, feb$ami_urchin)

mar$ami_urchin <- (mar$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mar$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((mar$temp^2*x2) + (mar$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mar$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
mar$ami_urchin <- ifelse(mar$ami_urchin <= 0, NA, mar$ami_urchin)

apr$ami_urchin <- (apr$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(apr$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((apr$temp^2*x2) + (apr$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(apr$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
apr$ami_urchin <- ifelse(apr$ami_urchin <= 0, NA, apr$ami_urchin)

may$ami_urchin <- (may$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(may$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((may$temp^2*x2) + (may$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(may$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
may$ami_urchin <- ifelse(may$ami_urchin <= 0, NA, may$ami_urchin)

jun$ami_urchin <- (jun$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jun$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jun$temp^2*x2) + (jun$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jun$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jun$ami_urchin <- ifelse(jun$ami_urchin <= 0, NA, jun$ami_urchin)

jul$ami_urchin <- (jul$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jul$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jul$temp^2*x2) + (jul$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jul$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jul$ami_urchin <- ifelse(jul$ami_urchin <= 0, NA, jul$ami_urchin)

aug$ami_urchin <- (aug$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(aug$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((aug$temp^2*x2) + (aug$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(aug$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
aug$ami_urchin <- ifelse(aug$ami_urchin <= 0, NA, aug$ami_urchin)

sep$ami_urchin <- (sep$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(sep$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((sep$temp^2*x2) + (sep$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(sep$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
sep$ami_urchin <- ifelse(sep$ami_urchin <= 0, NA, sep$ami_urchin)

oct$ami_urchin <- (oct$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oct$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((oct$temp^2*x2) + (oct$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oct$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
oct$ami_urchin <- ifelse(oct$ami_urchin <= 0, NA, oct$ami_urchin)

nov$ami_urchin <- (nov$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(nov$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((nov$temp^2*x2) + (nov$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(nov$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
nov$ami_urchin <- ifelse(nov$ami_urchin <= 0, NA, nov$ami_urchin)

dec$ami_urchin <- (dec$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(dec$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((dec$temp^2*x2) + (dec$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(dec$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
dec$ami_urchin <- ifelse(dec$ami_urchin <= 0, NA, dec$ami_urchin)

## AMI

jan_ami_urchin <- resample(griddify(dplyr::select((jan),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
feb_ami_urchin <- resample(griddify(dplyr::select((feb),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
mar_ami_urchin <- resample(griddify(dplyr::select((mar),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
apr_ami_urchin <- resample(griddify(dplyr::select((apr),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
may_ami_urchin <- resample(griddify(dplyr::select((may),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
jun_ami_urchin <- resample(griddify(dplyr::select((jun),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
jul_ami_urchin <- resample(griddify(dplyr::select((jul),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
aug_ami_urchin <- resample(griddify(dplyr::select((aug),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
sep_ami_urchin <- resample(griddify(dplyr::select((sep),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
oct_ami_urchin <- resample(griddify(dplyr::select((oct),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
nov_ami_urchin <- resample(griddify(dplyr::select((nov),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
dec_ami_urchin <- resample(griddify(dplyr::select((dec),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')

mean_ami_raster_urchin <- mean(jan_ami_urchin, feb_ami_urchin, mar_ami_urchin, apr_ami_urchin, may_ami_urchin, jun_ami_urchin, jul_ami_urchin, aug_ami_urchin, sep_ami_urchin, oct_ami_urchin, nov_ami_urchin, dec_ami_urchin)
min_ami_raster_urchin <- min(jan_ami_urchin, feb_ami_urchin, mar_ami_urchin, apr_ami_urchin, may_ami_urchin, jun_ami_urchin, jul_ami_urchin, aug_ami_urchin, sep_ami_urchin, oct_ami_urchin, nov_ami_urchin, dec_ami_urchin)
max_ami_raster_urchin <- max(jan_ami_urchin, feb_ami_urchin, mar_ami_urchin, apr_ami_urchin, may_ami_urchin, jun_ami_urchin, jul_ami_urchin, aug_ami_urchin, sep_ami_urchin, oct_ami_urchin, nov_ami_urchin, dec_ami_urchin)

library(rnaturalearth)
library(sf)
library(ggspatial)

countries <- ne_countries(country = c('mexico','united states of america','canada'), scale = 'medium', returnclass = 'sf')

##----- relative AMI -----##

d <- d_urchin
colnames(d)[1] <- "ami_norm_urchin"
head(d)
d$oxygen <- as.numeric(d$oxygen)
d$code <- paste(d$oxygen,d$temp)
d <- subset(d, selec = -c(temp, oxygen))

#jan
jan$oxygen_round <- round(jan$kpa)
jan$temp_round <- round(jan$temp)
jan$code <- paste(jan$oxygen_round,jan$temp_round)
jan <- left_join(jan, d, by = "code")
jan_rel_ami_urchin <- resample(griddify(dplyr::select((jan),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#feb
feb$oxygen_round <- round(feb$kpa)
feb$temp_round <- round(feb$temp)
feb$code <- paste(feb$oxygen_round,feb$temp_round)
feb <- left_join(feb, d, by = "code")
feb_rel_ami_urchin <- resample(griddify(dplyr::select((feb),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#mar
mar$oxygen_round <- round(mar$kpa)
mar$temp_round <- round(mar$temp)
mar$code <- paste(mar$oxygen_round,mar$temp_round)
mar <- left_join(mar, d, by = "code")
mar_rel_ami_urchin <- resample(griddify(dplyr::select((mar),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#apr
apr$oxygen_round <- round(apr$kpa)
apr$temp_round <- round(apr$temp)
apr$code <- paste(apr$oxygen_round,apr$temp_round)
apr <- left_join(apr, d, by = "code")
apr_rel_ami_urchin <- resample(griddify(dplyr::select((apr),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#may
may$oxygen_round <- round(may$kpa)
may$temp_round <- round(may$temp)
may$code <- paste(may$oxygen_round,may$temp_round)
may <- left_join(may, d, by = "code")
may_rel_ami_urchin <- resample(griddify(dplyr::select((may),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#jun
jun$oxygen_round <- round(jun$kpa)
jun$temp_round <- round(jun$temp)
jun$code <- paste(jun$oxygen_round,jun$temp_round)
jun <- left_join(jun, d, by = "code")
jun_rel_ami_urchin <- resample(griddify(dplyr::select((jun),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#jul
jul$oxygen_round <- round(jul$kpa)
jul$temp_round <- round(jul$temp)
jul$code <- paste(jul$oxygen_round,jul$temp_round)
jul <- left_join(jul, d, by = "code")
jul_rel_ami_urchin <- resample(griddify(dplyr::select((jul),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#aug
aug$oxygen_round <- round(aug$kpa)
aug$temp_round <- round(aug$temp)
aug$code <- paste(aug$oxygen_round,aug$temp_round)
aug <- left_join(aug, d, by = "code")
aug_rel_ami_urchin <- resample(griddify(dplyr::select((aug),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#sep
sep$oxygen_round <- round(sep$kpa)
sep$temp_round <- round(sep$temp)
sep$code <- paste(sep$oxygen_round,sep$temp_round)
sep <- left_join(sep, d, by = "code")
sep_rel_ami_urchin <- resample(griddify(dplyr::select((sep),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#oct
oct$oxygen_round <- round(oct$kpa)
oct$temp_round <- round(oct$temp)
oct$code <- paste(oct$oxygen_round,oct$temp_round)
oct <- left_join(oct, d, by = "code")
oct_rel_ami_urchin <- resample(griddify(dplyr::select((oct),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#nov
nov$oxygen_round <- round(nov$kpa)
nov$temp_round <- round(nov$temp)
nov$code <- paste(nov$oxygen_round,nov$temp_round)
nov <- left_join(nov, d, by = "code")
nov_rel_ami_urchin <- resample(griddify(dplyr::select((nov),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#dec
dec$oxygen_round <- round(dec$kpa)
dec$temp_round <- round(dec$temp)
dec$code <- paste(dec$oxygen_round,dec$temp_round)
dec <- left_join(dec, d, by = "code")
dec_rel_ami_urchin <- resample(griddify(dplyr::select((dec),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

mean_rel_ami_raster_urchin <- mean(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)
min_rel_ami_raster_urchin <- min(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)
max_rel_ami_raster_urchin <- max(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)

head(d_urchin)

##--------------------------
##  calculating AMI crit bin
##--------------------------

head(d_urchin)
head(test_urchin)

# round urchin occurence data
test_urchin_round <- test_urchin
test_urchin_round[,-3] <- round(test_urchin_round[,-3],0)
test_urchin_round <- test_urchin_round[,-3]
names(test_urchin_round)[names(test_urchin_round) == 'kpa'] <- 'oxygen'

ami_crit_bin_df_urchin <- left_join(test_urchin_round, d_urchin, by = c('temp','oxygen'))
hist(ami_crit_bin_df_urchin$ami)

thres_urchin <- as.numeric(quantile(ami_crit_bin_df_urchin$ami, probs = 0.01))

ami_crit_urchin <- ggplot(data = ami_crit_bin_df_urchin, aes(x = ami)) +
  geom_histogram(bins = 100, alpha = 0.7, colour = 'black', fill = 'grey') +
  theme_bw() +
  geom_vline(xintercept = thres_urchin, linetype = 'dashed') +
  scale_x_continuous(breaks = seq(0,1,0.1), limits = c(0.5,1))+
  scale_y_continuous(limits = c(0, 650))+
  labs(x=expression(paste("normalized AMI"))) +
  labs(y=expression(paste("Count"))) +
  ggtitle("Purple Urchin") +
  annotate(geom="text", x=0.5, y=650, label="(a)", fontface = "bold")

d_urchin$bin_ami <- ifelse(d_urchin$ami > thres_urchin, 1, 2)

inset_urchin <- ggplot()+
  geom_tile(data = d_urchin, aes(x = temp, y = oxygen, fill = as.factor(bin_ami)))+
  scale_fill_manual(values = c('purple','transparent'),
                    labels = c('suitable','unsuitable'),
                    name = 'normalized AMI')+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  geom_point(data = test_urchin, aes(x = temp, y = kpa), alpha = 0.1, size = 0.1)+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.text = element_text(colour = "black", size = 6),
        axis.line=element_line(colour="black", size=0.3),
        axis.title = element_text(colour = "black", size = 6))+
  theme(legend.position="none")+
  theme(plot.background = element_rect(colour = "black", size=0.5))

min_rel_ami_raster_df_urchin <- na.omit(as.data.frame(min_rel_ami_raster_urchin, xy = TRUE))

min_rel_ami_raster_df_urchin$suitable <- ifelse(min_rel_ami_raster_df_urchin$layer >= thres_urchin, 1, 0)

min_rel_ami_raster_df_urchin <- subset(min_rel_ami_raster_df_urchin, suitable > 0)

main_urchin_current <- ggplot()+
  geom_tile(data = min_rel_ami_raster_df_urchin, aes(x = x, y = y, fill = as.factor(suitable)))+
  scale_fill_manual(values = 'purple',
                    labels = 'viable \nhabitat')+
  geom_sf(data = countries, colour = "black", size = 0.01, fill = "grey", alpha = 1) +
  theme_bw()+
  coord_sf(xlim = c(-145, -105), ylim = c(22, 65),
           expand = FALSE)+
  scale_x_continuous(breaks =  seq(-145, -104, by = 10))+
  geom_label(aes(x = -114, y = 63.5, label = "habitat suitability"), size = 3, label.padding = unit(0.12, "lines"))+
  theme(panel.border = element_rect(fill = NA, size = 1),
        axis.text = element_text(colour = "black", size =8),
        axis.ticks = element_line(colour = "black"))+
  annotate("text", label = "Mexico", x = -109, y = 30, size = 3.5,  colour = "black")+
  annotate("text", label = "PACIFIC\nOCEAN", x = -125, y = 25, size = 3,  colour = "black", fontface="italic")+
  annotate("text", label = "United States", x = -113, y = 42, size = 3.5,  colour = "black")+
  annotate("text", label = "Canada", x = -115, y = 50, size = 3.5,  colour = "black")+
  geom_point(aes(x=-114.2639,y=28.665808), size = 1.5)+
  geom_point(aes(x=-117.154780,y=32.715380), size = 1.5)+
  geom_point(aes(x=-122.422286,y=37.739536), size = 1.5)+
  geom_point(aes(x=-123.1749,y=49.24639), size = 1.5)+
  annotate("text", label = "Santa Rosaliita", x=-114.2639,y=28.265808, size = 3,  colour = "black")+
  annotate("text", label = "San Diego", x=-117.154780,y=32.315380, size = 3,  colour = "black")+
  annotate("text", label = "San Francisco", x=-122.422286,y=37.339536, size = 3,  colour = "black")+
  annotate("text", label = "Vancouver", x=-123.1749,y=48.84639, size = 3,  colour = "black")+
  annotate("text", label = "(a)", x=-143.0, y= 63, size = 3,  colour = "black", fontface = "bold")+
  geom_segment(aes(x = -137, y = 54, xend = -125, yend = 54), linetype = "dashed")+
  geom_segment(aes(x = -120, y = 24, xend = -107, yend = 24), linetype = "dashed")+
  theme(legend.position=c(.2, .3))+
  theme(legend.background = element_rect(fill="white", linetype = "solid", colour = "black"))+
  theme(legend.title = element_text(size = 10))+
  theme(legend.text = element_text(size = 10))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.key.size = unit(0.4, "cm"))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.title=element_blank())


urchin_current_fig <- main_urchin_current +
  annotation_custom(grob = ggplotGrob(inset_urchin),
  xmin = -125,
  xmax = -106,
  ymin = 51,
  ymax = 62)

#ggsave("urchin_current_map.pdf", plot = urchin_current_fig, width = 3, height =  4)



##---------##
## abalone ##
##---------##

mass_abalone <- 500

jan$ami_abalone <- (jan$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jan$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jan$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jan$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jan$ami_abalone <- ifelse(jan$ami_abalone <= 0, NA, jan$ami_abalone)

feb$ami_abalone <- (feb$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(feb$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(feb$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(feb$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
feb$ami_abalone <- ifelse(feb$ami_abalone <= 0, NA, feb$ami_abalone)

mar$ami_abalone <- (mar$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mar$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(mar$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mar$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
mar$ami_abalone <- ifelse(mar$ami_abalone <= 0, NA, mar$ami_abalone)

apr$ami_abalone <- (apr$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(apr$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(apr$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(apr$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
apr$ami_abalone <- ifelse(apr$ami_abalone <= 0, NA, apr$ami_abalone)

may$ami_abalone <- (may$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(may$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(may$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(may$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
may$ami_abalone <- ifelse(may$ami_abalone <= 0, NA, may$ami_abalone)

jun$ami_abalone <- (jun$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jun$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jun$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jun$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jun$ami_abalone <- ifelse(jun$ami_abalone <= 0, NA, jun$ami_abalone)

jul$ami_abalone <- (jul$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jul$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jul$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jul$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jul$ami_abalone <- ifelse(jul$ami_abalone <= 0, NA, jul$ami_abalone)

aug$ami_abalone <- (aug$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(aug$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(aug$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(aug$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
aug$ami_abalone <- ifelse(aug$ami_abalone <= 0, NA, aug$ami_abalone)

sep$ami_abalone <- (sep$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(sep$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(sep$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(sep$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
sep$ami_abalone <- ifelse(sep$ami_abalone <= 0, NA, sep$ami_abalone)

oct$ami_abalone <- (oct$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oct$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(oct$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oct$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
oct$ami_abalone <- ifelse(oct$ami_abalone <= 0, NA, oct$ami_abalone)

nov$ami_abalone <- (nov$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(nov$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(nov$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(nov$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
nov$ami_abalone <- ifelse(nov$ami_abalone <= 0, NA, nov$ami_abalone)

dec$ami_abalone <- (dec$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(dec$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(dec$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(dec$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
dec$ami_abalone <- ifelse(dec$ami_abalone <= 0, NA, dec$ami_abalone)

## AMI

jan_ami_abalone <- resample(griddify(dplyr::select((jan),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
feb_ami_abalone <- resample(griddify(dplyr::select((feb),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
mar_ami_abalone <- resample(griddify(dplyr::select((mar),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
apr_ami_abalone <- resample(griddify(dplyr::select((apr),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
may_ami_abalone <- resample(griddify(dplyr::select((may),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
jun_ami_abalone <- resample(griddify(dplyr::select((jun),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
jul_ami_abalone <- resample(griddify(dplyr::select((jul),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
aug_ami_abalone <- resample(griddify(dplyr::select((aug),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
sep_ami_abalone <- resample(griddify(dplyr::select((sep),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
oct_ami_abalone <- resample(griddify(dplyr::select((oct),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
nov_ami_abalone <- resample(griddify(dplyr::select((nov),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')
dec_ami_abalone <- resample(griddify(dplyr::select((dec),1,2,14), nlon = 322, nlat=450), bath, method='bilinear')

mean_ami_raster_abalone <- mean(jan_ami_abalone, feb_ami_abalone, mar_ami_abalone, apr_ami_abalone, may_ami_abalone, jun_ami_abalone, jul_ami_abalone, aug_ami_abalone, sep_ami_abalone, oct_ami_abalone, nov_ami_abalone, dec_ami_abalone)
min_ami_raster_abalone <- min(jan_ami_abalone, feb_ami_abalone, mar_ami_abalone, apr_ami_abalone, may_ami_abalone, jun_ami_abalone, jul_ami_abalone, aug_ami_abalone, sep_ami_abalone, oct_ami_abalone, nov_ami_abalone, dec_ami_abalone)
max_ami_raster_abalone <- max(jan_ami_abalone, feb_ami_abalone, mar_ami_abalone, apr_ami_abalone, may_ami_abalone, jun_ami_abalone, jul_ami_abalone, aug_ami_abalone, sep_ami_abalone, oct_ami_abalone, nov_ami_abalone, dec_ami_abalone)

##----- relative AMI -----##

d <- d_abalone
colnames(d)[1] <- "ami_norm_abalone"
head(d)
d$oxygen <- as.numeric(d$oxygen)
d$code <- paste(d$oxygen,d$temp)
d <- subset(d, selec = -c(temp, oxygen))

#jan

jan <- left_join(jan, d, by = "code")
jan_rel_ami_abalone <- resample(griddify(dplyr::select((jan),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#feb

feb <- left_join(feb, d, by = "code")
feb_rel_ami_abalone <- resample(griddify(dplyr::select((feb),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#mar

mar <- left_join(mar, d, by = "code")
mar_rel_ami_abalone <- resample(griddify(dplyr::select((mar),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#apr

apr <- left_join(apr, d, by = "code")
apr_rel_ami_abalone <- resample(griddify(dplyr::select((apr),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#may

may <- left_join(may, d, by = "code")
may_rel_ami_abalone <- resample(griddify(dplyr::select((may),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#jun

jun <- left_join(jun, d, by = "code")
jun_rel_ami_abalone <- resample(griddify(dplyr::select((jun),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#jul

jul <- left_join(jul, d, by = "code")
jul_rel_ami_abalone <- resample(griddify(dplyr::select((jul),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#aug

aug <- left_join(aug, d, by = "code")
aug_rel_ami_abalone <- resample(griddify(dplyr::select((aug),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#sep

sep <- left_join(sep, d, by = "code")
sep_rel_ami_abalone <- resample(griddify(dplyr::select((sep),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#oct

oct <- left_join(oct, d, by = "code")
oct_rel_ami_abalone <- resample(griddify(dplyr::select((oct),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#nov

nov <- left_join(nov, d, by = "code")
nov_rel_ami_abalone <- resample(griddify(dplyr::select((nov),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

#dec

dec <- left_join(dec, d, by = "code")
dec_rel_ami_abalone <- resample(griddify(dplyr::select((dec),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

mean_rel_ami_raster_abalone <- mean(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)
min_rel_ami_raster_abalone <- min(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)
max_rel_ami_raster_abalone <- max(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)

head(d_abalone)

##--------------------------
##  calculating AMI crit bin
##--------------------------

head(d_abalone)
head(test_abalone)

# round abalone occurence data
test_abalone_round <- test_abalone
test_abalone_round[,-3] <- round(test_abalone_round[,-3],0)
test_abalone_round <- test_abalone_round[,-3]
names(test_abalone_round)[names(test_abalone_round) == 'kpa'] <- 'oxygen'

ami_crit_bin_df_abalone <- left_join(test_abalone_round, d_abalone, by = c('temp','oxygen'))
hist(ami_crit_bin_df_abalone$ami)

thres_abalone <- quantile(ami_crit_bin_df_abalone$ami, probs = 0.01)

ami_crit_abalone <- ggplot(data = ami_crit_bin_df_abalone, aes(x = ami)) +
  geom_histogram(bins = 100, alpha = 0.7, colour = 'black', fill = 'grey') +
  theme_bw() +
  geom_vline(xintercept = thres_abalone, linetype = 'dashed') +
  scale_x_continuous(breaks = seq(0,1,0.1), limits = c(0.5,1)) +
  scale_y_continuous(limits = c(0, 300)) +
  labs(x=expression(paste("normalized AMI"))) +
  labs(y=expression(paste("Count"))) +
  ggtitle("Red Abalone") +
  annotate(geom="text", x=0.5, y=300, label="(b)", fontface = "bold")


fig_s3 <- grid.arrange(ami_crit_urchin, ami_crit_abalone, ncol = 2)
ggsave("fig_s3.pdf", plot = fig_s2, width = 8, height = 3)



d_abalone$bin_ami <- ifelse(d_abalone$ami > thres_abalone, 1, 2)

inset_abalone <- ggplot()+
  geom_tile(data = d_abalone, aes(x = temp, y = oxygen, fill = as.factor(bin_ami)))+
  scale_fill_manual(values = c('firebrick','transparent'),
                    labels = c('suitable','unsuitable'),
                    name = 'normalized AMI')+
  scale_x_continuous(breaks = seq(0,35,5), limits = c(0,35), expand = c(0,0)) +
  scale_y_continuous(breaks = seq(0,23,3), limits = c(0,23), expand = c(0,0)) +
  geom_point(data = test_abalone, aes(x = temp, y = kpa), alpha = 0.1, size = 0.1)+
  labs(x=expression(paste("Temperature (" ^{o},"C)")))+
  labs(y=expression(paste("Oxygen (kPa)")))+
  theme_bw() +
  theme(panel.border = element_rect(colour="black", fill=NA, size=0.4), 
        axis.text = element_text(colour = "black", size = 6),
        axis.line=element_line(colour="black", size=0.3),
        axis.title = element_text(colour = "black", size = 6))+
  theme(legend.position="none")+
  theme(plot.background = element_rect(colour = "black", size=0.5))

min_rel_ami_raster_df_abalone <- na.omit(as.data.frame(min_rel_ami_raster_abalone, xy = TRUE))

min_rel_ami_raster_df_abalone$suitable <- ifelse(min_rel_ami_raster_df_abalone$layer >= thres_abalone, 1, 0)

min_rel_ami_raster_df_abalone <- subset(min_rel_ami_raster_df_abalone, suitable > 0)

main_abalone_current <- ggplot()+
  geom_tile(data = min_rel_ami_raster_df_abalone, aes(x = x, y = y, fill = as.factor(suitable)))+
  scale_fill_manual(values = 'firebrick',
                    labels = 'viable \nhabitat')+
  geom_sf(data = countries, colour = "black", size = 0.01, fill = "grey", alpha = 1) +
  theme_bw()+
  coord_sf(xlim = c(-145, -105), ylim = c(22, 65),
           expand = FALSE)+
  scale_x_continuous(breaks =  seq(-145, -104, by = 10))+
  geom_label(aes(x = -114, y = 63.5, label = "habitat suitability"), size = 3, label.padding = unit(0.12, "lines"))+
  theme(panel.border = element_rect(fill = NA, size = 1),
        axis.text = element_text(colour = "black", size =8),
        axis.ticks = element_line(colour = "black"))+
  annotate("text", label = "Mexico", x = -109, y = 30, size = 3.5,  colour = "black")+
  annotate("text", label = "PACIFIC\nOCEAN", x = -125, y = 25, size = 3,  colour = "black", fontface="italic")+
  annotate("text", label = "United States", x = -113, y = 42, size = 3.5,  colour = "black")+
  annotate("text", label = "Canada", x = -115, y = 50, size = 3.5,  colour = "black")+
  geom_point(aes(x=-114.2639,y=28.665808), size = 1.5)+
  geom_point(aes(x=-117.154780,y=32.715380), size = 1.5)+
  geom_point(aes(x=-122.422286,y=37.739536), size = 1.5)+
  geom_point(aes(x=-123.1749,y=49.24639), size = 1.5)+
  annotate("text", label = "Santa Rosaliita", x=-114.2639,y=28.265808, size = 3,  colour = "black")+
  annotate("text", label = "San Diego", x=-117.154780,y=32.315380, size = 3,  colour = "black")+
  annotate("text", label = "San Francisco", x=-122.422286,y=37.339536, size = 3,  colour = "black")+
  annotate("text", label = "Vancouver", x=-123.1749,y=48.84639, size = 3,  colour = "black")+
  annotate("text", label = "(b)", x=-143.0, y= 63, size = 3,  colour = "black", fontface = "bold")+
  geom_segment(aes(x = -132, y = 43.3, xend = -120, yend = 43.3), linetype = "dashed")+
  geom_segment(aes(x = -120, y = 24, xend = -107, yend = 24), linetype = "dashed")+
  theme(legend.position=c(.2, .3))+
  theme(legend.background = element_rect(fill="white", linetype = "solid", colour = "black"))+
  theme(legend.title = element_text(size = 10))+
  theme(legend.text = element_text(size = 10))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.key.size = unit(0.4, "cm"))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.title=element_blank())

  abalone_current_fig <- main_abalone_current +
  annotation_custom(grob = ggplotGrob(inset_abalone),
                    xmin = -125,
                    xmax = -106,
                    ymin = 51,
                    ymax = 62)

#ggsave("abalone_current_map.pdf", plot =  abalone_current_fig, width = 3, height =  4)


##--------------------------##
##                          ##
## future spatial modeling  ##
##                          ##
##--------------------------##

##----- load future layers and manipulate -----##

load("../ocean_model/future_bottom_temp_monthly.RDA") # load environmental dataframes
temp_df <- data.frame(df)

load("../ocean_model/future_bottom_oxy_monthly.RDA")
oxy_df <- data.frame(df)

load("../ocean_model/future_bottom_salt_monthly.RDA")
salt_df <- data.frame(df)

remove(df)

bath <- raster("../ocean_model/bath_grid.tif") #load bath from different folder
bath <- aggregate(bath, 8, fun=mean) #aggregate further to reduce size

depth <- resample(griddify(dplyr::select(temp_df,1,2,3), nlon = 322, nlat=450), bath, method='bilinear') #resample to create rasters for each month
temp <- resample(griddify(dplyr::select(temp_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,4), nlon = 322, nlat=450), bath, method='bilinear') 

jan <- stack(depth,temp,oxy,salt)
names(jan) <- c('depth','temp','oxy','salt')

temp <- resample(griddify(dplyr::select(temp_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,5), nlon = 322, nlat=450), bath, method='bilinear')

feb <- stack(depth,temp,oxy,salt)
names(feb) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,6), nlon = 322, nlat=450), bath, method='bilinear')

mar <- stack(depth,temp,oxy,salt)
names(mar) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,7), nlon = 322, nlat=450), bath, method='bilinear')

apr <- stack(depth,temp,oxy,salt)
names(apr) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,8), nlon = 322, nlat=450), bath, method='bilinear')

may <- stack(depth,temp,oxy,salt)
names(may) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,9), nlon = 322, nlat=450), bath, method='bilinear')

jun <- stack(depth,temp,oxy,salt)
names(jun) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,10), nlon = 322, nlat=450), bath, method='bilinear')

jul <- stack(depth,temp,oxy,salt)
names(jul) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,11), nlon = 322, nlat=450), bath, method='bilinear')

aug <- stack(depth,temp,oxy,salt)
names(aug) <- c('depth','temp','oxy','salt')

temp <- resample(griddify(dplyr::select(temp_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,12), nlon = 322, nlat=450), bath, method='bilinear')

sep <- stack(depth,temp,oxy,salt)
names(sep) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

oct <- stack(depth,temp,oxy,salt)
names(oct) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,14), nlon = 322, nlat=450), bath, method='bilinear')

nov <- stack(depth,temp,oxy,salt)
names(nov) <- c('depth','temp','oxy','salt')


temp <- resample(griddify(dplyr::select(temp_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear') 
oxy <- resample(griddify(dplyr::select(oxy_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear') 
salt <- resample(griddify(dplyr::select(salt_df,1,2,15), nlon = 322, nlat=450), bath, method='bilinear')

dec <- stack(depth,temp,oxy,salt)
names(dec) <- c('depth','temp','oxy','salt')


##--------------------------##
## future ocean model maps  ##
##--------------------------##

##---------##
## urchins
##---------##


## jan
jan <- as.data.frame(jan, xy = TRUE)
jan <- drop_na(jan)
jan$depth_pressure <- swPressure((jan$depth*-1), jan$y, eos="unesco")*100 #calculate pressure in mbar
jan$depth_pressure <- ifelse(jan$depth_pressure < 0, 0, jan$depth_pressure) #make anything above sea level = 0
jan$depth_pressure <- jan$depth_pressure + 1013.25 #add air pressure at surface
jan$kpa <- conv_o2(o2 = jan$oxy, from="umol_per_l", to = "kPa", 
                   temp = jan$temp, 
                   sal = jan$salt ,
                   atm_pres = jan$depth_pressure)
jan <- subset(jan, depth > -150 & depth < 50)

## feb
feb <- as.data.frame(feb, xy = TRUE)
feb <- drop_na(feb)
feb$depth_pressure <- swPressure((feb$depth*-1), feb$y, eos="unesco")*100 #calculate pressure in mbar
feb$depth_pressure <- ifelse(feb$depth_pressure < 0, 0, feb$depth_pressure) #make anything above sea level = 0
feb$depth_pressure <- feb$depth_pressure + 1013.25 #add air pressure at surface
feb$kpa <- conv_o2(o2 = feb$oxy, from="umol_per_l", to = "kPa", 
                   temp = feb$temp, 
                   sal = feb$salt ,
                   atm_pres = feb$depth_pressure)
feb <- subset(feb, depth > -150 & depth < 50)

## mar
mar <- as.data.frame(mar, xy = TRUE)
mar <- drop_na(mar)
mar$depth_pressure <- swPressure((mar$depth*-1), mar$y, eos="unesco")*100 #calculate pressure in mbar
mar$depth_pressure <- ifelse(mar$depth_pressure < 0, 0, mar$depth_pressure) #make anything above sea level = 0
mar$depth_pressure <- mar$depth_pressure + 1013.25 #add air pressure at surface
mar$kpa <- conv_o2(o2 = mar$oxy, from="umol_per_l", to = "kPa", 
                   temp = mar$temp, 
                   sal = mar$salt ,
                   atm_pres = mar$depth_pressure)
mar <- subset(mar, depth > -150 & depth < 50)

## apr
apr <- as.data.frame(apr, xy = TRUE)
apr <- drop_na(apr)
apr$depth_pressure <- swPressure((apr$depth*-1), apr$y, eos="unesco")*100 #calculate pressure in mbar
apr$depth_pressure <- ifelse(apr$depth_pressure < 0, 0, apr$depth_pressure) #make anything above sea level = 0
apr$depth_pressure <- apr$depth_pressure + 1013.25 #add air pressure at surface
apr$kpa <- conv_o2(o2 = apr$oxy, from="umol_per_l", to = "kPa", 
                   temp = apr$temp, 
                   sal = apr$salt ,
                   atm_pres = apr$depth_pressure)
apr <- subset(apr, depth > -150 & depth < 50)

## may
may <- as.data.frame(may, xy = TRUE)
may <- drop_na(may)
may$depth_pressure <- swPressure((may$depth*-1), may$y, eos="unesco")*100 #calculate pressure in mbar
may$depth_pressure <- ifelse(may$depth_pressure < 0, 0, may$depth_pressure) #make anything above sea level = 0
may$depth_pressure <- may$depth_pressure + 1013.25 #add air pressure at surface
may$kpa <- conv_o2(o2 = may$oxy, from="umol_per_l", to = "kPa", 
                   temp = may$temp, 
                   sal = may$salt ,
                   atm_pres = may$depth_pressure)
may <- subset(may, depth > -150 & depth < 50)

## jun
jun <- as.data.frame(jun, xy = TRUE)
jun <- drop_na(jun)
jun$depth_pressure <- swPressure((jun$depth*-1), jun$y, eos="unesco")*100 #calculate pressure in mbar
jun$depth_pressure <- ifelse(jun$depth_pressure < 0, 0, jun$depth_pressure) #make anything above sea level = 0
jun$depth_pressure <- jun$depth_pressure + 1013.25 #add air pressure at surface
jun$kpa <- conv_o2(o2 = jun$oxy, from="umol_per_l", to = "kPa", 
                   temp = jun$temp, 
                   sal = jun$salt ,
                   atm_pres = jun$depth_pressure)
jun <- subset(jun, depth > -150 & depth < 50)

## jul
jul <- as.data.frame(jul, xy = TRUE)
jul <- drop_na(jul)
jul$depth_pressure <- swPressure((jul$depth*-1), jul$y, eos="unesco")*100 #calculate pressure in mbar
jul$depth_pressure <- ifelse(jul$depth_pressure < 0, 0, jul$depth_pressure) #make anything above sea level = 0
jul$depth_pressure <- jul$depth_pressure + 1013.25 #add air pressure at surface
jul$kpa <- conv_o2(o2 = jul$oxy, from="umol_per_l", to = "kPa", 
                   temp = jul$temp, 
                   sal = jul$salt ,
                   atm_pres = jul$depth_pressure)
jul <- subset(jul, depth > -150 & depth < 50)

## aug
aug <- as.data.frame(aug, xy = TRUE)
aug <- drop_na(aug)
aug$depth_pressure <- swPressure((aug$depth*-1), aug$y, eos="unesco")*100 #calculate pressure in mbar
aug$depth_pressure <- ifelse(aug$depth_pressure < 0, 0, aug$depth_pressure) #make anything above sea level = 0
aug$depth_pressure <- aug$depth_pressure + 1013.25 #add air pressure at surface
aug$kpa <- conv_o2(o2 = aug$oxy, from="umol_per_l", to = "kPa", 
                   temp = aug$temp, 
                   sal = aug$salt ,
                   atm_pres = aug$depth_pressure)
aug <- subset(aug, depth > -150 & depth < 50)

## sep
sep <- as.data.frame(sep, xy = TRUE)
sep <- drop_na(sep)
sep$depth_pressure <- swPressure((sep$depth*-1), sep$y, eos="unesco")*100 #calculate pressure in mbar
sep$depth_pressure <- ifelse(sep$depth_pressure < 0, 0, sep$depth_pressure) #make anything above sea level = 0
sep$depth_pressure <- sep$depth_pressure + 1013.25 #add air pressure at surface
sep$kpa <- conv_o2(o2 = sep$oxy, from="umol_per_l", to = "kPa", 
                   temp = sep$temp, 
                   sal = sep$salt ,
                   atm_pres = sep$depth_pressure)
sep <- subset(sep, depth > -150 & depth < 50)

## oct
oct <- as.data.frame(oct, xy = TRUE)
oct <- drop_na(oct)
oct$depth_pressure <- swPressure((oct$depth*-1), oct$y, eos="unesco")*100 #calculate pressure in mbar
oct$depth_pressure <- ifelse(oct$depth_pressure < 0, 0, oct$depth_pressure) #make anything above sea level = 0
oct$depth_pressure <- oct$depth_pressure + 1013.25 #add air pressure at surface
oct$kpa <- conv_o2(o2 = oct$oxy, from="umol_per_l", to = "kPa", 
                   temp = oct$temp, 
                   sal = oct$salt ,
                   atm_pres = oct$depth_pressure)
oct <- subset(oct, depth > -150 & depth < 50)

## nov
nov <- as.data.frame(nov, xy = TRUE)
nov <- drop_na(nov)
nov$depth_pressure <- swPressure((nov$depth*-1), nov$y, eos="unesco")*100 #calculate pressure in mbar
nov$depth_pressure <- ifelse(nov$depth_pressure < 0, 0, nov$depth_pressure) #make anything above sea level = 0
nov$depth_pressure <- nov$depth_pressure + 1013.25 #add air pressure at surface
nov$kpa <- conv_o2(o2 = nov$oxy, from="umol_per_l", to = "kPa", 
                   temp = nov$temp, 
                   sal = nov$salt ,
                   atm_pres = nov$depth_pressure)
nov <- subset(nov, depth > -150 & depth < 50)

## dec
dec <- as.data.frame(dec, xy = TRUE)
dec <- drop_na(dec)
dec$depth_pressure <- swPressure((dec$depth*-1), dec$y, eos="unesco")*100 #calculate pressure in mbar
dec$depth_pressure <- ifelse(dec$depth_pressure < 0, 0, dec$depth_pressure) #make anything above sea level = 0
dec$depth_pressure <- dec$depth_pressure + 1013.25 #add air pressure at surface
dec$kpa <- conv_o2(o2 = dec$oxy, from="umol_per_l", to = "kPa", 
                   temp = dec$temp, 
                   sal = dec$salt ,
                   atm_pres = dec$depth_pressure)
dec <- subset(dec, depth > -150 & depth < 50)

##-------##
##  AMI  ##
##-------##

jan$ami_urchin <- (jan$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jan$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jan$temp^2*x2) + (jan$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jan$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jan$ami_urchin <- ifelse(jan$ami_urchin <= 0, NA, jan$ami_urchin)

feb$ami_urchin <- (feb$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(feb$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((feb$temp^2*x2) + (feb$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(feb$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
feb$ami_urchin <- ifelse(feb$ami_urchin <= 0, NA, feb$ami_urchin)

mar$ami_urchin <- (mar$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mar$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((mar$temp^2*x2) + (mar$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(mar$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
mar$ami_urchin <- ifelse(mar$ami_urchin <= 0, NA, mar$ami_urchin)

apr$ami_urchin <- (apr$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(apr$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((apr$temp^2*x2) + (apr$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(apr$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
apr$ami_urchin <- ifelse(apr$ami_urchin <= 0, NA, apr$ami_urchin)

may$ami_urchin <- (may$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(may$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((may$temp^2*x2) + (may$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(may$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
may$ami_urchin <- ifelse(may$ami_urchin <= 0, NA, may$ami_urchin)

jun$ami_urchin <- (jun$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jun$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jun$temp^2*x2) + (jun$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jun$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jun$ami_urchin <- ifelse(jun$ami_urchin <= 0, NA, jun$ami_urchin)

jul$ami_urchin <- (jul$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jul$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((jul$temp^2*x2) + (jul$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(jul$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
jul$ami_urchin <- ifelse(jul$ami_urchin <= 0, NA, jul$ami_urchin)

aug$ami_urchin <- (aug$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(aug$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((aug$temp^2*x2) + (aug$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(aug$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
aug$ami_urchin <- ifelse(aug$ami_urchin <= 0, NA, aug$ami_urchin)

sep$ami_urchin <- (sep$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(sep$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((sep$temp^2*x2) + (sep$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(sep$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
sep$ami_urchin <- ifelse(sep$ami_urchin <= 0, NA, sep$ami_urchin)

oct$ami_urchin <- (oct$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oct$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((oct$temp^2*x2) + (oct$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(oct$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
oct$ami_urchin <- ifelse(oct$ami_urchin <= 0, NA, oct$ami_urchin)

nov$ami_urchin <- (nov$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(nov$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((nov$temp^2*x2) + (nov$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(nov$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
nov$ami_urchin <- ifelse(nov$ami_urchin <= 0, NA, nov$ami_urchin)

dec$ami_urchin <- (dec$kpa*(((smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(dec$temp+kelvin)))))*(mass_urchin^mass_scaler_urchin))/
                              (((dec$temp^2*x2) + (dec$temp*x) + p)*(mass_urchin^mass_scaler_pcrit_urchin)))) - 
  (smr_intercept_urchin*(exp(smr_slope_urchin/(kb*(dec$temp+kelvin))))*(mass_urchin^mass_scaler_urchin))
dec$ami_urchin <- ifelse(dec$ami_urchin <= 0, NA, dec$ami_urchin)

## AMI

jan_ami_urchin_future <- resample(griddify(dplyr::select((jan),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
feb_ami_urchin_future <- resample(griddify(dplyr::select((feb),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
mar_ami_urchin_future <- resample(griddify(dplyr::select((mar),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
apr_ami_urchin_future <- resample(griddify(dplyr::select((apr),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
may_ami_urchin_future <- resample(griddify(dplyr::select((may),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
jun_ami_urchin_future <- resample(griddify(dplyr::select((jun),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
jul_ami_urchin_future <- resample(griddify(dplyr::select((jul),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
aug_ami_urchin_future <- resample(griddify(dplyr::select((aug),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
sep_ami_urchin_future <- resample(griddify(dplyr::select((sep),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
oct_ami_urchin_future <- resample(griddify(dplyr::select((oct),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
nov_ami_urchin_future <- resample(griddify(dplyr::select((nov),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')
dec_ami_urchin_future <- resample(griddify(dplyr::select((dec),1,2,9), nlon = 322, nlat=450), bath, method='bilinear')


mean_ami_raster_urchin_future <- mean(jan_ami_urchin_future, feb_ami_urchin_future, mar_ami_urchin_future, apr_ami_urchin_future, may_ami_urchin_future, jun_ami_urchin_future, jul_ami_urchin_future, aug_ami_urchin_future, sep_ami_urchin_future, oct_ami_urchin_future, nov_ami_urchin_future, dec_ami_urchin_future)
min_ami_raster_urchin_future <- min(jan_ami_urchin_future, feb_ami_urchin_future, mar_ami_urchin_future, apr_ami_urchin_future, may_ami_urchin_future, jun_ami_urchin_future, jul_ami_urchin_future, aug_ami_urchin_future, sep_ami_urchin_future, oct_ami_urchin_future, nov_ami_urchin_future, dec_ami_urchin_future)
max_ami_raster_urchin_future <- max(jan_ami_urchin_future, feb_ami_urchin_future, mar_ami_urchin_future, apr_ami_urchin_future, may_ami_urchin_future, jun_ami_urchin_future, jul_ami_urchin_future, aug_ami_urchin_future, sep_ami_urchin_future, oct_ami_urchin_future, nov_ami_urchin_future, dec_ami_urchin_future)

##----- relative AMI -----##

d <- d_urchin
colnames(d)[1] <- "ami_norm_urchin"
head(d)
d$oxygen <- as.numeric(d$oxygen)
d$code <- paste(d$oxygen,d$temp)
d <- subset(d, selec = -c(temp, oxygen))

#jan
jan$oxygen_round <- round(jan$kpa)
jan$temp_round <- round(jan$temp)
jan$code <- paste(jan$oxygen_round,jan$temp_round)
jan <- left_join(jan, d, by = "code")
jan_rel_ami_urchin <- resample(griddify(dplyr::select((jan),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#feb
feb$oxygen_round <- round(feb$kpa)
feb$temp_round <- round(feb$temp)
feb$code <- paste(feb$oxygen_round,feb$temp_round)
feb <- left_join(feb, d, by = "code")
feb_rel_ami_urchin <- resample(griddify(dplyr::select((feb),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#mar
mar$oxygen_round <- round(mar$kpa)
mar$temp_round <- round(mar$temp)
mar$code <- paste(mar$oxygen_round,mar$temp_round)
mar <- left_join(mar, d, by = "code")
mar_rel_ami_urchin <- resample(griddify(dplyr::select((mar),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#apr
apr$oxygen_round <- round(apr$kpa)
apr$temp_round <- round(apr$temp)
apr$code <- paste(apr$oxygen_round,apr$temp_round)
apr <- left_join(apr, d, by = "code")
apr_rel_ami_urchin <- resample(griddify(dplyr::select((apr),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#may
may$oxygen_round <- round(may$kpa)
may$temp_round <- round(may$temp)
may$code <- paste(may$oxygen_round,may$temp_round)
may <- left_join(may, d, by = "code")
may_rel_ami_urchin <- resample(griddify(dplyr::select((may),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#jun
jun$oxygen_round <- round(jun$kpa)
jun$temp_round <- round(jun$temp)
jun$code <- paste(jun$oxygen_round,jun$temp_round)
jun <- left_join(jun, d, by = "code")
jun_rel_ami_urchin <- resample(griddify(dplyr::select((jun),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#jul
jul$oxygen_round <- round(jul$kpa)
jul$temp_round <- round(jul$temp)
jul$code <- paste(jul$oxygen_round,jul$temp_round)
jul <- left_join(jul, d, by = "code")
jul_rel_ami_urchin <- resample(griddify(dplyr::select((jul),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#aug
aug$oxygen_round <- round(aug$kpa)
aug$temp_round <- round(aug$temp)
aug$code <- paste(aug$oxygen_round,aug$temp_round)
aug <- left_join(aug, d, by = "code")
aug_rel_ami_urchin <- resample(griddify(dplyr::select((aug),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#sep
sep$oxygen_round <- round(sep$kpa)
sep$temp_round <- round(sep$temp)
sep$code <- paste(sep$oxygen_round,sep$temp_round)
sep <- left_join(sep, d, by = "code")
sep_rel_ami_urchin <- resample(griddify(dplyr::select((sep),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#oct
oct$oxygen_round <- round(oct$kpa)
oct$temp_round <- round(oct$temp)
oct$code <- paste(oct$oxygen_round,oct$temp_round)
oct <- left_join(oct, d, by = "code")
oct_rel_ami_urchin <- resample(griddify(dplyr::select((oct),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#nov
nov$oxygen_round <- round(nov$kpa)
nov$temp_round <- round(nov$temp)
nov$code <- paste(nov$oxygen_round,nov$temp_round)
nov <- left_join(nov, d, by = "code")
nov_rel_ami_urchin <- resample(griddify(dplyr::select((nov),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')

#dec
dec$oxygen_round <- round(dec$kpa)
dec$temp_round <- round(dec$temp)
dec$code <- paste(dec$oxygen_round,dec$temp_round)
dec <- left_join(dec, d, by = "code")
dec_rel_ami_urchin <- resample(griddify(dplyr::select((dec),1,2,13), nlon = 322, nlat=450), bath, method='bilinear')


mean_rel_ami_raster_urchin_future <- mean(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)
min_rel_ami_raster_urchin_future  <- min(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)
max_rel_ami_raster_urchin_future  <- max(jan_rel_ami_urchin, feb_rel_ami_urchin, mar_rel_ami_urchin, apr_rel_ami_urchin, may_rel_ami_urchin, jun_rel_ami_urchin, jul_rel_ami_urchin, aug_rel_ami_urchin, sep_rel_ami_urchin, oct_rel_ami_urchin, nov_rel_ami_urchin, dec_rel_ami_urchin)

min_rel_ami_raster_future_df_urchin <- na.omit(as.data.frame(min_rel_ami_raster_urchin_future, xy = TRUE))

min_rel_ami_raster_future_df_urchin$suitable <- ifelse(min_rel_ami_raster_future_df_urchin$layer >= thres_urchin, 1, 0)

min_rel_ami_raster_future_df_urchin <- subset(min_rel_ami_raster_future_df_urchin, suitable > 0)

main_urchin_future <- ggplot()+
  geom_tile(data = min_rel_ami_raster_future_df_urchin, aes(x = x, y = y, fill = as.factor(suitable)))+
  scale_fill_manual(values = 'purple',
                    labels = 'viable \nhabitat')+
  geom_sf(data = countries, colour = "black", size = 0.01, fill = "grey", alpha = 1) +
  theme_bw()+
  coord_sf(xlim = c(-145, -105), ylim = c(22, 65),
           expand = FALSE)+
  scale_x_continuous(breaks =  seq(-145, -104, by = 10))+
  geom_label(aes(x = -114, y = 62.5, label = "future\n habitat suitability"), size = 3, label.padding = unit(0.12, "lines"))+
  theme(panel.border = element_rect(fill = NA, size = 1),
        axis.text = element_text(colour = "black", size =8),
        axis.ticks = element_line(colour = "black"))+
  annotate("text", label = "Mexico", x = -109, y = 30, size = 3.5,  colour = "black")+
  annotate("text", label = "PACIFIC\nOCEAN", x = -125, y = 25, size = 3,  colour = "black", fontface="italic")+
  annotate("text", label = "United States", x = -113, y = 42, size = 3.5,  colour = "black")+
  annotate("text", label = "Canada", x = -115, y = 50, size = 3.5,  colour = "black")+
  geom_point(aes(x=-114.2639,y=28.665808), size = 1.5)+
  geom_point(aes(x=-117.154780,y=32.715380), size = 1.5)+
  geom_point(aes(x=-122.422286,y=37.739536), size = 1.5)+
  geom_point(aes(x=-123.1749,y=49.24639), size = 1.5)+
  annotate("text", label = "Santa Rosaliita", x=-114.2639,y=28.265808, size = 3,  colour = "black")+
  annotate("text", label = "San Diego", x=-117.154780,y=32.315380, size = 3,  colour = "black")+
  annotate("text", label = "San Francisco", x=-122.422286,y=37.339536, size = 3,  colour = "black")+
  annotate("text", label = "Vancouver", x=-123.1749,y=48.84639, size = 3,  colour = "black")+
  annotate("text", label = "(c)", x=-143.0, y= 63, size = 3,  colour = "black", fontface = "bold")+
  theme(legend.position=c(.2, .3))+
  theme(legend.background = element_rect(fill="white", linetype = "solid", colour = "black"))+
  theme(legend.title = element_text(size = 10))+
  theme(legend.text = element_text(size = 10))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.key.size = unit(0.4, "cm"))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.title=element_blank())

##---------##
## abalone ##
##---------##

##-------##
##  AMI  ##
##-------##

jan$ami_abalone <- (jan$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jan$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jan$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jan$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jan$ami_abalone <- ifelse(jan$ami_abalone <= 0, NA, jan$ami_abalone)

feb$ami_abalone <- (feb$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(feb$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(feb$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(feb$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
feb$ami_abalone <- ifelse(feb$ami_abalone <= 0, NA, feb$ami_abalone)

mar$ami_abalone <- (mar$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mar$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(mar$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(mar$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
mar$ami_abalone <- ifelse(mar$ami_abalone <= 0, NA, mar$ami_abalone)

apr$ami_abalone <- (apr$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(apr$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(apr$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(apr$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
apr$ami_abalone <- ifelse(apr$ami_abalone <= 0, NA, apr$ami_abalone)

may$ami_abalone <- (may$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(may$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(may$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(may$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
may$ami_abalone <- ifelse(may$ami_abalone <= 0, NA, may$ami_abalone)

jun$ami_abalone <- (jun$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jun$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jun$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jun$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jun$ami_abalone <- ifelse(jun$ami_abalone <= 0, NA, jun$ami_abalone)

jul$ami_abalone <- (jul$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jul$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(jul$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(jul$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
jul$ami_abalone <- ifelse(jul$ami_abalone <= 0, NA, jul$ami_abalone)

aug$ami_abalone <- (aug$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(aug$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(aug$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(aug$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
aug$ami_abalone <- ifelse(aug$ami_abalone <= 0, NA, aug$ami_abalone)

sep$ami_abalone <- (sep$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(sep$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(sep$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(sep$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
sep$ami_abalone <- ifelse(sep$ami_abalone <= 0, NA, sep$ami_abalone)

oct$ami_abalone <- (oct$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oct$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(oct$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(oct$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
oct$ami_abalone <- ifelse(oct$ami_abalone <= 0, NA, oct$ami_abalone)

nov$ami_abalone <- (nov$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(nov$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(nov$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(nov$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
nov$ami_abalone <- ifelse(nov$ami_abalone <= 0, NA, nov$ami_abalone)

dec$ami_abalone <- (dec$kpa*(((smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(dec$temp+kelvin))))*(mass_abalone^mass_scaler_abalone)))/
                               ((m*(exp(dec$temp*n)))*(mass_abalone^mass_scaler_pcrit_abalone)))) - 
  (smr_intercept_abalone*(exp(smr_slope_abalone/(kb*(dec$temp+kelvin))))*(mass_abalone^mass_scaler_abalone))
dec$ami_abalone <- ifelse(dec$ami_abalone <= 0, NA, dec$ami_abalone)


## AMI

jan_ami_abalone_future <- resample(griddify(dplyr::select((jan),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
feb_ami_abalone_future <- resample(griddify(dplyr::select((feb),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
mar_ami_abalone_future <- resample(griddify(dplyr::select((mar),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
apr_ami_abalone_future <- resample(griddify(dplyr::select((apr),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
may_ami_abalone_future <- resample(griddify(dplyr::select((may),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
jun_ami_abalone_future <- resample(griddify(dplyr::select((jun),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
jul_ami_abalone_future <- resample(griddify(dplyr::select((jul),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
aug_ami_abalone_future <- resample(griddify(dplyr::select((aug),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
sep_ami_abalone_future <- resample(griddify(dplyr::select((sep),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
oct_ami_abalone_future <- resample(griddify(dplyr::select((oct),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
nov_ami_abalone_future <- resample(griddify(dplyr::select((nov),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')
dec_ami_abalone_future <- resample(griddify(dplyr::select((dec),1,2,15), nlon = 322, nlat=450), bath, method='bilinear')


mean_ami_raster_abalone_future <- mean(jan_ami_abalone_future, feb_ami_abalone_future, mar_ami_abalone_future, apr_ami_abalone_future, may_ami_abalone_future, jun_ami_abalone_future, jul_ami_abalone_future, aug_ami_abalone_future, sep_ami_abalone_future, oct_ami_abalone_future, nov_ami_abalone_future, dec_ami_abalone_future)
min_ami_raster_abalone_future <- min(jan_ami_abalone_future, feb_ami_abalone_future, mar_ami_abalone_future, apr_ami_abalone_future, may_ami_abalone_future, jun_ami_abalone_future, jul_ami_abalone_future, aug_ami_abalone_future, sep_ami_abalone_future, oct_ami_abalone_future, nov_ami_abalone_future, dec_ami_abalone_future)
max_ami_raster_abalone_future <- max(jan_ami_abalone_future, feb_ami_abalone_future, mar_ami_abalone_future, apr_ami_abalone_future, may_ami_abalone_future, jun_ami_abalone_future, jul_ami_abalone_future, aug_ami_abalone_future, sep_ami_abalone_future, oct_ami_abalone_future, nov_ami_abalone_future, dec_ami_abalone_future)


##----- relative AMI -----##

d <- d_abalone
colnames(d)[1] <- "ami_norm_abalone"
head(d)
d$oxygen <- as.numeric(d$oxygen)
d$code <- paste(d$oxygen,d$temp)
d <- subset(d, selec = -c(temp, oxygen))

#jan

jan <- left_join(jan, d, by = "code")
jan_rel_ami_abalone <- resample(griddify(dplyr::select((jan),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#feb

feb <- left_join(feb, d, by = "code")
feb_rel_ami_abalone <- resample(griddify(dplyr::select((feb),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#mar

mar <- left_join(mar, d, by = "code")
mar_rel_ami_abalone <- resample(griddify(dplyr::select((mar),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#apr

apr <- left_join(apr, d, by = "code")
apr_rel_ami_abalone <- resample(griddify(dplyr::select((apr),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#may

may <- left_join(may, d, by = "code")
may_rel_ami_abalone <- resample(griddify(dplyr::select((may),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#jun

jun <- left_join(jun, d, by = "code")
jun_rel_ami_abalone <- resample(griddify(dplyr::select((jun),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#jul

jul <- left_join(jul, d, by = "code")
jul_rel_ami_abalone <- resample(griddify(dplyr::select((jul),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#aug

aug <- left_join(aug, d, by = "code")
aug_rel_ami_abalone <- resample(griddify(dplyr::select((aug),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#sep

sep <- left_join(sep, d, by = "code")
sep_rel_ami_abalone <- resample(griddify(dplyr::select((sep),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#oct

oct <- left_join(oct, d, by = "code")
oct_rel_ami_abalone <- resample(griddify(dplyr::select((oct),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#nov

nov <- left_join(nov, d, by = "code")
nov_rel_ami_abalone <- resample(griddify(dplyr::select((nov),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')

#dec

dec <- left_join(dec, d, by = "code")
dec_rel_ami_abalone <- resample(griddify(dplyr::select((dec),1,2,16), nlon = 322, nlat=450), bath, method='bilinear')


mean_rel_ami_raster_abalone_future <- mean(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)
min_rel_ami_raster_abalone_future  <- min(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)
max_rel_ami_raster_abalone_future  <- max(jan_rel_ami_abalone, feb_rel_ami_abalone, mar_rel_ami_abalone, apr_rel_ami_abalone, may_rel_ami_abalone, jun_rel_ami_abalone, jul_rel_ami_abalone, aug_rel_ami_abalone, sep_rel_ami_abalone, oct_rel_ami_abalone, nov_rel_ami_abalone, dec_rel_ami_abalone)


min_rel_ami_raster_future_df_abalone <- na.omit(as.data.frame(min_rel_ami_raster_abalone_future, xy = TRUE))

min_rel_ami_raster_future_df_abalone$suitable <- ifelse(min_rel_ami_raster_future_df_abalone$layer >= thres_abalone, 1, 0)

min_rel_ami_raster_future_df_abalone <- subset(min_rel_ami_raster_future_df_abalone, suitable > 0)

main_abalone_future <- ggplot()+
  geom_tile(data = min_rel_ami_raster_future_df_abalone, aes(x = x, y = y, fill = as.factor(suitable)))+
  scale_fill_manual(values = 'firebrick',
                    labels = 'viable \nhabitat')+
  geom_sf(data = countries, colour = "black", size = 0.01, fill = "grey", alpha = 1) +
  theme_bw()+
  coord_sf(xlim = c(-145, -105), ylim = c(22, 65),
           expand = FALSE)+
  scale_x_continuous(breaks =  seq(-145, -104, by = 10))+
  geom_label(aes(x = -114, y = 62.5, label = "future\n habitat suitability"), size = 3, label.padding = unit(0.12, "lines"))+
  theme(panel.border = element_rect(fill = NA, size = 1),
        axis.text = element_text(colour = "black", size =8),
        axis.ticks = element_line(colour = "black"))+
  annotate("text", label = "Mexico", x = -109, y = 30, size = 3.5,  colour = "black")+
  annotate("text", label = "PACIFIC\nOCEAN", x = -125, y = 25, size = 3,  colour = "black", fontface="italic")+
  annotate("text", label = "United States", x = -113, y = 42, size = 3.5,  colour = "black")+
  annotate("text", label = "Canada", x = -115, y = 50, size = 3.5,  colour = "black")+
  geom_point(aes(x=-114.2639,y=28.665808), size = 1.5)+
  geom_point(aes(x=-117.154780,y=32.715380), size = 1.5)+
  geom_point(aes(x=-122.422286,y=37.739536), size = 1.5)+
  geom_point(aes(x=-123.1749,y=49.24639), size = 1.5)+
  annotate("text", label = "Santa Rosaliita", x=-114.2639,y=28.265808, size = 3,  colour = "black")+
  annotate("text", label = "San Diego", x=-117.154780,y=32.315380, size = 3,  colour = "black")+
  annotate("text", label = "San Francisco", x=-122.422286,y=37.339536, size = 3,  colour = "black")+
  annotate("text", label = "Vancouver", x=-123.1749,y=48.84639, size = 3,  colour = "black")+
  annotate("text", label = "(d)", x=-143.0, y= 63, size = 3,  colour = "black", fontface = "bold")+
  theme(legend.position=c(.2, .3))+
  theme(legend.background = element_rect(fill="white", linetype = "solid", colour = "black"))+
  theme(legend.title = element_text(size = 10))+
  theme(legend.text = element_text(size = 10))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.key.size = unit(0.4, "cm"))+
  theme(axis.title.x=element_blank())+
  theme(axis.title.y=element_blank())+
  theme(legend.title=element_blank()) 


fig_7 <- grid.arrange(urchin_current_fig,abalone_current_fig,main_urchin_future, main_abalone_future, ncol = 2)
ggsave("fig_7.pdf", plot = fig_7, width = 6, height = 8)