function [unwarped] = vBlebb(video,mechanics,options,preComp)
% vBlebb - main virtual blebbistatin unwarping file
% Louis Woodhams and John Boyle 2016

% Get first frame interpolant
[~,RGBImage,~,~] = getVideoFrame(video,video.Frames(1),options,false);
imsize = size(RGBImage);
warpingFcn = options.fwdWarpingFcn; % grab the warping function
P = mechanics.P; % warping parameters (numRegions x 6 x numFrames)
numRegions = size(P,1);
numFrames = numel(video.Frames);
if ndims(RGBImage) == 3
    colorChannels = 3;
else
    colorChannels = 1;
end

% create default camera calibration
xv = 1:imsize(1);
yv = 1:imsize(2);
[X.x, X.y] = ndgrid(xv,yv);
% get normalization matrices(fwd,inv)
options.CameraCalibration = normalizeCoordinateSystem(X); 

boxSize = options.Meshing.boxSize; % LW 7/7/2019
rangeXY = zeros(numRegions,4); % x,y pixel ranges per region
for i = 1:numRegions
    X = preComp(i).X.x; % get the original X coordinates for region
    Y = preComp(i).X.y; % get the original Y coordinates
    % convert normalized coordinates(-0.5 to 0.5) -> Image coordinates
    fcn = @(X) options.CameraCalibration.inv*X; 
    [x,y] = transform2D(X,Y,fcn); % transform to image coordinates
    
    % Get the x and y pixel ranges 
    % line below giving dimension mismatch errors
    % range(i,:) = ceil([min(x(:)) max(x(:)) min(y(:)) max(y(:))]');
    minxy = ceil([min(x(:)) min(y(:))]);
    rangeXY(i,:) = reshape([minxy; minxy+boxSize-1],[1 4]); % LW 7/7/2019
end

%% Code below is for if you want to write video as frames are unwarped
% nmov = VideoWriter(videoname);
% nmov.FrameRate = 30;
% open(nmov)
% h = figure;

%% loop over frames
unwarped = zeros(imsize(1), imsize(2), colorChannels, numFrames);
h = waitbar(0,'Unwarping frames');
for f = 1:1:numFrames % for each frame
    
    P_ = P(:,:,f); % grab the warping parameters for current frame
    
    % load the current image
    [~, RGBImage] = getVideoFrame(video,video.Frames(f),options); 
    nIm = im2double(RGBImage);
    
    if f == 1 % get normalized coordinates (-0.5 to 0.5) for whole frame
        K = options.CameraCalibration;
        sz = size(RGBImage);
        [X,Y] = ndgrid(1:sz(1),1:sz(2));
        [xIm,yIm] = transform2D(X,Y,@(X) K.fwd*X);
    end
    
    % create interpolant for current frame in normalized coordinates
    if  ndims(RGBImage) == 3
        warped.R = griddedInterpolant(xIm,yIm,nIm(:,:,1));
        warped.G = griddedInterpolant(xIm,yIm,nIm(:,:,2));
        warped.B = griddedInterpolant(xIm,yIm,nIm(:,:,3));
    else
        warped.R = griddedInterpolant(xIm,yIm,nIm(:,:,1));
    end

%% Code below for plotting results as calculated / writing frames    
%     subplot(2,1,1)
%     image(RGBImage)
%     axis equal
%     axis tight

    for i = 1:numRegions % for each region
        % get coordinates of region
        X = preComp(i).X.x; % get the original X coordinates
        Y = preComp(i).X.y; % get the original Y coordinates
        N = preComp(i).N;   % get the normalization parameters
        Pc = P_(i,:);       % grab the warp parameters
        
        % get warped normalized coordinates for region
        [x,y] = transform2D(X, Y, warpingFcn, Pc, N);
        
        % interpolate pixel values
        if  ndims(RGBImage) == 3
            R = warped.R(x, y);
            G = warped.G(x, y);
            B = warped.B(x, y);
        else
            R = warped.R(x, y);
        end
        cR = rangeXY(i,:);
        % construct an image (assign pixel values for current region)
        if  ndims(RGBImage) == 3
            nIm(cR(1):cR(2),cR(3):cR(4),1) = R;
            nIm(cR(1):cR(2),cR(3):cR(4),2) = G;
            nIm(cR(1):cR(2),cR(3):cR(4),3) = B;
        else
            nIm(cR(1):cR(2),cR(3):cR(4),1) = R;
        end
    end

%% Code below for plotting results as calculated / writing frames
%     subplot(2,1,2)
%     image(im2uint8(nIm))
%     axis equal
%     axis tight
%     colormap gray
%     title(['Frame ' num2str(f)])
%     %if f == 1
%     %    pause
%     %end
%     nframe = getframe(h);
%     writeVideo(nmov,nframe)

    unwarped(:,:,:,f) = nIm;
    waitbar(f/numFrames,h)
end
delete(h)
% close(nmov)
