function [regions,centers,polygon] = meshRegions(varargin)
% this function takes in an image and options, then allows a user to select
% a region for analysis, along with a region for exclusion (if desired). it
% outputs the initial points in a numberOfPoints x 2 matrix, with  y x
% (image) coordinates.

% process general inputs
im = varargin{1}; % set the image
imsize = size(im); % get it's size
options = varargin{2}; % get the options

%% switch the Meshing Mode

switch options.Meshing.mode
    case 'boxes' % simple box mode
        if nargin == 2 % check if a polygon was given
            figure; imshow(im); % if it was show the image
            hold on % set up for more plots
            % fill the points in according to the polygon and the spacing
            [regions,centers,polygon] = meshBoxes(options.Meshing.spacing,imsize,...
                options.Meshing.boxSize);
        elseif nargin == 3; % if polygon was given
            polygon = varargin{3}; % just use it
            % fill the points in according to the polygon and the spacing
            % using given polygon
            [regions,centers,polygon] = meshBoxes(options.Meshing.spacing,imsize,...
                options.Meshing.boxSize,polygon);
        end
    case 'points'
        if nargin == 3
            centers = varargin{3};
            regions = makeBoxRegions(centers,options.Meshing.boxSize);
        else
            figure; imshow(im); % if it was show the image
            hold on % set up for more plots
            [regions,centers] = meshPoints(options.Meshing.boxSize); % get the regions by selection
        end
    case 'multiline'
        if nargin == 2 % check if a polygon was given
            figure; imshow(im); % if it was show the image
            hold on % set up for more plots
            % fill the points in according to the polygon and the spacing
            [regions,centers,polygon] = meshBoxesLine(options.Meshing.spacing,...
                options.Meshing.boxSize);
        elseif nargin == 3; % if polygon was given
            points = varargin{3}; % just use it
            % fill the points in according to the polygon and the spacing
            % using given polygon
            [regions,centers,polygon] = meshBoxesLine(options.Meshing.spacing,...
                options.Meshing.boxSize,points);
        end
    case 'centerBoxes' % simple box mode
        [regions,centers] = meshCenterBoxes(options.Meshing.spacing,imsize,...
            options.Meshing.boxSize, options.Meshing.AnalysisRegionPrecentage);
end

% get the camera calibration
if isa(options.CameraCalibration,'calibrationSession');
    K = calibrationParameterstoMatrix(options.CameraCalibration);
else
    K  = options.CameraCalibration;
end
% normalize the coordinates in the regions
if options.is3d
    for i = 1:numel(regions) % loop over regions
        % get X,Y,Z values
        X = regions(i).x; Y = regions(i).y; Z = regions(i).z;
        [x,y,z] = transform3D(X,Y,Z,@(X) K.fwd*X); % transform them with the calibraiton
        regions(i).x = x; regions(i).y = y; regions(i).z = z; % save the results
    end
else
    for i = 1:numel(regions)
        X = regions(i).x; Y = regions(i).y;
        fcn = @(X) K.fwd*X;
        [x,y] = transform2D(X,Y,fcn); % transform to normalized image coordinates
        XY.x = x;
        XY.y = y;
        [regions(i).N] = normalizeCoordinateSystem(XY); % normalize hte new coordinate system
        % perform additional processing
        switch options.Meshing.mode
            case 'multiline'
                N = regions(i).N;
                R = regions(i).R;
                fcn = @(X) N.inv * R * N.fwd * X;
                [x,y] = transform2D(x,y,fcn);
        end
        regions(i).x = x; regions(i).y = y;
    end
end
%close all
end

function [regions,centers,polygon] = meshBoxes(varargin)
spacing = varargin{1}; % get the spacing
imsize = varargin{2}; % get the image size
boxSize = varargin{3}; % get the box size
if nargin == 4 % check if we the user gave a polygon
    polygon = varargin{4};
    centers = getBoxCenters(polygon,spacing,imsize);
    regions = makeBoxRegions(centers,boxSize);
    finished = true;
else
    finished = false;
end
while ~finished
    % ask the user to draw a polygon around the region of interest
    polygon = polymesh('Create a polygon around the region of interest');
    centers = getBoxCenters(polygon,spacing,imsize); % get the centers in the polygon
    %bxs = plotBoxes(centers,boxSize);
    % removed exclusion for later
    %     h1 = plot(x(in),y(in),'.b');
    %     polyInner = polymesh('Create a polygon around the region to exclude');
    %     yv = polyInner(:,2);
    %     xv = polyInner(:,1);
    %     out = inpolygon(x,y,xv,yv);
    %     h2 = plot(x(out),y(out),'.r');
    %     h3 = plot(x(in & ~out),y(in & ~out),'.g');
    %     points = [y(in & ~out) x(in & ~out) (1:sum(in & ~out))'];
    % nPoints = sum(in & ~out);
    
    choice = questdlg('Accept or repeat?','Confirm Selection','Accept','Repeat','Accept');
    % Handle response
    switch choice
        case 'Accept'
            finished = true;
            close(gcf)
        case 'Repeat'
            finished = false;
            delete(h1); delete(h2); delete(h3);
    end
end
regions = makeBoxRegions(centers,boxSize);
end

function [regions,centers] = meshPoints(boxSize)
[x,y] = getpts;
centers = round([y x]);
regions = makeBoxRegions(centers,boxSize);
end

function centers = getBoxCenters(polygon,spacing,imsize)
xv = polygon(:,1); % get the x points of the polygon
yv = polygon(:,2); % get the y points of the polygon
% create a grid of x and y points over the whole image according to
% spacing
x = 1:spacing(2):imsize(2); % create x positions
y = 1:spacing(1):imsize(1); % create y positions
[x,y] = ndgrid(x,y);
x = x(:); y = y(:); % linearize x and y
in = inpolygon(x,y,xv,yv); % check which are within the polygon
centers = [y(in) x(in)]; % arrange the result
end

function bxs = showBoxes(centers,boxSize)
% show boxes
% create a vertices array around 0
dist = [(boxSize(1)-1)/2 (boxSize(2)-1)/2];
v = [-dist(1)  dist(1)  dist(1) -dist(1)
    -dist(2) -dist(2)  dist(2)  dist(2)];
% preallocate arrays
nv = zeros(4,2,nPoints); % preallocate the vertices
bxs = zeros(1,nPoints);
% draw boxes around each point
nPoints = size(centers,2); % get the number of points
for i = 1:nPoints
    nv(:,:,i) = v'+repmat(fliplr(centers(i,:)),4,1);
    bxs(i) = fill(nv(:,1,i),nv(:,2,i),'r','FaceColor','none');
end
end

function poly = polymesh(user_message)
title(user_message)
new_poly = impoly;
poly = wait(new_poly);
delete(new_poly)
end

function [regions,centers] = meshCenterBoxes(spacing,sz,boxSize,percent)
ndims = numel(percent);
if ndims == 3
    is3d = true;
else is3d = false;
end
center = sz(1:ndims)/2; % find the center of the image
width = sz(1:ndims).*real(percent); % find the width of the region we're going to track
edges = round([center-width/2 ; center+width/2]); % find the box we're going to rack
xv = edges(1,1):spacing(1):edges(2,1); % get the x center values
yv = edges(1,2):spacing(2):edges(2,2); % get the y center values
if is3d
    zv = edges(1,3):spacing(3):edges(2,3); % get the z center values
end
% adjust if the edges are less than the range
if range(edges(:,1))<spacing(1)
    xv = round(center(1));
end
if range(edges(:,2))<spacing(2)
    yv = round(center(2));
end
if is3d
    if range(edges(:,3))<spacing(3)
        zv = round(center(3));
    end
end
if is3d
    [x,y,z] = ndgrid(xv,yv,zv); % grid the center vectors
else
    [x,y] = ndgrid(xv,yv);
end
% remove centers from center
if ~isreal(percent); % we have some points to remove from center (has an imaginary part)
    isvalid = false(1,numel(x));
    r = mean((imag(percent)*sz(1:ndims)));
    for i = 1:numel(x)
        if is3d
            d = norm([[x(i) y(i) z(i)] - center]);
        else d = norm([[x(i) y(i)] - center]);
            isvalid(i) = d>r;
        end
        x(~isvalid) = [];
        y(~isvalid) = [];
        if is3d
            z(~isvalid) = [];
        end
    end
end
if is3d
    centers = [x(:) y(:) z(:) x(:) y(:) z(:)]; % make an array of the centers
else
    centers = [x(:) y(:) x(:) y(:)]; % make an array of the centers
end
nregions = size(centers,1); % find out how mnay regions we have
v = [-(boxSize-1)/2 (boxSize-1)/2]; % create a vertices array
vertices = centers+repmat(v,nregions,1); % find the vertices of each region
regions = struct(); % preallocate the regions
for n = 1:nregions % loop over each region
    c = vertices(n,:); % get the specific vertices for that region
    xv = c(1):c(ndims+1); % create x vector of points
    yv = c(2):c(ndims+2); % create y vector of points
    if is3d
        zv = c(3):c(6); % create z vector of points
        [x,y,z] = ndgrid(xv,yv,zv); % grid the points
        regions(n).x = x; regions(n).y = y; regions(n).z = z; % save the region
    else
        [x,y] = ndgrid(xv,yv); % grid the points
        regions(n).x = x; regions(n).y = y; % save the region
    end
end
centers = centers(:,1:ndims); % truncate the centers for saving
end


function [regions,centers,poly] = meshBoxesLine(spacing,boxSize,varargin)
if nargin < 3
    new_poly = impoly(gca,'Closed',false);
    poly = wait(new_poly);
    close all
else poly = varargin{1};
end
man = manifold(poly,'spacing',spacing(1),'method','pchip');
centers = [man.y ;man.x]';
regions = makeBoxRegions(centers,boxSize);
R = man.rotationMatrices;
t = man.t;
for i = 1:numel(regions)
    regions(i).R = R(:,:,i)';
    regions(i).t = t(i);
end
end
