# Author: Fabi Prezja <faprezja@fairn.fi>
# Copyright (C) 2023 Fabi Prezja
# License: MIT License (see LICENSE.txt for details)

import os
import numpy as np
from tensorflow.keras.preprocessing.image import ImageDataGenerator

def create_predict_datagen(preprocess_input_func):
    """
    Create an ImageDataGenerator instance for prediction.

    Args:
        preprocess_input_func (function): Preprocessing function for input images.

    Returns:
        ImageDataGenerator: An ImageDataGenerator instance for prediction.
    """
    return ImageDataGenerator(preprocessing_function=preprocess_input_func)

def create_predict_generator(path_to_folder, image_size, target_type, batch_size, preprocess_input_func):
    """
    Create an image generator for prediction.

    Args:
        path_to_folder (str): Path to the directory containing images.
        image_size (tuple): Tuple of (height, width) for resizing images.
        target_type (str): Type of target labels ('binary' or 'categorical').
        batch_size (int): Number of samples per gradient update.
        preprocess_input_func (function): Preprocessing function for input images.

    Returns:
        DirectoryIterator: An iterator that yields (data, labels) tuples.
    """
    predict_datagen = create_predict_datagen(preprocess_input_func)

    predict_generator = predict_datagen.flow_from_directory(
        path_to_folder,
        target_size=image_size,
        batch_size=batch_size,
        class_mode=target_type,
        color_mode='rgb',
        shuffle=False
    )

    return predict_generator

def generate_predictions(model, predict_generator):
    """
    Generate predictions using a trained model and a predict_generator.

    Args:
        model (Model): A trained Keras model.
        predict_generator (DirectoryIterator): An iterator that yields (data, labels) tuples.

    Returns:
        tuple: A tuple containing the predictions and the filenames.
    """
    predictions = model.predict(predict_generator)
    filenames = predict_generator.filenames
    return predictions, filenames

def process_predictions(class_labels, path_to_folder, predictions, filenames, target_type, verbose, sort_by=None):
    """
    Process the predictions and generate a list of dictionaries containing the results.

    Args:
        class_labels (list): A list of class labels.
        path_to_folder (str): Path to the directory containing images.
        predictions (np.array): Predictions generated by the model.
        filenames (list): A list of filenames.
        target_type (str): Type of target labels ('binary' or 'categorical').
        verbose (bool): Whether to print results.
        sort_by (str, optional): Sorting criteria for results ('confidence' or 'variance').

    Returns:
        list: A list of dictionaries containing prediction results.
    """
    results = []

    for i, prediction in enumerate(predictions):
        file_path = os.path.join(path_to_folder, filenames[i])
        file_name = filenames[i]

        if target_type == 'binary':
            positive_class_prob = prediction[0]
            negative_class_prob = 1 - positive_class_prob
            label = 1 if positive_class_prob >= 0.5 else 0
            confidence = positive_class_prob if label == 1 else negative_class_prob
            confidence_variance = np.var([positive_class_prob, negative_class_prob])
        else:
            label = np.argmax(prediction)
            confidence = prediction[label]
            confidence_variance = np.var(prediction)

        original_label = class_labels[label]

        result = {
            'predicted_label': original_label,
            'confidence': confidence,
            'confidence_variance': confidence_variance,
            'file_name': file_name,
            'file_path': file_path
        }
        results.append(result)

    if sort_by == 'confidence':
        results.sort(key=lambda x: x['confidence'], reverse=True)
    elif sort_by == 'variance':
        results.sort(key=lambda x: x['confidence_variance'], reverse=True)

    if verbose:
        for result in results:
            print(result)

    return results

def model_predict(model, preprocess_input_func, path_to_folder, class_labels, image_size, target_type, batch_size=32,
                  verbose=True, sort_by='variance'):
    """
    Predict the classes of images in a folder using a trained model.

    Args:
        model (Model): A trained Keras model.
        preprocess_input_func (function): Preprocessing function for input images.
        path_to_folder (str): Path to the directory containing images.
        class_labels (list): A list of class labels.
        image_size (tuple): Tuple of (height, width) for resizing images.
        target_type (str): Type of target labels ('binary' or 'categorical').
        batch_size (int, optional): Number of image samples per batch. Defaults to 32.
        verbose (bool, optional): Whether to print results. Defaults to True.
        sort_by (str, optional): Sorting criteria for results ('confidence' or 'variance'). Defaults to 'variance'.

    Returns:
        list: A list of dictionaries containing prediction results.
    """
    predict_generator = create_predict_generator(path_to_folder, image_size, target_type,
                                                 batch_size, preprocess_input_func)
    predictions, filenames = generate_predictions(model, predict_generator)
    results = process_predictions(class_labels, path_to_folder, predictions, filenames,
                                  target_type, verbose, sort_by)
    return results
