#' Calculates the mean estimation and calibrated residuals for individual patients
#'
#' @param df data frame containing patient data
#' @param df_normal data frame containing normal data
#' @param variable_name name of the variable for which to estimate the mean
#' 
#' @return data frame containing calibrated residuals, mean estimations, sigma, and updated columns
#'
#' @examples
#' mean_estimator_ind_patient(df, df_normal, "Flow")
#'
mean_estimator_ind_patient<- function(df,df_normal,variable_name){
  
  # make a column to save residuals
  df$calibrated_resid<-0
  
  # The algorithm will iterate over calibration of 90 days or first 180 observations
  # you can change this paramter as you wish based on your data
  iter<-180
  
  # Number of observations in the df
  kl<-nrow(df)
  
  # if number of observations in df is less than the required calibration size,
  # we will put the number of observations as the calibration size
  if (iter> kl) {
    iter<-kl
  }
  
  #Selecting the first 28 observations from the stable patients
  bank_data<- df_normal %>%group_by(ID)%>%slice(1:28)
  
  #
  bank_data$Datetime<- as.Date(bank_data$Datetime)
  
  # make two other columns in the stable patients datasets for saving residulas
  bank_data$calibrated_resid<-0
  bank_data$mean_estimation<-0
  df$mean_estimation<-0
  online_mean<-data.frame()
  patient_num = df$ID[1]
  
  # we first check the name of the variable
  if (variable_name == "Flow") {
    
    # iterating in the calibration to update patient-specific mean
    for (j in 1:(iter)) {
      a<- df[1:j,]
      
      new_df<-data.frame()
      
      new_df<- rbind(bank_data, a)
      
      new_model<- lmer(Flow ~  (1 + norm_speed |ID) + HCT 
                       , data = new_df, REML = F, 
                       control = lmerControl(optimizer ="bobyqa", calc.derivs = F))
      
      # We are going to get the prediction for values
      the_estimated_mean<- 
        coef(new_model)$ID[which( row.names(coef(new_model)$ID)==patient_num),3]* a$HCT+
        a$norm_speed *coef(new_model)$ID[which( row.names(coef(new_model)$ID)==patient_num),1]+
        coef(new_model)$ID[which( row.names(coef(new_model)$ID)==patient_num),2]
      online_mean<- rbind(online_mean, the_estimated_mean)
      
      # now getting the residuals
      df[j,"calibrated_resid"]<- as.numeric(df$Flow[j]- online_mean[j,1])
      df[j,"mean_estimation"]<-  online_mean[j,1]
    }
    
    # getting thresholds for after calibration until the next event
    if (kl>iter) {
      df[((iter+1): kl), "calibrated_resid"]<- df[((iter+1):kl), "Flow"] - online_mean[iter,1]
      df[((iter+1): kl), "mean_estimation"]<- online_mean[iter,1]
    }
    
    # obtaining the sd of measurements in the calibration
    df$sigma<-0
    for (s in 1:iter) {
      df$sigma[s]<- sqrt(var(df[1:s,"calibrated_resid"])[1])
    }
    
    # and fixing the sd after the calibration
    df$sigma[1]<-2 # this number is a initial number and can be set to any number
    if (kl>iter) {
      df[((iter+1): kl), "sigma"]<- df[iter,"sigma"]
    }
   
    # now we do the same when variable is Motor_power
  }else if(variable_name == "Motor_power"){
    for (j in 1:(iter)) {
      
      a<- df[1:j,]
      
      new_df<- rbind(bank_data, a)
      
      new_model<- lmer(Motor_power ~  (1 + norm_speed |ID) 
                       
                       , data = new_df, REML = F, 
                       control = lmerControl(optimizer ="bobyqa", calc.derivs = F))
      
      
      the_estimated_mean<- coef(new_model)$ID[which( row.names(coef(new_model)$ID)==patient_num),2]+
        a$norm_speed *coef(new_model)$ID[which( row.names(coef(new_model)$ID)==patient_num),1]
      
      
      online_mean<- rbind(online_mean, the_estimated_mean)
      
      df[j,"calibrated_resid"]<- as.numeric(df$Motor_power[j]- online_mean[j,1])
      df[j,"mean_estimation"]<-  online_mean[j,1]
      
      
    }
    df[((iter+1): kl), "calibrated_resid"]<- df[((iter+1):kl), "Motor_power"] - online_mean[iter,1]
    df[((iter+1): kl), "mean_estimation"]<- online_mean[iter,1]
    
    df$sigma<-0
    
    for (s in 1:iter) {
      
      df$sigma[s]<- sqrt(var(df[1:s,"calibrated_resid"])[1])
      
      
    }
    df$sigma[1]<-2
    df[((iter+1): kl), "sigma"]<- df[iter,"sigma"]
    
    
  }
  
  
  # making another column to notify the datasets when an update occurs.
  # this column will help us to obtain out-of-control points
  df$updated<-0
  df$updated[1]<-1
  control_limits<- data.frame(lcl = rep(0, iter), ucl = rep(0,iter), mean_estimate = rep(0, iter))
  
  
  
  return(df[, c("calibrated_resid", "mean_estimation","sigma", "updated")])
}

