##### Data #####

diet <- read.csv("DietaryData.csv")
names(diet)
rownames(diet) <- diet[,1]
dietprey <- diet[,20:70]
inverts <- read.csv("Weather_Inverts.csv")
names(inverts)
rownames(inverts) <- inverts[,1]
invertcomm <- inverts[,9:81]

##### Diet Hill diversity #####

library(vegan)
library(gridExtra)
library(SpadeR)
library(iNEXT)
library(tidyverse)

d.species.col <- read.csv("dietincidence.csv")
d.spiders.zeta <- d.species.col %>% remove_rownames %>% column_to_rownames(var = "X")

d.mat <- as.list(d.spiders.zeta)
str(d.mat)
d.out <- iNEXT(d.mat, q = c(0,1,2), datatype = "abundance")
d.out
par(mfrow=c(2,2))

d.t1 <- ggiNEXT(d.out, type = 1, facet.var = "none", color.var = "order", grey=FALSE) + 
  theme_bw(base_size = 15) + theme(legend.position = "none") +
  labs(x = "Number of detections", y = "Species diversity (Hill)")
d.t2 <- ggiNEXT(d.out, type = 2, facet.var = "none", color.var = "order", grey=FALSE) + 
  theme_bw(base_size = 15) + theme(legend.position = "none") + 
  labs(x = "Number of detections") 
d.t3 <- ggiNEXT(d.out, type = 3, facet.var = "none", color.var = "order", grey=FALSE) +
  theme_bw(base_size = 15) + theme(legend.position = "none") +
  labs(y = "Species diversity (Hill)")
grid.arrange(d.t1, d.t2, d.t3, ncol=3)

pdf("Outputs/Spider diet Hill diversity.pdf", width = 12, height = 6) 
grid.arrange(d.t1, d.t2, d.t3, ncol=3)
dev.off()

tiff(file = "Outputs/Spider diet Hill diversity.tiff", width = 24, height = 12, res = 600, units = "cm")
grid.arrange(d.t1, d.t2, d.t3, ncol=3)
dev.off()

##### Invert Hill diversity #####

i.species.col <- read.csv("invertincidence.csv")
i.spiders.zeta <- i.species.col %>% remove_rownames %>% column_to_rownames(var = "X")

i.mat <- as.list(i.spiders.zeta)
str(i.mat)
i.out <- iNEXT(i.mat, q = c(0,1,2), datatype = "abundance")
i.out
par(mfrow=c(2,2))

i.t1 <- ggiNEXT(i.out, type = 1, facet.var = "none", color.var = "order", grey=FALSE) + 
  theme_bw(base_size = 15) + theme(legend.position = "none") +
  labs(x = "Number of detections", y = "Species diversity (Hill)")
i.t2 <- ggiNEXT(i.out, type = 2, facet.var = "none", color.var = "order", grey=FALSE) + 
  theme_bw(base_size = 15) + theme(legend.position = "none") + 
  labs(x = "Number of detections") 
i.t3 <- ggiNEXT(i.out, type = 3, facet.var = "none", color.var = "order", grey=FALSE) +
  theme_bw(base_size = 15) + theme(legend.position = "none") +
  labs(y = "Species diversity (Hill)")
grid.arrange(i.t1, i.t2, i.t3, ncol=3)

pdf("Outputs/Invertebrate community Hill diversity.pdf", width = 12, height = 6) 
grid.arrange(i.t1, i.t2, i.t3, ncol=3)
dev.off()

tiff(file = "Outputs/Invertebrate community Hill diversity.tiff", width = 24, height = 12, res = 600, units = "cm")
grid.arrange(i.t1, i.t2, i.t3, ncol=3)
dev.off()


##### Weather correlogram #####

## Functions from http://www.sthda.com/english/wiki/correlation-matrix-formatting-and-visualization and http://www.sthda.com/english/wiki/visualize-correlation-matrix-using-correlogram

flattenCorrMatrix <- function(cormat, pmat) {
  ut <- upper.tri(cormat)
  data.frame(
    row = rownames(cormat)[row(cormat)[ut]],
    column = rownames(cormat)[col(cormat)[ut]],
    cor  =(cormat)[ut],
    p = pmat[ut]
  )
}

cor.mtest <- function(mat, ...) {
  mat <- as.matrix(mat)
  n <- ncol(mat)
  p.mat<- matrix(NA, n, n)
  diag(p.mat) <- 0
  for (i in 1:(n - 1)) {
    for (j in (i + 1):n) {
      tmp <- cor.test(mat[, i], mat[, j], ...)
      p.mat[i, j] <- p.mat[j, i] <- tmp$p.value
    }
  }
  colnames(p.mat) <- rownames(p.mat) <- colnames(mat)
  p.mat
}

library("Hmisc")
library("corrplot")
library("RColorBrewer")
library("dplyr")

weather <- read.csv("WeatherData.csv")
weather <- rename(weather, Day = Julian.Day, Temperature = MeanWeekTemp, Precipitation = MeanWeekPrecipitation, Dewpoint = MeanWeekDew, Wind = MeanWeekWind, Pressure = MeanWeekPressure)
p.mat <- cor.mtest(weather)
head(p.mat[, 1:5])
palo <- viridisLite::viridis(100)

weathcor <- cor(weather, method = "spearman")
weathres <- rcorr(as.matrix(weather), type = "spearman")  ####
weathres
weathp.mat <- cor.mtest(weather, conf.level=.95)
weathp.mat
weathcorplot <- corrplot(weathcor, order ="alphabet", type = "lower", tl.col="black", tl.srt=45, #addCoef.col="black", 
                         col=palo, p.mat=weathp.mat, sig.level=0.05)
WalesCorMat<-flattenCorrMatrix(weathres$r, weathres$P)
write.table(WalesCorMat, file = "Outputs/WeatherCorrMatrix.csv", sep = ";", quote = FALSE, row.names = T)
WalesSigTable <- symnum(weathcor, abbr.colnames = FALSE)
write.table(WalesSigTable, file = "Outputs/WeatherCorrSigTable.csv", sep = ";", quote = FALSE, row.names = T)
weathcorplot

tiff(file = "Outputs/Weather Correlogram.tiff", width = 20, height = 20, res = 600, units = "cm")
corrplot(weathcor, order ="alphabet", type = "lower", tl.col="black", tl.srt=45,# addCoef.col="black", 
         col=palo, p.mat=weathp.mat, sig.level=0.05)
dev.off()



##### Diet MGLMs #####

library(mvabund)
options(max.print = 9999)

mvdiet <- mvabund(diet[,20:70])
meanvar.plot(mvdiet)

dietmT<-manyglm(mvdiet ~ Julian.Day + Genus + MeanWeekDaylength + Julian.Day:MeanWeekDaylength +
                  Genus:Julian.Day + Genus:MeanWeekDaylength
                , data=diet, family="binomial(cloglog")
plot(dietmT)
dietmT_ano <- anova(dietmT, p.uni="adjusted", resamp="montecarlo")
dietmT_ano

dietmWin <-manyglm(mvdiet ~ MeanWeekTemp + MeanWeekPrecipitation + MeanWeekDew + MeanWeekWind + MeanWeekPressure +
                     MeanWeekTemp:MeanWeekPrecipitation + MeanWeekTemp:MeanWeekDew + MeanWeekTemp:MeanWeekWind + MeanWeekTemp:MeanWeekPressure + 
                     MeanWeekPrecipitation:MeanWeekDew + MeanWeekPrecipitation:MeanWeekWind + MeanWeekPrecipitation:MeanWeekPressure + 
                     MeanWeekDew:MeanWeekWind + MeanWeekDew:MeanWeekPressure + 
                     MeanWeekWind:MeanWeekPressure 
                   , data=diet, family="binomial")
plot(dietmWin)
dietmWin_ano <- anova(dietmWin, p.uni="adjusted", resamp="montecarlo")
dietmWin_ano

dietmWjd <-manyglm(mvdiet ~ Julian.Day + Genus +
                     MeanWeekTemp + MeanWeekPrecipitation + MeanWeekDew + MeanWeekWind + MeanWeekPressure +
                     Julian.Day:MeanWeekTemp + Julian.Day:MeanWeekPrecipitation + Julian.Day:MeanWeekDew + Julian.Day:MeanWeekWind + Julian.Day:MeanWeekPressure +
                     Genus:MeanWeekTemp + Genus:MeanWeekPrecipitation + Genus:MeanWeekDew + Genus:MeanWeekWind + Genus:MeanWeekPressure
                   , data=diet, family="binomial")
plot(dietmWjd)
dietmWjd_ano <- anova(dietmWjd, p.uni="adjusted", resamp="montecarlo")
dietmWjd_ano


##### Invert MGLMs #####

library(mvabund)

mvinverts <- mvabund(inverts[,9:81])
meanvar.plot(mvinverts)

invertsmT<-manyglm(mvinverts ~ Julian.Day + MeanWeekDaylength + Julian.Day:MeanWeekDaylength
                , data=inverts, family="poisson")
plot(invertsmT)
invertsmT_ano <- anova(invertsmT, p.uni="adjusted", resamp="montecarlo")
invertsmT_ano

invertsmWin <-manyglm(mvinverts ~ MeanWeekTemp + MeanWeekPrecipitation + MeanWeekDew + MeanWeekWind + MeanWeekPressure +
                        MeanWeekTemp:MeanWeekPrecipitation + MeanWeekTemp:MeanWeekDew + MeanWeekTemp:MeanWeekWind + MeanWeekTemp:MeanWeekPressure + 
                        MeanWeekPrecipitation:MeanWeekDew + MeanWeekPrecipitation:MeanWeekWind + MeanWeekPrecipitation:MeanWeekPressure + 
                        MeanWeekDew:MeanWeekWind + MeanWeekDew:MeanWeekPressure + 
                        MeanWeekWind:MeanWeekPressure 
                      , data=inverts, family="poisson")
plot(invertsmWin)
invertsmWin_ano <- anova(invertsmWin, p.uni="adjusted", resamp="montecarlo")
invertsmWin_ano

invertsmWjd <-manyglm(mvinverts ~ Julian.Day + 
                        MeanWeekTemp + MeanWeekPrecipitation + MeanWeekDew + MeanWeekWind + MeanWeekPressure +
                        Julian.Day:MeanWeekTemp + Julian.Day:MeanWeekPrecipitation + Julian.Day:MeanWeekDew + Julian.Day:MeanWeekWind + Julian.Day:MeanWeekPressure
                      , data=inverts, family="poisson")
plot(invertsmWjd)
invertsmWjd_ano <- anova(invertsmWjd, p.uni="adjusted", resamp="montecarlo")
invertsmWjd_ano

##### Diet MDS #####

library("devtools")
library("vegan")
library("ggplot2")
library("ggthemes")
library("RColorBrewer")
library("viridis")
library("metR")

dietnmds <- read.csv("DietaryData_NMDS.csv")

names(dietnmds)

rownames(dietnmds) <- dietnmds[,1]

dietpreynmds <- dietnmds[,20:70]

diet.mds <- metaMDS(comm = dietpreynmds, distance = "jaccard", trymax=999, k=2, trace = FALSE, autotransform = FALSE)

par(mfrow=c(1,1))
plot(diet.mds$points); text(diet.mds, row.names(diet.mds))

#identify(diet.mds$points)

diet.mds$stress # Lower the better, great <0.1, ideally <0.2. If >0.2, include additional dimension
#consider transforming matrix pre-MDS to reduce the influence of extreme values


##### Diet MDS - Calendar day ordisurf #####

calendardiet <- ordisurf(diet.mds, dietnmds$Julian.Day, main="", col="forestgreen")

head(calendardiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$Julian.Day <- dietnmds$Julian.Day
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

calendardiet.contour.vals <- extract.xyz(obj = calendardiet)
head(calendardiet.contour.vals)

calendar.surf <- ggplot(data=calendardiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "A", name = "Day") + ggtitle("Day")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour() 
calendar.surf

pdf("Outputs/Diet Calendar Surf Plot.pdf", width = 12, height = 10) 
calendar.surf
dev.off()

tiff(file = "Outputs/Diet calendar Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
calendar.surf
dev.off()

calendar.surf.sp <- calendar.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

calendar.surf.sp

pdf("Outputs/Diet calendar Surf Plot Species.pdf", width = 12, height = 10) 
calendar.surf.sp
dev.off()

tiff(file = "Outputs/Diet calendar Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
calendar.surf.sp
dev.off()


##### Diet MDS - Daylength ordisurf #####

daylengthdiet <- ordisurf(diet.mds, dietnmds$MeanWeekDaylength, main="", col="forestgreen")

head(daylengthdiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekDaylength <- dietnmds$MeanWeekDaylength
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

daylengthdiet.contour.vals <- extract.xyz(obj = daylengthdiet)
head(daylengthdiet.contour.vals)

daylength.surf <- ggplot(data=daylengthdiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(name = "Day length (min)") + ggtitle("Day length")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

daylength.surf

pdf("Outputs/Diet Daylength Surf Plot.pdf", width = 12, height = 10) 
daylength.surf
dev.off()

tiff(file = "Outputs/Diet Daylength Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
daylength.surf
dev.off()

daylength.surf.sp <- daylength.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

daylength.surf.sp

pdf("Outputs/Diet Daylength Surf Plot Species.pdf", width = 12, height = 10) 
daylength.surf.sp
dev.off()

tiff(file = "Outputs/Diet Daylength Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
daylength.surf.sp
dev.off()

##### Diet MDS - Temperature ordisurf #####

tempdiet <- ordisurf(diet.mds, dietnmds$MeanWeekTemp, main="", col="forestgreen")

head(tempdiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekTemp <- dietnmds$MeanWeekTemp
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

tempdiet.contour.vals <- extract.xyz(obj = tempdiet)
head(tempdiet.contour.vals)

temp.surf <- ggplot(data=tempdiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "B", name = "Temperature (°C)") + ggtitle("Temperature")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

temp.surf

pdf("Outputs/Diet Temp Surf Plot.pdf", width = 12, height = 10) 
temp.surf
dev.off()

tiff(file = "Outputs/Diet Temp Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
temp.surf
dev.off()

temp.surf.sp <- temp.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

temp.surf.sp

pdf("Outputs/Diet Temp Surf Plot Species.pdf", width = 12, height = 10) 
temp.surf.sp
dev.off()

tiff(file = "Outputs/Diet Temp Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
temp.surf.sp
dev.off()

##### Diet MDS - Precipitation ordisurf #####

precipdiet <- ordisurf(diet.mds, dietnmds$MeanWeekPrecipitation, main="", col="forestgreen")

head(precipdiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekPrecipitation <- dietnmds$MeanWeekPrecipitation
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

precipdiet.contour.vals <- extract.xyz(obj = precipdiet)
head(precipdiet.contour.vals)

precip.surf <- ggplot(data=precipdiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "C", name = "Precipitation (mm)") + ggtitle("Precipitation")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

precip.surf

pdf("Outputs/Diet Precip Surf Plot.pdf", width = 12, height = 10) 
precip.surf
dev.off()

tiff(file = "Outputs/Diet Precip Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
precip.surf
dev.off()

precip.surf.sp <- precip.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

precip.surf.sp

pdf("Outputs/Diet Precip Surf Plot Species.pdf", width = 12, height = 10) 
precip.surf.sp
dev.off()

tiff(file = "Outputs/Diet Precip Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
precip.surf.sp
dev.off()

##### Diet MDS - Wind ordisurf #####

winddiet <- ordisurf(diet.mds, dietnmds$MeanWeekWind, main="", col="forestgreen")

head(winddiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekWind <- dietnmds$MeanWeekWind
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

winddiet.contour.vals <- extract.xyz(obj = winddiet)
head(winddiet.contour.vals)

wind.surf <- ggplot(data=winddiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "E", name = "Wind (km/h)") + ggtitle("Wind")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

wind.surf

pdf("Outputs/Diet Wind Surf Plot.pdf", width = 12, height = 10) 
wind.surf
dev.off()

tiff(file = "Outputs/Diet Wind Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
wind.surf
dev.off()

wind.surf.sp <- wind.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

wind.surf.sp

pdf("Outputs/Diet Wind Surf Plot Species.pdf", width = 12, height = 10) 
wind.surf.sp
dev.off()

tiff(file = "Outputs/Diet Wind Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
wind.surf.sp
dev.off()

##### Diet MDS - Pressure ordisurf #####

pressdiet <- ordisurf(diet.mds, dietnmds$MeanWeekPressure, main="", col="forestgreen")

head(pressdiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekPressure <- dietnmds$MeanWeekPressure
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

pressdiet.contour.vals <- extract.xyz(obj = pressdiet)
head(pressdiet.contour.vals)

press.surf <- ggplot(data=pressdiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "F", name = "Pressure (hPa)") + ggtitle("Pressure")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

press.surf

pdf("Outputs/Diet Pressure Surf Plot.pdf", width = 12, height = 10) 
press.surf
dev.off()

tiff(file = "Outputs/Diet Pressure Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
press.surf
dev.off()

press.surf.sp <- press.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

press.surf.sp

pdf("Outputs/Diet Pressure Surf Plot Species.pdf", width = 12, height = 10) 
press.surf.sp
dev.off()

tiff(file = "Outputs/Diet Pressure Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
press.surf.sp
dev.off()

##### Diet MDS - Dewpoint ordisurf #####

dewdiet <- ordisurf(diet.mds, dietnmds$MeanWeekDew, main="", col="forestgreen")

head(dewdiet)

species.scores <- as.data.frame(scores(diet.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(diet.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekDew <- dietnmds$MeanWeekDew
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

dewdiet.contour.vals <- extract.xyz(obj = dewdiet)
head(dewdiet.contour.vals)

dew.surf <- ggplot(data=dewdiet.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "G", name = "Dewpoint (°C)") + ggtitle("Dewpoint")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

dew.surf

pdf("Outputs/Diet Dewpoint Surf Plot.pdf", width = 12, height = 10) 
dew.surf
dev.off()

tiff(file = "Outputs/Diet Dewpoint Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
dew.surf
dev.off()

dew.surf.sp <- dew.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

dew.surf.sp

pdf("Outputs/Diet Dewpoint Surf Plot Species.pdf", width = 12, height = 10) 
dew.surf.sp
dev.off()

tiff(file = "Outputs/Diet Dewpoint Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
dew.surf.sp
dev.off()

##### Diet MDS - combined #####

library("gridExtra")
library("egg")

dietmegasurfplot <- egg::ggarrange(calendar.surf, daylength.surf, temp.surf, precip.surf, wind.surf, press.surf, dew.surf, ncol=2, nrow=4)

pdf("Outputs/Diet Meterological diet NMDS matrix.pdf", width = 19, height = 26) 
dietmegasurfplot
dev.off()

tiff(file = "Outputs/Diet Meterological diet NMDS matrix.tiff", width = 38, height = 52, res = 600, units = "cm")
dietmegasurfplot
dev.off()

dietmegasurfplot.sp <- egg::ggarrange(calendar.surf.sp, daylength.surf.sp, temp.surf.sp, precip.surf.sp, wind.surf.sp, press.surf.sp, dew.surf.sp, ncol=2, nrow=4)

pdf("Outputs/Diet Meterological diet NMDS matrix.species.pdf", width = 19, height = 26) 
dietmegasurfplot.sp
dev.off()

tiff(file = "Outputs/Diet Meterological diet NMDS matrix.species.tiff", width = 38, height = 52, res = 600, units = "cm")
dietmegasurfplot.sp
dev.off()


##### Invert MDS #####

library("devtools")
library("vegan")
library("ggplot2")
library("ggthemes")
library("RColorBrewer")
library("viridis")

invnmds <- read.csv("Weather_Inverts.csv")

names(inverts)

rownames(inverts) <- inverts[,1]

invertcomm <- invnmds[,9:81]

inv.mds <- metaMDS(comm = invertcomm, distance = "bray", trymax=999, k=2, trace = FALSE, autotransform = FALSE)

par(mfrow=c(1,1))
plot(inv.mds$points); text(inv.mds, row.names(inv.mds))

#identify(diet.mds$points)

inv.mds$stress # Lower the better, great <0.1, ideally <0.2. If >0.2, include additional dimension
#consider transforming matrix pre-MDS to reduce the influence of extreme values


##### Invert MDS - calendar day ordisurf #####

calendarinv <- ordisurf(inv.mds, invnmds$Julian.Day, main="", col="forestgreen")

head(calendarinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$Julian.Day <- invnmds$Julian.Day
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

calendarinv.contour.vals <- extract.xyz(obj = calendarinv)
head(calendarinv.contour.vals)

calendar.surf <- ggplot(data=calendarinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "A", name = "Day") + ggtitle("Day")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

calendar.surf

pdf("Outputs/Invert calendar Surf Plot.pdf", width = 12, height = 10) 
calendar.surf
dev.off()

tiff(file = "Outputs/Invert calendar Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
calendar.surf
dev.off()

calendar.surf.sp <- calendar.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

calendar.surf.sp

pdf("Outputs/Invert calendar Surf Plot Species.pdf", width = 12, height = 10) 
calendar.surf.sp
dev.off()

tiff(file = "Outputs/Invert calendar Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
calendar.surf.sp
dev.off()


##### Invert MDS - Daylength ordisurf #####

daylengthinv <- ordisurf(inv.mds, invnmds$MeanWeekDaylength, main="", col="forestgreen")

head(daylengthinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekDaylength <- invnmds$MeanWeekDaylength
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

daylengthinv.contour.vals <- extract.xyz(obj = daylengthinv)
head(daylengthinv.contour.vals)

daylength.surf <- ggplot(data=daylengthinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(name = "Day length (min)") + ggtitle("Day length")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

daylength.surf

pdf("Outputs/Invert Daylength Surf Plot.pdf", width = 12, height = 10) 
daylength.surf
dev.off()

tiff(file = "Outputs/Invert Daylength Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
daylength.surf
dev.off()

daylength.surf.sp <- daylength.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

daylength.surf.sp

pdf("Outputs/Invert Daylength Surf Plot Species.pdf", width = 12, height = 10) 
daylength.surf.sp
dev.off()

tiff(file = "Outputs/Invert Daylength Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
daylength.surf.sp
dev.off()

##### Invert MDS - Temperature ordisurf #####

tempinv <- ordisurf(inv.mds, invnmds$MeanWeekTemp, main="", col="forestgreen")

head(tempinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekTemp <- invnmds$MeanWeekTemp
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

tempinv.contour.vals <- extract.xyz(obj = tempinv)
head(tempinv.contour.vals)

temp.surf <- ggplot(data=tempinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "B", name = "Temperature (°C)") + ggtitle("Temperature")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

temp.surf

pdf("Outputs/Invert Temp Surf Plot.pdf", width = 12, height = 10) 
temp.surf
dev.off()

tiff(file = "Outputs/Invert Temp Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
temp.surf
dev.off()

temp.surf.sp <- temp.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

temp.surf.sp

pdf("Outputs/Invert Temp Surf Plot Species.pdf", width = 12, height = 10) 
temp.surf.sp
dev.off()

tiff(file = "Outputs/Invert Temp Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
temp.surf.sp
dev.off()

##### Invert MDS - Precipitation ordisurf #####

precipinv <- ordisurf(inv.mds, invnmds$MeanWeekPrecipitation, main="", col="forestgreen")

head(precipinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekPrecipitation <- invnmds$MeanWeekPrecipitation
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

precipinv.contour.vals <- extract.xyz(obj = precipinv)
head(precipinv.contour.vals)

precip.surf <- ggplot(data=precipinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "C", name = "Precipitation (mm)") + ggtitle("Precipitation")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

precip.surf

pdf("Outputs/Invert Precip Surf Plot.pdf", width = 12, height = 10) 
precip.surf
dev.off()

tiff(file = "Outputs/Invert Precip Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
precip.surf
dev.off()

precip.surf.sp <- precip.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

precip.surf.sp

pdf("Outputs/Invert Precip Surf Plot Species.pdf", width = 12, height = 10) 
precip.surf.sp
dev.off()

tiff(file = "Outputs/Invert Precip Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
precip.surf.sp
dev.off()

##### Invert MDS - Wind ordisurf #####

windinv <- ordisurf(inv.mds, invnmds$MeanWeekWind, main="", col="forestgreen")

head(windinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekWind <- invnmds$MeanWeekWind
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

windinv.contour.vals <- extract.xyz(obj = windinv)
head(windinv.contour.vals)

wind.surf <- ggplot(data=windinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "E", name = "Wind (km/h)") + ggtitle("Wind")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

wind.surf

pdf("Outputs/Invert Wind Surf Plot.pdf", width = 12, height = 10) 
wind.surf
dev.off()

tiff(file = "Outputs/Invert Wind Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
wind.surf
dev.off()

wind.surf.sp <- wind.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

wind.surf.sp

pdf("Outputs/Invert Wind Surf Plot Species.pdf", width = 12, height = 10) 
wind.surf.sp
dev.off()

tiff(file = "Outputs/Invert Wind Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
wind.surf.sp
dev.off()

##### Invert MDS - Pressure ordisurf #####

pressinv <- ordisurf(inv.mds, invnmds$MeanWeekPressure, main="", col="forestgreen")

head(pressinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekPressure <- invnmds$MeanWeekPressure
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

pressinv.contour.vals <- extract.xyz(obj = pressinv)
head(pressinv.contour.vals)

press.surf <- ggplot(data=pressinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "F", name = "Pressure (hPa)") + ggtitle("Pressure")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

press.surf

pdf("Outputs/Invert Pressure Surf Plot.pdf", width = 12, height = 10) 
press.surf
dev.off()

tiff(file = "Outputs/Invert Pressure Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
press.surf
dev.off()

press.surf.sp <- press.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

press.surf.sp

pdf("Outputs/Invert Pressure Surf Plot Species.pdf", width = 12, height = 10) 
press.surf.sp
dev.off()

tiff(file = "Outputs/Invert Pressure Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
press.surf.sp
dev.off()

##### Invert MDS - Dewpoint ordisurf #####

dewinv <- ordisurf(inv.mds, invnmds$MeanWeekDew, main="", col="forestgreen")

head(dewinv)

species.scores <- as.data.frame(scores(inv.mds, "species"))
species.scores$species <- rownames(species.scores)
names(species.scores)[c(1, 2)] <- c("x", "y")
species.scores$z <- NA

data.scores <- as.data.frame(scores(inv.mds, "sites"))
data.scores$site <- rownames(data.scores)
data.scores$MeanWeekDew <- invnmds$MeanWeekDew
head(data.scores)

head(species.scores)

extract.xyz <- function(obj) {
  xy <- expand.grid(x = obj$grid$x, y = obj$grid$y)
  xyz <- cbind(xy, c(obj$grid$z))
  names(xyz) <- c("x", "y", "z")
  return(xyz)
}

dewinv.contour.vals <- extract.xyz(obj = dewinv)
head(dewinv.contour.vals)

dew.surf <- ggplot(data=dewinv.contour.vals, aes(x,y,z=z)) + 
  geom_contour(aes(colour = ..level..), alpha = 1, size = 1.5) + scale_colour_viridis(option = "G", name = "Dewpoint (°C)") + ggtitle("Dewpoint")+
  geom_point(data=data.scores, aes(x=NMDS1, y=NMDS2), inherit.aes = FALSE) + 
  labs(x = "NMDS1", y = "NMDS2") + theme_bw()+
  theme(legend.position = 'right', panel.border = element_rect(fill = NA),axis.text=element_text(size=14), axis.title=element_text(size=18,face="bold"), plot.title=element_text(size=26,face="bold", hjust=0.5))+
  geom_label_contour()

dew.surf

pdf("Outputs/Invert Dewpoint Surf Plot.pdf", width = 12, height = 10) 
dew.surf
dev.off()

tiff(file = "Outputs/Invert Dewpoint Surf Plot.tiff", width = 24, height = 20, res = 600, units = "cm")
dew.surf
dev.off()

dew.surf.sp <- dew.surf +
  geom_text(data=species.scores,aes(x=x,y=y,label=species), color="black", size=3,alpha=0.5, angle=0)

dew.surf.sp

pdf("Outputs/Invert Dewpoint Surf Plot Species.pdf", width = 12, height = 10) 
dew.surf.sp
dev.off()

tiff(file = "Outputs/Invert Dewpoint Surf Plot Species.tiff", width = 24, height = 20, res = 600, units = "cm")
dew.surf.sp
dev.off()

##### Invert MDS - combined #####

invertmegasurfplot <- egg::ggarrange(calendar.surf, daylength.surf, temp.surf, precip.surf, wind.surf, press.surf, dew.surf, ncol = 2, nrow = 4)
invertmegasurfplot

pdf("Outputs/Invert Meterological inv NMDS matrix.pdf", width = 19, height = 26) 
invertmegasurfplot.sp
dev.off()

tiff(file = "Outputs/Invert Meterological inv NMDS matrix.tiff", width = 38, height = 52, res = 600, units = "cm")
invertmegasurfplot.sp
dev.off()

invertmegasurfplot.sp <- egg::ggarrange(calendar.surf.sp, daylength.surf.sp, temp.surf.sp, precip.surf.sp, wind.surf.sp, press.surf.sp, dew.surf.sp, ncol = 2, nrow = 4)
invertmegasurfplot.sp

pdf("Outputs/Invert Meterological inv NMDS matrix.species.pdf", width = 19, height = 26) 
invertmegasurfplot.sp
dev.off()

tiff(file = "Outputs/Invert Meterological inv NMDS matrix.species.tiff", width = 38, height = 52, res = 600, units = "cm")
invertmegasurfplot.sp
dev.off()


##### Webs #####

library("mvabund")
library("ggplot2")
library("RColorBrewer")

webs <- read.csv("DietaryData_webs.csv")

#webs <- droplevels(webs[!webs$Genus == 'Microlinyphia',])
#webs <- droplevels(webs[!webs$Genus == 'Erigone',])

mvweb <- mvabund(webs[,4:5])

meanvar.plot(mvweb)

webm1<-manylm(log(mvweb+1) ~ Julian.Day + Genus +
                MeanWeekTemp + MeanWeekPrecipitation + MeanWeekDew + MeanWeekWind + MeanWeekPressure +
                Genus:Julian.Day + Genus:MeanWeekTemp + Genus:MeanWeekPrecipitation + Genus:MeanWeekDew + Genus:MeanWeekWind + Genus:MeanWeekPressure +
                Julian.Day:MeanWeekTemp + Julian.Day:MeanWeekPrecipitation + Julian.Day:MeanWeekDew + Julian.Day:MeanWeekWind + Julian.Day:MeanWeekPressure
              , data=webs)

options(max.print = 9999)

plot(webm1)

webm1_ano <- anova(webm1, p.uni="adjusted")

webm1_ano

##### Webs vs. meteorology figure #####

weatherclusterpal <- c("cyan", "green","#FF6666","blue","magenta")

webvweather <- ggplot(data=webs, aes(x = Web.Area, y = Web.Height))+
                      geom_point(pch = 21, size = 2, alpha = 0.5, aes(fill = Weather.cluster))+
                      scale_fill_manual(name = "Weather Cluster", values=weatherclusterpal)+
                      labs(x = expression(paste('Web Area cm'^2)), y = 'Web Height cm') + theme_bw() + theme(axis.text = element_text(size = 8), axis.title = element_text(size = 10), legend.title = element_text(size = 10), legend.text = element_text(size = 8), legend.key.size = unit(0.5, 'cm'))

webvweather

tiff(file = "Outputs/Webs by weather cluster.tiff", width = 20, height = 10, res = 600, units = "cm")
webvweather
dev.off()
 
##### Meteorology cluster method determination #####

weather <- read.csv("WeatherData.csv")
rownames(weather) <- weather[,1]
weather_sample <- weather$Julian.Day
weather <- weather[2:6]
summary(weather)


weather_sc <- as.data.frame(scale(weather))
summary(weather_sc)

weatherdist<- dist(weather_sc, method = "euclidean")

# average

weathertreeAVG <- hclust(weatherdist, method = "average")
plot(weathertreeAVG, main="")

x <- c(5:24)
for (i in x) {
  weathercut_avg <- cutree(weathertreeAVG, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_avg, method= 'euclidean') 
  print(weatherdunn)
}

plot(weathertreeAVG, main="")
rect.hclust(weathertreeAVG, k = 7, border = 2:28)

weathercut_avg7 <- cutree(weathertreeAVG, k = 7)
weatherdunn_avg7 <- dunn(distance= weatherdist, clusters = weathercut_avg7, method= 'euclidean') 
weatherdunn_avg7


# single

weathertreeSIN <- hclust(weatherdist, method = "single")
plot(weathertreeSIN, main="")

x <- c(5:24)
for (i in x) {
  weathercut_sin <- cutree(weathertreeSIN, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_sin, method= 'euclidean') 
  print(weatherdunn)
}

rect.hclust(weathertreeSIN, k = 6, border = 2:28)


weathercut_sin6 <- cutree(weathertreeSIN, k = 6)
weatherdunn_sin6 <- dunn(distance= weatherdist, clusters = weathercut_sin6, method= 'euclidean') 
weatherdunn_sin6


# complete

weathertreeCOM <- hclust(weatherdist, method = "complete")
plot(weathertreeCOM, main="")

x <- c(5:24)
for (i in x) {
  weathercut_com <- cutree(weathertreeCOM, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_com, method= 'euclidean') 
  print(weatherdunn)
}

rect.hclust(weathertreeCOM, k = 5, border = 2:28)

weathercut_com5 <- cutree(weathertreeCOM, k = 5)
weatherdunn_com5 <- dunn(distance= weatherdist, clusters = weathercut_com5, method= 'euclidean') 
weatherdunn_com5


# mcquitty

weathertreeMCQ <- hclust(weatherdist, "mcquitty")
plot(weathertreeMCQ, main="")

x <- c(5:24)
for (i in x) {
  weathercut_mcq <- cutree(weathertreeMCQ, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_mcq, method= 'euclidean') 
  print(weatherdunn)
}

rect.hclust(weathertreeMCQ, k = 5, border = 2:28)

weathercut_mcq5 <- cutree(weathertreeMCQ, k = 5)
weatherdunn_mcq5 <- dunn(distance= weatherdist, clusters = weathercut_mcq5, method= 'euclidean') 
weatherdunn_mcq5


# median

weathertreeMED <- hclust(weatherdist, "median")
plot(weathertreeMED, main="")

x <- c(5:24)
for (i in x) {
  weathercut_med <- cutree(weathertreeMED, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_med, method= 'euclidean') 
  print(weatherdunn)
}

rect.hclust(weathertreeMED, k = 14, border = 2:28)

weathercut_med14 <- cutree(weathertreeMED, k = 14)
weatherdunn_med14 <- dunn(distance= weatherdist, clusters = weathercut_med14, method= 'euclidean') 
weatherdunn_med14


# centroid

weathertreeCEN <- hclust(weatherdist, "centroid")
plot(weathertreeCEN, main="")

x <- c(5:24)
for (i in x) {
  weathercut_cen <- cutree(weathertreeCEN, k = i )
  weatherdunn <- dunn(distance= weatherdist, clusters = weathercut_cen, method= 'euclidean') 
  print(weatherdunn)
}

rect.hclust(weathertreeCEN, k = 6, border = 2:28)

weathercut_cen6 <- cutree(weathertreeAVG, k = 6)
weatherdunn_cen6 <- dunn(distance= weatherdist, clusters = weathercut_cen6, method= 'euclidean') 
weatherdunn_cen6




##### Meteorology clustering #####

weathertreeCOM <- hclust(weatherdist, method = "complete")
plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 5, border = 2:28)

weathercut_com5 <- cutree(weathertreeCOM, k = 5)
weatherdunn_com5 <- dunn(distance= weatherdist, clusters = weathercut_com5, method= 'euclidean') 
weatherdunn_com5

weather_cl <- mutate(weather, cluster = weathercut_com5)
count(weather_cl$cluster)

ggplot(weather_cl, aes(x=MeanWeekTemp, y = MeanWeekPrecipitation, color = factor(cluster))) + geom_point()

WeatherClusters <- table(weather_cl$cluster,weather_sample)

write.csv(WeatherClusters, "WeatherClusters.csv")

plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 5, border = 2:28)
abline(h = 3.7, col = 'red')

pdf("Outputs/Weather dendrogram.pdf", width = 12, height = 8) 
plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 5, border = 2:28)
dev.off()

hclustfunc <- function(x) hclust(x, method="complete")
distfunc <- function(x) dist(x,method="euclidean")
d <- distfunc(weather_sc)
fit <- hclustfunc(d)
clusters <- cutree(fit, h=3.7)
nofclust.height <-  length(unique(as.vector(clusters)));

cl.row <- hclustfunc(distfunc(weather_sc))
cl.col <- hclustfunc(distfunc(t(weather_sc)))

hmcols <- rev(viridis(2750))
selcol <- colorRampPalette(brewer.pal(12,"Set3"))
selcol2 <- colorRampPalette(brewer.pal(8,"Accent"))
clustcol.height = selcol2(nofclust.height);

heatmap.2(as.matrix(weather_sc), 
          trace='none', 
          dendrogram='both', 
          key=F,
          Colv=T, 
          scale='row',
          hclust=hclustfunc, distfun=distfunc, col=hmcols,
          symbreak=T,
          margins=c(7,10), keysize=0.1,
          lwid=c(5,0.5,3), lhei=c(0.05,0.5),
          lmat=rbind(c(5,0,4),c(3,1,2)),
          labRow=rownames(weather_sc),
          RowSideColors=clustcol.height[clusters], cexRow = 0.8, cexCol = 1.5)

pdf("Outputs/Weather dendroheatmap.pdf", width = 8, height = 12) 
heatmap.2(as.matrix(weather_sc), 
          trace='none', 
          dendrogram='both', 
          key=F,
          Colv=T, 
          scale='row',
          hclust=hclustfunc, distfun=distfunc, col=hmcols,
          symbreak=T,
          margins=c(7,10), keysize=0.1,
          lwid=c(5,0.5,3), lhei=c(0.05,0.5),
          lmat=rbind(c(5,0,4),c(3,1,2)),
          labRow=rownames(weather_sc),
          RowSideColors=clustcol.height[clusters], cexRow = 0.8, cexCol = 1.5)
dev.off()

##### 2017 weather cluster #####

weather <- read.csv("WeatherData_with2017.csv")
rownames(weather) <- weather[,1]
weather_sample <- weather$Julian.Day
weather <- weather[2:6]
summary(weather)

library("clValid")

weather_sc <- as.data.frame(scale(weather))
summary(weather_sc)

weatherdist<- dist(weather_sc, method = "euclidean")

weathertreeCOM <- hclust(weatherdist, method = "complete")
plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 6, border = 2:28)

weathercut_com6 <- cutree(weathertreeCOM, k = 6)
weatherdunn_com6 <- dunn(distance= weatherdist, clusters = weathercut_com6, method= 'euclidean') 
weatherdunn_com6

weather_cl <- mutate(weather, cluster = weathercut_com6)
#count(weather_cl$cluster)

ggplot(weather_cl, aes(x=MeanWeekTemp, y = MeanWeekPrecipitation, color = factor(cluster))) + geom_point()

WeatherClusters <- table(weather_cl$cluster,weather_sample)

write.csv(WeatherClusters, "WeatherClusters_with2017.csv")

plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 6, border = 2:28)
abline(h = 3.5, col = 'red')

pdf("Outputs/Weather dendrogram_with2017.pdf", width = 12, height = 8) 
plot(weathertreeCOM, main="")
rect.hclust(weathertreeCOM, k = 6, border = 2:28)
dev.off()

hclustfunc <- function(x) hclust(x, method="complete")
distfunc <- function(x) dist(x,method="euclidean")
d <- distfunc(weather_sc)
fit <- hclustfunc(d)
clusters <- cutree(fit, h=3.5)
nofclust.height <-  length(unique(as.vector(clusters)));

cl.row <- hclustfunc(distfunc(weather_sc))
cl.col <- hclustfunc(distfunc(t(weather_sc)))

hmcols <- rev(viridis(2750))
selcol <- colorRampPalette(brewer.pal(12,"Set3"))
selcol2 <- colorRampPalette(brewer.pal(8,"Accent"))
clustcol.height = selcol2(nofclust.height);

library("gplots")

heatmap.2(as.matrix(weather_sc), 
          trace='none', 
          dendrogram='both', 
          key=F,
          Colv=T, 
          scale='row',
          hclust=hclustfunc, distfun=distfunc, col=hmcols,
          symbreak=T,
          margins=c(7,10), keysize=0.1,
          lwid=c(5,0.5,3), lhei=c(0.05,0.5),
          lmat=rbind(c(5,0,4),c(3,1,2)),
          labRow=rownames(weather_sc),
          RowSideColors=clustcol.height[clusters], cexRow = 0.8, cexCol = 1.5)

pdf("Outputs/Weather dendroheatmap_with2017.pdf", width = 8, height = 12) 
heatmap.2(as.matrix(weather_sc), 
          trace='none', 
          dendrogram='both', 
          key=F,
          Colv=T, 
          scale='row',
          hclust=hclustfunc, distfun=distfunc, col=hmcols,
          symbreak=T,
          margins=c(7,10), keysize=0.1,
          lwid=c(5,0.5,3), lhei=c(0.05,0.5),
          lmat=rbind(c(5,0,4),c(3,1,2)),
          labRow=rownames(weather_sc),
          RowSideColors=clustcol.height[clusters], cexRow = 0.8, cexCol = 1.5)
dev.off()

library("devtools")
library("vegan")
library("ggplot2")
library("ggthemes")
library("RColorBrewer")
library("viridis")

weathernmds <- read.csv("WeatherData_with2017_withclusters.csv")
rownames(weathernmds) <- weathernmds[,1]
weathernmds <- weathernmds[2:7]
summary(weathernmds)

weather.mds <- metaMDS(comm = weathernmds[,2:6], distance = "euclidean", trymax=999, k=2, trace = FALSE, autotransform = FALSE)

par(mfrow=c(1,1))
plot(weather.mds$points); text(weather.mds, row.names(weather.mds))

weather.mds$stress

scrs <- scores(weather.mds, display = 'sites')
scrs <- cbind(as.data.frame(scrs), Cluster = weathernmds$Cluster)
cent <- aggregate(cbind(NMDS1, NMDS2) ~ Cluster, data = scrs, FUN = mean)
segs <- merge(scrs, setNames(cent, c('Cluster', 'oNMDS1', 'oNMDS2')), by = 'Cluster', sort = FALSE)

weatherdataspiplot <- ggplot(scrs, aes(x = NMDS1, y = NMDS2, fill = Cluster)) + 
  scale_fill_brewer(palette = "Dark2") + 
  scale_color_brewer(palette = "Dark2") + 
  geom_segment(data = segs, mapping = aes(xend = oNMDS1, yend = oNMDS2, color = Cluster), alpha=0.25) + 
  geom_point(data = cent, size = 5, pch = 21, alpha = 0.4) + 
  geom_point(alpha=0.25, pch = 21, aes(fill = Cluster)) + coord_fixed() + theme_bw()

weatherdataspiplot

pdf("Outputs/Weather cluster NMDS.pdf", width = 12, height = 6) 
weatherdataspiplot
dev.off()

tiff(file = "Outputs/Weather cluster NMDS.tiff", width = 24, height = 12, res = 600, units = "cm")
weatherdataspiplot
dev.off()

write.csv(cent, "Weather cluster NMDS centroids.csv")


##### Weather cluster ENNR #####

library("econullnetr")

ennr <- read.csv("ENNR_Diet_Weather.csv")

invertsennr <- read.csv("ENNR_Inverts.csv")

weather.null <- generate_null_net(ennr[,2:83], invertsennr[,2:82],
                                sims = 999, data.type = "names",
                                summary.type = "sum",
                                r.samples = invertsennr[,1],
                                c.samples = ennr[,1])

weather.null.tab <- test_interactions(weather.null)

weather.null.tab

edgelist_18 <- generate_edgelist(weather.null, signif.level = 0.95, 
                                 edge.cols = c("#2c7bb6", "#000000", "#d7191c"))

write.table(edgelist_18, "2018 Weather ENNR output - edgelist.csv")

write.csv(weather.null.tab, "WeatherENNR.csv")

#par(mfrow = c(2,3)) 
par(mfrow = c(1,1))
plot_preferences(weather.null, "1", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)

pdf("Outputs/Weather1_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null, "1", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

plot_preferences(weather.null, "2", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)

pdf("Outputs/Weather2_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null, "2", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

plot_preferences(weather.null, "3", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)

pdf("Outputs/Weather3_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null, "3", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

plot_preferences(weather.null, "4", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)

pdf("Outputs/Weather4_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null, "4", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

plot_preferences(weather.null, "5", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)

pdf("Outputs/Weather5_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null, "5", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()



weather.links <- test_interactions(weather.null, signif.level = 0.95)

weather.links

write.table(weather.links, "Weather ENNR output.csv")

export1 <- generate_edgelist(weather.null, signif.level = 0.95, 
            edge.cols = c("#2c7bb6", "#000000", "#d7191c"))

par(mfrow=c(1,1))

if (requireNamespace("igraph", quietly = TRUE)) {
  
  net.1 <- igraph::graph_from_edgelist(as.matrix(export1[, c("Resource", 
                                                             "Consumer")]),
                                       directed = TRUE)
  
  # Add in the null modelling results 
  igraph::E(net.1)$obs.str <- export1$Observed
  igraph::E(net.1)$test.res <- export1$Test
  igraph::E(net.1)$edge.cols <- export1$edge.col
  
  igraph::plot.igraph(net.1, layout = igraph::layout_in_circle, 
                      edge.color = igraph::E(net.1)$edge.cols, 
                      edge.width = sqrt(igraph::E(net.1)$obs.str))
}

pdf("Outputs/Weather Cluster ENNR Net.pdf", width = 14, height = 14) 
if (requireNamespace("igraph", quietly = TRUE)) {
  
  net.1 <- igraph::graph_from_edgelist(as.matrix(export1[, c("Resource", 
                                                             "Consumer")]),
                                       directed = TRUE)
  
  # Add in the null modelling results 
  igraph::E(net.1)$obs.str <- export1$Observed
  igraph::E(net.1)$test.res <- export1$Test
  igraph::E(net.1)$edge.cols <- export1$edge.col
  
  igraph::plot.igraph(net.1, layout = igraph::layout_in_circle, 
                      edge.color = igraph::E(net.1)$edge.cols, 
                      edge.width = sqrt(igraph::E(net.1)$obs.str))
}
dev.off()

##### 2018 Weather cluster ENNR metrics and bipartite #####

mean.abunds <- colMeans(invertsennr[,2:82])

plot_bipartite(weather.null, signif.level = 0.95, 
               edge.cols = c("#67a9cf", "#F7F7F7", "#ef8a62"),
               low.abun = mean.abunds, abuns.type = "independent", 
               low.abun.col = "black", high.abun.col = "black",
               high.lablength = 25, low.lablength = 5)

tiff(file = "Outputs/2018 Weather ENNR bipartite.tiff", width = 30, height = 18, res = 600, units = "cm")
plot_bipartite(weather.null, signif.level = 0.95, 
               edge.cols = c("#67a9cf", "#F7F7F7", "#ef8a62"),
               low.abun = mean.abunds, abuns.type = "independent", 
               low.abun.col = "black", high.abun.col = "black",
               text.rot = 90, labsize = 0.8,
               high.lablength = 25, low.lablength = 25)
dev.off()

ennrweather.species.stats <- bipartite_stats(weather.null, index.type = "specieslevel", 
                                        indices = c("normalised degree"), logbase = exp(1))
ennrweather.species.stats
write.csv(ennrweather.species.stats$normalised.degree$higher, "2018 Weather ENNR_specstats.normalised.degree.higher.csv")


##### 2018 Mega preference plot #####

library("dplyr")

ennr2018out <- read.csv("WeatherENNR.csv")

ennr2018out$Consumer <- as.factor(ennr2018out$Consumer)
ennr2018out$Consumer <- recode_factor(ennr2018out$Consumer, '1' = "WLD", '2' = "WMD", '3' = "HPR", '4' = "DWI", '5' = "HOT")

megaprefplot <- ggplot(ennr2018out, aes(x = Consumer, y = Resource))+
                      geom_point(aes(size = abs(SES), fill = Test), pch = 21, alpha = 0.8)+
                      scale_fill_manual(name = "Significance", labels = c("Stronger", "n.s.", "Weaker"), breaks = c("Stronger", "ns", "Weaker"),
                      values = c("#fc8d59", "#b2b2b2", "#7f7fff"))+
                      scale_size(name = "SES", labels = c("Low", "High"), breaks = c(4,16))+
                      theme_bw() + theme(axis.text.y = element_text(size = 6), axis.title = element_text(size = 10, face = "bold"), legend.title = element_text(size = 10, face = "bold"), legend.text = element_text(size = 8), legend.key.size = unit(0.5, 'cm')) #+ guides(size=guide_legend(title="SES", label = c("Low", "", "", "High")))

megaprefplot

tiff(file = "Outputs/Mega preference plot_obs by null.tiff", width = 13, height = 20, res = 600, units = "cm")
megaprefplot
dev.off()

##### 2018 Weather cluster OvE calculations #####

library("janitor")

ennr2018out <- read.csv("WeatherENNR.csv")

ennr2018out$Consumer <- as.factor(ennr2018out$Consumer)
ennr2018out$Consumer <- recode_factor(ennr2018out$Consumer, '1' = "WLD", '2' = "WMD", '3' = "HPR", '4' = "DWI", '5' = "HOT")


ennr2018out$OvE <- ennr2018out$Observed/ennr2018out$Null
ennr2018OvE <- ennr2018out[c(2,3,10)]
ennrweights <- ennr2018OvE %>% pivot_wider(names_from=Resource, values_from=OvE)
ennrweights <- ennrweights %>% remove_rownames %>% column_to_rownames(var = "Consumer")
ennrweights <- as.data.frame(t(ennrweights))
ennrweights$WLD <- ennrweights$WLD %>% replace(is.infinite(.), NaN) 
ennrweights$WLD <- ennrweights$WLD%>% replace(is.na(.), max(ennrweights$WLD, na.rm = TRUE))
ennrweights$WMD <- ennrweights$WMD %>% replace(is.infinite(.), NaN) 
ennrweights$WMD <- ennrweights$WMD %>% replace(is.na(.), max(ennrweights$WMD, na.rm = TRUE))
ennrweights$HPR <- ennrweights$HPR %>% replace(is.infinite(.), NaN) 
ennrweights$HPR <- ennrweights$HPR %>% replace(is.na(.), max(ennrweights$HPR, na.rm = TRUE))
ennrweights$DWI <- ennrweights$DWI %>% replace(is.infinite(.), NaN) 
ennrweights$DWI <- ennrweights$DWI %>% replace(is.na(.), max(ennrweights$DWI, na.rm = TRUE))
ennrweights$HOT <- ennrweights$HOT %>% replace(is.infinite(.), NaN) 
ennrweights$HOT <- ennrweights$HOT %>% replace(is.na(.), max(ennrweights$HOT, na.rm = TRUE))
range0.1_1 <- function(x){((x-min(x))/(max(x)-min(x))*(1-0.1)+0.1)}
ennrweights$WLD <- range0.1_1(ennrweights$WLD)
ennrweights$WMD <- range0.1_1(ennrweights$WMD)
ennrweights$HPR <- range0.1_1(ennrweights$HPR)
ennrweights$DWI <- range0.1_1(ennrweights$DWI)
ennrweights$HOT <- range0.1_1(ennrweights$HOT)

fl2017blank <- read.csv("2017ENNR_fl_blank_OvE.csv")
ennrweights <- rownames_to_column(ennrweights, "Taxon")
ennr.fl.WLD <- as.data.frame(ennrweights[c(1,2)])
fl.WLD <- merge(fl2017blank, ennr.fl.WLD, by.x = "Predator", by.y = "Taxon", all.x = TRUE)
fl.WLD$WLD <- fl.WLD$WLD %>% replace(is.na(.), 1)
names(fl.WLD)[names(fl.WLD) == "WLD"] <- "Pre-harvest"
fl.WLD$'Post-harvest' <- fl.WLD$'Pre-harvest'
ennr.fl.WMD <- as.data.frame(ennrweights[c(1,3)])
fl.WMD <- merge(fl2017blank, ennr.fl.WMD, by.x = "Predator", by.y = "Taxon", all.x = TRUE)
fl.WMD$WMD <- fl.WMD$WMD %>% replace(is.na(.), 1)
names(fl.WMD)[names(fl.WMD) == "WMD"] <- "Pre-harvest"
fl.WMD$'Post-harvest' <- fl.WMD$'Pre-harvest'
ennr.fl.HPR <- as.data.frame(ennrweights[c(1,4)])
fl.HPR <- merge(fl2017blank, ennr.fl.HPR, by.x = "Predator", by.y = "Taxon", all.x = TRUE)
fl.HPR$HPR <- fl.HPR$HPR %>% replace(is.na(.), 1)
names(fl.HPR)[names(fl.HPR) == "HPR"] <- "Pre-harvest"
fl.HPR$'Post-harvest' <- fl.HPR$'Pre-harvest'
ennr.fl.DWI <- as.data.frame(ennrweights[c(1,5)])
fl.DWI <- merge(fl2017blank, ennr.fl.DWI, by.x = "Predator", by.y = "Taxon", all.x = TRUE)
fl.DWI$DWI <- fl.DWI$DWI %>% replace(is.na(.), 1)
names(fl.DWI)[names(fl.DWI) == "DWI"] <- "Pre-harvest"
fl.DWI$'Post-harvest' <- fl.DWI$'Pre-harvest'
ennr.fl.HOT <- as.data.frame(ennrweights[c(1,6)])
fl.HOT <- merge(fl2017blank, ennr.fl.HOT, by.x = "Predator", by.y = "Taxon", all.x = TRUE)
fl.HOT$HOT <- fl.HOT$HOT %>% replace(is.na(.), 1)
names(fl.HOT)[names(fl.HOT) == "HOT"] <- "Pre-harvest"
fl.HOT$'Post-harvest' <- fl.HOT$'Pre-harvest'

tfl.HPR <- as.data.frame(t(fl.HPR))
tfl.HPR <- tfl.HPR %>% rownames_to_column("Predator") %>% row_to_names(row_number = 1)
write.csv(tfl.HPR, "2017ENNR_fl_HPR_OvE.csv")
tfl.WLD <- as.data.frame(t(fl.WLD))
tfl.WLD <- tfl.WLD %>% rownames_to_column("Predator") %>% row_to_names(row_number = 1)
write.csv(tfl.WLD, "2017ENNR_fl_WLD_OvE.csv")
tfl.WMD <- as.data.frame(t(fl.WMD))
tfl.WMD <- tfl.WMD %>% rownames_to_column("Predator") %>% row_to_names(row_number = 1)
write.csv(tfl.WMD, "2017ENNR_fl_WMD_OvE.csv")
tfl.DWI <- as.data.frame(t(fl.DWI))
tfl.DWI <- tfl.DWI %>% rownames_to_column("Predator") %>% row_to_names(row_number = 1)
write.csv(tfl.DWI, "2017ENNR_fl_DWI_OvE.csv")
tfl.HOT <- as.data.frame(t(fl.HOT))
tfl.HOT <- tfl.HOT %>% rownames_to_column("Predator") %>% row_to_names(row_number = 1)
write.csv(tfl.HOT, "2017ENNR_fl_HOT_OvE.csv")

##### 2017 Weather cluster ENNR without SES #####

library("econullnetr")
ennr17 <- read.csv("2017ENNR_Diet_Weather.csv")
invertsennr17 <- read.csv("2017ENNR_Inverts.csv")

weather.null_17 <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                                  sims = 999, data.type = "names",
                                  summary.type = "sum",
                                  r.samples = invertsennr17[,1],
                                  c.samples = ennr17[,1])

weather.null.tab_17 <- test_interactions(weather.null_17)
write.csv(weather.null.tab_17, "2017WeatherENNR_withoutSES.csv")

plot_preferences(weather.null_17, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withoutSES_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null_17, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(weather.null_17, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withoutSES_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null_17, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17 <- test_interactions(weather.null_17, signif.level = 0.95)
write.table(weather.links_17, "2017 Weather ENNR output without SES.csv")
edgelist_17 <- generate_edgelist(weather.null_17, signif.level = 0.95, 
                             edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17, "2017 Weather ENNR output without SES - edgelist.csv")

##### 2017 without SES simulations extraction #####

weather.null_17_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                            sims = 999, data.type = "names",
                            summary.type = "none",
                            r.samples = invertsennr17[,1],
                            c.samples = ennr17[,1])

null.17.rand <- weather.null_17_sim$rand.data
null.17.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17_sim$n.iterations)
ind.cons.avg.17 <- aggregate(null.17.rand[, 3:69], by = list(null.17.rand$Consumer, null.17.rand$indiv), mean)
colnames(ind.cons.avg.17)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17, "ENNR17_ENNRSimOut_withoutSES.csv")

##### 2017 HPR Weather cluster ENNR with SES ####

ennrfl17HPR <- read.csv("2017ENNR_fl_HPR_OvE.csv")
ennrfl17HPR <- ennrfl17HPR[-1]

null.1.17HPR <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                            sims = 999, data.type = "names",
                            summary.type = "sum",
                            r.samples = invertsennr17[,1],
                            c.samples = ennr17[,1],
                            r.weights = ennrfl17HPR)

weather.null.tab_17HPR <- test_interactions(null.1.17HPR)
write.csv(weather.null.tab_17HPR, "2017WeatherENNR_withSES_HPR_OvE.csv")

plot_preferences(null.1.17HPR, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withSES_ENNR_HPR_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17HPR, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(null.1.17HPR, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withSES_ENNR_HPR_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17HPR, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17HPR <- test_interactions(null.1.17HPR, signif.level = 0.95)
write.table(weather.links_17HPR, "2017 Weather ENNR output with SES input_HPR_OvE.csv")
edgelist_17HPR <- generate_edgelist(null.1.17HPR, signif.level = 0.95, 
                                    edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17HPR, "2017 Weather ENNR output with SES input - edgelist_HPR_OvE.csv")

##### 2017 HPR SES simulations extraction #####

weather.null_17HPR_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                                          sims = 999, data.type = "names",
                                          summary.type = "none",
                                          r.samples = invertsennr17[,1],
                                          c.samples = ennr17[,1],
                                          r.weights = ennrfl17HPR)

null.17HPR.rand <- weather.null_17HPR_sim$rand.data
null.17HPR.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17HPR_sim$n.iterations)
ind.cons.avg.17HPR <- aggregate(null.17HPR.rand[, 3:69], by = list(null.17HPR.rand$Consumer, null.17HPR.rand$indiv), mean)
colnames(ind.cons.avg.17HPR)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17HPR, "ENNR17HPR_ENNRSimOut_withSES_OvE.csv")

##### 2017 WMD Weather cluster ENNR with SES ####

ennrfl17WMD <- read.csv("2017ENNR_fl_WMD_OvE.csv")
ennrfl17WMD <- ennrfl17WMD[-1]

null.1.17WMD <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                                  sims = 999, data.type = "names",
                                  summary.type = "sum",
                                  r.samples = invertsennr17[,1],
                                  c.samples = ennr17[,1],
                                  r.weights = ennrfl17WMD)

weather.null.tab_17WMD <- test_interactions(null.1.17WMD)
write.csv(weather.null.tab_17WMD, "2017WeatherENNR_withSES_WMD_OvE.csv")

plot_preferences(null.1.17WMD, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withSES_ENNR_WMD_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17WMD, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(null.1.17WMD, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withSES_ENNR_WMD_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17WMD, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17WMD <- test_interactions(null.1.17WMD, signif.level = 0.95)
write.table(weather.links_17WMD, "2017 Weather ENNR output with SES input_WMD_OvE.csv")
edgelist_17WMD <- generate_edgelist(null.1.17WMD, signif.level = 0.95, 
                                    edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17WMD, "2017 Weather ENNR output with SES input - edgelist_WMD_OvE.csv")

##### 2017 WMD SES simulations extraction #####

weather.null_17WMD_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                                             sims = 999, data.type = "names",
                                             summary.type = "none",
                                             r.samples = invertsennr17[,1],
                                             c.samples = ennr17[,1],
                                             r.weights = ennrfl17WMD)

null.17WMD.rand <- weather.null_17WMD_sim$rand.data
null.17WMD.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17WMD_sim$n.iterations)
ind.cons.avg.17WMD <- aggregate(null.17WMD.rand[, 3:69], by = list(null.17WMD.rand$Consumer, null.17WMD.rand$indiv), mean)
colnames(ind.cons.avg.17WMD)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17WMD, "ENNR17WMD_ENNRSimOut_withSES_OvE.csv")

##### 2017 HOT Weather cluster ENNR with SES ####

ennrfl17HOT <- read.csv("2017ENNR_fl_HOT_OvE.csv")
ennrfl17HOT <- ennrfl17HOT[-1]

null.1.17HOT <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                                  sims = 999, data.type = "names",
                                  summary.type = "sum",
                                  r.samples = invertsennr17[,1],
                                  c.samples = ennr17[,1],
                                  r.weights = ennrfl17HOT)

weather.null.tab_17HOT <- test_interactions(null.1.17HOT)
write.csv(weather.null.tab_17HOT, "2017WeatherENNR_withSES_HOT_OvE.csv")

plot_preferences(null.1.17HOT, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withSES_ENNR_HOT_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17HOT, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(null.1.17HOT, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withSES_ENNR_HOT_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17HOT, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17HOT <- test_interactions(null.1.17HOT, signif.level = 0.95)
write.table(weather.links_17HOT, "2017 Weather ENNR output with SES input_HOT_OvE.csv")
edgelist_17HOT <- generate_edgelist(null.1.17HOT, signif.level = 0.95, 
                                    edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17HOT, "2017 Weather ENNR output with SES input - edgelist_HOT_OvE.csv")

##### 2017 HOT SES simulations extraction #####

weather.null_17HOT_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                                             sims = 999, data.type = "names",
                                             summary.type = "none",
                                             r.samples = invertsennr17[,1],
                                             c.samples = ennr17[,1],
                                             r.weights = ennrfl17HOT)

null.17HOT.rand <- weather.null_17HOT_sim$rand.data
null.17HOT.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17HOT_sim$n.iterations)
ind.cons.avg.17HOT <- aggregate(null.17HOT.rand[, 3:69], by = list(null.17HOT.rand$Consumer, null.17HOT.rand$indiv), mean)
colnames(ind.cons.avg.17HOT)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17HOT, "ENNR17HOT_ENNRSimOut_withSES_OvE.csv")

##### 2017 WLD Weather cluster ENNR with SES ####

ennrfl17WLD <- read.csv("2017ENNR_fl_HOT_OvE.csv")
ennrfl17WLD <- ennrfl17WLD[-1]

null.1.17WLD <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                                  sims = 999, data.type = "names",
                                  summary.type = "sum",
                                  r.samples = invertsennr17[,1],
                                  c.samples = ennr17[,1],
                                  r.weights = ennrfl17WLD)

weather.null.tab_17WLD <- test_interactions(null.1.17WLD)
write.csv(weather.null.tab_17WLD, "2017WeatherENNR_withSES_WLD_OvE.csv")

plot_preferences(null.1.17WLD, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withSES_ENNR_WLD_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17WLD, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(null.1.17WLD, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withSES_ENNR_WLD_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17WLD, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17WLD <- test_interactions(null.1.17WLD, signif.level = 0.95)
write.table(weather.links_17WLD, "2017 Weather ENNR output with SES input_WLD_OvE.csv")
edgelist_17WLD <- generate_edgelist(null.1.17WLD, signif.level = 0.95, 
                                    edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17WLD, "2017 Weather ENNR output with SES input - edgelist_WLD_OvE.csv")

##### 2017 WLD SES simulations extraction #####

weather.null_17WLD_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                                             sims = 999, data.type = "names",
                                             summary.type = "none",
                                             r.samples = invertsennr17[,1],
                                             c.samples = ennr17[,1],
                                             r.weights = ennrfl17WLD)

null.17WLD.rand <- weather.null_17WLD_sim$rand.data
null.17WLD.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17WLD_sim$n.iterations)
ind.cons.avg.17WLD <- aggregate(null.17WLD.rand[, 3:69], by = list(null.17WLD.rand$Consumer, null.17WLD.rand$indiv), mean)
colnames(ind.cons.avg.17WLD)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17WLD, "ENNR17WLD_ENNRSimOut_withSES_OvE.csv")

##### 2017 DWI Weather cluster ENNR with SES ####

ennrfl17DWI <- read.csv("2017ENNR_fl_DWI_OvE.csv")
ennrfl17DWI <- ennrfl17DWI[-1]

null.1.17DWI <- generate_null_net(ennr17[,2:68], invertsennr17[,2:67],
                                  sims = 999, data.type = "names",
                                  summary.type = "sum",
                                  r.samples = invertsennr17[,1],
                                  c.samples = ennr17[,1],
                                  r.weights = ennrfl17DWI)

weather.null.tab_17DWI <- test_interactions(null.1.17DWI)
write.csv(weather.null.tab_17DWI, "2017WeatherENNR_withSES_DWI_OvE.csv")

plot_preferences(null.1.17DWI, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withSES_ENNR_DWI_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17DWI, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(null.1.17DWI, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withSES_ENNR_DWI_OvE.pdf", width = 6, height = 14) 
plot_preferences(null.1.17DWI, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17DWI <- test_interactions(null.1.17DWI, signif.level = 0.95)
write.table(weather.links_17DWI, "2017 Weather ENNR output with SES input_DWI_OvE.csv")
edgelist_17DWI <- generate_edgelist(null.1.17DWI, signif.level = 0.95, 
                                    edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17DWI, "2017 Weather ENNR output with SES input - edgelist_DWI_OvE.csv")

##### 2017 DWI SES simulations extraction #####

weather.null_17DWI_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17[,2:67],
                                             sims = 999, data.type = "names",
                                             summary.type = "none",
                                             r.samples = invertsennr17[,1],
                                             c.samples = ennr17[,1],
                                             r.weights = ennrfl17DWI)

null.17DWI.rand <- weather.null_17DWI_sim$rand.data
null.17DWI.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17DWI_sim$n.iterations)
ind.cons.avg.17DWI <- aggregate(null.17DWI.rand[, 3:69], by = list(null.17DWI.rand$Consumer, null.17DWI.rand$indiv), mean)
colnames(ind.cons.avg.17DWI)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17DWI, "ENNR17DWI_ENNRSimOut_withSES_OvE.csv")

##### 2017 Weather cluster ENNR with equal prey abundance #####

library("econullnetr")

invertsennr17even <- read.csv("2017ENNR_Inverts_even.csv")

weather.null_17even <- generate_null_net(ennr17[,2:68], invertsennr17even[,2:67],
                                         sims = 999, data.type = "names",
                                         summary.type = "sum",
                                         r.samples = invertsennr17even[,1],
                                         c.samples = ennr17[,1])

weather.null.tab_17even <- test_interactions(weather.null_17even)
write.csv(weather.null.tab_17even, "2017WeatherENNR_withoutSES_evenpreyabund.csv")

plot_preferences(weather.null_17even, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Pre-harvest_withoutSES_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null_17even, "Pre-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()
plot_preferences(weather.null_17even, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
pdf("Outputs/Post-harvest_withoutSES_ENNR.pdf", width = 6, height = 14) 
plot_preferences(weather.null_17even, "Post-harvest", signif.level = 0.95, type = "counts",
                 xlab = "Num. of prey detections", p.cex = 1.5, l.cex = 0.8, lwd = 2)
dev.off()

weather.links_17even <- test_interactions(weather.null_17even, signif.level = 0.95)
write.table(weather.links_17even, "2017 Weather ENNR output without SES_evenpreyabund.csv")
edgelist_17even <- generate_edgelist(weather.null_17even, signif.level = 0.95, 
                                     edge.cols = c("#91bfdb", "#C9CAA3", "#fc8d59"))
write.table(edgelist_17even, "2017 Weather ENNR output without SES_evenpreyabund - edgelist.csv")

##### 2017 with equal prey abundance simulations extraction #####

weather.null_17even_sim <- generate_null_net_indiv(ennr17[,2:68], invertsennr17even[,2:67],
                                              sims = 999, data.type = "names",
                                              summary.type = "none",
                                              r.samples = invertsennr17even[,1],
                                              c.samples = ennr17[,1])

null.17even.rand <- weather.null_17even_sim$rand.data
null.17even.rand$indiv <- rep(seq(1:nrow(ennr17)), weather.null_17even_sim$n.iterations)
ind.cons.avg.17even <- aggregate(null.17even.rand[, 3:69], by = list(null.17even.rand$Consumer, null.17even.rand$indiv), mean)
colnames(ind.cons.avg.17even)[1:2] <- c("Consumer", "Individual")
write.csv(ind.cons.avg.17even, "ENNR17_ENNRSimOut_withoutSES_evenpreyabund.csv")

##### Processing 2017 ENNR outputs #####

# Model outputs
DWI.ennrout <- read.csv("2017 Weather ENNR output with SES input_DWI_OvE.csv", sep=" ")
HPR.ennrout <- read.csv("2017 Weather ENNR output with SES input_HPR_OvE.csv", sep=" ")
HOT.ennrout <- read.csv("2017 Weather ENNR output with SES input_HOT_OvE.csv", sep=" ")
WLD.ennrout <- read.csv("2017 Weather ENNR output with SES input_WLD_OvE.csv", sep=" ")
WMD.ennrout <- read.csv("2017 Weather ENNR output with SES input_WMD_OvE.csv", sep=" ")
unweight.ennrout <- read.csv("2017 Weather ENNR output without SES.csv", sep=" ")
even.ennrout <- read.csv("2017 Weather ENNR output without SES_evenpreyabund.csv", sep=" ")
SES.clusters <- unweight.ennrout[1:2]
SES.clusters$Unweighted <- unweight.ennrout$SES
SES.clusters$DWI <- DWI.ennrout$SES
SES.clusters$HPR <- HPR.ennrout$SES
SES.clusters$HOT <- HOT.ennrout$SES
SES.clusters$WLD <- WLD.ennrout$SES
SES.clusters$WMD <- WMD.ennrout$SES
SES.clusters$'Equal prey abundance' <- even.ennrout$SES
SES.clusters <- rename(SES.clusters,  Period = Consumer)
SES.clusters <- rename(SES.clusters, Prey = Resource)

write.csv(SES.clusters, "2017 effect size comparison.csv")


# Simulations
unweighted.ennrsims <- read.csv("ENNR17_ENNRSimOut_withoutSES.csv")
unweighted.ennrsims$Data <- rep("Unweighted")
unweighted.ennrsims <- unweighted.ennrsims[c((ncol(unweighted.ennrsims)), 2:(ncol(unweighted.ennrsims)-2))]
DWI.ennrsims <- read.csv("ENNR17DWI_ENNRSimOut_withSES_OvE.csv")
DWI.ennrsims$Data <- rep("DWI")
DWI.ennrsims <- DWI.ennrsims[c((ncol(DWI.ennrsims)), 2:(ncol(DWI.ennrsims)-2))]
HOT.ennrsims <- read.csv("ENNR17HOT_ENNRSimOut_withSES_OvE.csv")
HOT.ennrsims$Data <- rep("HOT")
HOT.ennrsims <- HOT.ennrsims[c((ncol(HOT.ennrsims)), 2:(ncol(HOT.ennrsims)-2))]
HPR.ennrsims <- read.csv("ENNR17HPR_ENNRSimOut_withSES_OvE.csv")
HPR.ennrsims$Data <- rep("HPR")
HPR.ennrsims <- HPR.ennrsims[c((ncol(HPR.ennrsims)), 2:(ncol(HPR.ennrsims)-2))]
WLD.ennrsims <- read.csv("ENNR17WLD_ENNRSimOut_withSES_OvE.csv")
WLD.ennrsims$Data <- rep("WLD")
WLD.ennrsims <- WLD.ennrsims[c((ncol(WLD.ennrsims)), 2:(ncol(WLD.ennrsims)-2))]
WMD.ennrsims <- read.csv("ENNR17WMD_ENNRSimOut_withSES_OvE.csv")
WMD.ennrsims$Data <- rep("WMD")
WMD.ennrsims <- WMD.ennrsims[c((ncol(WMD.ennrsims)), 2:(ncol(WMD.ennrsims)-2))]
even.ennrsims <- read.csv("ENNR17_ENNRSimOut_withoutSES_evenpreyabund.csv")
even.ennrsims$Data <- rep("Equal prey abundance")
even.ennrsims <- even.ennrsims[c((ncol(even.ennrsims)), 2:(ncol(even.ennrsims)-2))]
obs.ennrsims <- read.csv("2017ENNR_Diet_Weather.csv")
obs.ennrsims$Data <- rep("Observed")
obs.ennrsims$Consumer <- obs.ennrsims$Predator
obs.ennrsims$Individual <- rep(1:46)
obs.ennrsims <- obs.ennrsims[c((ncol(obs.ennrsims)-2),(ncol(obs.ennrsims)-1), (ncol(obs.ennrsims)), 3:(ncol(obs.ennrsims)-3))]

simvobs <- bind_rows(unweighted.ennrsims, DWI.ennrsims, HOT.ennrsims, HPR.ennrsims, WLD.ennrsims, WMD.ennrsims, even.ennrsims, obs.ennrsims)

write.csv(simvobs, "ENNRSimObvsCompare.csv")


##### ENNR sim vs obvs diet NMDS #####

library("devtools")
library("vegan")
library("ggplot2")
library("ggthemes")
library("RColorBrewer")
library("viridis")

simobs <- read.csv("ENNRSimObvsCompare.csv")
simobs <- simobs[-1]
simobsnmds <- simobs[,4:69]

simobs.mds <- metaMDS(comm = simobsnmds, distance = "euclidean", trymax=999, k=2, trace = FALSE, autotransform = FALSE)
plot(simobs.mds$points); text(simobs.mds, row.names(simobs.mds))
simobs.mds$stress

scrs <- scores(simobs.mds, display = 'sites')
scrs <- cbind(as.data.frame(scrs), Data = simobs$Data)
cent <- aggregate(cbind(NMDS1, NMDS2) ~ Data, data = scrs, FUN = mean)
segs <- merge(scrs, setNames(cent, c('Data', 'oNMDS1', 'oNMDS2')), by = 'Data', sort = FALSE)

ove.pal <- c("gray80", "orange", "purple", "tomato", "skyblue", "yellow", "green", "gray20")

dataspiplot <- ggplot(scrs, aes(x = NMDS1, y = NMDS2, fill = Data)) + 
  scale_fill_manual(values = ove.pal, breaks=c('Unweighted', 'HPR', 'DWI', 'WLD', 'WMD', 'HOT', 'Equal prey abundance', 'Observed')) + 
  scale_color_manual(values = ove.pal, breaks=c('Unweighted', 'HPR', 'DWI', 'WLD', 'WMD', 'HOT', 'Equal prey abundance', 'Observed')) + 
  geom_segment(data = segs, mapping = aes(xend = oNMDS1, yend = oNMDS2, color = Data), alpha=0.15) + 
  geom_point(data = cent, size = 4, pch = 21, alpha = 0.8) + 
  #geom_point(alpha=0.1, pch = 21, size = 0.5, aes(fill = Data)) + 
  coord_fixed() + theme_bw()

dataspiplot

tiff(file = "Outputs/SimVsObsSpiderplot.tiff", width = 24, height = 12, res = 600, units = "cm")
dataspiplot
dev.off()

cent$Distance <- sqrt(((0.24659706-cent$NMDS1)^2)+((-0.33512596-cent$NMDS2)^2))
write.csv(cent, "Exp vs Obs NMDS centroids.csv")

##### Compare model effect sizes #####

library("reshape2")

comp17 <- read.csv("2017 effect size comparison.csv")
comp17 <- comp17[-1]
hist(comp17$Unweighted)
hist(comp17$HPR)
hist(comp17$WMD)
hist(comp17$WLD)
hist(comp17$HOT)
hist(comp17$DWI)
hist(log10(comp17$Equal.prey.abundance))

t.comp17_HPR <- t.test(comp17$Unweighted, comp17$HPR, paired = TRUE)
t.comp17_HPR

t.comp17_WMD <- t.test(comp17$Unweighted, comp17$WMD, paired = TRUE)

t.comp17_WMD

t.comp17_WLD <- t.test(comp17$Unweighted, comp17$WLD, paired = TRUE)

t.comp17_WLD

t.comp17_HOT <- t.test(comp17$Unweighted, comp17$HOT, paired = TRUE)

t.comp17_HOT

t.comp17_DWI <- t.test(comp17$Unweighted, comp17$DWI, paired = TRUE)

t.comp17_DWI

t.comp17_EPA <- t.test(comp17$Unweighted, log10(comp17$Equal.prey.abundance), paired = TRUE)

t.comp17_EPA

comp17stack <- melt(comp17[2:9], id.var = "Prey", variable.name = "Model", value.name = "SES")

comp17agg <- aggregate(SES~Prey+Model, comp17stack, mean)

comp17agg$Model <- factor(comp17agg$Model, levels=c('Unweighted', 'HPR', 'DWI', 'WLD', 'WMD', 'HOT', 'Equal.prey.abundance'))

mod.pal <- c("gray80", "orange", "purple", "tomato", "skyblue", "yellow", "green")

pairedses <-  ggplot(data = comp17agg, aes(x = Model, y = SES))+
              geom_point(shape = 21, size = 2, aes(fill = Model), alpha = 0.5) +
              geom_line(aes(group = Prey), alpha = 0.2) +
              theme_bw() + 
              scale_fill_manual(labels = c("Equal.prey.abundance" = "Equal prey abundance"), values = mod.pal)+
              theme(axis.text.x = element_text(angle = 20, vjust = 1.1, hjust=1))+
              scale_x_discrete(labels=c("Equal.prey.abundance" = "Equal prey abundance"))
pairedses

tiff(file = "Outputs/Sim vs Obs SES paired comparison.tiff", width = 24, height = 12, res = 600, units = "cm")
pairedses
dev.off()

##### 2018 bipartite network #####

library(ggplot2)
library(ggnetwork)
library(igraph)
library(reshape2)
library(dplyr)

int.df <- read.csv("2018 Weather ENNR output - edgelist.csv")

node.df <- read.csv("spider_nodes.csv")

nut1 <- graph_from_data_frame(d = int.df, vertices = node.df, directed = TRUE)
nut1

plot(nut1)

node.df$x.coord <- 1
node.df[node.df$tr.height == 1, ]$x.coord <- 
  seq(0, 1, length.out = (sum(node.df$tr.height == 1)))
node.df[node.df$tr.height == 2, ]$x.coord <- 
  seq(.2, .8, length.out = (sum(node.df$tr.height == 2)))

plot(nut1, layout = as.matrix(node.df[, c("x.coord", "tr.height")]))

nut2 <- fortify(nut1, layout = as.matrix(node.df[, c("x.coord", "tr.height")]))

nut2$tr.height <- (nut2$tr.height - 1) / 2
nut2$y <- nut2$y / 2
nut2$yend <- nut2$yend / 2
node.df$tr.height <- (node.df$tr.height - 1) / 2

ggplot(nut2, aes(x = x, y = y, xend = xend, yend = yend)) +
  geom_edges() +
  geom_point(data = node.df, aes(x = x.coord, y = tr.height),
             inherit.aes = FALSE, size = 5) +
  theme_blank()

node.df$level <- as.factor(node.df$tr.height)
node.df$level <- recode_factor(node.df$level, '0' = "Lower", '0.5' = "Higher")

nodepal <- c("darkgreen", "#FDE725FF")

pointsToLabel <- c("DWI", "HPR", "HOT", "WMD", "WLD")

bipweathernet <- ggplot(nut2, aes(x = x, y = y, xend = xend, yend = yend)) +
  geom_edges(alpha = 0.5, aes(col = Test, size = (abs(Observed)))) +
  geom_point(data = node.df, aes(x = x.coord, y = tr.height, shape = level, fill = level),
             inherit.aes = FALSE, alpha = 0.8, size = 4) +
  theme_blank() + coord_fixed() +
  scale_shape_manual(guide = 'none', values = c(21,22)) +
  scale_fill_manual(guide = 'none', values = nodepal)+
  theme(legend.key.size = unit(0.5, 'cm'), legend.text = element_text(size=10), legend.title = element_text(size=12, face = "bold"), legend.key.width = unit(0.4, "cm")) +
  scale_color_manual(name = "Preference", labels = c("Stronger", "n.s.", "Weaker"), values = c("#fc8d59", "#b2b2b2", "#7f7fff")) +
  scale_size(guide = "none") + 
  geom_text(aes(x = x, y = (y+0.035), label = name), data = nut2[nut2$name %in% pointsToLabel,])
bipweathernet

tiff(file = "Outputs/Weather cluster bipartite network.tiff", width = 20, height = 20, res = 300, units = "cm")
bipweathernet
dev.off()

##### 2018 circular network plot #####

library(igraph)

## Data entry ##
net18 <- read.csv("2018 Weather ENNR output - edgelist.csv")
net18$edge.col <- recode_factor(as.factor(net18$edge.col), '#d7191c' = "#fc8d59", '#000000' = "#b2b2b2", '#2c7bb6' = "#7f7fff")
net18$edge.col <- as.character(net18$edge.col)

## Plotting the  network
# Generate an igraph object from the matrix
net18_network <- graph_from_edgelist(as.matrix(net18[1:2]))
# Add the weights
E(net18_network)$weights <- sqrt(net18$Observed)
# Add the color
E(net18_network)$colour <- net18$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net18_dataframe <- igraph::as_data_frame(net18_network, "both")
# Add other information about the taxa
V(net18_network)$shape <- c("square","circle","circle","circle","circle","circle","square","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","square","circle","circle","circle","circle","circle","square","circle","circle","square","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle","circle")
# Change the colours for plotting
V(net18_network)$colour <- V(net18_network)$shape
V(net18_network)$colour <- gsub("square","#FDE725FF", V(net18_network)$colour)
V(net18_network)$colour <- gsub("circle","darkgreen", V(net18_network)$colour)
# Layout for the network
la18 <- layout.circle(net18_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net18_network, edge.width = E(net18_network)$weights, edge.color = adjustcolor(E(net18_network)$colour, alpha.f = 0.8), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net18_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la18, edge.arrow.size = 0, vertex.color.alpha = 0.5)
#legend("topleft", legend = c("Spider", "Other"), col = c("firebrick","darkgreen"), bty = "n", pch = 20, pt.cex = 3, cex = 1, horiz = F, inset = c(0.05,0.05))
# Specify the adjustment values
x_adj <- c(1.32,1.6,1.7,1.73,1.68,1.72,1.34,1.55,1.72,1.68,1.62,1.58,1.52,1.75,1.52,1.65,1.52,1.55,1.75,1.34,1.61,1.61,1.61,1.6,1.58,1.36,1.6,1.75,1.36,1.7,1.65,1.66,1.72,1.68,1.58,1.52,1.68,1.58,1.52,1.5,1.6,1.64,1.55,1.62)
y_adj <- c(1.32,1.6,1.7,1.73,1.68,1.72,1.34,1.55,1.72,1.68,1.62,1.58,1.52,1.75,1.52,1.65,1.52,1.55,1.75,1.34,1.61,1.61,1.61,1.6,1.58,1.36,1.6,1.75,1.36,1.7,1.65,1.66,1.72,1.68,1.58,1.52,1.68,1.58,1.52,1.5,1.6,1.64,1.55,1.62)
x_adj <- x_adj-.12
y_adj <- y_adj-.12

# Specify x and y coordinates of labels, adjust outward as desired
x = (la18[,1])*x_adj
y = (la18[,2])*y_adj
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la18[,1]/la18[,2]))*(180/pi) < 0,
               90 + atan(- (la18[,1]/la18[,2]))*(180/pi), 270 + atan(-la18[,1]/la18[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net18_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col="black", srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2018 network.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net18_network, edge.width = E(net18_network)$weights, edge.color = adjustcolor(E(net18_network)$colour, alpha.f = 0.8), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net18_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la18, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj <- c(1.45,1.6,1.65,1.68,1.63,1.67,1.45,1.55,1.62,1.60,1.58,1.53,1.52,1.62,1.52,1.58,1.52,1.52,1.65,1.45,1.57,1.57,1.57,1.55,1.54,1.45,1.56,1.62,1.45,1.62,1.61,1.6,1.64,1.62,1.57,1.52,1.64,1.57,1.52,1.5,1.58,1.61,1.55,1.61)
y_adj <- c(1.45,1.6,1.65,1.68,1.63,1.67,1.45,1.55,1.62,1.60,1.58,1.53,1.52,1.62,1.52,1.58,1.52,1.52,1.65,1.45,1.57,1.57,1.57,1.55,1.54,1.45,1.56,1.62,1.45,1.62,1.61,1.6,1.64,1.62,1.57,1.52,1.64,1.57,1.52,1.5,1.58,1.61,1.55,1.61)
x_adj <- x_adj-.25
y_adj <- y_adj-.25
x = (la18[,1])*x_adj
y = (la18[,2])*y_adj
angle = ifelse(atan(-(la18[,1]/la18[,2]))*(180/pi) < 0,
               90 + atan(- (la18[,1]/la18[,2]))*(180/pi), 270 + atan(-la18[,1]/la18[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net18_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col="black", srt=angle[i], xpd=T)
}
dev.off()

##### 2017 circular network plots #####

net17 <- read.csv("2017 Weather ENNR output without SES - edgelist.csv", sep = " ")
net17$edge.col <- recode_factor(as.factor(net17$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17$edge.col <- as.character(net17$edge.col)
net17dwi <- read.csv("2017 Weather ENNR output with SES input - edgelist_DWI_OvE.csv", sep = " ")
net17dwi$edge.col <- recode_factor(as.factor(net17dwi$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17dwi$edge.col <- as.character(net17dwi$edge.col)
net17wmd <- read.csv("2017 Weather ENNR output with SES input - edgelist_WMD_OvE.csv", sep = " ")
net17wmd$edge.col <- recode_factor(as.factor(net17wmd$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17wmd$edge.col <- as.character(net17wmd$edge.col)
net17hot <- read.csv("2017 Weather ENNR output with SES input - edgelist_HOT_OvE.csv", sep = " ")
net17hot$edge.col <- recode_factor(as.factor(net17hot$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17hot$edge.col <- as.character(net17hot$edge.col)
net17hpr <- read.csv("2017 Weather ENNR output with SES input - edgelist_HPR_OvE.csv", sep = " ")
net17hpr$edge.col <- recode_factor(as.factor(net17hpr$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17hpr$edge.col <- as.character(net17hpr$edge.col)
net17wld <- read.csv("2017 Weather ENNR output with SES input - edgelist_WLD_OvE.csv", sep = " ")
net17wld$edge.col <- recode_factor(as.factor(net17wld$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17wld$edge.col <- as.character(net17wld$edge.col)
net17even <- read.csv("2017 Weather ENNR output without SES_evenpreyabund - edgelist.csv", sep = " ")
net17even$edge.col <- recode_factor(as.factor(net17even$edge.col), '#fc8d59' = "#fc8d59", '#C9CAA3' = "#b2b2b2", '#91bfdb' = "#7f7fff")
net17even$edge.col <- as.character(net17even$edge.col)

### 2017 network (non-SES-weighted network) ###
# Generate an igraph object from the matrix
net17_network <- graph_from_edgelist(as.matrix(net17[1:2]))
# Add the weights
E(net17_network)$weights <- abs(net17$SES)
# Add the color
E(net17_network)$colour <- net17$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17_dataframe <- igraph::as_data_frame(net17_network, "both")
# Add other information about the taxa
V(net17_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                             "circle","circle","circle","circle","square","circle","circle","circle")
V(net17_network)$text <- c("black","black","black","black","black","black","black","black","black",
                            "black","black","black","black","black","black","black","black")
# Change the colours for plotting
V(net17_network)$colour <- V(net17_network)$shape
V(net17_network)$colour <- gsub("square","#FDE725FF", V(net17_network)$colour)
V(net17_network)$colour <- gsub("circle","darkgreen", V(net17_network)$colour)
# Layout for the network
la17w <- layout.circle(net17_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17_network, edge.width = E(net17_network)$weights, edge.color = adjustcolor(E(net17_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17w, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17w <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17w <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17w <- x_adj17w +0.05
y_adj17w <- y_adj17w +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17w[,1]*x_adj17w
y = la17w[,2]*y_adj17w
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17w[,1]/la17w[,2]))*(180/pi) < 0,
               90 + atan(- (la17w[,1]/la17w[,2]))*(180/pi), 270 + atan(-la17w[,1]/la17w[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17_network, edge.width = E(net17_network)$weights, edge.color = adjustcolor(E(net17_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17w, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17w <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17w <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17w <- x_adj17w -0.12
y_adj17w <- y_adj17w -0.12
x = la17w[,1]*x_adj17w
y = la17w[,2]*y_adj17w
angle = ifelse(atan(-(la17w[,1]/la17w[,2]))*(180/pi) < 0,
               90 + atan(- (la17w[,1]/la17w[,2]))*(180/pi), 270 + atan(-la17w[,1]/la17w[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()

### 2017 WMD network ###
# Generate an igraph object from the matrix
net17wmd_network <- graph_from_edgelist(as.matrix(net17wmd[1:2]))
# Add the weights
E(net17wmd_network)$weights <- abs(net17wmd$SES)
# Add the color
E(net17wmd_network)$colour <- net17wmd$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17wmd_dataframe <- igraph::as_data_frame(net17wmd_network, "both")
# Add other information about the taxa
V(net17wmd_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                               "circle","circle","circle","circle","square","circle","circle","circle")
V(net17wmd_network)$text <- c("black", "black", "limegreen", "black", "limegreen", "black", "black", "black", "limegreen", "black", "black", "black", "limegreen", "black", "black", "black", "black")
# Change the colours for plotting
V(net17wmd_network)$colour <- V(net17wmd_network)$shape
V(net17wmd_network)$colour <- gsub("square","#FDE725FF", V(net17wmd_network)$colour)
V(net17wmd_network)$colour <- gsub("circle","darkgreen", V(net17wmd_network)$colour)
# Layout for the network
la17wmd <- layout.circle(net17wmd_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17wmd_network, edge.width = E(net17wmd_network)$weights, edge.color = adjustcolor(E(net17wmd_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17wmd_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17wmd, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17wmd <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17wmd <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17wmd <- x_adj17wmd +0.05
y_adj17wmd <- y_adj17wmd +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17wmd[,1]*x_adj17wmd
y = la17wmd[,2]*y_adj17wmd
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17wmd[,1]/la17wmd[,2]))*(180/pi) < 0,
               90 + atan(- (la17wmd[,1]/la17wmd[,2]))*(180/pi), 270 + atan(-la17wmd[,1]/la17wmd[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17wmd_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17wmd_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network wmd.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17wmd_network, edge.width = E(net17wmd_network)$weights, edge.color = adjustcolor(E(net17wmd_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17wmd_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17wmd, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17wmd <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17wmd <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17wmd <- x_adj17wmd -0.12
y_adj17wmd <- y_adj17wmd -0.12
x = la17wmd[,1]*x_adj17wmd
y = la17wmd[,2]*y_adj17wmd
angle = ifelse(atan(-(la17wmd[,1]/la17wmd[,2]))*(180/pi) < 0,
               90 + atan(- (la17wmd[,1]/la17wmd[,2]))*(180/pi), 270 + atan(-la17wmd[,1]/la17wmd[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17wmd_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17wmd_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()

### 2017 hpr network ###
# Generate an igraph object from the matrix
net17hpr_network <- graph_from_edgelist(as.matrix(net17hpr[1:2]))
# Add the weights
E(net17hpr_network)$weights <- abs(net17hpr$SES)
# Add the color
E(net17hpr_network)$colour <- net17hpr$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17hpr_dataframe <- igraph::as_data_frame(net17hpr_network, "both")
# Add other information about the taxa
V(net17hpr_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                               "circle","circle","circle","circle","square","circle","circle","circle")
V(net17hpr_network)$text <- c("black", "black", "black", "black", "black", "black", "black", "black", "limegreen", "black", "black", "black", "limegreen", "black", "black", "black", "goldenrod")
# Change the colours for plotting
V(net17hpr_network)$colour <- V(net17hpr_network)$shape
V(net17hpr_network)$colour <- gsub("square","#FDE725FF", V(net17hpr_network)$colour)
V(net17hpr_network)$colour <- gsub("circle","darkgreen", V(net17hpr_network)$colour)
# Layout for the network
la17hpr <- layout.circle(net17hpr_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17hpr_network, edge.width = E(net17hpr_network)$weights, edge.color = adjustcolor(E(net17hpr_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17hpr_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17hpr, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17hpr <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17hpr <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17hpr <- x_adj17hpr +0.05
y_adj17hpr <- y_adj17hpr +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17hpr[,1]*x_adj17hpr
y = la17hpr[,2]*y_adj17hpr
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17hpr[,1]/la17hpr[,2]))*(180/pi) < 0,
               90 + atan(- (la17hpr[,1]/la17hpr[,2]))*(180/pi), 270 + atan(-la17hpr[,1]/la17hpr[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17hpr_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17hpr_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network hpr.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17hpr_network, edge.width = E(net17hpr_network)$weights, edge.color = adjustcolor(E(net17hpr_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17hpr_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17hpr, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17hpr <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17hpr <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17hpr <- x_adj17hpr -0.12
y_adj17hpr <- y_adj17hpr -0.12
x = la17hpr[,1]*x_adj17hpr
y = la17hpr[,2]*y_adj17hpr
angle = ifelse(atan(-(la17hpr[,1]/la17hpr[,2]))*(180/pi) < 0,
               90 + atan(- (la17hpr[,1]/la17hpr[,2]))*(180/pi), 270 + atan(-la17hpr[,1]/la17hpr[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17hpr_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17hpr_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()

### 2017 hot network ###
# Generate an igraph object from the matrix
net17hot_network <- graph_from_edgelist(as.matrix(net17hot[1:2]))
# Add the weights
E(net17hot_network)$weights <- abs(net17hot$SES)
# Add the color
E(net17hot_network)$colour <- net17hot$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17hot_dataframe <- igraph::as_data_frame(net17hot_network, "both")
# Add other information about the taxa
V(net17hot_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                               "circle","circle","circle","circle","square","circle","circle","circle")
V(net17hot_network)$text <- c("black", "black", "black", "black", "black", "black", "black", "black", "limegreen", "black", "black", "black", "limegreen", "black", "black", "black", "black")
# Change the colours for plotting
V(net17hot_network)$colour <- V(net17hot_network)$shape
V(net17hot_network)$colour <- gsub("square","#FDE725FF", V(net17hot_network)$colour)
V(net17hot_network)$colour <- gsub("circle","darkgreen", V(net17hot_network)$colour)
# Layout for the network
la17hot <- layout.circle(net17hot_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17hot_network, edge.width = E(net17hot_network)$weights, edge.color = adjustcolor(E(net17hot_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17hot_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17hot, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17hot <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17hot <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17hot <- x_adj17hot +0.05
y_adj17hot <- y_adj17hot +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17hot[,1]*x_adj17hot
y = la17hot[,2]*y_adj17hot
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17hot[,1]/la17hot[,2]))*(180/pi) < 0,
               90 + atan(- (la17hot[,1]/la17hot[,2]))*(180/pi), 270 + atan(-la17hot[,1]/la17hot[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17hot_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17hot_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network hot.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17hot_network, edge.width = E(net17hot_network)$weights, edge.color = adjustcolor(E(net17hot_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17hot_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17hot, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17hot <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17hot <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17hot <- x_adj17hot -0.12
y_adj17hot <- y_adj17hot -0.12
x = la17hot[,1]*x_adj17hot
y = la17hot[,2]*y_adj17hot
angle = ifelse(atan(-(la17hot[,1]/la17hot[,2]))*(180/pi) < 0,
               90 + atan(- (la17hot[,1]/la17hot[,2]))*(180/pi), 270 + atan(-la17hot[,1]/la17hot[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17hot_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17hot_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()


### 2017 dwi network ###
# Generate an igraph object from the matrix
net17dwi_network <- graph_from_edgelist(as.matrix(net17dwi[1:2]))
# Add the weights
E(net17dwi_network)$weights <- abs(net17dwi$SES)
# Add the color
E(net17dwi_network)$colour <- net17dwi$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17dwi_dataframe <- igraph::as_data_frame(net17dwi_network, "both")
# Add other information about the taxa
V(net17dwi_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                               "circle","circle","circle","circle","square","circle","circle","circle")
V(net17dwi_network)$text <- c("black", "black", "limegreen", "limegreen", "black", "black", "black", "black", "limegreen", "limegreen", "black", "black", "limegreen", "black", "black", "darkorchid", "goldenrod")
# Change the colours for plotting
V(net17dwi_network)$colour <- V(net17dwi_network)$shape
V(net17dwi_network)$colour <- gsub("square","#FDE725FF", V(net17dwi_network)$colour)
V(net17dwi_network)$colour <- gsub("circle","darkgreen", V(net17dwi_network)$colour)
# Layout for the network
la17dwi <- layout.circle(net17dwi_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17dwi_network, edge.width = E(net17dwi_network)$weights, edge.color = adjustcolor(E(net17dwi_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17dwi_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17dwi, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17dwi <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17dwi <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17dwi <- x_adj17dwi +0.05
y_adj17dwi <- y_adj17dwi +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17dwi[,1]*x_adj17dwi
y = la17dwi[,2]*y_adj17dwi
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17dwi[,1]/la17dwi[,2]))*(180/pi) < 0,
               90 + atan(- (la17dwi[,1]/la17dwi[,2]))*(180/pi), 270 + atan(-la17dwi[,1]/la17dwi[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17dwi_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17dwi_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network dwi.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17dwi_network, edge.width = E(net17dwi_network)$weights, edge.color = adjustcolor(E(net17dwi_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17dwi_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17dwi, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17dwi <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17dwi <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17dwi <- x_adj17dwi -0.12
y_adj17dwi <- y_adj17dwi -0.12
x = la17dwi[,1]*x_adj17dwi
y = la17dwi[,2]*y_adj17dwi
angle = ifelse(atan(-(la17dwi[,1]/la17dwi[,2]))*(180/pi) < 0,
               90 + atan(- (la17dwi[,1]/la17dwi[,2]))*(180/pi), 270 + atan(-la17dwi[,1]/la17dwi[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17dwi_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17dwi_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()

### 2017 wld network ###
# Generate an igraph object from the matrix
net17wld_network <- graph_from_edgelist(as.matrix(net17wld[1:2]))
# Add the weights
E(net17wld_network)$weights <- abs(net17wld$SES)
# Add the color
E(net17wld_network)$colour <- net17wld$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17wld_dataframe <- igraph::as_data_frame(net17wld_network, "both")
# Add other information about the taxa
V(net17wld_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                               "circle","circle","circle","circle","square","circle","circle","circle")
V(net17wld_network)$text <- c("black", "black", "black", "black", "black", "black", "black", "black", "limegreen", "black", "black", "black", "limegreen", "black", "black", "black", "black")
# Change the colours for plotting
V(net17wld_network)$colour <- V(net17wld_network)$shape
V(net17wld_network)$colour <- gsub("square","#FDE725FF", V(net17wld_network)$colour)
V(net17wld_network)$colour <- gsub("circle","darkgreen", V(net17wld_network)$colour)
# Layout for the network
la17wld <- layout.circle(net17wld_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17wld_network, edge.width = E(net17wld_network)$weights, edge.color = adjustcolor(E(net17wld_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17wld_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17wld, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17wld <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17wld <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17wld <- x_adj17wld +0.05
y_adj17wld <- y_adj17wld +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17wld[,1]*x_adj17wld
y = la17wld[,2]*y_adj17wld
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17wld[,1]/la17wld[,2]))*(180/pi) < 0,
               90 + atan(- (la17wld[,1]/la17wld[,2]))*(180/pi), 270 + atan(-la17wld[,1]/la17wld[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17wld_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17wld_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network wld.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17wld_network, edge.width = E(net17wld_network)$weights, edge.color = adjustcolor(E(net17wld_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17wld_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17wld, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17wld <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17wld <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17wld <- x_adj17wld -0.12
y_adj17wld <- y_adj17wld -0.12
x = la17wld[,1]*x_adj17wld
y = la17wld[,2]*y_adj17wld
angle = ifelse(atan(-(la17wld[,1]/la17wld[,2]))*(180/pi) < 0,
               90 + atan(- (la17wld[,1]/la17wld[,2]))*(180/pi), 270 + atan(-la17wld[,1]/la17wld[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17wld_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17wld_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()


### 2017 network (even prey abundance) ###
# Generate an igraph object from the matrix
net17even_network <- graph_from_edgelist(as.matrix(net17even[1:2]))
# Add the weights
E(net17even_network)$weights <- abs(net17even$SES)
# Add the color
E(net17even_network)$colour <- net17even$edge.col
# Extract the edges and vertices for some wrangling as igraph is annoying for adding in attribute data (i.e. hard to match vertices with dataframes)
net17even_dataframe <- igraph::as_data_frame(net17even_network, "both")
# Add other information about the taxa
V(net17even_network)$shape <- c("square","circle","circle","circle","circle","circle","circle","circle","circle",
                                "circle","circle","circle","circle","square","circle","circle","circle")
V(net17even_network)$text <- c("black", "black", "limegreen", "limegreen", "limegreen", "black", "limegreen", "black", "black", "limegreen", "black", "black", "limegreen", "black", "black", "darkorchid", "goldenrod")
# Change the colours for plotting
V(net17even_network)$colour <- V(net17even_network)$shape
V(net17even_network)$colour <- gsub("square","#FDE725FF", V(net17even_network)$colour)
V(net17even_network)$colour <- gsub("circle","darkgreen", V(net17even_network)$colour)
# Layout for the network
la17even <- layout.circle(net17even_network)
# Set the plot margins to fit in the labels
par(mar = c(5.5,5.5,5.5,5.5))
# Plot network
plot(net17even_network, edge.width = E(net17even_network)$weights, edge.color = adjustcolor(E(net17even_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17even_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17even, edge.arrow.size = 0, vertex.color.alpha = 0.5)
# Specify the adjustment values
x_adj17even <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
y_adj17even <- c(1.43,1.48,1.38,1.38,1.48,1.38,1.36,1.41,1.46,1.35,1.36,1.53,1.41,1.42,1.34,1.48,1.4)
x_adj17even <- x_adj17even +0.05
y_adj17even <- y_adj17even +0.05
# Specify x and y coordinates of labels, adjust outward as desired
x = la17even[,1]*x_adj17even
y = la17even[,2]*y_adj17even
#create vector of angles for text based on number of nodes (flipping the orientation of the words half way around so none appear upside down)
angle = ifelse(atan(-(la17even[,1]/la17even[,2]))*(180/pi) < 0,
               90 + atan(- (la17even[,1]/la17even[,2]))*(180/pi), 270 + atan(-la17even[,1]/la17even[,2])*(180/pi))
#Apply the text labels with a loop with angle as srt
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17even_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17even_network)$text[i], srt=angle[i], xpd=T)
}


tiff(file = "Outputs/2017 network evenpreyabund.tiff", width = 15, height = 15, res = 600, units = "cm")
par(mar = c(5.5,5.5,5.5,5.5))
plot(net17even_network, edge.width = E(net17even_network)$weights, edge.color = adjustcolor(E(net17even_network)$colour, alpha.f = 0.6), vertex.label.color = "black", vertex.label = "",
     vertex.color = adjustcolor(V(net17even_network)$colour, alpha.f = 0.9), vertex.size = 12, layout = la17even, edge.arrow.size = 0, vertex.color.alpha = 0.5)
x_adj17even <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
y_adj17even <- c(1.43,1.47,1.41,1.46,1.48,1.41,1.45,1.42,1.47,1.38,1.38,1.48,1.39,1.42,1.38,1.49,1.43)
x_adj17even <- x_adj17even -0.12
y_adj17even <- y_adj17even -0.12
x = la17even[,1]*x_adj17even
y = la17even[,2]*y_adj17even
angle = ifelse(atan(-(la17even[,1]/la17even[,2]))*(180/pi) < 0,
               90 + atan(- (la17even[,1]/la17even[,2]))*(180/pi), 270 + atan(-la17even[,1]/la17even[,2])*(180/pi))
for (i in 1:length(x)) {
  text(x=x[i], y=y[i], labels=V(net17even_network)$name[i], adj=NULL,
       pos=NULL, cex=0.8, col=V(net17even_network)$text[i], srt=angle[i], xpd=T)
}
dev.off()

##### Dietary diversity vs temperature #####

library(stringr)

ennr$dietdiv <- rowSums(ennr[3:ncol(ennr)])
invertsennr$invdiv <- rowSums(invertsennr[2:ncol(invertsennr)])

divcomp <- merge(ennr, invertsennr, by = 'ENNRcode')
diet$ENNRcode <- str_sub(diet$Sample, 1, 3)
divcomp <- merge(divcomp, diet, by = 'ENNRcode')
divcomp <- divcomp[c('dietdiv', 'invdiv', 'MeanWeekTemp')]

tempdiv <- ggplot(divcomp, aes(x=MeanWeekTemp, y = dietdiv))+
  geom_point(size = 1, alpha = 0.4, aes(colour = MeanWeekTemp)) +
  scale_colour_viridis(name = "Temperature", option = "A")+
  geom_smooth(method = 'loess', colour = "tomato", fill = "tomato") +
  labs(x = ('Temperature'), y = 'Dietary diversity') + theme(axis.text = element_text(size = 8), axis.title = element_text(size = 10), legend.title = element_text(size = 10), legend.text = element_text(size = 8), legend.key.size = unit(0.5, 'cm')) + theme_bw()
invdiv <- ggplot(divcomp, aes(x=invdiv, y = dietdiv))+
  geom_point(size = 1, alpha = 0.4, aes(colour = invdiv)) +
  scale_colour_viridis(name = "Prey abundance")+
  geom_smooth(method = 'loess', colour = "forestgreen", fill = "forestgreen") +
  labs(x = ('Prey abundance'), y = 'Dietary diversity') + theme(axis.text = element_text(size = 8), axis.title = element_text(size = 10), legend.title = element_text(size = 10), legend.text = element_text(size = 8), legend.key.size = unit(0.5, 'cm')) + theme_bw()

tiff(file = "Outputs/Prey and dietary diversity and temperature.tiff", width = 20, height = 8, res = 300, units = "cm")
ggarrange(tempdiv, invdiv, ncol = 2)
dev.off()
