
/******************************************************************************/
/* Code to produce figures and graphs in "Is the Social Safety Net a Long-Term Investment?
 Large-Scale Evidence from the Food Stamps Program,” 
 by Martha J. Bailey, Hilary Hoynes, Maya Rossin-Slater, and Reed Walker	
 
 The inputs are Excel files with output disclosed from the RDC.
/******************************************************************************/
*/

set scheme s2color

*Make sure necessary ados are installed
cap which labutil 
if !_rc ssc install labutil
ssc install labutil
ssc install sxpose

/****************************************************************************/
********************* FIGURE 3 *********************************************
****************************************************************************/
	local omitexp = 10		// Omitted event-time
	local omitdum = 17		// Omitted event-time dummy

	*foreach g in All WF WM NF NM {
	foreach g in all  {


	foreach y in z_composite {


	/* BRING IN OUTREG FILE */
	import excel using "`y'.xls", clear sheet("`g'") cellrange(A3) ///
		firstrow
	replace VARIABLES = subinstr(VARIABLES,"TB_exp_","",.)	
	ren VARIABLES year
	ren EventStudy0 x0
	ren G x1
	ren H x2
	rename I x3 //this is labeled as event study 0 as well
	rename Spline* lsr*

	*Drop FE indicators
	drop if regexm(year, "Birth") == 1 | regexm(year, "Survey") == 1 ///
		| regexm(year, "County") == 1
	quietly destring _all, replace ignore(",")

	/*ADJUST S.E.'S USING ADDED STATS TO THE OUTREG FILE*/
	**make locals out of the number of clusters and the number of regressors to fix the areg dF
	**do it for each "variable" here--ie. specification--in case the obs chnage and because the number of regressors changes
	replace year = "N" if year=="Observations"

	foreach var of varlist x*  {
	
		foreach stat in  Counties k Cells {
		di "This summarizes `stat'"
			sum `var' if year=="`stat'"
			local `stat' = r(mean)
		}
		*areg degrees of freedom adjustment--Cameron and Miller (the if exp=="" part ensures that it only adjusts the s.e.'s)
		replace `var' = `var'*((`Cells'-`k'-1)/(`Cells'-`k'-`Ng'-1)) if year != ""
		}	

	
	* Identify betas
	gen b = year != ""
	replace year = year[_n-1] if year== ""

	rename year stryear
		g year = real(stryear)
	
	compress

	/************* 
	Capture spline
	*************/

	*Get point estimates
	preserve
	import excel using "splinem5_p17_`y'.xls", ///
	clear sheet("`g'") cellrange(A2) ///
		firstrow
	rename VARIABLES year
	keep year Model3
	rename Model* lsr*

	keep if substr(year,1,6)=="spline"
	destring _all, replace
	local s1 = lsr3 in 1
	local s2 = lsr3 in 2
	local s3 = lsr3 in 3
	local s4 = lsr3 in 4
	restore

	*Now get SEs
	preserve
	import excel using "splinem5_p17_`y'.xls", ///
	clear sheet("`g'") cellrange(A2) ///
		firstrow
	rename VARIABLES year
	keep year Model3
	rename Model* lsr*
	g b = regexm(year, "spline") == 1
		replace year = year[_n-1] if substr(year[_n-1],1,6)=="spline" & ///
		year == ""

	keep if b==0 & lsr3 != "" & regexm(year, "spline") == 1
	destring _all, replace
	local se1 = lsr3 in 1
	local se2 = lsr3 in 2
	local se3 = lsr3 in 3
	local se4 = lsr3 in 4
	restore

	* Now expand to cover [-5,10], the balanced panel
	preserve
	clear
	local obs = 18+5 + 1
	set obs `obs'
	gen year = _n - 7
	summ year, d

	
	gen lsr3 = inrange(year,-5,-2)*(`s1'*(year+1)-7*`s2'-4*`s3') ///
		+ inrange(year,-1,5)*(-`s2'*(6-year)-4*`s3') ///
		+ inrange(year,6,11)*(`s3'*(year-10)) ///
		+ inrange(year,12,18)*(`s3'+`s4'*(year-11))
	
	*Get upper bound of CI
	gen lsr3_ub = (lsr3 + 1.96*`se1')*inrange(year,-5,-2) + ///
		(lsr3 + 1.96*`se2')*inrange(year,-1,5) + ///
		(lsr3 + 1.96*`se3')*inrange(year,6,11) + ///
		(lsr3 + 1.96*`se4')*inrange(year,12,18)
	di "got here"
	*Get lower bound of CI
	g lsr3_lb = (lsr3 - 1.96*`se1')*inrange(year,-5,-2) + ///
		(lsr3 - 1.96*`se2')*inrange(year,-1,5) + ///
		(lsr3 - 1.96*`se3')*inrange(year,6,11) + ///
		(lsr3 - 1.96*`se4')*inrange(year,12,18)


	su
	tempfile spline
	save "`spline'", replace
	di "Finished saving spline"
	restore



	/*************************
	Prepare event study coeffs
	*************************/

	keep if substr(stryear,1,6)!="spline"	// Drop spline coefficients and SEs
		*Just keep years
		keep if year != .
	*destring year, replace
	local obs = _N+1
	set obs `obs'						// Add extra observation for omitted category
	replace year = `omitdum' in `obs'	// Assign correct event-time dummy
	foreach var of varlist x* {		// 
		replace `var'=0 in `obs'
	}
	replace b = 1 in `obs'

	replace year = year - (`omitdum' - `omitexp')	// Adjust dummy variable numbers to event time

	/*RESHAPE SO S.E.'S ARE NEXT TO COEFS*/
	for var _all: ren X X_ 
	ren year_ year
	ren b_ b
	reshape wide *_, i(year) j(b)

	/*CONFIDENCE INTERVALS*/
	cap drop lb* ub*

	forvalues a=0/3 {

	gen lb_`a' = x`a'_1 - 1.96*x`a'_0 if year!=`omitexp'
	gen ub_`a' = x`a'_1 + 1.96*x`a'_0 if year!=`omitexp'

	}



	* Merge spline onto event-study coefficients
	merge 1:1 year using "`spline'"
	assert _merge!=2
	drop _merge
	su year
	drop if year==`r(min)' | year==`r(max)'		// Drop endpoints


	/********
	Labels
	*********/
	* Subgroups
	if "`g'"=="all" local x ""
	if "`g'"=="WF" local x ", White females"
	if "`g'"=="WM" local x ", White males"
	if "`g'"=="NF" local x ", Nonwhite females"
	if "`g'"=="NM" local x ", Nonwhite males"

	* Outcomes
	if "`y'"=="hc" local ti "Human capital index`x'"
	if "`y'"=="z_composite" local ti "Composite index`x'"
	if "`y'"=="ess" local ti "Economic self-sufficiency index`x'"
	if "`y'"=="nei" local ti "Neighborhood quality index`x'"
	if "`y'"=="dis" local ti "Physical disability index`x'"
	if "`y'"=="dis_7 Survive to 2000" local ti "Survive to 2000`x'"
	if "`y'"=="dis_8 Age at Death" local ti "Age at death`x'"
	if "`y'"=="dis_9 Survive to 2012" local ti "Survive to 2012`x'"
	if "`y'"== "ess_7 Incarceration" local ti "Not incarcerated`x'"


	
	twoway (scatter x2_1 x3_1 ub_3 lb_3 year, ///
			lpattern(solid solid  dash dash ) ///
			lcolor(maroon navy navy navy) ///
			lwidth(medthin medthick medthick medthick) ///
			msymbol(O Sh i i) mcolor(maroon navy navy navy) ///
			c(l l l l)  cmissing( n n n n) ///
			yline(0, lcolor(black)) ), ///
			legend(order(1 "birth yr, survey yr, county FE, county trends" ///
			2 "+ state x birth yr FEs") ///
			rows(2)  ) ///
			xtitle("Age at FS Rollout") ytitle("") ///
			yscale(range(-.02,.04)) ///
			ylabel(-0.02(0.01)0.04) ///
			graphregion(fcolor(white) color(white) icolor(white) lwidth(vthin)) 
	gr save "Figure3a.gph", replace
	gr export "Figure3a.pdf", replace
	gr export "Figure3a.eps", replace
	cap gr export "Figure3a.png", replace


	*Round point estimates and ses
	forvalues x = 1/4 {
		foreach pt in s se {
		
		local round`pt'`x' = round(``pt'`x'', 0.0001)
			
		di ``pt'`x''
		di `round`pt'`x''
		
			local r`pt'`x' = substr("``pt'`x''", 1, strpos("``pt'`x''", ".") - 1) + "0." + ///
				substr("`round`pt'`x''", strpos("`round`pt'`x''", ".") + 1, 4)
			di "`r`pt'`x''"
		
	
			if ``pt'`x'' >= 0 local addl = "	" 
			if ``pt'`x'' < 0 local addl = ""
			
			}
			}
			
	local textpos0 = 0.014
	local textpos1 = `textpos0' - (0.003)
	local textpos2 = `textpos0' - 0.0035
	
	twoway (scatter x3_1 lsr3 year, ///
			lpattern(solid solid) ///
			lcolor(navy maroon ) lwidth(medthick medthin ) ///
			msymbol(Sh i ) mcolor(navy maroon ) ///
			msize(small small) c(l l l l l)  cmissing(n n n n n) ///
			yline(0, lcolor(black)) ) /*(rarea lsr3_ub lsr3_lb year, color(maroon%30))*/, ///
			legend(order(1 "Event study" ///
			2 "Spline") size(small) rows(2) region(lstyle(none)) ) ///
			xtitle("Age at FS Rollout") ytitle("") ///
			yscale(range(-.01,.02)) ///
			ylabel(-0.01(0.005)0.02) ///
			graphregion(fcolor(white) color(white) icolor(white)) ///
			text( `textpos0' 6 "{bf:Spline estimates (SE)}", just(left) ///
						place(e) /*box*/ size(small)) ///
			text( `textpos2' 6	 "-5 to IU:" ///
						"IU to age 5:" ///
						"Age 6 to 11:" ///
						"Age 12 to 17:", just(left) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos2' 8.5 "`addl'`rs1'" ///
						"`addl'`rs2'" ///
						"`addl'`rs3'" ///
						"`addl'`rs4'", just(center) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos2' 12 "(`rse1')" ///
						"(`rse2')" ///
						"(`rse3')" ///
						"(`rse4')", just(center) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos1' 5.9 " " " " " " " " " " " ", /// empty box for outline
			just(left) width(48) ///
			place(e) size(small) bcolor(none) box lcolor(black) ///
			margin(l+1 b+1) )
					
			
	cap gr export "Figure3b.pdf", replace
	cap gr export "Figure3b.eps", replace
	cap gr export "Figure3b.png", replace

	} // End y loop

	} // End g loop
	
	
	
/****************************************************************************/
********************* FIGURE 4 / APPENDIX FIGURES 3, 4, 5 *********************************************
****************************************************************************/
		local omitexp = 10		// Omitted event-time
	local omitdum = 17		// Omitted event-time dummy

*foreach g in All WF WM NF NM {
	foreach g in all wf wm nf nm  {

		*Locals for figure #
		if "`g'" == "all" local fignum = 4
		if "`g'" != "all" local fignum = .

	

	foreach y in z_hc z_ess z_nei z_dis dis_9  ess_7 {
		*Define locals to save graphs with corresponding panels
		if "`g'" == "all" {
			if "`y'" == "z_hc" local panel = "A"
			if "`y'" == "z_ess" local panel = "B"
			if "`y'" == "z_nei" local panel = "C"
			if "`y'" == "z_dis" local panel = "D"
			if "`y'" == "dis_9" local panel = "E"
			if "`y'" == "dis_7" local panel = "F"
			if "`y'" == "ess_7" local panel = "G"
			}
		
		*Figures slightly different for other groups
		if "`g'" != "all" {
			if "`y'" == "z_hc" local fignum = 4
			if "`y'" == "z_ess" local fignum = 5
			if "`y'" == "z_nei" local fignum = 6
			if "`y'" == "z_dis" local fignum = 7
			if "`y'" == "dis_9" local fignum = 8
			if "`y'" == "ess_7" local fignum = 9
			if "`y'" == "dis_7" local fignum = 10
		
		
			if "`g'" == "wm" local panel = "A"
			if "`g'" == "wf" local panel = "B"
			if "`g'" == "nm" local panel = "C"
			if "`g'" == "nf" local panel = "D"
			}


	/* BRING IN OUTREG FILE */
	import excel using "`y'.xls", clear sheet("`g'") cellrange(A3) ///
		firstrow
	replace VARIABLES = subinstr(VARIABLES,"TB_exp_","",.)	
	ren VARIABLES year
	ren EventStudy0 x0
	if "`g'" == "all" {
		ren G x1
		ren H x2
		rename I x3 //this is labeled as event study 0 as well
		}
		else {
			rename E x3
			}
	rename Spline* lsr*

	*Drop FE indicators
	drop if regexm(year, "Birth") == 1 | regexm(year, "Survey") == 1 ///
		| regexm(year, "County") == 1
	quietly destring _all, replace ignore(",")

	/*ADJUST S.E.'S USING ADDED STATS TO THE OUTREG FILE*/
	**make locals out of the number of clusters and the number of regressors to fix the areg dF
	**do it for each "variable" here--ie. specification--in case the obs chnage and because the number of regressors changes
	replace year = "N" if year=="Observations"

	foreach var of varlist x* lsr* {
	
		foreach stat in  Counties k Cells {
		di "This summarizes `stat'"
			sum `var' if year=="`stat'"
			local `stat' = r(mean)
		}
		*areg degrees of freedom adjustment--Cameron and Miller (the if exp=="" part ensures that it only adjusts the s.e.'s)
		replace `var' = `var'*((`Cells'-`k'-1)/(`Cells'-`k'-`Ng'-1)) if year==""
		}	

	
	* Identify betas
	gen b = year!=""
	replace year = year[_n-1] if year==""

	*Create variable for real year 
	rename year stryear
		g year = real(stryear)
	
	compress

	****************
	/*Spline*/
	****************

	*Get point estimates
	preserve
	import excel using "splinem5_p17_`y'.xls", ///
	clear sheet("`g'") cellrange(A2) ///
		firstrow
	rename VARIABLES year
	keep year Model3
	rename Model* lsr*

	keep if substr(year,1,6)=="spline"
	destring _all, replace
	local s1 = lsr3 in 1
	local s2 = lsr3 in 2
	local s3 = lsr3 in 3
	local s4 = lsr3 in 4
	restore

	*Now get SEs
	preserve
	import excel using "splinem5_p17_`y'.xls", ///
	clear sheet("`g'") cellrange(A2) ///
		firstrow
	rename VARIABLES year
	keep year Model3
	rename Model* lsr*
	g b = regexm(year, "spline") == 1
		replace year = year[_n-1] if substr(year[_n-1],1,6)=="spline" & ///
		year == ""

	keep if b==0 & lsr3 != "" & regexm(year, "spline") == 1
	destring _all, replace
	local se1 = lsr3 in 1
	local se2 = lsr3 in 2
	local se3 = lsr3 in 3
	local se4 = lsr3 in 4
	restore

	* Now expand to cover [-5,10], the balanced panel
	
	preserve
	clear
	local obs = 18+5 + 1
	set obs `obs'
	gen year = _n - 7
	summ year, d

	
	gen lsr3 = inrange(year,-5,-2)*(`s1'*(year+1)-7*`s2'-4*`s3') ///
		+ inrange(year,-1,5)*(-`s2'*(6-year)-4*`s3') ///
		+ inrange(year,6,11)*(`s3'*(year-10)) ///
		+ inrange(year,12,18)*(`s3'+`s4'*(year-11))
	
	*Get upper bound of CI
	gen lsr3_ub = (lsr3 + 1.96*`se1')*inrange(year,-5,-2) + ///
		(lsr3 + 1.96*`se2')*inrange(year,-1,5) + ///
		(lsr3 + 1.96*`se3')*inrange(year,6,11) + ///
		(lsr3 + 1.96*`se4')*inrange(year,12,18)

	*Get lower bound of CI
	g lsr3_lb = (lsr3 - 1.96*`se1')*inrange(year,-5,-2) + ///
		(lsr3 - 1.96*`se2')*inrange(year,-1,5) + ///
		(lsr3 - 1.96*`se3')*inrange(year,6,11) + ///
		(lsr3 - 1.96*`se4')*inrange(year,12,18)

	su
	tempfile spline
	save "`spline'", replace
	di "Finished saving spline"
	restore


	/*************************
	Prepare event study coeffs
	*************************/

	keep if substr(stryear,1,6)!="spline"	// Drop spline coefficients and SEs
		*Just keep years
		keep if year != .
	*destring year, replace
	local obs = _N+1
	set obs `obs'						// Add extra observation for omitted category
	replace year = `omitdum' in `obs'	// Assign correct event-time dummy
	foreach var of varlist x* {		// 
		replace `var'=0 in `obs'
	}
	replace b = 1 in `obs'

	replace year = year - (`omitdum' - `omitexp')	// Adjust dummy variable numbers to event time

	/*RESHAPE SO S.E.'S ARE NEXT TO COEFS*/
	for var _all: ren X X_ 
	ren year_ year
	ren b_ b
	reshape wide *_, i(year) j(b)

	/*CONFIDENCE INTERVALS*/
	cap drop lb* ub*


	foreach a in 0 3 {

	gen lb_`a' = x`a'_1 - 1.96*x`a'_0 if year!=`omitexp'
	gen ub_`a' = x`a'_1 + 1.96*x`a'_0 if year!=`omitexp'

	}

	

	* Merge spline onto event-study coefficients
	merge 1:1 year using "`spline'"
	assert _merge!=2
	drop _merge

	su year
	drop if year==`r(min)' | year==`r(max)'		// Drop endpoints


	/********
	Labels
	*********/
	* Subgroups
	if "`g'"=="all" local x ""
	if "`g'"=="wf" local x ", White females"
	if "`g'"=="wm" local x ", White males"
	if "`g'"=="nf" local x ", Nonwhite females"
	if "`g'"=="nm" local x ", Nonwhite males"

	* Outcomes
	if "`y'"=="z_hc" local ti "Human capital index`x'"
	if "`y'"=="z_composite" local ti "Composite index`x'"
	if "`y'"=="z_ess" local ti "Economic self-sufficiency index`x'"
	if "`y'"=="z_nei" local ti "Neighborhood quality index`x'"
	if "`y'"=="z_dis" local ti "Physical disability index`x'"
	if "`y'"=="dis_8 Age at Death" local ti "Age at death`x'"
	if "`y'"=="dis_9" local ti "Survive to 2012`x'"
	if "`y'"== "ess_7" local ti "Not incarcerated`x'"
	if "`y'"=="dis_7" local ti "Survive to 2000`x'"
	local xname = subinstr("`x'", ", ", "", .)
		local xname = strtrim("`xname'")
	
	if "`y'"=="dis_9" & regexm("`x'", "ale") == 1 local ti "`xname'"
	if "`y'"== "ess_7" & regexm("`x'", "ale") == 1  local ti "`xname'"
	if "`y'"== "z_nei" & regexm("`x'", "ale") == 1  local ti "`xname'"


	*For y-scale
	if "`y'" =="z_hc" {
		local yscale = "-0.06, 0.06"
		local ylab = "-0.06(0.02)0.06"

		local yscalealt = "-0.03, 0.03"
		local ylabalt = "-0.03(0.015)0.03"
		local midpt = (0.03-0)/2

		}
	
	if "`y'" == "z_ess" {
		local yscale = "-0.04 0.06"
		local ylab = "-0.04(0.02)0.06"
	
		local yscalealt = "-0.025 0.025"
		local ylabalt = "-0.025(0.01)0.025"
		local midpt = (0.025-0)/2
		}
	
	if "`y'"=="z_nei" {
		local ylab = "-0.04(0.02)0.06"	
		local yscale = "-0.04, 0.06"

		local yscalealt = "-0.04 0.04"
		local ylabalt = "-0.04(0.02)0.04"
		local midpt = (0.04-0)/2
		}

	if "`y'"=="z_dis" {
		local yscale = "-0.03, .03"
		local ylab = "-0.04(0.01).03"
	
		local yscalealt = "-0.02, .01"
		local ylabalt = "-0.02(0.01).01"
		local midpt = (0.01-0)/2
		}

	if "`y'"=="dis_9" { 
		local yscale = "-0.01, 0.008"
		local ylab = "-0.01(0.004)0.008"
	
		local yscalealt = "-0.005, 0.005"
		local ylabalt = "-0.005(0.002)0.005"
		local midpt = (0.005-0)/2
		}
	
	if "`y'"== "ess_7" & (regexm("`g'", "White") == 1 | ///
		regexm("`g'", "Nonwhite females") == 1)  { 
		local yscale = "-0.01, 0.01"
		local ylab = "-0.01(0.005)0.01"
	
		local yscalealt = "-0.01, 0.01"
		local ylabalt = "-0.01(0.005)0.01"
		local midpt = (0.01-0)/2
		}

	if "`y'"=="dis_7" {
		local yscale = "-0.01, 0.01"
		local ylab = "-0.01(0.005)0.01"
	
		local yscalealt = "-0.01, 0.01"
		local ylabalt = "-0.01(0.005)0.01"
		local midpt = (0.01-0)/2
		}

	if "`y'"=="dis_7" & "`g'" == "all" {
		local yscale = "-0.003, 0.003"
		local ylab = "-0.003(0.001)0.003"
	
		local yscalealt = "-0.003, 0.003"
		local ylabalt = "-0.003(0.001)0.003"
		local midpt = (0.003-0)/2
		}

	if ("`y'"=="ess_7" | "`y'"=="dis_9") & "`g'" == "all" {
		local yscale = "-0.003, 0.003"
		local ylab = "-0.003(0.001)0.003"
	
		local yscalealt = "-0.003, 0.003"
		local ylabalt = "-0.003(0.001)0.003"
		local midpt = (0.003-0)/2
		}

	if "`y'"== "ess_7" & regexm("`g'", "onwhite mal") == 1 { 
		local yscale = "-0.06, 0.06"
		local ylab = "-0.06(0.02)0.06"
	
		local yscalealt = "-0.06, 0.06"
		local ylabalt = "-0.06(0.02)0.06"
		local midpt = (0.06-0)/2
		}




	if "`g'" == "all" & "`y'"=="dis_7" & "`y'"== "ess_7" & "`y'"=="dis_7" {
		 local yscale = "-0.02, 0.05"
		local ylab = "-0.02(0.01)0.05"
		local yscalealt = "-0.02, 0.02"
		local ylabalt = "-0.02(0.01)0.02"
		local midpt = (0.02-0)/2
	}

	*Index graphs
	if "`fignum'" == "2" & substr("`y'", 1, 2) == "z_" {
		 local yscale = "-0.04, 0.06"
		local ylab = "-0.04(0.02)0.06"
		local yscalealt = "-0.02, 0.02"
		local ylabalt = "-0.02(0.01)0.02"
		local midpt = (0.02-0)/2
	}

	if `fignum' == 9 & "`panel'" == "C" { 
		local yscale = "-0.06, 0.06"
		local ylab = "-0.06(0.02)0.06"
	
		local yscalealt = "-0.06, 0.06"
		local ylabalt = "-0.06(0.02)0.06"
		local midpt = (0.06-0)/2
		}

	
	/*** GRAPHS ***/
	
	*4 decimals for indices
	*5 decimals for suvival and incarceration
	*4 decimals for indices
	*5 decimals for suvival and incarceration
	if "`y'"=="dis_9" | "`y'"== "ess_7" local num = 5
	if  "`y'"!="dis_9" & "`y'"!= "ess_7" local num = 4
	
	*Round point estimates and ses
	forvalues x = 1/4 {
		foreach pt in s se {
		di ``pt'`x''
			local roundnum = 10^(-`num')
			di "roundnum is `roundnum'"

			local round`pt'`x' = round( ``pt'`x'', `roundnum')

			di "rounded values is `round`pt'`x''"
			
		if length("`round`pt'`x''") < `num' + 2 {
			local round`pt'`x' = "`round`pt'`x''" + "0"
			}

		if length("`round`pt'`x''") < `num' + 2 {
			local round`pt'`x' = "`round`pt'`x''" + "0"
			}

		if length("`round`pt'`x''") < `num' + 2 {
			local round`pt'`x' = "`round`pt'`x''" + "0"
			}	

		if length("`round`pt'`x''") < `num' + 2 {
			local round`pt'`x' = "`round`pt'`x''" + "0"
			}

		if (length("`round`pt'`x''") < `num' + 3 & regexm("`round`pt'`x''", "-") == 1) {
			local round`pt'`x' = "`round`pt'`x''" + "0"
			}

		di " ``pt'`x''"

			local r`pt'`x' = substr("`round`pt'`x''", 1, strpos("`round`pt'`x''", ".") - 1) + "0." + ///
				substr("`round`pt'`x''", strpos("`round`pt'`x''", ".") + 1, `num')
			di "`r`pt'`x''"
			
			}
			}
			
	di "before calculating midpoint"
	*Organize y-placement
	di "Midpoint of yaxis is `midpt'"
		local midpt8 = `midpt'/8
		forvalues x = 1/8 {
			local midpt_minus`x' = `midpt' - `midpt8'*`x'
			local midpt_plus`x' = `midpt' + `midpt8'*`x'
			}
	*Whether text is above/below intercept
	summ x3_1 if year == -4
		local pos = r(mean)
		if `pos' < 0 {
			local minus = ""
			
			local textpos0 = `minus'`midpt_plus6'
			local textpos1 = `minus'`midpt_plus4'
			local textpos2 = `minus'`midpt_plus2'
			local textpos3 =  `minus'`midpt'
			local textpos4 =  `minus'`midpt_minus2'
			}
			
		if `pos' >= 0 {
			local minus = "-"
		
			local textpos0 = `minus'`midpt_minus2' 
			local textpos1 = `minus'`midpt'
			local textpos2 = `minus'`midpt_plus2'
			local textpos3 =  `minus'`midpt_plus4'
			local textpos4 =  `minus'`midpt_plus6'
			
			}
		
	di "This will graph `rs1' `rs2' `rs3' `rs4'"
	di "and se (`rse1') (`rse2') (`rse3') (`rse4')"

		
	
	*With spline CI
	twoway (scatter x3_1 lsr3 year, ///
			lpattern(solid solid) ///
			lcolor(navy maroon ) ///
			lwidth(medthick medthin ) ///
			msymbol(Sh i ) mcolor(navy maroon ) ///
			msize(small small) c(l l l l l)  cmissing(n n n n n) ///
			yline(0, lcolor(black)) ) , ///
			legend(order(1 "Event study" ///
			2 "Spline") size(small)	rows(2) region(lstyle(none)) ) ///
			xtitle("Age at FS Rollout") ytitle("") ///
			title("`ti'") ///
			ysc(r(`yscalealt')) ylab(`ylabalt') ///
			graphregion(fcolor(white) color(white) icolor(white)) ///
			text( `textpos0' -4 "{bf:Spline estimates (SE)}", just(left) ///
						place(e) /*box*/ size(small)) ///
			text( `textpos2' -4	 "-5 to IU:" ///
						"IU to age 5:" ///
						"Age 6 to 11:" ///
						"Age 12 to 17:", just(left) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos2' -.6 "`rs1'" ///
						"`rs2'" ///
						"`rs3'" ///
						"`rs4'", just(center) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos2' 1.7 "(`rse1')" ///
						"(`rse2')" ///
						"(`rse3')" ///
						"(`rse4')", just(center) ///
						place(e) size(small) bcolor(none) /*box lcolor(black)*/ ///
						margin(l+1 t+1 b+1) ) ///
			text( `textpos2' -4.1 " " " " " ", /// empty box for outline
			just(left) width(45) ///
			place(e) size(small) bcolor(none) box lcolor(black) ///
			margin(l+1 t+10 b+0) )
					
	gr save "Figure`fignum'`panel'.gph", replace
	cap gr export "Figure`fignum'`panel'.pdf", replace
	cap gr export "Figure`fignum'`panel'.eps", replace
	cap gr export "Figure`fignum'`panel'.png", replace


	} // End y loop

	} // End g loop

/****************************************************************************/
********************* FIGURE 6 *********************************************
****************************************************************************/


* Prepare Romano Wolf p-values
	cap prog drop subindex
		prog def subindex
			args subi num l1 l2 l3 l4 l5 l6 l7
			cap g subindex = ""
			local label = ""
			forvalues x = 1/7 {
				local label = "`label' " + "`l`x'' "
				}
				local label = strtrim(subinstr("`label'", "  ", " ", .))
			replace subindex = "`label'" if VARIABLES == "z_`subi'_`num'_all"
		end
	

* Tidy program for the Romano-Wolf p-values
cap program drop tidy2
program define tidy2
	args index grp table
	
	*local index z_ess
	*local grp all

	** Import
	import excel "rwolf_`index'_`grp'_subindices.xls", cellrange(A2) /*firstrow*/ clear

	** Transpose
	tostring *, replace
	sxpose, clear
	
	
	
	drop in 1
	rename _var1 VARIABLES
	rename _var2 rw_p_`grp'
	destring rw_p_`grp', replace

	** Clean up

	*Label variables

		subindex ess 1	In labor force
		subindex ess 2	Worked last yr
		subindex ess 3 	# wks worked last yr
		subindex ess 4 	Usual hrs worked/wk
		subindex ess 5	Log labor income
		subindex ess 6	Log non-labor income, excl public source
		subindex ess 8	Log fam income to pov
		subindex ess 9	Not in pov
		subindex ess 10	-1 x Log(public source income)
		/*foreach v in b ub lb {
			replace `v' = `v'*-1 if VARIABLES=="z_ess_10_all"
		}*/

		subindex ess 5_1	Labor income >0
		subindex ess 6_1	Non-labor income excl public source >0
		subindex ess 8_1 	Family income >0
		subindex ess 10_1	No public source income

		subindex hc 1	Yrs schooling
		subindex hc 2 	HS/GED or more
		subindex hc 3	Some college or more
		subindex hc 4	4 yrs college or more
		subindex hc 5	Graduate degree or more
		subindex hc 6	Professional occupation
		subindex nei 1	Log house value
		subindex nei 2	Log gross rent
		subindex nei 3	Home ownership
		subindex nei 4	Single family residence
		subindex nei 5	Log mean fam income to pov (tract)
		subindex nei 6	-1 x teen pregnancy rate (tract)
		subindex nei 7 	-1 x share single HOH (tract)
		subindex nei 8	-1 x child pov rate (tract)
		subindex nei 9	Mean home ownership (tract)
		subindex nei 10	Log median value of home (tract)
		subindex nei 11	Log median gross rent (tract)
		subindex nei 12	Absolute upward mobility (CH)
		subindex nei 1_1 	House value >0
		subindex nei 5_1 Mean fam income (tract) >0
		subindex dis 1	No work disability
		subindex dis 2 	No ambulatory difficulty
		subindex dis 3	No cognitive difficulty
		subindex dis 4	No independent learning difficulty
		subindex dis 5	No vision/hearing difficulty
		subindex dis 6 	No self-care difficulty


	drop VARIABLES
	
end

** Clean all
foreach group in all{
	foreach index in z_ess z_hc z_nei z_dis{
		tidy2 `index' `group'
		tempfile sub`index'`group'
		save `sub`index'`group'', replace
	}
}

* Append indexes
use `subz_essall', clear
foreach index in z_hc z_nei z_dis{
	append using `sub`index'all'
}

order subindex

tempfile rw_p
save `rw_p'


*Bring in file
import excel using "fs_subindices_all.xlsx", firstrow clear ///
	cellrange(A1)

	*Drop what's not needed
	desc *
	*Just keep point estimates and standard errors
	replace A = "b" if regexm(A, "shareFSPage") ==1 
	replace A = "se" if A[_n-1] == "b"
		keep if A == "b" | A == "se" | A == "VARIABLES" | ///
			regexm(A, "State x Birth") == 1

		*Transpose and create upper and lower CIs
		sxpose, clear 
			foreach var of var _all {
			local vname = `var'[1]
				local vname = subinstr("`vname'", " ", "", .) //take out spaces
				rename `var' `vname'
				}
				*Drop first row (variable name)
				drop if _n == 1
					destring _all, replace
				
			replace b = b
		
			*Confidence intervals
			g ub = b+ (1.96*se)
			g lb = b - (1.96*se)
		
	*Label variables
	cap prog drop subindex
		prog def subindex
			args subi num l1 l2 l3 l4 l5 l6 l7
			cap g subindex = ""
			local label = ""
			forvalues x = 1/7 {
				local label = "`label' " + "`l`x'' "
				}
				local label = strtrim(subinstr("`label'", "  ", " ", .))
			replace subindex = "`label'" if VARIABLES == "z_`subi'_`num'_all"
		end
	
subindex ess 1	In labor force
subindex ess 2	Worked last yr
subindex ess 3 	# wks worked last yr
subindex ess 4 	Usual hrs worked/wk
subindex ess 5	Log labor income
subindex ess 6	Log non-labor income, excl public source
subindex ess 8	Log fam income to pov
subindex ess 9	Not in pov
subindex ess 10	-1 x Log(public source income)


subindex ess 5_1	Labor income >0
subindex ess 6_1	Non-labor income excl public source >0
subindex ess 8_1 	Family income >0
subindex ess 10_1	No public source income

subindex hc 1	Yrs schooling
subindex hc 2 	HS/GED or more
subindex hc 3	Some college or more
subindex hc 4	4 yrs college or more
subindex hc 5	Graduate degree or more
subindex hc 6	Professional occupation
subindex nei 1	Log house value
subindex nei 2	Log gross rent
subindex nei 3	Home ownership
subindex nei 4	Single family residence
subindex nei 5	Log mean fam income to pov (tract)
subindex nei 6	-1 x teen pregnancy rate (tract)
subindex nei 7 	-1 x share single HOH (tract)
subindex nei 8	-1 x child pov rate (tract)
subindex nei 9	Mean home ownership (tract)
subindex nei 10	Log median value of home (tract)
subindex nei 11	Log median gross rent (tract)
subindex nei 12	Absolute upward mobility (CH)
subindex nei 1_1 	House value >0
subindex nei 5_1 Mean fam income (tract) >0
subindex dis 1	No work disability
subindex dis 2 	No ambulatory difficulty
subindex dis 3	No cognitive difficulty
subindex dis 4	No independent learning difficulty
subindex dis 5	No vision/hearing difficulty
subindex dis 6 	No self-care difficulty

* Add stars from Romano-Wolf p-values
merge m:1 subindex using `rw_p', keep(1 3) nogen

gen stars = ""
replace stars = "*" if rw_p_all < 0.1
replace stars = "**" if rw_p_all < 0.05
replace stars = "***" if rw_p_all < 0.01
egen subindex_stars = concat(subindex stars)
*replace subindex = subindex_stars
drop subindex_stars //stars	
	
*Just keep components that are in the indices
keep if subindex != ""
drop if subindex == "House value >0" 

*Cut points for each index
	*1-26 = ESS
	*27-44: HCI
	*45-62: DIS
	*63-96 = NEI
g index = "ess" if regexm(VARIABLES, "ess") == 1
foreach sub in hc dis nei {
	replace index = "`sub'" if regexm(VARIABLES, "`sub'") == 1
	}

*Dummy variable for spacing/counting
cap drop order
gen order = .
	local n = 1
	local v1 "hc_1 hc_2 hc_3 hc_4 hc_5 hc_6"
	local v2 "ess_1	ess_2 ess_3 ess_4 ess_5 ess_6 ess_8	ess_9 ess_10"
	local v3 "ess_5_1 ess_6_1 ess_8_1 ess_10_1"
	local v4 "nei_1	nei_2 nei_3	nei_4 nei_5	nei_6 nei_7 nei_8 nei_9 nei_10 nei_11 nei_12"
	local v5 "nei_1_1 nei_5_1"
	local v6 "dis_1	dis_2 dis_3	dis_4 dis_5	dis_6"
	local vars "`v1' `v2' `v3' `v4' `v5' `v6'"
	foreach var in `vars' {
		replace order = `n' if VARIABLES=="z_`var'_all"
		qui sum order
		di "`r(max)'"
		if `r(max)'==`n' local n = `n'+1
	}
sort order StatexBirthYear
drop order
	
foreach index in ess hc dis nei {
cap g n = _n if index == "`index'"
	summ n if n != 1, d
		local max = r(max)
	replace n = (-1)*n + `max' + 1 if index == "`index'"
		summ n, d

	g `index'n = n if index == "`index'"
	labmask `index'n, values(subindex)
	
	* Generate subindex stars
	g `index'n_stars = `index'n
	labmask `index'n_stars, values(stars)
	replace `index'n_stars = . if stars == ""

	cap drop n
	}	
	
	
*Figure: Loop over with and without stateXbirth trends
foreach sb in No Yes {
	if "`sb'" == "No" local panel = "A"
		if "`sb'" == "No" local title = "No state X birth year"
	if "`sb'" == "Yes" local panel = "B"
		if "`sb'" == "Yes" local title = "State X birth year"
	
*Loop over each index for graphs, then combine to 2x2
foreach index in hc ess nei  dis {
*Labels
	if "`index'" == "ess" local ilab = "Econ self-sufficiency"
	if "`index'" == "hc" local ilab = "Human capital"
	if "`index'" == "dis" local ilab = "Disability"
	if "`index'" == "nei" local ilab = "Neighborhood quality"
	
*Get ranges 
	qui summ `index'n if index == "`index'" & StatexBirthYear == "`sb'", d
		local min = r(min)
		local max = r(max)
		
	* Get list of where to put stars 
	levelsof `index'n if stars != "" & index == "`index'" &  StatexBirthYear == "`sb'",  local(str_list) 	
		
	tw dot b `index'n if StatexBirthYear == "`sb'" & index == "`index'", ///
			horizontal color(navy) msize(small) ndots(0)  || ///
		rspike ub lb `index'n if StatexBirthYear == "`sb'" & index == "`index'", ///
			horizontal lcolor(navy)lwidth(narrow) || ///
		rspike b b `index'n_stars if StatexBirthYear == "`sb'" & index == "`index'", ///
			horizontal lcolor(navy)lwidth(narrow) yaxis(2) ///
		title("`ilab'", size(medium)) ytitle("") ///
		ylab(`min'(2)`max',valuelabel labsize(small) angle(horizontal)) ///
		xlab(-0.04(.02)0.04) xsc(r(-0.04 0.04)) ///
		legend( region(lstyle(none))off) xline(0, lcolor(black) lpattern(dash)) ///
		graphregion(fcolor(white) color(white) icolor(white)) 	///
		ytick(`min'(2)`max', axis(2)) ///
		ylab(`str_list', axis(2) valuelabel angle(horizontal) labsize(small)) ///
		ytitle("", axis(2)) xtitle("")
		
		*Save graph for combining
		gr save "Figure6`index'_rwolf.gph", replace
		} //end subindex loop


	
*Combined all graphs together and save final version
*Combine
local c = " " //empty local
foreach index in hc ess nei  dis {
	local c = "`c' " + "Figure6`index'_rwolf.gph "
	}
	di "`c'"
	graph combine `c', col(2) scheme(s1color) /*title("`title'")*/ ///
		saving("Figure6`panel'_rwolf.gph", replace)

cap gr export "Figure6`panel'_rwolf.pdf", replace



	}


/******************************************************************************/
********************* FIGURE 5 / APPENDIX FIGURE 6*********************
/******************************************************************************/



foreach g in all  {
	* For testing
	local g all
	
	if "`g'" == "all" local tlab = " "
	if "`g'" != "all" { //other dem groups
		if substr("`g'", 1, 1) == "w" local r = "White"
		if substr("`g'", 1, 1) != "w" local r = "Non-white"
		if substr("`g'", 2, 1) == "f" local s = "women"
		if substr("`g'", 2, 1) != "f" local s = "men"
	
		local tlab = "`r' `s'"
	
	}

	di "Loop for `tlab'"

	*Panels 
	if "`g'" == "all" local panel = "A"
	if "`g'" == "wm" local panel = "B"
	if "`g'" == "wf" local panel = "C"
	if "`g'" == "nm" local panel = "D"
	if "`g'" == "nf" local panel = "E"


foreach y in z_composite z_hc z_ess z_nei z_dis dis_9 ess_7 { //start file loop
/*
				z_hc: Human capital index (Panel A)
				z_ess: Econ self-sufficiency index (Panel B)
				z_nei: Living quality index (Panel C)
				z_dis: Physical disability index (Panel D) 
				dis_9: Share that lived to 2012 (Panel E) 
				ess_7: Share not incarcerated (Panel F) */


/* BRING IN OUTREG FILE */
if "`y'" != "LE3"{
import excel using "splinem5_p17_`y'.xls", clear sheet("`g'") ///
	cellrange(A2) firstrow
}

if "`y'" == "LE3"{	
import excel using "splinem5_p17_`y'.xls", clear sheet("`g'") ///
	cellrange(A2) firstrow	
	drop in 10
	drop in 10
}

	*Just keep what we need
	rename Model3 Spline3
	keep VARIABLES Spline3
		*just keep beta and CI
	
		replace VARIABLES = VARIABLES[_n-1] + "se" ///
			if VARIABLES == "" & VARIABLES[_n-1 ] != ""
		keep if (regexm(VARIABLES, "spline2_1") == 1 | regexm(VARIABLES, "spline2_2") == 1 | ///
			regexm(VARIABLES, "spline2_3") == 1 | regexm(VARIABLES, "spline2_4") == 1) & Spline3 != ""

		*Rename spline the index
		rename Spline3 `y'
	
	
		*Save tempfile
			tempfile t`y'
				save `t`y'', replace
				} //end file loop
			
*Merge file with all indices together -- start at bottom and work up
use `tess_7', clear
	foreach y in dis_9 z_dis z_nei z_ess z_hc z_composite { //start index merge
	*Make sure everything matches
	merge 1:1 VARIABLES using `t`y'', nogen assert(match)
	} //end index merge

	*Transpose
	rename dis_9 z_dis_9
	rename ess_7 z_ess7
		reshape long z_, i(VARIABLES) j(index) string
		reshape wide z_, i(index) j(VARIABLES) string
		*Rename spline variables
		rename z_* *
		rename spline2_* spline*
	destring _all, replace

	*Label variables
	g n = 1 if regexm(index, "ess7") == 1 
	replace n = 2 if regexm(index, "dis_9") == 1 & n == .
	replace n = 3 if regexm(index, "dis") == 1 & n == .
	replace n = 4 if regexm(index, "nei") == 1 & n == .
	replace n = 5 if regexm(index, "ess") == 1 & n == .
	replace n = 6 if regexm(index, "hc") == 1 & n == .
	replace n = 7 if regexm(index, "composite") == 1 & n == .

	g label = "Composite index" if regexm(index, "composite") == 1
	replace label = "Human capital index" if regexm(index, "hc") == 1
	replace label = "Econ self-sufficiency index" if regexm(index, "ess") == 1
	replace label = "Neighborhood quality index" if regexm(index, "nei") == 1
	replace label = "Disability index" if regexm(index, "dis") == 1
	replace label = "Survive to 2012" if regexm(index, "dis_9") == 1
	replace label = "Not incarcerated" if regexm(index, "ess7") == 1

	labmask n, values(label)

	sort n //make sure everything's in correct order

	
	forvalues x = 1/4 {
		replace spline`x' = spline`x'*-1
		}

	*Get upper and lower bounds of CI
	forvalues x = 1/4 { //loop over specs
	g ub`x' = spline`x' + 1.96*spline`x'se
	g lb`x' = spline`x' - 1.96*spline`x'se
	}

	summ n, d
		local min = r(min)
		local max = r(max)
	
*Expand so slightly jiggered
expand 4
bys n: g splineno = _n 
	tab splineno //make sure this worked
	forvalues x = 1/4 {
	foreach v in spline ub lb {
		replace `v'`x' = . if splineno != `x'
		} //var
		} //spline #
	
	*slightly modify n
	replace n = n + (3-splineno)*0.18
		tab n


***Version with pre-treatment
*Indices other than "dis_9" "ess7"
preserve 
	keep if n >= 2.6
	tw dot spline1 n if splineno == 1, horizontal color(teal) ndots(0)  || ///
		rspike ub1 lb1 n if splineno == 1, horizontal lcolor(teal) lwidth(narrow) lpattern(dash) || ///
		dot spline2 n if splineno == 2, horizontal color(navy) ndots(0) msymbol(T)  || ///
		rspike ub2 lb2 n if splineno == 2, horizontal lcolor(navy)lwidth(narrow) || ///
		dot spline3 n if splineno == 3, horizontal color(maroon) ndots(0) msymbol(Dh)  || ///
		rspike ub3 lb3 n if splineno == 3, horizontal lcolor(maroon)lwidth(narrow) || ///
		dot spline4 n if splineno == 4, horizontal color(green) ndots(0) msymbol(X)  || ///
		rspike ub4 lb4 n if splineno == 4, horizontal lcolor(green)lwidth(narrow) ///
		title("`tlab'", size(large)) ytitle("") scale(0.8) ///
		xlab(-0.002(0.001)0.005) xsc(r(-0.002, 0.005)) ///
		ylab(3(1)`max',valuelabel labsize(small) angle(horizontal)) ///
		legend( region(lstyle(none))order(1 3 5 7) label(1 "-5 to -2") label(3 "-1 to 5") ///
			label(5 "6 to 11") label(7 "12 to 17") rows(1)) ///
		xline(0, lcolor(black) lpattern(dash)) ysize(4.5) xsize(5.5) ///
		graphregion(fcolor(white) color(white) icolor(white)) //fxsize(100)
		
gr save "Figure5_splineall`panel'_parta.gph", replace
cap gr export "Figure5_splineall`panel'_parta.pdf", replace
cap gr export "Figure5_splineall`panel'_parta.eps", replace
restore

*Indices with  "dis_9" "ess7"
preserve 
	keep if n <= 2.4
	tw dot spline1 n if splineno == 1, horizontal color(teal) ndots(0)  || ///
		rspike ub1 lb1 n if splineno == 1, horizontal lcolor(teal) lwidth(narrow) lpattern(dash) || ///
		dot spline2 n if splineno == 2, horizontal color(navy) ndots(0) msymbol(T) || ///
		rspike ub2 lb2 n if splineno == 2, horizontal lcolor(navy)lwidth(narrow) || ///
		dot spline3 n if splineno == 3, horizontal color(maroon) ndots(0) msymbol(Dh)  || ///
		rspike ub3 lb3 n if splineno == 3, horizontal lcolor(maroon)lwidth(narrow) || ///
		dot spline4 n if splineno == 4, horizontal color(green) ndots(0)  msymbol(X) || ///
		rspike ub4 lb4 n if splineno == 4, horizontal lcolor(green)lwidth(narrow) ///
		title(" ", size(medium)) ytitle("") ysize(2) xsize(5.5) scale(2.25) ///
		/*xlab(-0.0004(0.0002)0.0004) xsc(r(-0.0004 0.0004))*/ ///
		xlab(-0.04(0.02)0.04) xsc(r(-0.04 0.04)) ///
		ylab(1(1)2,valuelabel labsize(small) angle(horizontal)) ///
		legend( region(lstyle(none))order(1 3 5 7) label(1 "-5 to -2") label(3 "-1 to 5") ///
			label(5 "6 to 11") label(7 "12 to 17") rows(1)) ///
		xline(0, lcolor(black) lpattern(dash)) ///
		graphregion(fcolor(white) color(white) icolor(white)) 
gr save "Figure5_splineall`panel'_partb.gph", replace
cap gr export "Figure5_splineall`panel'_partb.pdf", replace
cap gr export "Figure5_splineall`panel'_partb.eps", replace
restore





cap log close


cap log close
}
