* Event Study Revisions: create versions of the event study with 
* an extend time horizon on x-axis and with 3-year age bins
* 5/5/2021 Kate Moulton

clear all 
set more off 
set trace on
set tracedepth 2 
*set matsize 11000
set scheme s1color
	
capture program drop main 
program define main 
	paths
	cap log close 
	log using "$output/eventstudy_RESTUD.log", replace

	*** Exposure, Event study, and spline models ***
	* Exposure model data 
	local data1 "$datatemp/fs_hsacr_1_newsub_exactmatch0.dta"
	local stayerData1 "$datatemp/fs_hsacr_1_newsub_exactmatch0_withStayers.dta" 
	* Eventstudy and spline data 
	local data2 "$datatemp/fs_dtacr_2_exactmatch0_newsub.dta"		
	local stayerData2 "$datatemp/fs_dtacr_2_exactmatch0_newsub_withStayers.dta"
	* Sample sets what percent of the data to use
	local sample = 100

	*** Extended event study (age -7 to 20) *** 	
	* tables for index outcomes
	foreach o in z_composite z_ess z_hc z_dis z_nei{
		eventStudyExtendedSpecial, dataEvent(`data2') sample(`sample') indexType("`o'")
	}
	
	*outcomes for special subindices 
	foreach o in dis_9 ess_7{
		eventStudyExtendedSpecial, dataEvent(`data2') sample(`sample') indexType("`o'")
	}
	
	* outcomes for life expectancy
	foreach o in LE3{
		eventStudyExtendedLE, dataEvent(`stayerData2') indexType("`o'") sample(`sample') 
	}
	
	*** Binned event study (3 year bins -5 - 18, omitting bin 10-12) ***
	* tables for index outcomes
	foreach o in z_composite z_ess z_hc z_dis z_nei{
		eventStudyBins, dataEvent(`data2') sample(`sample') indexType("`o'")
	}
	
	*outcomes for special subindices 
	foreach o in dis_9 ess_7{
		eventStudyBinsSpecial, dataEvent(`data2') sample(`sample') indexType("`o'")
	}	
	
	* outcomes for life expectancy
	foreach o in LE3{
		eventStudyBinsLE, dataEvent(`stayerData2') indexType("`o'") sample(`sample') 
	}

end 

capture program drop paths
program define paths

	global datatemp"/projects/programs/foodstamps/master_folder/datatemp"
	global output "/projects/programs/foodstamps/master_folder/RESTUD_2021/output/eventstudy"
	global data "/projects/data"		
	global countydata "/projects/data/countyLevel"
	global empData "/projects/programs/rwalker/chris/dataSTATA"

end 

/******************************************************************************
* eventStudyExtended
*
* Produces tables for extended event-study models. 
* 
*******************************************************************************/ 

capture program drop eventStudyExtended
program define eventStudyExtended
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]
	
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create extended event time dummies: use age 10 as omitted category
	gen exp_ext = exp2
	replace exp_ext = -8 if exp_ext < -8
	replace exp_ext = 21 if exp_ext > 21 & exp_ext < .
	char 	exp_ext[omit] 10
	xi, 	prefix(TBe_) i.exp_ext

	tabstat TBe_*, by(exp_ext)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)
	
	* Check if balanced
		bysort fips: gen count_fips = _n == 1
		forvalues i = 1/18{
		bysort fips: egen any_TBe_exp_ext_`i' = max(TBe_exp_ext_`i')
		tab any_TBe_exp_ext_`i' if count_fips == 1
		}
		
		forvalues i = 20/30{
		bysort fips: egen any_TBe_exp_ext_`i' = max(TBe_exp_ext_`i')
		tab any_TBe_exp_ext_`i' if count_fips == 1
		}
		
		forvalues i = 1/16{
		bysort fips: egen any_TB_exp_`i' = max(_TB_exp_`i')
		tab any_TB_exp_`i' if count_fips == 1
		}
	
	foreach g in all wm wf nm nf{
		* Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TBe_*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext* if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust), 100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6 
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8 // test whether exposure -4 to -1 are = 0
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
			local r "append"
		}
		
		reghdfe `outcome' TBe_exp_ext* if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust), 100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		local r "append"
		
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext* `trends'  if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust), 100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		
		}
		
		*/
		
		reghdfe `outcome' TBe_exp_ext* `trends' if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year  ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', Counties , `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta

		restore
	}

	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_extended_dta, clear 
		export excel using $output/`indexType'_extended, sheet("`g'", replace )  
	}
	
end 


/******************************************************************************
* eventStudyExtendedSpecial
*
* Similar to previous subprogram but specific to a specific set of indices. 
*
*******************************************************************************/
capture program drop eventStudyExtendedSpecial
program define eventStudyExtendedSpecial
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]
	
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create extended event time dummies: use age 10 as omitted category
	gen exp_ext = exp2
	replace exp_ext = -8 if exp_ext < -8
	replace exp_ext = 21 if exp_ext > 21 & exp_ext < .
	char 	exp_ext[omit] 10
	xi, 	prefix(TBe_) i.exp_ext

	tabstat TBe_*, by(exp_ext)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)	
	
	* Some modifications to certain indices 
	if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
		* These observations are at the county/birth-month level. No survey-year variation.
		* So get rid of the extra survey years of data
		egen tag = tag(fips birthyr  exactmatch)
		tab birthyr tag, m
		tab year tag, m
		keep if tag==1 & exactmatch==0
		drop tag
	} 
	
	
	foreach g in all wm wf nm nf{
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		if "`indexType'"=="ess_7" local obs "obs_inc_`g'"

		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TBe_* spline1_1 spline1_2 spline1_3)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Calculate mean for 1950-54
		sum `outcome' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1
		local mdv5054 = r(mean)
		local sd5054 = r(sd)
		
		* Calculate mean for 1950-80
		sum `outcome' [aw=`obs'] if sample==1
		local mdv = r(mean)
		local sd = r(sd)
		local nobs = r(sum_w)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext_* if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year  ) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
			local r "append"
		}
		
		reghdfe `outcome' TBe_exp_ext_*  if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips)
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		
		local r "append"
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext_* `trends'  if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster nobs nor2 ///
				keep(*exp*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		
		}	
		*/
		
		reghdfe `outcome' TBe_exp_ext_* `trends'  if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta

		restore
	}

	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_extended_dta, clear 
		export excel using $output/`indexType'_extended, sheet("`g'", replace )  
	}
	
end


/******************************************************************************
* eventStudyExtendedLE
*
* Produces tables for extended event-study models, adapted for the life expectancy estimates
* 
*
*******************************************************************************/ 

capture program drop eventStudyExtendedLE
program define eventStudyExtendedLE
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]
	
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create extended event time dummies: use age 10 as omitted category
	gen exp_ext = exp2
	replace exp_ext = -8 if exp_ext < -8
	replace exp_ext = 21 if exp_ext > 21 & exp_ext < .
	char 	exp_ext[omit] 10
	xi, 	prefix(TBe_) i.exp_ext

	tabstat TBe_*, by(exp_ext)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)
	
	* Check if balanced
		bysort fips: gen count_fips = _n == 1
		forvalues i = 1/18{
		bysort fips: egen any_TBe_exp_ext_`i' = max(TBe_exp_ext_`i')
		tab any_TBe_exp_ext_`i' if count_fips == 1
		}
		
		forvalues i = 20/30{
		bysort fips: egen any_TBe_exp_ext_`i' = max(TBe_exp_ext_`i')
		tab any_TBe_exp_ext_`i' if count_fips == 1
		}
		
		forvalues i = 1/16{
		bysort fips: egen any_TB_exp_`i' = max(_TB_exp_`i')
		tab any_TB_exp_`i' if count_fips == 1
		}
	
	foreach g in all wm wf nm nf{
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		*** LE weights using cohort size
		if "`indexType'_`g'"=="LE1_`g'" local obs "obs_LE1_`g'"
		if "`indexType'_`g'"=="LE2_`g'" local obs "obs_LE2_`g'"
		if "`indexType'_`g'"=="LE3_`g'" local obs "obs_LE3_`g'"
		if "`indexType'_`g'"=="LE4_`g'" local obs "obs_LE4_`g'"
		if "`indexType'_`g'"=="LE4_`g'"{
			drop if LE4_all>100
			}
		if "`indexType'_`g'"=="LE_chetty`g'" local obs "obs_dis_`g'"
		
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TBe_*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext* if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6 
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8 // test whether exposure -4 to -1 are = 0
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
			local r "append"
		}
		
		reghdfe `outcome' TBe_exp_ext* if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		local r "append"
		
		if "`g'" == "all"{
			reghdfe `outcome' TBe_exp_ext* `trends'  if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
			outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		}
		*/
		
		reghdfe `outcome' TBe_exp_ext* `trends' if exp_ext!=. & sample==1 [pw=`obs'], absorb(fips state_year year  ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TBe_exp_ext_5 TBe_exp_ext_6 TBe_exp_ext_7 TBe_exp_ext_8
		outreg2 using "$output/table_`outcome'_extended", `r' noparen noaster noobs nor2 ///
				keep(*exp*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', Counties , `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
				

		restore
	}

	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_extended_dta, clear 
		export excel using $output/`indexType'_extended, sheet("`g'", replace )  
	}
	
end 


********************************************************************************
* eventStudyBins
*
* Produces tables for event-study models with 3 year bins
* 
********************************************************************************

capture program drop eventStudyBins
program define eventStudyBins
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]

	
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create binned event time dummies: use age 10-12 as omitted category
	gen TB_bin_1 = exp2 < -5
	gen TB_bin_2 = inrange(exp2, -5, -3)
	gen TB_bin_3 = inrange(exp2, -2, 0)
	gen TB_bin_4 = inrange(exp2, 1, 3)
	gen TB_bin_5 = inrange(exp2, 4, 6)
	gen TB_bin_6 = inrange(exp2, 7, 9)
	gen TB_bin_8 = inrange(exp2, 13, 15)
	gen TB_bin_9 = inrange(exp2, 16, 18)
	gen TB_bin_10 = exp2 > 18 & exp2 != .

	tabstat TB_bin*, by(exp2)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)	
	
	foreach g in all wm wf nm nf{
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		
		* Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' ///
		lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TB_bin*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			return list
			ereturn list
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta	
			local r "append"
		}
		
		reghdfe `outcome' TB_bin* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
				
		local r "append"
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		}	
		*/
		
		reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year  ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', Counties , `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta

		restore
	}

	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_bins_dta, clear 
		export excel using $output/`indexType'_bins, sheet("`g'", replace )  
	}
	
end 

/******************************************************************************
* eventStudyBinsSpecial 
*
* Similar to previous subprogram but specific to a specific set of indices. 
*
*******************************************************************************/
capture program drop eventStudyBinsSpecial 
program eventStudyBinsSpecial 
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]
		
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create binned event time dummies: use age 10 as omitted category
	gen TB_bin_1 = exp2 < -5
	gen TB_bin_2 = inrange(exp2, -5, -3)
	gen TB_bin_3 = inrange(exp2, -2, 0)
	gen TB_bin_4 = inrange(exp2, 1, 3)
	gen TB_bin_5 = inrange(exp2, 4, 6)
	gen TB_bin_6 = inrange(exp2, 7, 9)
	gen TB_bin_8 = inrange(exp2, 13, 15)
	gen TB_bin_9 = inrange(exp2, 16, 18)
	gen TB_bin_10 = exp2 > 18 & exp2 != .

	tabstat TB_bin*, by(exp2)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)	
	
	* Some modifications to certain indices 
	if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
		* These observations are at the county/birth-month level. No survey-year variation.
		* So get rid of the extra survey years of data
		egen tag = tag(fips birthyr  exactmatch)
		tab birthyr tag, m
		tab year tag, m
		keep if tag==1 & exactmatch==0
		drop tag
	} 
	
	
	foreach g in all wm wf nm nf{
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		if "`indexType'" == "ess_7" local title "Share Not Incarcerated, `g'"
		
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		if "`indexType'"=="ess_7" local obs "obs_inc_`g'"

		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TB_bin* spline1_1 spline1_2 spline1_3)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Calculate mean for 1950-54
		sum `outcome' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1
		local mdv5054 = r(mean)
		local sd5054 = r(sd)
		
		* Calculate mean for 1950-80
		sum `outcome' [aw=`obs'] if sample==1
		local mdv = r(mean)
		local sd = r(sd)
		local nobs = r(sum_w)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year  ) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
			local r "append"
		}
		reghdfe `outcome' TB_bin_* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips)
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		local r "append"
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster nobs nor2 ///
				keep(TB_bin*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		
		}
		*/
		
		reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', mdv,`mdv',sd,`sd',mdv5054,`mdv5054',sd5054,`sd5054', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		
		restore
	}

	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_bins_dta, clear 
		export excel using $output/`indexType'_bins, sheet("`g'", replace )  
	}
	
end


/********************************************************************************
* eventStudyBinsLE

* Produces tables for event-study models with 3 year bins, adapted for the Life Expectancy estimates

********************************************************************************/

capture program drop eventStudyBinsLE
program define eventStudyBinsLE
syntax, [dataEvent(string)  sample(int 100) indexType(string) ]

	
	*** Estimate event-study and spline models using other data 
	use if runiform() <= `sample'/100 using `dataEvent', clear 	 
	keep if inrange(birthyr,1950,1980)
	
	* Create binned event time dummies: use age 10 as omitted category
	gen TB_bin_1 = exp2 < -5
	gen TB_bin_2 = inrange(exp2, -5, -3)
	gen TB_bin_3 = inrange(exp2, -2, 0)
	gen TB_bin_4 = inrange(exp2, 1, 3)
	gen TB_bin_5 = inrange(exp2, 4, 6)
	gen TB_bin_6 = inrange(exp2, 7, 9)
	gen TB_bin_8 = inrange(exp2, 13, 15)
	gen TB_bin_9 = inrange(exp2, 16, 18)
	gen TB_bin_10 = exp2 > 18 & exp2 != .

	tabstat TB_bin*, by(exp2)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)	
	
	foreach g in all wm wf nm nf{
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		*** LE weights using cohort size
		if "`indexType'_`g'"=="LE1_`g'" local obs "obs_LE1_`g'"
		if "`indexType'_`g'"=="LE2_`g'" local obs "obs_LE2_`g'"
		if "`indexType'_`g'"=="LE3_`g'" local obs "obs_LE3_`g'"
		if "`indexType'_`g'"=="LE4_`g'" local obs "obs_LE4_`g'"
		if "`indexType'_`g'"=="LE4_`g'"{
			drop if LE4_all>100
			}
		if "`indexType'_`g'"=="LE_chetty`g'" local obs "obs_dis_`g'"
		
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' ///
		lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 TB_bin*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Run event study models	
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 
		
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		
		/*
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			return list
			ereturn list
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 0) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"No") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta	
			local r "append"
		}
		
		reghdfe `outcome' TB_bin* if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 1) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"No") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
				
		local r "append"
		if "`g'" == "all"{
			reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips birthyr year) vce(cluster fips) 
			local cells = round(`e(N)', 1000)
			local R2 = round(`e(r2)', .0001)
			local counties = round(e(N_clust),100)
			*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
			test TB_bin_2 TB_bin_3
			outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 2) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"Yes", Birth month, "No", Survey year,"Yes", ///
				State x Birth Year,"No",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		}
		*/
		
		reghdfe `outcome' TB_bin* `trends'  if exp2 !=. & sample==1 [pw=`obs'], absorb(fips state_year year  ) vce(cluster fips) 
		local cells = round(`e(N)', 1000)
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust),100)
		*test TB_exp_3 TB_exp_4 TB_exp_5 TB_exp_6
		test TB_bin_2 TB_bin_3
		outreg2 using "$output/table_`outcome'_bins", `r' noparen noaster noobs nor2 ///
				keep(TB_bin*) ctitle(Model 3) ///
				addstat(Cells, `cells', R squared, `R2', Counties , `counties', k, `e(df_m)',F,`r(F)',df,`r(df)',p,`r(p)',nobs,`nobs') ///
				addtext(Birth year,"No", Birth month, "No", Survey year,"Yes",  ///
				State x Birth Year,"Yes",County Char Trends,"Yes") ///
				dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta

		restore
	}
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/table_`indexType'_`g'_bins_dta, clear 
		export excel using $output/`indexType'_bins, sheet("`g'", replace )  
	}
	
end 


main

log close 
