* Split counties into "early", "middle", and "late" adopters and use the raw data to plot long-run outcomes for each of these groups separately. Additionally, run the exposure model separately for early and late adopters.

clear all 
set more off 
set trace on
set tracedepth 2 
*set matsize 11000
set scheme s1color

	
capture program drop main 
program define main 
	paths

	*** Exposure, Event study, and spline models ***
	* Exposure model data 
	local data1 "$datatemp/fs_hsacr_1_newsub_exactmatch0.dta"
	* Eventstudy and spline data 
	local data2 "$datatemp/fs_dtacr_2_exactmatch0_newsub.dta"
	local sample = 100
	
	*Characterize the counties into early, middle, late adopter
	cap log close 
	log using "$output/make_adopter_groups.log", replace
	make_adopter_groups, dataExposure(`data1') dataEvent(`data2')
	log close
	
	* Exposure model data with adopters
	local data3 "$datatemp/fs_hsacr_1_newsub_exactmatch0_adopters.dta"
	* Eventstudy and spline data with adopters
	local data4 "$datatemp/fs_dtacr_2_exactmatch0_newsub_adopters.dta"
	
	cap log close 
	log using "$output/figures/plot_by_adopter.log", replace
	foreach o in z_composite z_ess z_hc z_dis z_nei{
		PlotByAdopter, indexType("`o'")
	}
	
	foreach o in ess_7 dis_9{
		PlotByAdopterSpecial, indexType("`o'")
	}
	
	* Combine means to create excel sheet for export
	use "$output/figures/z_composite_all_data.dta", clear
	replace z_composite_all = round(z_composite_all, .0001)
	export excel using "$output/figures/z_composite_adopter_group_means.xls", firstrow(variables) replace keepcellfmt
	
	foreach o in z_ess z_hc z_dis z_nei ess_7 dis_9{
		merge 1:1 birthyr early_adopt mid_adopt late_adopt using "$output/figures/`o'_all_data.dta", nogen
		replace `o' = round(`o', .0001)
	}
	export excel using "$output/figures/adopter_group_means.xls", firstrow(variables) replace keepcellfmt
	log close
	
	/* Exposure DD model: stratified and using interaction term model
	cap log close 
	log using "$output/exposure/adopter_groups_DD.log", replace
	foreach o in z_composite z_ess z_hc z_dis z_nei{
		ExposureByAdopter, indexType("`o'") dataExposure(`data3') sample(`sample')
	}
	
	foreach o in ess_7 dis_9{
		ExposureByAdopterSpecial, indexType("`o'") dataExposure(`data3') sample(`sample')
	}
	log close
	*/

end 

capture program drop paths
program define paths

	global datatemp"/projects/programs/foodstamps/master_folder/datatemp"
	global output "/projects/programs/foodstamps/master_folder/RESTUD_2021/output/adopter_groups"
	global data "/projects/data"		// project-wide data 
	global countydata "/projects/data/countyLevel"
	global empData "/projects/programs/rwalker/chris/dataSTATA"


end 

capture program drop make_adopter_groups
program make_adopter_groups
syntax, [dataExposure(string) dataEvent(string)]

	*local dataEvent "$datatemp/fs_dtacr_2_exactmatch0_newsub.dta"
	use `dataEvent', clear 
	gen fips = st_fips*1000+co_fips_1
	
	* Restrict data to z_composite analysis sample
	local indexType z_composite
	
	foreach g in all /*wm wf nm nf*/{
		* Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		*preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 _TB_exp_*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
	}
	
	keep if sample==1
	
	* Identify number of analysis observations in each county x birth year
	bysort fips birthyr: egen num_obs = total(obs_all)
	
	* Work with relevant relevant variables
	keep fips fs_year_1 birthyr num_obs
	sort fips fs_year_1 birthyr num_obs
	duplicates drop
	
	* Use population size from year (or closest year) food stamps was introduced
	* keep if birthyr == fs_year_1 // not all counties have a cohort from their FS year
	gen diff = birthyr - fs_year_1
	bysort fips: egen max_diff = max(diff)
	replace diff = . if diff < 0 & max_diff >= 0 // ignore if there are cohorts after FS
	replace diff = abs(diff)
	bysort fips: egen min_diff = min(diff)
	tab min_diff
	keep if diff == min_diff
	isid fips
	
	* Generate population and share variables
	sort fs_year_1 birthyr fips
	egen total_pop = total(num_obs)
	bysort fs_year_1: egen new_exposed_pop = total(num_obs)
	gen new_exposed_share = new_exposed_pop / total_pop
	
	* Keep only relevant variables
	keep fs_year_1 new_exposed_share total_pop
	duplicates drop
	gen exposed_share = sum(new_exposed_share) 
		
	* Identify years when 1/3 and 2/3 of total analysis population has been exposed to foodstamps
	gen dist_1third = abs(exposed_share - (1/3))
	egen min1 = min(dist_1third)
	gen closest_cohort_1third = dist_1third == min1
	gen early_year = fs_year_1 if closest_cohort_1third == 1
	egen early_time = min(early_year)
	
	gen dist_2third = abs(exposed_share - (2/3))
	egen min2 = min(dist_2third)
	gen closest_cohort_2third = dist_2third == min2
	gen late_year = fs_year_1 if closest_cohort_2third == 1
	egen late_time = min(late_year)
	
	* Identify year when median share of population has been exposed
	sum exposed_share, detail
	local median_share = `r(p50)'
	gen dist_median = abs(exposed_share - `median_share')
	egen min3 = min(dist_median)
	gen closest_cohort_median = dist_median == min3
	gen median_year = fs_year_1 if closest_cohort_median == 1
	egen median_time = min(median_year)
		
	* Check which year have been picked
	keep fs_year_1 exposed_share early_time late_time median_time
	tab early_time 
	tab late_time 
	sum exposed_share if fs_year_1 <= early_time
	sum exposed_share if fs_year_1 >= late_time
	
	tab median_time
	sum exposed_share if fs_year_1 <= median_time
	sum exposed_share if fs_year_1 > median_time
		
	* Store cutoffs as locals
	sum early_time
	local early_t = `r(mean)'
	sum late_time
	local late_t = `r(mean)'
	sum median_time
	local median_t = `r(mean)'
	
	* Add adopter group indicators to event study dataset
	use "`dataEvent'", clear 
	
	gen early_adopt = fs_year_1 <= `early_t'
	gen mid_adopt = inrange(fs_year_1, `early_t' + 1, `late_t')
	gen late_adopt = fs_year_1 > `late_t'
	
	tab early_adopt
	tab mid_adopt
	tab late_adopt
	
	tab fs_year_1 if early_adopt == 1
	tab fs_year_1 if mid_adopt == 1
	tab fs_year_1 if late_adopt == 1
	
	save "$datatemp/fs_dtacr_2_exactmatch0_newsub_adopters.dta", replace
	
	* Record which counties are in which adopter group
	use "$datatemp/fs_dtacr_2_exactmatch0_newsub_adopters.dta", clear
	gen fips = st_fips*1000+co_fips_1
	keep year st_fips co_fips fips early_adopt mid_adopt late_adopt
	duplicates drop
	rename st_fips STATE
	rename co_fips COUNTY
	rename year Year 
	merge 1:1 STATE COUNTY Year using "$datatemp/county_population_estimates.dta", keep(1 3) keepusing(Persons state_name county_name)
	order state_name county_name STATE COUNTY fips Year Persons early_adopt mid_adopt late_adopt
	bysort fips: egen State_name = mode(state_name)
	bysort fips: egen County_name = mode(county_name)
	drop state_name county_name _merge
	
	*keep st_fips co_fips fips early_adopt mid_adopt late_adopt
	*duplicates drop
	export excel using "$output/adopter_group_fips.xls", firstrow(variables) replace keepcellfmt
	
	* Add adopter group indicators to exposure dataset
	use "`dataExposure'", clear 
	
	* Here the early adopter group is just cohorts born before median exposure
	gen early_adopt = fs_year_1 <= `median_t'
	tab early_adopt
	tab fs_year_1 if early_adopt == 1
	
	la var early_adopt "Adopted Food Stamps before median population exposure"
	
	save "$datatemp/fs_hsacr_1_newsub_exactmatch0_adopters.dta", replace
	
end

capture program drop PlotByAdopter
program PlotByAdopter
syntax, [indexType(string)]

	use "$datatemp/fs_dtacr_2_exactmatch0_newsub_adopters.dta", clear 
	keep if inrange(birthyr,1950,1980)
	
	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)
	
	* Isolate to our analysis sample
	foreach g in all /*wm wf nm nf*/{
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		if "`indexType'" == "ess_7" local title "Share Not Incarcerated, `g'"
		
		* Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		*preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60  _TB_exp_*)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
	
	keep if sample==1
	
	* Find mean outcomes for each birth year cohort separately for each group
	preserve
	
	collapse `outcome', by(birthyr early_adopt mid_adopt late)
	
	graph twoway (line `outcome' birthyr if early_adopt == 1) ///
	(line `outcome' birthyr if mid_adopt == 1) ///
	(line `outcome' birthyr if late_adopt == 1), ///
	legend(label(1 "Early Adopter (< 1967)") label(2 "Middle Adopter") label(3 "Late Adopter (> 1969)")) ///
	ytitle("Mean `title'") xtitle("Birth Cohort")
	graph export "$output/figures/`indexType'_`g'_by_adopter_group.pdf", replace
	
	*replace `outcome' = round(`outcome', .0001)
	save "$output/figures/`outcome'_data.dta", replace
	
	restore
	
	if "`indexType'" == "z_composite"{
	preserve
	* Regress composite index on county FE, state x year FE, and linear trends X 1960 county chars
	local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
	reghdfe `outcome' `trends', absorb(fips state_year) resid(`outcome'_resid)
	
	* Take residuals, then plot for each of the three adopter groups
	collapse `outcome'_resid, by(birthyr early_adopt mid_adopt late)
	graph twoway (line `outcome'_resid birthyr if early_adopt == 1) ///
	(line `outcome'_resid birthyr if mid_adopt == 1) ///
	(line `outcome'_resid birthyr if late_adopt == 1), ///
	legend(label(1 "Early Adopter (< 1967)") label(2 "Middle Adopter") label(3 "Late Adopter (> 1969)")) ///
	ytitle("Mean Residuals `title'") xtitle("Birth Cohort")
	graph export "$output/figures/residual_`indexType'_`g'_by_adopter_group.pdf", replace
	
	restore
	}
	
	
	}
	
end

/******************************************************************************
* PlotByAdopterSpecial 
*
* Similar to previous subprogram but specific to a specific set of indices. 
*
*******************************************************************************/
capture program drop PlotByAdopterSpecial
program PlotByAdopterSpecial
syntax, [indexType(string)]
	
	use "$datatemp/fs_dtacr_2_exactmatch0_newsub_adopters.dta", clear 
	keep if inrange(birthyr,1950,1980)
	
	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)
	
	* Some modifications to certain indices
	if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
		* These observations are at the county/birth-month level. No survey-year variation.
		* So get rid of the extra survey years of data
		egen tag = tag(fips birthyr  exactmatch)
		tab birthyr tag, m
		tab year tag, m
		keep if tag==1 & exactmatch==0
		drop tag
	} 
	
	* Isolate to our analysis sample
	foreach g in all /*wm wf nm nf*/{
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		if "`indexType'" == "ess_7" local title "Share Not Incarcerated, `g'"
		
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		if "`indexType'"=="ess_7" local obs "obs_inc_`g'"

		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"
		* Identify sample for event study/spline models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60  _TB_exp_* spline1_1 spline1_2 spline1_3)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1	
		local mdv = r(mean)
		
		* Calculate mean for 1950-54 // special
		sum `outcome' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1
		local mdv5054 = r(mean)
		local sd5054 = r(sd)
		
		* Calculate mean for 1950-80
		sum `outcome' [aw=`obs'] if sample==1
		local mdv = r(mean)
		local sd = r(sd)
		local nobs = r(sum_w)
	
	keep if sample==1
	
	* Find mean outcomes for each birth year cohort separately for each group
	collapse `outcome', by(birthyr early_adopt mid_adopt late)
	
	graph twoway (line `outcome' birthyr if early_adopt == 1) ///
	(line `outcome' birthyr if mid_adopt == 1) ///
	(line `outcome' birthyr if late_adopt == 1), ///
	legend(label(1 "Early Adopter (< 1967)") label(2 "Middle Adopter") label(3 "Late Adopter (> 1969)")) ///
	ytitle("Mean `title'") xtitle("Birth Cohort")
	
	graph export "$output/figures/`indexType'_`g'_by_adopter_group.pdf", replace
	
	*replace `outcome' = round(`outcome', .0001)
	save "$output/figures/`outcome'_data.dta", replace
	
	
	}
	
end

*****************************************************************************
* ExposurebyAdopter
*****************************************************************************

capture program drop ExposureByAdopter
program ExposureByAdopter
syntax, [indexType(string) dataExposure(string) sample(int 100)]
	
	* First do exposure model specifications and then add event-study and spline due to different datasets 
	use if runiform() <= `sample'/100 using `dataExposure', clear

	keep if inrange(birthyr,1950,1980)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1	
	sort fips birthyr
	egen state_year = group(st_fips birthyr)
	gen shareFSP_early = shareFSPageIU_5 * early_adopt

	foreach g in all wm wf nm nf {
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		
		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"

		* Identify sample for exposure models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' shareFSPageIU_5 birthyr birthmo year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60)
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1 & early_adopt == 1
		local nobs_early = round(r(sum), 1000)
		sum `obs' if sample==1 & early_adopt == 0
		local nobs_late = round(r(sum), 1000)
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1 & early_adopt == 1
		local mdv_early = round(r(mean), .0001)
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1 & early_adopt == 0
		local mdv_late = round(r(mean), .0001)
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1
		local mdv = round(r(mean), .0001)
		
		*Order of models 
		* 0. birthyr year (w/o trends) 
		* 1. state_year year (w/o trends) * in subgroup  
		* 2. year birthyr (w trends) 
		* 3. state_year year (w trends) * in subgroup 

		* Run exposure models
		* Model 3 (preferred model)
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		reghdfe `outcome' shareFSPageIU_5 `trends'  if early_adopt == 1 & sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", `r' noparen noaster noobs nor2 ///
				keep(shareFSP*) ctitle("Early Model 3" ) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', mdv, `mdv_early',nobs,`nobs_early') ///
				addtext(Birth month, "Yes", Survey year,"Yes", ///
				State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "No", Early, "Yes") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta	
		local r "append"
		reghdfe `outcome' shareFSPageIU_5 `trends'  if early_adopt == 0 & sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", `r' noparen noaster noobs nor2 ///
				keep(shareFSP*) ctitle("Late Model 3" ) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', mdv, `mdv_late',nobs,`nobs_late') ///
				addtext(Birth month, "Yes", Survey year,"Yes", ///
				State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "No", Early, "No") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
				
		reghdfe `outcome' shareFSPageIU_5 shareFSP_early `trends'  if sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", ///
		`r' noparen noaster noobs nor2 ///
		keep(shareFSP*) ctitle("Interaction Model 3" ) ///
		addstat(Cells, `cells', R squared, `R2', Counties, ///
		`counties', mdv, `mdv',nobs,`nobs') ///
		addtext(Birth month, "Yes", Survey year,"Yes", ///
		State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "Yes", Early, "N/A") ///
		title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		restore 
	
	}
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/table_`indexType'_`g'_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/exposure/table_`indexType'_`g'_adopter_dta, clear 
		export excel using $output/exposure/`indexType'_adopter, sheet("`g'", replace )  
	}
	
	
end

*****************************************************************************
* ExposurebyAdopterSpecial
*****************************************************************************

capture program drop ExposureByAdopterSpecial
program define ExposureByAdopterSpecial
syntax, [indexType(string) dataExposure(string) sample(int 100)]

	* First do exposure model specifications and then add event-study and spline due to different datasets 
	use if runiform() <= `sample'/100 using `dataExposure', clear

	keep if inrange(birthyr,1950,1980)

	* Create fips for clustering
	capture drop fips
	gen fips = st_fips*1000+co_fips_1	
	sort fips birthyr
	egen inNumidentSamp=tag(fips birthyr)
	replace inNumidentSamp=. if fips==. | !inrange(birthyr,1950,1980)
	egen state_year = group(st_fips birthyr)
	gen shareFSP_early = shareFSPageIU_5 * early_adopt
	
	* Some modifications to certain indices 
	if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
		* These observations are at the county/birth-month level. No survey-year variation.
		* So get rid of the extra survey years of data
		egen tag = tag(fips birthyr birthmo early_adopt exactmatch)
		tab birthyr tag, m
		tab year tag, m
		keep if tag==1 & exactmatch==0
		drop tag
	} 
	

	foreach g in all wm wf nm nf {
		
		* Table names 
		if "`indexType'" == "z_composite" local title "Composite Index, `g'"
		if "`indexType'" == "z_ess" local title "Economic Self Sufficiency Index, `g'"
		if "`indexType'" == "z_hc"  local title  "Human Capital Index, `g'" 
		if "`indexType'" == "z_dis" local title "Disability Index, `g'" 
		if "`indexType'" == "z_nei" local title "Living Quality Index"	
		if "`indexType'" == "dis_7" local title "Share That Lived to 2000, `g'"
		if "`indexType'" == "dis_8" local title "Age at Death, `g'"
		if "`indexType'" == "dis_9" local title "Share That Lived to 2012, `g'"	
		if "`indexType'" == "ess_7" local title "Share Not Incarcerated, `g'"
			
		*** Set weights 
		if "`indexType'_`g'"=="z_dis_`g'" local obs "obs_dis_`g'"		
		else local obs "obs_`g'"
		* Specific weights for these indices 
		if "`indexType'" == "dis_7" | "`indexType'" == "dis_8" | "`indexType'" == "dis_9" {
			if "`indexType'_`g'"=="dis_8_`g'" local obs "num_deaths_`g'"
			else local obs "num_births_`g'"
		}
		if "`indexType'"=="ess_7" local obs "obs_inc_`g'"

		* outcome of interest for subgroup g 
		local outcome "`indexType'_`g'"



		
		* Identify sample for exposure models 
		preserve 
		cap drop sample
		egen tmp = rowmiss(`indexType'_`g' shareFSPageIU_5 birthyr year fips `obs' lpop60 inc3k60 urban60 black60 age560 age6560 farmlandpct60 )
		gen sample = tmp==0
		tab sample tmp, m			// Total county/birth year/survey year cells
		tabstat `obs', by(sample) stats(sum)	// Total underlying individual observations
		drop tmp
		
		* Capture number of underlying observations and round 			
		sum `obs' if sample==1 & early_adopt == 1
		local nobs_early = round(r(sum), 1000)
		sum `obs' if sample==1 & early_adopt == 0
		local nobs_late = round(r(sum), 1000)
		sum `obs' if sample==1
		local nobs = round(r(sum), 1000)
		
		* Mean of dependent variable, 1950-54 cohorts
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1 & early_adopt == 1
		local mdv_early = round(r(mean), .0001)
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1 & early_adopt == 0
		local mdv_late = round(r(mean), .0001)
		sum `indexType'_`g' [aw=`obs'] if inrange(birthyr,1950,1954) & sample==1
		local mdv = round(r(mean), .0001)


	
		* Run exposure models
		* Model 3 (preferred model)
		local r "replace"
		local trends "c.birthyr#(c.lpop60 c.inc3k60 c.urban60 c.black60 c.age560 c.age6560)"
		reghdfe `outcome' shareFSPageIU_5 `trends'  if early_adopt == 1 & sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", `r' noparen noaster noobs nor2 ///
				keep(shareFSP*) ctitle("Early Model 3" ) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', mdv, `mdv_early',nobs,`nobs_early') ///
				addtext(Birth month, "Yes", Survey year,"Yes", ///
				State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "No", Early, "Yes") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta	
		local r "append"
		reghdfe `outcome' shareFSPageIU_5 `trends'  if early_adopt == 0 & sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", `r' noparen noaster noobs nor2 ///
				keep(shareFSP*) ctitle("Late Model 3" ) ///
				addstat(Cells, `cells', R squared, `R2', Counties, `counties', mdv, `mdv_late',nobs,`nobs_late') ///
				addtext(Birth month, "Yes", Survey year,"Yes", ///
				State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "No", Early, "No") ///
				title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
				
		reghdfe `outcome' shareFSPageIU_5 shareFSP_early `trends'  if sample==1 [aw=`obs'], absorb(fips state_year birthmo year) vce(cluster fips) 
		local cells = round(`e(N)', 1000) 
		local R2 = round(`e(r2)', .0001)
		local counties = round(e(N_clust), 1000)
		outreg2 using "$output/exposure/table_`outcome'_adopter", ///
		`r' noparen noaster noobs nor2 ///
		keep(shareFSP*) ctitle("Interaction Model 3" ) ///
		addstat(Cells, `cells', R squared, `R2', Counties, ///
		`counties', mdv, `mdv',nobs,`nobs') ///
		addtext(Birth month, "Yes", Survey year,"Yes", ///
		State x Birth Year,"Yes",County Char Trends,"Yes", Interaction Share x Early, "Yes", Early, "N/A") ///
		title("`title'") dec(4) bdec(4) sdec(4) rdec(4) fmt(g) dta
		restore 
	
	}
	
	
	* Make table 	
	* Now save each onto a different sheet within an excel file
	*capture erase $output/exposure/table_`indexType'_`g'_adopter_dta.dta 
	foreach g in all wm wf nm nf {
		use $output/exposure/table_`indexType'_`g'_adopter_dta, clear 
		export excel using $output/exposure/`indexType'_adopter, sheet("`g'", replace )  
	}
	
end


main
