#---------------- Siqueira et al ----------------#
#----- Evolution of fish-coral interactions -----#
#-- Reconstruction of association through time --#

#----- Loading packages -----#
library(tidyverse)  #Version 2.0.0
library(scales)     #Version 1.2.0
library(igraph)     #Version 1.4.1


#----- Setting WD -----#
wd <- "XXX"   #Set your personal working directory here
setwd(wd)


#----- Loading data -----#
data <- read.table("RevBayes/MuHiSSE/output/coral_association_MuHiSSE_coral_association.log",header=TRUE)

# Removing 10% burnin
data <- data[2003:nrow(data),]

#----- Plotting diversification rates -----#
# Filtering evolutionary rates from the posterior dataset
data_spe <- data[,grep("speciation.", colnames(data), value = T)]
data_ext <- data[,grep("extinction.", colnames(data), value = T)]

data_netdiv <- data_spe[,1:6] - data_ext[,1:6]
colnames(data_netdiv) <- c("Nonassociated","Moderate","Strong",
                           "NonassociatedHidden","ModerateHidden","StrongHidden")

data_netdiv <- data_netdiv %>% 
  pivot_longer(cols = Nonassociated:StrongHidden, names_to = "rate", values_to = "value")

## Plotting - Figure 4A
ggplot(data_netdiv, aes(x=value, fill=rate)) +
  geom_density(size = 0.3, col = "white") +
  scale_x_continuous(limits = c(-0.01, 0.25), expand = c(0,0)) +
  scale_fill_manual(breaks=c("Nonassociated","Moderate","Strong",
                             "NonassociatedHidden","ModerateHidden","StrongHidden"),
                    values = c("#86BBD8","#1E84AD","#2F4858",
                               alpha("#86BBD8",0.5),alpha("#1E84AD",0.5),alpha("#2F4858",0.5))) +
  scale_y_continuous(expand = c(0.01,0.01)) +
  theme_bw() +
  ylab("Posterior Density") + xlab("Net Diversification (lineages/Ma)") +
  theme(text = element_text(size=12),
        legend.title = element_blank(),
        legend.position = c(.8,.8),
        legend.background = element_rect(fill = "transparent", colour = NA),
        panel.background = element_rect(fill = "grey95",
                                        colour = "grey95",
                                        linetype = "solid"),
        panel.grid.major = element_line(size = 0.5, linetype = 'solid',
                                        colour = "white"), 
        panel.grid.minor = element_line(size = 0.25, linetype = 'solid',
                                        colour = "white"))

#ggsave("Figures/Siqueira_etal_Fig4A.pdf", height = 5, width = 6)


#----- Plotting transition rates -----#
# Filtering transition rates from posterior dataset
data_trans <- data[,grep("transition", colnames(data), value = T)]

mat_trans <- matrix(nrow = 3, ncol = 3, dimnames = list(c("Nonassociated","Moderate","Strong"),
                                                        c("Nonassociated","Moderate","Strong")))
mat_trans[1,2] <- round(median(data_trans$transition_rates.1.), 5)
mat_trans[1,3] <- round(median(data_trans$transition_rates.2.), 5)
mat_trans[2,1] <- round(median(data_trans$transition_rates.3.), 5)
mat_trans[2,3] <- round(median(data_trans$transition_rates.4.), 5)
mat_trans[3,1] <- round(median(data_trans$transition_rates.5.), 5)
mat_trans[3,2] <- round(median(data_trans$transition_rates.6.), 5)

# Defining colours
cols<-setNames(c("#86BBD8","#1E84AD","#2F4858"),colnames(mat_trans))

#----- Igraph functions -----#
source("~/Desktop/Ongoing_papers/Coral_association/Scripts/Siqueira_etal_functions.R")
environment(plot.igraph2) <- asNamespace('igraph')
environment(igraph.Arrows2) <- asNamespace('igraph')

g2 <- graph_from_adjacency_matrix(mat_trans, weighted=TRUE,diag=F)

#pdf("Figures/Siqueira_etal_Fig4B.pdf", height = 8, width = 8)
plot.igraph2(g2, layout = layout.circle,
            vertex.size = 50,
            vertex.label.cex = 1.2,
            vertex.color = cols,
            vertex.frame.color = "white",
            vertex.width = 2,
            vertex.label.color = "grey99",
            vertex.label.family = "Helvetica",
            edge.color=c(cols[1],cols[3],cols[2],cols[2],cols[1],cols[3]),
            edge.width=(E(g2)$weight/max(E(g2)$weight))*25,
            edge.arrow.size=(E(g2)$weight/max(E(g2)$weight))*5, 
            edge.label.family = "Helvetica",
            edge.label.cex = 0.8,
            edge.curved = 0.5,
            rescale = F, axes = F)

#dev.off()

####################End of the script###########################
