# AUTOGENERATED! DO NOT EDIT! File to edit: nbs/05_ecg_utils.ipynb (unless otherwise specified).

__all__ = ['get_available_channels', 'channel_stoi_default', 'resample_data', 'get_filename_out', 'reformat_data_ptb',
           'prepare_data_ptb', 'mat_to_np', 'reformat_data_icbeb', 'prepare_data_icbeb', 'prepare_data_ptb_xl',
           'filter_ptb_xl', 'thew_to_np', 'reformat_data_thew', 'prepare_data_thew', 'prepare_data_cinc',
           'prepare_data_zheng', 'prepare_data_ribeiro']

# Cell
import wfdb

import scipy.io

import numpy as np
import pandas as pd

from skimage import transform
from scipy.ndimage import zoom
from tqdm import tqdm_notebook as tqdm
from pathlib import Path

#thew
#from ishneholterlib import Holter
from .stratify import stratify

#ribeiro
#import h5py

#from clinical_ts.misc_utils import *
from .timeseries_utils import *

# Cell
channel_stoi_default = {"i": 0, "ii": 1, "v1":2, "v2":3, "v3":4, "v4":5, "v5":6, "v6":7, "iii":8, "avr":9, "avl":10, "avf":11, "vx":12, "vy":13, "vz":14}

def get_available_channels(channel_labels, channel_stoi):
    if(channel_stoi is None):
        return range(len(channel_labels))
    else:
        return sorted([channel_stoi[c] for c in channel_labels if c in channel_stoi.keys()])

# Cell
def resample_data(sigbufs, channel_labels, fs, target_fs, channels=8, channel_stoi=None,skimage_transform=True,interpolation_order=3):
    channel_labels = [c.lower() for c in channel_labels]
    #https://github.com/scipy/scipy/issues/7324 zoom issues
    factor = target_fs/fs
    timesteps_new = int(len(sigbufs)*factor)
    if(channel_stoi is not None):
        data = np.zeros((timesteps_new, channels), dtype=np.float32)
        for i,cl in enumerate(channel_labels):
            if(cl in channel_stoi.keys() and channel_stoi[cl]<channels):
                if(skimage_transform):
                    data[:,channel_stoi[cl]]=transform.resize(sigbufs[:,i],(timesteps_new,),order=interpolation_order).astype(np.float32)
                else:
                    data[:,channel_stoi[cl]]=zoom(sigbufs[:,i],timesteps_new/len(sigbufs),order=interpolation_order).astype(np.float32)
    else:
        if(skimage_transform):
            data=transform.resize(sigbufs,(timesteps_new,channels),order=interpolation_order).astype(np.float32)
        else:
            data=zoom(sigbufs,(timesteps_new/len(sigbufs),1),order=interpolation_order).astype(np.float32)
    return data

# Cell
def get_filename_out(filename_in, target_folder=None, suffix=""):
    if target_folder is None:
        #absolute path here
        filename_out = filename_in.parent/(filename_in.stem+suffix+".npy")
        filename_out_relative = filename_out
    else:
        if("train" in filename_in.parts):
            target_folder_train = target_folder/"train"
            # relative path here
            filename_out = target_folder_train/(filename_in.stem+suffix+".npy")
            filename_out_relative = filename_out.relative_to(target_folder)

            target_folder_train.mkdir(parents=True, exist_ok=True)
        elif("eval" in filename_in.parts or "dev_test" in filename_in.parts or "valid" in filename_in.parts or "valtest" in filename_in.parts):
            target_folder_valid = target_folder/"valid"
            filename_out = target_folder_valid/(filename_in.stem+suffix+".npy")
            filename_out_relative = filename_out.relative_to(target_folder)
            target_folder_valid.mkdir(parents=True, exist_ok=True)
        else:
            filename_out = target_folder/(filename_in.stem+suffix+".npy")
            filename_out_relative = filename_out.relative_to(target_folder)
            target_folder.mkdir(parents=True, exist_ok=True)
    return filename_out, filename_out_relative

# Cell
def _age_to_categorical(age):
    if(np.isnan(age)):
        label_age = -1
    elif(age<30):
        label_age = 0
    elif(age<40):
        label_age = 1
    elif(age<50):
        label_age = 2
    elif(age<60):
        label_age = 3
    elif(age<70):
        label_age = 4
    elif(age<80):
        label_age = 5
    else:
        label_age = 6
    return label_age

def _sex_to_categorical(sex):
    sex_mapping = {"n/a":-1, "male":0, "female":1, "":-1}
    return sex_mapping[sex]

def reformat_data_ptb(datafiles, target_fs=200, channels=8, channel_stoi=channel_stoi_default, lbl_itos=None, target_folder=None, skimage_transform=True):
    rows = []
    for filename_in in tqdm(datafiles):
        filename_out, filename_out_relative = get_filename_out(filename_in,target_folder)

        sigbufs, header = wfdb.rdsamp(str(filename_in.parent/filename_in.stem))
        data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=header['sig_name'],fs=header['fs'],target_fs=target_fs, channels=channels,skimage_transform=skimage_transform)
        np.save(filename_out,data.astype(np.float32))

        metadata=[(":".join(h.split(":")[1:])).strip() for h in header['comments']]
        metadata_descr = [(h.split(":")[0]).strip() for h in header['comments']]
        #for i,(m1,m2) in enumerate(zip(metadata_descr,metadata)):
        #    print(i,m1,m2)
        #input()
        ecg = filename_in.stem
        patient = str(filename_in.parent)[-3:]
        channels_available = get_available_channels(header['sig_name'],channel_stoi)
        try:
            age=int(metadata[0])
        except ValueError:
            age=np.nan
        label_age = _age_to_categorical(age)

        diagnosis1 = metadata[4]
        diagnosis2 = [l.strip() for l in  metadata[7].split(",") if l.strip() not in ["no","unknown","n/a"]]
        diagnosis2 = [l.replace("Typ IIa","Type IIa").replace("Hypertension","hypertension").replace(" (intraventricular gradient 100-160mmHg)","").replace("Fibrillation","fibrillation").replace("Recurrent ventricular tachycardias","Recurrent ventricular tachycardia").replace("Thyriodectomy","Thyroidectomy").replace("Dilated Cardiomyopathy secondary to coronary artery disease. Mitral regurgitation (grade 2)","Dilated Cardiomyopathy") for l in diagnosis2]#clean up
        #remove type from Hyperlipoproteinemia Type xx TODO
        diagnosis2 = [l.replace("Hyperlipoproteinemia Type IIa","Hyperlipoproteinemia").replace("Hyperlipoproteinemia Type IV","Hyperlipoproteinemia").replace("Hyperlipoproteinemia Type IIb","Hyperlipoproteinemia") for l in diagnosis2]#clean up

        diagnosis12 = [diagnosis1]+diagnosis2

        smoker = metadata[8]
        smoker_mapping = {"no":0, "yes":1, "n/a":-1, "unknown":-1}
        label_smoker = smoker_mapping[smoker]

        sex = metadata[1]
        label_sex = _sex_to_categorical(sex)

        row={"data":filename_out_relative,"dataset":"PTB","channels_available":channels_available,"patient":"PTB_"+patient, "ecg": "PTB_"+ecg, "age": age, "label_age":label_age,"sex": sex, "label_sex": label_sex, "ecg_date":metadata[2], "diagnosis":diagnosis1, "infarction_localization":metadata[5], "diagnosis2": diagnosis2, "diagnosis12": diagnosis12,"smoker": smoker, "label_smoker":label_smoker, "infarction_date":metadata[10], "catheterization_date":metadata[14]}
        #add hemodynamics
        for i in range(14,36):
            row[metadata_descr[i]]=metadata[i]
        #clean up localization
        if(row["diagnosis"]!="Myocardial infarction"):
            row["infarction_localization"]=""
        if row["infarction_localization"] == "infero-latera":
            row["infarction_localization"] = "infero-lateral"
        if row["infarction_localization"] == "infero-posterior-lateral":
            row["infarction_localization"] = "infero-postero-lateral"
        if row["infarction_localization"] == "infero-poster-lateral":
            row["infarction_localization"] = "infero-postero-lateral"
        if row["infarction_localization"] == "no":
            row["infarction_localization"] = ""
        rows.append(row)
    df_all = pd.DataFrame(rows)
    _ptb_cleanup_hemodynamics(df_all)
    #fill label column
    if(lbl_itos is not None):
        lbl_stoi = {s:i for i,s in enumerate(lbl_itos)}
        df_all["label"] = df_all.diagnosis.apply(lambda x: (lbl_stoi[x] if x in lbl_stoi.keys() else len(lbl_itos)))#map everything that cannot be found in the dictionary to last entry
    else:
        lbl_itos = np.unique(df_all["diagnosis"])
        lbl_stoi = {s:i for i,s in enumerate(lbl_itos)}
        df_all["label"] = df_all.diagnosis.apply(lambda x: lbl_stoi[x])
    #convert datatypes (be careful the standard to_datetime does not work correctly here)
    #df_all.ecg_date=pd.to_datetime(df_all.ecg_date,errors='coerce')
    #df_all.infarction_date=pd.to_datetime(df_all.infarction_date,errors='coerce')
    #df_all.catheterization_date=pd.to_datetime(df_all.catheterization_date,errors='coerce')

    #df_all=df_all.set_index("ecg")
    return df_all, lbl_itos


def _ptb_cleanup_hemodynamics(df_ptb):
    def _extract_numeric(x,mode=0):
        if x=="n/a":
            return np.nan
        else:
            if mode==0:
                return float(x.split(" ")[0])
            elif mode==1:
                return float(x.split(" ")[0].split("/")[0])
            else:
                return float(x.split(" ")[0].split("/")[1])

    def _reformat_col(df, col_in, col_out1, col_out2=None):
        df[col_out1]= df[col_in].apply(lambda x: _extract_numeric(x,0 if col_out2 is None else 1))
        if col_out2 is not None:
            df[col_out2]= df[col_in].apply(lambda x: _extract_numeric(x,2))
        df.drop(col_in,axis=1,inplace=True)

    _reformat_col(df_ptb,"Aorta (at rest) (syst/diast)","aorta_at_rest_syst","aorta_at_rest_diast")
    _reformat_col(df_ptb,"Aorta (at rest) mean","aorta_at_rest_mean")

    df_ptb["Cardiac index (at rest)"]=df_ptb["Cardiac index (at rest)"].apply(lambda x: x.replace(",","."))
    _reformat_col(df_ptb,"Cardiac index (at rest)","cardiac_index_at_rest")
    df_ptb["Cardiac index (load)"]=df_ptb["Cardiac index (load)"].apply(lambda x: x.replace(",","."))
    _reformat_col(df_ptb,"Cardiac index (load)","cardiac_index_load")
    df_ptb["Cardiac output (at rest)"]=df_ptb["Cardiac output (at rest)"].apply(lambda x: x.replace(",","."))
    _reformat_col(df_ptb,"Cardiac output (at rest)","cardiac_output_at_rest")
    df_ptb["Cardiac output (load)"]=df_ptb["Cardiac output (load)"].apply(lambda x: x.replace(",","."))
    _reformat_col(df_ptb,"Cardiac output (load)","cardiac_output_load")

    df_ptb.drop('Chest X-ray',axis=1,inplace=True)
    df_ptb.drop('Left coronary artery stenoses (RCX)',axis=1,inplace=True)
    df_ptb.drop('Left coronary artery stenoses (RIVA)',axis=1,inplace=True)
    df_ptb.drop('Right coronary artery stenoses (RCA)',axis=1,inplace=True)
    df_ptb.drop('Ventriculography',axis=1,inplace=True)
    df_ptb.drop('Catheterization date',axis=1,inplace=True)

    df_ptb["Peripheral blood Pressure (syst/diast)"]=df_ptb["Peripheral blood Pressure (syst/diast)"].apply(lambda x: x.replace("120 /70","120/70"))
    _reformat_col(df_ptb,"Peripheral blood Pressure (syst/diast)","peripheral_bp_syst","peripheral_bp_diast")
    _reformat_col(df_ptb,"Pulmonary artery pressure (at rest) (mean)","pulmonary_ap_at_rest_mean")
    _reformat_col(df_ptb,"Pulmonary artery pressure (at rest) (syst/diast)","pulmonary_ap_at_rest_syst","pulmonary_ap_at_rest_diast")
    _reformat_col(df_ptb,"Pulmonary artery pressure (laod) (mean)","pulmonary_load_mean")
    _reformat_col(df_ptb,"Pulmonary artery pressure (laod) (syst/diast)","pulmonary_ap_load_syst","pulmonary_ap_load_diast")
    _reformat_col(df_ptb,"Pulmonary capillary wedge pressure (at rest)","pulmonary_cwp_at_rest")
    _reformat_col(df_ptb,"Pulmonary capillary wedge pressure (load)","pulmonary_cwp_load")

    df_ptb["Left ventricular enddiastolic pressure"]=df_ptb["Left ventricular enddiastolic pressure"].apply(lambda x:x[-8:].strip())#only post value
    _reformat_col(df_ptb,"Left ventricular enddiastolic pressure","left_ventricular_ed_pressure")


    _reformat_col(df_ptb,"Stroke volume index (at rest)","stroke_volume_index_at_rest")
    df_ptb["Stroke volume index (load)"]=df_ptb["Stroke volume index (load)"].apply(lambda x: x.replace(",","."))
    _reformat_col(df_ptb,"Stroke volume index (load)","stroke_volume_index_load")

def prepare_data_ptb(data_folder, target_folder=None, channel_stoi=channel_stoi_default, channels=8, target_fs=50, strat_folds=10, cv_random_state=42,recreate_data=False,ptb_permanently_remove_diagnosis_na = True, ptb_permanently_remove_multiple_ecgs = False,ptb_select_only_first_mi = True, ptb_remove_unknown_loc_mi = True,lbl_itos_ptb_in = ["Healthy control", "Myocardial infarction"],lbl_itos_ptb_regression = ["pulmonary_ap_at_rest_mean"],lbl_itos_ptb_all = ["Diabetes mellitus"],skimage_transform=True):


    target_root_ptb = Path(".") if target_folder is None else target_folder

    diagnosis_to_int_w_subdiagnosis = {
    #"Healthy control": 0,  # 80
    "anterior": 1,  # 47
    "antero-septal": 1,  # 77
    "antero-septo-lateral": 1,  # 2
    "antero-lateral": 1,  # 43
    "lateral": 1,  # 3
    "inferior": 2,  # 89
    "infero-posterior": 2,  # 1
    "infero-postero-lateral": 2,  # 16
    "infero-poster-lateral": 2,  # 3
    "infero-latera": 2,  # 3
    "infero-lateral": 2,  # 53
    "posterior": 2,  # 4
    "postero-lateral": 2,  # 5
    #"other": -1  # 123
    "":0
    }
    int_to_str=[""," anterior", "inferior"]

    if(recreate_data):
        #collect files
        datafiles_ptb = list(data_folder.glob('**/*.dat'))

        #reformat data
        df_ptb, lbl_itos_ptb=reformat_data_ptb(datafiles=datafiles_ptb,target_fs=target_fs, channels=channels,lbl_itos=lbl_itos_ptb_in, target_folder=target_folder, skimage_transform=skimage_transform)


        df_ptb["infarction_localization_mapped"]=df_ptb["infarction_localization"].apply(lambda x: int_to_str[diagnosis_to_int_w_subdiagnosis[x]])
        df_ptb["diagnosis_extended"]=df_ptb.apply(lambda row: row["diagnosis"]+" "+row["infarction_localization_mapped"], axis=1)
        #TODO move all this stuff into reformat_data_ptb
        #set regression label(s)
        df_ptb["label_regression"]=df_ptb.apply(lambda row: row[lbl_itos_ptb_regression[0]] if len(lbl_itos_ptb_regression)==1 else [row[regression_col] for regression_col in lbl_itos_ptb_regression] ,axis=1)
        df_ptb.label_regression.apply(lambda x: np.float32(np.log(x)))
        if(len(lbl_itos_ptb_regression)==1):
            df_ptb.label_regression=df_ptb.label_regression.astype(np.float32)

        #set diagnosis_all multilabels
        def extract_multilabel(x, lbl_itos):
            ret = np.zeros(len(lbl_itos),dtype=np.float32)
            for xi in x:
                if(xi in lbl_itos):
                    ret[lbl_itos.index(xi)]=1.0
            return ret

        df_ptb["label_multi_all"]=df_ptb.diagnosis12.apply(lambda x: extract_multilabel(x, lbl_itos_ptb_all) if len(lbl_itos_ptb_all)>1 else int(lbl_itos_ptb_all[0] in x))

        #permanently discard diagnosis n/a
        if(ptb_permanently_remove_diagnosis_na):
            df_ptb = df_ptb[df_ptb.diagnosis != "n/a"].copy()

        #permanently remove multiple ecgs
        if(ptb_permanently_remove_multiple_ecgs):
            df_ptb=df_ptb.sort_values(by='ecg')
            df_ptb["first_ecg"]=df_ptb.ecg.isin(df_ptb.groupby("patient").first().ecg)
            df_ptb["drop"]=df_ptb.ecg.isin(df_ptb[(df_ptb.diagnosis=="Myocardial infarction") & (df_ptb.first_ecg==False)].ecg)
            df_ptb=df_ptb[df_ptb["drop"]==False].reset_index().drop("index",axis=1)

        #discard other diagnosis classes that were not selected
        df_ptb2 = df_ptb[df_ptb.label <= len(lbl_itos_ptb)-1].copy()

        #select first record for MI patients
        if(ptb_select_only_first_mi):
            df_ptb2=df_ptb2.sort_values(by='ecg')
            df_ptb2["first_ecg"]=df_ptb2.ecg.isin(df_ptb2.groupby("patient").first().ecg)
            df_ptb2["drop"]=df_ptb2.ecg.isin(df_ptb2[(df_ptb2.diagnosis=="Myocardial infarction") & (df_ptb2.first_ecg==False)].ecg)
            df_ptb2=df_ptb2[df_ptb2["drop"]==False].reset_index().drop("index",axis=1)

        #remove MIs with unknown infarction status
        if(ptb_remove_unknown_loc_mi):
            df_ptb2=df_ptb2[~((df_ptb2.diagnosis=="Myocardial infarction") & (df_ptb2.infarction_localization==""))]

        #train test split for label (main diagnosis) based on patients
        label_patients=df_ptb2.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        df_ptb["strat_fold"] = -1
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            #now select from ptb2 (as second MIs etc might have been excluded)
            ecgs_valid = np.array(df_ptb2[df_ptb2.patient.isin(patients_valid)]["ecg"])
            df_ptb.loc[df_ptb.ecg.isin(ecgs_valid),"strat_fold"]=i

        #train test split for regression based on ecgs (but does not work properly anyway)
        df_ptb["strat_fold_regression"]=-1
        df_ptb_regression = df_ptb[df_ptb.label_regression.apply(lambda x: np.isfinite(x))].copy()
        label_patients=df_ptb_regression.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            #now select from ptb_regression
            ecgs_valid = np.array(df_ptb_regression[df_ptb_regression.patient.isin(patients_valid)]["ecg"])
            df_ptb.loc[df_ptb.ecg.isin(ecgs_valid),"strat_fold_regression"]=i

        #train test split for label_multi_all (main diagnosis) based on patients
        label_patients=df_ptb.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        df_ptb["strat_fold_all"] = -1
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            df_ptb.loc[df_ptb.patient.isin(patients_valid),"strat_fold_all"]=i

        #train test split for smoker based on patients
        df_ptb_smoker = df_ptb[df_ptb.label_smoker>=0].copy()
        label_patients=df_ptb_smoker.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        df_ptb["strat_fold_smoker"] = -1
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            #now select from ptb_smoker
            ecgs_valid = np.array(df_ptb_smoker[df_ptb_smoker.patient.isin(patients_valid)]["ecg"])
            df_ptb.loc[df_ptb.ecg.isin(ecgs_valid),"strat_fold_smoker"]=i

        #train test split for age based on patients
        df_ptb_age = df_ptb[df_ptb.label_age>=0].copy()
        label_patients=df_ptb_age.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        df_ptb["strat_fold_age"] = -1
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            #now select from ptb_age
            ecgs_valid = np.array(df_ptb_age[df_ptb_age.patient.isin(patients_valid)]["ecg"])
            df_ptb.loc[df_ptb.ecg.isin(ecgs_valid),"strat_fold_age"]=i

        #train test split for sex based on patients
        df_ptb_sex = df_ptb[df_ptb.label_sex>=0].copy()
        label_patients=df_ptb_sex.groupby("patient")["diagnosis_extended"].first()
        splits_patients = get_stratified_kfolds(np.array(label_patients),n_splits=strat_folds,random_state=cv_random_state)
        df_ptb["strat_fold_sex"] = -1
        for i,split in enumerate(splits_patients):
            patients_valid = np.array(label_patients.index)[split[1]]
            #now select from ptb_sex
            ecgs_valid = np.array(df_ptb_sex[df_ptb_sex.patient.isin(patients_valid)]["ecg"])
            df_ptb.loc[df_ptb.ecg.isin(ecgs_valid),"strat_fold_sex"]=i

        #DEBUG
        #importing old splits for debugging
        indexlst = np.load("./old/index_lst.npy")
        evallist = np.load("./old/evallist.npy")
        #trainlist = np.load("./old/trainlist.npy")
        df_ptb["strat_fold_old"] = -1
        for i in range(len(evallist)):
            df_ptb.loc[df_ptb.ecg.isin(["PTB_"+x for x in indexlst[evallist[i]][:,1]]),"strat_fold_old"]=i
        #export current splits for debugging
        evallist =[]
        trainlist =[]
        for i in range(strat_folds):
            valid_ecgs= np.array(df_ptb[(df_ptb.strat_fold>=0)&(df_ptb.strat_fold==i)].ecg.apply(lambda x: x[4:]))
            train_ecgs= np.array(df_ptb[(df_ptb.strat_fold>=0)&(df_ptb.strat_fold!=i)].ecg.apply(lambda x: x[4:]))
            evallist.append([np.where(indexlst==v)[0][0] for v in valid_ecgs])
            trainlist.append([np.where(indexlst==t)[0][0] for t in train_ecgs])
        np.save("./old/evallist_new.npy",evallist)
        np.save("./old/trainlist_new.npy",trainlist)


        #add means and std
        dataset_add_mean_col(df_ptb,data_folder=target_root_ptb)
        dataset_add_std_col(df_ptb,data_folder=target_root_ptb)
        dataset_add_length_col(df_ptb,data_folder=target_root_ptb)
        #dataset_add_median_col(df_ptb,data_folder=target_folder)
        #dataset_add_iqr_col(df_ptb,data_folder=target_folder)

        #compute means and stds
        mean_ptb, std_ptb = dataset_get_stats(df_ptb)

        #save
        save_dataset(df_ptb,lbl_itos_ptb,mean_ptb,std_ptb,target_root_ptb)
    else:
        df_ptb, lbl_itos_ptb, mean_ptb, std_ptb = load_dataset(target_root_ptb)
    return df_ptb, lbl_itos_ptb, mean_ptb, std_ptb

# Cell
def mat_to_np(filename_in, target_fs=100, channels=8, channel_stoi=None, fs=500, target_folder=None, skimage_transform=True):
    channel_labels=["i","ii", "iii", "avr", "avl", "avf", "v1", "v2", "v3", "v4", "v5", "v6"]

    filename_out, filename_out_relative = get_filename_out(filename_in, target_folder)

    channels_available = get_available_channels(channel_labels,channel_stoi)
    data_raw = scipy.io.loadmat(filename_in)
    sex,age,sigbufs=data_raw['ECG'][0][0]
    sigbufs =np.transpose(sigbufs)

    data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=channel_labels,fs=fs,target_fs=target_fs, channels=channels,skimage_transform=skimage_transform)
    np.save(filename_out, data.astype(np.float32))
    return sex[0], age[0,0], channels_available, filename_out, filename_out_relative

def reformat_data_icbeb(datafiles, labelfile, target_fs=100, channels=8, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True):
    #labels
    label_itos=["NORMAL","AF", "I-AVB","LBBB","RBBB","PAC","PVC","STD","STE"]
    labels=pd.read_csv(labelfile)
    labels.columns=["ecg","label1","label2","label3"]
    labels=labels.set_index("ecg")

    labels.label1=labels.label1.apply(lambda x: int(x) -1 if not np.isnan(x) else np.nan)
    labels.label2=labels.label2.apply(lambda x: int(x) -1 if not np.isnan(x) else np.nan)
    labels.label3=labels.label3.apply(lambda x: int(x) -1 if not np.isnan(x) else np.nan)
    #data
    rows=[]
    for d in tqdm(datafiles):
        sex,age,channels_available,filename_out,filename_out_relative=mat_to_np(filename_in=d, target_fs=target_fs, channels=channels, channel_stoi=channel_stoi, target_folder=target_folder,skimage_transform=skimage_transform)
        rows.append({"ecg":d.stem, "dataset":"ICBEB2018", "data":filename_out_relative, "age": (np.nan if np.isnan(age) else int(age)), "sex": sex.lower(), "channels_available":channels_available})
    df=pd.DataFrame(rows)
    df=df.set_index("ecg")
    #join
    df=df.join(labels)
    df=df.reset_index()
    #define actual label (label 2 and label 3 are multilabels)
    df["label"]=df["label1"]
    df["has_label2"]=~pd.isna(df["label2"])#i.e. multilabeled label 3 will only be set if label 2 is
    df["has_label3"]=~pd.isna(df["label3"])

    #age
    df["label_age"]=df.age.apply(lambda x: _age_to_categorical(x))
    #sex
    df["label_sex"]=df.sex.apply(lambda x: _sex_to_categorical(x))

    def multi_label(x):
        res = np.zeros(len(label_itos),dtype=np.float32)
        for xi in x:
            if(np.isfinite(xi) and int(xi) in list(range(len(label_itos)))):
                res[int(xi)]=1
        return res

    def combine_labels(x):
        res = [x["label1"]]
        if(np.isfinite(x["label2"])):
            res += [int(x["label2"])]
        if(np.isfinite(x["label3"])):
            res += [int(x["label3"])]
        return res

    df["labels"]=df.apply(lambda x: combine_labels(x),axis=1)

    df["label_multi"]=df.labels.apply(lambda x:multi_label(x))
    return df, label_itos

def prepare_data_icbeb(data_folder, target_folder=None, channel_stoi=channel_stoi_default, channels=8, target_fs=50, strat_folds=10, cv_random_state=42,recreate_data=False,discard_valtest=True,skimage_transform=True):
    target_root_icbeb = Path(".") if target_folder is None else target_folder
    if(recreate_data):
        #collect files
        datafiles_icbeb=list(data_folder.glob('**/*.mat'))
        labelfile_icbeb=data_folder/"REFERENCE.csv"

        #reformat data
        df_icbeb, lbl_itos_icbeb = reformat_data_icbeb(datafiles_icbeb,labelfile_icbeb,channel_stoi=channel_stoi,target_fs=target_fs,channels=channels,target_folder=target_folder,skimage_transform=skimage_transform)

        #TODO eventually move all the stuff below into reformat data
        #remove valid data
        df_icbeb["set"] = df_icbeb.data.apply(lambda x: (0 if "train" in x.parts else 1))
        if(discard_valtest is True):
            df_icbeb=df_icbeb[df_icbeb["set"]==0].reset_index()#only train set has label
        else:#reset labels that have been set erroneously
            df_icbeb.loc[df_icbeb["set"]==1,"label"]=np.nan
            df_icbeb.loc[df_icbeb["set"]==1,"label_multi"]=np.nan
            df_icbeb.loc[df_icbeb["set"]==1,"label1"]=np.nan
            df_icbeb.loc[df_icbeb["set"]==1,"label2"]=np.nan
            df_icbeb.loc[df_icbeb["set"]==1,"label3"]=np.nan

        #train test split (all)
        stratified_ids = stratify(list(df_icbeb["label_multi"]), range(len(lbl_itos_icbeb)), [1./strat_folds]*strat_folds)
        df_icbeb["strat_fold"]=-1
        idxs = np.array(df_icbeb.index.values)
        for i,split in enumerate(stratified_ids):
            df_icbeb.loc[idxs[split],"strat_fold"]=i

        #add means and std
        dataset_add_mean_col(df_icbeb,data_folder=target_folder)
        dataset_add_std_col(df_icbeb,data_folder=target_folder)
        dataset_add_length_col(df_icbeb,data_folder=target_folder)
        #dataset_add_median_col(df_icbeb,data_folder=target_folder)
        #dataset_add_iqr_col(df_icbeb,data_folder=target_folder)

        #save means and stds
        mean_icbeb, std_icbeb = dataset_get_stats(df_icbeb)

        #save
        save_dataset(df_icbeb, lbl_itos_icbeb, mean_icbeb, std_icbeb, target_root_icbeb)
    else:
        df_icbeb, lbl_itos_icbeb, mean_icbeb, std_icbeb = load_dataset(target_root_icbeb)
    return df_icbeb,lbl_itos_icbeb, mean_icbeb, std_icbeb

# Cell
def prepare_data_ptb_xl(data_path, min_cnt=50, target_fs=100, channels=8, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True, recreate_data=True):
    target_root_ptb_xl = Path(".") if target_folder is None else target_folder
    #print(target_root_ptb_xl)
    target_root_ptb_xl.mkdir(parents=True, exist_ok=True)

    if(recreate_data is True):
        # reading df
        ptb_xl_csv = data_path/"ptbxl_database.csv"
        df_ptb_xl=pd.read_csv(ptb_xl_csv,index_col="ecg_id")
        #print(df_ptb_xl.columns)
        df_ptb_xl.scp_codes=df_ptb_xl.scp_codes.apply(lambda x: eval(x.replace("nan","np.nan")))

        # preparing labels
        ptb_xl_label_df = pd.read_csv(data_path/"scp_statements.csv")
        ptb_xl_label_df=ptb_xl_label_df.set_index(ptb_xl_label_df.columns[0])

        ptb_xl_label_diag= ptb_xl_label_df[ptb_xl_label_df.diagnostic >0]
        ptb_xl_label_form= ptb_xl_label_df[ptb_xl_label_df.form >0]
        ptb_xl_label_rhythm= ptb_xl_label_df[ptb_xl_label_df.rhythm >0]

        diag_class_mapping={}
        diag_subclass_mapping={}
        for id,row in ptb_xl_label_diag.iterrows():
            if(isinstance(row["diagnostic_class"],str)):
                diag_class_mapping[id]=row["diagnostic_class"]
            if(isinstance(row["diagnostic_subclass"],str)):
                diag_subclass_mapping[id]=row["diagnostic_subclass"]

        df_ptb_xl["label_all"]= df_ptb_xl.scp_codes.apply(lambda x: [y for y in x.keys()])
        df_ptb_xl["label_diag"]= df_ptb_xl.scp_codes.apply(lambda x: [y for y in x.keys() if y in ptb_xl_label_diag.index])
        df_ptb_xl["label_form"]= df_ptb_xl.scp_codes.apply(lambda x: [y for y in x.keys() if y in ptb_xl_label_form.index])
        df_ptb_xl["label_rhythm"]= df_ptb_xl.scp_codes.apply(lambda x: [y for y in x.keys() if y in ptb_xl_label_rhythm.index])

        df_ptb_xl["label_diag_subclass"]= df_ptb_xl.label_diag.apply(lambda x: [diag_subclass_mapping[y] for y in x if y in diag_subclass_mapping])
        df_ptb_xl["label_diag_superclass"]= df_ptb_xl.label_diag.apply(lambda x: [diag_class_mapping[y] for y in x if y in diag_class_mapping])

        df_ptb_xl["dataset"]="ptb_xl"
        #filter (can be reapplied at any time)
        df_ptb_xl, lbl_itos_ptb_xl =filter_ptb_xl(df_ptb_xl,min_cnt=min_cnt)

        filenames = []
        for id, row in tqdm(list(df_ptb_xl.iterrows())):
            filename = data_path/row["filename_lr"] if target_fs<=100 else data_path/row["filename_hr"]
            sigbufs, header = wfdb.rdsamp(str(filename))
            data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=header['sig_name'],fs=header['fs'],target_fs=target_fs,channels=channels,skimage_transform=skimage_transform)
            assert(target_fs<=header['fs'])
            np.save(target_root_ptb_xl/(filename.stem+".npy"),data)
            filenames.append(Path(filename.stem+".npy"))
        df_ptb_xl["data"] = filenames

        #add means and std
        dataset_add_mean_col(df_ptb_xl,data_folder=target_root_ptb_xl)
        dataset_add_std_col(df_ptb_xl,data_folder=target_root_ptb_xl)
        dataset_add_length_col(df_ptb_xl,data_folder=target_root_ptb_xl)
        #dataset_add_median_col(df_ptb_xl,data_folder=target_root_ptb_xl)
        #dataset_add_iqr_col(df_ptb_xl,data_folder=target_root_ptb_xl)

        #save means and stds
        mean_ptb_xl, std_ptb_xl = dataset_get_stats(df_ptb_xl)

        #save
        save_dataset(df_ptb_xl,lbl_itos_ptb_xl,mean_ptb_xl,std_ptb_xl,target_root_ptb_xl)
    else:
        df_ptb_xl, lbl_itos_ptb_xl, mean_ptb_xl, std_ptb_xl = load_dataset(target_root_ptb_xl)
    return df_ptb_xl, lbl_itos_ptb_xl, mean_ptb_xl, std_ptb_xl

def filter_ptb_xl(df,min_cnt=10,categories=["label_all","label_diag","label_form","label_rhythm","label_diag_subclass","label_diag_superclass"]):
    #filter labels
    def select_labels(labels, min_cnt=10):
        lbl, cnt = np.unique([item for sublist in list(labels) for item in sublist], return_counts=True)
        return list(lbl[np.where(cnt>=min_cnt)[0]])
    df_ptb_xl = df.copy()
    lbl_itos_ptb_xl = {}
    for selection in categories:
        label_selected = select_labels(df_ptb_xl[selection],min_cnt=min_cnt)
        df_ptb_xl[selection+"_filtered"]=df_ptb_xl[selection].apply(lambda x:[y for y in x if y in label_selected])
        lbl_itos_ptb_xl[selection] = np.array(list(set([x for sublist in df_ptb_xl[selection+"_filtered"] for x in sublist])))
        lbl_stoi = {s:i for i,s in enumerate(lbl_itos_ptb_xl[selection])}
        df_ptb_xl[selection+"_filtered_numeric"]=df_ptb_xl[selection+"_filtered"].apply(lambda x:[lbl_stoi[y] for y in x])
    return df_ptb_xl, lbl_itos_ptb_xl


# Cell
def thew_to_np(filename_in, target_fs=100, channels=12, max_length_seconds=0, channel_stoi=None, fs=180, target_folder=None, skimage_transform=True):
    x = Holter(str(filename_in))
    metadata = {"id":x.id, "length":x.ecg_size, "record_time": str(x.record_date)+" "+str(x.start_time), "channel_labels":[xx.spec_str().lower() for xx in x.lead]}
    metadata["channels_available"] = get_available_channels(metadata["channel_labels"],channel_stoi)

    x.load_data()
    sigbufs = np.transpose(np.array([l.data for l in x.lead]))#timesteps,channel

    if(max_length_seconds>0):
        max_length = min(len(sigbufs),int(np.round(fs*max_length_seconds)))
        sigbufs = sigbufs[:max_length]

    data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=metadata["channel_labels"],fs=fs,target_fs=target_fs, channels=channels,skimage_transform=skimage_transform)

    filename_out, filename_out_relative = get_filename_out(filename_in,target_folder)
    np.save(filename_out, data.astype(np.float32))
    metadata["length"]=len(data)

    return metadata, filename_out, filename_out_relative


def reformat_data_thew(data_path=".",max_length_seconds=0, target_fs=100, channels=12, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True):
    data_path = Path(data_path)

    #prepare clinical data
    df_clinical = pd.read_excel(data_path/"clinicalData"/"E-HOL-12-1172-012.xls",sheet_name=1)
    df_clinical = df_clinical[~df_clinical["ECG file"].isna()]#filter out nan entries
    label_itos=["Non ACS","Non-cardiac", "Unstb Angina","Non-ST Elev MI","ST-Elev MI"]#,"nan"]
    label_stoi={s:i for i,s in enumerate(label_itos)}
    df_clinical["label"]=df_clinical.FinalDx.apply(lambda x: label_stoi[x])
    df_clinical["label_txt"]=df_clinical.FinalDx
    df_clinical["dataset"]="THEW Chest Pain LR"
    df_clinical = df_clinical[~df_clinical.StudyID.isna()]
    df_clinical = df_clinical.astype({'StudyID': 'int32'})
    df_clinical = df_clinical.set_index("StudyID")


    res = []
    for file in tqdm(list(data_path.glob('**/*.ecg'))):
        metadata, filename_out, filename_out_relative = thew_to_np(file, target_fs=target_fs, channels=channels, max_length_seconds=max_length_seconds, channel_stoi=channel_stoi, fs=180, target_folder=target_folder, skimage_transform=skimage_transform)
        res.append({"ecg_id":file.stem.replace("S-","").replace("S","").lstrip(), "patient_id":int(file.stem.replace("S-","").replace("S","").lstrip()[:4]), "data":filename_out_relative, "length": metadata["length"], "channels_available": metadata["channels_available"]})
    df = pd.DataFrame(res)

    df = df.join(df_clinical, how="left", on="patient_id")
    df_error= list(df[df["ECG file"].isna()].ecg_id)
    if(len(df_error)>0):
        print("Skipping the following ECG files that could not be matched to diagnoses:",df_error)
        df = df[~df["ECG file"].isna()]
    return df, label_itos


def prepare_data_thew(data_folder, max_length_seconds=0, target_folder=None, channel_stoi=channel_stoi_default, channels=12, target_fs=100, strat_folds=10, cv_random_state=42,recreate_data=False,skimage_transform=True):
    target_root_thew = Path(".") if target_folder is None else target_folder
    if(recreate_data):
        #reformat data
        df_thew, lbl_itos_thew = reformat_data_thew(data_path=data_folder,max_length_seconds=max_length_seconds,channel_stoi=channel_stoi,target_fs=target_fs,channels=channels,target_folder=target_folder,skimage_transform=skimage_transform)

        #cv splits
        df_patients = df_thew.groupby("patient_id")["label"].apply(lambda x: list(x))
        patients_ids = list(df_patients.index)
        patients_labels = list(df_patients)
        patients_num_ecgs = list(df_patients.apply(len))

        stratified_ids = stratify(patients_labels, range(6), [1./strat_folds]*strat_folds, samples_per_group=patients_num_ecgs)
        stratified_patient_ids = [[patients_ids[i] for i in fold] for fold in stratified_ids]

        df_thew["strat_fold"]=-1
        for i,split in enumerate(stratified_patient_ids):
            df_thew.loc[df_thew.patient_id.isin(split),"strat_fold"]=i

        #add means and std
        dataset_add_mean_col(df_thew,data_folder=target_root_thew)
        dataset_add_std_col(df_thew,data_folder=target_root_thew)
        dataset_add_length_col(df_thew,data_folder=target_root_thew)
        #dataset_add_median_col(df_thew,data_folder=target_root_thew)
        #dataset_add_iqr_col(df_thew,data_folder=target_root_thew)

        #save means and stds
        mean_thew, std_thew = dataset_get_stats(df_thew)

        #save
        save_dataset(df_thew, lbl_itos_thew, mean_thew, std_thew, target_root_thew)
    else:
        df_thew, lbl_itos_thew, mean_thew, std_thew = load_dataset(target_root_thew)
    return df_thew,lbl_itos_thew, mean_thew, std_thew

# Cell

def prepare_data_cinc(data_path, datasets=["ICBEB2018","ICBEB2018_2","INCART","PTB","PTB-XL","Georgia"], target_fs=100, strat_folds=10, channels=8, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True, recreate_data=True):
    '''unzip archives into separate folders with dataset names from above'''
    target_root = Path(".") if target_folder is None else target_folder
    target_root.mkdir(parents=True, exist_ok=True)

    if(recreate_data is True):
        dx_meta = pd.concat([pd.read_csv(data_path/"dx_mapping_scored.csv"),pd.read_csv(data_path/"dx_mapping_unscored.csv")],sort=True)
        dx_mapping_snomed_abbrev = {a:b for [a,b] in list(dx_meta.apply(lambda row: [row["SNOMED CT Code"],row["Abbreviation"]],axis=1))}

        metadata = []
        for filename in tqdm(list(data_path.glob('**/*.hea'))):
            if(not(filename.parts[-2] in datasets)):
                continue
            sigbufs, header = wfdb.rdsamp(str(filename)[:-4])
            #print(filename,sigbufs.shape,np.min(sigbufs,axis=0),np.any(np.isnan(sigbufs)))
            if(np.any(np.isnan(sigbufs))):
                print("Warning:",str(filename),"is corrupt. Skipping.")
                continue
            data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=header['sig_name'],fs=header['fs'],target_fs=target_fs,channels=channels,skimage_transform=skimage_transform)
            assert(target_fs<=header['fs'])
            np.save(target_root/(filename.stem+".npy"),data)
            labels=[]
            age=np.nan
            sex="nan"
            for l in header["comments"]:
                arrs = l.strip().split(' ')
                if l.startswith('Dx:'):
                    labels = [dx_mapping_snomed_abbrev[int(x)] for x in arrs[1].split(',')]
                elif l.startswith('Age:'):
                    try:
                        age = int(arrs[1])
                    except:
                        age= np.nan
                elif l.startswith('Sex:'):
                    sex = arrs[1].strip().lower()
                    if(sex=="m"):
                        sex="male"
                    elif(sex=="f"):
                        sex="female"

            metadata.append({"data":Path(filename.stem+".npy"),"label":labels,"sex":sex,"age":age,"dataset":"cinc_"+filename.parts[-2]})
        df =pd.DataFrame(metadata)
        lbl_itos = np.unique([item for sublist in list(df.label) for item in sublist])
        lbl_stoi = {s:i for i,s in enumerate(lbl_itos)}
        df["label"] = df["label"].apply(lambda x: [lbl_stoi[y] for y in x])

        #does not incorporate patient-level split
        df["strat_fold"]=-1
        for ds in np.unique(df["dataset"]):
            print("Creating CV folds:",ds)
            dfx = df[df.dataset==ds]
            idxs = np.array(dfx.index.values)
            lbl_itosx = np.unique([item for sublist in list(dfx.label) for item in sublist])
            stratified_ids = stratify(list(dfx["label"]), lbl_itosx, [1./strat_folds]*strat_folds)

            for i,split in enumerate(stratified_ids):
                df.loc[idxs[split],"strat_fold"]=i

        #add means and std
        dataset_add_mean_col(df,data_folder=target_root)
        dataset_add_std_col(df,data_folder=target_root)
        dataset_add_length_col(df,data_folder=target_root)

        #save means and stds
        mean, std = dataset_get_stats(df)

        #save
        save_dataset(df, lbl_itos, mean, std, target_root)
    else:
        df, lbl_itos, mean, std = load_dataset(target_root)
    return df, lbl_itos, mean, std

# Cell
def prepare_data_zheng(data_path, denoised=False, target_fs=100, strat_folds=10, channels=8, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True, recreate_data=True):
    '''prepares the Zheng et al 2020 dataset'''
    target_root = Path(".") if target_folder is None else target_folder
    target_root.mkdir(parents=True, exist_ok=True)

    if(recreate_data is True):
        #df_attributes = pd.read_excel("./AttributesDictionary.xlsx")
        #df_conditions = pd.read_excel("./ConditionNames.xlsx")
        #df_rhythm = pd.read_excel("./RhythmNames.xlsx")
        df = pd.read_excel(data_path/"Diagnostics.xlsx")
        df["id"]=df.FileName
        df["data"]=df.FileName.apply(lambda x: x+".npy")
        df["label_condition_txt"]=df.Beat.apply(lambda x: [y for y in x.split(" ") if x!="NONE"])
        df["label_rhythm_txt"]=df.Rhythm.apply(lambda x: x.split(" "))
        df["label_txt"]=df.apply(lambda row: row["label_condition_txt"]+row["label_rhythm_txt"],axis=1)
        df["sex"]=df.Gender.apply(lambda x:x.lower())
        df["age"]=df.PatientAge
        df.drop(["Gender","PatientAge","Rhythm","Beat","FileName"],inplace=True,axis=1)

        #map to numerical indices
        lbl_itos={}
        lbl_stoi={}
        lbl_itos["all"] = np.unique([item for sublist in list(df.label_txt) for item in sublist])
        lbl_stoi["all"] = {s:i for i,s in enumerate(lbl_itos["all"])}
        df["label"] = df["label_txt"].apply(lambda x: [lbl_stoi["all"][y] for y in x])
        lbl_itos["condition"] = np.unique([item for sublist in list(df.label_condition_txt) for item in sublist])
        lbl_stoi["condition"] = {s:i for i,s in enumerate(lbl_itos["condition"])}
        df["label_condition"] = df["label_condition_txt"].apply(lambda x: [lbl_stoi["condition"][y] for y in x])
        lbl_itos["rhythm"] = np.unique([item for sublist in list(df.label_rhythm_txt) for item in sublist])
        lbl_stoi["rhythm"] = {s:i for i,s in enumerate(lbl_itos["rhythm"])}
        df["label_rhythm"] = df["label_rhythm_txt"].apply(lambda x: [lbl_stoi["rhythm"][y] for y in x])
        df["dataset"]="Zheng2020"

        for id,row in tqdm(list(df.iterrows())):
            fs = 500.

            df_tmp = pd.read_csv(data_path/("ECGDataDenoised" if denoised else "ECGData")/(row["id"]+".csv"))
            channel_labels = list(df_tmp.columns)
            sigbufs = np.array(df_tmp)*0.001 #assuming data is given in muV

            data = resample_data(sigbufs=sigbufs,channel_stoi=channel_stoi,channel_labels=channel_labels,fs=fs,target_fs=target_fs,channels=channels,skimage_transform=skimage_transform)
            assert(target_fs<=fs)
            np.save(target_root/(row["id"]+".npy"),data)

        stratified_ids = stratify(list(df["label_txt"]), lbl_itos["all"], [1./strat_folds]*strat_folds)
        df["strat_fold"]=-1
        idxs = np.array(df.index.values)
        for i,split in enumerate(stratified_ids):
            df.loc[idxs[split],"strat_fold"]=i

        #add means and std
        dataset_add_mean_col(df,data_folder=target_root)
        dataset_add_std_col(df,data_folder=target_root)
        dataset_add_length_col(df,data_folder=target_root)

        #save means and stds
        mean, std = dataset_get_stats(df)

        #save
        save_dataset(df, lbl_itos, mean, std, target_root)
    else:
        df, lbl_itos, mean, std = load_dataset(target_root)
    return df, lbl_itos, mean, std

# Cell
def prepare_data_ribeiro(data_path, denoised=False, target_fs=100, strat_folds=10, channels=8, channel_stoi=channel_stoi_default, target_folder=None, skimage_transform=True, recreate_data=True):
    '''prepares test set of Ribeiro et al Nat Comm 2020'''
    data_path = Path(data_path)
    target_root = Path(".") if target_folder is None else target_folder
    target_root.mkdir(parents=True, exist_ok=True)

    if(recreate_data is True):
        lbl_itos = ["1AVB","RBBB","LBBB","SBRAD","AFIB","STACH"]
        channel_labels = ["i","ii","iii","avr","avl","avf","v1","v2","v3","v4","v5","v6"]
        fs= 400.
        #prepare df
        df_cardiologist1 = pd.read_csv(data_path/"annotations"/"cardiologist1.csv")
        df_cardiologist2 = pd.read_csv(data_path/"annotations"/"cardiologist2.csv")
        df_gold = pd.read_csv(data_path/"annotations"/"gold_standard.csv")
        df_cardiology_residents = pd.read_csv(data_path/"annotations"/"cardiology_residents.csv")
        df_dnn = pd.read_csv(data_path/"annotations"/"dnn.csv")
        df_emergency_residents = pd.read_csv(data_path/"annotations"/"emergency_residents.csv")
        df_medical_students = pd.read_csv(data_path/"annotations"/"medical_students.csv")
        df_attributes = pd.read_csv(data_path/"attributes.csv")
        sex_map = {"M":"male", "F":"female"}
        df_attributes.sex = df_attributes.sex.apply(lambda x: sex_map[x])

        def reformat_predictions(df, colname_txt="label_txt", colname_num="label", lbl_itos=["1AVB","RBBB","LBBB","SBRAD","AFIB","STACH"]):
            lbl_stoi = {s:i for i,s in enumerate(lbl_itos)}
            df[colname_txt]=df.apply(lambda row: ("1AVB " if row["1dAVb"] else "")+("RBBB " if row["RBBB"] else "")+("LBBB " if row["LBBB"] else "")+("SBRAD " if row["SB"] else "")+("AFIB " if row["AF"] else "")+("STACH " if row["ST"] else ""),axis=1)
            df[colname_txt]=df[colname_txt].apply(lambda x: [y for y in x.strip().split(" ") if y!=""])
            df[colname_num]=df[colname_txt].apply(lambda x: [lbl_stoi[y] for y in x if y in lbl_stoi.keys()])
            df.drop(["1dAVb","RBBB","LBBB","SB","AF","ST"],axis=1,inplace=True)

        reformat_predictions(df_cardiologist1,"label_cardiologist1_txt","label_cardiologist1")
        reformat_predictions(df_cardiologist2,"label_cardiologist2_txt","label_cardiologist2")
        reformat_predictions(df_gold,"label_txt","label")
        reformat_predictions(df_cardiology_residents,"label_cardiology_residents_txt","label_cardiology_residents")
        reformat_predictions(df_emergency_residents,"label_emergency_residents_txt","label_emergency_residents")
        reformat_predictions(df_medical_students,"label_medical_students_txt","label_medical_students")
        reformat_predictions(df_dnn,"label_dnn_txt","label_dnn")

        df=df_gold.join([df_cardiologist1,df_cardiologist2,df_cardiology_residents,df_emergency_residents,df_medical_students,df_dnn,df_attributes])
        df["data"]=["Ribeiro_test_"+str(i)+".npy" for i in range(len(df))]
        df["dataset"]="Ribeiro_test"
        #prepare raw data
        with h5py.File(data_path/"ecg_tracings.hdf5", "r") as f:
            sigbufs = np.array(f['tracings'])
        start_idxs=[ np.where(np.sum(np.abs(sigbufs[i]),axis=1)==0.)[0] for i in range(len(sigbufs))] #discard zeros at beginning/end
        start_idxs = [len(a)//2 for a in start_idxs]


        for id,row in tqdm(list(df.iterrows())):
            data = resample_data(sigbufs=sigbufs[id][start_idxs[id]:-start_idxs[id] or None],channel_stoi=channel_stoi,channel_labels=channel_labels,fs=fs,target_fs=target_fs,channels=channels,skimage_transform=skimage_transform)
            assert(target_fs<=fs)
            np.save(target_root/(row["data"]),data)

        stratified_ids = stratify(list(df["label_txt"]), lbl_itos, [1./strat_folds]*strat_folds)
        df["strat_fold"]=-1
        idxs = np.array(df.index.values)
        for i,split in enumerate(stratified_ids):
            df.loc[idxs[split],"strat_fold"]=i

        #add means and std
        dataset_add_mean_col(df,data_folder=target_root)
        dataset_add_std_col(df,data_folder=target_root)
        dataset_add_length_col(df,data_folder=target_root)

        #save means and stds
        mean, std = dataset_get_stats(df)

        #save
        save_dataset(df, lbl_itos, mean, std, target_root)
    else:
        df, lbl_itos, mean, std = load_dataset(target_root)
    return df, lbl_itos, mean, std
